#include <windows.h>
#include <memory.h>
#include "view.h"

extern HINSTANCE _HInstance;
inline long ScanBytes(int pixWidth, int bitsPixel) {
  return (((long)pixWidth*bitsPixel+31) / 32) * 4;
  }


struct LocalInfo {
	BYTE Buffer[0x2000];
	WORD BufferSize;
	WORD State;
	RGBQUAD	OrigPal[256];		// original palette
	int		OrigBits;			// original bit count (in case of reduction)
	int		SrcWidth;			// how many bytes (including pad) per source line
	int		DestWidth;			// how many bytes (including pad) per destline
	BYTE 		huge *DestBits;	// where to put everything
	ReduceFn Reduce;				// how to do the transfer
	int		CurLine;				// current display line in output bitmap
	BOOL		IsCore;				// uses bitmap CORE header (old style)

	enum { FILEHEADER, IMAGEHEADER, PALETTE, ENLARGE, DESTPALETTE, BODY };
	void TrimBuff( int size )
		{
		BufferSize -= size;
		memcpy( Buffer, Buffer+size, BufferSize );
		}
	void BuffCopy( void *dest, int size )
		{
		memcpy( dest, Buffer, size );
		BufferSize -= size;
		memcpy( Buffer, Buffer+size, BufferSize );
		}
	int StateCheck( ViewInfo *vi, ViewDisplayFn, LPVOID );
	int FileHeader( ViewInfo *vi );
	int ImageHeader( ViewInfo *vi );
	int GetPalette( ViewInfo *vi );
	int Enlarge( ViewInfo *vi );
	int DestPalette( ViewInfo *vi );
	int ImageData( ViewInfo *vi, ViewDisplayFn view, LPVOID user );
	};

int FAR PASCAL _export
LibMain( HINSTANCE, WORD, WORD, LPSTR )
	{
	void SetupDither();
   SetupDither();
	return TRUE;
	}

int FAR PASCAL _export
WEP ( int )
	{
	return 1;
	}

WORD FAR PASCAL _export
ViewInit( ViewInfo *vi )
	{
	vi->Private = new LocalInfo;
	LocalInfo &li = *(LocalInfo*)vi->Private;
	li.BufferSize = 0;
	li.State = LocalInfo::FILEHEADER;
	vi->ImageData = li.Buffer;
	vi->ImageDataSize = sizeof( li.Buffer );
	return TRUE;
	}

void FAR PASCAL _export
ViewTerm( ViewInfo *vi )
	{
	delete (LocalInfo*)(vi->Private);
	}

int LocalInfo::
FileHeader( ViewInfo *vi )
	{
	if ( BufferSize < sizeof( BITMAPFILEHEADER ) )
		return ViewInfo::MORE;

	BITMAPFILEHEADER *bmfh = (BITMAPFILEHEADER*)(Buffer);
	if ( bmfh->bfType != 'BM' ) {
		vi->Error = ViewInfo::BADFORMAT;
		return ViewInfo::ERR;
		}

	TrimBuff( sizeof(BITMAPFILEHEADER) );
	State = LocalInfo::IMAGEHEADER;
	return 0;
	}

int LocalInfo::
ImageHeader( ViewInfo *vi )
	{
	if ( BufferSize < sizeof( BITMAPINFOHEADER ) )
		return ViewInfo::MORE;

	BITMAPINFOHEADER *bmih = (BITMAPINFOHEADER*)Buffer;
	if ( bmih->biSize == sizeof( BITMAPINFOHEADER ) ) {
		BuffCopy( vi->Bitmap, sizeof( BITMAPINFOHEADER ) );
		IsCore = FALSE;
		}
	else if ( bmih->biSize == sizeof( BITMAPCOREHEADER ) ) {
		BITMAPCOREHEADER *bmch = (BITMAPCOREHEADER*)Buffer;

		vi->Bitmap->biSize = sizeof( BITMAPINFOHEADER );
		vi->Bitmap->biPlanes = 1;
		vi->Bitmap->biBitCount = bmch->bcBitCount;
		vi->Bitmap->biWidth = bmch->bcWidth;
		vi->Bitmap->biHeight = bmch->bcHeight;
		vi->Bitmap->biCompression = BI_RGB;
		vi->Bitmap->biXPelsPerMeter = 0;
		vi->Bitmap->biYPelsPerMeter = 0;
		vi->Bitmap->biClrUsed = 0;
		vi->Bitmap->biClrImportant = 0;
		TrimBuff( sizeof( BITMAPCOREHEADER ) );
		IsCore = TRUE;
		}
	else {
		vi->Error = ViewInfo::BADFORMAT;
		return ViewInfo::ERR;
		}

	// ok, so far, so good... now is there a palette?
	OrigBits = vi->Bitmap->biBitCount;

	if ( OrigBits <=8 )
		State = PALETTE;
	else
		State = ENLARGE;
	return 0;
	}

int LocalInfo::
GetPalette( ViewInfo *vi )
	{
	if ( IsCore ) {
		if ( BufferSize < 3*(1<<OrigBits) )
			return ViewInfo::MORE;

		RGBTRIPLE *from = (RGBTRIPLE*)Buffer;
		RGBQUAD *to = OrigPal;
		int limit=1<<OrigBits;
		while ( limit-- ) {
			to->rgbBlue = from->rgbtBlue;
			to->rgbGreen = from->rgbtGreen;
			to->rgbRed = from->rgbtRed;
			to->rgbReserved = 0;
			to++;
			from++;
			}
		TrimBuff( 3 * (1<<OrigBits) );
		}
	else {
		if ( BufferSize < (4<<OrigBits) )
			return ViewInfo::MORE;
		BuffCopy( OrigPal, 4<<OrigBits );
		}

	memcpy( vi->Bitmap+1, OrigPal, 4<<OrigBits );
	State = ENLARGE;
	return 0;
	}

int LocalInfo::
Enlarge( ViewInfo *vi )
	{
	if ( vi->ImageBits > OrigBits ) {
		if ( ! vi->FattenPixels )
			vi->ImageBits = OrigBits;
		}
	vi->Bitmap->biBitCount = vi->ImageBits;

	vi->EnlargeSize = sizeof( BITMAPINFOHEADER );

	int colorsize = 0;
	if ( vi->Bitmap->biBitCount <= 8 )
		colorsize = 4 << vi->Bitmap->biBitCount;

	vi->EnlargeSize += colorsize;

	vi->EnlargeSize += vi->Bitmap->biHeight
						* ScanBytes( vi->Bitmap->biHeight, vi->Bitmap->biBitCount );

	if ( OrigBits != vi->Bitmap->biBitCount ) {
		// gotta fill in our own palette (currently ignoring any real request)
		RGBQUAD *pal = (RGBQUAD*)(vi->Bitmap+1);
		if ( vi->Bitmap->biBitCount == 8 ) {
			int i=0;
			for ( int r=0; r<256; r+=51 ) {
				for ( int g=0; g<256; g+=51 ) {
					for ( int b=0; b<256; b+=51 ) {
						pal[i].rgbRed = r;
						pal[i].rgbGreen = g;
						pal[i].rgbBlue = b;
						i++;
						}
					}
				}
			}
		if ( vi->Bitmap->biBitCount == 4 ) {
			static int graypal[4] = { 0, 128, 192, 255 };
			for ( int i=0; i<4; i++ ) {
				pal[i].rgbRed = graypal[i];
				pal[i].rgbGreen = graypal[i];
				pal[i].rgbBlue = graypal[i];
				}
			}
		else {
			pal[0].rgbRed = 0;
			pal[0].rgbGreen = 0;
			pal[0].rgbBlue = 0;
			pal[0].rgbReserved = 0;
			pal[1].rgbRed = 255;
			pal[1].rgbGreen = 255;
			pal[1].rgbBlue = 255;
			pal[1].rgbReserved = 0;
			}
		}

	State = DESTPALETTE;
	return ViewInfo::ENLARGE;
	}

int LocalInfo::
DestPalette( ViewInfo *vi )
	{
	// now select transfer method
	Reduce = PickReduction( vi->Bitmap->biBitCount, OrigBits );
	DestWidth = ScanBytes( vi->Bitmap->biWidth, vi->Bitmap->biBitCount );
	SrcWidth = ScanBytes( vi->Bitmap->biWidth, OrigBits );

	// find the destination bit start
	DestBits = (BYTE*)vi->Bitmap;
	DestBits += sizeof( BITMAPINFOHEADER );
	int colorsize = 0;
	if ( vi->Bitmap->biBitCount <= 8 )
		colorsize = 4 << vi->Bitmap->biBitCount;
	DestBits += colorsize;

	CurLine = 0;
	State = BODY;
	return 0;
	}

int LocalInfo::
ImageData( ViewInfo *vi, ViewDisplayFn view, LPVOID user )
	{
	if ( BufferSize < SrcWidth )
		return ViewInfo::MORE;

	(*Reduce)( CurLine, DestBits, ((BITMAPINFO*)(vi->Bitmap))->bmiColors,
				Buffer, OrigPal, vi->Bitmap->biWidth );
	TrimBuff( SrcWidth );
	DestBits += DestWidth;
	(view)( vi->Bitmap, CurLine++, 1, user );
	return 0;
	}

int LocalInfo::
StateCheck( ViewInfo *vi, ViewDisplayFn view, LPVOID data )
	{
	int retval = 0;

	for (;;) {
		switch( State ) {
			case FILEHEADER:	retval = FileHeader( vi );		break;
			case IMAGEHEADER:	retval = ImageHeader( vi );	break;
			case PALETTE:		retval = GetPalette( vi );		break;
			case ENLARGE:		retval = Enlarge( vi );			break;
			case DESTPALETTE:	retval = DestPalette( vi );	break;
			case BODY:			retval = ImageData( vi, view, data );		break;
			}
		if ( retval ) {
			if ( retval==ViewInfo::MORE ) {
				vi->ImageData = Buffer + BufferSize;
				vi->ImageDataSize = sizeof( Buffer ) - BufferSize;
				}
			return retval;
			}
		}
	}

#pragma argsused
WORD FAR PASCAL _export
ViewData( ViewInfo *vi, ViewDisplayFn fn, LPVOID user )
	{
	LocalInfo &li = *(LocalInfo*)vi->Private;
	li.BufferSize += vi->ImageDataSize;

	return li.StateCheck( vi, fn, user );
	}

