unit WcUserDb;

interface

uses
  WcMisc,
  WcType,
  Filer,
  WcGlobal,
  WcDb;

const
  {keys for User database}
  UserNameKey    = 1;
  UserSecKey     = 2;
  UserExpDateKey = 3;
  UserAliasKey   = 4;
  UserIdKey      = 5;
  UserRealKey    = 6;

type
  PUserConfDatabase = ^TUserConfDatabase;
  TUserConfDatabase = object
  private { shouldn't need to access anything in here from outside this unit }
    constructor Init(PageRecs : Word);
    destructor Done; virtual;
    procedure GetPage(var UserRec : TUserRec; Page : Word; var ConfPage : TUserConfPage);
    procedure SavePage(const UserRec : TUserRec; Page : Word; var ConfPage : TUserConfPage);
  private
    F : File;
    IndexSize : Word;
    DataSize : Word;
  end;

  PUserDatabase = ^TUserDatabase;
  TUserDatabase = object(TFileBlock)
    UserConfDb : TUserConfDatabase;
    constructor Init;
    destructor Done; virtual;
    procedure GetCreateInfo(var DataLen : Word; var Keys : Integer; var IID : IsamIndDescr); virtual;
    function BuildKey(const Data; Key : Integer) : IsamKeyStr; virtual;
{$IFDEF UserDatabaseAdd}
    function AddRecord(var RefNr : LongInt; var Data) : Boolean; virtual;
{$ENDIF}
    procedure UpdateRecordID(ID : LongInt; var NewData);
    procedure FatalDBError(const S : String); virtual;
    procedure LogDBError(const S : String); virtual;
  end;

  PUserWrapper = ^TUserWrapper;
  TUserWrapper = object
    PageSize : Word;
    UserPtr  : PUserRec;
    CurPage  : Integer;
    ConfPage : TUserConfPage;
    constructor Init(var UserRec : TUserRec);
    destructor Done; virtual;
    procedure SetDirty;
    function GetFlags(Conf : Word) : Byte;
    function FlagIsSet(Mask : Byte; Conf : Word) : Boolean;
    function NextSet(Mask : Byte; Current : Word) : Word;
    function PrevSet(Mask : Byte; Current : Word) : Word;
    function FirstSet(Mask : Byte) : Word;
    function LastSet(Mask : Byte) : Word;
    function FlagsSet(Mask : Byte) : Word;
    procedure SetAllFlags(Mask : Byte);
    procedure ClearAllFlags(Mask : Byte);
    procedure ToggleFlag(Mask : Byte; Conf : Word);
    function GetLastRead(Conf : Word) : Word;
    function GetFirstUnread(Conf : Word) : Word;
    procedure SetFlags(Conf : Word; NewFlags : Byte);
    procedure SetLastRead(Conf, NewLastRead : Word);
    procedure SetFirstUnread(Conf, NewFirstUnread: Word);
    procedure SetFlags_LastRead(Conf : Word; NewFlags : Byte; NewLastRead : Word);
    function GetConfPage(Conf : Word) : Integer;
    procedure LoadConfPage(Conf : Word; ForceLoad : Boolean);
    procedure SaveConfPage;
  end;

function SwitchLast(const Name : String) : String;
function BuildUserNameKey(const Name : String; UserID : LongInt) : IsamKeyStr;
function BuildUserIDKey(IDName : LongInt) : IsamKeyStr;

var
  UserDb : TUserDatabase;
  UserConfPtr : PUserWrapper;

const
  UserDBOpen : Boolean = False;

implementation

type
  TUserConfFileHeader = record
    TotalConfs: Word;
  end;


  function LongFlagIsSet(Flags, FlagMask : LongInt) : Boolean;
    inline($5B/$59/$58/$5A/$21/$D8/$21/$CA/$09/$D0/$74/$02/$B0/$01);


  function UserConfMaxPages : Word;
  const
    MaxChunk = 1024;

  begin
    UserConfMaxPages := (LongInt(MwConfig.MaxConfAreas) + MaxChunk - 1) div MaxChunk;
  end;


  constructor TUserConfDatabase.Init(PageRecs : Word);
  var
    Header : TUserConfFileHeader;
    Nr : Word;
    Fm : Byte;

  begin
    Indexsize := 3 + UserConfMaxPages * SizeOf(Longint);
    DataSize := SizeOf(TUserConfPageHeader) + PageRecs * SizeOf(TUserConfData);
    Assign(F, MwConfig.UserDatabasePath+'USERCONF.DAT');
    Fm := FileMode;
    FileMode := $42;
    Reset(F, 1);
    FileMode := Fm;
    if IoResult <> 0 then begin
      ReWrite(F, 1);
      if IoResult <> 0 then
        begin
          IsamError := 10075; {return Write Error in this case}
          Fail;
        end;
      Header.TotalConfs := MwConfig.MaxConfAreas;
      BlockWrite(F, Header, SizeOf(Header));
      Close(F);
      if IoResult = 0 then
        {ignore};
      Fm := FileMode;
      FileMode := $42;
      Reset(F, 1);
      FileMode := Fm;
      if IoResult <> 0 then
        begin
          IsamError := 10070; {return Open Error in this case}
          Fail;
        end;
    end;
    Seek(F, 0);
    BlockRead(F, Header, SizeOf(Header), Nr);
    if (IoResult <> 0) or (Nr <> SizeOf(Header)) or (Header.TotalConfs <> MwConfig.MaxConfAreas) then
      begin
        Close(F);
        if IoResult = 0 then
          {ignore};
        IsamError := 9907; {return Error in file size}
        Fail;
      end;
  end;


  destructor TUserConfDatabase.Done;
  begin
    Close(f);
    if IoResult <> 0 then
      {ignore};
  end;


  procedure TUserConfDatabase.GetPage(var UserRec: TUserRec; Page: Word; var ConfPage: TUserConfPage);
  var
    Io : Word;
    OldUser : TUserRec;
    Offsets : TUserConfIndex;
    Ofs : Longint;

  begin
    if UserRec.UserConfData > 0 then begin
      Seek(F, UserRec.UserConfData);
      BlockRead(F, Offsets, IndexSize);
      Io := IoResult;
      if Io <> 0 then
        LogFatalError('I/O Error reading USERCONF.DAT', Io);
    end else begin
      OldUser := Userrec;
      UserRec.UserConfData := FileSize(F);
      UserDb.UpdateRecord(OldUser, UserRec);
      FillChar(Offsets, IndexSize, 0);
      Offsets.RecLen := IndexSize;
      Offsets.RecType := ucrIndex;
      Seek(F, UserRec.UserConfData);
      BlockWrite(F, Offsets, IndexSize);
      Io := IoResult;
      if Io <> 0 then
        LogFatalError('I/O Error writing USERCONF.DAT', Io);
    end;
    if Offsets.Offsets[Page] > 0 then begin
      Seek(F, Offsets.Offsets[Page]);
      BlockRead(F, ConfPage, DataSize);
      Io := IoResult;
      if Io <> 0 then
        LogFatalError('I/O Error reading USERCONF.DAT', Io);
      if ConfPage.Page <> Page then
        LogFatalError('USERCONF.DAT needs repair - run WCREPAIR', 0);
      end
    else begin
      Offsets.Offsets[Page] := FileSize(F);
      Seek(F, UserRec.UserConfData);
      BlockWrite(F, Offsets, IndexSize);
      Io := IoResult;
      if Io <> 0 then
        LogFatalError('I/O Error writing USERCONF.DAT', Io);
      Ofs := Offsets.Offsets[Page];
      FillChar(confpage, SizeOf(ConfPage), 0);
      ConfPage.RecLen := DataSize;
      ConfPage.RecType := ucrData;
      ConfPage.UserID := UserRec.UserID;
      ConfPage.Page := Page;
      ConfPage.This := Ofs;
      Seek(F, Ofs);
      BlockWrite(F, ConfPage, DataSize);
      Io := IoResult;
      if Io <> 0 then
        LogFatalError('I/O Error writing USERCONF.DAT', Io);
    end;
  end;


  procedure TUserConfDatabase.SavePage(const UserRec : TUserRec; Page: Word; var ConfPage: TUserConfPage);
  var
    Io: Word;

  begin
    if ConfPage.Page <> Page then {sanity check}
      Exit;
    Seek(F, ConfPage.This);
    BlockWrite(F, ConfPage, DataSize);
    Io := IoResult;
    if Io <> 0 then
      LogFatalError('I/O Error writing USERCONF.DAT', Io);
  end;


(**********************************************)


{$IFDEF R+}
{$DEFINE rwasplus}
{R-}
{$ENDIF}
  function SwitchLast(const Name : String) : String;
  var
    X, Y : Byte;

  begin
    Y := Length(Name);
    X := Y;
    while (Y > 0) and (Name[Y] <> ' ') do
      Dec(Y);
    if Y = 0 then
      SwitchLast := Name
    else
      SwitchLast := Copy(Name, Succ(Y), X-Y) + ' ' + Copy(Name, 1, Pred(Y));
  end;
{$IFDEF rwasplus}
{$UNDEF rwasplus}
{$R+}
{$ENDIF}


  function BuildUserNameKey(const Name : String; UserID : LongInt) : IsamKeyStr;
  begin
    BuildUserNameKey := Pad(StUpcase(SwitchLast(Name)), 25)+Long2Key(UserID);
  end;


  function BuildUserIDKey(IDName : LongInt) : IsamKeyStr;
  begin
    BuildUserIDKey := Long2Key(IDName);
  end;


  function UserConfPageRecords : Word;
  const
    MaxChunk = 1024;

  var
    Chunks : Word;

  begin
    Chunks := (LongInt(MwConfig.MaxConfAreas) + MaxChunk - 1) div MaxChunk;
    UserConfPageRecords := ((LongInt(MwConfig.MaxConfAreas) + Chunks - 1) div Chunks);
  end;


  constructor TUserDatabase.Init;
  begin
    with MwConfig do
      begin
        if not inherited Init(UserDatabasePath+'ALLUSERS', DatabaseMode = dbSaveMode,
                              Network <> WcType.NoNet, UserIdKey, SizeOf(TUserRec)) then
          Fail;
        if not UserConfDb.Init(UserConfPageRecords) then
          begin
            inherited Done;
            LogFatalError('Error opening USERCONF.DAT', IsamError);
            Fail;
          end;
        UserDBOpen := True;
      end;
  end;


  destructor TUserDatabase.Done;
  begin
    UserConfDb.Done;
    inherited Done;
    UserDBOpen := False;
  end;


  procedure TUserDatabase.GetCreateInfo(var DataLen : Word; var Keys : Integer; var IID : IsamIndDescr);
  begin
    DataLen := SizeOf(TUserRec);
    Keys := 6;
    IID[1].KeyL := 30;         {UserName Key}
    IID[1].AllowDupK := False;
    IID[2].KeyL := 35;         {SecLevel + UserName Key}
    IID[2].AllowDupK := True;
    IID[3].KeyL := 3;          {Expired Date Key}
    IID[3].AllowDupK := True;
    IID[4].KeyL := 25;         {User Alias Key}
    IID[4].AllowDupK := False;
    IID[5].KeyL := 5;          {User ID Key}
    IID[5].AllowDupK := False;
    IID[6].KeyL := 25;         {User Real Name}
    IID[6].AllowDupK := True;
  end;


  function TUserDatabase.BuildKey(const Data; Key : Integer) : IsamKeyStr;
  var
    UserRec : TUserRec absolute Data;

  begin
    with UserRec do
      case Key of
        1 : BuildKey := BuildUserNameKey(UserName, UserID);
        2 : BuildKey := Pad(SecLevel, 10)+StUpcase(SwitchLast(UserName));
        3 : BuildKey := Word2Key(ExpireDate);
        4 : if UserRec.Alias = '' then
              BuildKey := ''
            else
              BuildKey := StUpcase(Alias);
        5 : BuildKey  := BuildUserIDKey(UserId);
        6 : BuildKey  := StUpcase(UserName);
      end;
  end;


{$IFDEF UserDatabaseAdd}
  function TUserDatabase.AddRecord(var RefNr : LongInt; var Data) : Boolean;
  var
    UserRec : TUserRec absolute Data;
    ConfPage : TUserConfPage;
    RefKey : IsamKeyStr;
    UserRef : LongInt;

  begin
    AddRecord := False;
    Lock;
    if MwConfig.DupUserLevel <> duAllow then
      if FindKey(UserRealKey, RefNr, BuildKey(UserRec, UserRealKey)) then
        begin
          Unlock;
          Exit;
        end;
    Unlock; {we unlock here to prevent deadlock with master file}
    ReadMInfo(True);
    ClearKey(UserIDKey);
    PrevKey(UserIDKey, UserRef, RefKey);
    if IsamOk then
      UserRef := Key2Long(RefKey)
    else
      UserRef := 0;
    if UserRef > MasterInfo.HighestUserId then
      MasterInfo.HighestUserId := UserRef + 1
    else
      Inc(MasterInfo.HighestUserId);
    WriteMInfo;
    Lock;
    (*
      we now have to recheck the duplicate situation, in weird cases we make
      increment the highest user id without adding a new user but this is
      required to prevent deadlock situations
    *)
    if MwConfig.DupUserLevel <> duAllow then
      if FindKey(UserRealKey, RefNr, BuildKey(UserRec, UserRealKey)) then
        begin
          Unlock;
          Exit;
        end;
    UserRec.UserId := MasterInfo.HighestUserId;
    UserRec.UserConfData := 0;
    if inherited AddRecord(RefNr, Data) then
      begin
        AddRecord := True;
        UserConfDb.GetPage(UserRec, 0, ConfPage);
      end;
    Unlock;
  end;
{$ENDIF}


  procedure TUserDatabase.UpdateRecordID(ID : LongInt; var NewData);
  begin
    inherited UpdateRecordKey(BuildUserIDKey(ID), NewData);
  end;


  procedure TUserDatabase.FatalDBError(const S : String);
  begin
    LogFatalError('USERS DATABASE : '+ S, IsamError);
  end;


  procedure TUserDatabase.LogDBError(const S : String);
  begin
    LogError('USERS DATABASE : '+ S, IsamError);
  end;


(*********************************************)


  constructor TUserWrapper.Init(var UserRec : TUserRec);
  begin
    PageSize := UserConfPageRecords;
    UserPtr := @UserRec;
    CurPage := -1;
  end;


  destructor TUserWrapper.Done;
  begin
  end;


  procedure TUserWrapper.SetDirty;
  begin
    CurPage := -1;
  end;


  function TUserWrapper.GetFlags(Conf : Word) : Byte;
  begin
    LoadConfPage(Conf, False);
    GetFlags := ConfPage.UserConfData[Conf mod PageSize].cuFlags;
  end;


  function TUserWrapper.GetLastRead(Conf : Word) : Word;
  begin
    LoadConfPage(Conf, False);
    GetLastRead := ConfPage.UserConfData[Conf mod PageSize].cuLastRead;
  end;


  function TUserWrapper.GetFirstUnread(Conf : Word) : Word;
  begin
    LoadConfPage(Conf, False);
    GetFirstUnread := ConfPage.UserConfData[Conf mod PageSize].cuFirstUnread;
  end;


  procedure TUserWrapper.SetFlags(Conf : Word; NewFlags : Byte);
  begin
    UserDb.Lock;
    LoadConfPage(Conf, True);
    ConfPage.UserConfData[Conf mod PageSize].cuFlags := NewFlags;
    SaveConfPage;
    UserDb.UnLock;
  end;


  procedure TUserWrapper.SetLastRead(Conf, NewLastRead : Word);
  begin
    UserDb.Lock;
    LoadConfPage(Conf, True);
    ConfPage.UserConfData[Conf mod PageSize].cuLastRead := NewLastRead;
    SaveConfPage;
    UserDb.UnLock;
  end;


  procedure TUserWrapper.SetFirstUnread(Conf, NewFirstUnread: Word);
  begin
    UserDb.Lock;
    LoadConfPage(Conf, True);
    ConfPage.UserConfData[Conf mod PageSize].cuFirstUnread := NewFirstUnread;
    SaveConfPage;
    UserDb.UnLock;
  end;


  procedure TUserWrapper.SetFlags_LastRead(Conf : Word; NewFlags : Byte; NewLastRead : Word);
  begin
    UserDb.Lock;
    LoadConfPage(Conf, True);
    ConfPage.UserConfData[Conf mod PageSize].cuFlags := NewFlags;
    ConfPage.UserConfData[Conf mod PageSize].cuLastRead := NewLastRead;
    SaveConfPage;
    UserDb.UnLock;
  end;


  function TUserWrapper.GetConfPage(Conf : Word) : Integer;
  begin
    GetConfPage := Conf div PageSize;
  end;


  procedure TUserWrapper.LoadConfPage(Conf : Word; ForceLoad : Boolean);
  var
    LoadPage : Word;

  begin
    LoadPage := Conf div PageSize;
    if (LoadPage <> CurPage) or ForceLoad then
      begin
        CurPage := LoadPage;
        UserDb.Lock;
        UserDb.UserConfDb.GetPage(UserPtr^, CurPage, ConfPage);
        UserDb.UnLock;
      end;
  end;


  procedure TUserWrapper.SaveConfPage;
  begin
    if CurPage <> -1 then begin
      UserDb.Lock;
      UserDb.UserConfDb.SavePage(UserPtr^, CurPage, ConfPage);
      UserDb.UnLock;
    end;
  end;


  function TUserWrapper.FlagIsSet(Mask : Byte; Conf : Word) : Boolean;
  begin
    FlagIsSet := GetFlags(Conf) and Mask = Mask;
  end;


  function TUserWrapper.NextSet(Mask : Byte; Current : Word) : Word;
  var
    I : Word;

  begin
    I := Current;
    Inc(I);
    while (I < MwConfig.MaxConfAreas) and not FlagIsSet(Mask, I) do
      Inc(I);
    if I < MwConfig.MaxConfAreas then
      NextSet := I
    else
      NextSet := NoMoreBits;
  end;


  function TUserWrapper.PrevSet(Mask : Byte; Current: Word) : Word;
  var
    I : Word;

  begin
    I := Current;
    Dec(I);
    while (I >= 0) and not FlagIsSet(Mask, I) do
      Dec(I);
    if I >= 0 then
      PrevSet := I
    else
      PrevSet := NoMoreBits;
  end;


  function TUserWrapper.FirstSet(Mask : Byte) : Word;
  begin
    FirstSet := NextSet(Mask, Word(-1));
  end;


  function TUserWrapper.LastSet(Mask : Byte) : Word;
  begin
    LastSet := PrevSet(Mask, MwConfig.MaxConfAreas);
  end;


  function TUserWrapper.FlagsSet(Mask : Byte) : Word;
  var
    I, Total : Word;

  begin
    Total := 0;
    for I := 0 to MwConfig.MaxConfAreas - 1 do
      if FlagIsSet(Mask, I) then
        Inc(Total);
    FlagsSet := Total;
  end;


  procedure TUserWrapper.SetAllFlags(Mask : Byte);
  var
    I : Word;

  begin
    SetDirty; {!! Needed to make sure we reload from disk before doing
                  any IO, else we might overwrite changes made by someone
                  else
              }

    for I := 0 to MwConfig.MaxConfAreas - 1 do
      SetFlags(I, GetFlags(I) or Mask);
  end;


  procedure TUserWrapper.ClearAllFlags(Mask : Byte);
  var
    I : Word;

  begin
    SetDirty; {!! Needed to make sure we reload from disk before doing
                  any IO, else we might overwrite changes made by someone
                  else
              }
    for I := 0 to MwConfig.MaxConfAreas - 1 do
      SetFlags(I, GetFlags(I) and not Mask);
  end;


  procedure TUserWrapper.ToggleFlag(Mask : Byte; Conf : Word);
  var
    Flags : Byte;

  begin
    SetDirty; {!! Needed to make sure we reload from disk before doing
                  any IO, else we might overwrite changes made by someone
                  else
              }
    Flags := GetFlags(Conf);
    if Flags and Mask = Mask then
      Flags := Flags and not Mask
    else
      Flags := Flags or Mask;
    SetFlags(Conf, Flags);
  end;


end.
