unit WcFileDb;

interface

uses
  WcType,
  Filer,
  QxStub,
  QxIndex,
  WcDb,
  WcMisc,
  WcGlobal;

const
  FileAreaKey = 1;
  FileNameKey = 2;
  FileDateKey = 3;
  FileUpKey   = 4;

type
  PFileDatabase = ^TFileDatabase;
  TFileDatabase = object(TVFileBlock)
    IndexDb : TIndexFile;
    constructor Init;
    destructor Done; virtual;
    procedure GetCreateInfo(var Len : Word; var Keys : Integer; var IID : IsamIndDescr); virtual;
    function DataLen(const Data) : Word; virtual;
    function BuildKey(const Rec; Key : Integer) : IsamKeyStr; virtual;
    function AddRecord(var RefNr : LongInt; var Data) : Boolean; virtual;
    procedure PutRecordPrim(RefNr : LongInt; var OldData, NewData); virtual;
    procedure UpdateRecordName(Area : Word; const Name : String; var NewData);
    procedure FatalDBError(const S : String); virtual;
    procedure LogDBError(const S : String); virtual;
  end;

procedure AddIndexRecord(RefNr : LongInt; const Data);
function PackFileArea(Area : Word; const Name : String) : IsamKeyStr;
function PackFileName(const Name : String; Area : Word) : IsamKeyStr;

var
  FileDb : TFileDatabase;

const
  FileDBOpen : Boolean = False;

implementation

const
  WordChars : set of Char = ['!', '#', '$', '''', '-', '.', '0'..'9', '@'];
  StartChars : set of Char = ['!', '#', '$', '0'..'9'];
  EndChars : set of Char = ['0'..'9'];


  function LongFlagIsSet(Flags, FlagMask : LongInt) : Boolean;
    inline($5B/$59/$58/$5A/$21/$D8/$21/$CA/$09/$D0/$74/$02/$B0/$01);


  function PackFileArea(Area : Word; const Name : String) : IsamKeyStr;
  begin
    PackFileArea := Word2Key(Area)+StUpcase(Pad(Name, 12));
  end;


  function PackFileName(const Name : String; Area : Word) : IsamKeyStr;
  begin
    PackFileName := StUpcase(Pad(Name, 12))+Word2Key(Area);
  end;


  procedure TFileDatabase.GetCreateInfo(var Len : Word; var Keys : Integer; var IID : IsamIndDescr);
  begin
    Len := 316;
    Keys := 4;
    IID[1].KeyL := 15;  {Area + Filename}
    IID[1].AllowDupK := False;
    IID[2].KeyL := 15;  {Filename + Area}
    IID[2].AllowDupK := False;
    IID[3].KeyL := 11;   {File date}
    IID[3].AllowDupK := True;
    IID[4].KeyL := 30;  {Uploader name + Uploader ID key}
    IID[4].AllowDupK := True;
  end;


  constructor TFileDatabase.Init;
  begin
    if not inherited Init(MwConfig.FileDatabasePath+'ALLFILES',
                          MwConfig.DatabaseMode = dbSaveMode,
                          MwConfig.Network <> WcType.NoNet,
                          FileAreaKey, SizeOf(TFileRec)) then
      Fail;
    if not IndexDb.Init(MwConfig.FileDatabasePath+'ALLFILES.QX') then begin
      inherited Done;
      Fail;
    end;
    FileDBOpen := True;
  end;


  destructor TFileDatabase.Done;
  begin
    IndexDb.Done;
    inherited Done;
    FileDBOpen := False;
  end;


  function TFileDatabase.DataLen(const Data) : Word;
  var
    FileRec : TFileRec absolute Data;

  begin
    DataLen := SizeOf(TFileHeader) + FileRec.MsgBytes;
  end;


  function TFileDatabase.BuildKey(const Rec; Key : Integer) : IsamKeyStr;
  var
    FileRec : TFileRec absolute Rec;
    KeyStr : IsamKeyStr;

  begin
    with FileRec do
      case Key of
        FileAreaKey : BuildKey := PackFileArea(Area, StUpcase(FileName));
        FileNameKey : BuildKey := PackFileName(StUpcase(FileName), Area);
        FileDateKey : BuildKey := Word2Key(Area)+Word2Key(FileTime.D)+Long2Key(FileTime.T);
        FileUpKey   : BuildKey := Pad(StUpcase(Uploader), 25)+Long2key(UploaderID);
      end;
  end;


  function AddWords(oldf: PFileRec; P : PChar; Len : Word; RefNr : LongInt) : Boolean;
  var
    I : Word;
    Q, T : PChar;
    S : String;

  begin
    AddWords := False;
    I := 0;
    while I < Len do
      begin
        while (I < Len) and not (P^ in StartChars) do
          begin
            Inc(P);
            Inc(I);
          end;
        if I >= Len then
          Break;
        Q := P;
        while (I < Len) and (P^ in WordChars) do
          begin
            Inc(P);
            Inc(I);
          end;
        T := P;
        while (Word(T) > Word(Q)) and not (T[$FFFF] in EndChars) do
          Dec(T);
        if Word(T) > Word(Q) then
          begin
            Move(Q^, S[1], Word(T) - Word(Q));
            S[0] := Chr(Word(T) - Word(Q));
            S := StUpcase(S);
            if (oldf = nil) or (SearchUC(oldf^, sizeof(TFileHeader)+oldf^.MsgBytes, s[1], Length(s)) = $FFFF) then
              FileDb.IndexDb.Add(S, RefNr);
          end;
      end;
    AddWords := True;
  end;


  function AddWordsStr(oldf: PFileRec; const S : String; RefNr : LongInt): Boolean;
  begin
    AddWordsStr := AddWords(oldf, @S[1], Length(S), RefNr);
  end;


  procedure AddIndexRecord(RefNr : LongInt; const Data);
  var
    FileRec : TFileRec absolute Data;
    Counter : Byte;

  begin
    AddWordsStr(nil, FileRec.FileName, RefNr);
    if FileRec.Uploader <> '' then
      AddWordsStr(nil, FileRec.Uploader, RefNr);
    if FileRec.Desc <> '' then
      AddWordsStr(nil, FileRec.Desc, RefNr);
    if LongFlagIsSet(MwConfig.mwFlags, mwIndexLongDesc) and (FileRec.MsgBytes > 0) then
      AddWords(nil, @FileRec.MsgText, FileRec.MsgBytes, RefNr);
    for Counter := 1 to 6 do
      if FileRec.KeyWords[Counter] <> '' then
        AddWordsStr(nil, FileRec.KeyWords[Counter], RefNr);
  end;


  function TFileDatabase.AddRecord(var RefNr : LongInt; var Data) : Boolean;
  var
    FileRec : TFileRec absolute Data;

  begin
    Lock;
    if inherited AddRecord(RefNr, Data) then
      begin
        AddRecord := True;
        AddIndexRecord(RefNr, Data);
      end
    else
      AddRecord := False;
    Unlock;
  end;


  procedure UpdateIndexRecord(RefNr : LongInt; var OldRec; var NewRec);
  var
    Counter : Byte;
    OldFile : TFileRec absolute OldRec;
    NewFile : TFileRec absolute NewRec;

  begin
    if (OldFile.FileName <> NewFile.FileName) then
      AddWordsStr(@OldFile, NewFile.FileName, RefNr);
    if (OldFile.Uploader <> NewFile.Uploader) and (NewFile.Uploader <> '') then
      AddWordsStr(@OldFile, NewFile.Uploader, RefNr);
    if (OldFile.Desc <> NewFile.Desc) and (NewFile.Desc <> '') then
      AddWordsStr(@OldFile, NewFile.Desc, RefNr);
    if LongFlagIsSet(MwConfig.mwFlags, mwIndexLongDesc) and (NewFile.MsgBytes > 0) then
      if (OldFile.MsgBytes <> NewFile.MsgBytes)
        or not SameStruct(OldFile.MsgText, NewFile.MsgText, OldFile.MsgBytes) then
          AddWords(@OldFile, @NewFile.MsgText, NewFile.MsgBytes, RefNr);
    for Counter := 1 to 6 do
      if (OldFile.Keywords[Counter] <> NewFile.Keywords[Counter]) and (NewFile.KeyWords[Counter] <> '') then
        AddWordsStr(@OldFile, NewFile.KeyWords[Counter], RefNr);
  end;


  procedure TFileDatabase.PutRecordPrim(RefNr : LongInt; var OldData, NewData);
  var
    OldFileRec : TFileRec absolute OldData;
    NewFileRec : TFileRec absolute NewData;

  begin
    Lock;
    UpdateIndexRecord(RefNr, OldData, NewData);
    inherited PutRecordPrim(RefNr, OldData, NewData);
    Unlock;
  end;


  procedure TFileDatabase.UpdateRecordName(Area : Word; const Name : String; var NewData);
  begin
    inherited UpdateRecordKey(PackFileArea(Area, Name), NewData);
  end;


  procedure TFileDatabase.FatalDBError(const S : String);
  begin
    LogFatalError('FILE DATABASE : ' + S, IsamError);
  end;


  procedure TFileDatabase.LogDBError(const S : String);
  begin
    LogError('FILE DATABASE : ' + S, IsamError);
  end;


  procedure InitWordChars;
  var
    C : Char;

  begin
    for C := #0 to #255 do
      if (Upcase(C) <> C) then begin
        Include(WordChars, C);
        Include(WordChars, Upcase(C));
        Include(StartChars, C);
        Include(StartChars, Upcase(C));
        Include(EndChars, C);
        Include(EndChars, Upcase(C));
      end;
  end;


begin
  InitWordChars;
end.
