unit WcDb;

interface

uses
  Crt,
  WcMisc,
  NumKeys,
  IsamTool,
  Desq,
  Filer,
  VRec;

const
  MaxLockRetries = 50;

type
  PFileBlock = ^TFileBlock;
  TFileBlock = object
    IFBPtr      : IsamFileBlockPtr;
    LockCount   : Integer;
    Retries     : Integer;
    UniqueKey   : Integer;
    MaxDataLen  : Word;

    constructor Init(FName : IsamFileBlockName; Save, Net : Boolean;
                     AUniqueKey : Integer; AMaxDataLen : Word);
    destructor Done; virtual;
    procedure GetCreateInfo(var Len : Word; var Keys : Integer; var IID : IsamIndDescr); virtual;
    procedure PostCreate; virtual;
    function LockOkay(const what : String) : Boolean;
    procedure Lock;
    procedure Unlock;
    function BuildKey(const Data; Key : Integer) : IsamKeyStr; virtual;
    procedure GetRec(RefNr : LongInt; var Data); virtual;
    procedure AddRec(var RefNr : LongInt; var Data); virtual;
    procedure PutRec(RefNr : LongInt; var Data); virtual;
    procedure DelRec(RefNr : LongInt); virtual;
    procedure AddKey(KeyNr : Integer; RefNr : LongInt; Key : IsamKeyStr);
    procedure DeleteKey(KeyNr : Integer; RefNr : LongInt; Key : IsamKeyStr);
    procedure DeleteAllKeys(KeyNr : Integer);
    function FindKey(KeyNr : Integer; var RefNr : LongInt; Key : IsamKeyStr) : Boolean;
    function RecLen : Word;
    function FileLen : LongInt;
    function FreeRecs : LongInt;
    function UsedRecs : LongInt;
    function UsedKeys(KeyNr : Integer) : LongInt;
    procedure ClearKey(KeyNr : Integer);
    function NextKey(KeyNr : Integer; var RefNr : LongInt; var Key : IsamKeyStr) : Boolean;
    procedure NextDiffKey(KeyNr : Integer; var RefNr : LongInt; var Key : IsamKeyStr);
    function PrevKey(KeyNr : Integer; var RefNr : LongInt; var Key : IsamKeyStr) : Boolean;
    procedure PrevDiffKey(KeyNr : Integer; var RefNr : LongInt; var Key : IsamKeyStr);
    procedure SearchKey(KeyNr : Integer; var RefNr : LongInt; var Key : IsamKeyStr);
    function KeyExists(KeyNr : Integer; RefNr : LongInt; Key : IsamKeyStr) : Boolean;
    function AddRecord(var RefNr : LongInt; var Data) : Boolean; virtual;
    procedure PutRecordPrim(RefNr : LongInt; var OldData, NewData); virtual;
    procedure UpdateRecord(var OldData, NewData);
    procedure UpdateRecordKey(const Key : IsamKeyStr; var NewData);
    procedure DeleteRecordPrim(RefNr : LongInt; var Data); virtual;
    function DeleteRecord(var Data) : Boolean;
    function DeleteRecordKey(const Key : IsamKeyStr) : Boolean;
    procedure SearchKeyAndRef(KeyNr : Integer; RefNr : LongInt; Key : IsamKeyStr);
    procedure FindKeyAndRef(KeyNr : Integer; RefNr : LongInt; Key : IsamKeyStr; SVal : Integer);
    procedure FatalDBError(const S : String); virtual;
    procedure LogDBError(const S : String); virtual;
  end;

  PVFileBlock = ^TVFileBlock;
  TVFileBlock = object(TFileBlock)
    function DataLen(const Data) : Word; virtual;
    procedure GetRec(RefNr : LongInt; var Data); virtual;
    procedure AddRec(var RefNr : LongInt; var Data); virtual;
    procedure PutRec(RefNr : LongInt; var Data); virtual;
    procedure DelRec(RefNr : LongInt); virtual;
    procedure GetRecPart(RefNr : LongInt; var Data; Len : Word);
    procedure GetFixedRec(RefNr : LongInt; var Data);
    procedure AddFixedRec(var RefNr : LongInt; var Data);
    procedure DelFixedRec(RefNr : LongInt);
    procedure PutFixedRec(RefNr : LongInt; var Data);
  end;

function Word2Key(Num : Word) : String;
function Long2Key(Num : LongInt) : String;
function Key2Long(Key : IsamKeyStr) : LongInt;

procedure LogFatalError(const S : String; ErrorCode : Word);
procedure LogError(const S : String; ErrorCode : Word);

implementation

uses
  WcGlobal;


  function Word2Key(Num : Word) : String;
  begin
    Word2Key := CStyleNumKey(WordToKey(Num));
  end;


  function Long2Key(Num : LongInt) : String;
  begin
    Long2Key := CStyleNumKey(LongToKey(Num));
  end;


  function Key2Long(Key : IsamKeyStr) : LongInt;
  begin
    Key2Long := KeyToLong(PascalStyleNumKey(Key));
  end;


  procedure LogFatalError(const S : String; ErrorCode : Word);
  begin
    LogFatalError(S, ErrorCode);
  end;


  procedure LogError(const S : String; ErrorCode : Word);
  begin
    NoteError(S+' :'+Long2Str(ErrorCode));
  end;


  procedure Abstract;
  begin
    RunError(211);
  end;


  constructor TFileBlock.Init(FName : IsamFileBlockName; Save, Net : Boolean;
                              AUniqueKey : Integer; AMaxDataLen : Word);
  var
    Keys : Integer;
    IID : IsamIndDescr;
    DataLen : Word;
    Created : Boolean;

  begin
    LockCount := 0;
    Retries := 0;
    UniqueKey := AUniqueKey;
    MaxDataLen := AMaxDataLen;
    if not ExistFile(FName+'.DAT') then
      begin
        GetCreateInfo(DataLen, Keys, IID);
        BtCreateFileBlock(FName, DataLen, Keys, IID);
        if not IsamOk then
          Fail;
        Created := True;
      end
    else
      Created := False;
    repeat
      BtOpenFileBlock(IFBPtr, FName, False, False, Save, Net);
    until LockOkay('open');
    if Created then
      PostCreate;
  end;


  procedure TFileBlock.GetCreateInfo(var Len : Word; var Keys : Integer; var IID : IsamIndDescr);
  begin
    Abstract;
  end;


  procedure TFileBlock.PostCreate;
  begin
  end;


  destructor TFileBlock.Done;
  begin
    repeat
      BtCloseFileBlock(IFBPtr);
    until LockOkay('close');
  end;


  function TFileBlock.LockOkay(const what : String) : Boolean;
  begin
    LockOkay := True;
    if not IsamOk then
      begin
        case BtIsamErrorClass of
          1 : {ignore};
          2 : if Retries < MaxLockRetries then
                begin
                  LockOkay := False;
                  WriteTopRight('Lock retry #'+Long2Str(Retries));
                  Inc(Retries);
                  WcDelay(500 + Random(500));
                end
              else
                FatalDBError('Unable lock database after 50 retries!');
          3,
          4 : FatalDBError('Unable to '+what+' database!');
        end;
      end
    else
      begin
        if Retries > 0 then
          WriteTopRight('              ');
        Retries := 0;
      end;
  end;


  procedure TFileBlock.Lock;
  begin
    if LockCount = 0 then
      repeat
        BtLockFileBlock(IFBPtr);
      until LockOkay('lock');
    Inc(LockCount);
  end;


  procedure TFileBlock.Unlock;
  begin
    Dec(LockCount);
    if LockCount = 0 then
      repeat
        BtUnLockFileBlock(IFBPtr);
      until LockOkay('unlock');
  end;


  function TFileBlock.BuildKey(const Data; Key : Integer) : IsamKeyStr;
  begin
    Abstract;
  end;


  procedure TFileBlock.GetRec(RefNr : LongInt; var Data);
  begin
    repeat
      BtGetRec(IFBPtr, RefNr, Data, False);
    until LockOkay('read');
  end;


  procedure TFileBlock.AddRec(var RefNr : LongInt; var Data);
  begin
    repeat
      BtAddRec(IFBPtr, RefNr, Data);
    until LockOkay('write');
  end;


  procedure TFileBlock.PutRec(RefNr : LongInt; var Data);
  begin
    repeat
      BtPutRec(IFBPtr, RefNr, Data, False);
    until LockOkay('write');
  end;


  procedure TFileBlock.DelRec(RefNr : LongInt);
  begin
    repeat
      BtDeleteRec(IFBPtr, RefNr);
    until LockOkay('write');
  end;


  procedure TFileBlock.AddKey(KeyNr : Integer; RefNr : LongInt; Key : IsamKeyStr);
  begin
    repeat
      BtAddKey(IFBPtr, KeyNr, RefNr, Key);
    until LockOkay('write');
  end;


  procedure TFileBlock.DeleteKey(KeyNr : Integer; RefNr : LongInt; Key : IsamKeyStr);
  begin
    repeat
      BtDeleteKey(IFBPtr, KeyNr, RefNr, Key);
    until LockOkay('write');
  end;


  function TFileBlock.FindKey(KeyNr : Integer; var RefNr : LongInt; Key : IsamKeyStr) : Boolean;
  begin
    repeat
      BtFindKey(IFBPtr, KeyNr, RefNr, Key);
    until LockOkay('write');
    FindKey := IsamOk;
  end;


  function TFileBlock.RecLen : Word;
  begin
    RecLen := BtDatRecordSize(IFBPtr);
  end;


  function TFileBlock.FileLen : LongInt;
  begin
    repeat
      FileLen := BtFileLen(IFBPtr);
    until LockOkay('read');
  end;


  function TFileBlock.FreeRecs : LongInt;
  begin
    repeat
      FreeRecs := BtFreeRecs(IFBPtr);
    until LockOkay('read');
  end;


  function TFileBlock.UsedRecs : LongInt;
  begin
    repeat
      UsedRecs := BtUsedRecs(IFBPtr);
    until LockOkay('read');
  end;


  function TFileBlock.UsedKeys(KeyNr : Integer) : LongInt;
  begin
    repeat
      UsedKeys := BtUsedKeys(IFBPtr, KeyNr);
    until LockOkay('read');
  end;


  procedure TFileBlock.ClearKey(KeyNr : Integer);
  begin
    repeat
      BtClearKey(IFBPtr, KeyNr);
    until LockOkay('read');
  end;


  function TFileBlock.NextKey(KeyNr : Integer; var RefNr : LongInt; var Key : IsamKeyStr) : Boolean;
  begin
    repeat
      BtNextKey(IFBPtr, KeyNr, RefNr, Key);
    until LockOkay('read');
    NextKey := IsamOk;
  end;


  procedure TFileBlock.NextDiffKey(KeyNr : Integer; var RefNr : LongInt; var Key : IsamKeyStr);
  begin
    repeat
      BtNextDiffKey(IFBPtr, KeyNr, RefNr, Key);
    until LockOkay('read');
  end;


  function TFileBlock.PrevKey(KeyNr : Integer; var RefNr : LongInt; var Key : IsamKeyStr) : Boolean;
  begin
    repeat
      BtPrevKey(IFBPtr, KeyNr, RefNr, Key);
    until LockOkay('read');
    PrevKey := IsamOk;
  end;


  procedure TFileBlock.PrevDiffKey(KeyNr : Integer; var RefNr : LongInt; var Key : IsamKeyStr);
  begin
    repeat
      BtPrevDiffKey(IFBPtr, KeyNr, RefNr, Key);
    until LockOkay('read');
  end;


  procedure TFileBlock.SearchKey(KeyNr : Integer; var RefNr : LongInt; var Key : IsamKeyStr);
  begin
    repeat
      BtSearchKey(IFBPtr, KeyNr, RefNr, Key);
    until LockOkay('read');
  end;


  function TFileBlock.KeyExists(KeyNr : Integer; RefNr : LongInt; Key : IsamKeyStr) : Boolean;
  begin
    repeat
      KeyExists := BtKeyExists(IFBPtr, KeyNr, RefNr, Key);
    until LockOkay('read');
  end;


  procedure TFileBlock.SearchKeyAndRef(KeyNr : Integer; RefNr : LongInt; Key : IsamKeyStr);
  begin
    repeat
      BTSearchKeyAndRef(IFBPtr, KeyNr, RefNr, Key);
    until LockOkay('read');
  end;


  procedure TFileBlock.FindKeyAndRef(KeyNr : Integer; RefNr : LongInt; Key : IsamKeyStr; SVal : Integer);
  begin
    repeat
      BTFindKeyAndRef(IFBPtr, KeyNr, RefNr, Key, SVal);
    until LockOkay('read');
  end;


  function TFileBlock.AddRecord(var RefNr : LongInt; var Data) : Boolean;
  var
    I : Byte;
    Key : IsamKeyStr;

  begin
    Lock;
    if FindKey(UniqueKey, RefNr, BuildKey(Data, UniqueKey)) then
      begin
        Unlock;
        AddRecord := False;
        Exit;
      end;
    AddRec(RefNr, Data);
    if not IsamOk then
      FatalDBError('Unable to add record to database!');
    for I := 1 to IFBPtr^.NrOfKeys do
      begin
        Key := BuildKey(data, I);
        if Key <> '' then
          AddKey(I, RefNr, Key);
        if not IsamOk then
          FatalDBError('Unable to add key to index!');
      end;
    Unlock;
    AddRecord := True;
  end;


  procedure TFileBlock.PutRecordPrim(RefNr : LongInt; var OldData, NewData);
  var
    I : Integer;
    Key : IsamKeyStr;

  begin
    Lock;
    for I := 1 to IFBPtr^.NrOfKeys do
      begin
        Key := BuildKey(OldData, I);
        if (Key <> '') and (Key <> BuildKey(NewData, I)) then
          begin
            DeleteKey(I, RefNr, Key);
            if not IsamOk then
              FatalDBError('Unable to delete key!');
          end;
      end;
    PutRec(RefNr, NewData);
    if not IsamOk then
      FatalDBError('Unable to delete record!');
    for I := 1 to IFBPtr^.NrOfKeys do
      begin
        Key := BuildKey(NewData, I);
        if (Key <> '') and (Key <> BuildKey(OldData, I)) then
          begin
            AddKey(I, RefNr, Key);
            if not IsamOk then
              FatalDBError('Unable to add key to index!');
          end;
      end;
    Unlock;
  end;


  procedure TFileBlock.UpdateRecord(var OldData, NewData);
  var
    RefNr : LongInt;

  begin
    Lock;
    if FindKey(UniqueKey, RefNr, BuildKey(OldData, UniqueKey)) then
      GetRec(RefNr, OldData);
    if IsamOk then
      PutRecordPrim(RefNr, OldData, NewData);
    Unlock;
  end;


  procedure TFileBlock.UpdateRecordKey(const Key : IsamKeyStr; var NewData);
  var
    RefNr : LongInt;
    OldData : Pointer;

  begin
    if not GetMemCheck(OldData, MaxDataLen) then
      Exit;
    Lock;
    if FindKey(UniqueKey, RefNr, Key) then
      GetRec(RefNr, OldData^);
    if IsamOk then
      PutRecordPrim(RefNr, OldData^, NewData);
    Unlock;
    FreeMemCheck(OldData, MaxDataLen);
  end;


  procedure TFileBlock.DeleteRecordPrim(RefNr : LongInt; var Data);
  var
    Key : IsamKeyStr;
    I : Integer;

  begin
    Lock;
    for I := 1 to IFBPtr^.NrOfKeys do
      begin
        Key := BuildKey(Data, I);
        if (Key <> '') then
          DeleteKey(I, RefNr, Key);
        if not IsamOk then
          FatalDBError('Unable to delete key from index!');
      end;
    DelRec(RefNr);
    if not IsamOk then
      FatalDBError('Unable to delete record from database!');
    Unlock;
  end;


  function TFileBlock.DeleteRecord(var Data) : Boolean;
  var
    RefNr : LongInt;

  begin
    Lock;
    if FindKey(UniqueKey, RefNr, BuildKey(Data, UniqueKey)) then
      begin
        DeleteRecord := True;
        GetRec(RefNr, Data);
        DeleteRecordPrim(RefNr, Data);
      end
    else
      DeleteRecord := False;
    Unlock;
  end;


  function TFileBlock.DeleteRecordKey(const Key : IsamKeyStr) : Boolean;
  var
    Data : Pointer;
    RefNr : LongInt;

  begin
    DeleteRecordKey := False;
    if not GetMemCheck(Data, MaxDataLen) then
      Exit;
    Lock;
    if FindKey(UniqueKey, RefNr, Key) then
      begin
        DeleteRecordKey := True;
        GetRec(RefNr, Data^);
        DeleteRecordPrim(RefNr, Data^);
      end;
    Unlock;
    FreeMemCheck(Data, MaxDataLen);
  end;


  procedure TFileBlock.DeleteAllKeys(KeyNr : Integer);
  begin
    Lock;
    BTDeleteAllKeys(IFBPtr, KeyNr);
    Unlock;
  end;


  procedure TFileBlock.FatalDBError(const S : String);
  begin
    LogFatalError(S + IsamErrorMessage(IsamError), IsamError);
  end;


  procedure TFileBlock.LogDBError(const S : String);
  begin
    LogError(S + IsamErrorMessage(IsamError), IsamError);
  end;


  function TVFileBlock.DataLen(const Data) : Word;
  begin
    Abstract;
  end;


  procedure TVFileBlock.GetRec(RefNr : LongInt; var Data);
  var
    Len : Word;

  begin
    repeat
      BtGetVariableRec(IFBPtr, RefNr, Data, Len);
    until LockOkay('read');
  end;


  procedure TVFileBlock.AddRec(var RefNr : LongInt; var Data);
  begin
    repeat
      BtAddVariableRec(IFBPtr, RefNr, Data, DataLen(Data));
    until LockOkay('write');
  end;


  procedure TVFileBlock.PutRec(RefNr : LongInt; var Data);
  begin
    repeat
      BtPutVariableRec(IFBPtr, RefNr, Data, DataLen(Data));
    until LockOkay('write');
  end;


  procedure TVFileBlock.DelRec(RefNr : LongInt);
  begin
    repeat
      BtDeleteVariableRec(IFBPtr, RefNr);
    until LockOkay('write');
  end;


  procedure TVFileBlock.GetRecPart(RefNr : LongInt; var Data; Len : Word);
  begin
    repeat
      BtGetVariableRecPart(IFBPtr, RefNr, Data, Len);
    until LockOkay('read');
  end;


  procedure TVFileBlock.GetFixedRec(RefNr : LongInt; var Data);
  begin
    TFileBlock.GetRec(RefNr, Data);
  end;


  procedure TVFileBlock.AddFixedRec(var RefNr : LongInt; var Data);
  begin
    TFileBlock.AddRec(RefNr, Data);
  end;


  procedure TVFileBlock.DelFixedRec(RefNr : LongInt);
  begin
    TFileBlock.DelRec(RefNr);
  end;


  procedure TVFileBlock.PutFixedRec(RefNr : LongInt; var Data);
  begin
    TFileBlock.PutRec(RefNr, Data);
  end;


end.
