/************************************************************************/
/*			SAMPLE2.C						05-01-94					*/
/*											ver 1.00					*/
/* The following sample cource code for C compilers for DOS shows the	*/
/* simple case of two axes moving together (Linear Interpolation)  with */
/* a common vector velocity and vector acceleration, to a given target 	*/
/* position. For simplicity, it is assumed that the PID values (Servo)	*/
/* or Steps/Lines (Stepper - CL) are already set for this Axis.			*/
/* This source code also shows how to properly break a vector velocity 	*/
/* into its 'x' and 'y' components.										*/
/************************************************************************/

#include <stdio.h>
#include <conio.h>
#include <math.h> 		/*additional header file needed for math functions */
#include <nulib02.h>

#define PROF_COMPL		0x400			/*profile complete bit in 'per Axis' hardware status */
#define MOTOR_OFF		0x04			/*motor off bit */
#define MOTOR_RUNNING	0x80           	/*motor running bit */

int WaitForMoveComplete(WORD BoardAddr, WORD BoardType, BYTE Axis, WORD TimeOutSecs);
int CalcVectComponents(double Vel, double Accel,  /*desired vector velocity and acceleration */
					   long int PresPosX, long int PresPosY,/*present known positions */
					   long int TarPosX, long int TarPosY,	/*target positions */
					   double *VelX, double *VelY, 			/*component vel vals returned */
					   double *AccelX, double *AccelY); 	/*component accel vals returned */


void main(void) {
  int retstat;
  WORD BoardAddr, BoardType;
  WORD StepsPerRev;
  WORD TimeOutSecs;
  double Vel, Accel;
  BYTE xAxis, yAxis;
  long int PresPosX, PresPosY, TarPosX, TarPosY;
  double VelX, VelY, AccelX, AccelY;

/* * * * * * * * * * * * *SETUP PARAMETERS* * * * * * * * * * * * * * * * * * * * * */
	/*Set up parameters for this board and motors*/
  	BoardAddr = 0x398;		/*Board address is set with DIP switches on the board */
  	BoardType = 1;			/*Board Type 1 means an open loop stepper */
  	StepsPerRev = 1000;     /*Steps Per Revolution (stepper) or Lines/Rev (servo) on your motor */

  	/*consider this to be your target position, vector velocity, and vector acceleration */
  	xAxis = 1;				/*Controlling Axis 1	*/
  	yAxis = 2;				/* and...	  Axis 2	*/
  	Vel = 500; /*500 RPM for vector, the X and Y components will be less or the same */
  	Accel = 200; /*200 RPS/s for vector, the X and Y components will be less */
  	TarPosX = 12000;	/*arbitrary targets	*/
  	TarPosY = 17000;	/*	"			"	*/
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	/*First, you want to read your present positions*/
  	retstat = read_pos(BoardAddr, BoardType, xAxis, &PresPosX);
  	if (retstat) printf("Read Pos Error!");
	printf("X Position is now = %d\n",PresPosX);

  	retstat = read_pos(BoardAddr, BoardType, xAxis, &PresPosY);
  	if (retstat) printf("Read Pos Error!");
	printf("Y Position is now = %d\n",PresPosY);

	/*Now, you've got all the info necessary to calulate the X and Y components for your vector*/
	retstat = CalcVectComponents(Vel, Accel, PresPosX, PresPosY, TarPosX, TarPosY,
													&VelX, &VelY, &AccelX, &AccelY);

    /*Now load vels and accels for both axes */
  	retstat =   load_rpm(BoardAddr, BoardType, xAxis, StepsPerRev, VelX); /*load velocity (RPM)*/
  	if (retstat) printf("  Load RPM Error!");

  	retstat =   load_rpm(BoardAddr, BoardType, yAxis, StepsPerRev, VelY);
  	if (retstat) printf("  Load RPM Error!");

  	retstat = load_rpsps(BoardAddr, BoardType, xAxis, StepsPerRev, AccelX);/*load accel (rpsps)*/
  	if (retstat) printf("Load RPSPS Error!");

  	retstat = load_rpsps(BoardAddr, BoardType, yAxis, StepsPerRev, AccelY);
  	if (retstat) printf("Load RPSPS Error!");

    /*load target positions for both X and Y axes */
    retstat = load_target_pos(BoardAddr, BoardType, xAxis, TarPosX);/*load target pos (counts)*/
  	if (retstat) printf("Load Target Pos Error!");

    retstat = load_target_pos(BoardAddr, BoardType, yAxis, TarPosY);
  	if (retstat) printf("Load Target Pos Error!");

    /*start motion on both axes 1 and 2 */
    /*We chose axes 1 and 2 for this demo, but they could be any combination */
  	retstat = multi_start(BoardAddr, BoardType, 1);
  	if (retstat) printf("Multi Start Error!");

	/*Wait for move complete on one axis. Once that routine returns, the other axis move 	*/
	/* should also be complete, since both axes are now synchronized. Since the second axis */
	/* should finish at the same time, the timeout does not need to be high. */
    TimeOutSecs = 15; /*wait 15 seconds before giving up and returning a timeout error */
    retstat = WaitForMoveComplete(BoardAddr, BoardType, xAxis, TimeOutSecs);
  	if (retstat) printf("Timeout Error!");

    TimeOutSecs = 2; /*wait 2 seconds before giving up and returning a timeout error */
    retstat = WaitForMoveComplete(BoardAddr, BoardType, yAxis, TimeOutSecs);
  	if (retstat) printf("Timeout Error!");

    printf("Multi-Axis Vector Trajectory is now complete!\n"); /*yeah!*/

  	/* Now display your new positions */
  	retstat = read_pos(BoardAddr, BoardType, xAxis, &PresPosX);
  	if (retstat) printf("Read Pos Error!");
	printf("X Position is now = %d\n",PresPosX);

  	retstat = read_pos(BoardAddr, BoardType, yAxis, &PresPosY);
  	if (retstat) printf("Read Pos Error!");
	printf("X Position is now = %d\n",PresPosY);

}

/*****************************************************************************************/
/* Calculate Vector Components -- This routine takes a two-dimensional target, and vector*/
/* velocities, accelerations, and calculates the X and Y components of the vector to give*/
/* you the per axis parameters for a linearly interpolated move. This routine is very	 */
/* useful for a cartesian robot, or XY Table.						 */
/*****************************************************************************************/
/* Vel - Desired Vector Velocity, supplied by the caller								 */
/* Accel - Desired Vector Accel, supplied by the caller									 */
/* PresPosX, PresPosY - Starting Positions, supplied by the caller						 */
/* TarPosX, TarPosY - Desired Target Positions, supplied by the caller					 */
/* VelX, VelY - Vector Velocity X and Y Components, returned by the routine				 */
/* AccelX, AccelY - Vector Accel X and Y Components, returned by the routine			 */
/* always returns a 0																	 */
/*****************************************************************************************/
int CalcVectComponents(double Vel, double Accel, /*desired vector velocity and acceleration */
					   long int PresPosX, long int PresPosY, /*present known positions */
					   long int TarPosX, long int TarPosY,	  /*target positions */
					   double *VelX, double *VelY, /*component vel vals returned */
					   double *AccelX, double *AccelY) /*component accel vals returned */
{
	double rads;
	long int dX, dY; /*difference between present X and target X, present Y and target Y*/

	dX = TarPosX - PresPosX;	/* Target X - present X*/
	dY = TarPosY - PresPosY;	/* Target Y - present Y*/

	if (dX==0) { 			/*vertical or no movement*/
    if (dY==0) {
    	*VelX = 0.000;		/*special case of no movement*/
    	*VelY = 0.000;
    	*AccelX = 0.000;
    	*AccelY = 0.000;
    } else {
    	*VelX = 0.000;      /*special case of vertical movement*/
    	*VelY = Vel;
    	*AccelX = 0.000;
    	*AccelY = Accel;
    }
    return(0);
  }

  /* Only the absolute values of velocity and acceleration are needed for these trajectories */
  rads = atan((double)dY / (double)dX );/*vector angle is in radians*/
  *VelX = fabs(Vel* cos(rads)); 			/*absolute value*/
  *VelY = fabs(Vel* sin(rads)); 			/*	"	 	"	*/
  *AccelX = fabs(Accel * cos(rads)); 		/*  "	    " 	*/
  *AccelY = fabs(Accel * sin(rads));		/*  " 	    " 	*/
  return(0);

}




/************************************************************************************/
/* Wait For Move Complete -- This function waits for an Axis to complete its move	*/
/*  and returns a 0 status. If the number of timeout seconds specified is greater 	*/
/*  than 10000, this routine will return an error (-1) . 							*/
/* If you give a timeout of 0, the routine will only return when the move completes.*/
/* In other words a timeout value of 0 is effectively an infinite timeout.			*/
/* If the timeout actually happens, we return a (-2);								*/
/* Other error numbers can be returned from the library call, see your library doc- */
/* ument for more detail.															*/
/************************************************************************************/
/* BoardAddr - The ISA bus board address											*/
/* BoardType - The nuLogic board type (1-stepper ol, 2-stepper cl, 3-servo4a, etc.	*/
/* TimeOutSecs - Number of seconds before routine quits looking for move complete	*/
/* returns a 0 if successful, -1 if timeout parm too large, -2 if timeout occurred	*/
/************************************************************************************/
int WaitForMoveComplete(WORD BoardAddr, WORD BoardType, BYTE Axis, WORD TimeOutSecs)
{
  int retstat;
  WORD AxisStat;
  DWORD TimeOutCycles, i;

  	if (TimeOutSecs>10000) return(-1); /*too big a number */

  	TimeOutCycles = 240 * TimeOutSecs; /*this is a rough estimate of cycles per second */
  	/*approx 240 read_axis_stat's per sec, can vary by how many axes are moving, etc. */
    /*since this is just an error condition limit, it doesn't need to be very accurate*/

    AxisStat = 0; /*if timeoutsecs = 0, then timeout is infinite */
    for (i=0;((i<TimeOutCycles)||(TimeOutSecs==0));i++)  {

    	retstat = read_axis_stat(BoardAddr, BoardType, Axis, &AxisStat);
    	if (retstat) return(retstat); /*error checking */

		/* check if motor is not running ,and either profile is complete or motor is off*/
    	if ( (AxisStat&(PROF_COMPL|MOTOR_OFF)) && (!(AxisStat&MOTOR_RUNNING)) )
    	  return(0);
    }

    return(-2); /*timeout*/
}
