/********************************************************************
 				COMM.C							05-01-94
	C Driver for 3 and 4 axis boards			rev1.02
	written by John Pelletier and Jon Leehey
 ********************************************************************
 This is the low level driver code written in C language, which formats
 the message packets for you, in order to talk correctly to nuLogic's
 boards.

	The following are error codes that this driver code will return
	should something go wrong. You should use these definitions as
	error messages to the end user.

        status 0: Successful, no errors"
        status 1: Communication error, timeout on ready to receive"
        status 2: Command error on current packet"
        status 3: Communication error, no data available in the return data buffer"
        status 4: Communication error, return packet is not complete"
        status 5: Board failure"
        status 6: Application error, bad axis number passed to driver"
        status 7: Communication error, command-in-process bit stuck on"
        status 8: Command error occured prior to current packet"
        status 9: Unable to clear command error bit"
        status 10: Applications error, bad command ID passed to driver"
        status 11: Communication error, returned packet ID is corrupted"
        status 12: Application error, bad board ID or board ID not configured"
        status 13: Application error, bad word count passed to 'send_command' "
        status 15: Communication error, unable to flush the return data buffer"
        status 21: Communication error, system reset did not occur within allotted time"
        status 25: Communication error, return data buffer is not empty"
*/
#include <conio.h>
#include <comm.h>

/***************************************************
 READ THE COMMUNICATIONS STATUS REGISTER
 ***************************************************
  generic command to read the comm status register
 ***************************************************/
int read_csr(WORD boardaddr, BYTE Is4AxBd, WORD FAR *csrptr)
{
int retstat;

/*get the CSR from I/O port*/
*csrptr = inpw(boardaddr + 6);

return(0);
}

/**************************************************
	SEND COMMAND
***************************************************
This routine is used to send a command to the board. A packet
is created based on the command code (cmd), axis (axis), and
command data size (wcount)

A packet is sent followed by a short wait for command processing
prior to command error checking.

A zero status is returned if no errors have occurred.
***************************************************/
int send_command(WORD boardaddr, BYTE Is4AxBd, BYTE axis, BYTE cmd, BYTE wcount, DWORD cmddata)
{
int retstat;
WORD packet[4];

/*range check on command*/
if (cmd < 0x41) return(10); /*status 10: bad command number*/

/*range check on word count*/
if ((wcount < 2) || (wcount > 4)) return(13); /*status 13: bad word count*/

/*limit check on the axis number*/
if (Is4AxBd) {
	if ((axis < 1) || (axis > 4)) return(6); /*status 6: bad axis number*/
}
else {
	/*range check on axis*/
	if ((axis < 1) || (axis > 3)) return(6); /*status 6: bad axis number*/
}

/*build the command packet and send it to the board*/
if (wcount == 2) {
	packet[0] = (WORD) (((axis | 0x20) << 8) | cmd);  /*command word*/
	packet[1] = TERMINATOR;
	retstat = sendpacket(boardaddr,Is4AxBd,packet,2);
	}
else if (wcount == 3) {
	packet[0] = (WORD) (((axis | 0x30) << 8) | cmd);  /*command word*/
	packet[1] = (WORD) cmddata;
	packet[2] = TERMINATOR;
	retstat = sendpacket(boardaddr,Is4AxBd,packet,3);
	}
else {
	packet[0] = (WORD) (((axis | 0x40) << 8) | cmd);  /*command word*/
	packet[1] = (WORD) ((cmddata >> 16) & 0x0000FFFF); /*data high word*/
	packet[2] = (WORD) cmddata;		/*data low word*/
	packet[3] = TERMINATOR;
	retstat = sendpacket(boardaddr,Is4AxBd,packet,4);
    }

return(retstat);
}

/****************************************************************************
 							SEND PACKET
 ****************************************************************************
 This routine is used to send a command packet to the board.
 A packet is sent followed by a short wait for command processing prior to
 command error checking.
 A zero status is returned if no errors have occurred.
 ****************************************************************************/
WORD sendpacket(WORD boardaddr, BYTE Is4AxBd, WORD *packetptr, BYTE packetsize)
{
BYTE i;
WORD retstat, swappedword;

/*read status port and check for an existing board level error*/
retstat = checkboarderr(boardaddr, Is4AxBd);
if (retstat) return(retstat);

/*send the packet: check RTR and CE before each word is sent*/
for (i = 0; i < packetsize; i++) {
	retstat = pollrtrandce(boardaddr, Is4AxBd);	/*poll RTR and check cmd err*/
	if (retstat) return(retstat);
    swappedword = byteswap(*(packetptr + i));/*for PC, each word needs to be byteswapped*/
    outpw(boardaddr, swappedword);
	}

/*wait for command processing and check for command error*/
retstat = chkcmdprocess(boardaddr, Is4AxBd);
if (retstat) return(retstat);
else return(0);    /*status 0: successful transmission of packet*/
}

/****************************************************************************
 						READ THE RETURN DATA BUFFER
 ****************************************************************************
 Generic command to read the return data buffer.
 Intended to be called by the application at some asynchronous time after
 a 'READ' command has been issued which in turn has left data in the
 return data buffer.
 Using pointers to axis, command, and rdb, three arrays will be filled with
 sequentially corresponding data for the number of packets specified.
 If fewer than the number of packets requested can be read, then the number
 of packets read will be indicated in the location pointed to by numptr.
 ****************************************************************************/
int read_rdb(WORD boardaddr, BYTE Is4AxBd, WORD FAR *numptr, WORD FAR *axisptr,
						  WORD FAR *cmdptr, DWORD FAR *rdbptr)
{
WORD packetID,dataloword,datahiword,trash;
BYTE wordcount;
int retstat,i;
WORD j, boardtype;
DWORD datahidword;

/*read status port and check for an existing board level error*/
retstat = checkpreviouserr(boardaddr,Is4AxBd);
if (retstat) return(retstat);

/*check for data pending*/
if (!datapending(boardaddr,Is4AxBd)) return(3);  /*status 3: no data pending*/

for (j = 0; j < *numptr; j++) {
	/*read the first word: the packet identifier*/
	packetID = byteswap(inpw(boardaddr));

	/*check the word count for validity*/
	wordcount = (BYTE) (packetID >> 12);  /*isolate the data word count*/
	if ((wordcount != 2) && (wordcount != 3)) {
		/*try to flush data buffer if packetID is bad*/
		for (i = 0; i < 10000; i++) {
			if (!datapending(boardaddr,Is4AxBd)) break;
			trash = inpw(boardaddr);
    		}
		if (i == 10000) return(15); /*status 15: unable to flush RDB*/
		else return(11);     /*status 11: corrupted return packet ID*/
		}
	/*set the return axis and command*/
	*(axisptr + j) = (WORD) ((packetID & 0x0F00) >> 8);
	*(cmdptr + j) = (WORD) (packetID & 0x00ff);

	/*read the second word*/
	if (!datapending(boardaddr,Is4AxBd)) return(4); /*status 4: incomplete packet*/
	dataloword = byteswap(inpw(boardaddr));

	/*read the third word, if needed, and build the return data double word*/
	if (wordcount == 3) {
		if (!datapending(boardaddr,Is4AxBd)) return(4); /*status 4: incomplete packet*/
    	datahiword = byteswap(inpw(boardaddr));
    	datahidword = (DWORD) datahiword;
    	*(rdbptr + j) = (DWORD) ((datahidword << 16) | dataloword);
    	}
	else *(rdbptr + j) = (DWORD) dataloword;   /*no high word if only one data*/
	/*check to see if more data is available before continuing*/
	if (!datapending(boardaddr,Is4AxBd)) {
		*numptr = j + 1;
		return(0);
		}
} /*end for (numpackets)*/

return(0);      /*status 0: successful reading of data buffer*/
}



/***************************************************************************
 							POLL RTR AND CE
 ****************************************************************************
 This routine is called from sendpacket().
 The 3 axis and 4 axis versions differ by the command error bit mask,
 the call to clearcmderr() and the need for pulseholdoff().
 ****************************************************************************/
WORD pollrtrandce(WORD boardaddr, BYTE Is4AxBd)
{
DWORD i;
WORD csr, retstat;

for (i = 0; i < 40000; i++) {
	csr = inpw(boardaddr + 6);	/*get contents of csr

	/*check the command error bit
	Is4AxBd is for the 4 axis version*/
	if ((Is4AxBd && (csr & 0x0010)) || (!Is4AxBd && (csr & 0x0004))) {
		retstat = clearcmderr(boardaddr, Is4AxBd);
		if (retstat) return (retstat);
		else return(2);			/*status 2: command error on current packet*/
	    }

	/*check ready to receive*/
	if (csr & 0x0001) {
        if (!Is4AxBd) pulseholdoff(boardaddr);
        return(0);      	/*0: no command error, ready to receive*/
        }

	} /*end for*/
return(1);  			/*status 1: RTR timeout*/
}

/****************************************************************************
 							POLL RTS AND CE
 ****************************************************************************
 The 3 axis and 4 axis versions differ by the command error bit mask,
 the call to clearcmderr() and the need for pulseholdoff().
 ****************************************************************************/
WORD pollrtsandce(WORD boardaddr, BYTE Is4AxBd)
{
DWORD i;
WORD csr, retstat;

for (i = 0; i < 40000; i++) {
	csr = inpw(boardaddr + 6);	/*get contents of csr*/

	/*check command error bit*/
	/*Is4AxBd is for the 4 axis version*/
	if ((Is4AxBd && (csr & 0x0010)) || (!Is4AxBd && (csr & 0x0004))) {
		retstat = clearcmderr(boardaddr, Is4AxBd);
		if (retstat) return (retstat);
		else return(2);			/*status 2: command error on current packet*/
	    }

	/*check ready to send (data pending)*/
	if (csr & 0x0002) {
		if (!Is4AxBd) pulseholdoff(boardaddr);
		return(0);      	/*0: no command error and data pending*/
        }

	} /*end for*/
return(3);  			/*status 3: RTS timeout*/
}

/* ****************************************************************************
/* 							READY TO RECEIVE
/* ****************************************************************************
/* Returns 1 if board is ready to receive, else returns 0.
/* ****************************************************************************/
WORD readytoreceive(WORD boardaddr, BYTE Is4AxBd)
{
DWORD i;
WORD csr;

for (i = 0; i < 40000; i++) {
	csr = inpw(boardaddr + 6);	/*get contents of csr*/
	if (csr & 0x0001) {
        if (!Is4AxBd) pulseholdoff(boardaddr);
        return(1);		/*1: ready to receive*/
        }
	} /*end for*/
return(0);				/*not ready to receive*/
}

/* ****************************************************************************
/* 							DATAPENDING
/* ****************************************************************************
/* Returns 1 if data is pending, returns 0 otherwise.
/* Called prior to any read of the data buffer.
/* ****************************************************************************/
WORD datapending(WORD boardaddr, BYTE Is4AxBd)
{
DWORD i;
WORD csr;

for (i = 0; i < 40000; i++) {
	csr = inpw(boardaddr + 6);	/*get contents of csr*/
	if (csr & 0x0002) {
        if (!Is4AxBd) pulseholdoff(boardaddr);
        return(1);      /*1: data pending*/
        }
	} /*end for*/
return(0);				/*no data pending*/
}

/* ****************************************************************************
/* 							CHKCMDPROCESS
/* ****************************************************************************
/* Waits for the command to finish by polling the Command-In-Process (CIP) bit
/* (Is4AxBd) or waiting 250 microsec in a loop (3 axis driver) and then
/* checks the command error bit.
/* ****************************************************************************/
WORD chkcmdprocess(WORD boardaddr, BYTE Is4AxBd)
{
DWORD i;
WORD csr, retstat;

/*wait for last word to be taken and check cmd err*/
retstat = pollrtrandce(boardaddr, Is4AxBd);
if (retstat) return(retstat);

/*wait for the command to finish*/
if (Is4AxBd) {			/*4 axis version*/
	for (i = 0; i < 40000; i++) {
		csr = inpw(boardaddr + 6);	/*get contents of csr*/
		if (!(csr & 0x0008)) break;
	}
	if (i == 40000) return(7);	/*status 7: CIP bit stuck on*/
}
else {						/*3 axis version*/
	for (i = 0; i < 80; i++) {		/*250 loops - 400 usec on Mac IIfx*/
		csr = inpw(boardaddr + 6);	/*80 loops - 400 usec on 486/33*/
	}
}

/*check the command error bit*/
if ((Is4AxBd && (csr & 0x0010)) || (!Is4AxBd && (csr & 0x0004))) {
	retstat = clearcmderr(boardaddr, Is4AxBd);
	if (retstat) return (retstat);
	else return(2);		/*status 2: command error on current packet*/
    }

return(0);   /*0: no errors*/
}

/* ****************************************************************************
/* 							CHECKBOARDERR
/* ****************************************************************************
/* This routine is called from sendpacket().
/* The command error and system failure bits are checked, and the
/* appropriate status is returned.
/* The 3 axis and 4 axis versions differ by the error bit masks and
/* the call to clearcmderr().
/* ****************************************************************************/
WORD checkboarderr(WORD boardaddr, BYTE Is4AxBd)
{
WORD csr, retstat;

csr = inpw(boardaddr + 6);	/*get contents of csr*/

if (Is4AxBd) {		/*4 axis version*/
	if (csr & 0x0040) return(5);	/*if system failure, return status 5*/
	if (csr & 0x0010) {				/*check command error bit*/
		retstat = clearcmderr(boardaddr, Is4AxBd);	/*try to clear error if needed*/
		if (retstat) return(retstat);	/*if unable to clear, return*/
		else return(8); /*status 8: command error occurred in between packets*/
		}
	}
else {					/*3 axis version*/
	if (csr & 0x0010) return(5);	/*if system failure, return status 5*/
	if (csr & 0x0004) {				/*check command error bit*/
		retstat = clearcmderr(boardaddr, Is4AxBd);	/*try to clear error if needed*/
		if (retstat) return(retstat);	/*if unable to clear, return*/
		else return(8); /*status 8: command error occurred in between packets*/
		}
	}

/*return 0 if all is OK*/
return(0);
}

/****************************************************************************
 							CLEARCMDERR
 ****************************************************************************
 Attempts to clear an existing command error and returns status.
 The 3 axis and 4 axis versions differ by the command error bit mask.
 ****************************************************************************/
WORD clearcmderr(WORD boardaddr, BYTE Is4AxBd)
{
WORD csr, gotrtr,swapterm;
DWORD i,j;

/*since the maximum packet size is 4 words*/
/*sending 5 terminators should be more than sufficient*/
for (i = 0; i < 5; i++) {
	if (!readytoreceive(boardaddr, Is4AxBd)) return (9);
		/*status 9: unable to clear command error*/
    swapterm = SWAPTERM;  /* send SWAPTERM value to output*/
	outpw(boardaddr, swapterm);     /*send swapped terminator word*/

	/*check RTR -- to make sure terminator was taken*/
	gotrtr = 0;
	for (j = 0; j < 40000; j++) {
		csr = inpw(boardaddr + 6);	/*get contents of csr*/
		if (csr & 0x0001) break;
		} /*end for*/
	if (j == 40000) return(9);   /*status 9: unable to clear command error*/

    /*check command error*/
    if (Is4AxBd && !(csr & 0x0010)) return(0);		/*4 axis version*/
	if (!Is4AxBd && !(csr & 0x0004)) return(0);		/*3 axis version*/
    } /*end for*/
return(9);   /*unable to clear command error*/
}

/* ****************************************************************************
/* 							PULSEHOLDOFF
/* ****************************************************************************
/* Only used with the 3 axis board driver.
/* Short wait to allow RTR and RTS pulses to complete.
/* Tested delay: 5 loops - approx 10 microsec on Mac IIfx,
/* 3 loops - approx 25 usec on 486/33
/* which is about twice what we may actually need
/* ****************************************************************************/
void pulseholdoff(WORD boardaddr)
{
WORD	i, csr;

for (i = 0; i < 3; i++) {
	csr = inpw(boardaddr + 6);	/*get contents of csr*/
	}
}


/* ****************************************************************************
/* 							CHECKPREVIOUSERR
/* ****************************************************************************
/* This routine is called before sending a command packet, or before
/* processing any application issued call. This error checking is provided
/* just in case a board level error has occurred since the last completion
/* of a command.
/* The command error and system failure bits are checked, and the
/* appropriate status is returned.
/* ****************************************************************************/
int checkpreviouserr(WORD boardaddr, BYTE Is4AxBd)
{
int retstat;

/*read status port and check for an existing board level error*/
retstat = checkboarderr(boardaddr,Is4AxBd);
switch (retstat) {
	case 0:
		return(0);
	case 2:
		retstat = clearcmderr(boardaddr,Is4AxBd); /*try to clear error if needed*/
		if (retstat)
			return(retstat); /*if unable to clear, return*/
		else return(8); /*status 8: command error occurred in between packets*/
	case 5:
		return(5);  /*system failure bit was set*/
	}
}


/* ****************************************************************************
/* 							BYTESWAP
/* ****************************************************************************/
WORD byteswap(WORD wordtoswap)
{
WORD swappedword;

/*all words sent to the board need to be byte swapped*/
swappedword = (wordtoswap << 8 | wordtoswap >> 8);

return(swappedword);
}


