'Copyright 1993 nuLogic, Inc., Needham, MA
'All rights reserved.
'
'******************************README*************************************
'This source code is written in MS-DOS QBASIC.
'Its purpose is to provide routines that support the
'development of BASIC code that interfaces to nuLogic's
'motion control boards.
'
'Three essential routines are provided:
'       1) SENDCOMMAND
'       -takes a hexadecimal board address, axis #, hex command ID,
'       -packet word count, and command data as input
'       -then builds a packet and sends it to the board
'       2) READRDB
'       -takes a hex board address, number of packets, and three arrays
'       -consisting of axis, command ID, and data
'       -the return data buffer is read for the number of packets
'       -or until the RDB is empty, the actual number of packets
'       -read is returned in the "numpacks" variable,
'       -for eack packet read, values are stuffed in the axis,
'       -command, and data arrays
'       3) READCSR
'       -takes a hex board address and integer variable as input
'       -the current contents of the communications status register
'       -for the given board is returned in the integer
'
'Important note: INITASM needs to be called before attempting
'to use the above routines. INITASM sets up the machine language arrays
'that support 16 bit I/O transfers.
'
'The following error statuses may be returned by the above functions:
'       status 0: successful, no errors
'       status 1: communications error, ready to receive timeout
'       status 2: command error on current packet
'       status 3: no data available in the return data buffer
'       status 4: communications error, incomplete returned data packet
'       status 5: board failure
'       status 8: command error occurred prior to current packet
'       status 9: command error, unable to clear command error bit
'       status 11: communications error, returned packet ID is corrupted
'       status 15: communications error, unable to flush the data buffer
'
'The other routines in this source code are required to support the
'above functions, and are not intended to be called directly.
'
'The following code provides a relatively simple example of how
'to use SENDCOMMAND, READRDB and READCSR to program a simple move
'and read data back from the board.
'
'       'set up some variable for readrdb function
'       DIM axis(0 TO 10) AS INTEGER
'       DIM cmd(0 TO 10) AS INTEGER
'       DIM rdbdata(0 TO 10) AS LONG
'
'       'initialize the machine code arrays
'       initasm
'
'       'initialize board address and axis number
'       boardaddr% = &H248
'       axis% = 1
'
'       'reset position counter to zero
'       retstat% = sendcommand(boardaddr%, axis%, &H50, 2, 0)
'
'       'load target
'       retstat% = sendcommand(boardaddr%, axis%, &H4E, 4, 2000)
'
'       'load velocity
'       retstat% = sendcommand(boardaddr%, axis%, &H4F, 4, 1000)
'
'       'load accel
'       retstat% = sendcommand(boardaddr%, axis%, &H62, 4, 1000)
'
'       'start motion
'       retstat% = sendcommand(boardaddr%, axis%, &H61, 2, 0)
'
'       'wait for profile complete
'       FOR i& = 0 TO 100000
'               'read hardware status
'               retstat% = sendcommand(boardaddr%, axis%, &H49, 2, 0)
'               retstat% = readrdb(boardaddr%, 1, axis(), cmd(), rdbdata())
'               IF ((rdbdata(0) AND &H400) OR (rdbdata(0) AND &H4)) THEN
'                       if NOT (rdbdata(0) AND &H80) THEN EXIT FOR
'               END IF
'       NEXT i&
'
'
'       'read position
'       retstat% = sendcommand(boardaddr%, axis%, &H53, 2, 0)
'       numpack% = 1
'       retstat% = readrdb(boardaddr%, numpack%, axis(), cmd(), rdbdata())
'
'       'read the communications status register
'       retstat% = readcsr(boardaddr%, csr%)
'
'
'
'For further technical information on programming a nuLogic board
'please refer to the appropriate User Manual.
'***************************END OF README*******************************

'************************BEGIN CODE*************************************
'function prototypes
DECLARE FUNCTION readrdb% (boardaddr%, numpacks%, axis%(), cmd%(), rdbdata&())
DECLARE FUNCTION readcsr% (boardaddr%, csr%)
DECLARE FUNCTION byteswap% (wordtoswap%)
DECLARE FUNCTION clearcmderr% (boardaddr%)
DECLARE FUNCTION checkboarderr% (boardaddr%)
DECLARE FUNCTION checkpreviouserr% (boardaddr%)
DECLARE FUNCTION chkcmdprocess% (boardaddr%)
DECLARE FUNCTION datapending% (boardaddr%)
DECLARE FUNCTION readytoreceive% (boardaddr%)
DECLARE FUNCTION pollrtrandce% (boardaddr%)
DECLARE FUNCTION sendpacket% (boardaddr%, packet%(), packetsize%)
DECLARE FUNCTION sendcommand% (boardaddr%, axis%, cmd%, wcount%, cmddata&)
DECLARE SUB waitforpulse (boardaddr%)
DECLARE SUB initasm ()

'****GLOBALS****
DIM in16asm(0 TO 8) AS INTEGER
COMMON SHARED in16asm() AS INTEGER
DIM out16asm(0 TO 8) AS INTEGER
COMMON SHARED out16asm() AS INTEGER
'****END OF GLOBALS****

'packet terminator value
CONST TERM = &HA

'machine code to perform 16 bit I/O read
in16code:
	DATA &H55,&H8B,&HEC,&H8B,&H5E,&H08,&H8B,&H17
	DATA &HED,&H8B,&H5E,&H06,&H89,&H07,&H5D,&HCA
	DATA &H04,&H00

'machine code to perform 16 bit I/O write
out16code:
	DATA &H55,&H8B,&HEC,&H8B,&H5E,&H08,&H8B,&H17
	DATA &H8B,&H5E,&H06,&H8B,&H07,&HEF,&H5D,&HCA
	DATA &H04,&H00

FUNCTION byteswap% (wordtoswap%)

'this routine takes a word, swaps the high byte
'with the low byte and returns the result
'all words sent to and from the board need
'to be byte-swapped

'some tricks are required to deal with signed numbers
IF wordtoswap% AND &H80 THEN
	lo8% = wordtoswap% OR &HFF00
ELSE
	lo8% = wordtoswap% AND &HFF
END IF
hi8% = wordtoswap% AND &HFF00
byteswap% = (lo8% * 256) OR ((hi8% / 256) AND &HFF)

END FUNCTION

FUNCTION checkboarderr% (boardaddr%)

'this routine reads the communications status register
'and checks the board failure and command bits,
'returning with the appropriate status

'get comm status register
DEF SEG = VARSEG(in16asm(0))
offset% = VARPTR(in16asm(0))
CALL absolute(boardaddr% + 6, csr%, offset%)
DEF SEG

'check board failure bit
IF (csr% AND &H10) THEN
	checkboarderr% = 5    'status 5: board failure
	EXIT FUNCTION
END IF

'check command error bit
IF (csr% AND &H4) THEN
	checkboarderr% = 2    'status 2: command error
	EXIT FUNCTION
END IF

'return 0 if no errors
checkboarderr% = 0

END FUNCTION

FUNCTION checkpreviouserr% (boardaddr%)

'checks for a pre-existing error before attempting
'to send a command to the board
'the appropriate error status is returned

retstat% = checkboarderr(boardaddr%)
SELECT CASE boardaddr%
	CASE 0
		checkpreviouserr% = 0     'status 0: no errors
	CASE 2
		retstat% = clearcmderr(boardaddr%)
		IF retstat% THEN
			checkpreviouserr% = retstat%
		ELSE
			checkpreviouserr% = 8  'status 8: command error
		END IF                         'occurred prior to current packet
	CASE 5
		checkpreviouserr% = 5  'status 5: board failure
END SELECT

END FUNCTION

FUNCTION chkcmdprocess% (boardaddr%)

'this routine first checks the ready to receive bit
'to make sure the packet has been received by the board
'after which the command error bit is polled for a
'minimum of 300 usec

'check ready to receive bit
retstat% = pollrtrandce(boardaddr%)
IF retstat% THEN
	chkcmdprocess% = retstat%
	EXIT FUNCTION
END IF

FOR i% = 0 TO 20

	'get comm status register
	DEF SEG = VARSEG(in16asm(0))
	offset% = VARPTR(in16asm(0))
	CALL absolute(boardaddr% + 6, csr%, offset%)
	DEF SEG

	'check command error bit, clear it if an error is found
	IF (csr% AND 4) THEN
		retstat% = clearcmderr(boardaddr%)
		IF retstat% THEN
			chkcmdprocess% = retstat%
		ELSE
			chkcmdprocess% = 2    'status 2: command error
		END IF
		EXIT FUNCTION
	END IF
NEXT i%

'no command errors if reached here
chkcmdprocess% = 0

END FUNCTION

FUNCTION clearcmderr% (boardaddr%)

'this routine attempts to clear a command error
'by sending up to five terminators to the board

FOR i% = 0 TO 4

	'check ready to receive
	retstat% = readytoreceive(boardaddr%)
	IF retstat% = 0 THEN
		clearcmderr% = 9  'status 9: unable to clear command error
		EXIT FUNCTION
	END IF

	'send a terminator word
	DEF SEG = VARSEG(out16asm(0))
	offset% = VARPTR(out16asm(0))
	CALL absolute(boardaddr%, &HA00, offset%)
	DEF SEG

	'check RTR - make sure terminator was taken
	gotrtr% = 0
	FOR j& = 0 TO 40
		'get comm status register
		DEF SEG = VARSEG(in16asm(0))
		offset% = VARPTR(in16asm(0))
		CALL absolute(boardaddr% + 6, csr%, offset%)
		DEF SEG

		'check ready to receive bit
		IF (csr% AND 1) THEN
		       gotrtr% = 1
		       EXIT FOR
		END IF
	NEXT j&
	IF gotrtr% = 0 THEN
		clearcmderr% = 9 'status 9: unable to clear command error
		EXIT FUNCTION
	END IF

	'check command error bit
	DEF SEG = VARSEG(in16asm(0))
	offset% = VARPTR(in16asm(0))
	CALL absolute(boardaddr% + 6, csr%, offset%)
	DEF SEG

	'if command error bit is cleared then return successfully
	IF (csr% AND 4) = 0 THEN
		clearcmderr% = 0
		EXIT FUNCTION
	END IF

NEXT i%

'unable to clear command error if here
clearcmderr% = 9

END FUNCTION

FUNCTION datapending% (boardaddr%)

'this routine polls for data pending, and returns a 1
'when data is available
'a zero is returned in the case of a timeout,

FOR i& = 0 TO 200
	'get comm status register
	DEF SEG = VARSEG(in16asm(0))
	offset% = VARPTR(in16asm(0))
	CALL absolute(boardaddr% + 6, csr%, offset%)
	DEF SEG

	'check data pending bit
	IF (csr% AND 2) THEN
		waitforpulse (boardaddr%)
		datapending% = 1
		EXIT FUNCTION
	END IF
NEXT i&

'not ready to receive if reached here, timeout condition
datapending% = 0

END FUNCTION

SUB initasm

'this routine loads the machine language for
'16 bit I/O transfers

'set up in16 assembly language array
DEF SEG = VARSEG(in16asm(0))
offset% = VARPTR(in16asm(0))
RESTORE in16code

FOR i% = 0 TO 17
	READ byte%
	POKE offset% + i%, byte%
NEXT i%

'set up out16 assembly language array
DEF SEG = VARSEG(out16asm(0))
offset% = VARPTR(out16asm(0))
RESTORE out16code

FOR i% = 0 TO 17
	READ byte%
	POKE offset% + i%, byte%
NEXT i%

DEF SEG

END SUB

FUNCTION pollrtrandce% (boardaddr%)

'this routine polls for the ready to receive bit
'to go active, and returns a 0 when it does,
'meanwhile the command error bit is checked
'and a status is returned if an error occurs
'a 1 is returned in the case of an RTR timeout

FOR i& = 0 TO 200

	'get comm status register
	DEF SEG = VARSEG(in16asm(0))
	offset% = VARPTR(in16asm(0))
	CALL absolute(boardaddr% + 6, csr%, offset%)
	DEF SEG

	'check command error
	IF (csr% AND 4) THEN
		retstat% = clearcmderr(boardaddr%)
		IF retstat% <> 0 THEN
			pollrtrandce% = retstat%
			EXIT FUNCTION
		ELSE
			pollrtrandce% = 2   'status 2: command error
			EXIT FUNCTION
		END IF
	END IF

	'check ready to receive
	IF (csr% AND 1) THEN
		waitforpulse (boardaddr%)
		pollrtrandce% = 0
		EXIT FUNCTION
	END IF
NEXT i&

'RTR timeout if reached here
pollrtrandce% = 1

END FUNCTION

FUNCTION readcsr% (boardaddr%, csr%)

'this routine will return the current contents of the
'communications status register

'get comm status register
DEF SEG = VARSEG(in16asm(0))
offset% = VARPTR(in16asm(0))
CALL absolute(boardaddr% + 6, csr%, offset%)
DEF SEG

readcsr% = 0   'return 0: successful

END FUNCTION

FUNCTION readrdb% (boardaddr%, numpacks%, axis%(), cmd%(), rdbdata&())

'this routine will read the contents of the return data buffer
'for the number of packets specified by "numpacks", or until
'the RDB is empty
'the actual number of packets read will be returned in numpacks%
'the data collected from the board will be stuffed into
'three arrays: axis, command and data for each packet read

'check for existing board level error
retstat% = checkpreviouserr(boardaddr%)
IF retstat% THEN
	readrdb% = retstat%
	EXIT FUNCTION
END IF

'check for data pending
retstat% = datapending(boardaddr%)
IF retstat% = 0 THEN
	readrdb% = 3       'status 3: no data pending
	EXIT FUNCTION
END IF

FOR j% = 0 TO (numpacks% - 1)

	'read 1st word: the packet identifier
	DEF SEG = VARSEG(in16asm(0))
	offset% = VARPTR(in16asm(0))
	CALL absolute(boardaddr%, packetid%, offset%)
	DEF SEG
	packetid% = byteswap(packetid%)

	'check the word count for validity
	wordcount% = packetid% / 4096     '12 bit shift
	IF (wordcount% = 2) OR (wordcount% = 3) THEN
	ELSE
		'try to flush data buffer if bad packet ID
		FOR i% = 0 TO 9999
			retstat% = datapending(boardaddr%)
			IF retstat% = 0 THEN EXIT FOR
			'read next word
			DEF SEG = VARSEG(in16asm(0))
			offset% = VARPTR(in16asm(0))
			CALL absolute(boardaddr%, trash%, offset%)
			DEF SEG
		NEXT i%
		IF i% = 10000 THEN
			readrdb% = 15  'status 15: unable to flush RDB
		ELSE
			readrdb% = 11 'status 11: corrupted packet ID
		END IF
		EXIT FUNCTION
	END IF

	'set the return axis and command
	axis%(j%) = (packetid% AND &HF00) / 256
	cmd%(j%) = packetid% AND &HFF

	'read the second word
	retstat% = datapending(boardaddr%)
	IF retstat% = 0 THEN
		readrdb% = 4       'status 4: incomplete packet
		EXIT FUNCTION
	END IF
	DEF SEG = VARSEG(in16asm(0))
	offset% = VARPTR(in16asm(0))
	CALL absolute(boardaddr%, dataloword%, offset%)
	DEF SEG
	dataloword% = byteswap(dataloword%)

	'read third word, if needed, and build return double word
	'some tricks are required to deal with signed integers
	IF wordcount% = 3 THEN
		retstat% = datapending(boardaddr%)
		IF retstat% = 0 THEN
			readrdb% = 4       'status 4: incomplete packet
			EXIT FUNCTION
		END IF
		DEF SEG = VARSEG(in16asm(0))
		offset% = VARPTR(in16asm(0))
		CALL absolute(boardaddr%, datahiword%, offset%)
		DEF SEG
		datahiword% = byteswap(datahiword%)
		'check datahiword for sign extension
		IF datahiword% AND &H8000 THEN
			hi16& = datahiword% OR &HFFFF0000
		ELSE
			hi16& = datahiword% AND 65535
		END IF
		lo16& = dataloword% AND 65535
		rdbdata&(j%) = (hi16& * 65536) OR lo16&
	ELSE
		rdbdata&(j%) = dataloword%
		rdbdata&(j%) = rdbdata&(j%) AND 65535
	END IF

	'check for more data before continuing
	retstat% = datapending(boardaddr%)
	IF retstat% = 0 THEN
		numpacks% = j% + 1   'set number of packets actually read
		readrdb% = 0
		EXIT FUNCTION
	END IF

NEXT j% 'end for (numpacks)

readrdb% = 0   'status 0: successful

END FUNCTION

FUNCTION readytoreceive% (boardaddr%)

'polls for ready to receive bit active
'returns 1 if bit is on
'returns a 0 if RTR timeout occurs

FOR i& = 0 TO 200
	'get comm status register
	DEF SEG = VARSEG(in16asm(0))
	offset% = VARPTR(in16asm(0))
	CALL absolute(boardaddr% + 6, csr%, offset%)
	DEF SEG

	'check ready to receive bit
	IF (csr% AND 1) THEN
		waitforpulse (boardaddr%)
		readytoreceive% = 1
		EXIT FUNCTION
	END IF
NEXT i&

'not ready to receive if reached here
readytoreceive% = 0

END FUNCTION

FUNCTION sendcommand% (boardaddr%, axis%, cmd%, wcount%, cmddata&)

DIM packet(0 TO 3) AS INTEGER

'this routine takes input consisting of axis, command
'word count and data, builds a packet and sends it to the board
'some tricks are required to deal with signed integers

IF wcount% = 2 THEN
	packet(0) = (((axis% OR &H20) * 256)) OR cmd%
	packet(1) = TERM
	retstat% = sendpacket(boardaddr%, packet(), 2)
ELSEIF wcount% = 3 THEN
	packet(0) = (((axis% OR &H30) * 256)) OR cmd%
	mask& = 65535
	lodata& = cmddata& AND mask&
	IF lodata& AND &H8000 THEN
		lodata& = lodata& OR &HFFFF0000
	END IF
	packet(1) = lodata&
	packet(2) = TERM
	retstat% = sendpacket(boardaddr%, packet(), 3)
ELSE
	packet(0) = (((axis% OR &H40) * 256)) OR cmd%
	packet(1) = (cmddata& AND &HFFFF0000) / 65536
	mask& = 65535
	lodata& = cmddata& AND mask&
	IF lodata& AND &H8000 THEN
		lodata& = lodata& OR &HFFFF0000
	END IF
	packet(2) = lodata&
	packet(3) = TERM
	retstat% = sendpacket(boardaddr%, packet(), 4)
END IF

sendcommand% = retstat%

END FUNCTION

FUNCTION sendpacket% (boardaddr%, packet%(), packetsize%)

'this routine is called by send command when a packet is
'ready to be sent to the board
'the packet is sent one word at a time, polling RTR

'check for existing command error
retstat% = checkpreviouserr(boardaddr%)
IF retstat% <> 0 THEN
	sendpacket% = retstat%
	EXIT FUNCTION
END IF

'send the packet
FOR i% = 0 TO (packetsize% - 1)

	'make sure ready to receive
	retstat% = pollrtrandce(boardaddr%)
	IF retstat% <> 0 THEN
		sendpacket% = retstat%
		EXIT FUNCTION
	END IF
	swappedword% = byteswap(packet%(i%))
	DEF SEG = VARSEG(out16asm(0))
	offset% = VARPTR(out16asm(0))
	CALL absolute(boardaddr%, swappedword%, offset%)
	DEF SEG

NEXT i%

'make sure command is sucessfully processed
retstat% = chkcmdprocess(boardaddr%)
IF retstat% <> 0 THEN
	sendpacket% = retstat%
	EXIT FUNCTION
END IF
sendpacket% = 0

END FUNCTION

SUB waitforpulse (boardaddr%)

'this routine provides a delay after a bit has been
'set by the board in the communications status register
'attempting to communicate too quickly can result in the
'bit getting stuck, minimum delay should be 25 usec

FOR i% = 0 TO 1
DEF SEG = VARSEG(in16asm(0))
offset% = VARPTR(in16asm(0))
CALL absolute(boardaddr% + 6, csr%, offset%)
DEF SEG
NEXT i%

END SUB

