/* SAMPLE1.C

   This program is mentioned in chapter 8 of OPTIMIZING SQL, by Peter
   Gulutzan + Trudy Pelzer; R&D Publications 1994. */

/* Conceptual explanation
   ----------------------
   This program tests back-truncation of index keys.  The concept is simple:
   read the key directly from the index, treating the index as a table.

   If the index is back truncated, then there will be characters missing in
   the retrieved host variables.

   As we warned in the book this is definitely not a standard SQL feature,
   and this particular example will work only with the SQL DBMS supplied on
   this diskette.  The SQL command on line 64 is not ODBC-sanctioned SQL
   syntax, and the SQL command on line 81 presumes one can select from 'i'.

   However, neither concept is unique.  Back truncation is found in several
   products, and selecting from indexes as if they were tables is possible in,
   for instance, both Ingres's and Ocelot's products. */

/* Style notes
   -----------

   We're passing SQL-statement strings using the ODBC call SQLExecDirect, so
   the DBMS will parse and execute them at runtime.  All these strings end
   with \0, which is what SQL_NTS means (NTS=null-terminated string).

   It's a lucky thing that SQL uses ' to enclose strings while C uses ".  Try
   to imagine how readability of this program would be affected if SQL used "
   to enclose strings, so that we'd have to fill our SQL statements with \042s
   rather than 's to avoid confusing the C compiler.

   The SQL-statement strings in this program have SQL keywords in upper case
   and non-keywords (such as table names or index names) in lower case. That's
   a fairly common convention, though we've emphasized that SQL is not a case-
   sensitive language. */

#include <stdio.h>
#include <stdlib.h>
#include "ocelot.h"	/* our equivalent of "sql.h" */
#define NULL 0

void main ()
{
  char country[11];	/* We'll fetch a 10-byte country name into this */
  HENV henv;		/* Handle of ENVironment */
  HDBC hdbc;		/* Handle of Data Base Connection */
  HSTMT hstmt;		/* HAndle of STaTeMent */

  /* This program assumes that the database named btrunc already exists.  You
     can use CHAP4.EXE for this: "CREATE DATABASE btrunc", "COMMIT", "EXIT" */

  SQLAllocEnv(&henv);
  SQLAllocConnect(henv,&hdbc);
  SQLConnect(hdbc,"btrunc",SQL_NTS,"",SQL_NTS,"",SQL_NTS);

  SQLAllocStmt(hdbc,&hstmt);

  /* Create a new table, create a new index file with a backtruncation flag
     that applies to all indexes which will be created in
     the file, create a table, create an index on the table, and insert 4
     records into the table. */
  SQLExecDirect(hstmt,"CREATE TABLE t (column1 CHAR(10))",SQL_NTS);

  if (SQLExecDirect(hstmt,"CREATE INDEXSPACE btrunc.ind BACKTRUNCATE",
  SQL_NTS)<0) {
    printf("Failed to create backtruncated index btrunc.ind\n");
    printf("Suggestion:  del btrunc.* and (re-)create database btrunc\n");
    exit(1); }

  SQLExecDirect(hstmt,"CREATE INDEX i ON t (column1)",SQL_NTS);

  SQLExecDirect(hstmt,"INSERT INTO t VALUES ('canada')",SQL_NTS);
  SQLExecDirect(hstmt,"INSERT INTO t VALUES ('china')",SQL_NTS);
  SQLExecDirect(hstmt,"INSERT INTO t VALUES ('colombia')",SQL_NTS);
  SQLExecDirect(hstmt,"INSERT INTO t VALUES ('costa rica')",SQL_NTS);

  /* Select all the records that are in i, which is the index of table t and
     should therefore contain the index keys for the 4 records we just
     inserted.  Fetch one row at a time and display them.  The display will
     be CAN+CHI+COL+COS, proving that the keys are in fact backtruncated. */
  SQLExecDirect(hstmt,"SELECT * FROM i",SQL_NTS);

  SQLBindCol(hstmt,1,SQL_C_CHAR,country,sizeof(country),NULL);
  while (SQLFetch(hstmt)==SQL_SUCCESS) printf("%s\n",country);

  /* Clean up */
  SQLFreeStmt(hstmt,SQL_DROP);
  SQLFreeConnect(hdbc);
  SQLFreeEnv(henv); }
