/* Find out if a column is indexed, with embedded SQL.

This program is mentioned in chapter 6 of Gulutzan + Pelzer's book.

The catalogue tables we're examining are OCELOT.SYSKEYS, which has one row 
for each index column, and OCELOT.SYSINDEXES, which has one row for each 
index.  Since one index can contain several columns, there's a many- to-one 
relationship between SYSINDEXES and SYSKEYS.

Note:  with IBM's DB2 (the real one, not the OS/2 DBMS with a similar name) 
the names would be have to be changed to SYSIBM.SYSKEYS+SYSIBM.SYSINDEXES, 
but everything else is precisely the same.  XDB also uses IBM-style catalogue 
names.  Oracle and Ingres have slightly different structures for the 
catalogue tables; we allowing for that in the way we create views at runtime, 
but only to show the concept for "non-IBM-style" catalogues in general. */

#include <stdio.h>
#include <string.h>
int indexed_column_check ();

void main ()
{
  int	i;
  exec sql include sqlca;

  exec sql connect to x;
  i=indexed_column_check();
  if (i==0) printf("not indexed\n");
  if (i==1) printf("not first in any index\n");
  if (i==2) printf("not alone in UNIQUE index\n");
  if (i==3) printf("alone in UNIQUE index\n"); }

int indexed_column_check ()
{
  exec sql include sqlca;
  exec sql begin declare section;
  char query_table_name[18+1];
  char query_column_name[18+1];
  int counter;
  exec sql end declare section;
  char c[300];

  printf("Enter table name:"); gets(query_table_name);
  printf("Enter column name:"); gets(query_column_name);

  /* First let's settle on what catalogue table we actually look up. We do 
  this by constructing a command string to be executed dynamically, which 
  makes a view of the table.  This dynamic-view-creation isolates the later 
  queries from the table definitions -- notice how if it's not a DB2-style 
  definition we can switch to an Oracle-style at runtime, and notice how we 
  avoid passing :query_table_name and :query_column_name in every search. */

  strcpy(c,"create view v1 ");
  strcat(c,"(index_name,table_name,index_type,column_name,column_seq) as ");
  strcat(c,"select i.name,i.tbname,i.uniquerule,k.colname,k.colseq ");
  strcat(c,"from ocelot.sysindexes i, ocelot.syskeys k ");
  strcat(c,"where i.name = k.ixname ");
  strcat(c,"and i.tbname = '"); strcat(c,query_table_name); strcat(c,"'");
  exec sql execute :c;
  if (sqlcode<0) {
    /* Error return.  Assume that the only possible error is that this is */
    /* not a DB2-style catalogue, so dynamically construct another view,  */
    /* this time based on an Oracle-style or Ingres-style definition. */
    strcpy(c,"create view v1 ");
    strcat(c,"(index_name,table_name,index_type,column_name,column_seq) ");
    strcat(c,"as select iname,tbname,indextype,colname,seq ");
    strcat(c,"from sysindexes ");
    strcat(c,"where tname='"); strcat(c,query_table_name); strcat(c,"'");
    exec sql execute :c;
    if (sqlcode<0) return (0); }

  strcpy(c,"create view v ");
  strcat(c,"(index_name,table_name,index_type,column_name,column_seq) ");
  strcat(c,"as select * from v1 where ");
  strcat(c,"column_name = '"); strcat(c,query_column_name); strcat(c,"'"); 
  exec sql execute :c;
  /* The simplest query is:  does this column name appear in any of the 
  indexes on this table name? */

  exec sql select count(*) into :counter from v;
  if (counter==0) return (0);			/* not indexed */

  /* Yes. The column is used in an index. UPDATEs will slow down whenever 
  this column's value changes. But we don't know much about the effect on 
  SELECTs yet, because  if this column is the second or third column in a 
  compound index, then the index isn't in order by this column and usually 
  won't be searched by this column. So add a condition:  is this the index's 
  first column? That is, is its sequence number 1? */

  exec sql select count(*) into :counter from v
  where column_seq = 1;
  if (counter==0) return (1);			/* Not first in any index */

  /* Yes. The column is used in an index, and it's the first in an index. So  
  we would answer the question "is column X indexed" with a "yes".  But is 
  the column uniquely indexed? Note that it isn't enough to find out that the 
  column is in a UNIQUE index; we should also find out that there's no other
  column in the same index, i.e. a column with a sequence number of 2.  */

  exec sql select count(*) into :counter
  from v
  where column_seq = 1
  and index_type like 'U%'
  and index_name not in (
  select index_name from v1 where column_seq = 2);
  if (counter==0) return (2);      /* not alone in UNIQUE index */

  /* The column is uniquely indexed. It's safe to throw out the word DISTINCT 
  if you SELECT this column, provided you aren't doing a join. It's likely 
  that this column will be given a high priority in ANDed predicates. 
  (Incidentally, if counter > 1 then there are unnecessary duplicate indexes 
  here.) */

  return (3);
  }
