/* Embedded-SQL C program for executing any SQL command 'online' with MS-DOS

   This program is mentioned in chapter 4 of OPTIMIZING SQL, by Peter
   Gulutzan + Trudy Pelzer; R&D Publications 1994. */
     
/* This program has one loop in main(), which prompts the user to type in any 
SQL command.  If the command is 'EXIT', the program exits the loop and 
returns to MS-DOS.  If the command is 'SELECT ...', the program calls the 
doselect() procedure, and the user will see a columnar display of the 
results.  If the command is anything else, the program simply executes it, 
and displays the returned error information.
     
The program can run into these problems which are common to all columnar-
display scenarios:

1) not enough room on one line (solved by shrinking 'screen column' widths 
   and allowing more than one line for each 'SQL row'),

2) not enough room on one page (solved by stopping at the end of a 'SQL row' 
   and prompting the user to hit a key to see more),

3) width of header greater than width of 'SQL column' (solved by allocating 
   more lines for the header). */

/* Implementation-specific notes: the assumed compiler is Borland C++ v3.1 
(we use a few Borland-specific library routines). The assumed DBMS is the one 
supplied with this book. Other SQL DBMSs may differ in these ways:

(a) they may require the use of EXEC SQL EXECUTE IMMEDIATE rather than EXEC 
SQL EXECUTE,

(b) they may require the use of PREPARE (we've omitted this step),

(c) they may be unable to convert all non-CHAR data to CHAR,

(d) 'sqlcode' may be called 'sqlca.sqlcode', and 'sqlmess_' may be either be 
absent or have another name.

Except for those minor differences, this program should work with any and all 
'embedded-SQL' database management systems. */

/* The screen is divided into a command area, an error-information display 
area, and a selected-columns display area.  For example: */

/* COMMAND: select author,title from book		starts at LINE_1  */
/*     									  */
/* Error code = 0, "OK"					starts at LINE_5  */
/*								          */
/* AUTHOR       |    TITLE				(header)	  */
/* ---------------------------------------				  */
/* Smith, J     |     My life in pictures	        (body)		  */
/* Smith, K     |     My life in sounds					  */
/* [Type any key to see next row]     			this is LINE_LAST */

#define    LINE_1          1
#define    LINE_5          5
#define    LINE_LAST       25

/* prototypes of 'gotoxy', 'clreol', 'clrscr' */
#include <conio.h>
/* prototype of 'memmove' */
#include <mem.h>
/* prototypes of 'printf', 'gets' */
#include <stdio.h>
/* prototype of 'strncmpi' */
#include <string.h>
/* prototypes of 'malloc', 'free' */
#include <stdlib.h>

/* Since we'll need sqlcode and sqlmess_, we define them: */
EXEC SQL INCLUDE SQLCA;

/* Since we'll need a SQLDA, we define it: */
EXEC SQL INCLUDE SQLDA;

/* The above SQL statement 'include sqlda' will generate a C statement 
similar to: '#include "sqlda_c.inc"'. Most database vendors provide an 
include file which has both the structure declaration and the allocation 
statement for an area named 'sqlda', with an arbitrary maximum of sqlvar 
occurrences. If the maximum is insufficient for your purposes, it is possible 
to either change the limit set by the vendor, or define your own 'sqlda' 
using the structure shown previously. */

/* The only host variable we need is 'command_string'. It will contain the 
string that the user types in (accepted with the C 'gets' function). We'll 
pass its value to the DBMS with 'exec sql describe :command_string;' and 
'exec sql execute :command_string;'. */

EXEC SQL BEGIN DECLARE SECTION;
  char command_string[255];
EXEC SQL END DECLARE SECTION;

/* Constants used for line drawing */
#define BAR '|'			/* we use BAR to separate columns */
#define UNDERLINE '-'		/* we use UNDERLINE to underline the header */

#define SCREENWIDTH 80

/* Procedure prototypes */
/* There are only 2 procedures: one if the command is SELECT, one to fill */
void doselect (void);
void display_sql_row (char *display_type);

void main ()
{
  char *p;

  clrscr();
  /* Loop: get string from user, execute it, repeat till 'EXIT' seen. */
  for (;;) {
    /* Screen lines 1 to 3 are for "Command:" prompt and user input. We begin
    by clearing them, since the last command might still be there. */
    gotoxy(1,1); clreol();
    gotoxy(1,2); clreol();
    gotoxy(1,3); clreol();
    gotoxy(1,1); printf("Command:");
    gets(command_string);			/* Get user input  */
    /* We must know in advance whether a command is SELECT or EXIT. */
    for (p=command_string; *p==' '; ++p) ;	/* skip lead spaces */
    if (strnicmp(p,"EXIT",4)==0) break;		/* break if 'EXIT' */
    if (strnicmp(p,"SELECT",6)==0) doselect();	/* do this if 'SELECT' */
    else {
      EXEC SQL EXECUTE :command_string; }	/* do this if not */
    if (sqlcode>=0) {
      EXEC SQL COMMIT; }
    clrscr();
    gotoxy(1,LINE_5);
    printf("sqlcode=%d\n",sqlcode);
    printf("%s",sqlmess_); }
  /* We've exited the loop because the user typed the word 'EXIT'.  */
  clrscr(); }					/* and return to MS-DOS now */

/* Describe and Execute a command that contains a SELECT statement */
void doselect ()
{
  int sql_column_number,sql_column_type,sql_column_width,j,k;
  long int offset;
  struct sqlvar *varptr;
  int saved_sqlcode;
  char saved_sqlmess_[255];
  char far *malloc_area=0L;

  /* If any SQL commands inside this procedure return an error, we'll jump to 
  f, which is a label towards the end of the procedure.  */
  EXEC SQL WHENEVER SQLERROR GOTO f;
  EXEC SQL DESCRIBE :command_string INTO :sqlda;      /* Fills sqlda/sqlvar */

  /* The 'describe' gives us SQL column names, types, and sizes -- but not 
  the addresses. Setting up the addresses is the program's responsibility. 
  For this application, we've decided to use malloc() to create a SINGLE area 
  that's just large enough to fit the data that can be returned. That means 
  we'll loop through the sqlda twice -- the first time we'll be adjusting the 
  types and calculating the widths, the second time we'll be adding the 
  address of the malloc'd area to the offsets we calculate. */

  offset=0;
  for (sql_column_number=0;			/* first loop through sqlda */
  sql_column_number<sqlda.sqld;++sql_column_number) { 
    varptr=&sqlda.sqlvar[sql_column_number]; 
    sql_column_type=varptr->sqltype;
    /* 448/449 is VARCHAR, 452/453 is CHAR */
    if ((sql_column_type & 0xfffe)!=448 && (sql_column_type & 0xfffe)!=452) {
      /* Next we force all non-character sqltypes to be 449 varying length 
      nullable character string (VARCHAR), and assume that our DBMS will be 
      able to automatically convert all numbers/dates/times/timestamps to 
      character strings when it passes to the host, and that none of those 
      types requires more than 20 bytes to store. */
      varptr->sqltype=449;			/* fiddle it to a varchar! */
      varptr->sqllen=20; }			/* fiddle the size to 20! */
    sql_column_width=varptr->sqllen;
    offset+=sql_column_width+2;			/* calc next col offset */
    offset+=sizeof(int); }			/* indicators are integers */
  if ((malloc_area=malloc(offset))==0L) goto f;
  offset=0;
  for (sql_column_number=0;			/* second loop through sqlda*/
  sql_column_number<sqlda.sqld;++sql_column_number) {
    varptr=&sqlda.sqlvar[sql_column_number];
    varptr->sqldata = malloc_area + offset;	/* set data's address */
    offset+=varptr->sqllen+2;
    varptr->sqlind = (int *)malloc_area+offset;	/* set indicator's address */
    offset+=sizeof(int); }
  /* The setup of the sqlda descriptor area is now complete. From now on the 
  only thing that changes is the data stored *malloc_area -- because of the 
  way that we set up the sqlda, each FETCH in the forthcoming loop will be 
  transferring data to designated addresses within malloc_area. 
  gotoxy(1,LINE_LAST);				/* put prompt on last line */
  printf("Type any key to fetch next row.");

  /* On line 5, display 'headings': field names as gotten from 'describe'. */
  display_sql_row("heading");			   /* display heading */
  for (k=0; k<SCREENWIDTH; ++k) putch(UNDERLINE);  /* underline heading */

  EXEC SQL DECLARE C CURSOR FOR :command_string;
  /* Notice the "dynamic" form of the DECLARE here. This works out to meaning 
  "declare c cursor for <a SQL command written to the host variable 
  'command_string'>;". */

  EXEC SQL OPEN C;				/* Do selection */
  for (;;) {					/* i.e. "infinitely" */
    EXEC SQL FETCH C USING DESCRIPTOR :sqlda;	/* get 1 row */
    /* The FETCH here is also a "dynamic" variant.  Instead of fetching 
    directly into variables as static SQL would, we fetch indirectly into 
    *malloc_area using the pointers supplied in sqlda. */
    if (sqlcode==100) break;			/* exit loop if no more rows*/
    display_sql_row("detail"); }		/* show row data on screen */
  gotoxy(1,LINE_LAST);
  printf("Enter any key to end"); clreol();
  getch();
f:
  EXEC SQL WHENEVER SQLERROR CONTINUE;

  /* When we return to the main loop, we want to display sqlcode + sqlmess_. 
  But the following 'close cursor' command will change those variables. So we 
  save them, close, and restore. Technically, the 'close cursor' could be 
  avoided because when we exit the loop we do a COMMIT. COMMIT closes all 
  open cursors. */

  saved_sqlcode=sqlcode; strcpy(saved_sqlmess_,sqlmess_);
  EXEC SQL CLOSE C;
  sqlcode=saved_sqlcode; strcpy(sqlmess_,saved_sqlmess_);
  if (malloc_area!=0L) free(malloc_area); }

void display_sql_row (char *displaytype)
{
  unsigned int	screen_column_width,line_number,max_line_number,k;
  char linebuf[LINE_LAST][SCREENWIDTH];/* store here before displaying rows */
  unsigned int	sql_column_number;
  int		sql_column_width;
  char far	*sql_column_pointer;
  char		*screen_column_pointer;
  struct sqlvar	*varptr;
  static char	question_mark[]="?";
  static int	current_line,first_display_line;

  /* Our method of dividing up the screen line into screen columns is very 
  simple: we'll just divide the screen width by the number of SQL columns, 
  which results in a situation where all screen columns are the same size. */

  screen_column_width=SCREENWIDTH / sqlda.sqld;
  setmem(linebuf,sizeof(linebuf),' ');		/* clear the buffer */
  max_line_number=0;
  for (sql_column_number=0;
  sql_column_number<sqlda.sqld;++sql_column_number) {
    varptr=&sqlda.sqlvar[sql_column_number];

    /* There are three possible data locations:
    (1) if this is the heading, then the data is in varptr->sqlname,
    (2) if this is a detail row and the indicator value is less than 0,
        we'll use a question mark to signify it (it's conventional to use "?"
        for displaying NULLs),
    (3) if this is a detail row and the indicator value is greater than or
        equal to 0, the data is in what varptr->sqldata points to. */

    if (strcmp(displaytype,"heading")==0) {
      sql_column_width=varptr->sqlname.length;
      sql_column_pointer=varptr->sqlname.data; }
    else {
      if (*(varptr->sqlind)<0) {
        sql_column_width=1;
        sql_column_pointer=question_mark; }		/* indicator NULL */
      else {
        sql_column_width=*((unsigned int *) varptr->sqldata);
        sql_column_pointer=varptr->sqldata+2; } }

    /* We've got an array linebuf[LINE_LAST][SCREENWIDTH] i.e. a 25x80 buffer.
    Transfer from the SQL data location to the appropriate buffer location. */

    for (line_number=0;;++line_number) {
      screen_column_pointer=
      &linebuf[line_number][sql_column_number*screen_column_width];
      *(screen_column_pointer++)=BAR;		/* | to separate cols */
      if (sql_column_width<=0) break;
      for (k=1; k<screen_column_width && --sql_column_width>=0;++k) {
        *(screen_column_pointer++)=*(sql_column_pointer++); } }
    if (line_number>max_line_number) max_line_number=line_number; }

  /* We now have 'max_line_number' lines filled within 'linebuffer'. First we 
  scroll the window up so there's enough room to display. Then we display. */
  if (strcmp(displaytype,"heading")==0) {
    for (current_line=LINE_LAST; current_line>=LINE_5; --current_line) {
    gotoxy(1,current_line);
    clreol(); }
    first_display_line=LINE_5+max_line_number+1; }
  if (current_line+max_line_number>=LINE_LAST-1) {
    gotoxy(1,LINE_LAST);
    printf("Type any key to see next row.");
    getch();
    gotoxy(1,LINE_LAST);
    clreol();
    gotoxy(1,first_display_line);
    for (k=0; k<max_line_number; ++k) delline();
    gotoxy(1,current_line-1); }
  else current_line+=max_line_number;
  for (line_number=0; line_number<max_line_number; ++line_number) {
    for (k=0; k<SCREENWIDTH; ++k) putch(linebuf[line_number][k]); } }

/* NOTES
1) For simplicity we have eliminated overflow testing. Both sqlda and 
   command_string could overflow, and this should be tested for.
2) We didn't do a separate malloc() for every variable. In large-model 
   Windows, the malloc() would probably be converted to GlobalAlloc(), and 
   there's a limited number of GlobalAlloc()s that can be done with Windows
   3.1. So the code calculates the total size, and subdivides that.
3) We didn't try to save a few bytes of data space by saying "if (the column 
   is defined as NOT NULL) then (we won't require a word for the indicator 
   integer)", on the theory that NOT NULL columns can't be NULL.  Actually, a 
   'select' with an outer join, e.g.: "SELECT A.COL,B.COL FROM A LEFT JOIN B 
   WHERE A.COL=B.COL" can return NULLs for B.COL. It doesn't matter that
   B.COL was defined as NOT NULL.
4) We use "sizeof(int)", not 2, because 32-bit code is becoming increasingly 
   common these days. Symantec C++ users should replace this with __INTSIZE.*/

/* Example ends */
