;clear all the variables and procedure which maybe left from previous run.
RELEASE VARS ALL
RELEASE PROCS ALL

;assign an error procedure to handle run time error
ErrorProc = "ErrorHandle"

;****************************************************************
; the error handler printed the informative error message and quit
; the execution.
;****************************************************************
PROC ErrorHandle()

  PRIVATE ErrorProc            ;in case of errors within Handler
  ERRORINFO TO ErrorData
  MESSAGE "A Paradox runtime error occured"

  CLEAR
  @5,5 ?? "Paradox runtime error : "
  @11,5 ?? "Lines : ",ErrorData["Line"],"     ", "Procedure : ", ErrorData["PROC"]

  IF ERRORCODE() <> 0 THEN
     @10,5 ?? "Error Code = ", ERRORCODE()
     @8,5 ?? ERRORMESSAGE()
  ENDIF
  SLEEP(8000)
  QUIT

ENDPROC

;************************************************************************
;create a message window to display the message 
;************************************************************************
PROC CLOSED CreateMsgWindow(MsgArray)

  ;set window height
  WH = 14
  ;window attributes array
  DYNARRAY WinArray[]      
  WinArray["ORIGINROW"] = 6
  WinArray["ORIGINCOL"] = 8
  WinArray["WIDTH"] = 64
  WinArray["HEIGHT"] = WH
  WinArray["STYLE"] = 38
  WinArray["FLOATING"] = TRUE
  WinArray["TITLE"] = "WARNING"
  WinArray["HASFRAME"] = False
   
  ;only allow to have 12 lines of messages, the extra message will be 
  ;truncated 
  If (DYNARRAYSIZE(MsgArray) <= (WH - 2)) then
    mLines = DYNARRAYSIZE(MsgArray)
  else 
    mLines = WH - 2
  endif
  
  ;Create the window and get the window handle, hMsgWin
  Window Create Attributes WinArray to hMsgWin

  ;Place information to the window
  ;first 3 line is aligned at center
  FOR i FROM 1 TO 2
    @ i,0 ?? Format("W64,AC", MsgArray[i])
  ENDFOR

  FOR i FROM 3 TO mLines
    @ i,0 ?? Format("W64", MsgArray[i])
  ENDFOR
  
  ;return the window handle to the calling procedure
  return(hMsgWin)
    
ENDPROC

;****************************************************************
; Close the message window, this procedure must be called 
; after the CreateMsgWindow(MsgArray), hWin is the handle of the
; window to be closed.
;****************************************************************
PROC CLOSED CloseMsgWindow(hWin)

  WINDOW SELECT hWin
  If Retval = TRUE then
    WINDOW CLOSE
    SETCANVAS DEFAULT
  Else
    message "Unable to close the selected message window"
  Endif
  
ENDPROC

;****************************************************************
; Display a welcome message for using this utilities
;****************************************************************
PROC CLOSED WelcomeMsg()
  DYNARRAY Msg[]
  Msg[1] = "A L C O M"
  Msg[2] = "Database Utilities for Lanfax Redirector 2.15-GL"
  Msg[3] = ""
  Msg[4] = "These utilities let you: "
  Msg[5] = "- Upgrade database tables to the new format"
  Msg[6] = "- Password protect your database tables"
  Msg[7] = "- Print reports and export data from the database tables"
  Msg[8] = "- Maintain the audit trail database"
  Msg[9] = "- See statistics and get information on fax server usage"
  Msg[10] = ""
  Msg[11] = "Press any key to continue"

  handle = CreateMsgWindow(Msg)
  while NOT CHARWAITING()
    sleep(10)
  endwhile
  CloseMsgWindow(handle)
  
ENDPROC

;****************************************************************
; Display a printing message window
;****************************************************************
PROC CLOSED PrintMsg()

  DYNARRAY Msg[]
  Msg[1] = "The job you selected is PRINTING now"
  Msg[2] = "Please Wait!!"
  Msg[3] = ""
  Msg[4] = "If you want to cancel the print job,"
  Msg[5] = "press ESC or SPACE now."

  Handle = CreateMsgWindow(Msg)
  return Handle
  
ENDPROC

;******************************************************************
; Display a message of missing template files when trying to
; print a report.
;******************************************************************
PROC CLOSED ReportMsg(template)

  DYNARRAY Msg[]
  Msg[1] = "ERROR"
  Msg[2] = ""
  Msg[3] = "The template files " + template + ".db and " + template + ".r1 "
  Msg[4] = "are missing from the current directory"
  Msg[5] = ""
  Msg[6] = "The result of the query is stored in the answer.db."

  Handle = CreateMsgWindow(Msg)
  sleep(3000)
  CloseMsgWindow(handle)

ENDPROC

;**********************************************************************
; Get password from the user, a user has three chances of tries
;**********************************************************************
PROC CLOSED GetPassword()

  SHOWDIALOG "Input Password" @7,12 HEIGHT 9 WIDTH 46

	  @2,3 ?? "Enter the password : "
	  ACCEPT @2,24 WIDTH 18
	      "A15"
	      HIDDEN 
	      ;REQUIRED
	      TAG "Password"
	      TO UserPass
   
	  PUSHBUTTON @4,10 WIDTH 6
	      "~O~K"
	      OK
	      DEFAULT
	      VALUE "Go On"
	      TAG "OKTag"
	      TO ButtonValue
   
	  PUSHBUTTON @4,22 WIDTH 10
	      "~C~ancel"
	      CANCEL
	      VALUE "Stop"
	      TAG "CancelTag"
	      TO ButtonValue
	    
       ENDDIALOG

       IF NOT Retval THEN
	  RETURN "N/A"
       ELSE
	  RETURN UserPass
       ENDIF
   
 ENDPROC

;***********************************************************************
; Get the user input for a name string. This procedure is used to get
; the group name or users name for the query on user or phone book.
;***********************************************************************
PROC CLOSED GetNameStrInput(DlgTitle,DlgPrompt)

  i = LEN(DlgPrompt) + 4  ;position to start the user input dialog box
  w = 70 - i - 3          ;the length of the user input dialog box
  
  SHOWDIALOG DlgTitle @5, 5 HEIGHT 9 WIDTH 70

	  @2,3 ?? DlgPrompt
	  ACCEPT @2,i WIDTH w
	      "A32"
	      TAG "Name"
	      TO NameStr
   
	  PUSHBUTTON @4,20 WIDTH 6
	      "~O~K"
	      OK
	      DEFAULT
	      VALUE "Go On"
	      TAG "OKTag"
	      TO ButtonValue
   
	  PUSHBUTTON @4,32 WIDTH 10
	      "~C~ancel"
	      CANCEL
	      VALUE "Stop"
	      TAG "CancelTag"
	      TO ButtonValue
	    
       ENDDIALOG

       IF NOT Retval THEN
	  RETURN "N/A"
       ELSE
	  RETURN NameStr
       ENDIF

ENDPROC       

;**********************************************************************
; This procedure prompt the user to input two dates as starting and
; ending date for processing activity
;**********************************************************************
PROC GetDateRange()

  startD = 1/1/90
  endD = TODAY()
  
  SHOWDIALOG "TIME FRAME" @ 8,8 HEIGHT 13 WIDTH 64

    @ 2, 2 ?? "Please input the start date : "
  
    ACCEPT @2,34 WIDTH 15
      "D" 
      REQUIRED
      TAG "StartDate"
      TO startD
    
    @ 5, 2 ?? "Please input the end date : "
  
    ACCEPT @5,34 WIDTH 15
      "D"
      REQUIRED
      TAG "EndDate"
      TO endD
    
    PUSHBUTTON @ 8, 20 WIDTH 10
      "~O~K"
      OK
      DEFAULT
      VALUE "Say Yes"
      TAG "OKTag"
      To ButtonValue
    
    PUSHBUTTON @ 8, 35 WIDTH 10
       "~C~ancel"
      CANCEL
      VALUE "Say No"
      TAG "CancelTag"
      TO ButtonValue
    
  ENDDIALOG

  RETURN retval

  CLEARALL
  CLEAR

ENDPROC


;**********************************************************************
; Get a new directory path
;**********************************************************************
PROC GetTablePath(TableName)

     CLEAR
  
     SHOWDIALOG "Table Directory" @ 5,4 HEIGHT 13 WIDTH 72

       @3, 2 ?? "Please input the directory path for the ", TableName, " table : "   
       ACCEPT @5,5 WIDTH 50
       "A50" 
       REQUIRED
       TAG "TablePath"
       TO TblPath
    
    
      PUSHBUTTON @ 8, 20 WIDTH 10
	"~O~K"
	OK
	DEFAULT
	VALUE "Say Yes"
	TAG "OKTag"
	To ButtonValue
    
      PUSHBUTTON @ 8, 35 WIDTH 10
	 "~C~ancel"
	 CANCEL
	 VALUE "Say No"
	 TAG "CancelTag"
	 TO ButtonValue
    
      ENDDIALOG
      
      IF RetVal then
	;save the current directory name
	IF DIREXISTS(TblPath) = 1 THEN
	   return TblPath
	ELSE
	   Message "Not a valid Directory Path"
	   sleep(2000)
	   message ""
	   RETURN "N/A"
	ENDIF   
      ELSE
	 message "User cancelled operation"
	 sleep(2000)
	 message ""
	 return "N/A"
      ENDIF   

ENDPROC

;**********************************************************************
; my version of checking whether a table exists. If it is not found in 
; the current directory, it prompts the user with a new directory and
; search the table there.
;**********************************************************************
PROC CLOSED myIsTable(TblName)
  
  if Not isTable(TblName) then 
     clear
     myPath = GetTablePath(TblName)
     If myPath  = "N/A" then 
       return FALSE
     ELSE
       ;check again if the table is in the new directory the user specify
       curPath = DIRECTORY()
       setdir myPath
       if not isTable(TblName) then
	  message TblName, " is not found in the new diretory"
	  sleep(2000)
	  message ""
	  setdir curPath
	  return FALSE
       else
	  message "Directory changed to ", myPath
	  sleep(2000)
	  message""
	  return TRUE
       endif      
     ENDIF
  ELSE
    RETURN TRUE   
  ENDIF  
 
ENDPROC

;**********************************************************************
; my version of open table. It checks the existence of the table and
; also ask the user to provide the password if the table is encrypted.
;**********************************************************************
PROC CLOSED myOpenTable(TblName)

  IF myIstable(TblName) = FALSE then
    ;the table doesn't exist in the current directory or the directory
    ;the user specify in the myIstable module
    reset
    return False
  ELSE
    ;the table does exist, check whether it is encrypted.
    IF isEncrypted(TblName) Then
       IF OpenEncryptedTable(TblName) = "N/A" Then
	  ;failed to open the encrypted table
	  message "Unable to open the encrypted table"
	  sleep(2000)
	  message ""
	  clear
	  reset
	  return False
       ENDIF   
    ENDIF

    RETURN TRUE 
     
  ENDIF
  
ENDPROC  

;*************************************************************************
; check to see if printer is ready
;*************************************************************************
PROC CLOSED IsPrinterReady()

  MESSAGE "Checking printer status ..."

  IF (PrinterStatus() <> TRUE) THEN
    BEEP
    MESSAGE "Printer is not ready"
    SLEEP 2500
    MESSAGE ""
    RETURN FALSE
  ENDIF 

  MESSAGE ""
  RETURN TRUE     

ENDPROC


;**********************************************************************
; The OpenEncryptedTable is a procedure to prompts the user to
; enter the password and verifies the password
;**********************************************************************
PROC CLOSED OpenEncryptedTable(TableName)
 
   CLEAR
   NumOfTries = 3
  
   FOR Attempts From 1 TO NumOfTries
       
       message "Please input the password for ", TableName, " table" 
       PW = GetPassword()
       message ""

       IF PW <> "N/A" then
	  
	  Password PW
	  IF TABLERIGHTS(TableName,"All") THEN
	     MESSAGE "Table opened successfully"
	     CLEAR
	     MESSAGE ""
	     RETURN PW
	  ELSE
	     MESSAGE "Incorrect Password"
	     ;reach the maximun number of tries given
	     IF Attempts = NumOfTries THEN
		MESSAGE "Sorry, you entered the wrong password"
		SLEEP(1000)
		CLEAR
		MESSAGE ""
		RETURN "N/A"
	     ENDIF
	  ENDIF

       ELSE
       
	 Message "User cancelled opening the table"
	 SLEEP(1000)
	 CLEAR
	 MESSAGE ""
	 return "N/A"
       
       ENDIF
	  
    ENDFOR
   
ENDPROC

;**********************************************************************
; The ConfirmDeleteTable is a procedure to let user to choose whether to 
; delete a table or not (rename is done instead)
;**********************************************************************

PROC ConfirmDeleteTable(TableName1, TableName2)

  CLEAR
  ARRAY ITEM[2]
  ARRAY EXPL[2]
  ITEM[1] = "YES"
  ITEM[2] = "NO "
  EXPL[1] = "Yes, delete the table"
  EXPL[2] = "No, do not delete the table"
  SHOWARRAY
    ITEM EXPL
    DEFAULT ITEM[1]
  TO x

  IF Retval THEN
     IF x = "YES" THEN
	MESSAGE "Deleting the table"
	DELETE TableName1
     ELSE
	RENAME TableName1 TableName2
     ENDIF
     CLEAR
  ELSE
     MESSAGE "ESC was pressed.  This program will now terminate"
     CLEAR
     RETURN 0
  ENDIF

ENDPROC

;**********************************************************************
; Get network type from user
;**********************************************************************
PROC CLOSED GetNetType()

  SHOWPOPUP " Please select the network type : " CENTERED
    "Banyan VINES" : "Banyan VINES" : "Banyan",
    "Novell/NetBIOS" : "Novell or any NetBIOS environment" : "IPX/NetBIOS"
  ENDMENU
  UNTIL "ESC" KEYTO KeyVar
  TO MenuVar
   
  IF KeyVar = FALSE THEN
    RETURN MenuVar
  ELSE
    RETURN "NULL"
  ENDIF
  
ENDPROC       

;**********************************************************************
; The UpGradeUser procedure is use to upgrade the users.db from LanFax
; Redirector 2.1 or earlier versions to LanFax 2.15GL
;**********************************************************************
PROC UpGradeUser()
  CLEAR
  @5,1 ?? "Upgrading USERS.DB (LFR2.1 -> LFR2.15GL)"

  IF ISTABLE("Users") THEN
 
    IF isEncrypted("USERS") Then
       IF OpenEncryptedTable("USERS") = "N/A" Then
	  message "Unable to open the encrypted table"
	  sleep(2000)
	  message ""
	  clear
	  return
	ENDIF   
    ENDIF
     
     IF ISTABLE("NewUser") THEN
	@3,5 ?? "Table NewUser is found, do you want to save it as NewUser0?"
	ConfirmDeleteTable("NewUser","NewUser0")
     ENDIF   

     NetTypeVar = GetNetType()
     SWITCH 
       CASE NetTypeVar = "Banyan" :
	 CREATE "NewUser"
	  "Name":"A65",
	  "MBox":"S",
	  "Routing":"A32",
	  "FaxName":"A32",
	  "RegExp":"S",
	  "InBoard":"N",
	  "OutBoard":"N",
	  "Rights":"S",
	  "Password":"A16",
	  "LastLogin":"D",
	  "FirstLogin":"D"
       CASE NetTypeVar = "IPX/NetBIOS" :   
	 CREATE "NewUser"
	  "Name":"A17",
	  "MBox":"S",
	  "Routing":"A32",
	  "FaxName":"A32",
	  "RegExp":"S",
	  "InBoard":"N",
	  "OutBoard":"N",
	  "Rights":"S",
	  "Password":"A16",
	  "LastLogin":"D",
	  "FirstLogin":"D"
       CASE NetTypeVar = "NULL" :
	 CLEAR
	 RETURN 0 
     ENDSWITCH

     VIEW "NewUser"
     HOME

     MESSAGE "Opening USERS"

     COEDIT "Users"
     HOME
     COPYTOARRAY SrcArray
     DB = ARRAYSIZE(SrcArray)

     SWITCH
       CASE DB = 12 :
	@3,1 ?? "Source database from LFR2.1c GL"
       CASE DB = 11 :
	@3,1 ?? "Source database from LFR2.1GL"
       CASE DB = 10 :
	@3,1 ?? "Source database from LFR2.1 or LFR2.05"
       CASE DB = 9 :
	@3,1 ?? "Source database from LFR2.0"
     ENDSWITCH

     IF DB <> 12 AND DB <> 11 AND DB <> 10 AND DB <> 9 THEN
	MESSAGE "Unknown database"
	RETURN 0
     ENDIF

     MESSAGE "Creating array"

     ARRAY DestArray[12]

     DestArray[1] = "NewUser"
     DestArray[4] = ""
     DestArray[5] = ""
     DestArray[6] = 0

     @15,1 ?? "Record#"
   
     FOR i FROM 1 TO NRECORDS("Users")

	@15,20 ?? i

	MOVETO "Users"
	COPYTOARRAY SrcArray
	DOWN
	IF NetTypeVar = "Banyan" THEN
	  DestArray[2] = UPPER(SrcArray["Name"])
	ELSE  
	  DestArray[2] = UPPER(SUBSTR(SrcArray["Name"],1,17))
	ENDIF     
	DestArray[3] = SrcArray["MBox"]
	IF DB=10 OR DB=11 THEN
	  DestArray[4] = SrcArray["FaxName"]
	  DestArray[5] = SrcArray["FaxName"]
	ELSE IF DB = 12 THEN
	       DestArray[4] = SrcArray["Routing"]
	       DestArray[5] = SrcArray["FaxName"]
	     ENDIF  
	ENDIF
	  
	IF DB=11 OR DB=12 THEN
	   DestArray[6] = SrcArray["RegExp"]
	ENDIF
	DestArray[7] = SrcArray["InBoard"]
	DestArray[8] = SrcArray["OutBoard"]
	DestArray[9] = SrcArray["Rights"]
	DestArray[10] = SrcArray["Password"]
	DestArray[11] = SrcArray["LastLogin"]
	DestArray[12] = SrcArray["FirstLogin"]

	MOVETO "NewUser"
	DOWN
	COPYFROMARRAY DestArray

     ENDFOR
   
     DO_IT!
   
     CLEAR

     @20,5 ?? "Old users.db has been renamed as OldUser.db"
     RENAME "Users" "OldUser"
     RENAME "NewUser" "Users"
     DO_IT!
     RELEASE VARS All
     MESSAGE "Done"
  ELSE
     @15,5 ?? "Table Users not found"
     SLEEP(2000)
  ENDIF
  
  CLEAR
  RESET
  
ENDPROC

;**********************************************************************
; The UpGradePhone procedure is use to upgrade the phone.db from LanFax
; Redirector 2.1 or earlier versions to LanFax 2.15GL
;**********************************************************************

PROC UpGradePhone()

  CLEAR

  @11,1 ?? "Upgrading PHONE.DB (LFR2.1 -> LFR2.15GL)"

  IF ISTABLE("Phone") THEN
 
    IF isEncrypted("PHONE") Then
       IF OpenEncryptedTable("PHONE") = "N/A" Then
	  message "Unable to open the encrypted table"
	  sleep(2000)
	  message ""
	  clear
	  return
	ENDIF   
    ENDIF
     
     IF ISTABLE("NewPhone") THEN
	@13,5 ?? "Table NewPhone is found, do you want to save it as NewTel0?"
	ConfirmDeleteTable("NewPhone","NewTel0")
     ENDIF   


     MESSAGE "Creating new PHONE.DB"

     CREATE "NEWPHONE"
	"Name"      : "A32*",
	"Owner"     : "S*",
	"ID"        : "S",
	"Type"      : "S",
	"Status"    : "S",
	"Company"   : "A32",
	"FaxC"      : "A4",
	"FaxA"      : "A5",
	"FaxN"      : "A32",
	"VoiceC"    : "A4",
	"VoiceA"    : "A5",
	"VoiceN"    : "A32",
	"Time"      : "S",
	"PollCode"  : "A2",
	"AuditInfo" : "A16",
	"BFTflag"   : "S",
	"Revised"   : "D"

     MESSAGE "Opening new PHONE.DB"

     VIEW "NEWPHONE"
     HOME

     MESSAGE "Opening PHONE.DB"

     COEDIT "PHONE"
     HOME
     COPYTOARRAY SrcArray
     DB = ARRAYSIZE(srcArray)
     ; all version of LFR has DB = 18

     @13,1 ?? "Start to trim the database"

     IF DB <> 18 THEN
	MESSAGE "Unknown database"
	RETURN 0
     ENDIF

     MESSAGE "Trimming"

     ARRAY DestArray[18]

     DestArray[1] = "NEWPHONE"

     @15,1 ?? "Record#"

     FOR i FROM 1 TO NRECORDS("PHONE")

	@15,20 ?? i

	MOVETO "PHONE"
	COPYTOARRAY SrcArray
	DOWN

	DestArray[2] = SrcArray["Name"]
	DestArray[3] = SrcArray["Owner"]
	DestArray[4] = SrcArray["ID"]
	DestArray[5] = SrcArray["Type"]
	DestArray[6] = SrcArray["Status"]
	DestArray[7] = SrcArray["Company"]
	DestArray[8] = SrcArray["FaxC"]
	DestArray[9] = SrcArray["FaxA"]
	DestArray[10] = SrcArray["FaxN"]
	DestArray[11] = SrcArray["VoiceC"]
	DestArray[12] = SrcArray["VoiceA"]
	DestArray[13] = SrcArray["VoiceN"]
	DestArray[14] = SrcArray["Time"]
	DestArray[15] = SrcArray["PollCode"]
	DestArray[16] = SrcArray["AuditInfo"]
	DestArray[17] = SrcArray["BFTflag"]
	DestArray[18] = SrcArray["Revised"]

	MOVETO "NEWPHONE"
	DOWN
	COPYFROMARRAY DestArray

     ENDFOR

     DO_IT!

     CLEAR

     @18,5 ?? "Old phone.db has been renamed as OldPhone.db"
     RENAME "Phone" "OldPhone"
     RENAME "NewPhone" "Phone"
     RELEASE VARS All
     MESSAGE "Done"
  ELSE
     @15,5 ?? "Table Phone not found"
     SLEEP(2000)
  ENDIF

  CLEAR

RESET

ENDPROC

;**********************************************************************
; The UpGradeAudit procedure is use to upgrade the Audit.db from LanFax
; Redirector 2.1 or earlier versions to LanFax 2.15GL
;**********************************************************************
PROC UpGradeAudit()

  CLEAR

  @11,1 ?? "Upgrading Audit.DB (LFR2.1 -> LFR2.15GL)"

  IF myISTABLE("Audit") = 1 THEN
    IF isEncrypted("Audit") Then
       IF OpenEncryptedTable("Audit") = "N/A" Then
	  message "Unable to open the encrypted table"
	  sleep(2000)
	  message ""
	  clear
	  return
	ENDIF   
    ENDIF
    
 
     IF ISTABLE("NewAudit") THEN
	@13,5 ?? "Table NewAudit is found, do you want to save it as NewAud0?"
	ConfirmDeleteTable("NewAudit","NewAud0")
     ENDIF   

     MESSAGE "Creating NEWAUDIT"
     CREATE "NEWAUDIT"
       "Date"     : "D",
       "Time"     : "A9",
       "Duration" : "A9",
       "Mode"   : "A5",
       "Type"   : "A5",
       "Status" : "S",
       "Pages"  : "S",
       "To"     : "A33",
       "From"   : "A33",
       "Phone"  : "A48",
       "Tag"    : "A65",
       "CSID"   : "A22",
       "Errors" : "S",
       "Retries"        : "S",
       "Board"  : "S",
       "MBox"   : "S",
       "Posted" : "S",
       "ID"     : "A9",
       "Noise"  : "S",
       "Speed"  : "S",
       "Quality"        : "S",
       "Strength"       : "S",
       "GFaxStat"       : "S"
       
     MESSAGE "Opening NEWAUDIT"

     VIEW "NEWAUDIT"
     HOME

     MESSAGE "Opening AUDIT.DB"

     COEDIT "AUDIT"
     HOME
     COPYTOARRAY SrcArray
     DB = ARRAYSIZE(SrcArray)
     ; LFR21GL    = 24
     ; LFR20, LFR21, 205 = 19

     IF DB=24 THEN
	@12,1 ?? "Source database from LFR2.1GL"
     ELSE
	IF DB=19 THEN
	   @12,1 ?? "Source database from LFR2.1, 2.05 or 2.0"
	ENDIF
     ENDIF

     IF DB <> 24 AND DB <> 19 THEN
	MESSAGE "Unknown database"
	RETURN 0
     ENDIF

     MESSAGE "Creating array"

     ARRAY DestArray[24]

     DestArray[1] = "NEWAUDIT"
     DestArray[20] = 0
     DestArray[21] = 0
     DestArray[22] = 0
     DestArray[23] = 0

     @14,1 ?? "Record#"

     FOR i FROM 1 TO NRECORDS("AUDIT")

	@14,20 ?? i

	MOVETO "AUDIT"
	COPYTOARRAY SrcArray
	DOWN

	DestArray[2] = SrcArray["Date"]
	DestArray[3] = SrcArray["Time"]
	DestArray[4] = SrcArray["Duration"]
	DestArray[5] = SrcArray["Mode"]
	DestArray[6] = SrcArray["Type"]
	DestArray[7] = SrcArray["Status"]
	DestArray[8] = SrcArray["Pages"]
	DestArray[9] = SrcArray["To"]
	DestArray[10] = SrcArray["From"]
	DestArray[11] = SrcArray["Phone"]
	DestArray[12] = SrcArray["Tag"]
	DestArray[13] = SrcArray["CSID"]
	DestArray[14] = SrcArray["Errors"]
	DestArray[15] = SrcArray["Retries"]
	DestArray[16] = SrcArray["Board"]
	DestArray[17] = SrcArray["MBox"]
	DestArray[18] = SrcArray["Posted"]
	DestArray[19] = SrcArray["ID"]
	IF DB=24 THEN
	   DestArray[20] = SrcArray["Noise"]
	   DestArray[21] = SrcArray["Speed"]
	   DestArray[22] = SrcArray["Quality"]
	   DestArray[23] = SrcArray["Strength"]
	   DestArray[24] = SrcArray["GFaxStat"]
	ELSE
	   DestArray[24] = SrcArray["Status"]
	ENDIF

	MOVETO "NEWAUDIT"
	DOWN
	COPYFROMARRAY DestArray

     ENDFOR

     DO_IT!

     CLEAR

     @18,5 ?? "Old Audit.db has been renamed as OldAudit.db"
     RENAME "Audit" "OldAudit"
     RENAME "NewAudit" "Audit"
     DO_IT!
     RELEASE VARS All
     MESSAGE "Done"
  ELSE
     @15,5 ?? "Table Audit not found"
     SLEEP(2000)
  ENDIF

  CLEAR

RESET

ENDPROC

;****************************************************************
; This procedure is used to set password to a table
;****************************************************************
PROC CLOSED PassProtectTable(TableName)

  DYNARRAY Msg[]
  
  CLEAR
  
  IF ISENCRYPTED(TableName) THEN 
     MESSAGE "ERROR: The table has been encrypted."
     SLEEP(2000)
     MESSAGE ""
     CLEAR
     RETURN 0
  ELSE
     Msg[1]="WARNING"
     Msg[2]=""
     Msg[3]="When Password Protecting your table, remember to make"
     Msg[4]="the appropriate changes to server configuration. Use"
     Msg[5]="the server SETUP program so that the server can continue"
     Msg[6]="to access the tables and work correctly."
     Msg[7]=""
     Msg[8]="Press any key to continue."

     Handle = CreateMsgWindow(Msg)
     while NOT CHARWAITING()
       sleep(10)
     endwhile
     CloseMsgWindow(Handle)
     
     PW = GetPassword()

     IF PW <> "N/A" THEN
	 ;verify the password again
	 @5,5 ?? "Please enter your password again for verification"
	 PW1 = GetPassword()
      
	 IF PW1 <> "N/A" THEN      
	     IF PW1 = PW THEN
		PROTECT TableName PW
		MESSAGE "The table is now password-protected."
		SLEEP(2000)
		MESSAGE ""
		CLEAR
		RETURN 1
	      ELSE
		MESSAGE "Password mismatch, table is not password-protected."
		SLEEP(2000)
		MESSAGE ""
		CLEAR
		RETURN 0
	      ENDIF
	  ELSE
	      MESSAGE "User cancelled operation."
	      SLEEP(2000)
	      MESSAGE ""
	      CLEAR
	      RETURN 0
	  ENDIF            
     ELSE             
	  MESSAGE "User cancelled operation."
	  SLEEP(2000)
	  MESSAGE ""
	  CLEAR
	  RETURN 0
     ENDIF

  ENDIF
  
ENDPROC

;*************************************************************************
; Print the report of database table
; ReportType is the report format for the table. The report format is 
; stored as a file in the name of tablename.r*. The user should have
; the correspond tablename.r1 exists at the local directory if ReportType
; 1 is choosen, same deal with the rest of the ReportType
;*************************************************************************
PROC CLOSED PrintReport(TableName, ReportName)
  
  IF IsPrinterReady() = TRUE THEN
     handle = PrintMsg()
     REPORT TableName ReportName
     CloseMsgWindow(handle)
     RETURN
  ELSE
     RETURN
  ENDIF

ENDPROC 


;***********************************************************************
;This procedure get the details information of the group members in a 
;group and place in GrpMbr table
;***********************************************************************
PROC CLOSED  GetGroupMemberInfo(GrpName)

   CLEAR
   ;check the existence and security of tables, group, phone and users
   IF myOpenTable("PHONE") = False then
     message "Unable to open PHONE table"
     sleep(2000)
     message ""
     reset
     return FALSE 
   ELSE
     ;the users.db and audit.db have to be in the same directory to do
     ;the query
     IF NOT isTable("GROUP") then
	message "Can't find the GROUP table in the current directory"
	sleep(2000)
	message ""
	reset
	return FALSE
     ELSE   
       IF NOT isTable("USERS") then
	 message "Can't find the USERS table in the current directory"
	 sleep(2000)
	 message ""
	 reset
	 return FALSE
       ELSE   
	 IF isEncrypted("USERS") = TRUE Then
	   IF OpenEncryptedTable("USERS") = "N/A" THEN
	      message "Unable to open the encrypted table"
	      sleep(2000)
	      message ""
	      reset
	      return FALSE
	   ENDIF
	ENDIF      
       ENDIF

       IF isEncrypted("GROUP") = TRUE Then
	   IF OpenEncryptedTable("GROUP") = "N/A" THEN
	      message "Unable to open the encrypted table"
	      sleep(2000)
	      message ""
	      reset
	      return FALSE
	   ENDIF
	ENDIF      
     ENDIF
   ENDIF   


   @ 5,5 ?? "processing the query on the group name now ....."

   ;find all the memberID for the group name
   Query
 
	 Phone |   Name   |   ID   | Type | 
	       | ~grpName | _grpID | 2    | 

	 Phone | 
	       | 

	 Group |   ID   |  Seq   |   M1   |   M2   |   M3   |   M4   |   M5   |
	       | _grpID | Check  | Check  | Check  | Check  | Check  | Check  | 

	 Group |   M6   |   M7   |   M8   | 
	       | Check  | Check  | Check  | 

     Endquery

     DO_IT!

     ;place all the member IDs to a field in a temp table GrpID000
     ;so that a query on the member details info can be done
     CLEAR

     IF NOT isEmpty("answer") then
       ;create a temp database table to hold all the group member ID
       CREATE "GrpID000"
	 "Group Name" : "A32",
	 "MemID" : "S"
	 
       VIEW "GrpID000"
       HOME  
	 
       VIEW "Answer"
       COEDIT "Answer"
       HOME

       @5,5 ?? "Reading the member ID : "       
       FOR i from 1 TO nRecords("Answer")
  
	  MOVETO "Answer" 
	  COPYTOARRAY srcArray
	  DOWN
	  
	  MOVETO "grpID000"
	  
	  ;skip the record number field and the sequence number field
	  FOR j from 3 to 10
	     if NOT isBlank(srcArray[j]) then
	       @5, 30 ?? srcArray[j]
	       [Group Name] = grpName
	       [memID] = srcArray[j]
	       DOWN
	     else
		QUITLOOP
	     ENDIF     
	  ENDFOR
	  
	ENDFOR
     
     DO_IT!        
 
  ELSE

    @5,5 ?? "There isn't a group named ", grpName, " or it has no member."      
    sleep(2000)
    message ""
    reset
    clear
    return FALSE

  ENDIF

  CLEAR
  @5,5 ?? "Processing query on the member details information now ....."  
  ;obtain all the details information of the group member 
  Query

	 Grpid000 | Group Name |   MemID   | 
		  | CHECK      | _memberID | 

	 Phone |  Name  |  Owner  |    ID     | Company |  FaxC  | 
	       | Check  | _userID | _memberID | Check   | Check  | 

	 Phone |  FaxA  |  FaxN  | VoiceC | VoiceA | VoiceN |  
	       | Check  | Check  | Check  | Check  | Check  | 

	 Users |      Name      |  MBox   | 
	       | Check As Owner | _userID | 

   Endquery
 
   DO_IT!
   
   DELETE "grpID000"
   
   CLEAR
   RESET
   RETURN TRUE
   
ENDPROC   

;**********************************************************************
;Print report of a group
;**********************************************************************
PROC CLOSED PrintGroup()

   CLEAR
   grpName=GetNameStrInput("Group Name", "Please input the group name :") 
   IF grpName <> "N/A" then
     IF GetGroupMemberInfo(grpName) then
       IF (isFile("grp.db")) AND (isFile("grp.r")) THEN
	  copyreport "grp" "R" "answer" "R"
	  PrintReport("answer", "R")
       ELSE
	 ReportMsg("grp")
	 RESET
	 RETURN
       ENDIF
     ELSE
       CLEAR
       reset
       return
     ENDIF  
   ELSE
     reset
     clear
     return
   ENDIF    

ENDPROC

;**********************************************************************
;Print report of phone book belongs to a user
;**********************************************************************
PROC CLOSED PrintPhone()
 
  CLEAR
  IF myOpenTable("Phone") = False then
     message "Unable to open PHONE table"
     sleep(2000)
     message ""
     reset
     return
  ELSE
     ;the users.db and audit.db have to be in the same directory to do
     ;the query
     IF NOT isTable("Users") then
	message "Can't find the USERS table in the current directory"
	sleep(2000)
	message ""
	reset
	return
     ELSE   
	IF isEncrypted("Users") = TRUE Then
	   IF OpenEncryptedTable("Users") = "N/A" THEN
	      message "Unable to open the encrypted table"
	      sleep(2000)
	      message ""
	      reset
	      return
	   ENDIF
	ENDIF      
     ENDIF
  ENDIF   

  CLEAR
  
  @3,5 ?? "Please key in the user login name to retrieve his/her phone book"
  UName = GetNameStrInput("User Login Name", "Please input the login name")

  IF UName <> "N/A" then
      UName = Upper(UName)
      Query

	 Phone |  Name  |  Owner   |  Type  | Status | Company |  FaxC  | 
	       | Check  | _OwnerID | Check  | Check  | Check   | Check  | 

	 Phone |  FaxA  |  FaxN  | VoiceC | VoiceA | VoiceN |  Time  | 
	       | Check  | Check  | Check  | Check  | Check  | Check  | 

	 Phone | BFTflag | Revised | 
	       | Check   | Check   | 

	 Users |   Name |   MBox   | 
	       | ~UName | _OwnerID | 

      Endquery

      DO_IT!
      
      CLEAR
      
      IF NOT isEmpty("Answer") then
	     IF (isFile("phn.db")) AND (isFile("phn.r")) THEN
		copyreport "phn" "R" "answer" "R"
		PrintReport("answer", "R")
	     ELSE
		ReportMsg("phn")  
		RESET
		CLEAR
		RETURN
	     ENDIF
      ELSE   
	 message "No record meets the query criteria"
	 sleep(2000)
	 message ""
	 CLEAR
	 RESET
	 RETURN
      ENDIF   
  ELSE
     reset
     clear
     return
  ENDIF    

ENDPROC

;**********************************************************************
;Print report of users list
;**********************************************************************
PROC CLOSED PrintUsers()
 
  CLEAR
  IF myOpenTable("Users") = False then
     message "Unable to open PHONE table"
     sleep(2000)
     message ""
     reset
     return
  ENDIF   

  CLEAR

  IF (isFile("usr.db")) AND (isFile("usr.r")) THEN
       copyreport "usr" "R" "users" "R"
       PrintReport("users", "R")
  ELSE
       ReportMsg("usr")  
       RESET
       CLEAR
       RETURN
  ENDIF
  
  CLEAR
  RESET
  RETURN

ENDPROC

;**********************************************************************
;Print the audit report. There is two choices for the users, either
;Print all entries or the audit of a specify user.
;**********************************************************************
PROC CLOSED PrintAudit()
  CLEAR
  CLEARALL
  
  IF myOpenTable("Audit") = FALSE THEN
    message "Failed to open Audit table"
    sleep(2000)
    message ""
    reset
    return
  ENDIF

  IF GetDateRange() = TRUE THEN
     SHOWPOPUP " All or Individual ? " CENTERED
       "ALL" : "Print all entries within the time frame" : "A",
       "Individual" : "Print entries of the individual only" : "I"
     ENDMENU
     
     TO MenuVar
     
     IF RetVal THEN
       
       switch 
	 
	 case MenuVar = "A" :
	  
	   @3,5 ?? "Processing the query now ....."
 
	   Query

	     Audit |           Date           |  Time  | Duration |  Mode  | 
		   | Check >=~StartD, <=~EndD | Check  | Check    | Check  | 

	     Audit |  Type  | Status | Pages  |   To   |  From  | Phone  | 
		   | Check  | Check  | Check  | Check  | Check  | Check  | 

	     Audit |  CSID  | Errors | Retries | Board  |   ID   | 
		   | Check  | Check  | Check   | Check  | Check  | 

	     Audit | 
		   | 
 
	  Endquery
	
	  DO_IT!
 
	  CLEAR
	  
	  IF NOT isEmpty("Answer") then
	     IF (isFile("adt.db")) AND (isFile("Adt.r")) THEN
		copyreport "adt" "R" "answer" "R"
		PrintReport("answer", "R")
	     ELSE
		ReportMsg("adt")  
		RESET
		CLEAR
		RETURN
	     ENDIF
	  ELSE   

		 message "No record meets the query criteria"
		 sleep(2000)
		 message ""
		 CLEAR
		 RESET
		 RETURN
	   ENDIF   

      case MenuVar = "I" :

	   Uname=GetNameStrInput("Input","Please input the user faxing name :") 

	   IF Uname <> "N/A" Then
	   
	     @3,5 ?? "Processing the query right now ....."

	     QUERY
		 Audit |           Date           |  Time  | Duration |  Mode  | 
		       | Check >=~StartD, <=~EndD | Check  | Check    | Check  | 

		 Audit |  Type  | Status | Pages  |  To   |     From     | Phone  | 
		       | Check  | Check  | Check  | Check | Check ~UName | Check  | 

		 Audit |  CSID  | Errors | Retries | Board  |   ID   | 
		       | Check  | Check  | Check   | Check  | Check  | 

		 Audit | 
		       | 
   
	     ENDQUERY
	     
	     DO_IT!
	     
	     CLEAR
	     
	      IF NOT isEmpty("Answer") then
		 
		 IF (isFile("adt.db")) AND (isFile("adt.r")) THEN
		   copyreport "adt" "R" "answer" "R"
		   PrintReport("answer", "R")
		 ELSE
		   ReportMsg("adt")  
		   RESET
		   CLEAR
		   RETURN
		 ENDIF
		 
	      ELSE   
		 message "No record meets the query criteria"
		 sleep(2000)
		 message ""
		 CLEAR
		 RESET
		 RETURN
	      ENDIF   
	     

	   ELSE  ;user cancelled the individual name input   
	     RESET
	     RETURN
	   ENDIF

       ENDSWITCH 
	   
     ELSE       ;user cancelled the date input
       CLEAR
       RESET
       RETURN
     ENDIF
     
  ELSE          ;user cancel the choice of printing
     message "User cancelled operation"
     sleep(2000)
     message ""
     reset
     return
  ENDIF 
  
ENDPROC
  
;*************************************************************************
; Export the database table to a file in ASCII format
;*************************************************************************
PROC CLOSED ExportToASCII()

  CLEAR
  SHOWTABLES DIRECTORY() "Select a table for exporting to ASCII file"
  To TableName
  
  IF RetVal = FALSE THEN
     RETURN
  ELSE
     IF TableName = "None" THEN
       MESSAGE "No table found in current directory"
       SLEEP (2000)
       CLEAR
       RETURN
     ELSE  
       CLEAR
       IF myOpenTable(TableName) = False then
	  CLEAR
	  RESET
	  RETURN
       ENDIF   
       @5,5 ?? "Please key in the filename : "
       ACCEPT "A40"
       REQUIRED
       TO FileName
     ENDIF
  ENDIF

  CLEAR
  IF RetVal = FALSE THEN
     Message "User cancelled operation"
     sleep(2000)
     message ""
     RETURN
  ELSE
     MESSAGE "Exporting the database table to an ASCII file ..."
     MENU {Tools} {ExportImport} {Export} {ASCII} {Delimited}
     SELECT TableName 
     SELECT FileName
     CLEAR
     MESSAGE ""
     RETURN 
  ENDIF
  
ENDPROC

;*************************************************************************
; Export the database table to a file in Lotus 1-2-3 release 2 format
;*************************************************************************
PROC CLOSED ExportToLotus123()

  CLEAR
  SHOWTABLES DIRECTORY() "Select a table for exporting to Lotus 1-2-3 file"
  To TableName
  
  IF RetVal = FALSE THEN
     RETURN
  ELSE
     IF TableName = "None" THEN
       MESSAGE "No table found in current directory"
       SLEEP (2000)
       MESSAGE ""
       CLEAR
       RETURN
     ELSE  
       CLEAR
       IF myOpenTable(TableName) = False then
	  CLEAR
	  RESET
	  RETURN
       ENDIF   
       @5,5 ?? "Please key in the output filename : "
       ACCEPT "A40"
       REQUIRED
       TO FileName
     ENDIF
  ENDIF

  CLEAR
  IF RetVal = FALSE THEN
     RETURN
  ELSE
     MESSAGE "Exporting the database table to an ASCII file ..."
     MENU {Tools} {ExportImport} {Export} {1-2-3} 
     DOWN ENTER
     SELECT TableName 
     SELECT FileName
     message "Exporting ....."
     CLEAR
     MESSAGE ""
     RETURN 
  ENDIF
  
ENDPROC


;*************************************************************************
; Export the database table to a file in Quattro Pro format
;*************************************************************************
PROC CLOSED ExportToQuattroPro()

  CLEAR
  SHOWTABLES DIRECTORY() "Select a table for exporting to Quattro Pro file"
  To TableName
  
  IF RetVal = FALSE THEN
     RETURN
  ELSE
     IF TableName = "None" THEN
       MESSAGE "No table found in current directory"
       SLEEP (2000)
       MESSAGE ""
       CLEAR
       RETURN
     ELSE  
       CLEAR
       IF myOpenTable(TableName) = False then
	  CLEAR
	  RESET
	  RETURN
       ENDIF   
       @5,5 ?? "Please key in the output filename : "
       ACCEPT "A40"
       REQUIRED
       TO FileName
     ENDIF
  ENDIF

  CLEAR
  IF RetVal = FALSE THEN
     RETURN
  ELSE
     MESSAGE "Exporting the database table to a Quattro Pro file ..."
     MENU {Tools} {ExportImport} {Export} {Quattro} 
     DOWN ENTER
     SELECT TableName 
     SELECT FileName
     message "Exporting ....."
     CLEAR
     MESSAGE ""
     RETURN 
  ENDIF
  
ENDPROC

;*************************************************************************
; Export the database table to a file in DBase IV format
;*************************************************************************
PROC CLOSED ExportToDBase()

  CLEAR
  SHOWTABLES DIRECTORY() "Select a table for exporting to DBase IV file"
  To TableName
  
  IF RetVal = FALSE THEN
     RETURN
  ELSE
     IF TableName = "None" THEN
       MESSAGE "No table found in current directory"
       SLEEP (2000)
       MESSAGE ""
       CLEAR
       RETURN
     ELSE  
       CLEAR
       IF myOpenTable(TableName) = False then
	  CLEAR
	  RESET
	  RETURN
       ENDIF   
       @5,5 ?? "Please key in the output filename : "
       ACCEPT "A40"
       REQUIRED
       TO FileName
     ENDIF
  ENDIF

  CLEAR
  IF RetVal = FALSE THEN
     RETURN
  ELSE
     MESSAGE "Exporting the database table to a DBase IV file ..."
     MENU {Tools} {ExportImport} {Export} {DBASE} 
     DOWN DOWN ENTER
     SELECT TableName 
     SELECT FileName
     message "Exporting ....."
     CLEAR
     MESSAGE ""
     RETURN 
  ENDIF
  
ENDPROC

;*************************************************************************
;set printer port
;*************************************************************************
PROC CLOSED SetPrinterPort()

  CLEAR
  SHOWMENU
    "LPT1" : "Select LPT1 as the printer port",
    "LPT2" : "Select LPT2 as the printer port",
    "LPT3" : "Select LPT3 as the printer port",
    "COM1" : "Select COM1 as the printer port",
    "COM2" : "Select COM2 as the printer port",
    "AUX" : "Select AUX as the printer port"
    DEFAULT "LPT1"
  TO choice

  IF RetVal = FALSE THEN
     RETURN
  ELSE
     SETPRINTER choice   
     RETURN      
  ENDIF
  
ENDPROC

;*************************************************************************
;prompt to ask user to input the printer initial and reset string
;*************************************************************************
PROC CLOSED PrinterInit()

  CLEAR
  SHOWDIALOG "Printer Setup String" @5,5 HEIGHT 10 WIDTH 70
    @2,2 ?? "The Setup String : "
      ACCEPT @2,22 WIDTH 38
      "A35"
      TAG "PrintSetup"
      TO SetupStr

  PUSHBUTTON @5,20 WIDTH 10
    "OK"
    OK
    DEFAULT
    VALUE "Say Yes"
    TAG "OKTag"
    TO ButtonValue

  PUSHBUTTON @5,40 WIDTH 10
    "Cancel"
    CANCEL
    VALUE "Say No"
    TAG "CancelTag"
    TO ButtonValue
    
  ENDDIALOG 

  IF RetVal = FALSE THEN
     RETURN
  ELSE
     SetupThePrinterInitString(SetupStr)
     RETURN
  ENDIF

ENDPROC


;*************************************************************************
;assign the printer setup string 
;*************************************************************************
PROC CLOSED SetupThePrinterInitString(SetupStr)

  MENU {REPORT} {SetPrinter} {Override} {Setup} 
  Select SetupStr

  RETURN
    
ENDPROC

;*************************************************************************
;Setup the reset string for the printer
;*************************************************************************
PROC CLOSED SetupThePrinterResetString(ResetStr)

  MENU {REPORT} {SetPrinter} {Override} {Reset} 
  Select ResetStr

  RETURN

ENDPROC
;*************************************************************************
; Reset the printer 
;*************************************************************************
PROC CLOSED PrinterReset()

  CLEAR
  SHOWDIALOG "Printer Reset String" @5,5 HEIGHT 10 WIDTH 70
    @2,2 ?? "Please input the Reset String : "
      ACCEPT @2,36 WIDTH 24
      "A20"
      TAG "PrintReset"
      TO ResetStr

  PUSHBUTTON @5,20 WIDTH 10
    "OK"
    OK
    DEFAULT
    VALUE "Say Yes"
    TAG "OKTag"
    TO ButtonValue

  PUSHBUTTON @5,40 WIDTH 10
    "Cancel"
    CANCEL
    VALUE "Say No"
    TAG "CancelTag"
    TO ButtonValue

  ENDDIALOG 

  IF RetVal = FALSE THEN
     RETURN
  ELSE
     SetupThePrinterResetString(ResetStr)
     RETURN
  ENDIF
  
ENDPROC

;**********************************************************************
; Ask the user to decide whether they want to keep the deleted audit
; records in a file, audit.dmp?
;**********************************************************************
PROC CLOSED AskToDumpFile()
 
  SHOWPOPUP " Do you want to save the deleted record to a file ? " CENTERED
    "Yes" : "Deleted audit records are saved to AUDIT.DMP" : "Yes",
    "No" : "The deleted records are not saved" : "No"
  ENDMENU
  UNTIL "ESC" KEYTO KeyVar
  TO MenuVar
   
  IF KeyVar = FALSE THEN
    RETURN MenuVar
  ELSE
    RETURN "Cancel"
  ENDIF
  
ENDPROC

;**********************************************************************
; This procedure deletes records from the audit.db based on the
; time interval the user specify.
;**********************************************************************
PROC CLOSED DeleteAuditRecord()
    
  CLEARALL
  CLEAR
  
  DYNARRAY Msg[]

  IF myOpenTable("Audit") = FALSE then
    message "Unable to open Audit table"
    sleep(2000)
    message ""
    reset
    return
  ENDIF

  i = 0
  ;get the time interval to delete record.
  Msg[1] = ""
  Msg[2] = "Trimming AUDIT.DB"
  Msg[3] = ""
  Msg[4] = "By entering a starting and ending date, records"
  Msg[5] = "for every event during this period will be marked"
  Msg[6] = "for deletion and will be deleted later."

  handle = CreateMsgWindow(Msg)
  sleep(3500)
  CloseMsgWindow(Handle)
  
  IF GetDateRange() = TRUE THEN
   
    @3,5 ?? "Opening the audit table .." 

    COEDIT "Audit"

    ;marked all the records within the timeframe as not needed
    ;the last record must be kept untouch for the server, else
    ;the server will not able to create new record.
    rCount = (NRECORDS("AUDIT") - 1)
    sleep(2000)

    j = 0
    FOR i FROM 1 TO rCount

	 @5,5 ?? "Scanning record ", i
	 MOVETO RECORD i
	 IF (([DATE] >= StartD) AND ([DATE] <= EndD)) THEN
	   MOVETo [Posted]
	   [Posted] = 1
	   MOVETO [DATE]
	   j = j + 1
	   @7,5 ?? j, " records have been marked."
	 ENDIF

    ENDFOR
    DO_IT!

    CLEAR
    IF j = 0 THEN
      @5,5 ?? "No record are found in the time specify,"
      @6,5 ?? "none is marked deleted."
      sleep(2000)
      CLEAR
      reset
      return
    ELSE  
      ;confirm from the user for the deletion
      SHOWDIALOG "Confirmation" @ 6,8 HEIGHT 10 WIDTH 64
	@ 3,3 ?? FORMAT("AC", "Are you sure you want to delete the marked events ?")
	
	PUSHBUTTON @ 5,10 WIDTH 10
	  "~O~K"
	  OK
	  DEFAULT
	  Value "Yes"
	  Tag "OKTag"
	  To ButtonValue
	  
	PUSHBUTTON @ 5,30 WIDTH 10
	  "~C~ANCEL"
	  CANCEL
	  Value "No"
	  Tag "CancelTag"
	  To ButtonValue

      ENDDIALOG    
	
      IF RetVal = TRUE THEN
	IF ButtonValue = "Yes" THEN
	  i = 0
	  @ 3,5 ?? "Deleting the marked events .."
	  COEDIT("AUDIT")
	  askDump = AskToDumpFile()
	  IF askDump = "Yes" THEN
	     mystring = "Date,Time,Duration,Mode,Type,Status,Pages,To,From,Tag,CSID,Errors,Retries,Board,MBox,Posted,ID,Noise,Speed,Quality,Strength,GFaxStat\n" 
	     FILEWRITE "audit.dmp" FROM mystring
	  ENDIF   
	  SCAN FOR [Posted] = 1
	    IF askDump = "Yes" THEN
	       mystring = ""
	       mystring = STRVAL([DATE])+","+[TIME]+","+[Duration]+","+[Mode]+","+[Type]+","+STRVAL([STATUS])+","+STRVAL([Pages])+","+[To]+","+[From]+","
	       mystring = mystring+[Phone]+","+[Tag]+","+[CSID]+","+STRVAL([Errors])+","+ STRVAL([Retries])+","+STRVAL([Board])+","+STRVAL([MBox])+","
	       mystring = mystring+STRVAL([Posted])+","+[ID]+","+STRVAL([Noise])+","+STRVAL([Speed])+","+STRVAL([Quality])+","+STRVAL([Strength])+","+STRVAL([GFaxStat])+"\n" 
	       FILEWRITE APPEND "audit.dmp" FROM mystring
	    ENDIF   
	    DEL
	    UP
	    i = i + 1
	    @10,3 ?? i, " records have been deleted."
	  ENDSCAN    
	  DO_IT!

	  CLEAR
	  reset
	  return

	ENDIF
      ENDIF

      @3,5 ?? "Undo the marking .."
      ;unmarked the events
      COEDIT("AUDIT")
      i = 1
      SCAN FOR [Posted] = 1 
	[Posted] = 0
	@ 5,5 ?? "Scanning ", i
	i = i + 1
      ENDSCAN  
      DO_IT!

      CLEAR
      reset
      return

    ENDIF
  
  ENDIF

ENDPROC     

;**********************************************************************
; Print a statistics graph (average pages of faxes sent per day). 
;**********************************************************************
PROC QueryPagePerDay()

  CLEAR
  CLEARALL
  
  IF myOpenTable("Audit") = FALSE THEN
    message "Failed to open Audit table"
    sleep(2000)
    message ""
    reset
    return
  Else
    IF isEncrypted("Audit") = TRUE Then
	   IF OpenEncryptedTable("Audit") = "N/A" THEN
	      message "Unable to open the encrypted table"
	      sleep(2000)
	      message ""
	      reset
	      return
	   ENDIF
    ENDIF      
  ENDIF           

  IF NOT isFile("bar.g") then
    message "Can't find graphic template bar.g"
    sleep(2000)
    message ""
    reset
    return
  ENDIF

  ;get the time frame to do the statistics graph
  IF GetDateRange() = TRUE THEN

    SHOWPOPUP "Output" CENTERED
      "Screen" : "Show on the screen" : "S",
      "Printer" : "Print to printer" : "P"
    ENDMENU  

   TO MenuVar
      
   IF RetVal = FALSE THEN
      message "User cancelled operation"
      sleep(2000)
      message ""
      reset
      return
   ENDIF

   @5,5 ?? "Processing the query right now ....."
       
      ;do the query on the table
      Query

	      Audit |           Date           |    Pages     | 
		    | Check >=~StartD, <=~EndD | calc average | 

	      Audit | 
		    | 

      Endquery
      DO_IT!

      If not isEmpty("Answer") then
	 MENU {IMAGE} {GRAPH} {LOAD} Select "Bar" 
	 IF MenuVar = "S" then
	   View "Answer"
	   MOVETO [Average of Pages]
	   ECHO NORMAL
	   GRAPHKEY
	   ECHO OFF
	 ELSE
	   VIEW "Answer"
	   MOVETO [Average of Pages]
	   MENU {IMAGE} {GRAPH} {VIEWGRAPH} {PRINTER}
	   @5,5 ?? "Printing the graph now ....."
	 ENDIF    
      else
	 message "No record meets the query criteria"
	 sleep(2000)
	 message ""
	 CLEAR
	 reset
	 return
      endif  

    ENDIF        

    CLEARALL
    CLEAR
    reset
    return
    
ENDPROC

;*************************************************************************
; Make a statistics calculation on the number of pages sent out per day
; by a user during a period of time
;*************************************************************************
Proc QueryPagePerUser()

  CLEAR
  CLEARALL

  IF myOpenTable("Audit") = False then
     message "Unable to open Audit table"
     sleep(2000)
     message ""
     reset
     return
  ELSE
     ;the users.db and audit.db have to be in the same directory to do
     ;the query
     IF NOT isTable("Users") then
	message "Can't find the USERS table in the current directory"
	sleep(2000)
	message ""
	reset
	return
     ELSE   
	IF isEncrypted("Audit") = TRUE Then
	   IF OpenEncryptedTable("Audit") = "N/A" THEN
	      message "Unable to open the encrypted table"
	      sleep(2000)
	      message ""
	      reset
	      return
	   ENDIF
	ENDIF      
	IF isEncrypted("Users") = TRUE Then
	   IF OpenEncryptedTable("Users") = "N/A" THEN
	      message "Unable to open the encrypted table"
	      sleep(2000)
	      message ""
	      reset
	      return
	   ENDIF
	ENDIF      
     ENDIF
  ENDIF   


  IF NOT isfile("pieuser.g") then
    message "Can't find graphic template pieuser.g in the current directory"
    sleep(2000)
    message ""
    reset
    return
  Endif
 
  IF GetDateRange() = TRUE THEN
  
    SHOWPOPUP "Output" CENTERED
      "Screen" : "Show on the screen" : "S",
      "Printer" : "Print to printer" : "P"
    ENDMENU  

   TO MenuVar
      
   IF RetVal = FALSE THEN
      message "User cancelled operation"
      sleep(2000)
      message ""
      reset
      return
   ENDIF
	 
   @5,5 ?? "Processing the query the table right now ....."
    
   Query

       Audit |       Date           |   Pages   | 
	     | >=~StartD, <=~EndD   | calc sum  | 

       Audit |  MBox   | 
	     | _sender | 

       Users |  Name  |  MBox   | 
	     | Check  | _sender | 

    Endquery

    DO_IT!

    IF NOT isEmpty("Answer") then
       MENU {IMAGE} {GRAPH} {LOAD} Select "PieUser" 
       IF MenuVar = "S" then 
	 ECHO NORMAL
	 View "Answer"
	 MOVETO [Sum of Pages]
	 GRAPHKEY
	 ECHO OFF
       ELSE
	 ECHO NORMAL
	 View "Answer"
	 MOVETO [Sum of Pages]
	 MENU {IMAGE} {GRAPH} {VIEWGRAPH} {PRINTER}  
       ENDIF  
    ELSE
       message "No record meets the query criteria"
       sleep(2000)
       message ""
       CLEAR
       RESET
       return
    ENDIF   

  ENDIF

  CLEARALL
  CLEAR

ENDPROC

PROC CLOSED myIsFile(filename)

  IF NOT isfile(fileName) then
    message "Can't find file ", filename, " in the current directory"
    sleep(2000)
    message ""
    reset
    return FALSE
  ELSE
    return TRUE
  Endif

ENDPROC

PROC CLOSED DEMOINIT()

  CLEAR
  CLEARALL

  IF isTable("Audit") = False then
     message "Unable to open Audit table"
     sleep(2000)
     message ""
     reset
     return
  ELSE
     ;the users.db and audit.db have to be in the same directory to do
     ;the query
     IF NOT isTable("Users") then
	message "Can't find the USERS table in the current directory"
	sleep(2000)
	message ""
	reset
	return
     ELSE   
	IF isEncrypted("Audit") = TRUE Then
	   IF OpenEncryptedTable("Audit") = "N/A" THEN
	      message "Unable to open the encrypted table"
	      sleep(2000)
	      message ""
	      reset
	      return
	   ENDIF
	ENDIF      
	IF isEncrypted("Users") = TRUE Then
	   IF OpenEncryptedTable("Users") = "N/A" THEN
	      message "Unable to open the encrypted table"
	      sleep(2000)
	      message ""
	      reset
	      return
	   ENDIF
	ENDIF      
     ENDIF
  ENDIF   

  IF NOT (Myisfile("qbar.g") OR MyIsfile("qpie.g")) then
     return FALSE
  ENDIF
  
  IF NOT (Myisfile("qbar.sc") OR MyisFile("Quser.sc") OR MyisFile("Qsrpie.sc")) then
     return FALSE
  ENDIF

  MENU {TOOLS} {NET} {AUTOREFRESH} select 10
  RETURN TRUE

ENDPROC
	
PROC CLOSED DEMO()

  CLEAR
  CLEARALL
  
  IF DEMOINIT() THEN      

     SHOWPOPUP "Which Demo ? " CENTERED
       "User/Page Bar Chart " : "" : "qu",
       "Type/Page Bar Chart " : "" : "qb",
       "Pie Chart of Send/Receive" : "" : "qp"
     ENDMENU
     
     TO MenuVar
     
     IF RetVal THEN
 
	switch 
	 
	   case MenuVar = "qu" :
	   
	     MENU {IMAGE} {GRAPH} {LOAD} Select "qBAR"
	     ;MENU {Scripts} {RepeatPlay} select "quser" select "c" 
	     while NOT CHARWAITING() 
	       PLAY "quser"
	     endwhile  
	     reset
	     return
	   
	   case MenuVar = "qb" :  
	     MENU {IMAGE} {GRAPH} {LOAD} Select "qBAR"
	     ;MENU {Scripts} {RepeatPlay} select "qbar" select "c" 
	     response = " "
	     while NOT CHARWAITING()
	       PLAY "qbar"
	     endwhile  
	     reset
	     return
	     
	   case MenuVar = "qp" : 
	     MENU {IMAGE} {GRAPH} {LOAD} Select "qpie"
	     ;MENU {Scripts} {RepeatPlay} select "qsrpie" select "c
	     while NOT CHARWAITING()
	       PLAY "qsrpie"
	     endwhile  
	     reset
	     return
       endswitch
       
     ELSE          ;user cancel the choice of printing
       message "User cancelled operation"
       sleep(2000)
       message ""
       reset
       return
     ENDIF    
       
  ELSE
    message "Monitor initialization failed."
    sleep(2000)
    message ""
    REset
    return
  ENDIF

ENDPROC

;*************************************************************************
; This procedure display a menu bar at top of the screen and let
; the use choose whatever functionality she/he wants
;*************************************************************************
PROC MENUBAR()

  CLEAR

  WHILE TRUE
    SHOWPULLDOWN
      
      "Upgrade" : "Upgrade previous version of database table to 2.1 GL's " : "Upgrade"
 
	 SUBMENU
	    "Users" : "Upgrade users.db" : "UpgradeU",
	    "Phone" : "Upgrade phone.db" : "UpgradeP",
	    "Audit" : "Upgrade audit.db" : "UpgradeA",
	    "A~l~l"   : "Upgrade all three's" : "UpgradeAll"
	 ENDSUBMENU,
		 
      "Protection" : "Encrypted the database table with password" : "SetPassword"

	 SUBMENU
	    "Users" : "Encrypted users.db" : "SetPassU",
	    "Phone" : "Encrypted phone.db" : "SetPassP",
	    "Audit" : "Encrypted audit.db" : "SetPassA",
	    "Group" : "Encrypted group.db" : "SetPassG"
	 ENDSUBMENU,

      "Report" : "Print reports to the printer" : "Report"
	 
	 SUBMENU
	   "PHONE" : "Print the phone book of a user" : "RPhone",           
	   "USERS" : "Print the whole users list" : "RUsers",           
	   "AUDIT" : "Print the audit trail" : "RAudit",
	   "GROUP" : "Print a group in the phone book" : "RGroup"
	 ENDSUBMENU,  
      
      "Export" : "Export the database table to ASCII file" : "ExportFile"
	  
	  SUBMENU
	    "ASCII" : "Export as ASCII file" : "ExportASCII",
	    "Lotus 123" : "Export as Lotus 1-2-3 rel. 2 file" : "ExportLotus",
	    "Quattro Pro" : "Export as Quattro Pro file" : "ExportQuattro",
	    "DBASE IV" : "Export as DBASE IV file" : "ExportDbase"
	  ENDSUBMENU,  

      ;"Printer" : "Set printer port, setup and reset string" : "SetPrinter"         

	; SUBMENU
	;   "Port" : "Select the printer port" : "SetPort",
	;   "SetupString" : "Set the setup string for printer" : "SStr",
	;   "ResetString" : "Set the reset string for printer" : "RStr"
	; ENDSUBMENU,           
      
      "Maintenance" : "Delete out of date entries in audit.db" : "Maintenance",
      
      "Statistics" : "Print statistic information from audit.db" : "Statistics"

	 SUBMENU
	    "PagesPerDay" : "Average pages sent or received per day" : "APage",
	    "PagePer~U~ser" : "Sum of pages sent or received per user" : "SPage",
	    "Demo" : "Real time demostrastraction" : "DPage"
	 ENDSUBMENU,

      "E~X~IT" : "Exit this application" : "Exit"
	 
	 SUBMENU
	    "Yes" : "Exit this application"         : "Exit/Yes",
	    "No"  : "Do not exit this application"  : "Exit/No"
	 ENDSUBMENU

     ENDMENU UNTIL 17               ;CTRL-Q to Quit

     GETMENUSELECTION KEYTO KeyVar TO MenuItemSelected
     IF KeyVar = 17 THEN
	QUITLOOP
	EXIT     
     ENDIF

     IF Retval THEN
       SWITCH
	 CASE (MenuItemSelected = "UpgradeU") :
	   UpGradeUser()
	 CASE (MenuItemSelected = "UpgradeP") :
	   UpgradePhone()
	 CASE (MenuItemSelected = "UpgradeA") :
	   UpgradeAudit()
	 CASE (MenuItemSelected = "UpgradeAll") :
	   UpGradeUser()
	   UpGradePhone()
	   UpGradeAudit()
	 CASE (MenuItemSelected = "SetPassU") :
	   PassProtectTable("Users")
	 CASE (MenuItemSelected = "SetPassP") :
	   PassProtectTable("Phone")
	 CASE (MenuItemSelected = "SetPassA") :          
	   PassProtectTable("Audit") 
	 CASE (MenuItemSelected = "SetPassG") :
	   PassProtectTable("Group")          
	 CASE (MenuItemSelected = "RPhone") :
	   PrintPhone()
	 CASE (MenuItemSelected = "RUsers") :
	   PrintUsers()
	 CASE (MenuItemSelected = "RAudit") :
	   PrintAudit()
	 CASE (MenuItemSelected = "RGroup") :
	   PrintGroup()
	 CASE (MenuItemSelected = "ExportASCII") :
	   ExportToASCII()
	 CASE (MenuItemSelected = "ExportLotus") :
	   ExportToLotus123()  
	 CASE (MenuItemSelected = "ExportQuattro") :
	   ExportToQuattroPro()
	 CASE (MenuItemSelected = "ExportDbase") :
	   ExportToDBASE()    
	 CASE (MenuItemSelected = "SetPort") :
	   SetPrinterPort()
	 CASE (MenuItemSelected = "SStr") :
	   PrinterInit()
	 CASE (MenuItemSelected = "RStr") :  
	   PrinterReset()
	 CASE (MenuItemSelected = "Maintenance") :
	   DeleteAuditRecord()
	   reset
	 CASE (MenuItemSelected = "APage") :
	   QueryPagePerDay()
	 CASE (MenuItemSelected = "SPage") :
	   QueryPagePerUser()
	 CASE (MenuItemSelected = "DPage") :
	   DEMO()  
	 CASE (MenuItemSelected = "Exit/Yes"):     
	   CLEARPULLDOWN
	   EXIT
	 CASE (MenuItemSelected = "Exit/No"):     
	   Dummy()
       ENDSWITCH      
   
     ENDIF
   
   ENDWHILE
   
   CLEAR

ENDPROC


PROC Dummy()
ENDPROC

WelcomeMsg()
sleep(2000)
MenuBar()

RESET
RELEASE VARS ALL
RELEASE PROCS ALL
