PROGRAM FileDescEditor;
{$A+,B-,D-,E-,F-,G+,I-,L+,N-,O-,R-,S-,V-,X-}
{$M 8192,0,0}

(* ----------------------------------------------------------------------
   A Simple 4DOS File Description Editor

   (c) 1992 Copyright by David Frey,         & Tom Bowden
                         Urdorferstrasse 30    1575 Canberra Drive
                         8952 Schlieren ZH     Stone Mountain, GA 30088-3629
                         Switzerland           USA

       Code created using Turbo Pascal 6.0 (c) Borland International 1990

   DISCLAIMER: This program is freeware: you are allowed to use, copy
               and change it free of charge, but you may not sell or hire
               4DESC. The copyright remains in our hands.

               If you make any (considerable) changes to the source code,
               please let us know. (send a copy or a listing).
               We would like to see what you have done.

               We, David Frey and Tom Bowden, the authors, provide absolutely
               no warranty of any kind. The user of this software takes the
               entire risk of damages, failures, data losses or other
               incidents.

   NOTES:      4DESC was modified extensively by Tom Bowden,
               August-October 1992.

               Among the changes:

               Screen layout now resembles the 4DOS SELECT screen.
               (The original screen had apparently been based on Larry
               Edwards' 4FILES).

               The display now is sorted.

               The program now is always in edit mode.

               Alt-T (cuT to buffer) now is Alt-M (Move to buffer).
               Alt-D now deletes a file description.
               Alt-X now exits the program.

               F1 now displays a help screen.
               F2 now changes drive.
               F3 now changes to the highlighted directory.
               F4 now changes to the parent directory.
               F10 now saves the current file descriptions.

               The screen colors were changed, and stored as CONST for
               easier maintenance.  VGA is no longer required.

               4DESC can now write and display file descriptions for
               directory entries.

               SaveDescriptions now strips trailing spaces from
               file extensions and leading and trailing spaces from
               file descriptions.

               4DESC does not presently write file descriptions longer
               than 40 characters.  When saving, any longer descriptions
               in the current directory will be truncated.  The user is
               warned when reading a directory having extended file
               descriptions.

               Handling of extended program information has not been tested.

   ADDITIONS TO TOM BOWDENS'S IMPROVEMENTS BY DAVID FREY:

||             I have split 4DESC.PAS into units:
||              StringDateHandling, DisplayKeyboardAndCursor, HandleINIFile
||
               Monochrome / Color display detection. /mono switch.

               Insert mode cursor is underline, overwrite is block cursor.
               (as in 4DOS)

               Since 4DOS 4.01 has introduced the `DescriptionMax' statement;
               references to fix description lengths have been removed.

               4DESC is now international: it chooses the appropriate date
               and time formats on startup. [by using DOS's function $38:
               Get/Set Country Data. DOS get the country information via
               COUNTRY= and COUNTRY.SYS].

               4TOOLS.INI file introduced. Colors and Time/Date formats can
               now be changed without recompiling 4DESC.PAS (for people
               without Turbo Pascal). 4DESC checks its startup directory,
               environment variable 4TOOLS and PATH to locate 4TOOLS.INI.

               "Change drive" will not change to drives which are not ready.

   A few new tweaks by Tom Bowden:

               "Change drive" will not change to drives which contain
               no files.

               New handling of command line parameters.  The /mono, /help,
               and selected directory params may be used together.  Note
               that the optional selected directory must be the last
               parameter entered.

               The status line now displays the 4DOS version (if running
               under 4DOS), and shows "Edit" and Cut" rather than "*"
               and "()".

    More additions by David Frey:

               Maximum number of files in a directory raised to 417
               descriptions. A warning ("Description file will be truncated")
               will appear if more than MaxDesc files are stored in a
               directory going to be edited with 4DESC.
               This prevents unintentional cutting of your description file.

               Yet another function key binding:
                F3 : View file (with list - whatever list may be
                                (internal 4DOS, external viewer))
                F4 : Change Dir
                F5 : Change to parent
                F6 : Change drive

               In 4TOOLS.INI the LeftJust variable has been added.

               Shelling out to 4DOS has been added (Alt-S or Shift-F10)

    More additions by Tom Bowden:

               In 4TOOLS.INI the FullSize and Viewer variables have
               been added.
               Get4DOSVer has been modified to display the correct
               minor version number and to check for NDOS.

   ----------------------------------------------------------------------- *)


USES Crt, Dos, StringDateHandling, DisplayKeyboardAndCursor, HandleINIFile;

CONST MaxDescLen = 42;

TYPE  NameExtStr = STRING[8+1+3];
      DescStr    = STRING[MaxDescLen];
      TFileData  = RECORD
                     DirSort  : CHAR;
                     Name     : NameExtStr;
                     Size     : STRING[8];
                     Date     : STRING[15];
                     ProgInfo : STRING[64]; (* I hope 64 characters are enough *)
                     Desc     : DescStr;
                   END;

CONST MaxDesc     = 61400 DIV SizeOf(TFileData);
      DirSize     = '  <DIR> ';

VAR   Description : ARRAY[1..MaxDesc] OF TFileData;
      NrOfFiles   : WORD;
      EdStart     : BYTE;
      DescLen     : BYTE;

      ActDir      : DirStr;
      StartDir    : DirStr;

      StartIndex  : BYTE;
      Index       : INTEGER;

      CutPasteDesc: DescStr;
      Changed     : BOOLEAN;
      IORes       : INTEGER;

      NewDir      : DirStr;
      NewName     : NameStr;
      NewExt      : ExtStr;

      FirstParam  : STRING[2];
      i           : BYTE;
      DoShowHelp  : BOOLEAN;

(*-------------------------------------------------------- Display-Routines *)
PROCEDURE WriteFileEntry(Index: INTEGER; Hilighted: BOOLEAN);

BEGIN
 GotoXY(1,2+Index-StartIndex);
 IF  Index <= NrOfFiles THEN
  WITH Description[Index] DO
   BEGIN
    IF Hilighted THEN
     BEGIN TextColor(SelectFg); TextBackGround(SelectBg); END
    ELSE
     BEGIN
      TextBackGround(NormBg);
      IF Size <> DirSize THEN TextColor(NormFg)
                         ELSE TextColor(DirFg)
     END;
    Write(' ',Name,Size,' ',Date,'  ');
    GotoXY(EdStart,2+Index-StartIndex); Write(Desc);
    IF Length(Desc) < DescLen THEN ClrEol;
   END
 ELSE ClrEol;
END;  (* WriteFileEntry *)

PROCEDURE DrawDirLine;

BEGIN
 {$I-}
 GetDir(0,ActDir);
 IF ActDir[Length(ActDir)] <> '\' THEN ActDir := ActDir + '\';
 UpString(ActDir);
 TextColor(DirFg); TextBackGround(NormBg);
 GotoXY(1,2); Write(' ',ActDir); ClrEol;
END; (* DrawDirLine *)

PROCEDURE ReDrawScreen;

VAR Index: INTEGER;

BEGIN
 {$I-}
 GetDir(0,ActDir);
 FOR Index := StartIndex+1 TO StartIndex+MaxLines-3 DO
  WriteFileEntry(Index,FALSE);
END; (* ReDrawScreen *)


(*-------------------------------------------------------- Sort-Directory *)
PROCEDURE SortArray;     (* Straight selection sort algorithm by N. Wirth *)

VAR i, j, k   : INTEGER;
    TempDesc : TFileData;

BEGIN
 FOR i := 1 TO NrOfFiles-1 DO
  BEGIN
   k := i;
   TempDesc := Description[i];
   FOR j := i+1 TO NrOfFiles DO
    IF Description[j].DirSort+Description[j].Name < TempDesc.DirSort+TempDesc.Name THEN
     BEGIN
      k := j;
      TempDesc := Description[j];
     END;
   Description[k] := Description[i];
   Description[i] := TempDesc;
  END;
END;  (* SortArray *)

(*-------------------------------------------------------- Read-Directory *)
PROCEDURE ReadFiles;

VAR Search         : SearchRec;
    Dir            : DirStr;
    BaseName       : NameStr;
    Ext            : ExtStr;
    Time           : DateTime;

    DescFileExists : BOOLEAN;
    DescFound      : BOOLEAN;
    DescLong       : BOOLEAN;
    DescFile       : TEXT;
    DescLine       : STRING;
    DescName       : NameExtStr;
    DescStart      : BYTE;
    DescEnd        : BYTE;
    i              : BYTE;
    ch             : WORD;

BEGIN
 NrOfFiles  := 0;
 Changed    := FALSE; DescLong   := FALSE;
 Index      := 1; StartIndex := 0;

 FindFirst('DESCRIPT.ION',Hidden + Archive,Search);
 DescFileExists := (DosError = 0);
 {$I-}
 IF DescFileExists THEN Assign(DescFile,'DESCRIPT.ION');

 FindFirst('*.*',ReadOnly+Hidden+Archive+Directory, Search);
 WHILE (DosError = 0) AND (NrOfFiles < MaxDesc) DO
  BEGIN
   DownString(Search.Name);
   INC(NrOfFiles);
   WITH Description[NrOfFiles] DO
    BEGIN
     UnpackTime(Search.Time,Time);
     Date     := FormDate(Time)+' '+FormTime(Time);
     ProgInfo := '';
     Desc     := '';

     IF Search.Attr AND Directory = Directory THEN
      BEGIN
       Name := UpStr(Search.Name);
       Size := DirSize;
       DirSort := '0';  (* Force directories ahead of files in sorted display. *)
      END
     ELSE
      BEGIN
       FSplit(Search.Name,Dir,Basename,Ext);
       IF NoJust   THEN Name := BaseName+Ext
                   ELSE Name := BaseName+Chars(' ',8-Length(BaseName))+Ext;
       IF FullSize THEN Str(Search.Size:8,Size)
                   ELSE Size := FormattedLongIntStr(Search.Size DIV 1024,7)+'K';
       DirSort := '1';
      END; (* if ... and directory ... else *)
     WHILE Length(Name) < 12 DO Name := Name+' ';

     IF DescFileExists THEN
      BEGIN
       Reset(DescFile);
       REPEAT
        ReadLn(DescFile,DescLine);
        DescStart := Pos(' ',DescLine);
        IF DescStart = 0 THEN DescStart := Length(DescLine)+1;
        DescName  := Copy(DescLine,1,DescStart-1);
        DownString(DescName);

        DescFound := (DescName = DownStr(Search.Name));
       UNTIL DescFound OR Eof(DescFile);

       IF DescFound THEN
        BEGIN
         DescEnd := Pos(#4,DescLine);
         IF DescEnd = 0 THEN DescEnd := Length(DescLine)+1;
         IF (DescEnd-1) - (DescStart+1) > DescLen THEN DescLong := TRUE;
         ProgInfo:= Copy(DescLine,DescEnd,255);
         Desc    := Copy(DescLine,DescStart+1,DescEnd-1);
         StripLeadingSpaces(Desc);
         StripTrailingSpaces(Desc);
        END;
      END; (* if DescFileExists *)
    END; (* with Description do *)
   FindNext(Search);
  END; (* while *)

 IF NrOfFiles = MaxDesc THEN
  BEGIN
   TextColor(NormFg); TextBackGround(NormBg);
   FOR i := 3 TO MaxLines-1 DO
    BEGIN
     GotoXY(1,i); ClrEol;
    END;
   ReportError('Warning! Too many files in directory, description file will be truncated! (Key)',(CutPasteDesc <> ''),Changed);
  END;

 {$I-}
 IF DescFileExists THEN Close(DescFile);

 IF NrOfFiles > 1 THEN SortArray;

 IF NrOfFiles > 0 THEN
  BEGIN
   DrawMainScreen(Index,NrOfFiles);
   DrawDirLine;
  END;

 IF DescLong THEN
  BEGIN
   TextColor(NormFg); TextBackGround(NormBg);
   FOR i := 3 TO MaxLines-1 DO
    BEGIN
     GotoXY(1,i); ClrEol;
    END;
   ReportError('Warning! Some descriptions are too long; they will be truncated. Press any key.',(CutPasteDesc <> ''),Changed);
  END;
END;  (* ReadFiles *)

(*-------------------------------------------------------- Save Descriptions *)
PROCEDURE SaveDescriptions;

VAR DescFile : TEXT;
    DescSaved: BOOLEAN;
    Dir      : DirStr;
    BaseName : NameStr;
    Ext      : ExtStr;
    Time     : DateTime;
    i        : INTEGER;
    ch       : WORD;

BEGIN
 DescSaved := FALSE;
 IF DiskFree(0) < NrOfFiles*SizeOf(TFileData) THEN
  ReportError('Probably out of disk space. Nevertheless trying to save DESCRIPT.ION...',(CutPasteDesc <> ''),Changed);

 {$I-}
 Assign(DescFile,'DESCRIPT.ION');
 SetFAttr(DescFile,Archive);
 Rewrite(DescFile);
 IF IOResult > 0 THEN ReportError('Unable to write DESCRIPT.ION !',(CutPasteDesc <> ''),Changed)
 ELSE
  BEGIN
   FOR i := 1 TO NrOfFiles DO
    WITH Description[i] DO
     IF Desc <> '' THEN
      BEGIN
       FSplit(Name,Dir,Basename,Ext);
       StripTrailingSpaces(BaseName);
       Write(DescFile,BaseName);

       StripLeadingSpaces(Ext);
       StripTrailingSpaces(Ext);
       IF Ext <> '' THEN Write(DescFile,Ext);

       StripLeadingSpaces(Desc);
       StripTrailingSpaces(Desc);
       Write(DescFile,' ',Desc);
       IF ProgInfo <> '' THEN Write(DescFile,#4,ProgInfo);
       WriteLn(DescFile);
       DescSaved := TRUE;
      END;
   {$I-}
   Close(DescFile);
   IF IOResult > 0 THEN ReportError('Unable to write DESCRIPT.ION !',(CutPasteDesc <> ''),Changed)
   ELSE
    BEGIN
     IF DescSaved THEN SetFAttr(DescFile, Archive + Hidden)
                  ELSE Erase(DescFile);  (* Don't keep zero-byte file. *)
     Changed := FALSE;
     DrawStatusLine(FALSE,(CutPasteDesc <> ''),Changed);
    END;
  END;  (* If IOResult > 0 then ... else begin *)
END;  (* SaveDescriptions *)

(*-------------------------------------------------------- Edit Descriptions *)
PROCEDURE EditDescriptions;

CONST kbLeft     = $4B00;   kbRight    = $4D00;
      kbUp       = $4800;   kbDown     = $5000;
      kbHome     = $4700;   kbEnd      = $4F00;
      kbPgUp     = $4900;   kbPgDn     = $5100;
      kbCtrlPgUp = $8400;   kbCtrlPgDn = $7600;
      kbCtrlHome = $7700;   kbCtrlEnd  = $7500;
      kbEnter    = $0D;     kbEsc      = $1B;

      kbIns      = $5200;   kbDel      = $5300;
      kbBack     = $08;

      kbGrayMinus= $4A2D;   kbGrayPlus = $4E2B;

      kbAltC     = $2E00;   kbAltP     = $1900;
      kbAltD     = $2000;   kbAltL     = $2600;
      kbAltM     = $3200;   kbAltT     = $1400;
      kbAltS     = $1F00;   kbAltV     = $2F00;
      kbAltX     = $2D00;

      kbF1       = $3B00;   kbF2       = $3C00;
      kbF3       = $3D00;   kbF4       = $3E00;
      kbF5       = $3F00;   kbF6       = $4000;
      kbF10      = $4400;   kbShiftF10 = $5D00;

VAR Key          : WORD;
    Drv          : STRING[3];
    LastDrv      : CHAR;
    x,y          : BYTE;
    EditStr      : DescStr;
    Overwrite    : BOOLEAN;
    Cursor       : WORD;
    OldDir       : DirStr;

 PROCEDURE UpdateLineNum(Index: INTEGER);

 BEGIN
  WriteFileEntry(Index,TRUE);
  TextColor(StatusFg); TextBackGround(StatusBg);
  GotoXY(70,1); Write(Index:3);

  IF Changed THEN DrawStatusLine(FALSE,(CutPasteDesc <> ''),Changed);

  x := 1;
  y := 2+Index-StartIndex;
  GotoXY(EdStart,y);
  TextColor(SelectFg); TextBackGround(SelectBg);
  EditStr := Description[Index].Desc;
  Write(EditStr);
  IF Length(EditStr) < MaxDescLen THEN ClrEol;
  GotoXY(EdStart+x-1,y);
 END;

 PROCEDURE PrevIndex(VAR Index: INTEGER);

 BEGIN
  Index := Max(Index-1,1);
  IF Index <= StartIndex THEN
   BEGIN
    StartIndex := Max(Index-ScreenSize,0);
    RedrawScreen;
   END;
  UpdateLineNum(Index);
 END; (* NextIndex *)

 PROCEDURE NextIndex(VAR Index: INTEGER);

 BEGIN
  Index := Min(Index+1,NrOfFiles);
  IF Index > StartIndex+ScreenSize THEN
   BEGIN
    StartIndex := Index-ScreenSize;
    RedrawScreen;
   END;
  UpdateLineNum(Index);
 END; (* NextIndex *)

  PROCEDURE QuerySaveDescriptions;

  VAR ch: CHAR;

  BEGIN
   TextColor(StatusFg); TextBackGround(StatusBg);
   IF Changed THEN
    BEGIN
     REPEAT
      GotoXY(1,MaxLines);
      Write(Chars(' ',11));
      Write('Descriptions have been edited. Shall they be saved (Y/N) ?');
      ClrEol;
      ch := UpCase(ReadKey);
     UNTIL (ch = 'Y') OR (ch = 'N');
     IF ch = 'Y' THEN SaveDescriptions;
    END;
  END; (* QuerySaveDescriptions *)

 PROCEDURE DirUp;

 BEGIN
  IF Changed THEN QuerySaveDescriptions;
  {$I-}
  ChDir('..');
  IF IOResult = 0 THEN
   BEGIN
    ReadFiles;
    RedrawScreen;

    DrawStatusLine(FALSE,(CutPasteDesc <> ''),Changed);
    Index := 1; UpdateLineNum(Index);
   END;
 END;  (* DirUp *)

 PROCEDURE DirDown;

 BEGIN
  IF (Description[Index].Size = DirSize) AND
     (Description[Index].Name[1] <> '.') THEN
   BEGIN
    IF Changed THEN QuerySaveDescriptions;
    {$I-}
    ChDir(Description[Index].Name);
    IF IOResult = 0 THEN
     BEGIN
      ReadFiles;
      RedrawScreen;
     END;
    DrawStatusLine(FALSE,(CutPasteDesc <> ''),Changed);
    Index := 1;
    UpdateLineNum(Index);
  END;  (* IF Description[Index].Size = DirSize *)
 END;  (* DirDown *)

BEGIN  (* EditDescriptions *)
 Index := 1;
 UpdateLineNum(Index);
 Overwrite := FALSE;
 ResetCursor(Overwrite);
 EditStr := Description[Index].Desc;

 REPEAT
  Key := GetKey;
  CASE Key OF
   kbUp       : BEGIN
                 Description[Index].Desc := EditStr;
                 WriteFileEntry(Index,FALSE);
                 PrevIndex(Index);
                END; (* Up *)

   kbDown     : BEGIN
                 Description[Index].Desc := EditStr;
                 WriteFileEntry(Index,FALSE);
                 NextIndex(Index);
                END; (* Down *)

   kbLeft     : BEGIN
                 x := Max(1,x-1);
                 GotoXY(EdStart+x-1,y);
                END; (* Left *)

   kbRight    : BEGIN
                 IF (x <= Length(EditStr)) AND (x < MaxDescLen) THEN INC(x);
                 GotoXY(EdStart+x-1,y);
                END; (* Right *)

   kbHome     : BEGIN
                 x := 1;
                 GotoXY(EdStart+x-1,y);
                END; (* Home *)

   kbEnd      : BEGIN
                 x := Length(EditStr);
                 IF x < MaxDescLen THEN INC(x);
                 GotoXY(EdStart+x-1,y);
                 END; (* End *)

   kbCtrlEnd  : BEGIN
                 Delete(EditStr,x,MaxDescLen);
                 Description[Index].Desc := EditStr;

                 Changed := TRUE;
                 DrawStatusLine(FALSE,(CutPasteDesc <> ''),Changed);

                 WriteFileEntry(Index,TRUE);
                END;  (* ^End *)

   kbIns      : BEGIN
                 Overwrite := NOT Overwrite;
                 ResetCursor(Overwrite);
                END; (* Ins *)

   kbDel      : BEGIN
                 Delete(EditStr,x,1);
                 Description[Index].Desc := EditStr;

                 IF x > Length(EditStr) THEN x := Max(Length(EditStr),1);

                 Changed := TRUE;
                 DrawStatusLine(FALSE,(CutPasteDesc <> ''),Changed);

                 WriteFileEntry(Index,TRUE);
                 GotoXY(EdStart+x-1,y);
                END; (* Del *)

   kbBack     : BEGIN
                 Delete(EditStr,x-1,1);
                 Description[Index].Desc := EditStr;
                 IF x > 1 THEN
                  BEGIN
                   DEC(x);
                   IF x > Length(EditStr) THEN x := Length(EditStr)+1;
                  END;

                 Changed := TRUE;
                 DrawStatusLine(FALSE,(CutPasteDesc <> ''),Changed);

                 WriteFileEntry(Index,TRUE);
                 GotoXY(EdStart+x-1,y);
                END; (* Back *)

   kbPgUp     : BEGIN
                 Description[Index].Desc := EditStr;
                 WriteFileEntry(Index,FALSE);
                 Index := Max(Index-ScreenSize,1);
                 StartIndex := Index-1;
                 RedrawScreen;
                 UpdateLineNum(Index);
                END; (* PgUp *)

   kbPgDn     : BEGIN
                 Description[Index].Desc := EditStr;
                 WriteFileEntry(Index,FALSE);
                 Index := Min(Index+ScreenSize,NrOfFiles);
                 StartIndex := Max(Index-ScreenSize,0);
                 RedrawScreen;
                 UpdateLineNum(Index);
                END; (* PgDn *)

   kbCtrlPgUp : BEGIN
                 Description[Index].Desc := EditStr;
                 WriteFileEntry(Index,FALSE);
                 StartIndex := 0; Index := 1;
                 RedrawScreen;
                 UpdateLineNum(Index);
                END; (* ^PgUp *)

   kbCtrlPgDn : BEGIN
                 Description[Index].Desc := EditStr;
                 WriteFileEntry(Index,FALSE);
                 StartIndex := Max(NrOfFiles-ScreenSize,0);
                 Index := NrOfFiles;
                 RedrawScreen;
                 UpdateLineNum(Index);
                END; (* ^PgDn *)

   kbAltD     : BEGIN
                 Description[Index].Desc := '';

                 Changed := TRUE;
                 DrawStatusLine(FALSE,(CutPasteDesc <> ''),Changed);

                 WriteFileEntry(Index,FALSE);
                 NextIndex(Index);
                END; (* Alt-D *)

   kbAltM,
   kbAltT     : BEGIN
                 CutPasteDesc := Description[Index].Desc;
                 Description[Index].Desc := '';
                 EditStr := '';

                 Changed := TRUE;
                 DrawStatusLine(FALSE,(CutPasteDesc <> ''),Changed);
                 WriteFileEntry(Index,FALSE);
                 NextIndex(Index);
                END; (* Alt-M / Alt-T *)

   kbAltC     : BEGIN
                 CutPasteDesc := Description[Index].Desc;
                 DrawStatusLine(FALSE,(CutPasteDesc <> ''),Changed);
                 WriteFileEntry(Index,TRUE);
                END; (* Alt-C *)

   kbAltP     : IF CutPasteDesc > '' THEN
                 BEGIN
                  Description[Index].Desc := CutPasteDesc;

                  Changed := TRUE;
                  DrawStatusLine(FALSE,(CutPasteDesc <> ''),Changed);

                  WriteFileEntry(Index,FALSE);
                  NextIndex(Index);
                 END; (* Alt-P *)

   kbF1       : BEGIN                                   (* F1: Help *)
                 ShowHelp(MaxDesc);
                 ResetCursor(Overwrite);
                 DrawMainScreen(Index,NrOfFiles);
                 DrawDirLine;
                 RedrawScreen;
                 UpdateLineNum(Index);
                END;  (* F1 *)

   kbAltL,
   kbF6       : BEGIN                                   (* F6: Change Drive *)
                 IF Changed THEN QuerySaveDescriptions;

                 ASM
                  mov ah,0eh       (* Select Disk *)
                  mov dl,3
                  int 21h
                  add al,'@'
                  mov LastDrv,al
                 END;

                 (* Tom's solution has been commented out, Dave.
                    Reason: LastDrive has not been detected correctly. *)

                 IF LastDrv > 'Z' THEN LastDrv := 'Z';

                 TextColor(StatusFg); TextBackGround(StatusBg); Drv := ' :';
                 REPEAT
                  GotoXY(1,MaxLines);
                  Write(' New drive letter (A..',LastDrv,'): ');
                  ClrEol;
                  Drv[1] := UpCase(ReadKey);
                 UNTIL (Drv[1] >= 'A') AND (Drv[1] <= LastDrv);
                 IF Drv[1] <= 'B' THEN Drv := Drv + '\';
                 OldDir := ActDir;
                 ChDir(Drv); IORes := IOResult;
                 IF IORes = 0 THEN
                  BEGIN
                   GetDir(0,ActDir); IORes := IOResult;
                   TextColor(StatusFg); TextBackGround(StatusBg);
                   GotoXY(1,MaxLines); Write('Scanning directory `',ActDir,'''... wait, please.'); ClrEol;
                   ReadFiles;
                   IF NrOfFiles = 0 THEN
                    BEGIN
                     IF (Length(OldDir) > 3) AND (OldDir[Length(OldDir)] = '\') THEN
                       Delete(OldDir,Length(OldDir),1);
                     ChDir(OldDir);
                     ReportError('There are no files on drive '+Drv+'. Press any key.',(CutPasteDesc <> ''),Changed);
                     ReadFiles;
                    END;
                   RedrawScreen;
                   DrawStatusLine(FALSE,(CutPasteDesc <> ''),Changed);;
                   Index := 1;
                   UpdateLineNum(Index);
                  END
                 ELSE
                  ReportError('Drive '+Drv+' not ready! Drive remains unchanged, press a key.',(CutPasteDesc <> ''),Changed);
                END;  (* F6 *)

   kbF4       : DirDown; (* F4 *)
   kbF5       : DirUp;   (* F5 *)

   kbEnter    : BEGIN
                 Description[Index].Desc := EditStr;
                 WriteFileEntry(Index,TRUE);
                 IF (Description[Index].Size = DirSize) THEN
                  IF (Description[Index].Name[1] = '.') AND
                     (Description[Index].Name[2] = '.') THEN DirUp
                  ELSE
                  IF Description[Index].Name[1] <> '.'  THEN DirDown;
                END; (* Enter *)
   kbF10,
   kbF2      : BEGIN                                   (* F10: Save *)
                SaveDescriptions;
                UpdateLineNum(Index);
               END; (* F10 or F2 *)
   kbAltS,
   kbShiftF10: BEGIN                                   (* Shell to 4DOS *)
                NormVideo; ClrScr;
                WriteLn('Press `Exit'' to return to 4DESC.');
                SwapVectors;
                Exec(GetEnv('COMSPEC'),'');
                SwapVectors;
                ClrScr;
                DrawMainScreen(Index,NrOfFiles);
                DrawStatusLine(TRUE,(CutPasteDesc <> ''),Changed);
                DrawDirLine;
                RedrawScreen;
                UpdateLineNum(Index);
               END;
   kbAltV,
   kbF3      : IF (Description[Index].Size <> DirSize) THEN
                BEGIN                                  (* F3: View File *)
                 SwapVectors;
                 FSplit(Description[Index].Name,NewDir,NewName,NewExt);
                 StripTrailingSpaces(NewName);
                 Exec(GetEnv('COMSPEC'),'/c '+ListCmd+' '+ActDir+'\'+NewName+NewExt);
                 SwapVectors;
                 ClrScr;
                 DrawMainScreen(Index,NrOfFiles);
                 DrawStatusLine(TRUE,(CutPasteDesc <> ''),Changed);
                 DrawDirLine;
                 RedrawScreen;
                 UpdateLineNum(Index);
               END;
  ELSE
   IF (Ord(Key) > 31) AND (Ord(Key) < 256) THEN
    BEGIN
     Changed := TRUE;
     DrawStatusLine(FALSE,(CutPasteDesc <> ''),Changed);

     IF Overwrite AND (x <= Length(EditStr)) THEN
      EditStr[x] := Chr(Key)
     ELSE
      EditStr := Copy(EditStr,1,x-1)+Chr(Key)+Copy(EditStr,x,255);
     INC(x);
     IF x > MaxDescLen THEN x := MaxDescLen;

     Description[Index].Desc := EditStr;
     WriteFileEntry(Index,TRUE);
     GotoXY(EdStart+x-1,y);
    END; (* all others *)

  END;  (* case *)
 UNTIL (Key = kbEsc) OR (Key = kbAltX);

 IF Changed THEN QuerySaveDescriptions;
END; (* EditDescriptions *)

(*-------------------------------------------------------- Main *)
BEGIN
 EdStart := 25+Length(DateFormat)+Length(TimeFormat);
 DescLen := ScreenWidth-EdStart;
 GetDir(0,StartDir); IORes := IOResult; DoShowHelp := FALSE;
 IF ParamCount > 0 THEN
  BEGIN
   FOR i := 1 TO Min(2,ParamCount) DO
    BEGIN
     FirstParam := ParamStr(i);
     IF (FirstParam[1] = '/') OR (FirstParam[1] = '-') THEN
      BEGIN
       IF NOT Monochrome THEN Monochrome := (UpCase(FirstParam[2]) = 'M');
       IF NOT DoShowHelp THEN DoShowHelp := (UpCase(FirstParam[2]) = 'H') OR
                                            (FirstParam[2] = '?');
      END;
    END;  (* for ... do begin *)
   FSplit(ParamStr(ParamCount), NewDir, NewName, NewExt);
   IF NewDir[Length(NewDir)] = '\' THEN NewDir[Length(NewDir)] := ' ';
   ChDir(NewDir);
  END;  (* if paramcount > 0 *)
 IORes := IOResult;
 Changed := FALSE; CutPasteDesc := '';
 ChooseColors(Monochrome);
 IF DoShowHelp THEN ShowHelp(MaxDesc);
 ReadFiles;
 RedrawScreen;
 EditDescriptions;
 ChDir(StartDir);
 SetCursorShape(OrigCursor);
 NormVideo;
 ClrScr;
 WriteLn('4DESC ',ver,' - (c) 1992 Copyright by David Frey & Tom Bowden');
 WriteLn;
 WriteLn('This program is freeware: you are allowed to use, copy it free');
 WriteLn('of charge, but you may not sell or hire 4DESC.');
END.
