/* Standard compiler headers */

#include <stdio.h>
#include <math.h>
#include <stdlib.h>

/* IXP header included with the IPAS toolkit */
#include "ixp.h"

/* PASTEL header file */
#include "pastel.h"

/* Defines for the dialog box variables */

#define SHADE 1
#define BRITE 2

/* The dialog box for the cat tails settings
   This sets up the whole dialog box with the layout being done
   by 3D Studio. See the IPAS Toolkit guide for the syntax. */

DlgEntry cdialog[]={
	0,"TITLE=\"3D Studio Special Effects\"",
	0,"TITLE=\"Keith A. Seifert\"",
	0,"TITLE=\"Pastel Shader\"",
	0,"TITLE=\"Version 1.0\"",
	0,"TITLE=\"\"",
	SHADE,"SLIDER=\"Color Fade:\",1,100,20",
	BRITE,"SLIDER=\"Brightness:\",1,100,20",
	0,"TITLE=\"\"",
	0,NULL
};

/* Each IPAS process needs a unique version number */

#define VERSION 0xAB03
	
/* The state structure contains all the variables which will be
   saved and restored by 3D Studio */ 

typedef struct {
	ulong version;
	int shade,brite;
} State;

/* The default settings for the state structure */

static State init_state = { VERSION,50,50 };
static State state = { VERSION,50,50 };

/* The standard function call for processing the current
   image buffer. All IXP routines need this function */

int ClientProcImage(FrameParm *f,FrameInfo *i,FieldInfo *d,int state,Report *rpt) {
	fi=i;
	fp=f;

	dev_width=fi->dev_width;
	dev_height=fi->dev_height;

   /* In this case we copy the input buffer to the output
      then change the output buffer directly */

	far_to_far(fp->outpix,fp->inpix,dev_width*dev_height*3);

   /* process the output buffer */

	process_image();

	return(1);
}

/* The standard function for setting the variables from the
   dialog box. All IXP routines need this function */

void ClientSetStateVar(int id, void *ptr) {
 	OVL o;
	ulong *ul;

	ul = (ulong *)ptr;
	o.ul = *ul;
	switch(id) {
		case SHADE:
         state.shade = o.i;
         break;
		case BRITE:
         state.brite = o.i;
         break;
	}
}

/* The standard function for getting the variables for the
   dialog box. All IXP routines need this function */

ulong ClientGetStateVar(int id) {
	OVL o;

	switch(id) {
		case SHADE:
         o.i = state.shade;
         break;
		case BRITE:
         o.i = state.brite;
         break;
   }
	return(o.ul);
}

/* The standard function for setting the variable size if
   the variable size is not four bytes.
   All IXP routines need this function */

int ClientVarSize(int id) {
	return(1);
}

/* The standard function for getting the size of the state
   structure. All IXP routines need this function */

char  *ClientGetState(int *size) {
	*size = sizeof(State);
	return((char *)&state);
}

/* The standard function for resetting the state structure
   with the default setup. All IXP routines need this function */

void ClientResetState() {	
	state = init_state;	
}

/* The standard function for starting the external process.
   At this time all the global setup should be done. At this
   point the external process begins to control the flow of
   the process. In this case no memory allocation or variable
   setup needs to be done.All IXP routines need this function */

void ClientStartup(void) {
}

/* The standard function for terminating. The external function
   should free all allocated data here. 3D Studio will call this
   function when unloading the process. All IXP routines need
   this function */

void ClientTerminate(void) { 
}

/* The standard function for returning the number lines in the
   dialog box. All IXP routines need this function */

DlgEntry *ClientDialog(int n) {	
	return(&cdialog[n]); 
}

/* The standard function for returning the name of process.
   All IXP routines need this function */

char *ClientName(void) {
   return("PASTEL.IXP");
}


/* The external process code itself */

/* The process_images function changes the output buffer which
   already contains a copy of the image buffer. Both the input
   and output images buffers are available globally at this point. */

int process_image(void) {
   int x,y,pixel,line;  /* Image position values */
   double r,g,b;        /* Current RGB values */
   double h,l,s;        /* Convert HLS values */
   double rc,gc,bc;     /* Calculation RGB values */
   double lper,sper;    /* Control variables */
   double cmin,cmax,dl,v1,v2;  /* Working variables */

   /* Setup the current control values as percentages to be used for
      changing the image buffer. */

   lper = (double)state.brite * 0.006;
   sper = 0.7 - ((double)state.shade * 0.007) + 0.3;

   /* For each line in the current image */

   for( y=0;y<dev_height;y++ ) {
      line = dev_width * y;

      /* For each pixel in the current line */

      for( x=0;x<dev_width;x++ ) {
         pixel = line + x;

         /* Get the current RGB value in a 0.0 - 1.0 floating
            point range */

         r = (double)fp->outpix[pixel].r / 255.0;
         g = (double)fp->outpix[pixel].g / 255.0;
         b = (double)fp->outpix[pixel].b / 255.0;

         /* Convert the RGB values to the correct HLS color
            model values. This formula is available in Computer
            Graphics: Principals and Practice. */

         cmin = cmax = r;
         if( b < cmin )
            cmin = b;
         if( b > cmax )
            cmax = b;
         if( g < cmin )
            cmin = g;
         if( g > cmax )
            cmax = g;
         l = (cmin + cmax) * 0.5;
         if( cmax == cmin )
            s = 0.0;
         else {
            if( l <= 0.5 )
               s = (cmax-cmin)/(cmax+cmin);
            else
               s = (cmax-cmin)/(2.0-cmax-cmin);
            dl = cmax-cmin;

            rc = (cmax - r) / dl;
            gc = (cmax - g) / dl;
            bc = (cmax - b) / dl;
	         if( r == cmax )
	            h = bc - gc;
	         else if( g == cmax )
	            h = 2.0 + rc - bc;
	         else
	            h = 4.0 + gc - rc;

            h *= 60.0;
            if( h < 0.0 )
               h += 360.0;
         }

         /* With the HLS values find the modified output value */

         if( s == 0 ) {

            /* Special case of no saturation */

            rc = (r * lper + 0.5) * 255.0;
            if( rc - floor(rc) > 0.5 )
               fp->outpix[pixel].r = (unsigned char)ceil(rc);
            else
               fp->outpix[pixel].r = (unsigned char)floor(rc);

            gc = (g * lper + 0.5) * 255.0;
            if( gc - floor(gc) > 0.5 )
               fp->outpix[pixel].g = (unsigned char)ceil(gc);
            else
               fp->outpix[pixel].g = (unsigned char)floor(gc);

            bc = (b * lper + 0.5) * 255.0;
            if( bc - floor(bc) > 0.5 )
               fp->outpix[pixel].b = (unsigned char)ceil(bc);
            else
               fp->outpix[pixel].b = (unsigned char)floor(bc);
         }
         else {

            /* General conversion case */
            /* Find the modified HLS values */

            l = l * lper + 0.4;
            s = s * sper;

            if( l <= 0.5 )
               v2 = l * (1.0 + s);
            else
               v2 = l + s - (l * s);
            v1 = 2.0 * l - v2;

            /* Change the values stored in the output buffer */

            fp->outpix[pixel].r = ColorVal(v1,v2,h+120.0);
            fp->outpix[pixel].g = ColorVal(v1,v2,h); 
            fp->outpix[pixel].b = ColorVal(v1,v2,h-120.0);
         }
      }
   }

   return 0;
}

/* The ColorVal function returns the color value from parameters
   used in the HLS color model system. Returns the color
   value in 0-255 value range. */

unsigned char ColorVal( double v1,double v2,double h ) {
   unsigned char c;
   double fc;

   if( h < 0.0 )
      h += 360.0;
   if( h > 360.0 )
      h -= 360.0;

   if( h < 60.0 )
      fc = v1 + ((v2 - v1) * h) / 60.0;
   else if( h < 180.0 )
      fc = v2;
   else if( h < 240.0 )
      fc = v1 + ((v2 - v1) * (240.0 - h)) / 60.0;
   else
      fc = v1;

   fc *= 255.0;
   if( fc - floor(fc) > 0.5 )
      c = (unsigned char)ceil(fc);
   else
      c = (unsigned char)floor(fc);
   return c;
}
