
/* This file contains error parsers to supplement those provided with
 * the Codewright distribution.  It will work with 2.0 or later versions.
 * To compile with 2.05 or later, define CWSTART.  The accompanying
 * makefile (errparse) will work with either version, and can be
 * used, with slight modifications, with either Microsoft or Borland
 * compilers.  See comments in that file for more information.
 */

/*
 * To add support for a new error parser for a compiler named 'Foo C'
 * compiler, complete the following steps:
 *
 *   1.  Add an entry like "FooC", to the  'parsers' array below.
 *   2.  Create a function _FooCErrorInfo() that will return the
 *       error file, error line, error column, error message, and
 *       error/warning level using the other functions as an example.
 *   3.  Run nmake or similar make utility to rebuild the DLL.
 *   4.  Move the DLL to a directory where Codewright can read it.
 *
 * The key to producing an error parser is to look for unique combinations
 * of characters to key on and begin processing from there.
 *
 */

#include <windows.h>
#include "exports.h"
#ifdef CWSTART
#include "cwstart.h"
#else
#include "startup.h"
#endif
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <limits.h>

#ifndef DECODE_PROJECT	// undefined in 2.0
#define	DECODE_PROJECT	(0x0008)
#endif

#ifdef __BORLANDC__
LIBMAIN    /* defines the LibMain() function for Borland C */
int DLL WEP(int nParam){return(1);}
#endif

#define ERRPARSE_SEARCH	(SEARCH_FORWARD | SEARCH_REGEX)

static void trim(LPSTR str, LPSTR set);
static LPSTR getNormalizedExtension( LPSTR ext );
static LPSTR getSourceBufferName(void);
static long bufNextTab(long oldcol);
static ERRCODE markReplacePos(void);

/* list of error parsers to produce names like _AcuCobolErrorInfo */
static LPSTR parsers[] =
{
	"2500AD_8051",
	"AcuCobol",
	"MicrotecC",
	"ObjectView",
	"WhitesmithC",
	"Asm56",
	"Sierra",
	"White68",
	"VLC",
	"Asm51",
	"LiantCOBOL85",
	"CFW",
	"Avocet_or_Z8",
	"Watcom",
	"IntelC386",
	"Archimedes",
	"MSCobol",
	"OracleSQL",
	"MapInfo",
	"Asm96",
	"PLM96",
	"iC96",
	"BorlandMake",
	"PCLint",
	"ThompsonAWK",
	"VHDL",
	"PLM51",
	"MSLink",
	"Gcc",
	"CrossCode",
	"Intermet",
	"UNIBench",
	"Fortran",
	NIL
};

/* When this DLL is first loaded either by LibPreLoad() or LibAutoLoad(),
 * the function _init() is invoked.  Within this function, all DLL specific
 * initialization should be performed such as:
 *
 *  - Exporting any functions to be accessed through LibFunctionExec()
 *  - Autoloading any other DLLs.
 *
 */
void DLL
_init()
{
	LPSTR *pp;
	char name[200];
	
	/* export error parser functions and add their names to the list */
	for (pp = parsers; *pp != NIL; pp++)
	{
		wsprintf(name, "int _%sErrorInfo", *pp);
		LibExport(name);
		_AddErrorInfoName(name + 4);
	}
}

/*
 ** _2500AD_8051ErrorInfo
 *
 *  This is the error info function to process the error file generated
 *  by the 2500AD 8051 compiler.  This also works for their Version 4
 *  series 68HC11 copmilers.  The function returns 0 for no more errors
 *  found, 1 for a warning and 2 for an error line in the file.
 *
 *  The format of the error file is:
 * col. 1234567890123456789012345678901234567890
 *                            Input  Filename : <filename>
 *
 *      <error line>
 *      <error line>
 *
 *  The format of each error line:
 *      <line>   <addr>  <opcode> ...
 *               ***** <error message> *****
 *
 *  NOTE: because the filename information is on a separate line from
 *        the error information, this error parser needs to search
 *        backward from any error message to discover the filename
 *        corresponding to the error message.
 *
 *  For further information see '_ErrorInfoDefault()'.
 *
 */
int DLL
_2500AD_8051ErrorInfo( void )
{
	int level = 0;
	LPSTR msg = NIL;
	LPSTR file = NIL;
	LPSTR p;

	/* look for a line containing an error message, leave cursor in the message */
	if (!SrchFind( "^ +\\*\\*\\*\\*\\* \\c", ERRPARSE_SEARCH | SEARCH_IGCASE, NIL ))
		return 0;

	/* search backward for the line containing the error file name */
	MarkSavePos();
	if (SrchFind( "Input[ \t]+Filename[ \t]+:[ \t]+\\c", SEARCH_REGEX | SEARCH_IGCASE, NIL ))
	{
		/* found line containing error file name */
		file = BufReadStr( 0x7FFFFFFF );
		trim(file, "\n\t ");
		ErrorSrcFile(file);
		StrFree(file);
	}
	MarkRestorePos();

	/* read the error message and strip off trailing junk */
	msg = BufReadStr( 0x7FFFFFFF );
	trim(msg, "\n\t *");
	ErrorMsgLine(msg);
	StrFree(msg);

	/* move to the previous line and get the line number */
	MarkSavePos();
	MovUp(1);
	MovHome();
	msg = BufReadStr( 0x7FFFFFFF );
	MarkRestorePos();
	for (p = msg; isspace(*p); p++)
		;
	ErrorLine(atol(p));
	ErrorColumn(0L);
	StrFree(msg);

	return 2;
}

/*
 ** _AcuCobolErrorInfo
 *
 *  This is the error info function to process the error file generated
 *  by the AcuCobol compiler.  The function returns 0 for no more errors
 *  found, 1 for a warning and 2 for an error line in the file.
 *
 *  The format of each line in the error file is:
 *     <filename>, line <line>: <message>
 *
 *  For example:
 *     c:\proj\test.cob, line 26: Verb expected, 03 found
 *
 *  For further information see '_ErrorInfoDefault()'.
 *
 */
int DLL
_AcuCobolErrorInfo( void )
{
	LPSTR srchStr = "^[a-zA-Z/\\_].+, line \\c[0-9]+:";
	int level = 0;
	LPSTR msg;
	LPSTR p;

	/* Search through the error file for the next matching error message
	 * leaving the cursor located over the error's line number.
	 */
	if (!SrchFind( srchStr, ERRPARSE_SEARCH | SEARCH_IGCASE, NIL ))
		return 0;

	/* Save the position of the line number and make a copy of the
	 * message line.
	 */
	MarkSavePos();
	MovHome();
	msg = BufReadStr( 0x7FFFFFFF );
	MarkRestorePos();

	/* Strip out the source file name from the error line */
	if ((p = strchr(msg, ',')) == NIL)
		goto done;
	*p++ = '\0';
	ErrorSrcFile(msg);

	/* skip over " line " and read the line number */
	ErrorLine(atol(p + 6));
	ErrorColumn(0L);

	/* move to the error message and determine error/warning */
	if ((p = strchr(p, ':')) != NIL)
	{
		p += 2;
		level = (strncmp(p, "Warning:", 8) == 0) ? 1 : 2;
		ErrorMsgLine(p);
	}

done:
	StrFree( msg );
	return level;
}

/*
 ** _MicrotecCErrorInfo
 *
 *  This is the error info function to process the error file generated
 *  by the Microtec compiler.  The function returns 0 for no more errors
 *  found, 1 for a warning and 2 for an error line in the file.
 *
 *  The format of each line in the error file is:
 *     "<filename>", line <line> pos <number>; (E) <message>
 *     "<filename>", line <line> pos <number>; (W) <message>
 *
 *  For example:
 *     "c:\proj\test.c", line 26 pos 5; (E) undeclared identifier "cd_len"
 *
 *  For further information see '_ErrorInfoDefault()'.
 *
 */
int DLL
_MicrotecCErrorInfo( void )
{
	LPSTR srchStr = "^\"[a-zA-Z/\\_].+\", line \\c[0-9]+ ";
	int level = 0;
	LPSTR msg;
	LPSTR p;

	/* Search through the error file for the next matching error message
	 * leaving the cursor located over the error's line number.
	 */
	if (!SrchFind( srchStr, ERRPARSE_SEARCH | SEARCH_IGCASE, NIL ))
		return 0;

	/* Save the position of the line number and make a copy of the
	 * message line.
	 */
	MarkSavePos();
	MovHome();
	msg = BufReadStr( 0x7FFFFFFF );
	MarkRestorePos();

	/* Strip out the source file name from the error line */
	if ((p = strchr(msg + 1, '"')) == NIL)
		goto done;
	*p++ = '\0';
	ErrorSrcFile(msg + 1);

	/* skip over " line " and read the line number */
	ErrorLine(atol(p + 7));
	ErrorColumn(0L);

	/* move to the error message and determine error/warning */
	if ((p = strchr(p, ';')) != NIL)
	{
		p += 2;
		level = (strncmp(p, "(E)", 3) == 0) ? 2 : 1;
		ErrorMsgLine(p);
	}

done:
	StrFree( msg );
	return level;
}

/*
 ** _ObjectViewErrorInfo
 *
 *  This is the error info function to process the error file generated
 *  by the ObjectView compiler.  The function returns 0 for no more errors
 *  found, 1 for a warning and 2 for an error line in the file.
 *
 *  The format of the error file is:
 * col. 1234567890
 *      Compilation messages for <filename>
 *
 *      <error line>
 *      <error line>
 *
 *  The format of each error line:
 *      *** [Line:<line>,Col:<col>]
 *          <error/warning message>
 *
 *  NOTE: because the filename information is on a separate line from
 *        the error information, this error parser needs to search
 *        backward from any error message to discover the filename
 *        corresponding to the error message.
 *
 *  For further information see '_ErrorInfoDefault()'.
 *
 */
int DLL
_ObjectViewErrorInfo( void )
{
	int level = 0;
	LPSTR msg = NIL;
	LPSTR file = NIL;
	long line;

	/* look for a line containing an error message, leave cursor on line number */
	if (!SrchFind( "^\\*\\*\\* \\[Line:\\c[0-9]+", ERRPARSE_SEARCH | SEARCH_IGCASE, NIL ))
		return 0;

	/* search backward for the line containing the error file name */
	MarkSavePos();
	if (SrchFind( "^Compilation messages for \\c", SEARCH_REGEX | SEARCH_IGCASE, NIL ))
		/* found line containing error file name */
		file = BufReadStr( 0x7FFFFFFF );
	MarkRestorePos();

	/* make a copy of the line to examine (should begin with line number */
	msg = BufReadStr( 0x7FFFFFFF );
	line = atol(msg);
	StrFree(msg);
	msg = NIL;

	/* move to the next line and get the message */
	if (SrchFind( "^    \\c", ERRPARSE_SEARCH | SEARCH_IGCASE, NIL ))
	{
		/* read error message */
		msg = BufReadStr( 0x7FFFFFFF );

		/* have all components, send information back */
		if (file != NIL)
			ErrorSrcFile(file);
		ErrorLine(line);
		ErrorColumn(0L);
		ErrorMsgLine(msg);
		level = 2;			// indicate error rather than warning
	}

	/* clean up */
	if (msg != NIL)
		StrFree( msg );
	if (file != NIL)
		StrFree(file);
	return level;
}

/*
 ** _WhitesmithCErrorInfo
 *
 *  This is the error info function to process the error file generated
 *  by the Whitesmith compiler.  The function returns 0 for no more errors
 *  found, 1 for a warning and 2 for an error line in the file.
 *
 *  The format of each line in the error file is:
 *     #error cp1(V3.32) <filename>:<line>  <message>
 *     #warning cp1(V3.32) <filename>:<line>  <message>
 *
 *  For example:
 *     #error cp1(V3.32) c:\proj\test.c:26  missing )
 *
 *  For further information see '_ErrorInfoDefault()'.
 *
 */
int DLL
_WhitesmithCErrorInfo( void )
{
	LPSTR srchStr = "^#\\c";
	int level = 0;
	LPSTR msg;
	LPSTR p;
	LPSTR file;
	LPSTR txt;

	/* Search through the error file for the next matching error message */
	if (!SrchFind( srchStr, ERRPARSE_SEARCH | SEARCH_IGCASE, NIL ))
		return 0;

	/* Save the position of the line number and make a copy of the
	 * message line.
	 */
	MarkSavePos();
	MovHome();
	msg = BufReadStr( 0x7FFFFFFF );
	MarkRestorePos();

	/* Strip out the source file name from the error line */
	if ((p = strchr(msg, ')')) == NIL)
		goto done;
	file = p + 2;
	if ((txt = strchr(file, ' ')) == NIL)
		goto done;
	if ((p = strrchr(file, ':')) == NIL)
		goto done;
	*p++ = '\0';
	ErrorSrcFile(file);

	/* read the line number */
	ErrorLine(atol(p));
	ErrorColumn(0L);

	/* move to the error message and determine error/warning */
	ErrorMsgLine(txt + 2);
	level = (strncmp(msg, "#error", 6) == 0) ? 2 : 1;

done:
	StrFree( msg );
	return level;
}



/*
 * _Asm56ErrorInfo
 *
 *    This function is an error file processor which parses both
 *    Motorola Asm56 and Lnk56 error log files.
 *    The function returns 0 for no more errors
 *    found, 1 for a warning and 2 for an error line in the file.
 *
 *  The format of each line in the Asm56 error file is:
 *	   **** <total line count>[<filename> <line>]: ERROR -- <message>
 *
 *  The format of each line in the Lnk56 error file is:
 *	   **** [File <filename>, Module <mod.>, Record <#>, Line <line>]: ERROR --- <message>
 *
 *  For example:
 *
 *		Asm56 errors:
 *		**** 288 [501_main.asm 85]: ERROR -- Unrecognized mnemonic: mve
 *
 *		Lnk56 errors:
 *		**** [File test.c, Module test.c, Record 72, Line 24]: ERROR --- Address or ...
 *
 *
 *  For further information see '_ErrorInfoDefault()'.
 *
 */

#define DONE_56	if (1) {StrFree(srcNameBuf); \
				StrFree(msgbase); MovEOL(); return(retval);} else

int DLL
_Asm56ErrorInfo( void )
{
	LPSTR msgbase = NULL, msgp;
	LPSTR srcNameBuf = NULL;
	int msgidx;
	int retval = 0;
	BOOL asm56;			/* FALSE if file is lnk56; TRUE if it's asm56	*/

	/* assign initial values	*/

	ErrorLine(0);
	ErrorColumn(0);
	ErrorMsgLine(NULL);

	/* Search through the error file for the next matching error message.
	 */
	if (!SrchFind("**** ", SEARCH_FORWARD, NIL ))
		DONE_56;

	/* Make a copy of the message line.
	 */
	if (!(msgp = msgbase = BufReadStr( 0x7FFFFFFF )))
		DONE_56;

	while (*msgp == ' ' || *msgp == '*')
		++msgp;

	if (isdigit((int)*msgp))
	{
		asm56 = TRUE;
		if (!(msgp = strchr(msgp, '[')))
			DONE_56;
	   ++msgp;
	}
	else
	{
		asm56 = FALSE;
		if (*msgp != '[')	/* don't recognize the format of this line; ignore it	*/
			DONE_56;
		if (!(msgp = strchr(msgp, ' ')))
			DONE_56;
		++msgp;
	}
	if (!*msgp)
		DONE_56;
	/* msgp now points to the filename.  Allocate a buffer to hold a copy of it.
	 */
	if (!(srcNameBuf = StrNew(msgp)));
		DONE_56;

	/* Copy the filename in order to null-terminate it.
	 */
	for (msgidx = 0; msgp[msgidx]; ++msgidx)
	{
		if (msgp[msgidx] == ',' || msgp[msgidx] == ' ')
			break;
		srcNameBuf[msgidx] = msgp[msgidx];
	}
	srcNameBuf[msgidx] = 0;
	msgp += msgidx;

	ErrorSrcFile(srcNameBuf);

	/* Act gracefully if the error message is truncated; assume no error
	 */
	if (!*msgp)
		DONE_56;

	if (asm56)						/* line number is not labeled		*/
	{
		if (!*++msgp)
			DONE_56;
	}
	else						    /* lnk56 is labeled with "Line "	*/
	{
		if (!(msgp = strstr(msgp, "Line ")))
			DONE_56;
		else
			msgp += 5;						/* Skip the string "Line "	*/
	}
	ErrorLine(atol(msgp));			/* atol stops at first nonnumeric	*/

	/* Look for colon which signals the beginning of the error text,
	 * including the string "ERROR" in the case of an error.
	 */
	if (!(msgp = strchr(msgp, ':')))
		DONE_56;
	if (!*++msgp)					/* should be a space */
		DONE_56;
	if (!*++msgp)					/* should be the first E in "ERROR" (or a W?) */
		DONE_56;
	/* If the line doesn't say "ERROR", assume it's a warning of some kind.
	 */
	if (!strncmp(msgp, "ERROR", 5))
		retval = 1;
	else
		retval = 2;

	/* Skip to the text describing the error or warning.
	 */
	if (!(msgp = strchr(msgp, '-')) || (!msgp[1] || !msgp[2]))
		DONE_56;
	while (*msgp == '-' || *msgp == ' ')
		++msgp;
	if (!*msgp)					/* first char of message	*/
		DONE_56;

	/* Finish up.
	 */
	if (!asm56)
		msgp[strlen(msgp) - 1] = 0;	/* delete trailing semicolon	*/
	ErrorMsgLine(msgp);
	DONE_56;
}

#undef DONE_56

/*
 ** _SierraErrorInfo()
 *
 *  This is the error info function to process the error file generated
 *  by the Sierra 68000 cross compiler.  The function returns 0 for no more
 *  errors found, 1 for a warning and 2 for an error line in the file.
 *
 *  The format of each line in the error file is:
 *     <filename>, <line>: <message>
 *     <filename>, <line>: warning: <message>
 *
 *  For example:
 *		bcmmain.c, 56: syntax error
 *		bcmmain.c, 59: warning: illegal combination of pointers and integers
 *
 *  For further information see '_ErrorInfoDefault()'.
 *
 */

#define DONE_Sierra	if (1) {MovEOL(); StrFree(msgbase); return(retval);} else

int DLL
_SierraErrorInfo( void )
{
	LPSTR msgbase = NULL, msgp, namebase;
	int retval = 0;

	/* assign initial values	*/

	ErrorLine(0);
	ErrorColumn(0);
	ErrorMsgLine(NULL);

	/* Search through the error file for the next matching error message.
	 */
	if (!SrchFind("^.*\\.[a-zA-Z]+, [0-9]+:", ERRPARSE_SEARCH, NIL ))
		DONE_Sierra;

	/* Make a copy of the message line.
	 */
	if (!(msgp = msgbase = BufReadStr( 0x7FFFFFFF )))
		DONE_Sierra;

	/* Find, isolate, and report the file name
	 */
	while (isspace(*msgp))
		++msgp;
	namebase = msgp;
	while (*msgp && *msgp != ',')
		++msgp;

	if (!*msgp)				/* line was truncated	*/
		DONE_Sierra;
	*msgp = 0;
	ErrorSrcFile(namebase);

	/* Find and report the line number
	 */

	while (*++msgp && !isdigit(*msgp))
			;
	if (!*msgp)
		DONE_Sierra;
	ErrorLine(atol(msgp));


	/* Report the body of the error message
	 */

	while (isdigit(*msgp))
		++msgp;
	while (isspace(*msgp) || *msgp == ':')
		++msgp;

	if (!strncmp(msgp, "warning: ", 9))
	{
		retval = 1;
		msgp += 9;
	}
	else
	{
		retval = 2;
	}

	if (*msgp)
		ErrorMsgLine(msgp);

	DONE_Sierra;
}

#undef DONE_Sierra

/*
 ** _White68ErrorInfo()
 *
 *  This is the error info function to process the error file generated
 *  by the Whitesmith 68HC11 cross compiler.  The function returns 0 for no more
 *  errors found, 1 for a warning and 2 for an error line in the file.
 *
 *  The format of each line in the error file is:
 *		<filename>:
 *		#error <compiler phase> <filename>:<line> <message>
 *
 *
 *  For example:
 *		conflict.c:
 *		#error pp conflict.c:526 invalid integer constant
 *
 *  For further information see _WhitesmithCErrorInfo and _ErrorInfoDefault().
 *
 */
#define DONE_White68   if (1) {MovEOL(); StrFree(msgbase); return(retval);} else

int DLL
_White68ErrorInfo( void )
{
	return(_WhitesmithCErrorInfo());
}

/*
 ** _VLCErrorInfo()
 *
 *  This is the error info function to process the error file generated
 *  by the Parity VLC compiler.  The function returns 0 for no more
 *  errors found, 1 for a warning and 2 for an error line in the file.
 *
 *  The format of each line in the error file is:
 *		<filename>(<line>) : <error code> <message>
 *		<filename>(<line>) (Warning): <message>
 *
 *	This function assumes that warnings are in an analagous format.
 *
 *  For example:
 *		TST.VS(16) : E1059 Function sc__play() not defined, .FUN file not found
 *		TST.VS(16) (Warning): Variable declared but not used 'i'
 *
 *  For further information see '_ErrorInfoDefault()'.
 *
 */

#define DONE_VLC	if (1) {MovEOL(); StrFree(msgbase); return(retval);} else

int DLL
_VLCErrorInfo( void )
{
	LPSTR msgbase = NULL, msgp;
	int retval = 0;

	/* assign initial values	*/

	ErrorLine(0);
	ErrorColumn(0);
	ErrorMsgLine(NULL);

	/* Search through the error file for the next matching error message.
	 */
	if (!SrchFind("^[a-zA-Z0-9_].*([0-9]+)", ERRPARSE_SEARCH, NIL ))
		DONE_VLC;

	/* Make a copy of the filename line.
	 */
	if (!(msgbase = BufReadStr( 0x7FFFFFFF )))
		DONE_VLC;

	/* Skip to source file name
	 */
	for (msgp = msgbase; *msgp && *msgp != '('; ++msgp)
		;
	*msgp = 0;
	ErrorSrcFile(msgbase);

	/* Go to and record line number
	 */
	if (!*++msgp)
		DONE_VLC;
	ErrorLine(atol(msgp));

	/* Skip line number
	 */
	while (isdigit(*msgp))
		++msgp;
	++msgp;					/* skip ')'	*/
	while (isspace(*msgp))
		++msgp;
	if (*msgp == ':')
	{
		retval = 2;
		while (isspace(*++msgp))
			;
		while (*msgp && !isspace(*msgp))
			++msgp;
	}
	else
	{
		retval = 1;
		while (*msgp && !isspace(*msgp))
			++msgp;
	}
	while (isspace(*msgp))
		++msgp;

	ErrorMsgLine(msgp);

	DONE_VLC;
}

#undef DONE_VLC

/*
 ** _Asm51ErrorInfo()
 *
 *  This is the error info function to process the error file generated
 *  by the Archimedes ASM51 cross assembler.  The function returns 0 for
 *  no more errors found, 1 for a warning and 2 for an error line in the file.
 *
 *  The format of each line in the error file is:
 *	Error in <line>: <message>
 *      <source text>
 *      <carat pointing to error column>
 *
 *  For example:
 *	Error in 23: Invalid instruction
 *      START:		.MACRO
 *			^
 *
 *  For further information see '_ErrorInfoDefault()'.
 *
 */

#define DONE_ASM51	if (1) {StrFree(msgbase); StrFree(srcName); \
	MarkRestorePos(); MovEOL(); return(retval);} else

int DLL
_Asm51ErrorInfo( void )
{
	LPSTR msgbase = NULL, msgp;
	long col;
	int ch;
	int retval = 0;
	LPSTR srcName = NULL;

	srcName = getSourceBufferName();

	/* assign initial values	*/

	ErrorLine(0);
	ErrorColumn(0);

	/* Search through the error file for the next matching error message.
	 */
	if (!SrchFind("^Error in [0-9]+:", ERRPARSE_SEARCH, NIL ))
	{
		StrFree(srcName);
		return(retval);
	}
	MarkSavePos();


	/* Make a copy of the errmsg line
	 */
	if (!(msgbase = BufReadStr( 0x7FFFFFFF )))
		DONE_ASM51;

	ErrorSrcFile(srcName);
	retval = 2;

	/* Skip to line number
	 */
	for (msgp = msgbase; *msgp && !isdigit(*msgp); ++msgp)
		;
	if (*msgp)
		ErrorLine(atol(msgp));
	while (isdigit(*msgp))
		++msgp;
	++msgp;					/* skip ':'	*/
	while (isspace(*msgp))
		++msgp;
	if (*msgp)
		ErrorMsgLine(msgp);
	MovDown(2);
	MovHome();
	col = 1;
	while ((ch = BufReadChar()) > 0)	/* assume no spaces within tabstops	*/
	{
		if (ch == '\t')
			col += 8;
		else if (ch == ' ')
			col++;
		else
			break;
		MovNextChar(1);
	}
	if (ch == '^')
		ErrorColumn(col);
	DONE_ASM51;
}

#undef DONE_ASM51

/*
 ** _LiantCOBOL85ErrorInfo()
 *
 *  This is the error info function to process the error file generated
 *  by the Liant RM/COBOL 85 compiler.  The function returns 0 for no more
 *  errors found, 1 for a warning and 2 for an error line in the file.
 *
 *  The format of each line in the error file is:
 *	Error in <line>: <message>
 *      <source text>
 *      <carat pointing to error column>
 *
 *  <line (5-column field)><source text>
 *  <dollar sign pointing to error column>
 *  *****   <error order on line>) <error code>: <E or W> <message>
 *
 *
 *  For example:
 *
 *
 *  229                            DISPLY "CLIENT #" LINE 12 POSITION 32
 *                                 $
 *  *****   1) 0465: E Verb expected. (scan suppressed).
 *
 *  233    000404              CLOS CLIENTS.
 *                             $           $
 *  *****   1) 0465: E Verb expected. (scan suppressed).
 *  *****   2) 0005: I Scan resumed.
 *
 */

#define LIANT_ERRCOL_OFFSET	(5)	// offset of source lines in error file

int DLL
_LiantCOBOL85ErrorInfo( void )
{
	LPSTR srchStr = "^\\**[ \\t]*[0-9]*\\)[ \\t]*[0-9]*:[ \\t]*[EW]";
	LPSTR msgbase, msgp;		// for handling message lines
	long ccol = 0;			// numeric value of error column
	int retval = 0;			// this function's return value
	static LPSTR srcName = NULL;

	srcName = getSourceBufferName();
	ErrorSrcFile(srcName);

	if (!SrchFind( srchStr, ERRPARSE_SEARCH, NIL ))
		return 0;			// no more error messages
	MarkSavePos();
	msgp = msgbase = BufReadStr( 0x7FFFFFFF );	// copy error message line
	while (*msgp != 'E' && *msgp != 'W')	// skip to E or W
		++msgp;
	if (*msgp == 'E')			// Error or Warning?
		retval = 2;
	else
		retval = 1;
	while (*++msgp && !isspace(*msgp))	// skip space before diagnostics
		++msgp;
	ErrorMsgLine(msgp);			// register error message
	StrFree(msgbase);

	MovUp(2);			// go to where the line number is
	MovHome();
	msgbase = BufReadStr( 0x7FFFFFFF );
	for (msgp = msgbase; isspace(*msgp); ++msgp)	// skip leading space
		;
	ErrorLine(atol(msgp));			// register line number
	StrFree(msgbase);

	MovDown(1);				// go to column marker
	MovHome();
	ccol = 1;
	msgbase = BufReadStr( 0x7FFFFFFF );
 	for (msgp = msgbase + LIANT_ERRCOL_OFFSET; isspace(*msgp); ++msgp)
 	{			// determine column position of '$'
 		if (*msgp == ' ')
 			++ccol;
 		else if (*msgp == '\t')
 			ccol = bufNextTab(ccol);
 	}
	ErrorColumn(ccol);			// register column number
	StrFree(msgbase);

	MarkRestorePos();
	MovEOL();

	return (retval);
}
#undef LIANT_ERRCOL_OFFSET

/*
 ** _CFWErrorInfo
 *
 *  This is the error info function to process the error file generated
 *  by the CFW language compiler.  The function returns 0 for no more
 *	errors found, 1 for a warning and 2 for an error in the file.
 *
 *	The source file name is dumped before any further diagnostics.
 *  The format of each line in the error file is:
 *
 *	Line <line>, <error message>
 *		or
 *	Line <line>, CAUTION!, <warning message>
 *
 *  For example:
 *
 *	C:\JUNK.CFW  05-11-1993  06:46:48
 *	Line 34, CAUTION!, blank line
 *	Line 43, Unknown statement, GOUB GET_SESSION
 *
 */

#define DONE_CFW	if (1) {MovEOL(); StrFree(msgbase); return(retval);} else

int DLL
_CFWErrorInfo( void )
{
	LPSTR msgbase = NULL, msgp;
	long col;
	int retval = 0;
	LPSTR ebuf;

	/* assign initial values	*/

	ErrorLine(0);
	ErrorColumn(0);

	MarkSavePos();
	MovAbs(1, 1);
	if (!(ebuf = BufReadStr(0x7FFFFFFF)))
		DONE_CFW;
	MarkRestorePos();
	for (col = 0; col < 0x7FFFFFFF; ++col)
		if (isspace(ebuf[col]))
		{
			ebuf[col] = 0;
			break;
		}
	ErrorSrcFile(ebuf);

	if (!SrchFind("^Line \\c[0-9]", ERRPARSE_SEARCH, NIL ))
		DONE_CFW;

	if (!(msgbase = BufReadStr( 0x7FFFFFFF )))
		DONE_CFW;

	ErrorLine(atol(msgbase));

	msgp = msgbase;

	while (isdigit(*msgp))
		++msgp;

	++msgp;	/* skip comma	*/
	while (isspace(*msgp))
		++msgp;

	if (!strncmp(msgp, "CAUTION!, ", 10))
	{
		msgp += 10;
		retval = 1;
	}
	else
	{
		retval = 2;
	}
	ErrorMsgLine(msgp);

	DONE_CFW;
}

#undef DONE_CFW


/*
 ** _AvocetErrorInfo
 *
 *  This is the error info function to process the error files generated
 *  by the Avocet 68HC05 cross assembler and the Zilog Z8 cross assembler.
 *  The function returns 0 for no more errors found, and 2 if there was
 *  an error in the file.
 *
 *  The format of each line in the error file is:
 *
 *  [Avocet]
 *
 *	<file>:<line: <message>
 *
 *  For example:
 *
 *	MAIN.ASM:306: CMOS/68HC05 instruction illegal
 *
 *  [Zilog]
 *
 *	<file>: <line: <message>
 *
 *  For example:
 *
 *	MAIN.ASM: 306: Blah Blah
 *
 */

#define DONE_AVOCET	if (1) {MovEOL(); StrFree(msgbase); return(retval);} else

int DLL
_Avocet_or_Z8ErrorInfo( void )
{
	LPSTR msgbase = NULL, msgp;
	long col;
	int retval = 0;

	/* assign initial values	*/

	ErrorLine(0);
	ErrorColumn(0);

	if (!SrchFind("^[^ ]*: ?[0-9]+:", ERRPARSE_SEARCH, NIL ))
		DONE_AVOCET;
	retval = 2;
	if (!(msgbase = BufReadStr(0x7FFFFFFF)))
		DONE_AVOCET;
	for (msgp = msgbase, col = 0; msgp[col] != ':'; ++col)
		;
	msgp[col] = 0;
	ErrorSrcFile(msgp);
	col++;
	while (isspace(msgp[col]))
		++col;
	msgp += col;
	for (col = 0; msgp[col] != ':'; ++col)
		;
	msgp[col] = 0;
	ErrorLine(atol(msgp));
	msgp += (col + 1);
	while (isspace(*msgp))
		++msgp;
	ErrorMsgLine(msgp);
	DONE_AVOCET;
}

#undef DONE_AVOCET

/*
 ** _WatcomErrorInfo
 *
 *  This is the error info function to process the error files generated
 *  by the Watcom C compiler.  It doesn't mistake Make errors for compiler
 *  errors.
 *  The function returns 0 for no more errors found, 1 if the next error
 *  is a warning, and 2 otherwise.
 *
 *  The format of each line in the error file is:
 *
 *	<file>(<line>) : Error! E<error number>: <message>
 *
 *  or
 *
 *	<file>(<line>) : Warning! W<error number>: <message>
 *
 *  For example:
 *
 *	Foxbat.c(20) : Error! E1009: Expecting ...
 *
 *  Make errors, by contrast, look like
 *
 *  Error <filename>: <message>
 *
 */

#define DONE_WATCOM	if (1) {MovEOL(); StrFree(msgbase); return(retval);} else

int DLL
_WatcomErrorInfo( void )
{
	LPSTR msgbase = NULL, msgp;
	long col;
	int retval = 0;

	/* assign initial values	*/

	ErrorLine(0);
	ErrorColumn(0);

	if (!SrchFind("^[^ ]*\\([0-9]+\\) ?: (Error|Warning)! [EW][0-9]+:", ERRPARSE_SEARCH, NIL ))
		DONE_WATCOM;
	if (!(msgbase = BufReadStr(0x7FFFFFFF)))
		DONE_WATCOM;
	for (msgp = msgbase, col = 0; msgp[col] != '('; ++col)
		;
	msgp[col] = 0;
	ErrorSrcFile(msgp);
	col++;
	while (isspace(msgp[col]))
		++col;
	msgp += col;
	for (col = 0; msgp[col] != ')'; ++col)
		;
	msgp[col] = 0;
	ErrorLine(atol(msgp));
	msgp += (col + 1);
	while (isspace(*msgp) || *msgp == ':')
		++msgp;
	for (col = 0; col < 6; ++col)
	{
		if (!msgp[col])
			DONE_WATCOM;
		if (toupper(msgp[col]) != toupper("Error!"[col]))
			break;
	}
	if (col >= 6)
		retval = 2;
	else
	{
		for (col = 0; col < 8; ++col)
		{
			if (!msgp[col])
				DONE_WATCOM;
			if (toupper(msgp[col]) != toupper("Warning!"[col]))
				break;		/* shouldn't really happen */
		}
		if (col < 8)
			DONE_WATCOM;
		retval = 1;
	}
	msgp += (col + 1);
	while (*msgp && *msgp != ':')
		++msgp;
	++msgp;
	while (isspace(*msgp))
		++msgp;
	ErrorMsgLine(msgp);
	DONE_WATCOM;
}

#undef DONE_WATCOM

/*
 ** _IntelC386
 *
 *  This is the error info function to process the error files generated
 *  by the Intel 386 C compiler.
 *  The function returns 0 for no more errors found, 1 if the next error
 *  is a warning, and 2 otherwise.
 *
 *  The format of each line in the error file is:
 *
 *	*** ERROR AT LINE <line> OF <file>: <message>
 *
 *  or
 *
 *	*** WARNING AT Line <line> OF <file>: <message>
 *
 *  For example:
 *
 *	*** ERROR AT LINE 5 OF hello.c: undefined identifier: it
 *  *** WARNING AT LINE 9 OF hello.c: illegal character: 5CH (hex)
 *
 */

#define DONE_INTELC386	if (1) {MovEOL(); StrFree(msgbase); return(retval);} else

int DLL
_IntelC386ErrorInfo( void )
{
	LPSTR msgbase = NULL, msgp;
	long col;
	int retval = 0;

	/* assign initial values	*/

	ErrorLine(0);
	ErrorColumn(0);

	if (!SrchFind("^\\*\\*\\* \\c(ERROR|WARNING) AT LINE [0-9]+ OF",
	ERRPARSE_SEARCH, NIL ))
		DONE_INTELC386;
	if (!(msgbase = BufReadStr(0x7FFFFFFF)))
		DONE_INTELC386;
	if (!strncmp(msgbase, "ERROR", 5))
		retval = 2;
	else
		retval = 1;
	for (msgp = msgbase; !isdigit(*msgp); ++msgp)
		;
	for (col = 0; isdigit(msgp[col]); ++col)
		;
	msgp[col] = 0;
	ErrorLine(atol(msgp));
	while (isspace(msgp[++col]))
		;
	while (msgp[col] && !isspace(msgp[col]))	/* skip "OF" */
		++col;
	while (isspace(msgp[col]))
		++col;
	msgp += col;
	for (col = 0; msgp[col] && !isspace(msgp[col]); ++col)
		;
	if (col && msgp[col - 1] == ':')
		--col;
	msgp[col] = 0;
	ErrorSrcFile(msgp);
	msgp += (col + 1);
	while (isspace(*msgp))
		++msgp;
	ErrorMsgLine(msgp);
	DONE_INTELC386;
}

#undef DONE_INTELC386

/*
 ** _ArchimedesErrorInfo
 *
 *  This is the error info function to process the error files generated
 *  by the Archimedes 80C51 cross compiler.
 *  The function returns 0 for no more errors found, 1 if the next error
 *  is a warning, and 2 otherwise.
 *
 *  The format of each line in the error file is:
 *
 *  <caret pointing to column>
 *	"<file>",<line>  Error[<error number>]: <message>
 *
 *  For example:
 *
 *  it call_in_flag;
 *  --------------^
 *	"delta.c",13  Error[32]: Invalid declaration syntax, unexpected identifier
 */

#define DONE_ARCHIMEDES if (1) {MovDown(1); MovEOL(); StrFree(msgbase); return(retval);} else

int DLL
_ArchimedesErrorInfo( void )
{
	LPSTR msgbase = NULL, msgp;
	long col;
	int retval = 0;

	/* assign initial values	*/

	ErrorLine(0);
	ErrorColumn(0);

	if (!SrchFind("^\".*\",[0-9]+[\t ]+(Error|Warning)\\[[0-9]+\\]:", ERRPARSE_SEARCH, NIL ))
		DONE_ARCHIMEDES;
	if (!(msgbase = BufReadStr(0x7FFFFFFF)))
		DONE_ARCHIMEDES;

	if (!MovUp(1))
		DONE_ARCHIMEDES;
	while (BufReadChar() == '-')
		MovRight(1);
	ErrorColumn(BufQCurrentCol());

	for (msgp = msgbase + 1, col = 0; msgp[col] != '"'; ++col)
		;
	msgp[col] = 0;
	ErrorSrcFile(msgp);
	msgp += (col + 2);
	for (col = 0; isdigit(msgp[col]); ++col)
		;
	msgp[col] = 0;
	ErrorLine(atol(msgp));
	msgp += (col + 1);
	while (isspace(*msgp))
		++msgp;
	if (!strncmp(msgp, "Error", 5))
		retval = 2;
	else
		retval = 1;
	while (*msgp && *msgp != ':')
		++msgp;
	while (*msgp == ':' || isspace(*msgp))
		++msgp;
	ErrorMsgLine(msgp);
	DONE_ARCHIMEDES;
}

#undef DONE_ARCHIMEDES

/*
 *
 *  This is the error info function to process the error file generated
 *  by the Microsoft/MicroFocus Cobol compiler (version 5.0).  This
 *  parser assumes that no errors are warnings; it returns 0 or 2.
 *
 *  The format of each line in the error file is:
 *
 *  <line (6-column field)><source text (with first 6 columns discarded)>
 *  *<error code><string of *s pointing to error column> <** at cols 79 & 80>
 *  **    <message>
 *
 *
 *  For example:
 *
 *    3593     05 FILLER                 PIC X(7)    VALUE "^graph ".
 *  *1014-E******                                                                **
 *  **    Period missing. Period assumed.
 *
 */


int DLL
_MSCobolErrorInfo( void )
{
	LPSTR srchStr = "^\\*[ \t]*\\c[0-9]+[-][A-Z]\\*+[ \\t]+\\*";
	LPSTR msgbase, msgp;		// for handling message lines
	LPSTR srcName = NULL;

	srcName = getSourceBufferName();

	ErrorSrcFile(srcName);

	if (!SrchFind( srchStr, ERRPARSE_SEARCH, NIL ))
		return 0;			// no more error messages
	MarkSavePos();

	msgbase = BufReadStr(0x7FFFFFFF);
	msgp = strrchr(msgbase, '*');
	while (msgp > msgbase && *msgp == '*')
		--msgp;
	while (msgp > msgbase && isspace(*msgp))
		--msgp;
	ErrorColumn(msgp - msgbase);
	StrFree(msgbase);

	if (!MovUp(1))
	{
		MarkRestorePos();
		return(0);
	}
	MovHome();
	msgp = msgbase = BufReadStr(0x7FFFFFFF);

	while (*msgp && !isdigit(*msgp))
		++msgp;

	ErrorLine(atol(msgp));
	StrFree(msgbase);

	if (!MovDown(2))
	{
		MarkRestorePos();
		return(0);
	}
	MovHome();
	msgp = msgbase = BufReadStr(0x7FFFFFFF);
	while (*msgp && (*msgp == '*' || isspace(*msgp)))
		++msgp;
	ErrorMsgLine(msgp);			// register error message
	StrFree(msgbase);
	StrFree(srcName);
	MarkRestorePos();
	MovEOL();
	return(2);
}


/*
 *
 *  This is the error info function to process the error file generated
 *  by the Oracle SQL->Cobol precompiler.  This parser assumes that no
 *  errors are warnings; it returns 0 or 2.
 *
 *  The file name is given just after the banner.
 *
 *  The format of each line in the error file is:
 *
 *  <line (6-column field)><source text (with first 6 columns discarded)>
 *  <line (6-column field)><dots leading to carat indicating error column>
 *  <6 spaces><error code>:  <error message>
 *
 *
 *  For example:
 *
 *
 *  ORACLE Precompiler: Version 1.3.17.0.8 - Production on Wed Aug 18 07:59:40 1993
 *
 *  Copyright (c) Oracle Corporation 1979, 1989.  All rights reserved.
 *
 *
 *  Precompiling INVOICE.PCO
 *   623
 *   623 ...........................................^
 *       PCC-S-0002:  Invalid syntax at column 44 in line 684 of file INVOICE.PCO
 *  5974            END-EXEC.
 *  5974 ...........^
 *       PCC-W-0021:  Oracle Error: ORA-00947: not enough values
 *  5974
 *  5974 ...........^
 *       PCC-S-0017:  Unable to parse statement at line 5974 in file INVOICE.PCO
 */

#define DONE_ORACLE_SQL	if (1){StrFree(msgbase); ename[0] = 0; return(0);} else

int DLL
_OracleSQLErrorInfo( void )
{
	LPSTR msgbase = NULL, msgp;		// for handling message lines
	static char ename[_MAX_PATH + 1] = "";	// pathname of source file

	if (!ename[0])
	{
		if (!SrchFind("Precompiling \\c", ERRPARSE_SEARCH, NIL))
		{
			ename[0] = 0;
			return(0);
		}
		msgp = msgbase = BufReadStr(0x7FFFFFFF);
		while (*msgp && !isspace(*msgp))
			++msgp;
		*msgp = 0;
		strncpy(ename, msgbase, _MAX_PATH);
		StrFree(msgbase);
		msgbase = NULL;
	}
	ErrorSrcFile(ename);

	if (!SrchFind("^[ \\t]*\\c[0-9]+[ \\t]*\\.*\\^", ERRPARSE_SEARCH, NIL ))
		DONE_ORACLE_SQL;			// no more error messages

	msgp = msgbase = BufReadStr(0x7FFFFFFF);
	while (*msgp && isdigit(*msgp))
		++msgp;
	*msgp = 0;
	if (msgp == msgbase)
		DONE_ORACLE_SQL;
	// OK, we know there's an error now.
	ErrorLine(atol(msgbase));
	while (*msgp != '^')
		++msgp;
	if (msgp > msgbase)
		ErrorColumn((msgp - msgbase) - 1);
	StrFree(msgbase);
	MovDown(1);
	MovHome();
	msgp = msgbase = BufReadStr(0x7FFFFFFF);
	while (isspace(*msgp))
   		++msgp;
   	ErrorMsgLine(msgp);
   	StrFree(msgbase);
   	return(2);
}

#undef DONE_ORACLE_SQL

/*
 ** _MapInfoErrorInfo
 *
 *  This is the error info function to process the error files generated
 *  by the MapInfo Basic compiler.
 *  The function returns 0 for no more errors found, or 2 if one was found.
 *
 *  The format of each line in the error file is:
 *
 *  (<file>:<line>) <message>. <message>. ...
 *
 *  Note that many messages may be associated with one error line.
 *  This parser treats each one separately.
 *
 *  For example:
 *
 *  (acad.mb:314) Unrecognized command: sub.  Too many errors.  Error information has been lost.
 *
 */

int DLL
_MapInfoErrorInfo( void )
{
	long col;
	int retval = 0;
	static LPSTR curErrBase = NULL, curErrPtr = NULL;
	const LPSTR searchStr = "\\(\\c.*\\..*:[0-9]+\\) +";

	if (!curErrPtr)
		curErrPtr = curErrBase;
	if (curErrBase && (col = StrMatch("[^ ]", curErrPtr, ERRPARSE_SEARCH, NIL)))
	{
		curErrPtr = curErrPtr + (col - 1);
	}
	else
	{
		StrFree(curErrBase);						// OK if curErrBase is NIL
		if (!SrchFind(searchStr, ERRPARSE_SEARCH, NIL ))
		{
			curErrBase = NULL;
			return(0);
		}
		if (!(curErrPtr = curErrBase = BufReadStr(0x7FFFFFFF)))
			return(0);
		for (col = 0; curErrBase[col] && curErrBase[col] != ':'; ++col)
			;
		curErrBase[col] = 0;
		ErrorSrcFile(curErrBase);
		curErrPtr = curErrBase + col + 1;
		for (col = 0; curErrPtr[col] != ')'; ++col)
			;
		curErrPtr[col] = 0;
		ErrorLine(atol(curErrPtr));
		curErrPtr += (col + 1);
		while (isspace(*curErrPtr))
			++curErrPtr;
	}

	// now curErrPtr points to the beginning of the next message.

	if ((col = StrMatch(". ", curErrPtr, SEARCH_FORWARD, NIL)) >= 1)
	{
		curErrPtr[col] = 0;
		ErrorMsgLine(curErrPtr);
		curErrPtr += (col + 1);
	}
	else
	{
		ErrorMsgLine(curErrPtr);
		StrFree(curErrBase);
		curErrBase = 0;
	}
	return(2);
}

/*
 ** _Asm96ErrorInfo
 *
 *  This is the error info function to process the error files generated
 *  by the R-Byte Asm96 assemlber.
 *  The function returns 0 for no more errors found, or 2 for an error.
 *
 *  Output is in a 3-section format.  The first section displays the
 *  names of the source and object files and command-line switches,
 *  followed by a table of error numbers and line numbers.  The second
 *  section shows the symbol table and the third section shows messages
 *  associated with the error numbers in secton 1.  The messages in
 *  section 3 are listed in numeric order, not in order of occurrence.
 *  The sections are demarcated by a copyright notice followed by a
 *  date/time/page# header.  I don't know whether fields are separated by
 *  spaces or tabs.
 *
 *  An example of the file:
 *
 *  (c) Copyright R-BYTE 1991 .\OBJ\JUNK.OBJ
 *                        08/31-93 15:07:16 PAGE   1
 *
 *  DOS 5.0 (038-N) (c) Copyright R-BYTE 1990,1991
 *
 *  SOURCE FILE: .\SRC\JUNK.SRC
 *  OBJECT FILE: .\OBJ\JUNK.OBJ
 *  ...
 *  ERR LOC  OBJECT                LINE        SOURCE STATEMENT
 *                                    1   ;
 *                                    2   ; This file demonstrates asm96 errors
 *                                    3   ;
 *  #34                               4           ld      0,0
 *  #10                               5           trc
 *  #30                               6           st      r0,junk
 *  #33
 *  (c) Copyright R-BYTE 1991 .\OBJ\JUNK.OBJ
 *                        08/31-93 15:07:16 PAGE   2
 *
 *  SYMBOL TABLE AND CROSS-REFERENCE LISTING
 *  ...
 *  (c) Copyright R-BYTE 1991 .\OBJ\JUNK.OBJ
 *                        08/31-93 15:07:16 PAGE   3
 *
 *  ERRORS DETECTED
 *
 *  #10  SYNTAX ERROR
 *  #30  INVALID LINE TERMINATOR
 *  #33  PREMATURE END OF FILE
 *  #34  STATEMENT NOT ALLOWED IN THIS CONTEXT
 *
 *  ASSEMBLY COMPLETED,    4 ERROR(s) FOUND.
 *
 */

#define DONE_ASM_96	if (1){StrFree(msgbase); StrFree(errorTag); \
	MarkRestorePos(); return(0);} else

int DLL
_Asm96ErrorInfo( void )
{
	int retval = 0;
	LPSTR msgp, msgbase;
	LPSTR errorTag = NULL;
	const LPSTR searchStr = "^#[0-9]+[ \t]*";

	if (!SrchFind(searchStr, ERRPARSE_SEARCH, NIL))
		return(0);

	if (!(msgp = msgbase = BufReadStr(0x7FFFFFFF)))
		return(0);
	while (isdigit(*++msgp))
		;
	*msgp = 0;
	errorTag = StrNew(msgbase);
	while (*++msgp && !isdigit(*msgp))
		;
	if (msgp)
		ErrorLine(atol(msgp));
	StrFree(msgbase);
	msgbase = NULL;
	MovEOL();
	MarkSavePos();
	// go back and get the source file name from the top of Section 1
	if (!SrchFind("SOURCE FILE: \\c", SEARCH_REGEX, NIL))
	{
		ErrorSrcFile(NULL);
	}
	else
	{
		if (!(msgp = msgbase = BufReadStr(0x7FFFFFFF)))
			DONE_ASM_96;
		ErrorSrcFile(msgbase);
		StrFree(msgbase);
		msgbase = NULL;
	}
	// jump forward to get the message associated with the error ID
	MovEOF();
	if (!SrchFind(errorTag, 0, NIL))
	{
		ErrorMsgLine(errorTag);
	}
	else
	{
		if (!(msgp = msgbase = BufReadStr(0x7FFFFFFF)))
			DONE_ASM_96;
		while (isdigit(*++msgp))
			;
		while (isspace(*msgp))
			++msgp;
		ErrorMsgLine(msgp);
		StrFree(msgbase);
		msgbase = NULL;
	}
	StrFree(errorTag);
	MarkRestorePos();
	return(2);
}

#undef	DONE_ASM_96

/*
 ** _PLM96ErrorInfo
 *
 *  This is the error info function to process the listing files generated
 *  by the PL/M-96 compiler.
 *  The function returns 0 for no more errors found, 1 for a warning,
 *  or 2 for an error.
 *
 *  Output is in the form of a listing file interspersed with error
 *  messages.  The name of the listing file is the basename of the
 *  source file with a .LST extension.  Statements in the listing file
 *  are numbered.  Error messages are printed immediately below the
 *  statements to which they refer.  The error message provides the
 *  error number, statement number, line number, and error description.
 *  Also provided is a specimen of code which gives an approximate indication
 *  of the error column.  The LST file is offset 14 columns wrt the source.
 *
 *  This parser ignores the context information for now.  Maybe someday.
 *
 *  This parser reads the LST file, grabs the line number and error
 *  description, and brings up the source file in the right spot.
 *
 *  Format of error line:
 *
 *  *** <ERROR|WARNING> <id> IN <statement #> (LINE <line>), NEAR '<context>': <msg>
 *
 *  or
 *
 *  *** <ERROR|WARNING> <id> IN <statement #> (<includename>,<line>), NEAR '<context>': <msg>
 *
 *  Example:
 *
 *  215   3              THEN DO I = 0 TO ;
 *  *** ERROR 41 IN 215 (LINE 117), NEAR 'I': INSERTED: " <CONST>"
 *
 *  or
 *
 *  215   3              THEN DO I = 0 TO ;
 *  *** ERROR 41 IN 215 (FILE.INC,117), NEAR 'I': INSERTED: " <CONST>"
 */

#define DONE_PLM_96	if (1){StrFree(msgbase); msgbase = NULL; \
	StrFree(srcName); srcName = NULL; return(0);} else

int DLL
_PLM96ErrorInfo( void )
{
	int retval = 0;
	LPSTR p, msgp, msgbase = NULL;
	LPSTR srcName = NULL;
	const LPSTR searchStr = "^\\*\\*\\* \\c";
	LPSTR repExtension(LPSTR, LPSTR, BOOL);

	if (!SrchFind(searchStr, ERRPARSE_SEARCH, NIL))
		DONE_PLM_96;
	if (!(msgp = msgbase = BufReadStr(0x7FFFFFFF)))
		DONE_PLM_96;
	if (!strncmp(msgbase, "ERROR", 5))
		retval = 2;
	else if (!strncmp(msgbase, "WARNING", 7))
		retval = 1;
	else
	{
		DONE_PLM_96;
	}
	if (!(msgp = strchr(msgbase, '(')))
	{
		DONE_PLM_96;
	}
	srcName = getSourceBufferName();
	/* make sure we get the source file, not the .LST file	*/
	srcName = repExtension(srcName, "PLM", TRUE);

	if (!strncmp(msgp, "(LINE ", 6))
	{
		msgp += 6;
		if (*msgp)
			ErrorLine(atol(msgp));
		ErrorSrcFile(srcName);
		p = msgp;
	}
	else
	{
		LPSTR incName;

		if (p = strchr(++msgp, ','))
			*p = 0;
		if (incName = StrNew((LPSTR) LibFunctionExec("ExecQCmndDir")))
		{
			incName = StrAppend(incName, "\\");
			incName = StrAppend(incName, msgp);
		}
		else
			incName = StrNew(msgp);
		ErrorSrcFile(incName);
		if (*++p)
			ErrorLine(atol(p));
		msgp = p;
		StrFree(incName);
	}
	if (p = strchr(p, ':'))
		if (*++p && *++p)
			ErrorMsgLine(p);
	MovEOL();
	StrFree(msgbase);
	StrFree(srcName);
	return(retval);
}

#undef	DONE_PLM_96

/*
 ** _IC96ErrorInfo
 *
 *  This is the error info function to process the listing files generated
 *  by the iC-96 compiler.
 *  The function returns 0 for no more errors found, 1 for a warning,
 *  or 2 for an error.  Compiler "remarks" are treated as warnings.
 *
 *  Output is in the form of a listing file interspersed with error
 *  messages.  The name of the listing file is the basename of the
 *  source file with a .LST extension.  Source lines in the listing file
 *  are numbered.  Error messages are printed immediately below the
 *  lines to which they refer.  The error message provides the
 *  line number, file name, and error description.
 *
 *  This parser reads the LST file, grabs the line number and error
 *  description, and brings up the source file in the right spot.
 *
 *  Format of error line:
 *
 *  *** <Error|Warning|Remark> at line <line>), of <file>: <message>
 *
 *   Line Level  Incl
 *     19              #include"global.h"
 *  *** Error at line 105 of global.h: syntax error near '}'
 *     21     1           t[i].integra = (long)t[i].valve*11L+t[i].p_der
 *  *** Error at line 21 of CONTROL.C: Invalid member name
 *
 */

#define DONE_IC_96	if (1){StrFree(msgbase); msgbase = NULL; \
	MovEOL(); return(retval);} else

int DLL
_iC96ErrorInfo( void )
{
	int retval = 0;
	LPSTR p, msgp, msgbase = NULL;
	const LPSTR searchStr = "^\\*\\*\\* \\c";

	if (!SrchFind(searchStr, ERRPARSE_SEARCH, NIL))
		DONE_IC_96;
	if (!(msgp = msgbase = BufReadStr(0x7FFFFFFF)))
		DONE_IC_96;
	if (!strncmp(msgp, "Error", 5))
		retval = 2;
	else if (!strncmp(msgp, "Warning", 7) || !strncmp(msgp, "Remark", 6))
		retval = 1;
	else
		DONE_IC_96;
	while (*msgp && !isdigit(*msgp))
		++msgp;

	if (!*msgp)
		DONE_IC_96;
	ErrorLine(atol(msgp));
	while (isdigit(*msgp))
		++msgp;
	if (strncmp(msgp, " of ", 4))
		DONE_IC_96;
	msgp += 4;
	for (p = msgp; *p && *p != ':'; ++p)
		;
	if (!*p)
		DONE_IC_96;
	*p = 0;
	ErrorSrcFile(msgp);
	msgp = p + 1;
	while (*msgp && isspace(*msgp))
		++msgp;
	ErrorMsgLine(msgp);
	MovEOL();
	StrFree(msgbase);
	return(retval);
}

#undef	DONE_IC_96

/*
 ** _BorlandMakeErrorInfo
 *
 *  This is the error info function to process the error files generated
 *  by Borland's Make utility.
 *  The function returns 2 if an error is indicated, or 0 if no errors
 *  were found.  Since Codewright has no notion of a project makefile,
 *  this parser always specifies the error file as the source file.
 *  It is intended to be used in the Error Parser 3 slot, after BorlandCPP
 *  and Tasm parsers.
 *
 *  The format of each error message is
 *
 *  <(Fatal|Error|Warning)>: <error message>
 *
 *  If there were errors, a message of this form appears at the end of
 *  the Make output:
 *
 *  ** error <errorID> ** <message>
 *
 *  For example:
 *
 *  MAKE Version 3.6 Copyright (c) 1992 Borland International
 *
 *  Available memory 27025408 bytes
 *
 *       1 file(s) copied
 *       BCC286@aimst100.rsp
 *  Borland C++ Version 3.1 Copyright (c) 1992 Borland International
 *  Fatal: Don't know how to make file 'aimst100.obj'
 *
 *       Available memory 4064204
 *  BCC286: 2.5-=Copyright(C) 1986-92 Phar Lap Software, Inc.
 *  Serial Number W1-12250
 *
 *  **error 1 ** deleting d:\54pos\aim\obj\aimst100.obj
 *
 */

int DLL
_BorlandMakeErrorInfo( void )
{
	int retval = 0;
	LPSTR msgp, msgbase = NULL;
	const LPSTR searchStr = "^(Fatal|Error|Warning): ";

	if (!SrchFind(searchStr, ERRPARSE_SEARCH, NIL ))
		return(0);
	if (!(msgp = msgbase = BufReadStr(0x7FFFFFFF)))
		return(0);
	if (!strncmp(msgp, "Fatal: ", 7) || !strncmp(msgp, "Error: ", 7))
	{
		retval = 2;
		msgp += 7;
	}
	else if (!strncmp(msgp, "Warning: ", 9))
	{
		retval = 1;
		msgp += 9;
	}
	else
		retval = 0;
	ErrorSrcFile(NULL);
	StrFree(msgbase);
	return(retval);
}

/*
 ** _PCLintErrorInfo
 *
 *  This is the error info function to process the listing files generated
 *  by the Gimpel PC-Lint C source code checker.
 *  The function returns 0 for no more errors found, 1 for a warning,
 *  or 2 for an error.  "Info" statements are treated as warnings.
 *
 *  Output messages are grouped by module.  Each module's messages are
 *  introduced by a module flag line.  After the line-by-line messages,
 *  there may be a wrap-up flag line which introduces a series of messages
 *  which apply to the whole module.
 *
 *  Messages come in two formats: source-line messages
 *  and module wrap-up messages.  In the former, a line of whitespace
 *  (mixed 8-column tabs and spaces) ends in an underscore, indicating
 *  the column to which the first error message(s) refer.  Then comes
 *  the source line, then one or more message line.  Following this (these)
 *  may come line-independent messages in the wrap-up message format.
 *
 *  The wrap-up format consists of a flag line.  Then come one or more
 *  wrap-up messages.  Then comes the next module flag, if any.
 *
 *  Any line of the output may be folded.  In this case, the next line will
 *  begin with four spaces.  The other kinds of line that begin with four
 *  spaces are the wrap-up flag line and possibly source code lines.
 *
 *  Format of error line:
 *
 *  --- Module:   <file>+single trailing space
 *  <pointer to column>
 *  <source line>
 *  <file>  <line>   (Info|Warning|Error) <number>: <message>
 *  ...
 *
 *      --- Wrap-up for Module: <file>
 *
 *  (Info|Warning|Error) <number>: <message>
 *  ...
 *
 *  Example:
 *
 *  --- Module:   clkticin.c
 *             _
 *  #define    CF    20
 *  clkticin.c  130  Error 40: Undeclared identifier (HALF_WIDTH_HALF_X_START)
 *  clkticin.c  130  Error 40: Undeclared identifier (SECTION_Y_START)
 *  Info 715: space_allocations_index (line 96, file srfse.c) not referenced
 *  Error 10: Expecting }
 *
 *      --- Wrap-up for Module: clkticin.c
 *
 *  Info 766: Header file 'd:\jd\include\graphics.h' not used in module
 *      'clkticin.c'
 *
 */

#define DONE_PCLINT	if (1){StrFree(msgbase); msgbase = NULL; \
	return(retval);} else

int DLL
_PCLintErrorInfo( void )
{
	int retval = 0;
	long col;
	int ch;
	LPSTR msgp, msgbase = NULL;
	const LPSTR searchStr =
	  "(^.*\\..*[ \\t]*[0-9]+[ \\t]*(Error|Warning|Info))|(Error|Warning|Info)";

	MarkSavePos();
	if (!SrchFind(searchStr, ERRPARSE_SEARCH | SEARCH_MAXIMAL, NIL))
		DONE_PCLINT;
	MovHome();
	MarkDropPos(1);
	MarkSavePos();
	if (!(msgp = msgbase = BufReadStr(0x7FFFFFFF)))
	{
		MarkDropPos(1);
		MovEOL();
		DONE_PCLINT;
	}
	/* this block takes care of wrap-up format messages	*/
	if (!strncmp(msgp, "Warning", 7) || !strncmp(msgp, "Error", 5) ||
	  !strncmp(msgp, "Info", 4))
	{
		if (!(SrchFind("^--- Module:[ \\t]*\\c", SEARCH_REGEX, NIL)))
		{
			MarkRestorePos();
			MovEOL();
			DONE_PCLINT;
		}
		/* process the error line first, then do the module name	*/
		if (!strncmp(msgp, "Error", 5))
			retval = 2;
		else
			retval = 1;
		while (*msgp && *msgp != ':')
			++msgp;
		while (*msgp == ':' || isspace(*msgp))
			++msgp;
		ErrorMsgLine(msgp);
		/* Cursor's still at "---Module: \c".  Get filename.	*/
		StrFree(msgbase);
		msgbase = NULL;
		if (!(msgp = msgbase = BufReadStr(0x7FFFFFFF)))
		{
			MarkRestorePos();
			MovEOL();
			DONE_PCLINT;
		}
		while (isspace(*msgp))
			++msgp;
		trim(msgp, " \t");	/* module name may have trailing space	*/
		ErrorSrcFile(msgp);
		MarkRestorePos();
		MovEOL();
		DONE_PCLINT;
	}

	/* the rest of the parser handles source=line format messages	*/
	if (!SrchFind("^[ \\t]*_", SEARCH_REGEX, NIL))
	{
		MarkRestorePos();
		MovEOL();
		DONE_PCLINT;
	}
	/* assume no spaces within tabstops	*/
	for (col = 1; (ch = BufReadChar()) > 0; MovNextChar(1))
	{
		if (ch == '\t')
			col += 8;
		else if (ch == ' ')
			col++;
		else
			break;
	}
	if (ch == '_')
		ErrorColumn(col);
	/* msgp was initialized near the top of this function	*/
	while (*msgp && !isspace(*msgp))
		++msgp;
	if (!*msgp)
	{
		MarkRestorePos();
		MovEOL();
		DONE_PCLINT;
	}
	*msgp++ = 0;
	ErrorSrcFile(msgbase);
	while (*msgp && isspace(*msgp))
		++msgp;
	if (isdigit(*msgp))
	{
		col = 0;
		while (msgp[++col] && !isdigit(msgp[col]))
			;
		if (!msgp[col])
		{
			MarkRestorePos();
			MovEOL();
			DONE_PCLINT;
		}
		msgp[col] = 0;
		ErrorLine(atol(msgp));
		msgp += (col + 1);
	}
	if (!*msgp)
	{
		MarkRestorePos();
		MovEOL();
		DONE_PCLINT;
	}
	while (isspace(*msgp))
		++msgp;
	if (!strncmp(msgp, "Error", 5))
		retval = 2;
	else
		retval = 1;
	while (*msgp && *msgp != ':')
		++msgp;
	if (*msgp == ':')
		++msgp;
	while (isspace(*msgp))
		++msgp;
	ErrorMsgLine(msgp);
	MarkRestorePos();
	MovEOL();
	DONE_PCLINT;
}

#undef	DONE_PCLINT

/*
 ** _ThompsonAWKErrorInfo()
 *
 *  This is the error parser for Thompson AWK.
 *
 *  The format of each line in the error file is:
 *
 *  awk: fatal error in <file> line <line>: <msg>
 *
 *  For example:
 *
 *	awk: fatal error in fu.awk line 2: end of line in  string
 *
 */

#define DONE_THOMPSONAWK if (1) {MovEOL(); StrFree(msgbase); return(retval);} else

int DLL
_ThompsonAWKErrorInfo( void )
{
 	LPSTR msgbase = NULL, msgp;
	int retval = 0;

	/* assign initial values	*/

	ErrorLine(0);
	ErrorColumn(0);

	/* Search through the error file for the next matching error message.
	 */
	if (!SrchFind("awk: fatal error in \\c", ERRPARSE_SEARCH, NIL ))
		DONE_THOMPSONAWK;

	/* Make a copy of the errmsg line
	 */
	if (!(msgbase = BufReadStr( 0x7FFFFFFF )))
		DONE_THOMPSONAWK;

	retval = 2;

	for (msgp = msgbase; *msgp && !isspace(*msgp); ++msgp)
		;
	*msgp++ = 0;
	ErrorSrcFile(msgbase);

	while (*msgp && !isdigit(*msgp))
		++msgp;
	if (!*msgp)
		DONE_THOMPSONAWK;

	ErrorLine(atol(msgp));

	while (*msgp != ':')
		++msgp;

	++msgp;

	while (isspace(*msgp))
		++msgp;
	ErrorMsgLine(msgp);
	DONE_THOMPSONAWK;
}
#undef DONE_THOMPSONAWK

/*
 ** _VHDLErrorInfo()
 *
 *  This is the error parser for the Model Technologies
 *  VHDL compiler.
 *
 *  The format of each line in the error file is:
 *
 *  ERROR: <file>(line): <message>
 *
 *  For example:
 *
 *	ERROR: counter.vhd(21): No feasible entries for infix op: "="
 *
 *  This parser will also recognize the messages that have the
 *  word WARNING instead of ERROR.
 *
 */

#define DONE_VHDL if (1) {MovEOL(); StrFree(msgbase); return(retval);} else

int DLL
_VHDLErrorInfo( void )
{
 	LPSTR msgbase = NULL, msgp;
	int retval = 0;

	/* assign initial values	*/

	ErrorLine(0);
	ErrorColumn(0);

	/* Search through the error file for the next matching error message.
	 */
	if (!SrchFind("(ERROR|WARNING):[ \t]*\\c[^ \t]*\\([0-9]*\\):",
	ERRPARSE_SEARCH | SEARCH_MAXIMAL, NIL ))
		DONE_VHDL;

	/* Make a copy of the errmsg line
	 */
	if (!(msgbase = BufReadStr( 0x7FFFFFFF )))
		DONE_VHDL;

	if (!strncmp(msgbase, "ERROR", 5))
		retval = 2;
	else	
		retval = 1;

	for (msgp = msgbase; *msgp && *msgp != '('; ++msgp)
		;
	*msgp++ = 0;
	ErrorSrcFile(msgbase);
	ErrorLine(atol(msgp));

	while (*msgp != ':')
		++msgp;

	++msgp;

	while (isspace(*msgp))
		++msgp;
	ErrorMsgLine(msgp);
	DONE_VHDL;
}
#undef DONE_VHDL

/*
 ** _PLM51ErrorInfo
 *
 *  This is the error info function to process the listing files generated
 *  by the PL/M-51 compiler.
 *  The function returns 0 for no more errors found, 1 for a warning,
 *  or 2 for an error.
 *
 *  Output is in the form of a listing file interspersed with error
 *  messages.  The name of the listing file is the basename of the
 *  source file with a .LST extension.  Statements in the listing file
 *  are numbered.  Error messages are printed immediately below the
 *  statements to which they refer.  The error message usually provides the
 *  error number, statement number, line number, and error description.
 *
 *  This parser reads the LST file, grabs the line number and error
 *  description, and brings up the source file in the right spot.
 *
 *  Presumably there is a variant of the format for errors that occur
 *  in an include file, in which the name of the file appears.  Since
 *  there's no such sample available to me now, I'll leave for the future
 *  the ability to handle errors in include files.
 *
 *  Format of error line:
 *
 *  *** <ERROR|WARNING> #<id>, STATEMENT #<statement #> (LINE #<line>), NEAR '<context>', <msg>
 *
 *  or
 *
 *  *** <ERROR|WARNING> #<id>, <msg>
 *
 *
 *  Examples:
 *
 *  *** ERROR #71, STATEMENT #53, LINE #77, NEAR 'TXBUF', IDENTIFIER IS OUT OF SCOPE
 *
 *  or
 *
 *  *** ERROR #180, TOO MANY ERRORS, COMPILATION TERMINATED
 */

#define DONE_PLM_51	if (1){StrFree(msgbase); msgbase = NULL; \
	StrFree(srcName); srcName = NULL; return(0);} else

int DLL
_PLM51ErrorInfo( void )
{
	int retval = 0;
	LPSTR msgp, msgbase = NULL;
	LPSTR srcName = NULL;
	const LPSTR searchStr = "^\\*\\*\\* \\c";
	LPSTR repExtension(LPSTR, LPSTR, BOOL);

	if (!SrchFind(searchStr, ERRPARSE_SEARCH, NIL))
		DONE_PLM_51;
	if (!(msgp = msgbase = BufReadStr(0x7FFFFFFF)))
		DONE_PLM_51;
	if (!strncmp(msgbase, "ERROR", 5))
		retval = 2;
	else if (!strncmp(msgbase, "WARNING", 7))
		retval = 1;
	else
		DONE_PLM_51;
	if (!(msgp = strstr(msgbase, "LINE #")))
	{
		if (!(msgp = strchr(msgbase, ',')))
			DONE_PLM_51;
		do ++msgp; while (isspace(*msgp));
		ErrorMsgLine(msgp);
		MovEOL();
		StrFree(msgbase);
		return(retval);
	}
	srcName = getSourceBufferName();
	/* make sure we get the source file, not the .LST file	*/
	srcName = repExtension(srcName, "PLM", TRUE);

	msgp += 6;
	if (*msgp)
		ErrorLine(atol(msgp));
	while (isdigit(*msgp))
		++msgp;
	++msgp;
	if ((msgp = strchr(msgp, ',')) && msgp[1])
		ErrorMsgLine(msgp + 2);

	MovEOL();
	StrFree(msgbase);
	StrFree(srcName);
	return(retval);
}

#undef	DONE_PLM_51

/*
 ** _MSLinkErrorInfo()
 *
 *  This is the error parser for the Microsoft Linker.
 *
 *  The format of each line in the error file is:
 *	<objectfile>(<sourcefile>) : error <error id>: <error message>
 *
 *  For example:
 *	foo.obj(foo.c) : error L2029: '_FGrep': unresolved external reference
 *
 *  MAIN.OBJ(main.cpp) : error L2029: 'public: __far __pascal Song::Song(void)__far' : unresolved external
 *
 */

#define DONE_MSLINK if (1) {MovEOL(); StrFree(msgbase); return(retval);} else

int DLL
_MSLinkErrorInfo( void )
{
	LPSTR msgbase = NULL, msgp;
	int retval = 0;

	/* assign initial values	*/

	ErrorLine(0);
	ErrorColumn(0);

	/* Search through the error file for the next matching error message.
	 */

// foo.obj(foo.c) : error L2029: '_FGrep': unresolved external reference

	if (!SrchFind("\\(\\c.*\\) : (error|warning) L[0-9]+: ",
	ERRPARSE_SEARCH, NIL ))
		DONE_MSLINK;

	/* Make a copy of the errmsg line
	 */
	if (!(msgbase = BufReadStr( 0x7FFFFFFF )))
		DONE_MSLINK;

	for (msgp = msgbase; *msgp != ')'; ++msgp)
		;
	*msgp = 0;
	ErrorSrcFile(msgbase);

	while (*msgp != ':')
		++msgp;
	++msgp;
	while (isspace(*msgp))
		++msgp;
	if (!strncmp(msgp, "error", 5))
		retval = 2;
	else
		retval = 1;
	while (*msgp != ':')
		++msgp;
	++msgp;
	while (isspace(*msgp))
		msgp++;

	ErrorMsgLine(msgp);
	DONE_MSLINK;
}
#undef DONE_MSLINK

/*
 ** _GccErrorInfo()
 *
 *  This is the error parser for the GNU C and C++ compilers.
 *  The formats of the error file are:
 *
 *	<file>:<line>: warning: <message>
 *	<file>:<line>: <message>
 *
 *  For example:
 *
 *	fu.c:2: warning: passing unsigned pointer to signed pointer
 *
 *  or
 *
 *	fu.c: In function `main':
 *  fu.c:11: `k' undeclared (first use this function)
 *  fu.c:11: (Each undeclared identifier is reported only once
 *  fu.c:11: for each function it appears in.)
 *
 *  or
 *
 *  fu.c: In function `main':
 *  fu.c:14: `s' has an incomplete type
 *  fu.c: At top level:
 *  fu.c:7: storage size of `s' isn't known
 *
 *  Error message from the gcc driver are of similar format, but
 *  the filename is replaced by the name of the driver (usually gcc.exe)
 */

#define DONE_GCC if (1) {MovEOL(); StrFree(msgbase); return(retval);} else

int DLL
_GccErrorInfo( void )
{
	LPSTR msgbase = NULL, msgp;
	int retval = 0;

	/* assign initial values	*/

	ErrorLine(0);
	ErrorColumn(0);

	/* Search through the error file for the next matching error message.
	 */

	if (!SrchFind("^[^ \t]*:", ERRPARSE_SEARCH, NIL ))
		DONE_GCC;

	/* Make a copy of the errmsg line
	 */
	if (!(msgbase = BufReadStr( 0x7FFFFFFF )))
		DONE_GCC;

	for (msgp = msgbase; *msgp != ':'; ++msgp)
		;
	*msgp = 0;
	if (!StrMatch(msgbase, "*.(exe|EXE)", ERRPARSE_SEARCH, NULL))
		ErrorSrcFile(msgbase);

	++msgp;
	if (isdigit(*msgp))
	{
		ErrorLine(atol(msgp));
		while (*msgp != ':')
			++msgp;
		++msgp;
	}
		
	while (isspace(*msgp))
		++msgp;
	if (!strncmp(msgp, "warning", 7))
	{
		retval = 1;
		msgp += 8;
	}
	else
		retval = 2;

	ErrorMsgLine(msgp);
	DONE_GCC;
}
#undef DONE_GCC

/*
 ** _CrossCodeErrorInfo()
 *
 *  This is the error parser for the Cross Code C cross-compiler.
 *  The formats of the error file are:
 *
 *	<compiler>:"<file>"line <line>: (warning|info(DD)|error): <message>
 *
 *  For example:
 *
 *  cc68000:"H:\work\appl\bout\boutinit.c"line 72: info(14): no effect
 *  cc68000:"H:\work\appl\bout\boutinit.c"line 73: error: invalid expression
 *  cc68000:"H:\work\appl\bout\boutinit.c"line 92: warning: casting to int
 *
 *  or
 *
 *  CCPRE:"H:\work\appl\bout\boutmain.c"line 758: error: too many macro args
 *
 *  or
 *
 *  CCPRE:"gemi.h"(H:\gemi.c)line 46: error: can't find/read "gemiperf.h"
 *
 *  In the last example, the root+ext name of the include file is given.
 *  The parser must search current dir and include path for the file.
 *
 */

#define DONE_CROSSCODE if (1) {MovEOL(); StrFree(msgbase); StrFree(envp); \
						StrFree(fullFilename); return(retval);} else

int DLL
_CrossCodeErrorInfo(void)
{
	LPSTR msgbase = NULL, msgp, msgFilename = NULL;
	LPSTR fullFilename = NULL;
	LPSTR envp = NULL;
	int retval = 0;
	BOOL searchInc = FALSE;
	LPSTR searchPath(LPSTR file, LPSTR path);

	/* assign initial values	*/

	ErrorLine(0);
	ErrorColumn(0);

	/* Search through the error file for the next matching error message.  */

	if (!SrchFind("^[^ \t]*:\"\\c.*\"(\\(.*\\))?line [0-9]+: (error|info|warning)",
	ERRPARSE_SEARCH, NIL ))
		DONE_CROSSCODE;

	/* Make a copy of the errmsg line */
	if (!(msgbase = BufReadStr( 0x7FFFFFFF )))
		DONE_CROSSCODE;
	
	msgFilename = msgbase;
	msgp = strchr(msgbase, '"');
	*msgp = 0;
	ErrorSrcFile(msgbase);
	if (*++msgp == '(')
		searchInc = TRUE;
	while (*msgp && !isdigit(*msgp))
		++msgp;
	ErrorLine(atol(msgp));
	while (isdigit(*msgp) || *msgp == ':' || isspace(*msgp))
		++msgp;
	
	if (!_strnicmp(msgp, "error", 5))
		retval = 2;
	else if (!_strnicmp(msgp, "warning", 7))
		retval = 1;
	else if (!_strnicmp(msgp, "info", 4))
		retval = 1;
	else
		DONE_CROSSCODE;		// this is just a safety net.
	ErrorMsgLine(msgp);
	
	if (!searchInc)
		DONE_CROSSCODE;
		
	/***  BEGIN INCLUDE PROCESSING  ***/
	/* Source file's not here, so check the INCLUDE path	*/
	if (!FileExists(msgFilename))
	{
		if (!(envp = SysGetEnv("INCLUDE")))
			DONE_CROSSCODE;
		if (!(fullFilename = searchPath(msgFilename, envp)))
			DONE_CROSSCODE;
		else
			ErrorSrcFile(fullFilename);
	}
	DONE_CROSSCODE;
}
#undef DONE_CROSSCODE


/*
 ** _IntermetErrorInfo
 *
 *  This is the error info function to process error generated by
 *  the TaskTools compiler.
 *
 *  The format of the error is:
 *    :FE:<filename>:<line no.>:	<message>
 * or :BE:<filename>:<line no.>:	<message>
 *
 * eg :FE:scan3.c:1993:	TRACKEM undefined, int type assumed
 *
 *
 *  For further information see '_ErrorInfoDefault()'.
 *
 *  This parser was contributed by ATI Unicam.
 *
 */
int DLL
_IntermetErrorInfo( void )
{
	LPSTR srchStr = "^:(B|F)E:.*:\\c";
	int   level;
	LONG  pos;
    LONG  len;
	LPSTR msg;
	LPSTR linenum;
	LPSTR srcName;

	/* Search through the error file for the next matching error message
	 * leaving the cursor located over the error's line number.
	 */
	if (!SrchFind( srchStr, SEARCH_FORWARD | SEARCH_REGEX | SEARCH_IGCASE, NIL ))
		return 0;

	/* Save the position of the line number and make a copy of the 
	 * message line.
	 */
	MarkSavePos();
	MovHome();
	msg = BufReadStr( 0x7FFFFFFF );

	/* Strip out the source file name from the error line.*/

	pos = StrMatch( "^:(B|F)E:\\c.*:", msg, SEARCH_FORWARD | SEARCH_REGEX , &len );
	if (pos)
	{
		ErrorSrcFile( srcName = StrSubStr( msg, (int)pos, (int)len - 1 ));
		StrFree( srcName );
	}
	else
		ErrorSrcFile ("");

	/* The current position is on the error line number.  Read the
	 * text into a string and convert it to a long integer.
	 */
	MarkRestorePos();
	ErrorLine( atol( linenum = BufReadStr( 12 )) );
	StrFree( linenum );
	ErrorColumn( 0L );

	/* Determine the error message text.  In this case, everything
	 * past the second last ':'. this includes the line no.
	 */

	/* I modified the search string to skip the last : and whitespace -DE */
	pos = StrMatch( "^:(B|F)E:.*:.*:[ \t]*\\c", msg,
	SEARCH_FORWARD | SEARCH_REGEX | SEARCH_MAXIMAL, NIL );

	ErrorMsgLine( msg + pos -1);
	StrFree( msg );

	/* Determine the severity of the error.
	 *
	 * return level == 1 for warning
	 *                 2 for error
	 */

	MovHome();

	if (SrchFind("warning",SEARCH_FORWARD | SEARCH_REGEX | SEARCH_IGCASE, NIL ))
      level = 1;   /* warning */
    else
      level = 2;   /* error */

	return level;

} /* end of _IntermetErrorInfo */


/*
 ** _UNIBenchErrorInfo()
 *
 *  This is the error parser for the UNI-Bench Checker
 *  Unisys Cobol syntax checker.  The checker creates its own error
 *  file which is the basename of the source file plus .ERR.  If
 *  no errors are found, the file is created with length 0.
 *
 *	Each error consists of two lines.  The first is the line from the
 *	Cobol source containing the syntax error.  The second line has 
 *	the following format:
 *
 *	Column          Description
 *	------          -------------------------------------------
 *	 1-45           Drive, path, filename, of the file in error
 *	46-50           Line number
 *	52-53           Column number
 *	55-             Checker syntax error message
 *
 *  For example:
 *
 *  030300     IF TEST-RUN                                                   RPP010
 *  c:\ub\sources\RPS010.CBL                     00299 14 ERR 28 : UNDECLARED IDENTIFIER                 
 *
 *  This parser does not distinguish between errors and warnings.
 *  All messages are treated as errors, so the return value is 0 or 2.
 *
 */

#define DONE_UNIBENCH if (1) {MovEOL(); StrFree(msgbase); return(retval);} else

int DLL
_UNIBenchErrorInfo( void )
{
 	LPSTR msgbase = NULL, msgp;
	int retval = 0;

	ErrorLine(0);
	ErrorColumn(0);

	if (!SrchFind("^[^0-9 \t]", ERRPARSE_SEARCH, NIL ))
		DONE_UNIBENCH;

	if (!(msgp = msgbase = BufReadStr( 0x7FFFFFFF )))
		DONE_UNIBENCH;
	
	if (strlen(msgbase) < 55)
		DONE_UNIBENCH;
	
	while (*msgp && !isspace(*msgp))
		++msgp;
		
	*msgp = 0;
	ErrorSrcFile(msgbase);
	ErrorLine(atol(msgbase + 45));
	ErrorColumn(atol(msgbase + 51));
	ErrorMsgLine(msgbase + 54);
	
	retval = 2;
	DONE_UNIBENCH;
}
#undef DONE_UNIBENCH


/*
 ** _FortranErrorInfo()
 *
 *  This is the error parser for the SGI FORTRAN77 compiler.
 *
 *	The error messages are in the following format:
 *
 *  Error on line <line> of <file>: <message>
 *
 *  For example:
 *
 *  Error on line 364  of mgrid.f: wrong number of subscripts on "nsvm"
 *
 */

#define DONE_FORTRAN if (1) {MovEOL(); StrFree(msgbase); return(retval);} else

int DLL
_FortranErrorInfo( void )
{
 	LPSTR msgbase = NULL, msgp, p;
	int retval = 0;

	ErrorLine(0);
	ErrorColumn(0);

	if (!SrchFind("^(Error|Warning) on line [0-9]+ of .*:", ERRPARSE_SEARCH, NIL ))
		DONE_FORTRAN;

	if (!(msgp = msgbase = BufReadStr( 0x7FFFFFFF )))
		DONE_FORTRAN;
	
	if (!strncmp(msgp, "Error", 5))	
		retval = 1;
	else
		retval = 2;
		
	while (*msgp && !isdigit(*msgp))
		++msgp;
	
	ErrorLine(atol(msgp));
	
	while (isdigit(*msgp))
		++msgp;
	if (strncmp(msgp, " of ", 4))
		DONE_FORTRAN;	/* Shouldn't happen	*/
		
	msgp += 4;
	if (!(p = strchr(msgp, ':')))
		DONE_FORTRAN;
	
	*p = 0;
	ErrorSrcFile(msgp);
	while (*++p && isspace(*p))
		;
	ErrorMsgLine(p);	
		
	DONE_FORTRAN;
}
#undef DONE_FORTRAN


/*
 ** Private functions
 */

 
/*
 * Given filename <file> and path string <path> in normal format
 * (directory names separated by semicolons and null-terminated)
 * find the file in the path, generate a full pathname for it,
 * and return an allocated copy.  Return NULL if the file couldn't
 * be found in those directories in the path that could be searched.
 */

LPSTR
searchPath(LPSTR file, LPSTR path)
{
	LPSTR pathp = path;
	char segbuf[_MAX_PATH];
	int len;
	char *nexttok(char *base, char *buf, char *sep);
	
	while (pathp = nexttok(pathp, segbuf, ";"))
	{
		len = strlen(segbuf);
		segbuf[len] = '\\';
		strcpy(segbuf + len + 1, file);
		if (FileExists(segbuf))
			return(StrNew(segbuf));
	}
	return(NULL);
}

/* Look for the next substring in <base> which is delimited by characters
 * in the string <sep>.	 If it's there, copy it into <buf> and return
 * a pointer to the next starting point in <base>.  Otherwise, or on
 * bad arguments, return NULL.  The stdlib function strtok() has similar
 * functionality, but it's less convenient and not reentrant.
 */

char *
nexttok(char *base, char *buf, char *sep)
{
	char *bufstart = buf;

	/* Handle aberrant arguments
	 */
	if (!base || !*base || !buf)
		return(NULL);
	if (!sep)
		sep = "";

	/* Skip any leading separators.
	 */
	while (strchr(sep, *base))
		++base;
	if (!*base)
		return(NULL);
	/* Copy the good stuff.
	 */
	while (*base && !strchr(sep, *base))
		*buf++ = *base++;
	*buf = 0;
	/* See if anything useful has happened.
	 */
	if (buf == bufstart)
		return(NULL);		/* Nope, no tokens left.		*/
	return(base);			/* Yep, give back this token	*/
}

/*
 ** trim
 *
 * Remove characters from the end of a string that belong to a specified
 * character set.
 *
 */
static void
trim(LPSTR str, LPSTR set)
{
	LPSTR p;

	for (p = str + strlen(str); p != str; )
	{
		if (strchr(set, *--p) != NIL)
			*p = '\0';
		else
			break;
	}
}

/*
 ** markReplacePos
 *
 * Assumes MarkSavePos has been called with no balancing MarkRestorePos.
 * Replaces top position on the stack with the current position.
 *
 */

ERRCODE
markReplacePos(void)
{
	ERRCODE retval;

	if (!(retval = MarkDropPos(1)))
		MarkSavePos();
	return(retval);
}

/*
 ** nexttab
 *
 * Given a column position and a tab settings string, return
 * the column position of the next tab stop.
 */

long
bufNextTab(long oldcol)
{
	long nexttab(long oldcol, LPSTR tabstr);

	return(nexttab(oldcol, BufQTabStr()));
}

long
nexttab(long oldcol, LPSTR tabstr)
{
	long newcol;
	LPSTR tabp;
	char interval;

	if (!tabstr || !*tabstr)	// bad tabstr argument
		return(-1);

	for (tabp = tabstr; *tabp; ++tabp)	// is there an explicit stop?
	{
		if (*tabp > oldcol || *tabp >= CHAR_MAX)
			break;
	}
	if (!*tabp || *tabp >= CHAR_MAX)	// no; calculate from interval
	{
		if (--tabp > tabstr && --tabp >= tabstr)	// find interval
			interval = tabp[1] - tabp[0];
		else
			interval = tabstr[0];
		newcol = tabp[1];
	/*
	 * Add intervals to the last explicit stop until we find
	 * a stop beyond the current column.
	 */
		newcol += ((((oldcol - newcol) / interval) + 1) * interval);
		return(newcol);
	}
	else			// yes, there was an explicit stop; return it.
	{
		return((long)*tabp);
	}
}


static LPSTR
getNormalizedExtension( LPSTR fname )
{
	LPSTR ext = NIL;

	if (!fname || !*fname)
		StrSplitpath( BufQFilename(), NIL, NIL, NIL, &ext );
	else
	{
		fname = StrNormalizePath( fname );
		StrSplitpath( fname, NIL, NIL, NIL, &ext );
		StrFree( fname );
	}

	return ext;
}

/*
 * Flips back from the error buffer to the source buffer, grabs
 * the name of the source file, then flip back to the error buffer.
 *
 * The string returned must be freed by the caller.
 */
static LPSTR
getSourceBufferName(void)
{
	LPSTR srcName;
	HBUFFER thisbuf, otherbuf;	// for hopping to source file and back

	if (otherbuf = BufQPrev(FALSE))	// find source buffer
	{
		thisbuf = BufSwitchBuffer(otherbuf);
		srcName = StrNew(BufQFilename());
		BufSwitchBuffer(thisbuf);
	}
	return(srcName);
}

/*
 ** repExtension
 *
 *  Given a filename and a new extension, create a new
 *  filename with the same basename as the old but with
 *  the given extension.  If freeFname is true, the free
 *  the space occupied by the old fname.
 *
 *  The string returned always points to either an allocated
 *  string or to the original string (as in the case wherein
 *  the filename already has the requested extension).
 */


static LPSTR
repExtension(LPSTR fname, LPSTR ext, BOOL freeFname)
{
	LPSTR fnamep;
	LPSTR newFname;

	if (!fname || !*fname)
		return(fname);
	if (!ext)                               // OK to replace ext with nothing
		ext = "";
	if (fnamep = strrchr(fname, '.'))  // disable the extension if any
	{
		*fnamep = 0;
	}
	newFname = StrNew(fname);               // copy the basename
	newFname = StrAppend(newFname, ".");
	newFname = StrAppend(newFname, ext);
	if (freeFname)
		StrFree(fname);
	return(newFname);
}
