
{***************************************************************************\
*       Module loader buffers and file pointer. These variables are static
*       instead of local so that a separate deallocation can be used which
*       will be called before exiting in error situations
\***************************************************************************}

var
    f : file;
    ms3m : PmpModule;
    instPtrs : ^wordArray;
    pattPtrs : ^wordArray;
    smpBuf : pointer;
    tempMem : pointer;




{****************************************************************************\
*
* Function:     s3mFreeModule(module : pointer; SD : pointer) : integer;
*
* Description:  Deallocates a Scream Tracker 3 module
*
* Input:        module : pointer        pointer to module to be deallocated
*               SD : pointer            Sound Device that has stored the
*                                       samples
*
* Returns:      MIDAS error code
*
\****************************************************************************}

function s3mFreeModule(module : pointer; SD : pointer) : integer;
var
    i, error : integer;
    sdev : ^SoundDevice;
begin
    ms3m := module
    sdev := SD;

    if ms3m = NIL then                 { valid module? }
    begin
        ERROR(errUndefined, ID_s3mFreeModule);
        s3mFreeModule := errUndefined;
        exit;
    end;

    { deallocate pattern orders if allocated: }
    if ms3m^.orders <> NIL then
    begin
        error := memFree(ms3m^.orders);
        if error <> OK then
        begin
            ERROR(error, ID_s3mFreeModule);
            s3mFreeModule := error;
            exit;
        end;
    end;

    { deallocate sample used flags: }
    if ms3m^.instsUsed <> NIL then
    begin
        error := memFree(ms3m^.instsUsed);
        if error <> OK then
        begin
            ERROR(error, ID_s3mFreeModule);
            s3mFreeModule := error;
            exit;
        end;
    end;


    if ms3m^.insts <> NIL then       { instruments? }
    begin
        for i := 0 to (ms3m^.numInsts-1) do
        begin
            { If the instrument has been added to Sound Device, remove
               it, otherwise just deallocate the sample if allocated }
            if ms3m&.insts^[i].sdInstHandle <> 0 then
            begin
                error := sdev^.RemInstrument(ms3m^.insts^[i].sdInstHandle);
                if error <> OK then
                begin
                    ERROR(error, ID_s3mFreeModule);
                    s3mFreeModule := error;
                    exit;
                end;
            end
            else
            begin
                if ms3m^.insts^[i].sample <> NIL then
                begin
                    error := memFree(ms3m^.insts^[i].sample));
                    if error <> OK then
                    begin
                        ERROR(error, ID_s3mFreeModule);
                        s3mFreeModule := error;
                        exit;
                    end;
                end;
            end;

{$IFDEF REALVUMETERS}
            #ifdef REALVUMETERS
            { remove VU meter information if used: }
            if realVU = 1 then
            begin
                if ms3m^.insts^[i].sdInstHandle <> 0 then
                begin
                    error := vuRemove(ms3m^.insts^[i].sdInstHandle);
                    if error <> OK then
                    begin
                        ERROR(error, ID_s3mFreeModule);
                        s3mFreeModule := error;
                        exit;
                    end;
                end;
            end;
{$ENDIF}
        end;

        { deallocate instrument structures: }
        error := memFree(ms3m^.insts));
        if error <> OK then
        begin
            ERROR(error, ID_s3mFreeModule);
            s3mFreeModule := error;
            exit;
        end;
    end;

    if (ms3m^.patterns <> NIL) and (ms3m^.pattEMS <> NIL) then
    begin
        for i := 0 to (ms3m^.numPatts-1) do
        begin
            { if the pattern has been allocated, deallocate it - either
                from conventional memory or from EMS }
            if ms3m^.patterns^[i] <> NIL then
            begin
                if ms3m^.pattEMS^[i] = 1 then
                begin
                    error := emsFree(ms3m^.patterns^[i]);
                    if error <> OK then
                    begin
                        ERROR(error, ID_s3mFreeModule);
                        s3mFreeModule := error;
                        exit;
                    end;
                end
                else
                begin
                    error := memFree(ms3m^.patterns^[i]);
                    if error <> OK then
                    begin
                        ERROR(error, ID_s3mFreeModule);
                        s3mFreeModule := error;
                        exit;
                    end;
                end;
            end;
        end;

        { deallocate pattern pointers: }
        error := memFree(ms3m^.patterns);
        if error <> OK then
        begin
            ERROR(error, ID_s3mFreeModule);
            s3mFreeModule := error;
            exit;
        end;

        { deallocate pattern EMS flags: }
        error := memFree(ms3m^.pattEMS);
        if error <> OK then
        begin
            ERROR(error, ID_s3mFreeModule);
            s3mFreeModule := error;
            exit;
        end;
    end;

    { deallocate the module: }
    error := memFree(ms3m);
    if error <> OK then
    begin
        ERROR(error, ID_s3mFreeModule);
        s3mFreeModule := error;
        exit;
    end;

    s3mFreeModule := OK;
end;




{***************************************************************************\
*
* Function:     procedure s3mLoadError(SD : PSoundDevice);
*
* Description:  Stops loading the module, deallocates all buffers and closes
*               the file.
*
* Input:        SD : PSoundDevice       Sound Device that has been used for
*                                       loading.
*
\***************************************************************************}

procedure s3mLoadError(SD : PSoundDevice);
begin
    if f <> NIL then                    { close file if opened }
        Close(f);

    { Attempt to deallocate module if allocated. Do not process errors. }
    if ms3m <> NIL then
        if s3mFreeModule(ms3m, SD) <> OK then
            exit;

    { Deallocate buffers if allocated. Do not process errors. }
    if smpBuf <> NIL then
        if memFree(smpBuf) <> OK then
            exit;
    if tempmem <> NIL then
        if memFree(tempmem) <> OK then
            exit;
    if instPtrs <> NIL then
        if memFree(instPtrs) <> OK then
            exit;
    if pattPtrs <> NIL then
        if memFree(pattPtrs) <> OK then
            exit;
end;




{****************************************************************************\
*
* Function:     s3mLoadModule(fileName : string; SD : pointer;
*                   module : Ppointer) : integer;
*
* Description:  Loads a Scream Tracker 3 module into memory
*
* Input:        fileName : string       name of module file to be loaded
*               SD : pointer            pointer to the Sound Device which will
*                                       store the samples
*               module : Ppointer       pointer to variable which will store
*                                       the module pointer.
*
* Returns:      MIDAS error code.
*               Pointer to module structure is stored in module^.
*
\****************************************************************************}

function s3mLoadModule(fileName : string; SD : pointer; module : Ppointer) :
    integer;

    { memcpy - copies bytes from src to dest }
    procedure memcpy(var dest; var src; bytes : word); assembler;
    asm
            push    ds
            les     di,dest
            lds     si,src
            mov     cx,bytes
            cld
            rep     movsb
            pop     ds
    end;

    { cmpstr - compares memory area m against string s. Returns 0 if
      identical. }
    function cmpstr(var m1; s : string) : integer;
    var
        b : ^byteArray;
        pos : byte;
    begin
        b := @m1;
        cmpstr := 0;
        for pos := 1 to ord(s[0]) do
            if b^[pos-1] <> ord(s[pos]) then
                cmpStr := 1;
    end;


var
    s3mh : s3mHeader;
    s3mi : s3mInstHdr;
    i : integer;
    inst : PmpInstrument;
    pattSize : word;
    pattData : PmpPattern;
    lend : word;
    maxSmpLength : longint;
    error : integer;
    ordersize : word;
    p : pointer;
    numRead : word;

begin
    { point file ptr and buffers to NIL so that s3mLoadError() can be
       called at any point }
    f := NIL;
    ms3m := NIL;
    instPtrs := NIL;
    pattPtrs := NIL;
    smpBuf := NIL;
    tempmem := NIL;


    { Open module file: }
    Assign(f, fileName);
    Reset(f, 1);
    if IOResult <> 0 then
    begin
        ERROR(errFileOpen, ID_s3mLoadModule);
        s3mLoadError(SD);
        s3mLoadModule := errFileOpen;
        exit;
    end;

    { Allocate memory for the module structure: }
    error := memAlloc(SizeOf(mpModule), @ms3m);
    if error <> OK then
    begin
        s3mLoadError(SD);
        ERROR(error, ID_s3mLoadModule);
        s3mLoadModule := error;
        exit;
    end;

    ms3m^.orders := NIL;                 { clear module structure so that }
    ms3m^.insts := NIL;                  { it can be deallocated with }
    ms3m^.patterns := NIL;               { s3mFree() at any point }
    ms3m^.pattEMS := NIL;
    ms3m^.instsUsed := NIL;

    { Read .S3M file header: }
    BlockRead(f, s3mh, SizeOf(s3mHeader), numRead);
    if numRead <> SizeOf(s3mHeader) then
    begin
        ERROR(errFileRead, ID_s3mLoadModule);
        s3mLoadError(SD);
        return errFileRead;
    end;

    { Check the "SCRM" signature in header: }
    if cmpstr(s3mh.SCRM, "SCRM") <> 0 then
    begin
        ERROR(errInvalidModule, ID_s3mLoadModule);
        s3mLoadError(SD);
        return errInvalidModule;
    end;

    memcpy(ms3m^.ID, s3mh.SCRM, 4);     { copy ID }
    ms3m^.IDnum := idS3M;               { S3M module ID }

    memcpy(ms3m^.songName, s3mh.name, 28);  { copy song name }
    ms3m^.songLength := s3mh.songLength;  { copy song length }
    ms3m^.numInsts := s3mh.numInsts;    { copy number of instruments }
    ms3m^.numPatts := s3mh.numPatts;    { copy number of patterns }
    ms3m^.flags := s3mh.flags;          { copy S3M flags }
    ms3m^.masterVol := s3mh.masterVol;  { copy master volume }
    ms3m^.speed := s3mh.speed;          { copy initial speed }
    ms3m^.tempo := s3mh.tempo;          { copy initial BPM tempo }
    ms3m^.masterMult := s3mh.masterMult and 15;     { copy master multiplier }
    ms3m^.stereo := (s3mh.masterMult shr 4) and 1;  { copy stereo flag }
    { copy channel settings: }
    memcpy(ms3m^.chanSettings, s3mh.chanSettings, 32);

    { Allocate memory for pattern orders: (length of pattern orders must be
       even) }
    ordersize = 2 * ((ms3m^.songLength+1) / 2);
    error = memAlloc(orderSize, @ms3m^.orders);
    if error <> OK then
    begin
        s3mLoadError(SD);
        ERROR(error, ID_s3mLoadModule);
        s3mLoadModule := error;
        exit;
    end;

    { Read pattern orders from file: }
    if fread(ms3m^.orders, ordersize, 1, f) <> 1 then
    begin
        ERROR(errFileRead, ID_s3mLoadModule);
        s3mLoadError(SD);
        return errFileRead;
    end;

    { Calculate real song length: (exclude 0xFF bytes from end) }
    for ( i = (ms3m^.songLength - 1); ms3m^.orders[i] := 0xFF; i-- );
    ms3m^.songLength := i + 1;

    { Allocate memory for instrument structures: }
    if ( (error = memAlloc(ms3m^.numInsts * sizeof(mpInstrument),
        (void**) &ms3m^.insts)) <> OK )
    begin
        s3mLoadError(SD);
        ERROR(error, ID_s3mLoadModule);
        s3mLoadModule := error;
        exit;
    end;

    { Clear all instruments: }
    for ( i = 0; i < ms3m^.numInsts; i++ )
    begin
        ms3m^.insts[i].sample := NIL;
        ms3m^.insts[i].sdInstHandle := 0;
    end;


    { Allocate memory for instrument paragraph pointers: }
    if (error = memAlloc(2 * ms3m^.numInsts, (void**) &instPtrs)) <> OK then
    begin
        s3mLoadError(SD);
        ERROR(error, ID_s3mLoadModule);
        s3mLoadModule := error;
        exit;
    end;

    { Read instrument pointers: }
    if fread(instPtrs, 2 * ms3m^.numInsts, 1, f) <> 1 then
    begin
        ERROR(errFileRead, ID_s3mLoadModule);
        s3mLoadError(SD);
        return errFileRead;
    end;

    { Allocate memory for S3M file pattern pointers: }
    if (error = memAlloc(2 * ms3m^.numPatts, (void**) &pattPtrs)) <> OK then
    begin
        s3mLoadError(SD);
        ERROR(error, ID_s3mLoadModule);
        s3mLoadModule := error;
        exit;
    end;

    { Read pattern pointers: }
    if fread(pattPtrs, 2 * ms3m^.numPatts, 1, f) <> 1 then
    begin
        ERROR(errFileRead, ID_s3mLoadModule);
        s3mLoadError(SD);
        return errFileRead;
    end;


    { Allocate memory for pattern pointers: }
    if ( (error = memAlloc(ms3m^.numPatts * sizeof(mpPattern*), (void**)
        &ms3m^.patterns)) <> OK )
    begin
        s3mLoadError(SD);
        ERROR(error, ID_s3mLoadModule);
        s3mLoadModule := error;
        exit;
    end;

    { Allocate memory for pattern EMS flags: }
    if (error = memAlloc(ms3m^.numPatts, (void**) &ms3m^.pattEMS)) <> OK then
    begin
        s3mLoadError(SD);
        ERROR(error, ID_s3mLoadModule);
        s3mLoadModule := error;
        exit;
    end;

    for ( i = 0; i < ms3m^.numPatts; i++ ) { point all unallocated patterns }
        ms3m^.patterns[i] := NIL;          { to NIL for safety }

    { Read all patterns to memory: }
    for ( i = 0; i < ms3m^.numPatts; i++ )
    begin
        { Seek to pattern beginning in file: }
        if fseek(f, 16L * pattPtrs[i], SEEK_SET) <> 0 then
        begin
            ERROR(errFileRead, ID_s3mLoadModule);
            s3mLoadError(SD);
            return errFileRead;
        end;

        { Read pattern length from file: }
        if fread(&pattSize, 2, 1, f) <> 1 then
        begin
            ERROR(errFileRead, ID_s3mLoadModule);
            s3mLoadError(SD);
            return errFileRead;
        end;

        if useEMS = 1 then
        begin
            { Try to allocate EMS memory for pattern: }
            if (error = emsAlloc(pattSize+2, (emsBlock**) &p)) <> OK then
            begin
                { failed - if only EMS memory should be used, or the
                    error is other than out of EMS memory, pass the error
                    on }
                if (forceEMS = 1) || (error <> errOutOfEMS) then
                begin
                    s3mLoadError(SD);
                    ERROR(error, ID_s3mLoadModule);
        s3mLoadModule := error;
        exit;
                end;
                else
                begin
                    { pattern not in EMS: }
                    ms3m^.pattEMS[i] := 0;

                    { try to allocate conventional memory instead: }
                    if (error = memAlloc(pattSize+2, (void**) &p)) <> OK then
                    begin
                        s3mLoadError(SD);
                        ERROR(error, ID_s3mLoadModule);
        s3mLoadModule := error;
        exit;
                    end;
                end;
            end;
            else
            begin
                { Pattern is in EMS - map pattern EMS block to conventional
                   memory and point pattData to it }
                ms3m^.pattEMS[i] := 1;

                { map EMS block to conventional memory and point pattData
                    to the memory area: }
                if ( (error = emsMap((emsBlock*) p, (void**) &pattData))
                    <> OK )
                begin
                    s3mLoadError(SD);
                    ERROR(error, ID_s3mLoadModule);
        s3mLoadModule := error;
        exit;
                end;
            end;
        end;
        else
        begin
            { No EMS memory used - allocate conventional memory for
               pattern: }
            ms3m^.pattEMS[i] := 0;

            if (error = memAlloc(pattSize+2, (void**) &p)) <> OK then
            begin
                s3mLoadError(SD);
                ERROR(error, ID_s3mLoadModule);
        s3mLoadModule := error;
        exit;
            end;

            pattData = p;
        end;

        ms3m^.patterns[i] := p;

        pattData^.length = pattSize;    { save pattern length }

        { Read pattern data from file: }
        if fread(&pattData^.data[0], pattSize, 1, f) <> 1 then
        begin
            ERROR(errFileRead, ID_s3mLoadModule);
            s3mLoadError(SD);
            return errFileRead;
        end;
    end;

    { deallocate pattern file pointers: }
    error := memFree(pattPtrs));
            if error <> OK then
    begin
        s3mLoadError(SD);
        ERROR(error, ID_s3mLoadModule);
        s3mLoadModule := error;
        exit;
    end;
    pattPtrs = NIL;

    { detect number of channels: }
    if (error = s3mDetectChannels(ms3m, &ms3m^.numChans)) <> OK then
    begin
        s3mLoadError(SD);
        ERROR(error, ID_s3mLoadModule);
        s3mLoadModule := error;
        exit;
    end;


    { detect instruments used: }

    if (error = memAlloc(ms3m^.numInsts, (void **) &ms3m^.instsUsed)) <> OK then
    begin
        s3mLoadError(SD);
        ERROR(error, ID_s3mLoadModule);
        s3mLoadModule := error;
        exit;
    end;
    if (error = s3mFindUsedInsts(ms3m, ms3m^.instsUsed)) <> OK then
    begin
        s3mLoadError(SD);
        ERROR(error, ID_s3mLoadModule);
        s3mLoadModule := error;
        exit;
    end;

    { Find maximum sample length: }
    maxSmpLength = 0;
    for ( i = 0; i < ms3m^.numInsts; i++ )
    begin
        { Seek to instrument header in file: }
        if fseek(f, instPtrs[i] * 16L, SEEK_SET) <> 0 then
        begin
            ERROR(errFileRead, ID_s3mLoadModule);
            s3mLoadError(SD);
            return errFileRead;
        end;

        { Read instrument header from file: }
        if fread(&s3mi, sizeof(s3mInstHdr), 1, f) <> 1 then
        begin
            ERROR(errFileRead, ID_s3mLoadModule);
            s3mLoadError(SD);
            return errFileRead;
        end;

        if maxSmpLength < s3mi.length then
            maxSmpLength = s3mi.length;
    end;

    { Check that no instrument is too long: }
    if maxSmpLength > SMPMAX then
    begin
        ERROR(errInvalidInst, ID_s3mLoadModule);
        s3mLoadError(SD);
        return errInvalidInst;
    end;

    { If EMS is used, allocate TEMPSIZE bytes of memory before the sample
       buffer and deallocate it after allocating all temporary loading
       buffers to minimize memory fragmentation }
    if useEMS then
    begin
        if (error = memAlloc(TEMPSIZE, &tempmem)) <> OK then
        begin
            s3mLoadError(SD);
            ERROR(error, ID_s3mLoadModule);
        s3mLoadModule := error;
        exit;
        end;
    end;


    { allocate memory for sample loading buffer: }
    if (error = memAlloc(maxSmpLength, (void**) &smpBuf)) <> OK then
    begin
        s3mLoadError(SD);
        ERROR(error, ID_s3mLoadModule);
        s3mLoadModule := error;
        exit;
    end;

    if useEMS then
    begin
        error := memFree(tempmem));
            if error <> OK then
        begin
            s3mLoadError(SD);
            ERROR(error, ID_s3mLoadModule);
        s3mLoadModule := error;
        exit;
        end;
        tempmem = NIL;
    end;

    for ( i = 0; i < ms3m^.numInsts; i++ )
    begin

        { point inst to current instrument structure }
        inst = &ms3m^.insts[i];

        { Seek to instrument header in file: }
        if fseek(f, instPtrs[i] * 16L, SEEK_SET) <> 0 then
        begin
            ERROR(errFileRead, ID_s3mLoadModule);
            s3mLoadError(SD);
            return errFileRead;
        end;

        { Read instrument header from file: }
        if fread(&s3mi, sizeof(s3mInstHdr), 1, f) <> 1 then
        begin
            ERROR(errFileRead, ID_s3mLoadModule);
            s3mLoadError(SD);
            return errFileRead;
        end;

        { Check if the instrument is valid - not too long, not stereo,
           16-bit or packed }
        if ( (s3mi.length > SMPMAX) || ((s3mi.flags & 6) <> 0) ||
            (s3mi.pack <> 0) )
        begin
            ERROR(errInvalidInst, ID_s3mLoadModule);
            s3mLoadError(SD);
            return errFileRead;
        end;

        memcpy(&inst^.fileName[0], &s3mi.dosName[0], 13); { copy filename }
        memcpy(&inst^.iname[0], &s3mi.iname[0], 28);  { copy inst name }
        inst^.length = s3mi.length;         { copy sample length }
        inst^.loopStart = s3mi.loopStart;   { copy sample loop start }
        inst^.loopEnd = s3mi.loopEnd;       { copy sample loop end }
        inst^.looping = s3mi.flags & 1;     { copy looping status }
        inst^.volume = s3mi.volume;         { copy default volume }
        inst^.c2Rate = s3mi.c2Rate;         { copy C2 playing rate }

        if(inst^.volume > 63) inst^.volume = 63;

        { Check if there is a sample for this instrument - type = 1 (sample),
           instrument signature is "SCRS" and length <> 0 }

        { Check if instrument is used }
        if (ms3m^.instsUsed[i] := 1)
        begin

            if ( (s3mi.type = 1) && (memcmp(&s3mi.SCRS[0], "SCRS", 4) = 0) &&
                (inst^.length <> 0) )
            begin
                { Seek to sample position in file: }
                if fseek(f, s3mi.samplePtr * 16L, SEEK_SET) <> 0 then
                begin
                    ERROR(errFileRead ,ID_s3mLoadModule);
                    s3mLoadError(SD);
                    return errFileRead;
                end;

                { Read sample to loading buffer: }
                if fread(smpBuf, inst^.length, 1, f) <> 1 then
                begin
                    ERROR(errFileRead ,ID_s3mLoadModule);
                    s3mLoadError(SD);
                    return errFileRead;
                end;
            end;

            { Point inst^.sample to NIL, as the instrument is not available
            - only the Sound Device has it }
            inst^.sample = NIL;

            { Add instrument to Sound Device: }
            error = SD^.AddInstrument(smpBuf, smp8bit, inst^.length,
                inst^.loopStart, inst^.loopEnd, inst^.volume, inst^.looping,
                &inst^.sdInstHandle);
            if error <> OK then
            begin
                s3mLoadError(SD);
                ERROR(error, ID_s3mLoadModule);
        s3mLoadModule := error;
        exit;
            end;

            #ifdef REALVUMETERS
            { if real VU meters are used, prepare VU meter information
                for this instrument }
            if realVU then
            begin
                if inst^.looping then
                    lend = inst^.loopEnd;
                else
                    lend = 0;           { no looping - set VU loop end to
                                           zero }

                if ( (error = vuPrepare(inst^.sdInstHandle, smpBuf, inst^.length,
                    inst^.loopStart, lend)) <> OK )
                begin
                    s3mLoadError(SD);
                    ERROR(error, ID_s3mLoadModule);
        s3mLoadModule := error;
        exit;
                end;
            end;
            #endif
        end;
    end;

    { deallocate instrument pointers: }
    error := memFree(instPtrs));
            if error <> OK then
    begin
        s3mLoadError(SD);
        ERROR(error, ID_s3mLoadModule);
        s3mLoadModule := error;
        exit;
    end;
    instPtrs = NIL;

    { deallocate sample loading buffer: }
    error := memFree(smpBuf));
            if error <> OK then
    begin
        s3mLoadError(SD);
        ERROR(error, ID_s3mLoadModule);
        s3mLoadModule := error;
        exit;
    end;
    smpBuf = NIL;

    fclose(f);
    f = NIL;

    *module = ms3m;                     { return module pointer in *module }

    return OK;
end;
