{*      MOD.PAS
 *
 * ProTracker Module Player, v1.00
 *
 * Copyright 1994 Petteri Kangaslampi and Jarno Paananen
 *
 * This file is part of the MIDAS Sound System, and may only be
 * used, modified and distributed under the terms of the MIDAS
 * Sound System license, LICENSE.TXT. By continuing to use,
 * modify or distribute this file you indicate that you have
 * read the license and understand and accept it fully.
*}


unit MOD;


interface



{****************************************************************************\
*       struct modInstHdr
*       -----------------
* Description:  Protracker module instrument header. Note that all word
*               fields are big-endian.
\****************************************************************************}

type
    modInstHdr = Record
        iname : array[0..21] of char;
        slength : word;
        finetune : byte;
        volume : byte;
        repoffset : word;
        replength : word;
    end;



{****************************************************************************\
*       struct modHeader
*       ----------------
* Description:  Protracker module header.
\****************************************************************************}

type
    modHeader = Record
        sname : array[0..19] of char;
        instruments : array[0..30] of modInstHdr;
        songlen : byte;
        restart : byte;
        positions : arary[0..127] of byte;
        sign : array[0..3] of byte;
    end;



{****************************************************************************\
*       struct modChannel
*       -----------------
* Description:  Protracker module player channel data structure
\****************************************************************************}

type
    modChannel = Record
        note : byte;
        inst : byte;
        cmd : byte;
        info : byte;
        comp : byte;

        sample : byte;
        volume : byte;

        period : word;
        snote : word;
        loff : byte;
        coff : byte;
        toperi : word;
        notepsp : byte;
        retrigc : byte;

        status : byte;

        vibcmd : byte;
        vibpos : byte;

        volbar : byte;
        playoff : word;
    end;




{****************************************************************************\
*
* Function:     modLoad(fileName : string; SD : pointer) : pointer;
*
* Description:  .MOD module loading routine
*
* Input:        fileName : string       name of module file to be loaded
*               SD : pointer            pointer to Sound Device into which
*                                       the samples will be added.
*
* Returns:      Pointer to module structure in memory or NULL if failure
*               in loading. loadError set according to the loading error.
*
\****************************************************************************}

function modLoad(fileName : string; SD : pointer) : pointer;




{****************************************************************************\
*
* Function:     modFree(module : pointer; SD : pointer) : integer;
*
* Description:  Deallocates an MOD module
*
* Input:        module : pointer        pointer to module to be deallocated
*               SD : pointer            pointer to Sound Device that has
*                                       stored the samples
*
* Returns:      1 if success, 0 if failure.
*
\****************************************************************************}

function modFree(module : pointer; SD : pointer) : integer;



procedure ConvSMP(sample : pointer; length : word);
function ConvTrack(track : pointer; tType : word) : word;

function modIdentify(module : pointer) : integer;
function modInit(SD : pointer) : integer;
function modClose : integer;
function modPlayModule(module : pointer; channelOffset, channelAmount,
    startPos, endPos : word) : integer;
function modStopModule : integer;
function modSetInt : integer;
function modRemInt : integer;
function modPlay : integer;
procedure modSetPos(pos : word);
procedure modGetInfo(info : pointer);

procedure mpMOD;                        { Protracker Module Player structure }



implementation


USES  sdevice, mmem, EMS, timer;


procedure ConvSMP(sample : pointer; length : word); external;
function ConvTrack(track : pointer; tType : word) : word; external;
function modIdentify(module : pointer) : integer; external;
function modInit(SD : pointer) : integer; external;
function modClose : integer; external;
function modPlayModule(module : pointer; channelOffset, channelAmount,
    startPos, endPos : word) : integer; external;
function modStopModule : integer; external;
function modSetInt : integer; external;
function modRemInt : integer; external;
function modPlay : integer; external;
procedure modSetPos(pos : word); external;
procedure modGetInfo(info : pointer); external;
procedure mpMOD; external;
{$L MOD.OBJ}




(*
{$I-}

{* Size of temporary memory area used for avoiding memory fragmentation
   if EMS is used *}
const
    TEMPSIZE = 8192;

type
    wordArray = array[0..8192] of word;



{****************************************************************************\
*       Module loader buffers and file pointer. These variables are static
*       instead of local so that a separate deallocation can be used which
*       will be called before exiting in error situations
\****************************************************************************}
var
    f : file;
    fileOpen : integer;
    s3mm : PmpModule;
    instPtrs : ^wordArray;
    pattPtrs : ^wordArray;
    smpBuf : pointer;
    tempMem : pointer;




{****************************************************************************\
*
* Function:     s3mFree(s3m : PmpModule; SD : pointer) : integer;
*
* Description:  Deallocates an S3M module
*
* Input:        module : PmpModule      module to be deallocated
*               SD : pointer            Sound Device that has stored the
*                                       samples
*
* Returns:      1 if success, 0 if failure.
*
\****************************************************************************}

function s3mFree(s3m : PmpModule; SD : pointer) : integer;
var
    sdev : ^SoundDevice;
    inst : ^mpInstrument;
    i : integer;
    tmp : integer;

begin
    sdev := SD;

    if s3m = NIL then                   { check if the module pointer is }
        exit;                           { valid }

    if s3m^.orders <> NIL then          { deallocate pattern orders if }
        memFree(s3m^.orders);           { allocated }

    if s3m^.insts <> NIL then           { have instrument structures been }
    begin                               { allocated? }
        for i := 0 to (s3m^.numInsts-1) do
        begin
            inst := @s3m^.insts^[i];

            { If the instrument has been added to Sound Device, remove
              it, otherwise just deallocate the sample if allocated }

            if (inst^.sdInstNum <> 0) and (sdev <> NIL) then
                tmp := sdev^.RemInstrument(inst^.sdInstNum)
            else
                if inst^.sample <> NIL then
                    memFree(inst^.sample);
        end;
        memFree(s3m^.insts);            { deallocate instrument structs }
    end;

    { if pattern pointers and pattern EMS status have been allocated,
      deallocate patterns: }
    if (s3m^.patterns <> NIL) and (s3m^.pattEMS <> NIL) then
    begin
        for i := 0 to (s3m^.numPatts-1) do
        begin
            if s3m^.patterns^[i] <> NIL then
            begin
                { if the pattern has been allocated, deallocate it - either
                  from conventional memory or from EMS }

                if s3m^.pattEMS^[i] = 1 then
                    tmp := emsFree(s3m^.patterns^[i])
                else
                    memFree(s3m^.patterns^[i]);
            end;
        end;
        memFree(s3m^.patterns);         { deallocate pattern pointers }
        memFree(s3m^.pattEMS);          { deallocate pattern EMS flags }
    end;

    memFree(s3m);                       { deallocate module structure }

    s3mFree := 1;
end;



{****************************************************************************\
*
* Function:     s3mLoadError(error : integer; SD : ^SoundDevice);
*
* Description:  Stops loading the module, deallocates all buffers, closes the
*               file and sets loadError variable.
*
* Input:        error : integer         error number
*               SoundDevice *sd         Sound Device that has been used for
*                                       loading.
*
\****************************************************************************}

procedure s3mLoadError(error : integer; SD : PSoundDevice);
var
    result : integer;

begin
    if fileOpen = 1 then                { close file if opened }
        close(f);

    if s3mm <> NIL then                 { deallocate module if allocated }
        result := s3mFree(s3mm, SD);

    if instPtrs <> NIL then             { deallocate instrument pointers }
        memFree(instPtrs);              { if allocated }

    if pattPtrs <> NIL then             { deallocate pattern pointers }
        memFree(pattPtrs);              { if allocated }

    if smpBuf <> NIL then               { deallocate sample buffer if }
        memFree(smpBuf);                { allocated }

    if tempmem <> NIL then              { deallocate temporary memory area }
        memFree(tempmem);               { if allocated }

    loadError := error;
end;


{****************************************************************************\
*
* Function:     s3mLoad(fileName : string; SD : pointer) : PmpModule
*
* Description:  .S3M module loading routine
*
* Input:        fileName : string       name of module file to be loaded
*               SD : pointer            Sound Device into which the samples
*                                       will be added
*
* Returns:      Pointer to module structure in memory or NULL if failure
*               in loading. loadError set according to loading error.
*
\****************************************************************************}

function s3mLoad(fileName : string; SD : pointer) : PmpModule;

    { memcpy - copies bytes from src to dest }
    procedure memcpy(var dest; var src; bytes : word); assembler;
    asm
            push    ds
            les     di,dest
            lds     si,src
            mov     cx,bytes
            cld
            rep     movsb
            pop     ds
    end;

var
    s3mh : s3mHeader;
    s3mi : s3mInstHdr;
    i : integer;
    inst : PmpInstrument;
    pattSize : word;
    pattData : PmpPattern;
    lend : word;
    maxSmpLength : longint;
    result : integer;
    numRead : word;
    orderLen : word;
    sdev : PSoundDevice;

begin
    sdev := SD;

    { point all buffers to NULL and set fileOpen to 0 so that s3mLoadError()
      can be called at any point }
    fileOpen := 0;
    s3mm := NIL;
    instPtrs := NIL;
    pattPtrs := NIL;
    smpBuf := NIL;
    tempMem := NIL;


    { Open module file: }
    Assign(f, fileName);
    Reset(f, 1);
    if IOResult <> 0 then
    begin
        s3mLoadError(errFileOpenError, sdev);
        s3mLoad := NIL;
        exit;
    end;

    { Allocate memory for the module structure: }
    s3mm := memAlloc(sizeof(mpModule));
    if s3mm = NIL then
    begin
        s3mLoadError(errOutOfMemory, sdev);
        s3mLoad := NIL;
        exit;
    end;

    s3mm^.orders := NIL;                { clear module structure so that }
    s3mm^.insts := NIL;                 { it can be deallocated with }
    s3mm^.patterns := NIL;              { s3mFree() at any point }

    { Read .S3M file header: }
    BlockRead(f, s3mh, sizeof(s3mHeader), numRead);
    if numRead <> sizeof(s3mHeader) then
    begin
        s3mLoadError(errFileReadError, sdev);
        s3mLoad := NIL;
        exit;
    end;

    { Check the "SCRM" signature in header: }
    if (s3mh.SCRM[0] <> 'S') or (s3mh.SCRM[1] <> 'C') or
        (s3mh.SCRM[2] <> 'R') or (s3mh.SCRM[3] <> 'M') then
    begin
        s3mLoadError(errInvalidModule, sdev);
        s3mLoad := NIL;
        exit;
    end;

    memcpy(s3mm^.ID, s3mh.SCRM, 4);     { copy ID }
    s3mm^.IDnum := 0;                   { S3M module ID }

    memcpy(s3mm^.songName, s3mh.name, 28);  { copy song name }
    s3mm^.songLength := s3mh.songLength;{ copy song length }
    s3mm^.numInsts := s3mh.numInsts;    { copy number of instruments }
    s3mm^.numPatts := s3mh.numPatts;    { copy number of patterns }
    s3mm^.flags := s3mh.flags;          { copy S3M flags }
    s3mm^.masterVol := s3mh.masterVol;  { copy master volume }
    s3mm^.speed := s3mh.speed;          { copy initial speed }
    s3mm^.tempo := s3mh.tempo;          { copy initial BPM tempo }
    s3mm^.masterMult := s3mh.masterMult and 15; { copy master multiplier }
    s3mm^.stereo := (s3mh.masterMult shr 4) and 1;  { copy stereo flag }

    { copy channel settings: }
    memcpy(s3mm^.chanSettings, s3mh.chanSettings, 32);

    { Allocate memory for pattern orders: (length of pattern orders must be
      even) }
    orderLen :=2 * ((s3mm^.songLength+1) div 2);
    s3mm^.orders := memAlloc(orderLen);
    if s3mm^.orders = NIL then
    begin
        s3mLoadError(errOutOfMemory, sdev);
        s3mLoad := NIL;
        exit;
    end;

    { Read pattern orders from file: }
    BlockRead(f, s3mm^.orders^, orderLen, numRead);
    if numRead <> orderLen then
    begin
        s3mLoadError(errFileReadError, sdev);
        s3mLoad := NIL;
        exit;
    end;

    { Allocate memory for instrument structures: }
    s3mm^.insts := memAlloc(s3mm^.numInsts * sizeof(mpInstrument));
    if s3mm^.insts = NIL then
    begin
        s3mLoadError(errOutOfMemory, sdev);
        s3mLoad := NIL;
        exit;
    end;

    { Clear all instruments: }
    for i := 0 to (s3mm^.numInsts-1) do
    begin
        s3mm^.insts^[i].sample := NIL;
        s3mm^.insts^[i].sdInstNum := 0;
    end;

    { If EMS is used, allocate TEMPSIZE bytes of memory before the sample
      buffer and deallocate it after allocating all temporary loading
      buffers to minimize memory fragmentation }
    if useEMS = 1 then
    begin
        tempmem := memAlloc(TEMPSIZE);
        if tempmem = NIL then
        begin
            s3mLoadError(errOutOfMemory, sdev);
            s3mLoad := NIL;
            exit;
        end;
    end;

    { Allocate memory for instrument paragraph pointers: }
    instPtrs := memAlloc(2 * s3mm^.numInsts);
    if instPtrs = NIL then
    begin
        s3mLoadError(errOutOfMemory, sdev);
        s3mLoad := NIL;
        exit;
    end;

    { Read instrument pointers: }
    BlockRead(f, instPtrs^, 2 * s3mm^.numInsts, numRead);
    if NumRead <> (2 * s3mm^.numInsts) then
    begin
        s3mLoadError(errFileReadError, sdev);
        s3mLoad := NIL;
        exit;
    end;

    { Allocate memory for S3M file pattern pointers: }
    pattPtrs := memAlloc(2 * s3mm^.numPatts);
    if pattPtrs = NIL then
    begin
        s3mLoadError(errOutOfMemory, sdev);
        s3mLoad := NIL;
        exit;
    end;

    { Read pattern pointers: }
    BlockRead(f, pattPtrs^, 2*s3mm^.numPatts, numRead);
    if numRead <> (2 * s3mm^.numPatts) then
    begin
        s3mLoadError(errFileReadError, sdev);
        s3mLoad := NIL;
        exit;
    end;

    { Find maximum sample length: }
    maxSmpLength := 0;
    for i := 0 to (s3mm^.numInsts-1) do
    begin
        { Seek to instrument header in file: }
        Seek(f, longint(instPtrs^[i]) * longint(16));
        if IOResult <> 0 then
        begin
            s3mLoadError(errFileReadError, sdev);
            s3mLoad := NIL;
            exit;
        end;

        { Read instrument header from file: }
        BlockRead(f, s3mi, sizeof(s3mInstHdr), numRead);
        if numRead <> sizeof(s3mInstHdr) then
        begin
            s3mLoadError(errFileReadError, sdev);
            s3mLoad := NIL;
            exit;
        end;

        if maxSmpLength < s3mi.length then
            maxSmpLength := s3mi.length;
    end;

    { Check that no instrument is too long: }
    if maxSmpLength > SMPMAX then
    begin
        s3mLoadError(errInvalidInsts, sdev);
        s3mLoad := NIL;
        exit;
    end;

    { Allocate memory for sample loading buffer: }
    smpBuf := memAlloc(maxSmpLength);
    if smpBuf = NIL then
    begin
        s3mLoadError(errOutOfMemory, sdev);
        s3mLoad := NIL;
        exit;
    end;

    if useEMS = 1 then
    begin
        memFree(tempmem);
        tempmem := NIL;
    end;

    for i := 0 to (s3mm^.numInsts-1) do
    begin
        { point inst to current instrument structure }
        inst := @s3mm^.insts^[i];

        { Seek to instrument header in file: }
        Seek(f, longint(instPtrs^[i]) * longint(16));
        if IOResult <> 0 then
        begin
            s3mLoadError(errFileReadError, sdev);
            s3mLoad := NIL;
            exit;
        end;

        { Read instrument header from file: }
        BlockRead(f, s3mi, sizeof(s3mInstHdr), numRead);
        if numRead <> sizeof(s3mInstHdr) then
        begin
            s3mLoadError(errFileReadError, sdev);
            s3mLoad := NIL;
            exit;
        end;

        { Check if the instrument is valid - not too long, not stereo,
          16-bit or packed }
        if (s3mi.length > SMPMAX) or ((s3mi.flags and 6) <> 0) or
            (s3mi.pack <> 0) then
        begin
            s3mLoadError(errInvalidInsts, sdev);
            s3mLoad := NIL;
            exit;
        end;

        memcpy(inst^.fileName, s3mi.dosName, 13);   { copy filename }
        memcpy(inst^.iname, s3mi.iname, 28);        { copy inst name }
        inst^.length := s3mi.length;        { copy sample length }
        inst^.loopStart := s3mi.loopStart;  { copy sample loop start }
        inst^.loopEnd := s3mi.loopEnd;      { copy sample loop end }
        inst^.looping := s3mi.flags and 1;  { copy looping status }
        inst^.volume := s3mi.volume;        { copy default volume }
        inst^.c2Rate := s3mi.c2Rate;        { copy C2 playing rate }

        { Check if there is a sample for this instrument - type = 1 (sample),
          instrument signature is "SCRS" and length != 0 }
        if (s3mi.itype = 1) and (s3mi.SCRS[0] = 'S') and (s3mi.SCRS[1] = 'C')
            and (s3mi.SCRS[2] = 'R') and (s3mi.SCRS[3] = 'S') and
            (inst^.length <> 0) then
        begin
            { Seek to sample position in file: }
            Seek(f, longint(s3mi.samplePtr) * longint(16));
            if IOResult <> 0 then
            begin
                s3mLoadError(errFileReadError, sdev);
                s3mLoad := NIL;
                exit;
            end;

            { Read sample to loading buffer: }
            BlockRead(f, smpBuf^, inst^.length, numRead);
            if numRead <> inst^.length then
            begin
                s3mLoadError(errFileReadError, sdev);
                s3mLoad := NIL;
                exit;
            end;
        end;

        { Point inst->sample to NULL, as the instrument is not available
           - only the Sound Device has it }
        inst^.sample := NIL;

        { Add instrument to Sound Device: }
        inst^.sdInstNum := sdev^.AddInstrument(smpBuf, smp8bit, inst^.length,
            inst^.loopStart, inst^.loopEnd, inst^.volume, inst^.looping,
            sdSmpConvCopy);
        if inst^.sdInstNum = 0 then
        begin
            s3mLoadError(errInstAddError, sdev);
            s3mLoad := NIL;
            exit;
        end;
    end;

    memFree(instPtrs);                  { deallocate instrument pointers }
    instPtrs := NIL;
    memFree(smpBuf);                    { deallocate sample buffer }
    smpBuf := NIL;

    { Allocate memory for pattern pointers: }
    s3mm^.patterns := memAlloc(4 * s3mm^.numPatts);
    if s3mm^.patterns = NIL then
    begin
        s3mLoadError(errOutOfMemory, sdev);
        s3mLoad := NIL;
        exit;
    end;

    { Allocate memory for pattern EMS flags: }
    s3mm^.pattEMS := memAlloc(s3mm^.numPatts);
    if s3mm^.pattEMS = NIL then
    begin
        s3mLoadError(errOutOfMemory, sdev);
        s3mLoad := NIL;
        exit;
    end;

    { point all unallocated patterns to NULL for safety: }
    for i := 0 to (s3mm^.numPatts-1) do
        s3mm^.patterns^[i] := NIL;

    { Read all patterns to memory: }
    for i := 0 to (s3mm^.numPatts-1) do
    begin
        { Seek to pattern beginning in file: }
        Seek(f, longint(pattPtrs^[i]) * longint(16));
        if IOResult <> 0 then
        begin
            s3mLoadError(errOutOfMemory, sdev);
            s3mLoad := NIL;
            exit;
        end;

        { Read pattern length from file: }
        BlockRead(f, pattSize, 2, numRead);
        if NumRead <> 2 then
        begin
            s3mLoadError(errFileReadError, sdev);
            s3mLoad := NIL;
            exit;
        end;

        if useEMS = 1 then
        begin
            { Try to allocate EMS memory for pattern: }
            s3mm^.patterns^[i] := emsAlloc(pattSize+2);
            if s3mm^.patterns^[i] = NIL then
            begin
                { Failed - if only EMS memory should be used, return
                  failure. Otherwise attempt to allocate conventional memory
                  instead. }
                if forceEMS = 1 then
                begin
                    s3mLoadError(errOutOfMemory, sdev);
                    s3mLoad := NIL;
                    exit;
                end
                else
                begin
                    s3mm^.pattEMS^[i] := 0;
                    s3mm^.patterns^[i] := memAlloc(pattSize+2);
                    pattData := s3mm^.patterns^[i];
                    if pattData = NIL then
                    begin
                        s3mLoadError(errOutOfMemory, sdev);
                        s3mLoad := NIL;
                        exit;
                    end;
                end;
            end
            else
            begin
                { Pattern is in EMS - map pattern EMS block to conventional
                  memory and point pattData to it }
                s3mm^.pattEMS^[i] := 1;
                pattData := emsMap(s3mm^.patterns^[i]);
            end;
        end
        else
        begin
            { No EMS memory used - allocate conventional memory for
              pattern: }
            s3mm^.pattEMS^[i] := 0;
            s3mm^.patterns^[i] := memAlloc(pattSize+2);
            pattData := s3mm^.patterns^[i];
            if pattData = NIL then
            begin
                s3mLoadError(errOutOfMemory, sdev);
                s3mLoad := NIL;
                exit;
            end;
        end;

        pattData^.length := pattSize;   { save pattern length }

        { Read pattern data from file: }
        BlockRead(f, pattData^.data, pattSize, numRead);
        if numRead <> pattSize then
        begin
            s3mLoadError(errFileReadError, sdev);
            s3mLoad := NIL;
            exit;
        end;
    end;

    memFree(pattPtrs);                  { deallocate pattern file pointers }
    pattPtrs := NIL;

    close(f);
    fileOpen := 0;
    loadError := loadOK;
    s3mLoad := s3mm;
end;


{$I+}


*)

END.
