PROCEDURE OnMouse
*--------------------------------------------------------------------
* DESCRIPTION
*   OnMouse demonstrates how you can use the new ON MOUSE in an
*   event loop to click between two Browse displays.  OnMouse
*   uses two DBF files, Transact and Stock.  For each Transact
*   record, there can be zero or more Stock records linked using
*   the OrderId field.  In addition to clicking, OnMouse shows
*   how you can use the F3 or F4 key to click between the Browse
*   windows.
*--------------------------------------------------------------------

  SET TALK OFF
  SET COLOR TO W+/B
  SET STATUS OFF
  USE Transact IN 1
  USE Stock ORDER Order_Id IN 2
  DEFINE WINDOW Transact FROM 2,2 TO 10,77  && Windows for browsing Transact
  DEFINE WINDOW Stock FROM 12,2 TO 19,77    && and Stock files

  @ 22, 36 SAY "  Quit  " COLOR w+/g    && Display the Quit button
  @ 22, 38 SAY "Q" COLOR gr+/g          && Highlite the Q for Alt-Q pick

  @ 1, 2 SAY "Transact: Click on lower browse to view related Stock"
  @ 20, 2 SAY "Stock: Click on upper browse to view new Transact record"
  @ 24, 18 SAY "Click on Quit button or press Alt-Q to exit."

  KEYBOARD "{27}"                       && Get out of browse after display
  DO ShowTran                           && Display the Transact browse
  KEYBOARD "{27}"
  DO ShowStock                          && Display the Stock browse
  ON MOUSE DO MouseHand WITH MROW(), MCOL()
  ON KEY LABEL Alt-Q DO KeyHand WITH "Q"
  ON KEY LABEL F3    DO KeyHand WITH "F3"
  ON KEY LABEL F4    DO KeyHand WITH "F4"
  nCurrent = 1                          && 1 for Transact, 2 for Stock
  lQuit = .F.
  *-- Do the event loop until the Quit button is pressed
  DO WHILE .NOT. lQuit                  && lQuit is set to .T. in MouseHand
    DO CASE
      CASE nCurrent = 1                 && Browse Transact file if 1
        DO ShowTran
      CASE nCurrent = 2                 && Browse Stock file if 2
        DO ShowStock
    ENDCASE
  ENDDO
  ON MOUSE                              && Clear the ON MOUSE
  ON KEY LABEL F3                       && and ON KEY settings
  ON KEY LABEL F4
  ON KEY LABEL Alt-Q

  RELEASE WINDOW Transact
  RELEASE WINDOW Stock
  USE IN 2
  USE IN 1
  SET STATUS ON
RETURN

PROCEDURE MouseHand
PARAMETERS pn_MRow, pn_MCol
*--------------------------------------------------------------------
* DESCRIPTION
*   MouseHand is the mouse event handler for the event loop.
*   MouseHand will respond to the following actions:
*
*     Click on quit button: Set lQuit flag to .T. to exit
*     Click in Transact:    Set nCurrent to 1 if in Stock window
*     Click in Stock:       Set nCurrent to 2 if in Transact window
*
*   MouseHand will then keyboard an Escape to exit the active browse
*   window and return control back to the event loop.
*--------------------------------------------------------------------

  DO CASE
    CASE pn_MRow = 22 .AND. pn_MCol >= 36 .AND. pn_MCol <= 43
      lQuit = .T.
    CASE pn_MRow >= 2 .AND. pn_MRow <= 10 .AND. ;
         pn_MCol >= 2 .AND. pn_MCol <= 77
      nCurrent = 1
    CASE pn_MRow >= 12 .AND. pn_MRow <= 19 .AND. ;
         pn_MCol >= 2 .AND. pn_MCol <= 77
      nCurrent = 2
  ENDCASE
  KEYBOARD "{27}"                       && Escape out of current browse

RETURN
*-- EOP: MouseHand


PROCEDURE KeyHand
PARAMETERS pc_KeyCode
*--------------------------------------------------------------------
* DESCRIPTION
*   KeyHand is the keyboard event handler for the event loop.
*   KeyHand will respond to the following actions:
*
*     On F3:    Flip nCurrent to 1 if 2 or 2 if 1
*     On F4:    Flip nCurrent to 1 if 2 or 2 if 1
*     On Alt-Q: Set lQuit flag to .T. to exit
*
*   KeyHand will then keyboard an Escape to exit the active browse
*   window and return control back to the event loop.
*--------------------------------------------------------------------

  DO CASE
    CASE pc_KeyCode = "Q"
      lQuit = .T.
    CASE pc_KeyCode = "F4"
      nCurrent = IIF( nCurrent = 1, 2, 1 )
    CASE pc_KeyCode = "F3"
      nCurrent = IIF( nCurrent = 1, 2, 1 )
  ENDCASE
  KEYBOARD "{27}"                       && Escape out of current browse

RETURN
*-- EOP: KeyHand


PROCEDURE ShowTran
*--------------------------------------------------------------------
* DESCRIPTION
*   Browse the Transact file inside of a window
*--------------------------------------------------------------------
  SELECT Transact
  BROWSE WINDOW Transact NOCLEAR COMPRESS NOINIT
RETURN
*-- EOP: ShowTran


PROCEDURE ShowStock
*--------------------------------------------------------------------
* DESCRIPTION
*   Browse the Stock records linked to the Transact file inside
*   of a window.  If there is no match, display all the Stock
*   records.
*--------------------------------------------------------------------
  SELECT Stock
  SET KEY TO Transact->Order_id         && Display records that match order
  IF EOF()                              && If no records match, then
    SET KEY TO                          && display all the Stock records
    GO TOP
  ENDIF
  BROWSE WINDOW Stock NOCLEAR COMPRESS NOINIT
RETURN
*-- EOP: ShowStock


