# Copyright (c) 1993 by Sanjay Ghemawat
##############################################################################
# NoteList
#
#	Maintains list of notices for a certain date.
#
# Description
# ===========
# A NoteList displays notices for a particular date.

# Autoload support
proc NoteList {} {}

class NoteList {name parent} {
    set slot(window) $name
    set slot(parent) $parent
    set slot(date) [date today]
    set slot(items) ""
    set slot(focus) ""
    set slot(width) 100
    set slot(iwidth) 100
    set slot(sbar)  0

    frame $name -bd 0
    scrollbar $name.s -relief raised -bd 1 -orient vertical\
	-command [list $name.c yview]
    canvas $name.c -bd 1 -relief raised -yscroll [list $self sbar_set]

    $self configure
    $self background

    pack append $name $name.c {left expand fill}

    # Establish bindings
    $name.c bind bg <1> [list $self new]
    bind $name.c <Configure> [list $self canvas_resize %w %h]
    bind $name.c <Any-KeyPress> [list $parent key %A]

    # Handle triggers
    trigger on add	[list $self change]
    trigger on delete	[list $self remove]
    trigger on change	[list $self change]
    trigger on text	[list $self textchange]
    trigger on exclude	[list $self exclude]
    trigger on include	[list $self rescan]
}

method NoteList set_date {date} {
    set slot(date) $date
    $self rescan
    $slot(window).c yview 0
}

# effects - Cleanup on destruction
method NoteList cleanup {} {
    # We have to be very careful here about making sure callbacks do
    # not occur in the wrong place (i.e. on already deleted objects).

    # Remove triggers as soon as possible
    trigger remove add		[list $self change]
    trigger remove delete	[list $self remove]
    trigger remove change	[list $self change]
    trigger remove text		[list $self textchange]
    trigger remove exclude	[list $self exclude]
    trigger remove include	[list $self rescan]

    # Now unfocus - do not want unfocus callbacks coming back later
    if {$slot(focus) != ""} {
	$slot(window.$slot(focus)) unfocus
    }

    # Should be safe to kill the items now
    foreach item $slot(items) {
	class_kill $slot(window.$item)
    }

    destroy $slot(window)
}

##############################################################################
# Internal Procedures

method NoteList reconfig {} {
    set name $slot(window)
    set slot(width)  [winfo pixels $name "[cal option ItemWidth]c"]
    set slot(iwidth) [expr $slot(width) - 2*[pref itemPad]]

    # Set canvas geometry
    $name.c configure\
	-width $slot(width)\
	-height "[cal option NoticeHeight]c"\
	-confine 1\
	-scrollregion [list 0 0 $slot(width) "[cal option NoticeHeight]c"]
    $name.c yview 0

    $self layout
}

# effects - Compute various dimensions for NoteList
method NoteList configure {} {
    set name $slot(window)

    $self reconfig

    # Allow vertical scrolling with middle mouse button
    #$name.c bind all <2> [list $name.c scan mark 0 %y]
    #$name.c bind all <B2-Motion> [list $name.c scan dragto 0 %y]
}

method NoteList background {} {
    $slot(window).c create rectangle 0 0 1 1\
	-fill ""\
	-outline ""\
	-width 0\
	-tags bg
}

method NoteList new {} {
    if {$slot(focus) != ""} {
	# Just unfocus
	$slot(window.$slot(focus)) unfocus
	return
    }

    if [cal readonly] {
	error_notify [winfo toplevel $slot(window)] "Permission denied"
	return
    }

    set id [notice]
    $id date $slot(date)
    $id earlywarning [cal option DefaultEarlyWarning]
    $id own
    run-hook item-create $id
    cal add $id

    set list $slot(items)
    lappend list $id
    set slot(items) $list

    $self make_window $id
    $slot(window.$id) focus

    trigger fire add $id
}

method NoteList change {item} {
    set list $slot(items)
    if {[$item is note] && [$item contains $slot(date)]} {
        if {[lsearch $list $item] < 0} {
            # Add item
            lappend list $item
            set slot(items) $list
            $self make_window $item
        } else {
            $slot(window.$item) read
        }
        $self layout
        return
    }

    if [lremove list $item] {
        set slot(items) $list
        class_kill $slot(window.$item)
        unset slot(window.$item)
        $self layout
    }
}

method NoteList textchange {item} {
    if {[$item is note] && [$item contains $slot(date)]} {
	$slot(window.$item) read
	$self layout
    }
}

method NoteList exclude {calendar} {
    set oldlist $slot(items)
    set slot(items) ""
    set newlist ""
    foreach item $oldlist {
	if {[$item calendar] == $calendar} {
	    class_kill $slot(window.$item)
	    unset slot(window.$item)
	} else {
	    lappend newlist $item
	}
    }
    set slot(items) $newlist
    $self layout
}

method NoteList remove {item} {
    set list $slot(items)
    if [lremove list $item] {
	set slot(items) $list
	class_kill $slot(window.$item)
	unset slot(window.$item)
	$self layout
    }
}

# args are ignored - they just allow trigger to call us directly.
method NoteList rescan {args} {
    set list $slot(items)
    set slot(items) ""

    foreach appt $list {
        class_kill $slot(window.$appt)
	unset slot(window.$appt)
    }

    set list {}
    cal query $slot(date) $slot(date) item d {
        if [$item is note] {
            lappend list $item
            $self make_window $item
        }
    }
    set slot(items) $list
    $self layout
}

method NoteList layout {} {
    set x [pref itemPad]
    set y 0
    foreach item $slot(items) {
	$slot(window.$item) geometry $x $y $slot(iwidth) 1
	set y [lindex [$slot(window.$item) bbox] 3]
    }

    # Set canvas geometry
    $slot(window).c configure -scrollregion [list 0 0 $slot(width) $y]
}

# Adjust scrollbar.  Unpack it completely if the whole canvas is visible,
method NoteList sbar_set {total size first last} {
    if {$total > $size} {
	# Need scrollbar
	if !$slot(sbar) {
	    pack append $slot(window) $slot(window).s {right filly}
	    set slot(sbar) 1
	}
    } else {
	if $slot(sbar) {
	    pack forget $slot(window).s
	    set slot(sbar) 0
	}
    }

    $slot(window).s set $total $size $first $last
}

method NoteList make_window {item} {
    set w [ItemWindow $slot(window).c $item $slot(date)]
    set slot(window.$item) $w

    $w set_focus_callback   [list $self focus]
    $w set_unfocus_callback [list $self unfocus]
}

method NoteList canvas_resize {w h} {
    $slot(window).c coord bg 0 0 $w [expr [pref itemLineHeight]*100]
}

method NoteList focus {item} {
    set slot(focus) $item
    $self layout

    $slot(parent) focus $item $slot(window.$item)
}

method NoteList unfocus {} {
    set slot(focus) ""
    $self layout
    $slot(parent) unfocus
}
