# $Id: picio.tcl,v 1.12 94/02/14 14:54:52 mangin Exp $

set drawingFile [pwd]
set loadFile    [pwd]
set lastSavedState {{} {} {} {}}

proc setDrawingName {name} {

  .frame1.name configure -text $name
  if {[string length $name] < 10} {
    .frame1.name configure -width 10
  } else {
    .frame1.name configure -width [string length $name]
  }
  update
}

proc SaveDrawing {} {
  global drawingFile

  if {![file writable $drawingFile]} {
    FSBox {Save drawing in file:} {$drawingFile} {
      focus .
      if {[file exists $fsBox(path)/$fsBox(name)]} {
	ConfirmBox "File exists. Save anyway ?" \
	  {save} {
	    global drawingFile
	    set drawingFile $fsBox(path)/$fsBox(name)
	    setDrawingName $fsBox(name)
	    SaveDrawingInFile $drawingFile } \
	  {abort} {}
      } else {
	global drawingFile
	set drawingFile $fsBox(path)/$fsBox(name)
	setDrawingName $fsBox(name)
	SaveDrawingInFile $drawingFile } {}
    } {}
  } else {
    SaveDrawingInFile $drawingFile
  }
}

proc SaveDrawingAs {} {
  global drawingFile
  
  FSBox {Save drawing in file:} {$drawingFile} {
    global drawingFile
    focus .
    if {[file exists $fsBox(path)/$fsBox(name)]} {
      ConfirmBox "File exists. Save anyway ?" \
	{save} {
	  global drawingFile
	  set drawingFile $fsBox(path)/$fsBox(name)
	  setDrawingName $fsBox(name)
	  SaveDrawingInFile $drawingFile } \
	{abort} {}
    } else {
      set drawingFile $fsBox(path)/$fsBox(name)
      setDrawingName $fsBox(name)
      SaveDrawingInFile $drawingFile
    }
  } {}
}

proc SaveDrawingInFile { path } {
  global lastSavedState ImageFile LatexText env

  waitCursor
  
  if {[file exists $path] &&
    (! [file writable $path])} {
      warn "Couldn't open $file for writing"
      restoreCursor
      return
    }

  if {[catch {open $path w} handle]} {
    warn "Open failed : $handle"
    restoreCursor
    return
  }

  #
  #    Write header
  #
  
  global PicassoVersion
  set header "##   Created by picasso version $PicassoVersion   ##"
  puts $handle [replicate "#" [clength $header]]
  puts $handle $header
  puts $handle [replicate "#" [clength $header]]
  puts $handle "\n"
  puts $handle "##  User : $env(USER)"
  puts $handle "##  Date : [fmtclock [getclock] "%B %d %Y %T"]\n\n"
  puts $handle "##"
  puts $handle {## Following the line "<<imagen>> bytesize", image data consists in:}
  puts $handle {##   - sizeof(int) bytes representing image width }
  puts $handle {##   - sizeof(int) bytes representing image height }
  puts $handle {##   - width*height*sizeof(float) bytes of float pixel values between 0.0 and 1.0}
  puts $handle "##\n\n"
  [cv] dtag all current
  set items [[cv] find withtag all]

  #
  #    Write tcl commands 
  #

  puts $handle "###############################"
  puts $handle "####  Start of TCL source  ####"
  puts $handle "###############################\n"

  #  Global variables  #
  puts $handle "global loadVersion"
  if {[lindex $PicassoVersion 0] == ""} {
    puts $handle "set loadVersion Exp"
  } else {
    puts $handle "set loadVersion [lindex $PicassoVersion 0]\n"
  }

  global LatexDrawingRatio
  puts $handle "global loadLatexDrawingRatio"
  puts $handle "set loadLatexDrawingRatio $LatexDrawingRatio\n"
  
  foreach item $items {
    set tags [[cv] gettags $item]

    #  Don't save control items  #
    if {[lmember $tags "ctl"]} {
      continue
    }
    
    #  Just take group and latex tags  #
    set stags {}
    foreach tag $tags {
      if {[string match {tGr*} $tag] ||
	[string match {Gr*} $tag] ||
	($tag == "latex") ||
	($tag == "ltxFr")} {
	  lappend stags $tag
	}
    }

    lappend stags inCreation

    set options [getSwitches $item]
    
    switch [[cv] type $item] {

      image {
	####  image items  ####
	if {[keylget LatexText $item text]} {
	  lappend options {-text} "$text"
	}
	lappend options {-tags} $stags
	puts $handle "loadCreateImage $item [[cv] coords $item] $options\n"
      }

      bitmap {
	####  bitmap items  ####
	lappend options {-tags} $stags
	puts $handle "loadCreateBitmap $item [[cv] coords $item] $options\n"
      }

      text {
	if {[keylget LatexText $item text]} {
	  #  latex items  #
	  set texti [lsearch -exact $options "-text"]
	  incr texti
	  set options \
	    [lreplace $options $texti $texti $text]
	  lappend options {-tags} $stags
	  
	  puts $handle "\[cv\] create text [[cv] coords $item] \
                          $options"
	} else {
	  lappend options {-tags} $stags
	  puts $handle "\[cv\] create [[cv] type $item] [[cv] coords $item] $options\n"
	}
      }
      
      default {
	####  other items  ####
	lappend options {-tags} $stags
	puts $handle "\[cv\] create [[cv] type $item] [[cv] coords $item] $options\n"
      }
    }
  }

  puts $handle "########  End of Tcl source  ########"
  puts $handle "return 0"
  close $handle
  ##  write image and bitmap data  ##

  foreach item $items {
    switch [[cv] type $item] {
      image {
	set picfile [lindex [[cv] itemconfigure $item -path] 4]
	set handle [open $path "a"]
	puts $handle "\n\n####  Image $item data  ####\n"
	puts -nonewline $handle "<<image$item>> [file size $picfile]\n"
	close $handle
	if {[catch {exec cat $picfile >> $path} err]} {
	  warn "Couldn't write image data"
	  continue
	}
	
      }

	bitmap {
	  if {[keylget ImageFile $item xbmfile]} {
	    set handle [open $path "a"]
	    puts $handle "\n\n####  Bitmap $item data  ####\n"
	    puts -nonewline $handle "<<bitmap$item>> [file size $xbmfile]\n"
	    close $handle
	    if {[catch {exec cat $xbmfile >> $path} err]} {
	      warn "Couldn't write bitmap data"
	      continue
	    }
	  } else {
	    warn "Couldn't write bitmap data"
	    continue
	  }
	}
    }
  }

  set lastSavedState [getCurrentState]
  restoreCursor
}

################################################################
####			Loading                             ####
################################################################

#  These procs are called from the drawing files to
#  recreate image and bitmap items, whose date
#  is stored at the end of the file

proc loadCreateBitmap {id args} {
  global loadFile loadVersion
  global ImageFile 

  ##  Create xbm file  ##
  set xbmfile [mkTmpFile xbm]
  set stout [open $xbmfile "w"]
  set stin [open $loadFile "r"]
  gets $stin line
  while {![regexp "<<bitmap$id>>" $line]} {
    if {[gets $stin line] < 0} {
      warn "Couldn't load bitmap data"
      return
    }
  }

  set fsize [lindex $line 1]

  copyfile -bytes $fsize $stin $stout
  close $stin
  close $stout

  set id [eval "[cv] create bitmap $args -bitmap @$xbmfile"]
  keylset ImageFile $id $xbmfile
}

########  Obsolete - can use LatexUncompile instead  ########
#  Conversion of latex items represented as images (until 3.7)
#  into new style latex items (text + frame)
#

#proc convertLatexItem {text x y} {
#  global LatexDims LatexDrawingRatio
#  
#  #  New latex items impose some modifs on the text  #
#  regsub {\\\\\\\\} $text {\\\\} text
#  if {[regsub -all {\\begin\{center\}} $text {} text]} {
#    regsub -all {\\end\{center\}} $text {} text
#  }
#  regsub -all "^\[ \]*\n" $text {} text
#  if {[regsub -all {\\\[} $text {$ \\displaystyle } text]} {
#    regsub -all {\\\]} $text { $} text
#  }
#  
#  #  try to get the bbox  #
#  if {[catch {computeLatexDims $text} dims]} {
#    error
#  }
#
#  set id [[cv] create text $x $y \
#	  -anchor center \
#	  -justify center \
#	  -text "$text"]
#
#  [cv] itemconfigure $id \
#    -tags "tGr.$id Gr.$id inCreation latex"
#
#  keylset LatexDims $id $dims
#  
#  set bbw2 [expr [lindex $dims 0] / (2*$LatexDrawingRatio)]
#  set bbh2 [expr [lindex $dims 1] / (2*$LatexDrawingRatio)]
#
#  [cv] create rectangle \
#    [expr $x - $bbw2] [expr $y - $bbh2] \
#    [expr $x + $bbw2] [expr $y + $bbh2] \
#    -outline green -fill green \
#    -tags "Gr.$id tGr.$id ltxFr"
#
#  return 0
#}

##
##  Image items creation
##  This proc also manages
##  latex image item creation and LatexText updating
##

proc loadCreateImage {id args} {
  global loadFile loadVersion
  global ImageFile LatexText

  set texti [lsearch -exact $args "-text"]
  if {$texti >= 0} {
    set text [lindex $args [expr $texti + 1]]
    set args [lreplace $args $texti [incr texti]]
  }
  
  ##  Create image data file  ##
  set picfile [mkTmpFile image]
  set stout [open $picfile "w"]
  set stin [open "$loadFile" "r"]
  gets $stin line
  while {![regexp "<<image$id>>" $line]} {
    if {[gets $stin line] < 0} {
      warn "Couldn't load image data"
      return
    }
  }

  set fsize [lindex $line 1]

  copyfile -bytes $fsize $stin $stout
  close $stin
  close $stout

  set id [eval "[cv] create image $args -path $picfile"]

  if {$texti >= 0} {
    keylset LatexText $id $text
  }
  keylset ImageFile $id $picfile
}
  
#
#   Interactive Load procs
#

proc IncludeDrawing {} {
  global loadFile drawingFile

  FSBox {Include drawing from file:} {$loadFile} {
    focus .
    IncludeDrawingFromFile $fsBox(path)/$fsBox(name) } {}
}
  
proc IncludeDrawingFromFile {path} {
  global loadFile

  set loadFile $path
  ReadDrawing $path
}  

proc LoadDrawing {} {
  global lastSavedState loadFile drawingFile

  set cstate [getCurrentState]
  if {($lastSavedState != $cstate) &&
    ($cstate != {{} {} {} {}})} {
      global saveCurrent
      ConfirmBox "Save current drawing ?" \
	{save} { global saveCurrent; set saveCurrent 1 } \
	{discard} { global saveCurrent; set saveCurrent 0}
      update
      if {$saveCurrent} {
	SaveDrawing
      }
    }

  FSBox {Load drawing from file:} {$loadFile} {
    focus .
    setDrawingName $fsBox(name)
    LoadDrawingFromFile $fsBox(path)/$fsBox(name) } {}
}

proc LoadDrawingFromFile {path} {
  global loadFile drawingFile

  set loadFile $path
  set drawingFile $path
  
  [cv] delete all
  ReadDrawing $path
}

##
##  ReadDrawing : makes the real job
##  of item creation and back-compatibility stuff
##

proc ReadDrawing {path} {

  waitCursor
  
  if {![file readable $path]} {
    warn "Couldn't open $path for reading"
    restoreCursor
    return
  }

  #  old versions don't set inCreation tag on  #
  #  loaded objects, so have to remember existing objects  #
  set oldIds [[cv] find withtag all]
  
  global loadVersion
  set loadVersion ""
  if {[catch {source $path} err]} {
    warn $err
  }

  if {![regexp {[0-9]*[.][0-9]*[a-z]} $loadVersion]} {
    #  Try to get version number from header  #
    set loadVersion ""
    set st [open $path r]
    loop i 0 5 {
      if {[gets $st line] < 0} {
	break
      }
      if {[regexp {^[ ]*[#]*[a-zA-Z ]*[vV]ersion[ ]+([0-9]*[.][0-9]*)} \
	   $line foo loadVersion]} {
	     break
	   }
    }
    close $st

    if {$loadVersion == ""} {
      warn "No version info found"
      set loadVersion 0
    }
  }

  ####
  ####  Back Compatibility stuff
  ####

  #  remove trailing letter from version string  #
  regexp {[0-9]*[.][0-9]} $loadVersion loadVersion
  
  if {$loadVersion <= 3.4} {
    # image anchor changed to center #
    foreach id [[cv] find withtag all] {
      if {[[cv] type $id] == "image"} {
	set w [lindex [[cv] itemconfigure $id -width] 4]
	set h [lindex [[cv] itemconfigure $id -height] 4]
	[cv] move $id [expr $w/2] [expr $h/2]
      }
    }
  }

  #  add inCreation tags if necessary  #
  if {($loadVersion <= 3.6) } {
    set newIds [lindex [intersect3 $oldIds [[cv] find withtag all]] 2]
    [cv] dtag all inCreation
    foreach id $newIds {
      [cv] addtag inCreation withtag $id
      # there was a bug with toRotate tags up to 3.6 #
      [cv] dtag $id toRotate
    }
  }

  #  update LatexText for latex items  #
  #  also update latex items bounding boxes  #

  global LatexText LatexDims LatexDrawingRatio loadLatexDrawingRatio

  if {($loadVersion <= 3.7) } {
    set loadLatexDrawingRatio 1
  }
  
  foreach id [[cv] find withtag latex inCreation] {

    keylset LatexText $id \
      [lindex [[cv] itemconfigure $id -text] 4]

    set both [[cv] find withtag \
	      [getTopGroupTags $id]]
    if {[lindex $both 0] == $id} {
      set frameId [lindex $both 1]
    } else {
      set frameId [lindex $both 0]
    }

    lassign [[cv] coords $frameId] x1 y1 x2 y2
    set w [expr $x2 - $x1]
    set h [expr $y2 - $y1]
    
    if {$loadLatexDrawingRatio == 1} {
      keylset LatexDims $id "$w $h"
    } else {
      set w [expr $w * $loadLatexDrawingRatio]
      set h [expr $h * $loadLatexDrawingRatio]
      keylset LatexDims $id "$w $h"
    }
    
    if {$loadLatexDrawingRatio != $LatexDrawingRatio} {
      updateLatexBboxes $id
    }

    eval "fitTextInRect $id [[cv] coords $frameId]"
  }

  # Replace obsolete groupList and topGroup tags #
  if {($loadVersion <= 3.7) } {
    foreach id [[cv] find withtag inCreation] {
      set tags [[cv] gettags $id]
      regsub {topGroup} $tags {tGr} tags
      regsub {groupList} $tags {Gr} tags
      [cv] itemconfigure $id -tags $tags
    }
  }    
  
  #  update group tags of newly created objects  #
  updateGroupTags inCreation

  [cv] dtag all inCreation
  [cv] raise ltxFr
  [cv] raise latex
  restoreCursor
}

################################################################
####		   Writing Postscript                       ####
################################################################

set currentWriteFile [pwd]

proc buildWriteDialog { fmt writeCmd } {

  toplevel .psDg
  
  # Window manager configurations
  global tkVersion
  wm positionfrom .psDg ""
  wm sizefrom .psDg ""
  wm maxsize .psDg 1000 900
  wm minsize .psDg 10 10
  wm title .psDg "Picasso Write $fmt"

  grab set .psDg
  
  ####  Label  ####

  label .psDg.titleLb \
    -text {Postscript Writing Options}

  ####  Path  ####

  frame .psDg.pathFr \
    -borderwidth 0

  label .psDg.pathFr.lb \
    -text {Write to file:} \
    -relief flat

  entry .psDg.pathFr.pathEn \
    -relief sunken \
    -width 40

  set cmd FSBox
  lappend cmd "$fmt File Name:"

  lappend cmd { .psDg.pathFr.pathEn delete 0 end
	  .psDg.pathFr.pathEn insert 0 $fsBox(path)/$fsBox(name)
	  grab set .psDg
	  focus . } {}
  
  button .psDg.pathFr.fsbBt \
    -relief raised \
    -text {File Box} \
    -command "
      grab release .psDg
      FSBox {$fmt File Name:} \[.psDg.pathFr.pathEn get]\ {
        .psDg.pathFr.pathEn delete 0 end
	  .psDg.pathFr.pathEn insert 0 \$fsBox(path)/\$fsBox(name)
	  grab set .psDg
	  focus . } {}"

  #  packing  #

  pack append .psDg.pathFr \
    .psDg.pathFr.lb {left padx 12} \
    .psDg.pathFr.pathEn {left} \
    .psDg.pathFr.fsbBt {left padx 12}

  ####  Dimensions  ####

  frame .psDg.dimFr -borderwidth 0

  label .psDg.dimFr.dimLb \
    -text {Region of drawing to be written:} \
    -relief flat
  
  frame .psDg.dimFr.butFr -borderwidth 0
  button .psDg.dimFr.butFr.wholeBt \
    -text {whole drawing} \
    -relief raised \
    -command {
      lassign [[cv] bbox all] x1 y1 x2 y2
      .psDg.dimFr.parFr.entryFr.x1En delete 0 end
      .psDg.dimFr.parFr.entryFr.x1En insert 0 $x1
      .psDg.dimFr.parFr.entryFr.y1En delete 0 end
      .psDg.dimFr.parFr.entryFr.y1En insert 0 $y1
      .psDg.dimFr.parFr.entryFr.widthEn delete 0 end
      .psDg.dimFr.parFr.entryFr.widthEn insert 0 [expr $x2-$x1]
      .psDg.dimFr.parFr.entryFr.heightEn delete 0 end
      .psDg.dimFr.parFr.entryFr.heightEn insert 0 [expr $y2-$y1]
    }

  button .psDg.dimFr.butFr.selBt \
    -text {select region} \
    -relief raised \
    -command {
      set RegionDone 0
      bind .psDg <ButtonPress-1>   {iregionStart %X %Y}
      bind .psDg <ButtonRelease-1> {iregionEnd}
      bind .psDg <Button1-Motion>  {iregion_move %X %Y}
      wm withdraw .psDg
      saveMsg
      msg "Drag region with mouse button 1"
      tkwait var RegionDone
      .psDg.dimFr.parFr.entryFr.x1En delete 0 end
      .psDg.dimFr.parFr.entryFr.x1En insert 0 $RegionX1
      .psDg.dimFr.parFr.entryFr.y1En delete 0 end
      .psDg.dimFr.parFr.entryFr.y1En insert 0 $RegionY1
      .psDg.dimFr.parFr.entryFr.widthEn delete 0 end
      .psDg.dimFr.parFr.entryFr.widthEn insert 0 $RegionWidth
      .psDg.dimFr.parFr.entryFr.heightEn delete 0 end
      .psDg.dimFr.parFr.entryFr.heightEn insert 0 $RegionHeight
      restoreMsg
      wm deiconify .psDg
    }

  pack .psDg.dimFr.butFr.wholeBt \
    -side left -padx 20 -ipadx 6 -ipady 4
  pack .psDg.dimFr.butFr.selBt \
    -side left -padx 20 -ipadx 6 -ipady 4
      
  label .psDg.dimFr.parFr -borderwidth 0
  
  frame .psDg.dimFr.parFr.labelFr -borderwidth 0
  label .psDg.dimFr.parFr.labelFr.x1Lb \
    -text {upper left x:} -relief flat
  label .psDg.dimFr.parFr.labelFr.y1Lb \
    -text {upper left y:} -relief flat
  label .psDg.dimFr.parFr.labelFr.widthLb \
    -text {width:} -relief flat
  label .psDg.dimFr.parFr.labelFr.heightLb \
    -text {height:} -relief flat

  pack append .psDg.dimFr.parFr.labelFr \
    .psDg.dimFr.parFr.labelFr.x1Lb    {top fill} \
    .psDg.dimFr.parFr.labelFr.y1Lb    {top fill} \
    .psDg.dimFr.parFr.labelFr.widthLb {top fill} \
    .psDg.dimFr.parFr.labelFr.heightLb {top fill}

  frame .psDg.dimFr.parFr.entryFr -borderwidth 0
  entry .psDg.dimFr.parFr.entryFr.x1En \
    -relief sunken -width 10
  entry .psDg.dimFr.parFr.entryFr.y1En \
    -relief sunken -width 10
  entry .psDg.dimFr.parFr.entryFr.widthEn \
    -relief sunken -width 10
  entry .psDg.dimFr.parFr.entryFr.heightEn \
    -relief sunken -width 10
  pack append .psDg.dimFr.parFr.entryFr \
    .psDg.dimFr.parFr.entryFr.x1En    {top fill} \
    .psDg.dimFr.parFr.entryFr.y1En {top fill} \
    .psDg.dimFr.parFr.entryFr.widthEn {top fill} \
    .psDg.dimFr.parFr.entryFr.heightEn {top fill}

  #  packing  #

  pack append .psDg.dimFr.parFr \
    .psDg.dimFr.parFr.labelFr {left padx 20 frame e} \
    .psDg.dimFr.parFr.entryFr {left padx 8 frame w}

  pack append .psDg.dimFr \
    .psDg.dimFr.dimLb {top frame w} \
    .psDg.dimFr.butFr {top frame center} \
    .psDg.dimFr.parFr {top fillx expand}
  
  ####  Buttons frame  ####

  frame .psDg.butFr -borderwidth 0

  set cmd {
    set wfile [.psDg.pathFr.pathEn get]

      set x1 \
	[string trim [.psDg.dimFr.parFr.entryFr.x1En get]]
      set y1 \
	[string trim [.psDg.dimFr.parFr.entryFr.y1En get]]
      set w \
	[string trim [.psDg.dimFr.parFr.entryFr.widthEn get]]
      set h \
	[string trim [.psDg.dimFr.parFr.entryFr.heightEn get]]

      focus .
      grab release .psDg
      destroy .psDg
  }
  append cmd "\n eval \"$writeCmd \$wfile \$x1 \$y1 \$w \$h\""
  
  button .psDg.butFr.okBt \
    -text {Ok} \
    -command $cmd \
    -relief raised

  button .psDg.butFr.cancelBt \
    -text {Cancel} \
    -command {
      focus .
      grab release .psDg
      destroy .psDg
    } \
    -relief raised

  #  packing  #
  pack append .psDg.butFr \
    .psDg.butFr.okBt {left fill expand} \
    .psDg.butFr.cancelBt {left fill expand}
  
  ####  Packing toplevel  ####

  pack append .psDg \
    .psDg.titleLb {top frame n fillx pady 10} \
    .psDg.pathFr {top fillx expand pady 16} \
    .psDg.dimFr  {top fillx expand pady 16} \
    .psDg.butFr  {top fillx expand frame s pady 10}

  ####  Entry bindings  ####

  global currentWriteFile
  .psDg.pathFr.pathEn insert 0 $currentWriteFile

  bind .psDg.pathFr.pathEn \
    <Return> { focus .psDg.dimFr.parFr.entryFr.x1En }
  bind .psDg.dimFr.parFr.entryFr.x1En \
    <Return> { focus .psDg.dimFr.parFr.entryFr.y1En }
  bind .psDg.dimFr.parFr.entryFr.y1En \
    <Return> { focus .psDg.dimFr.parFr.entryFr.widthEn }
  bind .psDg.dimFr.parFr.entryFr.widthEn \
    <Return> { focus .psDg.dimFr.parFr.entryFr.heightEn }
  bind .psDg.dimFr.parFr.entryFr.heightEn \
    <Return> { focus .psDg.pathFr.pathEn }

  bind .psDg.pathFr.pathEn \
    <Tab> { focus .psDg.dimFr.parFr.entryFr.x1En }
  bind .psDg.dimFr.parFr.entryFr.x1En \
    <Tab> { focus .psDg.dimFr.parFr.entryFr.y1En }
  bind .psDg.dimFr.parFr.entryFr.y1En \
    <Tab> { focus .psDg.dimFr.parFr.entryFr.widthEn }
  bind .psDg.dimFr.parFr.entryFr.widthEn \
    <Tab> { focus .psDg.dimFr.parFr.entryFr.heightEn }
  bind .psDg.dimFr.parFr.entryFr.heightEn \
    <Tab> { focus .psDg.pathFr.pathEn }

  ####  Initial bbox value  ####

  lassign [[cv] bbox all] x1 y1 x2 y2
  .psDg.dimFr.parFr.entryFr.x1En insert 0 $x1
  .psDg.dimFr.parFr.entryFr.y1En insert 0 $y1
  .psDg.dimFr.parFr.entryFr.widthEn insert 0 [expr $x2-$x1]
  .psDg.dimFr.parFr.entryFr.heightEn insert 0 [expr $y2-$y1]
}

########  Region Selection  ########

proc iregionStart {X Y} {
  global x1fixed y1fixed
  
  if {[winfo containing $X $Y] != [cv]} { return }
  deselect_all
  set x [[cv] canvasx [expr $X - [winfo rootx [cv]]]]
  set y [[cv] canvasy [expr $Y - [winfo rooty [cv]]]]

  [cv] create rectangle $x $y $x $y \
    -outline red -width 3 -tags {regionSelectFrame}
  set x1fixed 1
  set y1fixed 1
}

proc iregion_move {X Y} {
  set x [[cv] canvasx [expr $X - [winfo rootx [cv]]]]
  set y [[cv] canvasy [expr $Y - [winfo rooty [cv]]]]
  dragMotion regionSelectFrame $x $y
}

proc iregionEnd {} {
  global RegionDone RegionX1 RegionY1 RegionWidth RegionHeight

  lassign [[cv] coords regionSelectFrame] RegionX1 RegionY1 x2 y2
  [cv] delete regionSelectFrame
  set RegionWidth [expr $x2 - $RegionX1]
  set RegionHeight [expr $y2 - $RegionY1]

  set RegionDone 1
}

########  Writing Postscript  ########

proc WritePostscript {} {
  if {[llength [[cv] find withtag all]] > 0} {
    buildWriteDialog Postscript WritePostscriptInFile
  } else {
    warn "Empty drawing!"
  }
}

proc WritePostscriptInFile {path psx1 psy1 psw psh} {
  global currentWriteFile

  set currentWriteFile $path

  set path [sglob $path]

  if {[file exists $path] && ![file writable $path]} {
    warn "Couldn't open $path for writing"
  }

  if {$psx1 != ""} {
    waitCursor

    #  Move ctl and latex objects away  #
    set ctlXOffset ""
    if {![lempty [[cv] find withtag ctl]]} {
      set ctlXOffset \
	[expr $psx1 + ($psw) + 50 - ([lindex [[cv] bbox ctl] 0]) ]
      [cv] move ctl $ctlXOffset 0
    }
    set latexXOffset ""
    if {![lempty [[cv] find withtag latex]]} {
      set latexXOffset \
	[expr $psx1 + ($psw) + 50 - ([lindex [[cv] bbox latex] 0]) ]
      [cv] move latex $latexXOffset 0
    }
    set ltxFrXOffset ""
    if {![lempty [[cv] find withtag ltxFr]]} {
      set ltxFrXOffset \
	[expr $psx1 + ($psw) + 50 - ([lindex [[cv] bbox ltxFr] 0]) ]
      [cv] move ltxFr $ltxFrXOffset 0
    }

    [cv] postscript -file $path \
      -x $psx1 -y $psy1\
      -width $psw -height $psh

    #  bring ctl and latex objects back  #
    if {$ctlXOffset != ""} {
      [cv] move ctl [expr -($ctlXOffset)] 0
    }
    if {$latexXOffset != ""} {
      [cv] move latex [expr -($latexXOffset)] 0
    }
    if {$ltxFrXOffset != ""} {
      [cv] move ltxFr [expr -($ltxFrXOffset)] 0
    }
    
    restoreCursor
    
  } else {
    warn "Empty drawing !"
  }
}

################################################################
####                 Writing Latex                          ####
################################################################

proc WriteLatex {} {

  if {[llength [[cv] find withtag all]] > 0} {
    buildWriteDialog Latex WriteLatexInFile
  } else {
    warn "Empty drawing!"
  }
}

proc WriteLatexInFile {path psx1 psy1 psw psh} {
  global LatexText currentWriteFile

  set currentWriteFile $path
  
  set path [sglob $path]

  if {[file exists $path] && ![file writable $path]} {
    warn "Couldn't open $path for writing"
  }

  if {$psx1 == ""} { return }
  waitCursor

  #  Move ctl and latex objects away  #
  set ctlXOffset ""
  
  if {![lempty [[cv] find withtag ctl]]} {
    set ctlXOffset \
      [expr $psx1 + ($psw) + 50 - ([lindex [[cv] bbox ctl] 0]) ]
    [cv] move ctl $ctlXOffset 0
  }
  
  if {![lempty [[cv] find withtag latex]]} {
    set latexXOffset \
      [expr $psx1 + ($psw) + 50 - ([lindex [[cv] bbox latex] 0]) ]
    [cv] move latex $latexXOffset 0
  }
  
  ##  Replace latex frames by white filled rectangles  ##
  
  set latexIds [[cv] find withtag latex]

  #  make rectangles white #
  [cv] raise ltxFr
  [cv] itemconfigure ltxFr -outline white -fill white
  
  # generate postscript #
  set psbbox [[cv] postscript -file "$path.ps" \
	      -x $psx1 -y $psy1\
		-width $psw -height $psh]

  [cv] itemconfigure ltxFr -outline green -fill green

  #  create texPos items  #
  foreach id $latexIds {
    set both [[cv] find withtag [getTopGroupTags $id]]
    if {[lindex $both 0] == $id} {
      set frameId [lindex $both 1]
    } else {
      set frameId [lindex $both 0]
    }

    lassign [[cv] coords $frameId] x1 y1 x2 y2

    #  Compute the reference point #
    #  according to the text item anchor  #
    set anchor [lindex [[cv] itemconfigure $id -anchor] 4]
    switch -exact -- $anchor {
      w {
	set y1 [expr 0.5*($y2 + $y1)]
      }
      sw {
	set y1 $y2
      }
      n {
	set x1 [expr 0.5*($x2 + $x1)]
      }
      center {
	set x1 [expr 0.5*($x2 + $x1)]
	set y1 [expr 0.5*($y2 + $y1)]
      }
      s {
	set x1 [expr 0.5*($x2 + $x1)]
	set y1 $y2
      }
      ne {
	set x1 $x2
      }
      e {
	set x1 $x2
	set y1 [expr 0.5*($y2 + $y1)]
      }
      se {
	set x1 $x2
	set y1 $y2
      }
    }
	
    [cv] create texpos $x1 $y1 \
      -text $id \
      -tags {ctl texPos}
  }

  #  Create the auxiliary ps file and  #
  #  interpret it to get tex items positions  #

  set psfile [mkTmpFile ps]
  [cv] postscript -file $psfile \
    -x $psx1 -y $psy1\
    -width $psw -height $psh
  exec echo "quit" >> $psfile

  [cv] delete texPos
  
  global PicassoLib

  if {[catch {exec gs -q -DNOPAUSE -DNODISPLAY $psfile} \
	 texPositions]} {
	   warn "Couldn't get latex object positions"
	   catch {exec rm $psfile}
	   catch {exec rm $path.ps}
	   restoreCursor
	   return
	 }

  catch {exec rm $psfile} 

  if {[catch {open $path "w"} st]} {
    warn "Couldn't open $path for writing"
    catch {exec rm $psfile}
    catch {exec rm $path.ps}
    restoreCursor
    return
  }

  lassign $psbbox x1 y1 x2 y2

  #  Compute the latex drawing size that will  #
  #  preserve latex object bbox dims (depends on LatexDrawingRatio)  #

  global LatexDrawingRatio

  set xsize [expr $x2 - $x1]
  set ysize [expr $y2 - $y1]

  set ltxXSize [expr $xsize * $LatexDrawingRatio]
  set ltxYSize [expr $ysize * $LatexDrawingRatio]

  #  Write header  #

  global PicassoVersion env
  puts $st "%%
%% Generated by Picasso version [lindex $PicassoVersion 0]
%% User : $env(USER)
%% Date : [fmtclock [getclock] "%B %d %Y %T"]
%%
%% To get latex objects have the dimensions that were
%%   displayed under picasso, include this drawing by:
%%  
%%   \\input{$path}
%%
%% To resize this drawing, include it by:
%%
%%      \\epsfxsize=<dim> \\input{$path}
%% or
%%      \\epsfysize=<dim> \\input{$path}
%%
%% Resizing won't affect the positionning of latex
%%   object anchor points. The natural dims of the drawing (the ones
%%   preserving latex object dimensions) are:
%%      \\epsfxsize=${ltxXSize}pt
%%      \\epsfysize=${ltxYSize}pt
%%
"
  
  puts $st "\{"
  puts $st "
\\ifnum\\epsfxsize=0
  \\ifnum\\epsfysize=0
    \\epsfxsize=${ltxXSize}pt
    \\unitlength=[expr $ltxXSize / $xsize]pt
  \\else
    \\unitlength=\\epsfysize
    \\divide\\unitlength $ysize
  \\fi
\\else
  \\unitlength=\\epsfxsize
  \\divide\\unitlength $xsize
\\fi
"
  puts $st "\\begin{picture}([expr $x2-$x1],[expr $y2-$y1])"
  puts $st "\n\\put(0,0){\\epsfbox{$path.ps}}\n"

  foreach line [split $texPositions "\n"] {
    if {![regexp {[0-9 .]} $line]} { continue }
    lassign $line id x y
    set text [keylget LatexText $id]

    # compute makebox parameter #
    # according to the text item anchor #
    set anchor [lindex [[cv] itemconfigure $id -anchor] 4]
    switch -exact -- $anchor {
      nw { set mbopt "\[tl\]" }
      w  { set mbopt "\[l\]" }
      sw { set mbopt "\[bl\]" }
      n  { set mbopt "\[t\]" }
      center  { set mbopt "" }
      s  { set mbopt "\[b\]" }
      ne { set mbopt "\[tr\]" }
      e  { set mbopt "\[r\]" }
      se { set mbopt "\[br\]" }
    }

    #  See whether a tabular is necessary, depending on justify  #
    switch -exact -- [lindex [[cv] itemconfigure $id -justify] 4] {
      left {
        set begtab ""
        set endtab ""
      }
      center {
        set begtab "\\begin{tabular}{c}"
        set endtab "\\end{tabular}"
      }
      right {
        set begtab "\\begin{tabular}{r}"
        set endtab "\\end{tabular}"
      }
    }

    # accent commands are redefined in latex tabular environment #
    regsub -all {\\'} $text {\a'} text
    regsub -all {\\`} $text {\a`} text
    # remove blank lines that cause boxdims to break #
    regsub -all "^\[ \]*\n" $text {} text

    puts $st "
\\put([expr $x-$x1-2],[expr $y2-$y]){
  \\makebox(0,0)$mbopt{
    \\begin{minipage}{\\hsize}
    \\begin{tabbing} $begtab
    $text 
    $endtab \\end{tabbing}
    \\end{minipage}
  }
}
"
  }
  
  puts $st "\\end{picture}
\}"
  close $st

  #  bring ctl objects back  #
  if {![lempty [[cv] find withtag ctl]]} {
    [cv] move ctl [expr -($ctlXOffset)] 0
  }

  if {![lempty [[cv] find withtag latex]]} {
    [cv] move latex [expr -($latexXOffset)] 0
    [cv] raise latex
  }
  
  restoreCursor
  warn "See sizing information in the generated latex file"
}

################################################################
####		   Customization file                       ####
################################################################

proc customFileCreate {} {
  global env PicassoLib

  set path [sglob "$env(HOME)/.picassorc"]
  if {[file exists $path]} {
    warn "Customization file $path already exists !"
    return
  }
  
  if {[catch {open $path "w"} stout]} {
    warm "$stout"
    return
  }

  for_file line "$PicassoLib/picrc.tcl" {
    if {($line == "") || ([string index $line 0] == "#")} {
      puts $stout $line
    } else {
      puts $stout "# $line"
    }
  }

  close $stout
  warn "Created $path"
}
