#!/usr/bin/X11/wishx -f

######################################################################
#
# klondikeScore.tcl
#
# Copyright (C) 1993,1994 by John Heidemann <johnh@ficus.cs.ucla.edu>
# All rights reserved.  See the main klondike file for a full copyright
# notice.
#
# $Id: klondikeScore.tcl,v 1.10 1994/01/22 01:16:57 johnh Exp $
#
# $Log: klondikeScore.tcl,v $
# Revision 1.10  1994/01/22  01:16:57  johnh
# determineUser added for Solaris compatibility
#
# Revision 1.9  1994/01/22  01:04:56  johnh
# more robust score-file handling
#
# Revision 1.8  1994/01/13  03:55:14  johnh
# reportError was not calling unmenuHelp
#
# Revision 1.7  1994/01/13  03:30:15  johnh
# register scores without displaying new scores
#
# Revision 1.6  1994/01/12  07:47:40  johnh
# score(writeScores) allows permanent score file writing to be disabled
#
# Revision 1.5  1994/01/09  05:17:09  johnh
# frename ``commit'' of score file
#
# Revision 1.3  1994/01/06  06:15:24  johnh
# preliminary on-disk scores work
#
# Revision 1.2  1994/01/03  03:09:14  johnh
# in-memory scores work; preparation for on-disk scores
#
# Revision 1.1  1994/01/01  00:43:09  johnh
# Initial revision
#
#
######################################################################

set rcsid(klondikeScore.tcl) {$Id: klondikeScore.tcl,v 1.10 1994/01/22 01:16:57 johnh Exp $}



set score(scoreFileVersion) 758073443
set score(scoreFileHeader) "klondike score file version $score(scoreFileVersion)"

# this is used in making the writeScores cookie
set score(startTime) [getclock]


proc reportError {where error} {
	global table errorInfo errorCode

	set padValue $table(padValue)

	# Uncomment to enable standard Tk error handling mechanism.
	# return -code error -errorinfo $errorInfo -errorcode $errorCode $error

	set w ".error"
	catch {unmenuHelp $w}

	toplevel $w
	wmConfig $w "Klondike--error"
	wm transient $w .
	grab set $w

	frame $w.top

	label $w.top.icon -bitmap error
	pack $w.top.icon -padx $padValue -side left
	message $w.top.msg -text $error -width 4i
	pack $w.top.msg -padx $padValue -side right
	button $w.ok -text OK -padx [expr 2*$padValue] -command "unmenuHelp $w"
	pack $w.top $w.ok -side top -padx $padValue -pady $padValue
}


proc readScores {scoreMethod} {
	global table errorCode score

	if { $score(writeScores) == 0 } {
		# not allowed disk access.
		if { [info exists score(list,$scoreMethod)] == 0} {
			set score(list,$scoreMethod) ""
		}
		return
	}

	set catchval [catch {
			set score(list,$scoreMethod) ""
			set f [open "$score(scorefile).$scoreMethod" r]
			while {[gets $f line] >= 0} {
				lappend score(list,$scoreMethod) $line
			}
			close $f
			if { [llength $score(list,$scoreMethod)] > 0 } {
				if {[lindex $score(list,$scoreMethod) 0] !=
						$score(scoreFileHeader) } {
					set score(list,$scoreMethod) ""
					error "Old version of score file.\nHigh scores reset."
				}
				set score(list,$scoreMethod) [lrange $score(list,$scoreMethod) 1 end]
			}
		} error]
	if { $catchval == 0 } {
		return
	}
	switch -exact [lindex $errorCode 1] {
		ENOENT {}
		default { reportError "readScores" $error }
	}
}



proc writeScores {scoreMethod} {
	global table errorCode score

	if { $score(writeScores) == 0 } {
		return
	}

	#
	# The score file is updated optimistically.
	#
	# To write scores atomically without locking
	# we write the file to a tmp file and then use rename
	# to commit our changes.
	#
	# At worst, our change is lost because of concurrent
	# update.  The score file cannot be corrupted, though.
	#
	# NEEDSWORK: We should then check to make sure our
	# update made it and re-try if it didn't.
	#

	# Generate a (almost) guaranteed unique cookie to identify us.
	set cookie "$score(startTime).[random 10000]"

	set newPath "$score(scorefile).$scoreMethod"
	set oldPath "$newPath.$cookie"
	set catchval [catch {
			set f [open $oldPath w]
			puts $f $score(scoreFileHeader)
			foreach i $score(list,$scoreMethod) {
				puts $f $i
			}
			close $f
			# commit
			frename $oldPath $newPath
		} error]
	if { $catchval != 0 } {
		# Try to clean up.
		catch {unlink $oldPath}
		reportError "writeScores" $error
	}
}


proc determineUser {} {
	global score
	# cache the result
	if { [info exists score(user)] } {
		return $score(user)
	}
	# first try the environment
	# $USER is a bsd-ism
	if { [info exists env(USER)] } {
		return [set score(user) $env(USER)]
	}
	# $LOGNAME is the svr4-ism
	if { [info exists env(LOGNAME)] } {
		return [set score(user) $env(USER)]
	}
	# If these fail, try whoami.
	if { [catch {exec "whoami"} who] == 0 } {
		return [set score(user) $who]
	}
	# Give up.  Let the user know and disable writing scores.
	reportError "determineUser" {Could not determine user name.  \
Set your environment variable USER or LOGNAME and re-run klondike.  \
Score saving is disabled.}
	set score(writeScores) 1
	return [set score(user) "nobody"]
}


proc computeNewScoreListEntry {scoreMethod} {
	global table score env

	#
	# Get information for the score
	#
	if { $scoreMethod == "standard" } {
		set scoreValue $score(standardScore)
		set fancyScore $scoreValue
	} else {
		set scoreValue $score(casinoScore)
		set fancyScore "\$$score(casinoScore)"
	}
	# Add to scores to avoid sorting both negative and positive numbers.
	set scoreValue [expr $scoreValue + 10000]
	if { $scoreValue < 0 } { set scoreValue 0 }
	set scoreClock [getclock]
	set scoreDate [fmtclock $scoreClock "%e-%b-%Y"]
	set scoreName [determineUser]
	# Always ASCII sort by score key.
	# Switch the sign by subtracting from 2^30.
	set scoreKey [format "%08d:%08d" $scoreValue [expr 1073741824-$scoreClock]]

	set newListEntry [list $scoreKey \
			 $fancyScore $scoreDate $scoreName $scoreClock]

	return [list $newListEntry $scoreClock]
}


proc updateScoreList {newListEntry scoreMethod} {
	global score table

	#
	# NEEDSWORK: Currently don't do score locking.
	#
	readScores $scoreMethod
	set oldScoreList $score(list,$scoreMethod)

	#
	# Add score to score-list.
	# Scorelist format:
	# SortKey(score,clock) value date name clock
	#
	set score(list,$scoreMethod) [lsort -decreasing [linsert $score(list,$scoreMethod) 0 $newListEntry]]
	if { [llength $score(list,$scoreMethod)] > 100 } {
		set score(list,$scoreMethod) [lrange $score(list,$scoreMethod) 0 99]
	}

	#
	# NEEDSWORK: Currently don't do score locking.
	#
	if { $score(list,$scoreMethod) != $oldScoreList } {
		writeScores $scoreMethod
	}
}


proc computeNewScoreText {scoreMethod} {
	global score table

	if { [info exists table(lastScoreToken)] } {
		set ourScoreToken $table(lastScoreToken)
	} else {
		set ourScoreToken "xxx"
	}
	#
	# Regenerate score-text from score-list.
	#
	set fancyMethod "[string toupper [string index $scoreMethod 0]][string range $scoreMethod 1 end]"
	set score(text,$scoreMethod) "<big>${fancyMethod} Scores</big>\n\n<computer>"
	set j 0
	foreach i $score(list,$scoreMethod) {
		incr j
		set thisClock [lrange $i 4 4]
		set thisText ""
		if { $thisClock == $ourScoreToken } {
			set style "reverse"
		} else {
			set style ""
		}
		if { $style != "" } {
			set thisText "${thisText}<${style}>"
		}
		set thisText "${thisText}[format "%3d" $j].   [format "%8s" [lindex $i 1]]   [lindex $i 2]   [lindex $i 3] "
		if { $style != "" } {
			set thisText "${thisText}</${style}>"
		}
		set score(text,$scoreMethod) "$score(text,$scoreMethod)$thisText\n"
	}
	set score(text,$scoreMethod) "$score(text,$scoreMethod)</computer>"

	if { [llength $score(list,$scoreMethod)] == 0 } {
		set score(text,$scoreMethod) "$score(text,$scoreMethod)No current scores."
	}

	if { $score(writeScores) == 0 } {
		set score(text,$scoreMethod) "$score(text,$scoreMethod)\n<italic>Permanent storage of score file not enabled.</italic>"
	}
}





proc registerNewScore {showScores} {
	global table

	set scoreMethod $table(scoringMethod)

	#
	# Figure the new data.
	#
	set foo [computeNewScoreListEntry $scoreMethod]
	set newScoreListEntry [lindex $foo 0]
	set table(lastScoreToken) [lindex $foo 1]

	#
	# Add it to old scores.
	#
	updateScoreList $newScoreListEntry $scoreMethod

	#
	# Refigure score text.
	# 
	computeNewScoreText $scoreMethod

	if {$showScores} {
		#
		# Tell the jubliant user.
		#
		displayHighScores $scoreMethod
		#
		menuHelpScrollToTag $scoreMethod reverse 
	}
	
}


proc displayHighScores {method} {
	global score table help

	readScores $method
	computeNewScoreText $method
	
	#
	# re-use the help system code
	#

	# generate "help" text
	set help($method) $score(text,$method)
	menuHelp $method "${method} scores"
}


