################################################################
####                                                        ####
####      Procedures for interactive geometric              ####
####           transformation of objects                    ####
####                                                        ####
################################################################

proc iopStartHook {idList} {
  foreach id $idList {
    if {[[cv] type $id] == "image"} {
      #  image items  #
      [cv] addtag laterOp withtag $id
      lassign [[cv] coords $id] xc yc
      set w [lindex [[cv] itemconfigure $id -width] 4]
      set h [lindex [[cv] itemconfigure $id -height] 4]
      set angle [lindex [[cv] itemconfigure $id -angle] 4]
      [cv] hide $id
      [cv] create orectangle $xc $yc \
	-width $w -height $h -angle $angle \
	-outlinewidth 2 -outline red \
	-tags "toOp ctl opFr"
    } else {
      #  other items  #
      set clone \
	[eval [cv] create [[cv] type $id] \
	 [[cv] coords $id] [[cv] switches $id]]
      [cv] itemconfigure $clone -tags "toOp ctl"
      [cv] hide $id
    }
  }
}

proc iopEndHook {} {
  [cv] delete toOp
  [cv] show s
  [cv] raise ltxFr
  [cv] raise latex
}

proc opDelay {idList} {
  
  foreach id $idList {
    if {[[cv] type $id] == "image"} {
      [cv] addtag laterOp withtag $id
      lassign [[cv] coords $id] xc yc
      set w [lindex [[cv] itemconfigure $id -width] 4]
      set h [lindex [[cv] itemconfigure $id -height] 4]
      set angle [lindex [[cv] itemconfigure $id -angle] 4]
      [cv] create orectangle $xc $yc \
	-width $w -height $h -angle $angle \
	-outlinewidth 2 -outline red \
	-tags "toOp ctl opFr opSlave$id"
    } else {
      [cv] addtag toOp withtag $id
    }
  }
}

################################################################
####		   Object duplication                       ####
################################################################

proc Duplicate {} {
  global ImageFile LatexText LatexDims
  
  set selection [[cv] find withtag s]

  if {$selection == ""} {
    warn "No item selected"
    return
  }

  foreach item $selection {
    set type   [[cv] type $item]
    set coords [[cv] coords $item]
    set config [[cv] itemconfigure $item]
    set options {}
    foreach cf $config {
      if {[lindex $cf 0] == "-tags"} {
	set tags [lindex $cf 4]
	set newtag "inCreation"
	foreach tag $tags {
	  switch -glob -- $tag {
	    Gr* -
	    tGr*  -
	    latex      -
	    ltxFr      {
	      lappend newtag $tag
	    }
	  }
	}
	lappend options "-tags" $newtag
      } else {
	lappend options [lindex $cf 0] [lindex $cf 4]
      }
    }

    if {$type == "image"} {
      ####  image items => duplicate image file ####
      set dupfile [mkTmpFile image]
      if {[catch {exec cp [keylget ImageFile $item] $dupfile} err]} {
	warn "couldn't duplicate image object file: $err"
	continue
      }
      set ipath [expr "[lsearch -exact "$options" "-path"] + 1"]
      set options [lreplace $options $ipath $ipath $dupfile]
    }

    set dupid [eval "[cv] create $type $coords $options"]
    
    if {$type == "image"} {
      ####  image items => update ImageFile ####
      keylset ImageFile $dupid "$dupfile"
    }

    #  update keyed lists  #
    if {[keylget LatexText $id text]} {
      keylset LatexText $dupid $text
    }
    if {[keylget LatexDims $id dims]} {
      keylset LatexDims $dupid $dims
    }
  }
    
  set offset [grid 20 20]
  global GridSpacing
  if {[lindex $offset 0] == 0} {
    set offset "$GridSpacing [lindex $offset 1]"
  }
  if {[lindex $offset 1] == 0} {
    set offset "[lindex $offset 0] $GridSpacing"
  } else {
    set offset "[lindex $offset 0] -[lindex $offset 1]"
  }
  eval "[cv] move inCreation $offset"
  
  deselect_all

  #  Don't call createEndHook because of group tags  #
  updateGroupTags inCreation
  [cv] addtag s withtag inCreation
  [cv] addtag sOfr withtag inCreation
  [cv] dtag all inCreation
  redrawFrames
}

################################################################
####		     Moving objects                         ####
################################################################

proc Move {x y} {
  [cv] move sOfr $x $y
}

proc imoveSelectionStart {x y} {
  ##  Moves currently selected items, even if  ##
  ##  user didn't click on an item  ##
  global xdragLast ydragLast Dragging

  lassign [grid $x $y] x y
  
  ##  First include current in the selection  ##
  if {[lempty [select_for_drag]]} {
    warn "No item selected"
    deselect_for_drag
  } else {
    dragStartHook
    set xdragLast $x
    set ydragLast $y
  }
}

proc imoveStart {x y} {
  ##  Deselect all, and moves item under the mouse  ##
  global xdragLast ydragLast Dragging

  deselect_all
  ##  First select current ##
  if {[lempty [select_for_drag]]} {
    deselect_for_drag
  } else {
    dragStartHook
    set xdragLast $x
    set ydragLast $y
  }
}
  
proc imoveMotion {x y} {
  global xdragLast ydragLast Dragging
  
  if {!$Dragging} { return }

  lassign [grid $x $y] x y

  [cv] move sOfr \
    [expr $x - ($xdragLast)] \
    [expr $y - ($ydragLast)]

  set xdragLast $x
  set ydragLast $y
}

proc imoveEnd {} {
  deselect_for_drag
  dragEndHook
}

################################################################
####		    Resizing objects                        ####
################################################################

set xdragOrigin 0
set ydragOrigin 0

set ixdragAnchor 0
set iydragAnchor 0

set dragDox 0
set dragDoy 0

proc iresizeStart {x y} {
  global xdragAnchor ydragAnchor xdragOrigin ydragOrigin dragDox dragDoy
  global ixdragAnchor iydragAnchor

  ##  First include current in the selection  ##

  select_for_drag

  set latexSIds [[cv] find withtag s latex]
  if {![lempty $latexSIds]} {
    warn "Can't stretch latex objects - see latex help"
    [cv] addtag sLt withtag s latex
    [cv] addtag sLt withtag s ltxFr
    [cv] dtag sLt s
  }

  set ids [[cv] find withtag s]
  
  if {[lempty $ids]} {
    warn "No item selected"
    deselect_for_drag
    if {![lempty $latexSIds]} {
      [cv] addtag s withtag sLt
      [cv] dtag all sLt
    }
    return
  }
  
  dragStartHook
  lassign [grid $x $y] x y
  
  ##  Also protect objects that shouldn't be resized interactively  ##
  #  replace images by a polyline for visualisation  #

  opDelay $ids
  
  [cv] delete sFr
  
  set bb [[cv] bbox s]
  
  if {$bb == ""} {
    [cv] dtag all toOp
    [cv] dtag all laterOp
    deselect_for_drag
    redrawFrames
    dragEndHook
    return
  }

  set x1 [lindex $bb 0]
  set y1 [lindex $bb 1]
  set x2 [lindex $bb 2]
  set y2 [lindex $bb 3]
  
  set attach [cornerOrSide $x $y $bb]
  
  switch -exact $attach {
    none {
      return
    }
    c11 {
      set xdragOrigin $x2
      set ydragOrigin $y2
      set dragDox 1
      set dragDoy 1
    }
    c12 {
      set xdragOrigin $x2
      set ydragOrigin $y1
      set dragDox 1
      set dragDoy 1
    }
    c21 {
      set xdragOrigin $x1
      set ydragOrigin $y2
      set dragDox 1
      set dragDoy 1
    }
    c22 {
      set xdragOrigin $x1
      set ydragOrigin $y1
      set dragDox 1
      set dragDoy 1
    }
    sx1 {
      set xdragOrigin [expr "0.5*($x1+$x2)"]
      set ydragOrigin $y2
      set dragDox 0
      set dragDoy 1
    }
    sx2 {
      set xdragOrigin [expr "0.5*($x1+$x2)"]
      set ydragOrigin $y1
      set dragDox 0
      set dragDoy 1
    }
    sy1 {
      set xdragOrigin $x2
      set ydragOrigin [expr "0.5*($y1+$y2)"]
      set dragDox 1
      set dragDoy 0
    }
    sy2 {
      set xdragOrigin $x1
      set ydragOrigin [expr "0.5*($y1+$y2)"]
      set dragDox 1
      set dragDoy 0
    }
  }
  
  set xdragAnchor [expr "$x - $xdragOrigin"]
  set ixdragAnchor $xdragAnchor
  set ydragAnchor [expr "$y - $ydragOrigin"]
  set iydragAnchor $ydragAnchor
}

proc iresizeMotion {x y} {
  global xdragAnchor ydragAnchor xdragOrigin ydragOrigin dragDox dragDoy
  global Dragging
  
  if {!$Dragging} { return }
  
  lassign [grid $x $y] x y

  if { $dragDox } {
    set xfact [expr "($x - $xdragOrigin+0.0)/$xdragAnchor"]
  } else {
    set xfact 1.0
  }
  
  if { $dragDoy } {
    set yfact [expr "($y - $ydragOrigin+0.0)/$ydragAnchor"]
  } else {
    set yfact 1.0
  }
  
  if {($xfact > 0.01) && ($yfact > 0.01)} {
    set xdragAnchor [expr "$x - $xdragOrigin"]
    set ydragAnchor [expr "$y - $ydragOrigin"]
    [cv] scale toOp $xdragOrigin $ydragOrigin $xfact $yfact
  }
}

proc iresizeEnd {} {
  global Dragging GriddingMode
  global ixdragAnchor iydragAnchor xdragOrigin ydragOrigin dragDox dragDoy
  
  if {!$Dragging} {
    # never know! #
    dragEndHook
    return
  }

  #  Align objects to grid  #
  if {$GriddingMode} {
    gridObject toOp
  }
  
  ##  Actually resize heavy objects  ##
  waitCursor
  foreach id [[cv] find withtag laterOp] {
    [cv] dtag $id laterOp
    set fid [[cv] find withtag "opSlave$id"]
    if {$fid != ""} {
      eval "[cv] coords $id [[cv] coords $fid]"
      eval "[cv] itemconfigure $id \
              [[cv] coords $fid -config]"
    }
  }

  [cv] delete opFr
  deselect_for_drag
  [cv] dtag all toOp
  [cv] addtag s withtag sLt
  [cv] dtag all sLt
  redrawFrames
  restoreCursor
  dragEndHook
}

set dragDist 0
set initialDragDist 0

################################################################
####		   Reshaping objects                        ####
################################################################

##
##  Shape states:
##  
##   - ModeState = 0 <==> no handles drawn
##   - ModeState = 1 <==> handles are drawn for ShapeId
##   - ModeState = 2 <==> one of the handles is being dragged.
##  				(and ShapeHid = dragged handle id)
##  

set ShapeId {}
set ShapeHid {}
set ShapeCoords {}
set ShapeRank {}

##  Create one handle per item coord  ##
proc putShapeHandles {id} {
  global ShapeCoords

  switch [[cv] type $id] {
    frectangle -
    orectangle -
    image {
      set coords [[cv] coords $id -vertices]
    }
    polygon {
      set coords [[cv] coords $id]
      set coords [lrange $coords 0 \
		  [expr [llength $coords] - 3]]
      }
    default {
      set coords [[cv] coords $id]
    }
  }
  
  set ShapeCoords {}

  loop i 0 [llength $coords] 2 {

    set x [lindex $coords $i]
    set y [lindex $coords [expr $i + 1]]

    set hid [[cv] create rectangle \
	     [expr $x - 3] [expr $y - 3] [expr $x + 3] [expr $y + 3] \
	       -width 2 -outline red \
	       -tags {rsHandle ctl}]
    keylset ShapeCoords $hid "$x $y"
  }
}

proc ishapeTake {x y} {
  global ShapeId ShapeCoords ShapeHid ShapeRank 
  global xdragLast ydragLast ModeState

  if {![lempty [[cv] find withtag current latex]] ||
    ![lempty [[cv] find withtag current ltxFr]]} {
      warn "Can't reshape latex objects - See latex help"
      return
    }

  set id [[cv] find withtag current]
  
  if {$id == ""} {
    #  user clicked on nothing  #
    #  -> delete handles  #
    [cv] delete rsHandle
    [cv] delete opFr
    set ShapeId {}
    set ModeState 0
    return
  }

  if {($ShapeId == "")} {
    #  user clicked on an object, and no handles are drawn #
    #  -> draw handles for object  #
    [cv] delete rsHandle
    [cv] delete opFr
    putShapeHandles $id
    set ShapeId $id
    set ModeState 1
    return
  }

  if {[keylget ShapeCoords $id {}]} {
    # handles are drawn, and user clicked on one of them #
    # -> start dragging handle #
    set ModeState 2

    # replace images by frames #
    opDelay $id
    set id [[cv] find withtag toOp]
    
    set ShapeHid $id

    #  if reshaping an image or an rrect,  #
    #  delete other handles cause we're unable to  #
    #  redisplay them fast enough  #
    
    if {[lmember {orectangle frectangle} [[cv] type $ShapeId]]} {
      foreach hid [keylget ShapeCoords] {
	if {$hid != $ShapeHid} { [cv] delete $hid }
      }
      set ShapeRank [lsearch [keylget ShapeCoords] $ShapeHid]
      set ShapeCoords [keylget ShapeCoords $ShapeHid]
    }
      
    set xdragLast [[cv] canvasx $x]
    set ydragLast [[cv] canvasy $y]
  } else {
    # handles were drawn, but user clicked  #
    #  on another object -> redraw handles  #
    [cv] delete rsHandle
    putShapeHandles $id
    set ShapeId $id
    set ModeState 1
    return
  }
}

proc ishapeMotion {x y} {
  global xdragLast ydragLast
  global ModeState ShapeId ShapeHid ShapeCoords ShapeRank

  if {$ModeState != 2} { return }

  lassign [grid $x $y] x y
  
  if {($x == $xdragLast) &&
    ($y == $ydragLast)} { return }

  #  move handle  #
  [cv] move $ShapeHid \
    [expr $x - ($xdragLast)] [expr $y - ($ydragLast)]

  switch [[cv] type $ShapeId] {

    orectangle -
    frectangle -
    image {
      # update ShapeCoords #
      set ShapeCoords \
	"[expr [lindex $ShapeCoords 0] + $x - ($xdragLast)] \
         [expr [lindex $ShapeCoords 1] + $y - ($ydragLast)]"
      
      #  reconfigure object  #
      set ShapeRank [eval "[cv] coords $ShapeId -shape \
                           $ShapeRank $ShapeCoords"]     
    }      

    default {

      # update ShapeCoords #
      lassign [keylget ShapeCoords $ShapeHid] xcoord ycoord
      keylset ShapeCoords $ShapeHid \
	"[expr $xcoord + $x - ($xdragLast)] \
         [expr $ycoord + $y - ($ydragLast)]"

      #  reconfigure object  #
      set coords {}
      foreach assoc $ShapeCoords {
	set coords "$coords [lindex $assoc 1]"
      }
      eval "[cv] coords $ShapeId $coords"
      
    }
  }
  set xdragLast $x
  set ydragLast $y
  update
}

proc ishapeEnd {} {
  global ModeState ShapeHid ShapeId

  if {$ModeState != 2} { return }
  set ModeState 1
  set ShapeHid {}
  if {[lmember {orectangle frectangle image} [[cv] type $ShapeId]]} {
    [cv] delete rsHandle
    putShapeHandles $ShapeId
  }
  [cv] dtag all toOp
}

proc ishapeAdd {x y} {
  global ShapeId ShapeCoords ShapeHid

  set coords {}
  set newShapeCoords {}

  if {![lmember {line dline polygon} [[cv] type $ShapeId]]} {
    return
  }

  lassign [grid $x $y] x y
  
  foreach assoc $ShapeCoords {
    if {[lindex $assoc 0] == $ShapeHid} {
      set hid [[cv] create rectangle \
	       [expr $x - 3] [expr $y - 3] [expr $x + 3] [expr $y + 3] \
		 -width 2 -outline red \
		 -tags {rsHandle ctl}]
      lappend newShapeCoords \
	[list $hid "$x $y"]
      set coords "$coords $x $y"
    }
    lappend newShapeCoords $assoc
    set coords "$coords [lindex $assoc 1]"
  }
  set ShapeCoords $newShapeCoords
  eval "[cv] coords $ShapeId $coords"
}

proc ishapeDelete {} {
  global ShapeId ShapeCoords ShapeHid

  set coords {}

  switch [[cv] type $ShapeId] {
    line -
    dline {
      if {[llength $ShapeCoords] <= 2} { return }
    }
    polygon {
      if {[llength $ShapeCoords] <= 3} { return }
    }
    default {
      return
    }
  }

  set i [lsearch [keylget ShapeCoords] $ShapeHid]

  if {$i == 0} {
    [cv] delete [lindex [lindex $ShapeCoords 1] 0]
    set ShapeCoords [lreplace $ShapeCoords 1 1]
  } else {
    incr i -1
    [cv] delete [lindex [lindex $ShapeCoords $i] 0]
    set ShapeCoords [lreplace $ShapeCoords $i $i]
  }

  set coords {}
  foreach assoc $ShapeCoords {
    set coords "$coords [lindex $assoc 1]"
  }

  ####  Bug with polygon items: can't remove coordinates  ####
  ####  => recreate the item  ####
  if {[[cv] type $ShapeId] == "polygon"} {
    set cf [[cv] itemconfigure $ShapeId]
    set options {}
    foreach spec $cf {
      set val [lindex $spec 4]
      if {[lempty $val]} { continue }
      lappend options [lindex $spec 0] $val
    }
    [cv] delete $ShapeId
    set ShapeId \
      [eval "[cv] create polygon $coords $options"]
    [cv] raise rsHandle
  } else {
    eval "[cv] coords $ShapeId $coords [lrange $coords 0 1]"
  }

}
  
################################################################
####		    Rotating objects                        ####
################################################################

##  Non interactive rotations  ##
proc Rotate {alpha} {

  set latexSIds [[cv] find withtag s latex]
  if {![lempty $latexSIds]} {
    warn "Can't rotate latex objects - See latex help"
    [cv] addtag sLt withtag s latex
    [cv] addtag sLt withtag s ltxFr
    [cv] dtag sLt s
  }

  set ids [[cv] find withtag s]
  
  if {[lempty $ids]} {
    warn "No item selected"
    deselect_for_drag
    if {![lempty $latexSIds]} {
      [cv] addtag s withtag sLt
      [cv] dtag all sLt
    }
    return
  }
    
  [cv] delete sFr

  if {[llength $ids] == 1} {
    switch [[cv] type $ids] {
      frectangle -
      orectangle -
      image {
	lassign [[cv] coords $ids] xc yc
      }
      dline {
	lassign [[cv] coords $ids -bbox] x1 y1 x2 y2
	set xc [expr "0.5 * ($x1 + $x2)"]
	set yc [expr "0.5 * ($y1 + $y2)"]
      }
      default {
	lassign [[cv] bbox $ids] x1 y1 x2 y2
	set xc [expr "0.5 * ($x1 + $x2)"]
	set yc [expr "0.5 * ($y1 + $y2)"]
      }
    }
  } else {
    lassign [[cv] bbox s] x1 y1 x2 y2
    set xc [expr "0.5 * ($x1 + $x2)"]
    set yc [expr "0.5 * ($y1 + $y2)"]
  }

  foreach id $ids {
    switch [[cv] type $id] {

      arc {
	global ArcCoords
	if {[catch {lassign $ArcCoords($id) rayon xarcc yarcc xarcs yarcs}]} {
	  lassign [computeArcCoords $id] rayon xarcc yarcc xarcs yarcs
	}
	
	lassign [eval "rotateCoords -angle $alpha \
                         $xc $yc \
                         $xarcc $yarcc $xarcs $yarcs"] \
	  newxc newyc newxs newys

	[cv] coords $id \
	  [expr "$newxc - $rayon"] [expr "$newyc - $rayon"] \
	  [expr "$newxc + $rayon"] [expr "$newyc + $rayon"]

	[cv] itemconfigure $id -start \
	  [angle $newxc $newyc $newxs $newys]

	set ArcCoords($id) "$rayon $newxc $newyc $newxs $newys"
      }

      image {
	[cv] coords $id -rotate $xc $yc $alpha
      }

      default {
	eval "[cv] coords $id \
	  [eval "rotateCoords -angle $alpha \
	  $xc $yc \
	  [[cv] coords $id]"]"
      }
    }
  }

  [cv] addtag s withtag sLt
  [cv] dtag all sLt
  redrawFrames
}

proc irotateStart {x y} {
  global xdragOrigin ydragOrigin Dragging
  global xdragLast ydragLast xdragFirst ydragFirst 

  select_for_drag

  set latexSIds [[cv] find withtag s latex]
  if {![lempty $latexSIds]} {
    warn "Can't rescale latex objects - see latex help"
    [cv] addtag sLt withtag s latex
    [cv] addtag sLt withtag s ltxFr
    [cv] dtag sLt s
  }

  set ids [[cv] find withtag s]
  
  if {[lempty $ids]} {
    warn "No item selected"
    deselect_for_drag
    [cv] addtag s withtag sLt
    [cv] dtag all sLt
    return
  }

  dragStartHook
  iopStartHook $ids

  #  center of rotation  #
  if {[llength $ids] == 1} {
    switch [[cv] type $ids] {
      frectangle -
      orectangle -
      image {
	lassign [[cv] coords $ids] xdragOrigin ydragOrigin
      }
      dline {
	lassign [[cv] coords $ids -bbox] x1 y1 x2 y2
	set xdragOrigin [expr "0.5 * ($x1 + $x2)"]
	set ydragOrigin [expr "0.5 * ($y1 + $y2)"]
      }
      default {
	lassign [[cv] bbox $ids] x1 y1 x2 y2
	set xdragOrigin [expr "0.5 * ($x1 + $x2)"]
	set ydragOrigin [expr "0.5 * ($y1 + $y2)"]
      }
    }
  } else {
    lassign [[cv] bbox s] x1 y1 x2 y2
    set xdragOrigin [expr "0.5 * ($x1 + $x2)"]
    set ydragOrigin [expr "0.5 * ($y1 + $y2)"]
  }

  lassign [grid $x $y] xdragLast ydragLast
  set xdragFirst $xdragLast
  set ydragFirst $ydragLast

  [cv] delete sFr
}

proc irotateMotion {x y} {
  global xdragOrigin ydragOrigin Dragging
  global xdragLast ydragLast
  
  if {!$Dragging} { return }

  lassign [grid $x $y] x y
  
  set angle [pointAngle $xdragOrigin $ydragOrigin \
	     $xdragLast $ydragLast $x $y]
  
  foreach id [[cv] find withtag toOp] {
    switch [[cv] type $id] {

      arc {
	global ArcCoords
	if {[catch {lassign $ArcCoords($id) rayon xarcc yarcc xarcs yarcs}]} {
	  lassign [computeArcCoords $id] rayon xarcc yarcc xarcs yarcs
	}
	
	lassign [eval "rotateCoords -points $xdragOrigin $ydragOrigin \
                         $xdragLast $ydragLast $x $y \
                         $xarcc $yarcc $xarcs $yarcs"] \
	  newxc newyc newxs newys

	[cv] coords $id \
	  [expr "$newxc - $rayon"] [expr "$newyc - $rayon"] \
	  [expr "$newxc + $rayon"] [expr "$newyc + $rayon"]

	[cv] itemconfigure $id -start \
	  [angle $newxc $newyc $newxs $newys]

	set ArcCoords($id) "$rayon $newxc $newyc $newxs $newys"
      }

      frectangle -
      orectangle {
	[cv] coords $id -rotate \
	  $xdragOrigin $ydragOrigin $angle
      }

      default {
	eval "[cv] coords $id \
	  [eval "rotateCoords -points $xdragOrigin $ydragOrigin \
                   $xdragLast $ydragLast $x $y \
                   [[cv] coords $id]"]"
      }
    }
  }

  update idletasks
  set xdragLast $x
  set ydragLast $y
}

proc irotateEnd {x y} {
  global Dragging xdragOrigin ydragOrigin
  global xdragLast ydragLast xdragFirst ydragFirst 
  global GriddingMode

  if {$GriddingMode} {
    gridObject toOp
  }

  lassign [grid $x $y] x y

  # compute global rotation parameters #
  set angle [pointAngle $xdragOrigin $ydragOrigin \
	     $xdragFirst $ydragFirst $x $y]

  #  delete clones and rotate originals  #
  waitCursor
  foreach id [[cv] find withtag s] {
    switch [[cv] type $id] {
      
      arc {
	global ArcCoords
	lassign [computeArcCoords $id] rayon xarcc yarcc xarcs yarcs
	lassign [eval "rotateCoords -points $xdragOrigin $ydragOrigin \
                         $xdragFirst $ydragFirst $x $y \
                         $xarcc $yarcc $xarcs $yarcs"] \
	  newxc newyc newxs newys

	[cv] coords $id \
	  [expr "$newxc - $rayon"] [expr "$newyc - $rayon"] \
	  [expr "$newxc + $rayon"] [expr "$newyc + $rayon"]

	[cv] itemconfigure $id -start \
	  [angle $newxc $newyc $newxs $newys]
      }

      frectangle -
      orectangle -
      image {
	[cv] coords $id -rotate \
	  $xdragOrigin $ydragOrigin $angle
      }

      default {
	eval "[cv] coords $id \
	  [eval "rotateCoords -points $xdragOrigin $ydragOrigin \
	  $xdragFirst $ydragFirst $x $y \
	  [[cv] coords $id]"]"
      }
    }
  }
  
  iopEndHook
  restoreCursor

  deselect_for_drag
  [cv] addtag s withtag sLt
  [cv] dtag all sLt
  redrawFrames
  dragEndHook
}

################################################################
####		       Symetries                            ####
################################################################

proc FlipHorizontal {} {
  Flip hflipCoords
}

proc FlipVertical {} {
  Flip vflipCoords
}

proc Flip {procName} {

  set latexSIds [[cv] find withtag s latex]
  if {![lempty $latexSIds]} {
    warn "Can't flip latex objects - see latex help"
    [cv] addtag sLt withtag s latex
    [cv] addtag sLt withtag s ltxFr
    [cv] dtag sLt s
  }

  set ids [[cv] find withtag s]

  if {[lempty $ids]} {
    warn "No item selected"
    deselect_for_drag
    if {![lempty $latexSIds]} {
      [cv] addtag s withtag sLt
      [cv] dtag all sLt
    }
    return
  }

  [cv] delete sFr

  set bbox [eval "[cv] bbox $ids"]

  waitCursor
  set failed {}
  foreach id $ids {
    switch [[cv] type $id] {
      line -
      dline -
      polygon {
	eval "[cv] coords $id \
	  [eval "$procName $bbox [[cv] coords $id]"]"
      }
      image {
	set vertices "[[cv] coords $id] [[cv] coords $id -vertices]"
	set newvertices [eval "$procName $bbox $vertices"]
	set newcf [eval vertices2Config $newvertices]
	eval "[cv] coords $id [lrange $newcf 0 1]"
	eval "[cv] itemconfigure $id [lrange $newcf 2 end]"
      }

      default {
	lappend failed [[cv] type $id]
      }	
    }
  }

  [cv] addtag s withtag sLt
  [cv] dtag all sLt
  redrawFrames
  restoreCursor
  
  if {![lempty $failed]} {
    warn "Can't rotate [join $failed " or "] objects"
  }
}

####  Axial symetries  ####

proc AxialSymetry {} {
  #  Gets sym. axe from the user, then symetrize  #
  #  selected objects  #
  global CurrentMode lastMode

  set latexSIds [[cv] find withtag s latex]
  if {![lempty $latexSIds]} {
    warn "Can't rescale latex objects - see latex help"
    [cv] addtag sLt withtag s latex
    [cv] addtag sLt withtag s ltxFr
    [cv] dtag sLt s
  }
  
  set ids [[cv] find withtag s]
  if {[lempty $ids]} {
    warn "No item selected"
    if {![lempty $latexSIds]} {
      [cv] addtag s withtag sLt
      [cv] dtag all sLt
    }
    return
  }

  #  Get sym axe  #
  set lastMode $CurrentMode
  set CurrentMode SymetryMode

  setCursor diamond_cross
  saveMsg
  msg "Set axe of symetry with mouse button 1"
}

proc symaxeStart {x y} {
  dragStartHook
  lassign [grid $x $y] x y
  [cv] create dline $x $y $x $y \
    -fill blue -dashes {32 6 6 6} \
    -tags {symaxe ctl}
}

proc symaxeMotion {x y} {
  set id [[cv] find withtag symaxe]
  if {$id != ""} {
    eval "[cv] coords $id \
        [lrange [[cv] coords $id] 0 1] [grid $x $y]"
  }
}

proc symaxeEnd {x y} {
  global CurrentMode lastMode

  set axeid [[cv] find withtag symaxe]
  if {$axeid != ""} {

    lassign [grid $x $y] x y

    set CurrentMode $lastMode
    restoreCursor
    waitCursor

    lassign [[cv] coords $axeid] x0 y0
    [cv] delete $axeid
    [cv] delete sFr
    foreach id [[cv] find withtag s] {
      set newcoords \
	[eval "axialSymCoords $x0 $y0 $x $y \
	         [[cv] coords $id]"]
      switch [[cv] type $id] {
	image {
	  set vertices "[[cv] coords $id] [[cv] coords $id -vertices]"
	  set newvertices [eval "axialSymCoords $x0 $y0 $x $y \
                                 $vertices"]

	  set newcf [eval vertices2Config $newvertices]
	  eval "[cv] coords $id [lrange $newcf 0 1]"
	  eval "[cv] itemconfigure $id [lrange $newcf 2 end]"
	}
	default {
	  eval "[cv] coords $id $newcoords"
	}
      }
    }
    restoreCursor
  }

  [cv] addtag s withtag sLt
  [cv] dtag all sLt
  redrawFrames
  dragEndHook
}
  
################################################################
####		   Rescaling Objects                        ####
################################################################

proc iscaleStart {x y} {
  global dragDist initialDragDist xdragOrigin ydragOrigin

  select_for_drag

  set latexSIds [[cv] find withtag s latex]
  if {![lempty $latexSIds]} {
    warn "Can't rescale latex objects - see latex help"
    [cv] addtag sLt withtag s latex
    [cv] addtag sLt withtag s ltxFr
    [cv] dtag sLt s
  }

  set ids [[cv] find withtag s]
  
  if {[lempty $ids]} {
    warn "No item selected"
    deselect_for_drag
    if {![lempty $latexSIds]} {
      [cv] addtag s withtag sLt
      [cv] dtag all sLt
    }
    return
  }
  dragStartHook
  iopStartHook $ids
  lassign [grid $x $y] x y
  
  [cv] delete sFr
  if {([llength $ids] == 1) && ([[cv] type $ids] == "dline")} {
    lassign [[cv] coords $ids -bbox] x1 y1 x2 y2
  } else {
    lassign [[cv] bbox s] x1 y1 x2 y2
  }

  set xdragOrigin [expr "0.5 * ($x1 + $x2)"]
  set ydragOrigin [expr "0.5 * ($y1 + $y2)"]
    
  set dragDist \
    [expr "($x-$xdragOrigin)*($x-$xdragOrigin) +\
             ($y-$ydragOrigin)*($y-$ydragOrigin)"]
  set initialDragDist $dragDist
}

proc iscaleMotion {x y} {
  global dragDist xdragOrigin ydragOrigin
  global Dragging
  
  if {!$Dragging} { return }

  lassign [grid $x $y] x y
  set dist [expr "($x-$xdragOrigin)*($x-$xdragOrigin) +\
             ($y-$ydragOrigin)*($y-$ydragOrigin)"]
  set fact [expr (0.0 + $dist) / $dragDist]
  
  set dragDist $dist
  [cv] scale toOp \
    $xdragOrigin $ydragOrigin $fact $fact
}

proc iscaleEnd {x y} {
  global initialDragDist xdragOrigin ydragOrigin
  global Dragging GriddingMode
  
  if {!$Dragging} {
    dragEndHook
    return
  }

  # compute global scale parameters #
  lassign [grid $x $y] x y
  set dist \
    [expr "($x-$xdragOrigin)*($x-$xdragOrigin) +\
             ($y-$ydragOrigin)*($y-$ydragOrigin)"]
  set fact [expr ($dist + 0.0) / $initialDragDist]

  waitCursor
  [cv] scale s $xdragOrigin $ydragOrigin $fact $fact
  iopEndHook
  restoreCursor

  deselect_for_drag
  [cv] addtag s withtag sLt
  [cv] dtag all sLt
  redrawFrames
  dragEndHook
}

################################################################
####		    Aligning objects                        ####
################################################################

proc Align {} {
  global vertAlignMode horizAlignMode
  
  if { ($horizAlignMode == {none}) && ($vertAlignMode == {none}) } {
    return
  }

  ##  align on the older selected item if we find it  ##
  set bb [[cv] bbox firstS]
  if {$bb == ""} {
    if {[set bb [[cv] bbox s]] == ""} { return }
  }
  
  ##  find selected groups  ##
  set gtags [getTopGroupTags s]
  
  ##  set reference positions  ##
  switch $horizAlignMode {
    center { set x0 [expr "0.5 * ([lindex $bb 0] + [lindex $bb 2])"] }
    left   { set x0 [lindex $bb 0] }
    right  { set x0 [lindex $bb 2] }
  }
  switch $vertAlignMode {
    center { set y0 [expr "0.5 * ([lindex $bb 1] + [lindex $bb 3])"] }
    top    { set y0 [lindex $bb 1] }
    bottom { set y0 [lindex $bb 3] }
  }
  
  foreach gtag $gtags {
    lassign [[cv] bbox $gtag] x1 y1 x2 y2
    
    switch $horizAlignMode {
      center { set dx [expr "$x0 - 0.5 * ($x1 + $x2)"] }
      left   { set dx [expr "$x0 - $x1"] }
      right  { set dx [expr "$x0 - $x2"] }
      none   { set dx 0 }
      default { return }
    }
    
    switch $vertAlignMode {
      center  { set dy [expr "$y0 - 0.5 * ($y1 + $y2)"] }
      top     { set dy [expr "$y0 - $y1"] }
      bottom  { set dy [expr "$y0 - $y2"] }
      none    { set dy 0 }
      default { return }
    }
    
    [cv] move $gtag $dx $dy
  }
  
  redrawFrames
  ####  UNDO DISABLED  ####
#  histAdd
}

##  Aligning the selection to the grid  ##

proc AlignToGrid {} {

  ##  find selected groups  ##
  set gtags [getTopGroupTags s]

  foreach gtag $gtags {
    set ids [[cv] find withtag $gtag]
    if {[llength $ids] == 1} {
      set coords [[cv] coords $ids]
      if {[llength $coords] == 2} {
	##  Single object with 2 coords (text, ...)  ##
	##  => align coords  ##
	eval "[cv] coords $ids \
                [eval "grid $coords"]"
	continue
      }
    }

    ##  Grouped objects or multi-coordinate obj.  ##
    ##  => align bbox  ##
    lassign [[cv] bbox $gtag] x1 y1
    lassign [grid $x1 $y1] X1 Y1
    [cv] move $gtag \
      [expr $X1 - $x1] [expr $Y1 - $y1]
  }
  redrawFrames
}
  
##  Displaying the alignment mode into the small canvas  ##

proc alignShowModeInit {} {
  global alsx1 alsx2 alsx3 alsy1 alsy2 alsy3
  global alsit1 alsit2 alsit3
  
  set alsx1(none) {14.8 50.8}
  set alsx2(none) {40 58}
  set alsx3(none)  {68 82}
  
  set alsx1(left) {14.8 50.8}
  set alsx2(left) {14.8 32.8}
  set alsx3(left) {14.8 28.8}
  
  set alsx1(center) {31 67}
  set alsx2(center) {40 58}
  set alsx3(center) {42 56}
  
  set alsx1(right)  {46 82}
  set alsx2(right)  {64 82}
  set alsx3(right)  {68 82}
  
  set alsy1(none) {16 30.4}
  set alsy2(none) {22 52}
  set alsy3(none) {43 70}
  
  set alsy1(top)  {16 30.4}
  set alsy2(top)  {16 46}
  set alsy3(top)  {16 43}
  
  set alsy1(center) {29.8 47.2}
  set alsy2(center) {22 52}
  set alsy3(center) {23.5 50.5}
  
  set alsy1(bottom) {52.6 70}
  set alsy2(bottom) {40 70}
  set alsy3(bottom) {43 70}
  
  set alsit1 [[al] create rectangle 14.8 16 50.8 30.4 \
	      -width 2 -outline {#66ce92}]
  set alsit2 [[al] create oval 40 22 58 52 \
	      -width 2 -outline {#d67575}]
  set alsit3 [[al] create rectangle 68 43 82 70 \
	      -width 2 -outline {#657fa3}]
}

proc alignShowMode {} {
  global vertAlignMode horizAlignMode
  global alsx1 alsx2 alsx3 alsy1 alsy2 alsy3
  global alsit1 alsit2 alsit3
  
  [al] coords $alsit1\
    [lindex $alsx1($horizAlignMode) 0] \
    [lindex $alsy1($vertAlignMode) 0] \
    [lindex $alsx1($horizAlignMode) 1] \
    [lindex $alsy1($vertAlignMode) 1]
  [al] coords $alsit2 \
    [lindex $alsx2($horizAlignMode) 0] \
    [lindex $alsy2($vertAlignMode) 0] \
    [lindex $alsx2($horizAlignMode) 1] \
    [lindex $alsy2($vertAlignMode) 1]
  [al] coords $alsit3 \
    [lindex $alsx3($horizAlignMode) 0] \
    [lindex $alsy3($vertAlignMode) 0] \
    [lindex $alsx3($horizAlignMode) 1] \
    [lindex $alsy3($vertAlignMode) 1]

  Align
}
