#
# Quit the interface.  This consists of saving our newsrc and tknewsrc,
# and informing the logging server we have shut down.
#
proc tknews_exit {} {
    global CommandStream LogFile Resources

    save_newsrc
    save_tknewsrc
    if {[info exists LogFile]} {
	catch {puts $LogFile "$Resources(USER) quit"}
	catch {flush $LogFile}
    }
    destroy .
    exit
}

#
# Abort the interface.  In this case we don't save anything, but we
# still inform the logging server of our exit.
#
proc tknews_abort {} {
    global LogFile
    if {[info exists LogFile]} {
	catch {puts $LogFile "$Resources(USER) quit"}
	catch {flush $LogFile}
    } 
    destroy .
    exit
}

#
# Bind the tab key to 5 spaces.  Shuttle the focus to the metainfo
# frame when the text widget receives focus.
#
proc bind_text {w} {
    bind $w <Key-Tab> "$w insert end \"     \""
    bind $w <FocusIn> "focus .metainfo"
}

#
# This sets the global variable InterfaceState to the value indicated.
# It also turns the cursor into a watch if state is set to "busy" or 
# "wait".  This watch is implemented via a C proc because it is much
# faster (like one or two seconds faster).
#
proc interface_state {state} {
    global CursorList InterfaceBusy

    if {$state == "busy" || $state == "wait"} {
	SetCursor wait
	set InterfaceBusy 1
    } else {
	SetCursor normal
	set InterfaceBusy 0
    }
    update
    return
}

#
# This does nothing... kinda like me.
#
proc noop {} {}
	
    
#
# This takes a canvas widget, an item id, and a space
# argument and returns the next adjustment necessary to
# place the next item 'space' pixels away from the
# lower right corner of the item with id 'id'.
#
proc bbox_incr {w id space} {
    set bbox [$w bbox $id]
    set xa [expr [lindex $bbox 2]-[lindex $bbox 0]]
    incr xa $space
    set ya [expr [lindex $bbox 3]-[lindex $bbox 1]]
    incr ya $space
    return [list $xa $ya]
}


proc write_cpanel_help {count} {
    global ButtonList InButton

    set InButton $count
    write_message $ButtonList(help,$count)
    catch {.cpanel.buttons.button$count configure -state active}
}
    
proc undo_cpanel_help {count} {
    global InButton

    set InButton -1
    write_status
    catch {.cpanel.buttons.button$count configure -state normal}
}

proc cpanel_execute {count} {
    global InButton

    update
    .cpanel.buttons.button$count configure -relief raised
    if {$InButton == $count} {
	.cpanel.buttons.button$count invoke
    }
}

#
# This procedure binds the command button indicated to a short
# help message if it is pre-defined.  This message is displayed
# when the user moves over it.  If there is not message
# defined for the action of the button, nothing is bound to it.
#
proc bind_cpanel {} {
    global Resources ButtonList

    if {$Resources(commandMessages) == ""} {return}
    set list [multi_split $Resources(commandMessages) "\\n"]
    set list [lreplace $list 0 0]
    set count 0
    foreach mess $list {
	set ButtonList(help,$count) $mess
	catch {bind .cpanel.buttons.button$count <Enter> "write_cpanel_help $count"}
	catch {bind .cpanel.buttons.button$count <Leave> "undo_cpanel_help $count"}
	catch {bind .cpanel.buttons.button$count <ButtonRelease-1> "cpanel_execute $count"}
	incr count
    }
}

#
# This procedure builds the command button frame.  This isn't working
# right now....
#
proc build_cpanel {} {
    global Resources ButtonList

    if {$Resources(commandPanel) == ""} {
	return
    }
    frame .cpanel -borderwidth 0
    frame .cpanel.buttons -borderwidth 0
    pack configure .cpanel.buttons -side top -fill y
    # 
    # This section of code parses the button panel items, placing them
    # in the canvas in a row.  Buttons may be either text or bitmaps.
    #
    set c .cpanel.buttons
    set x 0
    set count 0
    set list [multi_split $Resources(commandPanel) "\\n"]
    set list [lreplace $list 0 0]
    foreach item $list {
	set a [split $item ":"]
	set name [string trim [lindex $a 0]]
	set action [string trim [lindex $a 1]]
	set ButtonList(name,$count) $name
	set ButtonList(action,$count) $name
	# is this a bitmap button?
	if {[string index $name 0] == "@"} {
	    if {[file exists [string range $name 1 end]]} {
		set name $name
	    } else {
		set name @$Resources(bitmapDir)/[string range $name 1 end]
	    }
	    button $c.button$count \
		-bitmap $name \
		-command $action \
		-borderwidth 2
	} else {
	    button $c.button$count \
		-text $name \
		-command $action \
		-borderwidth 2

	}
	if {$Resources(cpanelForeground) != ""} {
	    configure $c.button$count \
		-foreground $Resources(cpanelForeground) 
	}
	if {$Resources(cpanelBackground) != ""} {
	    configure $c.button$count \
		-background $Resources(cpanelBackground) 
	}
	pack configure $c.button$count -side left -fill none
	incr count
    }
    set ButtonList(NumButtons) $count
}

# Makes info a toplevel window instead of a frame
proc make_toplevel_info {} {
    global Options Resources LIBRARY_DIR
    toplevel .info
    if {[winfo ismapped .] == 0} {
	wm withdraw .info
    }
    wm minsize .info 0 0
    bind .info <Enter> "set inwin 1"
    bind .info <Leave> "set inwin 0"
    set Resources(infoGeometry) [option get .info infoGeometry Geometry]
    if {$Options(Iconify)} {
	wm geometry .info -4000-4000
    } else {
	if {$Resources(windowGeometry) != ""} {
 	   wm geometry .info $Resources(infoGeometry)
	}
    }
    wm maxsize .info [winfo vrootwidth .] [winfo vrootheight .]
    wm positionfrom .info user
    wm iconbitmap .info @$LIBRARY_DIR/bitmaps/newsicon
}

# Puts the text/scrollbar inside .info widget
proc make_info_window {} {
    global Resources
    scrollbar .info.vertical -command ".info.text yview"
    text .info.text \
	-yscrollcommand ".info.vertical set" \
	-exportselection 1 \
	-font $Resources(textFont)
    pack configure .info.vertical -side left -fill y
    pack configure .info.text -side top -fill both -expand 1
}

#
# This proc constructs the user interface.
#
proc build_widgets {} {
    global HasCpanel CpanelOrientation LIBRARY_DIR Options
    global inwin Resources InterfaceOrientation

    wm minsize . 0 0
    bind . <Enter> "set inwin 1"
    bind . <Leave> "set inwin 0"
    set Resources(windowGeometry) [option get . windowGeometry Geometry]
    if {$Options(Iconify)} {
	wm geometry . -4000-4000
	if {$InterfaceOrientation == "separate"} {
	    wm iconify .
	}
    } else {
	if {$Resources(windowGeometry) != ""} {
	    wm geometry . $Resources(windowGeometry)
	}
    }
    wm maxsize . [winfo vrootwidth .] [winfo vrootheight .]
    wm positionfrom . user
    wm iconbitmap . @$LIBRARY_DIR/bitmaps/newsicon
    frame .grabframe 
    frame .menubar
    button .menubar.about -command "version_info" -text "About"
    menubutton .menubar.file -menu .menubar.file.m
    menubutton .menubar.action -menu .menubar.action.m
    menubutton .menubar.view -menu .menubar.view.m
    menubutton .menubar.move -menu .menubar.move.m 
    menubutton .menubar.mark -menu .menubar.mark.m
    menubutton .menubar.options -menu .menubar.options.m
    menubutton .menubar.font -menu .menubar.font.m
    menubutton .menubar.help -menu .menubar.help.m
    pack configure \
	.menubar.file \
	.menubar.action \
	.menubar.move \
	.menubar.mark \
	.menubar.view \
	.menubar.font \
	.menubar.options \
	.menubar.help \
	-side left -fill none
    pack configure .menubar.about -side right -fill none
    frame .comm -borderwidth 1 -relief flat
    frame .comm.message
    message .comm.message.mess -aspect 3000
    canvas .comm.message.scroll
    pack configure .comm.message.mess -side top -fill both -expand 1
    pack configure .comm.message.scroll -side bottom -fill x -expand 1
    frame .comm.icon -borderwidth 1 -relief flat \
	-background white 
    button .comm.icon.object -bitmap @$LIBRARY_DIR/bitmaps/article.xbm \
	-background black -foreground white \
	-borderwidth 0 -relief raised \
	-activebackground black
    bind_for_dragndrop .comm.icon.object
    pack configure .comm.icon.object -side left -fill y 
    pack configure .comm.message -side left -fill both -expand 1
    pack configure .comm.icon -side right -fill y 
    frame .metainfo 
    scrollbar .metainfo.vertical -command ".metainfo.listbox yview"
    frame .metainfo.bottom -borderwidth 0
    scrollbar .metainfo.bottom.horizontal \
	-command ".metainfo.listbox xview" \
	-orient horizontal
    button .metainfo.bottom.adj -bitmap @$LIBRARY_DIR/bitmaps/grip.xbm \
	-borderwidth 2 -relief raised -anchor center
    configure_geometry_adjuster
    pack configure .metainfo.bottom.horizontal -side left -fill both -expand 1
    pack configure .metainfo.bottom.adj -side right -fill y

    listbox .metainfo.listbox \
	-exportselect false \
	-yscrollcommand ".metainfo.vertical set" \
	-xscrollcommand ".metainfo.bottom.horizontal set" 
    pack configure .metainfo.vertical -side left -fill y
    pack configure .metainfo.listbox -side top -fill both -expand 1
    pack configure .metainfo.bottom -side bottom -fill both
    frame .info
    make_info_window
    frame .holder -borderwidth 0
    bind_text .info.text
    build_cpanel
}

#
# This grip is the widget that allows the user to readjust the
# geometry of the metainfo and info frames.
#
proc start_grip {x y} {
    global grip_left grip_right grip_top grip_bottom grip_x grip_y
    global InterfaceOrientation
    
    incr x -[winfo rootx .]
    incr y -[winfo rooty .]
    set grip_x $x
    set grip_y $y
    #
    # extract information on the geometry of holder so we can compute
    # the end percentage and can limit the drawlines.
    #
    set grip_left [winfo x .holder]
    set grip_right [expr $grip_left+[winfo width .holder]]
    set grip_top [winfo y .holder]
    set grip_bottom [expr $grip_top+[winfo height .holder]]
    # set bounds on minimum sizes.
    incr grip_left 100
    incr grip_right -100
    incr grip_top 60
    incr grip_bottom -60
    if {$InterfaceOrientation == "vertical"} {
	DrawLine [winfo x .holder] $y \
	    [expr [winfo x .holder]+[winfo width .holder]] $y
    } else {
	DrawLine $x [winfo y .holder] $x \
	    [expr [winfo y .holder]+[winfo height .holder]]
    }
}

proc handle_grip {x y} {
    global grip_left grip_right grip_top grip_bottom grip_x grip_y
    global InterfaceOrientation
    
    incr x -[winfo rootx .]
    incr y -[winfo rooty .]
    if {$InterfaceOrientation == "vertical"} {
	if {$y < $grip_top} {set y $grip_top}
	if {$y > $grip_bottom} {set y $grip_bottom}
	set j [winfo x .holder]
	set k [expr $j + [winfo width .holder]]
	DrawLine $j $grip_y $k $grip_y
	DrawLine $j $y $k $y
    } else {
	if {$x < $grip_left} {set x $grip_left}
	if {$x > $grip_right} {set x $grip_right}
	set j [winfo y .holder]
	set k [expr $j+[winfo height .holder]]
	DrawLine $grip_x $j $grip_x $k
	DrawLine $x $j $x $k
    }
    set grip_x $x
    set grip_y $y
}

proc release_grip {x y} {
    global grip_top grip_bottom grip_left grip_right grip_x grip_y 
    global InterfaceOrientation Resources
    
    incr x -[winfo rootx .]
    incr y -[winfo rooty .]
    # These guys must not have a geometry, since they should resize
    # themselves to fit their frame's geometry.  Thus, we tell them
    # to be the smallest size possible.
    .metainfo.listbox configure -geometry 1x1
    .info.text configure -width 1 -height 1
    if {$InterfaceOrientation == "vertical"} {
	# adjust bounds
	if {$y < $grip_top} {set y $grip_top}
	if {$y > $grip_bottom} {set y $grip_bottom}
	# erase last line drawn.
	DrawLine [winfo x .holder] $grip_y \
	    [expr [winfo x .holder]+[winfo width .holder]] $grip_y
	# determine percentage of .holder above line.
	# i is how far down from the top of the .holder widget we are,
	# not necessarily how far from grip_top we are, since grip_top
	# will generally be used to limit the minimum size of the 
	# metainfo box.
	set i [expr $y-[winfo y .holder]]
	set height [winfo height .holder]
	set fraction [expr $i.0/$height.0]
	place configure .metainfo -relheight $fraction -relwidth 1.0 
	place configure .info -rely $fraction \
	    -relheight [expr 1.0 - $fraction] -relwidth 1.0
    } else {
	# adjust bounds
	if {$x < $grip_left} {set x $grip_left}
	if {$x > $grip_right} {set x $grip_right}
	# this stuff is the same as for the vertical case, but with
	# the natural alterations needed for the horizontal case.
	DrawLine $grip_x [winfo y .holder] $grip_x \
	    [expr [winfo y .holder]+[winfo height .holder]]
	set i [expr $x-[winfo x .holder]]
	set width [winfo width .holder]
	set fraction [expr $i.0/$width.0]
	place configure .metainfo -relwidth $fraction -relheight 1.0
	place configure .info -relx $fraction \
	    -relwidth [expr 1.0 - $fraction] -relheight 1.0
    }
    set Resources(fraction) $fraction
}

#
# This sets the text label for the geometry adjuster buttons
# based on the value of "InterfaceOrientation"
#
proc configure_geometry_adjuster {} {
    global InterfaceOrientation ButtonList

    # This is a bad way to do help for the grip, but it works.
    # We pretend the grip is the -2 button in the cpanel.
    set ButtonList(help,-2) \
	"Change the size of the window.  Hold down button and drag..."
    bind .metainfo.bottom.adj <Button-1>           {start_grip %X %Y}
    bind .metainfo.bottom.adj <B1-Motion>          {handle_grip %X %Y}
    bind .metainfo.bottom.adj <B1-ButtonRelease-1> {release_grip %X %Y}
    bind .metainfo.bottom.adj <Enter>              {write_cpanel_help -2}
    bind .metainfo.bottom.adj <Leave>              {undo_cpanel_help -2}
}


#
# This procedure sets the interface's orientation to the 
# value indicated by the ''which'' argument.
#
proc change_view {which} {
    global InterfaceOrientation NewGeom

    set InterfaceOrientation $which
    if {[info exists NewGeom]} {
	unset NewGeom
    }
    unpack_interface
    pack_interface
}

# 
# This procedure unpacks everything from the toplevel window.
#
proc unpack_interface {} {
    global HasCpanel OldGeometry Options Resources

    set OldGeometry [wm geometry .]
    catch {pack forget .cpanel}
    catch {pack forget .menubar}
    catch {pack forget .comm}
    catch {place forget .metainfo}
    catch {place forget .info}
    update
}

#
# This procedure sets the InterfaceOrientation variable to the
# correct value.
#
proc get_orientation {} {
    global InterfaceOrientation
    if {![info exists InterfaceOrientation]} {
	set i [catch {set answer [option get . layout Layout]}]
	set InterfaceOrientation "vertical"
	if {$i == 0} {
	    if {$answer == "horizontal"} {
		set InterfaceOrientation "horizontal"
	    } elseif {$answer == "separate"} {
		set InterfaceOrientation "separate"
	    }
	}
    }
}

#
# This procedure repacks the interface.
#
proc pack_interface { {do_cpanel 1} {do_text 1} {do_menubar 1} {do_comm 1}} {
    global HasCpanel OldGeometry  InterfaceOrientation Options Resources

    set packing $InterfaceOrientation
    .metainfo.listbox configure -geometry 1x1
    .info.text configure -width 1 -height 1
    if {$packing == "horizontal"} {
	if {$do_menubar} {
	    pack configure .menubar -side top -fill x
	}
	if {$do_comm} {
	    pack configure .comm -side top -fill both
	}
	# only show the command panel if it is defined and
	# the command argument said we should do it.
	if {$Resources(commandPanel) != "" && $do_cpanel} {
	    pack configure .cpanel -side top -fill x
	}
	pack configure .holder -side bottom -fill both -expand 1
	if {$do_text} {
	    place .metainfo -in .holder -relx 0.0 -rely 0.0 \
		-relheight 1.0 -relwidth $Resources(fraction)
	    if {([winfo class .info] == "Toplevel")} {
		destroy .info
		frame .info
		make_info_window
	    }
	    place .info -in .holder -relx $Resources(fraction) -rely 0.0 \
		-relheight 1.0 -relwidth [expr 1.0 - $Resources(fraction)]
	} else {
	    place .metainfo -in .holder -relx 0.0 -rely 0.0 \
		-relheight 1.0 -relwidth 1.0
	}
        pack configure .metainfo.bottom.adj -side right -fill y
    } elseif {$packing == "vertical"} {
	if {$do_menubar} {
	    pack configure .menubar -side top -fill x
	}
	if {$do_comm} {
	    pack configure .comm -side top -fill both
	}
	# only show the command panel if it is defined.
	if {$Resources(commandPanel) != "" && $do_cpanel} {
	    pack configure .cpanel -side top -fill x
	}
	pack configure .holder -side bottom -fill both -expand 1
	if {$do_text} {
	    place .metainfo -in .holder -relx 0.0 -rely 0.0 \
		-relheight $Resources(fraction) -relwidth 1.0
	    if {([winfo class .info] == "Toplevel")} {
		destroy .info
		frame .info
		make_info_window
	    }
	    place .info -in .holder -relx 0.0 -rely $Resources(fraction) \
		-relheight [expr 1.0 - $Resources(fraction)] -relwidth 1.0
	} else {
	    place .metainfo -in .holder -relx 0.0 -rely 0.0 \
		-relheight 1.0 -relwidth 1.0
	}
        pack configure .metainfo.bottom.adj -side right -fill y
    } else {
# Separate packing. Like Vertical, but replace frame .info with toplevel .info
	if {$do_text && ([winfo class .info] == "Frame")} {
	    destroy .info
	    make_toplevel_info
	    make_info_window
	}
	pack forget .metainfo.bottom.adj
	if {$do_menubar} {
	    pack configure .menubar -side top -fill x
	}
	if {$do_comm} {
	    pack configure .comm -side top -fill both
	}
	# only show the command panel if it is defined.
	if {$Resources(commandPanel) != "" && $do_cpanel} {
	    pack configure .cpanel -side top -fill x
	}
	pack configure .holder -side bottom -fill both -expand 1
	place .metainfo -in .holder -relx 0.0 -rely 0.0 \
		-relheight 1.0 -relwidth 1.0
    }
    lower .holder
    update
}

#
# This loads in all of the necessary libraries.
#
proc load_libs {} {
    global SCRIPT_DIR

    source $SCRIPT_DIR/actions.t
    source $SCRIPT_DIR/menus.t
    source $SCRIPT_DIR/utils.t
    source $SCRIPT_DIR/log.t
    source $SCRIPT_DIR/help.t
    source $SCRIPT_DIR/dragndrop.t
    source $SCRIPT_DIR/respond.t
    source $SCRIPT_DIR/appdef.t
    source $SCRIPT_DIR/dialog.t
    source $SCRIPT_DIR/tutorial.t
}

#
# Write a sticky message to the upper info line (stays there until you
# write somethign else to it).
#
proc write_message {mess} {

    .comm.message.mess config -text $mess
}

#
# This is the procedure used to write a message on the scrolling "marque".
#
proc write_scroll {mess} {
    global ScrollMess ScrollLength ScrollId ScrollPos ScrollFont ScrollFill
    global ScrollStop MESSAGE_DELAY ScrollOn

    set ScrollMess $mess
    set x [winfo width .comm.message.scroll]
    set x2 [winfo width .comm.message.mess]
    set y [winfo height .comm.message.scroll]
    .comm.message.scroll delete $ScrollId
    set ScrollId [.comm.message.scroll create text $x [expr $y/2] \
		  -font $ScrollFont \
		      -justify center -anchor center -text $mess \
		      -fill $ScrollFill]
    set bbox [.comm.message.scroll bbox $ScrollId]
    set ScrollLength [expr [lindex $bbox 2]-[lindex $bbox 0]]
    set ScrollStop [expr $x/2]
    set ScrollPos $x
    if {!$ScrollOn} {
	.comm.message.scroll move $ScrollId -[expr $ScrollPos-$ScrollStop] 0
	update
    } else {
	while {[update_scroll] != -1} { 
	}
    }
    after $MESSAGE_DELAY .comm.message.scroll delete $ScrollId
}

#
# Move the message along the canvas' marque.
#
proc update_scroll {} {
    global ScrollId ScrollSpeed ScrollPos
    global ScrollStop

    .comm.message.scroll move $ScrollId $ScrollSpeed 0
    incr ScrollPos $ScrollSpeed
    if {$ScrollPos < $ScrollStop} {
	.comm.message.scroll move $ScrollId [expr $ScrollStop-$ScrollPos] 0
	update
	return -1
    } else {
	update
	return 0
    }
}    

proc write_brief {mess} {
    global MESSAGE_DELAY

    write_scroll $mess
    return
    write_message $mess
    update
    after $MESSAGE_DELAY write_status
}


#
# split a string based on a multi-character string.
#
proc multi_split {s sequence} {
    
    set i [string length $sequence]
    set index 0
    set a 0
    set return {}
    while {$a >= 0} {
	set a [string first $sequence $s]
	incr a -1
	if {$a >= 0} {
	    set elem [string range $s 0 $a]
	    lappend return $elem
	    incr a [expr $i+1]
	    set s [string range $s $a end]
	    set index $a
	}
    }
    lappend return $s
    return $return
}
    
proc generate_cursorlist {} {
    global CursorList

    set CursorList {.app .root}
    set list [winfo children .]
    while {$list != ""} {
	set next {}
	foreach w $list {
	    set class [winfo class $w]
	    set cursor [lindex [$w config -cursor] 4]
	    lappend CursorList [list $w $cursor]
	    set next [concat $next [winfo children $w]]
	}
	set list $next
    }
}


#
# Load in the keyboard translations ala Motif.
#
proc set_translations {} {
    global CursorList BindingList 
    global Resources

    foreach item $CursorList {
	set w [lindex $item 0]
	set i [catch {set answer [option get $w translations Translations]}]
	if {$i == 0} { 
	    set list [multi_split $answer "\\n"]
	    set specified 1
	    # determine whether we are replacing, overriding, or augmenting 
	    set b [lindex $list 0]
	    case $b in {
		{#override} {set mode "override"} 
		{#augment}  {set mode "augment"}
		{#replace}  {set mode "replace"}
		{default}   {set mode "replace" ; set specified 0}
	    }
	    if {$specified} {
		# They specified a redirect, so we delete that from the list
		# now....
		set list [lreplace $list 0 0]
	    }
	    foreach elem $list {
		if {$elem == {}} {continue}
		set a [split $elem ":"]
		set binding [string trim [lindex $a 0]]
		set action [string trim [lindex $a 1]]
		if {$mode == "augment"} {
		    set action +$action
		}
		set i [catch {bind $w $binding $action}]
		if {!$i} {
		    lappend BindingList($action) [list $w $binding]
		}
	    }
	}
    }
}


#
# This does all sorts of things.  MESSAGE_DELAY is the time we wait before
# removing a message from the marque.  TIME_GRAIN is how often we check to
# see if it is time to rescan.  ControlC is used to simulate a real
# control-C.  
#
# The init process goes like this :
# 1)  Connect to nntp server if necessary.
# 2)  Source nntp and spool scripts if necessary.
# 3)  open the logfile
# 4)  start the auto-rescan checking (via self-perpetuating after comms).
# 5)  set the interface state to busy
# 6)  call get_subscribed to load in the active file and the newsrc.
# 7)  If we are checking for new newsgroups automatically then call all_groups
#
proc init_interface {} {
    global Resources
    global ControlC
    global MESSAGE_DELAY SCRIPT_DIR TIME_GRAIN
    global Mode Newsgroup NumGroups NumArticles NumActive
    global PostingList Resources CurrentArticle
    global Options NewGroups

    set MESSAGE_DELAY 8000
    set TIME_GRAIN [expr 5*1000]
    set Mode "Groups"
    set Newsgroup ""
    set NewGroups ""
    set NumGroups 0
    set NumArticles 0
    set CurrentArticle -1
    set ControlC 0
    set PostingList {}
    interface_state busy
    if {[info exists Resources(NNTPSERVER)]} {
	source $SCRIPT_DIR/nntp.t
	load_status "Connecting to nntp server $Resources(NNTPSERVER)"
	if {[info exists Resources(NNTPPORT)]} {
	    server_connect $Resources(NNTPSERVER) $Resources(NNTPPORT)
	} else {
	    server_connect $Resources(NNTPSERVER) nntp
	}
    } else {
	if {[info exists Resources(NNTPPOST)]} {
	    source $SCRIPT_DIR/nntp.t 
	    load_status "Connecting to nntp server $Resources(NNTPPOST)"
	    if {[info exists Resources(NNTPPORT)]} {
		server_connect $Resources(NNTPPOST) $Resources(NNTPPORT)
	    } else {
	        server_connect $Resources(NNTPPOST) nntp
	    }
	    source $SCRIPT_DIR/spool.t
	}  else {
	    source $SCRIPT_DIR/spool.t
	}
    }
    load_status "Loading active list"
    start_auto_rescan
    interface_state busy
    load_status "Getting list of subscribed newsgroups"
    get_subscribed 
    interface_state normal
    if {$Options(AutoNewCheck)} {
	add_new_groups
    }
}

#
# This procedure does the bare minimum to allow us to display messages
# to the user as the interface is initialized.
#
proc init_tknews {} {
    global CommandStream
    global TabString Version Focus env
    global Subjline AutoSubscribe ViewUnread
    global TclVersion
    

    set Version "1.2b"
    set TclVersion [info tclversion]
    wm title . "Tknews"
    set a [option get . focusWindow Focus]
    set Focus "."
    if {$a != ""} {
	set Focus $a
    } 
    focus default $Focus
    focus $Focus
    init_scroll
    set ViewUnread 0
}


#
# intialize the scrolling marquees.
#
proc init_scroll {} {
    global ScrollFont ScrollFill ScrollOn ScrollSpeed
    global ScrollMess ScrollId ScrollPos SCROLL_DELAY
    
    set ScrollMess "" 
    set ScrollId 0
    set ScrollPos 0
    set SCROLL_DELAY 5
    set SCROLL_INC -16
    
    set ScrollFont [option get . scrollFont Font]
    if {$ScrollFont == ""} {
	set ScrollFont "fixed"
    }
    set ScrollFill [option get . scrollFill Foreground]
    if {$ScrollFill == ""} {
	set ScrollFill "black"
    }
    set ScrollOn [option get . scrollOn Boolean]
    set ScrollOn [string tolower $ScrollOn]
    if {$ScrollOn == "false" || $ScrollOn == "no" || $ScrollOn == "0"} {
	set ScrollOn 0
    } else {
	set ScrollOn 1
    }
    set ScrollSpeed [option get . scrollSpeed Pixel]
    
#    set i [catch "set ScrollSpeed [expr $ScrollSpeed*-1]"]
    set i [catch "set ScrollSpeed [expr $ScrollSpeed-1]"]
    if {$i || $ScrollSpeed >= 0} {
	puts stdout "Can't parse ScrollSpeed"
	set ScrollSpeed -8
    } 
}

#
# This procedure sets the various parts of the Resources array
# that corrorpond to environemnt variables.  This is different from
# load_environment because the resource database is what is being
# queried as opposed to the environment variables.
#
proc load_env_resources {} {
    global Resources EnvList

    foreach var $EnvList {
	set s [option get . $var Environment]
	if {$s != ""} {
	    # there is an option for this variable
	    set Resources($var) $s
	}
	# only tilde_sub if there isn't a space in the name, otherwise
	# tilde_sub will enclose the return value in angle brackets.
	if {[info exists Resources($var)]} {
	    if {[string first " " $Resources($var)] == -1} {
		catch "set Resources($var) [tilde_sub $Resources($var)]"
	    }
	} 
    }
    set showall [option get . showallHeaders Boolean]
    set showall [string tolower $showall]
    if {$showall == "true" || $showall == "yes" || $showall == "1"} {
	set Resources(showallHeaders) 1
    } else {
	set Resources(showallHeaders) 0
    }
    set Resources(signature) [tilde_sub [option get . signature File]]
}

#
# This loads in the custom resources which don't corrospond to 
# environment variables.
#
proc load_resources {} {
    global Resources Options LIBRARY_DIR env

    # first load in the resource options corrosponding to environment
    # variables.
    load_env_resources
    # Now load in non-environment options
    set Resources(followupExpr) [option get . followupExpr Followup]
#    if {$Resources(followupExpr) == ""} {
#	set Resources(followupExpr) "^([ \t]*>)|^([ \t]*\|>)|^(\[ \t]*\()"
#    }
    set Resources(highlightExpr) [option get . highlightExpr Highlight]
    if {$Resources(highlightExpr) == ""} {
	set Resources(highlightExpr) "^Subject: "
    }
    set Resources(followupFont) [option get . followupFont Font]
    set Resources(bigFont) [option get . bigFont Font]
    set Resources(smallFont) [option get . smallFont Font]
    set Resources(boldFont) [option get . boldFont Font]
    set Resources(italicFont) [option get . italicFont Font]
    set Resources(highlightForeground) [option get . highlightForeground Foreground]
    set Resources(highlightBackground) [option get . highlightBackground Background]
    set Resources(highlightRelief) [option get . highlightRelief Relief]
    set Resources(highlightBorderWidth) [option get . highlightBorderWidth BorderWidth]
    set Resources(rescanDelay) [option get . rescanDelay Time]
    if {$Resources(rescanDelay) == ""} {
	set Resources(rescanDelay) 30
    }
    set Resources(rescanDelay) [expr $Resources(rescanDelay)*60000]
    set Resources(autoHelp) [option get . autoHelp Help]
    if {$Resources(autoHelp) == ""} {
	set Resources(autoHelp) 1
    }
    set Resources(maxGrab) [option get . maxGrab Int]
    if {$Resources(maxGrab) == ""} {
	set Resources(maxGrab) 100
    }
    set Resources(logAddr) [option get . logAddr String]
    if {$Resources(logAddr) == ""} {
	set Resources(logAddr) "dimetrodon.cis.ohio-state.edu"
    }
    set Resources(logPort) [option get . logPort String]
    if {$Resources(logPort) == ""} {
	set Resources(logPort) 5000
    }
    set Resources(defaultFont) [option get . defaultFont Font]
    if {$Resources(defaultFont) == ""} {
	set Resources(defaultFont) "fixed"
    }
    set Resources(textFont) [option get . textFont Font]
    if {$Resources(textFont) == ""} {
	set Resources(textFont) $Resources(defaultFont)
    }
    catch {.info.text configure -font $Resources(textFont)}
    set Resources(font1) [option get . font1 Font]
    if {$Resources(font1) == ""} {
	set Resources(font1) fixed
    }
    set Resources(font1Label) [option get . font1Label Text]
    if {$Resources(font1Label) == ""} {
	set Resources(font1Label) $Resources(font1)
    }
    set Resources(font2) [option get . font2 Font]
    if {$Resources(font2) == ""} {
	set Resources(font2) fixed
    }
    set Resources(font2Label) [option get . font2Label Text]
    if {$Resources(font2Label) == ""} {
	set Resources(font2Label) $Resources(font2)
    }
    set Resources(font3) [option get . font3 Font]
    if {$Resources(font3) == ""} {
	set Resources(font3) fixed
    }
    set Resources(font3Label) [option get . font3Label Text]
    if {$Resources(font3Label) == ""} {
	set Resources(font3Label) $Resources(font3)
    }
    set Resources(font4) [option get . font4 Font]
    if {$Resources(font4) == ""} {
	set Resources(font4) fixed
    }
    set Resources(font4Label) [option get . font4Label Text]
    if {$Resources(font4Label) == ""} {
	set Resources(font4Label) $Resources(font4)
    }
    set Resources(infoFont) [option get . infoFont Font]
    if {$Resources(infoFont) != ""} {
	catch {.metainfo.listbox configure -font $Resources(textFont)}
    }
    set Resources(bitmapDir) [option get . bitmapDir Directory]
    if {$Resources(bitmapDir) == ""} {
	set Resources(bitmapDir) $LIBRARY_DIR/bitmaps
    }
    set Resources(commandPanel) [option get . commandPanel CommandPanel]
    set Resources(commandMessages) [option get . commandMessages String]
    set Resources(cpanelForeground) [option get . cpanelForeground Foreground]
    set Resources(cpanelBackground) [option get . cpanelBackground Background]
    set Resources(cpanelSpace) [option get . cpanelSpace Pixel]
    if {$Resources(cpanelSpace) == ""} {
	set Resources(cpanelSpace) 5
    }
    set Resources(disableKeyboard) [option get . disableKeyboard Boolean]
    if {$Resources(disableKeyboard) == ""} {
	if {$Options(Gce)} {
	    set Resources(disableKeyboard) 1
	} else {
	    set Resources(disableKeyboard) 0
	}
    }
    set Resources(gceEditor) [option get . gceEditor Editor]
    if {$Resources(gceEditor) == ""} {
	set Resources(gceEditor) "emacs"
    }
    set Resources(gceMailer) [option get . gceMailer Mailer]
    if {$Resources(gceMailer) == ""} {
	set Resources(gceMailer) "mailt"
    }
    set Resources(fraction) [option get . fraction Fraction]
    if {$Resources(fraction) == ""} {
	set Resources(fraction) 0.3
    }
    set Resources(quotePrefix) [option get . quotePrefix String]
    if {$Resources(quotePrefix) == ""} {
	set Resources(quotePrefix) ">"
    }
    if {[info exists env(GCE_HOME)]} {
	set Resources(gceHome) $env(GCE_HOME)
    } else {
	set Resources(gceHome) [option get . gceHome String]
    }
    if {$Resources(gceHome) == ""} {
	set Resources(gceHome) "/usr/local/GCE"
    }
}


#
# This is the procedure called via send by the resedit application
# (found in the lib directory).  Its job is to apply all of the
# changes which might have been made in the resources.
#
proc resedit_apply {file} {
    global Mode

    option readfile $file interactive
    catch {exec /usr/bin/sh -c "rm $file"}
    load_resources
    if {$Mode == "Articles"} {
	set char [.info.text get 1.0]
	if {$char == "S" || $char == "s"} {
	    set_text [.info.text get 1.0 end]
	}
    }
}

#
# This procedure sets the appropriate resource variables to the
# values described in the user's environment variables.  This is
# called before the resource database is checked.  Thus, resources
# override environment variables.
#
proc load_environment {} { 
    global Resources env EnvList

    set EnvList [list NNTPSERVER NNTPPORT NNTPPOST NEWSACTIVEFILE NEWSRCFILE \
		 SPOOLDIR EDITOR VISUAL POSTINGPROG TKNEWSRC NAME \
		     ORGANIZATION PREFETCH]
    foreach var $EnvList {
	if {[info exists env($var)]} {
	    set Resources($var) $env($var)
	}
    }
}
    
# 
# This procedure is used as a last safety measure against missing
# parameters in the configuration variables.
#
proc check_environment {} {
    global Resources env

    if {![info exists Resources(PREFETCH)]} {
	set Resources(PREFETCH) 10
    }
    if {![info exists Resources(NEWSRCFILE)]} {
	set Resources(NEWSRCFILE) [tilde_sub ~/.newsrc]
    }
    if {![info exists Resources(NEWSACTIVEFILE)]} {
	set Resources(NEWSACTIVEFILE) "/usr/lib/news/active"
    }	
    if {[info exists Resources(VISUAL)]} {
	set Resources(EDITOR_COMMAND) $Resources(VISUAL)
    } else {
	if {[info exists Resources(EDITOR)]} {
	    set Resources(EDITOR_COMMAND) $Resources(EDITOR)
	    if {$Resources(EDITOR_COMMAND) == "vi"} {
		set $Resources(EDITOR_COMMAND) "xterm -e vi"
	    }
	} else {
	    if {[file exists "/usr/local/bin/emacs"]} {
		set Resources(EDITOR_COMMAND) "/usr/local/bin/emacs"
	    } else {
		if {[file exists "/usr/bin/emacs"]} {
		    set Resources(EDITOR_COMMAND) "/usr/bin/eamcs"
		} else {
		    error "No editor defined.... you probably need to set the VISUAL environment variable to an appropriate value."
		}
	    }
	}
    }
    if {![info exists Resources(POSTINGPROG)]} {
	set Resources(POSTINGPROG) "nntp"
    }    
    if {![file exists "/usr/ucb/whoami"]} {
	if {![info exists env(USER)]} {
	    if {![info exists env(LOGNAME)]} {
	        error "Cannot determine your login name... please set the USER environment variable."
	    } else {
	        set Resources(USER) $env(LOGNAME)
	    }
	} else {
	    set Resources(USER) $env(USER)
	}
    } else {
	set Resources(USER) [exec /usr/ucb/whoami]
    }
    if {![info exists Resources(TKNEWSRC)]} {
	set Resources(TKNEWSRC) [tilde_sub ~/.tknewsrc]
    }
    if {![info exists Resources(NAME)]} {
	set i [catch {set s [exec ypmatch $USER passwd]}]
	if {$i} {
	    set i [catch {set s [exec grep $USER /etc/passwd]}]
	    if {$i} {
		set NAME "??"
	    }
	}
	if {!$i} {
	    set s [split $s ":"]
	    set NAME [lindex $s 4]
	    set i [string first "," $NAME]
	    set NAME [string range $NAME 0 [expr $i-1]]
	}
    }
}

#
# This procedure obtains the qualified internet address for the host.
# 
proc get_hostaddr {} {
    global Resources env

    if {![info exists env(HOST)]} {
	# SEE IF WE CAN FIND OUT OUR HOSTNAME
	if {[file exists /usr/bin/hostname]} {
	    set env(HOST) [exec /usr/bin/hostname]
	} else {
	    if {[file exists /bin/hostname]} {
		set env(HOST) [exec /bin/hostname]
	    } else {
		if {[file exists /usr/local/bin/hostname]} {
		    set env(HOST) [exec /usr/local/bin/hostname]
		} else {
		    error_dialog "Unable to determine the hostname of your machine... Please set the HOST environment variable and startup tknews again.\n"
		    tknews_abort
		}
	    }
	}
    }
    set Resources(HOSTADDR) [GetHostname $env(HOST)]
}


#
# This parses the command line arguments.
#
proc parse_arguments {} {
    global argv Options

    # Set the default value for the command line options.
    set Options(AutoNewCheck) 1
    set Options(Iconify) 0
    set Options(Gce) 0
    set num [llength $argv]
    for {set i 0} {$i < $num} {incr i} {
	if {"[lindex $argv $i]" == "-nc"} {
	    # no check for new newsgroups.
	    set Options(AutoNewCheck) 0
	    continue
	}
	if {[lindex $argv $i] == "-iconic"} {
	    # start the interface up in iconic mode.
	    set Options(Iconify) 1
	    continue
	}
	if {[lindex $argv $i] == "-gce"} {
	    # configure the interface for use with GCE
	    set Options(Gce) 1
	    continue
	}
	puts stdout "Invalid command line option"
	puts stdout {Usage : tknews [-iconic] [-nc] [-gce]}
	puts stdout "Where"
	puts stdout "-iconic      : Bring the newsreader up in iconic mode."
	puts stdout "-nc          : Skip the new newsgroups check on startup."
	puts stdout "-gce         : Configure the newsreader for the GCE env."
	destroy .
	exit
    }
}


#
# This sets up the trace commands we need.
#
proc set_traces {} {
    global Resources

    # 
    # Put A trace on the disable keyboard option so that we can
    # effect the necessary changes whenever it is toggled.
    #
    update
    trace variable Resources(textFont) w change_text_font
    trace variable Resources(disableKeyboard) w toggle_keyboard
    trace variable Resources(showallHeaders) w prune_headers
    if {$Resources(disableKeyboard)} {
	# on init, we have to trigger the disable trace
	set Resources(disableKeyboard) 1
	return
    }
}

proc load_status {mess} {
    global LIBRARY_DIR  Options

    # create the load status widgets if necessary.
    if {$Options(Iconify)} {
	return
    }
    if {[info commands ".ls"] == {}} {
	toplevel .ls -borderwidth 0 
	wm title .ls "Loading Tknews..."
	wm positionfrom .ls user
	wm geometry .ls +0+0
	label .ls.title -bitmap @$LIBRARY_DIR/bitmaps/title.xbm \
	    -borderwidth 0	
	label .ls.label -text $mess -relief sunken -borderwidth 2 
	pack .ls.title -side top
	pack .ls.label -side bottom -fill x
	# make sure the toplevel doesn't resize itself if the message
	# to be displayed is longer in width than the bitmap.
	update
	wm maxsize .ls [winfo width .ls] [winfo height .ls]
	pack propagate .ls 0
    } else {
	.ls.label configure -text $mess
    }
    update
}


wm withdraw .
set TclVersion [info tclversion]
parse_arguments
load_libs
load_status "Loading environment"
load_environment
load_status "Checking environment"
check_environment
get_hostaddr
load_status "Loading Resources"
loadAppDefaults option {tknews Tknews} userDefault
get_orientation
load_resources
load_status "Building Interface"
build_widgets
pack_interface 1 1 1 1
generate_cursorlist
load_status "Loading Translations"
set_translations
load_status "Loading Menus"
load_menus
set_traces
load_status "Initializing"
init_tknews
if {$Resources(windowGeometry) != ""} {
    wm geometry . $Resources(windowGeometry)
}
init_interface
bind_cpanel
catch {destroy .ls}
if {!$Options(Iconify)} {
    wm deiconify .
    if {$InterfaceOrientation == "separate"} {
	wm deiconify .info
    }
} else {
    wm iconify .
}
update
if {$Resources(autoHelp)} {
    help_show grouplist_mode
}
write_scroll "Welcome to Tknews, version $Version"
