/*
 *	NetROM virtual device. This is a device which implements the IP over NetROM
 *	encapsulation. Frames are statically ARP'ed formatted up and thrown at the
 *	NetROM router.
 *
 *	30/12/93	Alan Cox	Created
 */

#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/ptrace.h>
#include <linux/fcntl.h>
#include <linux/ioport.h>
#include <linux/interrupt.h>
#include <linux/malloc.h>
#include <errno.h>

/* #include "inet/devinet.h" */
#include "skbuff.h"
#include "inet/arp.h"

#define NR_VERSION	"0.11 ALPHA"

#ifndef HAVE_ALLOC_SKB
#define alloc_skb(size, priority) (struct sk_buff *) kmalloc(size,priority)
#define kfree_skbmem(addr, size) kfree_s(addr,size);
#endif

int nr_init(void)
{
    	static unsigned char node_addr[]={'N'<<1,'O'<<1,'D'<<1,'E'<<1,'S'<<1,' '<<1,('0'<<1)&0x1E};
    	dev->base_addr = 0;
    	memcpy(dev->dev_addr,"DUMMYAD",7);
	
    	printk("%s: virtual NetROM device %s.\n",NR_VERSION);

    	dev->open = &nr_open;
    	dev->hard_start_xmit = &nr_start_xmit;
    	dev->stop = &nr_close;
    	dev->get_stats = NULL;
#ifdef HAVE_MULTICAST
    	dev->set_multicast_list = NULL;
#endif

    	/* Fill in the generic field of the device structure. */
    	for (i = 0; i < DEV_NUMBUFFS; i++)
		dev->buffs[i] = NULL;
	
    	dev->hard_header	= nr_header;
    	dev->add_arp	= nr_add_arp;
    	dev->queue_xmit	= nr_queue_xmit;
    	dev->rebuild_header	= nr_rebuild_header;
    	dev->type_trans	= nr_type_trans;

    	dev->type		= 0;
   	dev->hard_header_len = 17+20;	/* KISS + AX.25 + I + PID = 17, NetROM header is 20 for IP*/
    	dev->mtu		= 192; 
    	dev->addr_len	= 7;
    	memset(dev->broadcast,node_addr,7);

    	/* New-style flags. */
    	dev->flags		= IFF_BROADCAST;
   	dev->family		= AF_INET;
    	dev->pa_addr	= 0;
    	dev->pa_brdaddr	= 0;
    	dev->pa_mask	= 0;
    	dev->pa_alen	= sizeof(unsigned long);

    	return 0;
}

/* Open/initialize the board. */
static int nr_open(struct device *dev)
{
  	dev->start = 1;
  	return (0);
}

static int nr_start_xmit(struct sk_buff *skb, struct device *dev)
{
	char *tp;
	if (dev->tbusy) 
	{
		return 1;
	}

    	if (skb == NULL) 
    	{
		return 0;
    	}

	if(skb->arp==0)
	{
		printk("NetROM: Frame dropped. No static arp mapping for address.\n");
		kfree_skb(skb,FREE_WRITE);
		return 0;
	}
    	if (skb->len <= 0)
		return 0;
	/*
	 *	Fill in the NetROM level 3 and level 4 headers (what we can!)
	 *	(the destination call was done by rebuild_header). The source
	 *	call will be done by the NetROM router when it decides which
	 *	AX.25 port to use.
	 */
	tp=(unsigned char *)(skb+1);
	tp+=17;	/* Skip the AX.25 header (for use by the lower level) */
	*tp=0;	/* Mark source as 'to be inserted' */
	tp+=14;	/* Skip calls */
	*tp++=32;	/* TTL */
	*tp++=0x0C;	/* Netrom IP indicator */
	*tp++=0x0C;
	*tp++=0;
	*tp++=0;
	*tp=0;
	/* From here onwards is the IP header */
	nr_route(skb);
	return 0;
}


static void nr_close(struct device *dev)
{
    dev->tbusy = 1;
    dev->start = 0;
}


static int nr_header(unsigned char *buff,struct device *dev,unsigned short type,
	unsigned long daddr, unsigned long saddr,unsigned len)
{
	if(type!=ETH_P_IP)
		printk("Only IP frames supported by NetROM virtual device.\n");
	/* Do nothing here but let start_xmit do it all */
	buff+=17;	/* Skip AX.25 */
	buff+=7;	/* Point at destination call */
	if(arp_find(buff,daddr,dev,dev->pa_addr))
		return -dev->hard_header_len;
	return dev->hard_header_len;
}

static int nr_rebuild_header(void *buff, struct device *dev)
{
	printk("Error: no dynamic ARP on NetROM interfaces.\n");
	return 1;
}

static void nr_add_arp(unsigned long addr,struct sk_buff *skb, struct device *dev)
{
	printk("Error: no ARP on NetROM interfaces.\n");
}


static unsigned short nr_type_trans(struct sk_buff *skb, struct device *dev)
{
	return htons(ETH_P_IP);
}

void nr_receive_ip(struct sk_buff *skb)
{
	skb->dev=netrom_dev;
	netif_rx(skb);
}


