/*
 *
 * Simple VFS definitions for fileio. It contains both versions for normal
 * and for quota supported routines.
 *
 * Authors: Marco van Wieringen <v892273@si.hhs.nl> <mvw@mercury.mcs.nl.mugnet.org>
 *          Edvard Tuinder <v892231@si.hhs.nl> <ed@delirium.nl.mugnet.org>
 * Modified: Made loadable by Peter MacDonald.
 *
 * Version: $Id: fileio.h,v 1.1 1993/11/08 20:49:01 mvw Exp mvw $
 *
 */
#ifndef _LINUX_FILEIO_H
#define _LINUX_FILEIO_H

#include <linux/config.h>

#ifdef CONFIG_QUOTA

extern struct inode_operations* vfs_quota_ops;

#define vfs_create(dir, basename, namelen, mode, res_ino) \
vfs_quota_ops->create(dir, basename, namelen, mode, res_inode)

#define vfs_mknod(dir, basename, namelen, mode, dev) \
vfs_quota_ops->mknod((dir),(basename),(namelen),(mode),(dev))

#define vfs_mkdir(dir, basename, namelen, mode) \
vfs_quota_ops->mkdir((dir),(basename),(namelen),(mode))

#define vfs_rmdir(dir, basename, namelen) \
vfs_quota_ops->rmdir((dir),(basename),(namelen))

#define vfs_unlink(dir, basename, namelen) \
vfs_quota_ops->unlink((dir),(basename),(namelen))

#define vfs_symlink(dir, basename, namelen, oldname) \
vfs_quota_ops->symlink((dir),(basename),(namelen),(oldname))

static inline int vfs_truncate(struct inode *ino, int len)
{
	return ((int (*)(struct inode *, int))vfs_quota_ops->truncate)(ino, len);
}

static inline int vfs_chown(struct inode *ino, uid_t uid, gid_t gid)
{
	return ((int (*)(struct inode *, uid_t, gid_t))vfs_quota_ops->permission)(ino, uid, gid);
}

static inline int vfs_write(struct inode *ino, struct file *file,
                            char *addr, int bytes)
{
	return ((int (*)(struct inode *, struct file *, char *, int ))vfs_quota_ops->lookup)
		(ino, file, addr, bytes);
}

#else /* CONFIG_QUOTA */

static inline int vfs_write(struct inode *ino, struct file *file,
                            char *addr, int bytes)
{
   return file->f_op->write(ino, file, addr, bytes);
}

#define vfs_create(dir, basename, namelen, mode, res_ino) \
dir->i_op->create(dir, basename, namelen, mode, res_inode)

static inline int vfs_truncate(struct inode *ino, size_t lenght)
{
   int error;

   ino->i_size = lenght;
   if (ino->i_op && ino->i_op->truncate)
      ino->i_op->truncate(ino);
   if ((error = notify_change(NOTIFY_SIZE, ino))) {
      return error;
   }
   ino->i_dirt = 1;

   return error;
}

#define vfs_mknod(dir, basename, namelen, mode, dev) \
dir->i_op->mknod((dir),(basename),(namelen),(mode),(dev))

#define vfs_mkdir(dir, basename, namelen, mode) \
dir->i_op->mkdir((dir),(basename),(namelen),(mode))

#define vfs_rmdir(dir, basename, namelen) \
dir->i_op->rmdir((dir),(basename),(namelen));

#define vfs_unlink(dir, basename, namelen) \
dir->i_op->unlink((dir),(basename),(namelen));

#define vfs_symlink(dir, basename, namelen, oldname) \
dir->i_op->symlink((dir),(basename),(namelen),(oldname));

static inline int vfs_chown(struct inode *ino, uid_t uid, gid_t gid)
{
   ino->i_uid = uid;
   ino->i_gid = gid;
   ino->i_ctime = CURRENT_TIME;
   ino->i_dirt = 1;
   return notify_change(NOTIFY_UIDGID, ino);
}

#endif /* CONFIG_QUOTA */
#endif /* _LINUX_FILEIO_H */
