/*
 * QUOTA    An implementation of the diskquota system for the LINUX
 *          operating system. QUOTA is implemented using the BSD systemcall
 *          interface as the means of communication with the user level. Should
 *          work for all filesystems because of integration into the VFS layer
 *          of the operating system. This is based on the Melbourne quota system
 *          wich uses both user and group quota files.
 * 
 *          Main layer of quota management
 * 
 * Version: $Id: quota.c,v 3.3 1993/12/05 11:08:30 mvw Exp mvw $
 * 
 * Authors: Marco van Wieringen <v892273@si.hhs.nl> <mvw@mercury.mcs.nl.mugnet.org>
 *          Edvard Tuinder <v892231@si.hhs.nl> <ed@delirium.nl.mugnet.org>
 *          Fred N. van Kempen, <waltje@uwalt.nl.mugnet.org>
 * 
 *          This program is free software; you can redistribute it and/or
 *          modify it under the terms of the GNU General Public License as
 *          published by the Free Software Foundation; either version 2 of the
 *          License, or (at your option) any later version.
 */

#include <linux/config.h>
#include <linux/errno.h>
#include <linux/kernel.h>

#include <linux/types.h>
#include <linux/sched.h>
#include <linux/quota.h>
#include <linux/string.h>
#include <linux/fcntl.h>
#include <linux/stat.h>
#include <linux/segment.h>
#include <linux/fs.h>
#include <linux/tty.h>
#include <linux/malloc.h>
#include <linux/fileio.h>
#include <linux/module.h>

#include <asm/segment.h>
#include <sys/sysmacros.h>

char quota_version[] = UTS_RELEASE;

/* Linked list with enabled devices */
static struct device_list *devicelist[MAXQUOTAS];

/* Most recent used device */
static struct device_list *mru_device[MAXQUOTAS];

/* Most recent used id on the device in mru_device */
static struct dquot *mru_dquot[MAXQUOTAS];

static inline char *typetoname(int type)
{
   if (type == USRQUOTA)
      return "uid";
   else
      return "gid";
}

/*
 * Kick the message in quotamessage to the tty.
 */
static void quota_message(char *quotamessage)
{
   struct tty_struct *tty;

   if (current->tty >= 0) {
      tty = TTY_TABLE(current->tty);
      (void) tty_write_data(tty, quotamessage, strlen(quotamessage),
                            (void *)0, (void *)0);
   }
}

/*
 * Functions for management of devices.
 */
static struct device_list *lookup_device(dev_t dev, int type)
{
   register struct device_list *lptr;

   if (devicelist[type] == (struct device_list *) 0)
      return ((struct device_list *) 0);

   if (mru_device[type] != (struct device_list *) 0 &&
       mru_device[type]->dq_dev == dev)
      return (mru_device[type]);

   for (lptr = devicelist[type]; lptr != (struct device_list *)0; lptr = lptr->next)
      if (lptr->dq_dev == dev)
         return (lptr);

   return ((struct device_list *) 0);
}

static struct device_list *add_device(dev_t dev, char *devicename, int type)
{
   register struct device_list *lptr;
   int error;
   char *tmp;

   if ((lptr = (struct device_list *)
        kmalloc(sizeof(struct device_list), GFP_KERNEL)) == (struct device_list *) 0)
      return ((struct device_list *) 0);
   memset(lptr, 0, sizeof(struct device_list));

   lptr->dq_dev = dev;
   lptr->dq_dirt = 0;
   lptr->dq_quota = (struct dquot *) 0;

   if ((error = getname(devicename, &tmp)) != 0)
      return ((struct device_list *) 0);
   if ((lptr->dq_devicename = (char *)
        kmalloc(strlen(tmp), GFP_KERNEL)) == (char *)0)
      return ((struct device_list *) 0);
   strcpy(lptr->dq_devicename, tmp);
   putname(tmp);

   lptr->next = devicelist[type];
   devicelist[type] = lptr;

   return (lptr);
}

static inline int device_can_be_removed(struct device_list *device)
{
   register struct dquot *lptr;

   for(lptr = device->dq_quota; lptr != (struct dquot *)0; lptr = lptr->next)
      if (lptr->dq_flags & DQ_LOCKED)
         return 0;
   return 1;
}

static void remove_device(dev_t dev, int type)
{
   register struct device_list *lptr, *tmp;
   register struct dquot *idp, *tofree;

   if (devicelist[type] == (struct device_list *) 0)
      return;

   if (mru_device[type] != (struct device_list *) 0 &&
       mru_device[type]->dq_dev == dev) {
      mru_device[type] = (struct device_list *) 0;
      mru_dquot[type] = (struct dquot *) 0;
   }

   lptr = devicelist[type];
   if (lptr->dq_dev == dev)
      devicelist[type] = lptr->next;
   else {
      while (lptr->next != (struct device_list *) 0) {
         if (lptr->next->dq_dev == dev)
            break;
         lptr = lptr->next;
      }
      tmp = lptr->next;
      lptr->next = lptr->next->next;
      lptr = tmp;
   }

   idp = lptr->dq_quota;
   while (idp != (struct dquot *) 0) {
      tofree = idp;
      idp = idp->next;
      kfree_s(tofree, sizeof(struct dquot));
   }

   kfree_s(lptr->dq_devicename, strlen(lptr->dq_devicename));
   kfree_s(lptr, sizeof(struct device_list));
}

/*
 * Functions for management of dquot structs
 * 
 * Locking of dqblk for a user or group.
 */
static void wait_on_dqblk(struct dquot * dquot)
{
   struct wait_queue wait = {current, NULL};

   add_wait_queue(&dquot->dq_wait, &wait);
   while (dquot->dq_flags & DQ_LOCKED) {
      current->state = TASK_UNINTERRUPTIBLE;
      schedule();
   }
   remove_wait_queue(&dquot->dq_wait, &wait);
   current->state = TASK_RUNNING;
}

static inline void lock_dquot(struct dquot * dquot)
{
   if (dquot != (struct dquot *)0) {
      while (dquot->dq_flags & DQ_LOCKED) {
         dquot->dq_flags |= DQ_WANT;
         wait_on_dqblk(dquot);
      }
      dquot->dq_flags |= DQ_LOCKED;
   }
}

static inline void unlock_dquot(struct dquot * dquot)
{
   if (dquot != (struct dquot *)0) {
      dquot->dq_flags &= ~DQ_LOCKED;
      if (dquot->dq_flags & DQ_WANT) {
         dquot->dq_flags &= ~DQ_WANT;
         wake_up(&dquot->dq_wait);
      }
   }
}

/*
 * Functions for management of dquot's
 */
static struct dquot *lookup_dquot(struct device_list *device, int id, int type)
{
   register struct dquot *lptr = (struct dquot *) 0;

   if (id <= ID_NO_QUOTA)
      return ((struct dquot *) 0);

   /*
    * First fast lookup when same as used before.
    */
   if (mru_device[type] != (struct device_list *) 0 &&
       mru_dquot[type] != (struct dquot *) 0 &&
       mru_dquot[type]->dq_id == id &&
       mru_device[type]->dq_dev == device->dq_dev) {
      lock_dquot(mru_dquot[type]);
      return (mru_dquot[type]);
   }

   for (lptr = device->dq_quota; lptr != (struct dquot *) 0; lptr = lptr->next)
      if (lptr->dq_id == id) {
         mru_device[type] = device;
         mru_dquot[type] = lptr;
         lock_dquot(lptr);
         return (lptr);
      }
   return ((struct dquot *) 0);
}

static struct dquot *add_dquot(struct device_list * device, int id)
{
   register struct dquot *lptr;

   if ((lptr = (struct dquot *)
        kmalloc(sizeof(struct dquot), GFP_KERNEL)) == (struct dquot *) 0)
      return ((struct dquot *) 0);
   memset(lptr, 0, sizeof(struct dquot));

   lptr->dq_id = id;
   lptr->dq_btime = lptr->dq_itime = (time_t) 0;
   lptr->dq_flags |= DQ_LOCKED; /* lock it right away */

   /*
    * Stuff it in at front of the dquot list.
    */
   lptr->next = device->dq_quota;
   device->dq_quota = lptr;

   return (lptr);
}

static void remove_dquot(struct device_list * device, int id, int type)
{
   register struct dquot *lptr, *tmp;

   /*
    * Remove fast lookup references.
    */
   if (mru_dquot[type] != (struct dquot *) 0 && mru_dquot[type]->dq_id == id)
      mru_dquot[type] = (struct dquot *) 0;

   lptr = device->dq_quota;
   if (lptr->dq_id == id)
      device->dq_quota = lptr->next;
   else {
      while (lptr->next != (struct dquot *) 0) {
         if (lptr->next->dq_id == id)
            break;
         lptr = lptr->next;
      }

      tmp = lptr->next;
      lptr->next = lptr->next->next;
      lptr = tmp;
   }

   kfree_s(lptr, sizeof(struct dquot));
}

/*
 * Check quota for inodes. Returns QUOTA_OK if can allocate and
 * NO_QUOTA if it can't.
 */
static int check_idq(struct device_list * device, struct dquot * dquot,
                     int id, int type, u_long wanted_inodes)
{
   static char message[MAX_QUOTA_MESSAGE];

   if (wanted_inodes == 0 ||
      (dquot->dq_isoftlimit == 0 && dquot->dq_ihardlimit == 0))
      return QUOTA_OK;

   if (dquot->dq_ihardlimit &&
      (dquot->dq_curinodes + wanted_inodes) >= dquot->dq_ihardlimit) {
      if ((dquot->dq_flags & DQ_INODES) == 0) {
         sprintf(message,
                "File LIMIT reached on %s for %s %d. !! NO MORE !!\n\r",
                device->dq_devicename, typetoname(type), id);
         quota_message(message);
         dquot->dq_flags &= DQ_INODES;
      }
      return NO_QUOTA;
   }

   if (dquot->dq_isoftlimit &&
      (dquot->dq_curinodes + wanted_inodes) >= dquot->dq_isoftlimit &&
       dquot->dq_itime && CURRENT_TIME >= dquot->dq_itime) {
      sprintf(message,
              "File QUOTA exceeded TOO long on %s for %s %d. !! NO MORE !!\n\r",
              device->dq_devicename, typetoname(type), id);
      quota_message(message);
      return NO_QUOTA;
   }

   if (dquot->dq_isoftlimit &&
      (dquot->dq_curinodes + wanted_inodes) >= dquot->dq_isoftlimit &&
       dquot->dq_itime == 0) {
      sprintf(message,
              "File QUOTA exceeded on %s for %s %d\n\r",
              device->dq_devicename, typetoname(type), id);
      quota_message(message);
      dquot->dq_itime = CURRENT_TIME + device->dq_iexp;
   }

   return QUOTA_OK;
   /* NOTREACHED */
}

/*
 * Check quota for blocks. Returns QUOTA_OK if can allocate and
 * NO_QUOTA if it can't. When we can't allocate wanted_blocks you get
 * the number we can allocate in avail_blocks.
 */
static int check_bdq(struct device_list * device, struct dquot * dquot, int id,
                     int type, u_long wanted_blocks, u_long *avail_blocks)
{
   static char message[MAX_QUOTA_MESSAGE];

   if (wanted_blocks == 0 ||
      (dquot->dq_bsoftlimit == 0 && dquot->dq_bhardlimit == 0))
      return QUOTA_OK;

   if (dquot->dq_bhardlimit &&
      (dquot->dq_curblocks + wanted_blocks) >= dquot->dq_bhardlimit) {
      if ((dquot->dq_flags & DQ_BLKS) == 0) {
         sprintf(message,
                 "Block LIMIT reached on %s for %s %d. !! NO MORE !!\n\r",
                 device->dq_devicename, typetoname(type), id);
         quota_message(message);
         dquot->dq_flags &= DQ_BLKS;
      }
      if (dquot->dq_curblocks != dquot->dq_bhardlimit) {
         *avail_blocks = dquot->dq_bhardlimit - dquot->dq_curblocks;
         return QUOTA_OK;
      } else
         return NO_QUOTA;
   }

   if (dquot->dq_bsoftlimit &&
      (dquot->dq_curblocks + wanted_blocks) >= dquot->dq_bsoftlimit &&
       dquot->dq_btime && CURRENT_TIME >= dquot->dq_btime) {
      sprintf(message,
              "Block QUOTA exceeded TOO long on %s for %s %d. !! NO MORE !!\n\r",
              device->dq_devicename, typetoname(type), id);
         quota_message(message);
      return NO_QUOTA;
   }

   if (dquot->dq_bsoftlimit &&
      (dquot->dq_curblocks + wanted_blocks) >= dquot->dq_bsoftlimit &&
       dquot->dq_btime == 0) {
      sprintf(message,
              "Block QUOTA exceeded on %s for %s %d\n\r",
              device->dq_devicename, typetoname(type), id);
      quota_message(message);
      dquot->dq_btime = CURRENT_TIME + device->dq_bexp;
   }

   *avail_blocks = wanted_blocks;
   return QUOTA_OK;
   /* NOTREACHED */
}

/*
 * Remove inodes from a quota. Resets grace times if we go below
 * inode softlimit.
 */
static void remove_idq(struct dquot * dquot, u_long inodes)
{
   if (dquot->dq_curinodes >= inodes)
      dquot->dq_curinodes -= inodes;
   else
      dquot->dq_curinodes = 0;

   if (dquot->dq_curinodes < dquot->dq_isoftlimit) {
      dquot->dq_itime = (time_t) 0;
      dquot->dq_flags &= ~DQ_INODES;
   }
   dquot->dq_flags |= DQ_MOD;
}

/*
 * Remove blocks from a quota. Resets grace times if we go below
 * block softlimit.
 */
static void remove_bdq(struct dquot * dquot, u_long blocks)
{
   if (dquot->dq_curblocks >= blocks)
      dquot->dq_curblocks -= blocks;
   else
      dquot->dq_curblocks = 0;

   if (dquot->dq_curblocks < dquot->dq_bsoftlimit) {
      dquot->dq_btime = (time_t) 0;
      dquot->dq_flags &= ~DQ_BLKS;
   }
   dquot->dq_flags |= DQ_MOD;
}

/*
 * Realy sync the quota info to the quota file.
 */
static int sync_device(struct device_list * device, int type)
{
   register struct dquot *dquot = (struct dquot *) 0;
   struct dqblk exp_times;
   unsigned short fs;

   if (device->dq_dirt == 0)
      return 0;

   if (!device->dq_file.f_op->write)
      return -EIO;

   fs = get_fs();

   /* First write expire times */
   memset(&exp_times, 0, sizeof(struct dqblk));
   exp_times.dqb_itime = device->dq_iexp;
   exp_times.dqb_btime = device->dq_bexp;

   /*
    * Seek to absolute begin point of quota file.
    */
   if (device->dq_file.f_op->lseek) {
      if (device->dq_file.f_op->lseek(device->dq_file.f_inode,
                                      &device->dq_file, 0, 0) != 0)
         goto end_sync;
   } else
      device->dq_file.f_pos = 0;

   set_fs(KERNEL_DS);
   if (device->dq_file.f_op->write(device->dq_file.f_inode, &device->dq_file,
       (char *)&exp_times, sizeof(struct dqblk)) != sizeof(struct dqblk))
      goto end_sync;

   /*
    * Write out all quota-structs that are marked with DQ_MOD, those
    * are modified since the last time the were written to disk.
    */
   dquot = device->dq_quota;
   while (dquot != (struct dquot *)0) {
      if (dquot->dq_flags & DQ_MOD) {
         if (device->dq_file.f_op->lseek) {
            if (device->dq_file.f_op->lseek(device->dq_file.f_inode,
                                            &device->dq_file,
                                            dqoff(dquot->dq_id), 0) !=
                                            dqoff(dquot->dq_id))
               goto end_sync;
         } else
            device->dq_file.f_pos = dqoff(dquot->dq_id);

         if (device->dq_file.f_op->write(device->dq_file.f_inode,
                                         &device->dq_file,
                                         (char *) &dquot->dq_dqb,
                                         sizeof(struct dqblk)) !=
                                         sizeof(struct dqblk))
            goto end_sync;
         dquot->dq_flags &= ~DQ_MOD;
      }
      dquot = dquot->next;
   }

   set_fs(fs);
   device->dq_dirt = 0;
   return 0;

end_sync:
   set_fs(fs);
   return -EIO;
}

/*
 * Initialize a dquot-struct with new quota info. This is used by the
 * systemcall interface functions.
 */ 
static int set_dqblk(struct device_list * device, int id, int type, int flags,
                     struct dqblk *dqblk)
{
   register struct dquot *dquot = (struct dquot *) 0;
   struct dqblk dq_dqblk;
   int error;

   /* No quota enabled for users and groups below ID_NO_QUOTA */
   if (id > 0 && id <= ID_NO_QUOTA)
      return 0;

   if (dqblk == (struct dqblk *)0)
      return -EFAULT;

   if (flags & QUOTA_SYSCALL) {
      if ((error = verify_area(VERIFY_READ, dqblk, sizeof(struct dqblk))) != 0)
         return error;
      memcpy_fromfs(&dq_dqblk, dqblk, sizeof(struct dqblk));
   } else {
      memcpy(&dq_dqblk, dqblk, sizeof(struct dqblk));
   }

   /* set expiration times */
   if (id == 0) {
      device->dq_iexp = (dq_dqblk.dqb_itime) ? dq_dqblk.dqb_itime : MAX_IQ_TIME;
      device->dq_bexp = (dq_dqblk.dqb_btime) ? dq_dqblk.dqb_btime : MAX_DQ_TIME;
      device->dq_dirt = 1;
      return 0;
   }

   dquot = lookup_dquot(device, id, type);
   if (dq_dqblk.dqb_bhardlimit == 0 && dq_dqblk.dqb_bsoftlimit == 0 &&
       dq_dqblk.dqb_ihardlimit == 0 && dq_dqblk.dqb_isoftlimit == 0) {
      if (dquot == (struct dquot *) 0)
         return 0; /* No quota set */

      dquot->dq_flags |= DQ_REMOVE; /* sync and remove at the end */
   }

   if (dquot == (struct dquot *) 0)
      if ((dquot = add_dquot(device, id)) == (struct dquot *) 0)
         return -EUSERS;

   if ((flags & SET_QUOTA) || (flags & SET_QLIMIT)) {
      dquot->dq_bhardlimit = dq_dqblk.dqb_bhardlimit;
      dquot->dq_bsoftlimit = dq_dqblk.dqb_bsoftlimit;
      dquot->dq_ihardlimit = dq_dqblk.dqb_ihardlimit;
      dquot->dq_isoftlimit = dq_dqblk.dqb_isoftlimit;
      if ((flags & QUOTA_SYSCALL) == 0) {
         dquot->dq_btime = dq_dqblk.dqb_btime;
         dquot->dq_itime = dq_dqblk.dqb_itime;
      }
   }

   if ((flags & SET_QUOTA) || (flags & SET_USE)) {
      dquot->dq_curblocks = dq_dqblk.dqb_curblocks;
      dquot->dq_curinodes = dq_dqblk.dqb_curinodes;
      if (dquot->dq_btime && dquot->dq_curblocks < dquot->dq_bsoftlimit)
         dquot->dq_btime = (time_t) 0;
      if (dquot->dq_itime && dquot->dq_curinodes < dquot->dq_isoftlimit)
         dquot->dq_itime = (time_t) 0;
   }

   device->dq_dirt = 1;
   dquot->dq_flags |= DQ_MOD;

   if (dquot->dq_flags & DQ_REMOVE) {
      sync_device(device, type);
      remove_dquot(device, id, type);
   } else
      unlock_dquot(dquot);

   return 0;
}

/*
 * ====================== Entry points of module ======================
 * All functions under this line are callable from within the kernel.
 */

/*
 * This are two simple algorithms that calculates the size of a file in blocks
 * and from a number of blocks to a isize.
 * It is not perfect but works most of the time.
 */
u_long isize_to_blocks(size_t isize, size_t blksize)
{
   u_long blocks;
   u_long indirect;

   if (!blksize)
      blksize = BLOCK_SIZE;

   blocks = (isize / blksize) + ((isize % blksize) ? 1 : 0);
   if (blocks > 10) {
      indirect = ((blocks - 11) >> 8) + 1; /* single indirect blocks */
      if (blocks > (10 + 256)) {
         indirect += ((blocks - 267) >> 16) + 1; /* double indirect blocks */
         if (blocks > (10 + 256 + (256 << 8)))
            indirect++; /* triple indirect blocks */
      }
      blocks += indirect;
   }
   return blocks;
}

/*
 * And this works the other way around to calculate the inodesize
 * of a file if it may consume a certain number of blocks.
 */ 
size_t blocks_to_isize(u_long blocks, size_t blksize)
{
   size_t isize;
   u_long indirect;

   if (!blksize)
      blksize = BLOCK_SIZE;

   isize = blocks * blksize;
   if (blocks > 10) {
      indirect = ((blocks - 11) >> 8) + 1; /* single indirect blocks */
      if (blocks > (10 + 256)) {
         indirect += ((blocks - 267) >> 16) + 1; /* double indirect blocks */
         if (blocks > (10 + 256 + (256 << 8)))
            indirect++; /* triple indirect blocks */
      }
      isize -= indirect * blksize;
   }
   return isize;
}

/*
 * Allocate the number of inodes and blocks from a diskquota.
 */
int quota_alloc(dev_t dev, uid_t uid, gid_t gid, u_long wanted_inodes,
                u_long wanted_blocks, u_long *avail_blocks)
{
   u_long availblocks = 0, usr_avail = 0, grp_avail = 0;
   struct device_list *usr_device = (struct device_list *) 0,
                      *grp_device = (struct device_list *) 0;
   struct dquot *usr_dquot = (struct dquot *) 0,
                *grp_dquot = (struct dquot *) 0;

   availblocks = wanted_blocks;
   if (wanted_inodes > 0 || wanted_blocks > 0) {
      if ((usr_device = lookup_device(dev, USRQUOTA)) !=
          (struct device_list *)0) {
         if ((usr_dquot = lookup_dquot(usr_device, uid, USRQUOTA)) !=
             (struct dquot *) 0) {
            if (check_idq(usr_device, usr_dquot, uid, USRQUOTA,
                          wanted_inodes) == NO_QUOTA ||
                check_bdq(usr_device, usr_dquot, uid, USRQUOTA,
                          wanted_blocks, &usr_avail) == NO_QUOTA) {
               unlock_dquot(usr_dquot);
               return NO_QUOTA;
            }
            availblocks = usr_avail;
         }
      }

      if ((grp_device = lookup_device(dev, GRPQUOTA)) !=
          (struct device_list *)0) {
         if ((grp_dquot = lookup_dquot(grp_device, gid, GRPQUOTA)) !=
             (struct dquot *) 0) {
            if (check_idq(grp_device, grp_dquot, gid, GRPQUOTA,
                          wanted_inodes) == NO_QUOTA ||
                check_bdq(grp_device, grp_dquot, gid, GRPQUOTA,
                          wanted_blocks, &grp_avail) == NO_QUOTA) {
               unlock_dquot(usr_dquot);
               unlock_dquot(grp_dquot);
               return NO_QUOTA;
            }
            availblocks = (usr_avail == 0) ? grp_avail : \
                          min(usr_avail, grp_avail);
         }
      }

      /* Have exclusive lock on both quotas if needed and can allocate */
      if (usr_dquot != (struct dquot *) 0) {
         usr_dquot->dq_curinodes += wanted_inodes;
         usr_dquot->dq_curblocks += availblocks;
         usr_dquot->dq_flags |= DQ_MOD;
         unlock_dquot(usr_dquot);
         usr_device->dq_dirt = 1;
      }

      if (grp_dquot != (struct dquot *) 0) {
         grp_dquot->dq_curinodes += wanted_inodes;
         grp_dquot->dq_curblocks += availblocks;
         grp_dquot->dq_flags |= DQ_MOD;
         unlock_dquot(grp_dquot);
         grp_device->dq_dirt = 1;
      }
   }

   if (avail_blocks != (u_long *)0)
      *avail_blocks = availblocks;

   return QUOTA_OK;
   /* NOTREACHED */
}

/*
 * Transfer the number of inode and blocks from one diskquota to an other.
 */
int quota_transfer(dev_t dev, uid_t olduid, uid_t newuid, gid_t oldgid,
                   gid_t newgid, u_long inodes, u_long blocks)
{
   u_long availblocks = 0;
   struct device_list *usr_device = (struct device_list *) 0,
                      *grp_device = (struct device_list *) 0;
   struct dquot *old_usr_dquot = (struct dquot *) 0,
                *new_usr_dquot = (struct dquot *) 0,
                *old_grp_dquot = (struct dquot *) 0,
                *new_grp_dquot = (struct dquot *) 0;

   if (olduid != newuid &&
      (usr_device = lookup_device(dev, USRQUOTA)) != (struct device_list *)0) {
      if ((new_usr_dquot = lookup_dquot(usr_device, newuid, USRQUOTA)) !=
          (struct dquot *) 0) {
         if (check_idq(usr_device, new_usr_dquot, newuid, USRQUOTA,
                       inodes) == NO_QUOTA ||
             check_bdq(usr_device, new_usr_dquot, newuid, USRQUOTA,
                       blocks, &availblocks) == NO_QUOTA ||
             availblocks != blocks) {
            unlock_dquot(new_usr_dquot);
            return NO_QUOTA;
         }
      }
   }

   if (oldgid != newgid &&
      (grp_device = lookup_device(dev, GRPQUOTA)) != (struct device_list *)0) {
      if ((new_grp_dquot = lookup_dquot(grp_device, newgid, GRPQUOTA)) !=
          (struct dquot *) 0) {
         if (check_idq(grp_device, new_grp_dquot, newgid,
                       GRPQUOTA, inodes) == NO_QUOTA ||
             check_bdq(grp_device, new_grp_dquot, newgid, GRPQUOTA,
                       blocks, &availblocks) == NO_QUOTA ||
             availblocks != blocks) {
            unlock_dquot(new_usr_dquot);
            unlock_dquot(new_grp_dquot);
            return NO_QUOTA;
         }
      }
   }

   /* Have exclusive lock on both quotas if needed and can allocate */
   if (olduid != newuid && usr_device != (struct device_list *)0) {
      if ((old_usr_dquot = lookup_dquot(usr_device, olduid, USRQUOTA)) !=
          (struct dquot *) 0) {
         /* Remove it from old */
         if (inodes)
            remove_idq(old_usr_dquot, inodes);
         if (blocks)
            remove_bdq(old_usr_dquot, blocks);
         unlock_dquot(old_usr_dquot);
      }

      /* Move it to new diskquota */
      if (new_usr_dquot != (struct dquot *)0) {
         new_usr_dquot->dq_curinodes += inodes;
         new_usr_dquot->dq_curblocks += blocks;
         new_usr_dquot->dq_flags |= DQ_MOD;
         unlock_dquot(new_usr_dquot);
      }
      usr_device->dq_dirt = 1;
   }
   

   if (oldgid != newgid && grp_device != (struct device_list *)0) {
      if ((old_grp_dquot = lookup_dquot(grp_device, oldgid, GRPQUOTA)) !=
          (struct dquot *) 0) {
         /* Remove it from old */
         if (inodes)
            remove_idq(old_grp_dquot, inodes);
         if (blocks)
            remove_bdq(old_grp_dquot, blocks);
         unlock_dquot(old_grp_dquot);
      }

      /* Move it to new diskquota */
      if (new_grp_dquot != (struct dquot *) 0) {
         new_grp_dquot->dq_curinodes += inodes;
         new_grp_dquot->dq_curblocks += blocks;
         new_grp_dquot->dq_flags |= DQ_MOD;
         unlock_dquot(new_grp_dquot);
      }
      grp_device->dq_dirt = 1;
   }
   return QUOTA_OK;
   /* NOTREACHED */
}

/*
 * Remove the number of inodes and blocks from a diskquota.
 */
void quota_remove(dev_t dev, uid_t uid, gid_t gid, u_long inodes, u_long blocks
)
{
   struct device_list *usr_device, *grp_device;
   struct dquot *usr_dquot, *grp_dquot;

   if (inodes > 0 || blocks > 0) {
      if ((usr_device = lookup_device(dev, USRQUOTA)) !=
          (struct device_list *)0) {
         if ((usr_dquot = lookup_dquot(usr_device, uid, USRQUOTA)) !=
             (struct dquot *) 0) {
            if (inodes)
               remove_idq(usr_dquot, inodes);
            if (blocks)
               remove_bdq(usr_dquot, blocks);
            unlock_dquot(usr_dquot);
            usr_device->dq_dirt = 1;
         }
      }

      if ((grp_device = lookup_device(dev, GRPQUOTA)) !=
          (struct device_list *)0) {
         if ((grp_dquot = lookup_dquot(grp_device, gid, GRPQUOTA)) !=
             (struct dquot *) 0) {
            if (inodes)
               remove_idq(grp_dquot, inodes);
            if (blocks)
               remove_bdq(grp_dquot, blocks);
            unlock_dquot(grp_dquot);
            grp_device->dq_dirt = 1;
         }
      }
   }
}

/*
 * Following functions used by systemcall interface don't use it your self !!
 */
int set_quota(dev_t dev, int id, int type, struct dqblk * dqblk)
{
   register struct device_list *device;

   if ((device = lookup_device(dev, type)) != (struct device_list *) 0)
      return set_dqblk(device, id, type, SET_QUOTA | QUOTA_SYSCALL, dqblk);
   else
      return -ESRCH;
}

int set_use(dev_t dev, int id, int type, struct dqblk * dqblk)
{
   register struct device_list *device;

   if ((device = lookup_device(dev, type)) != (struct device_list *) 0)
      return set_dqblk(device, id, type, SET_USE | QUOTA_SYSCALL, dqblk);
   else
      return -ESRCH;
}

int set_qlimit(dev_t dev, int id, int type, struct dqblk * dqblk)
{
   register struct device_list *device;

   if ((device = lookup_device(dev, type)) != (struct device_list *) 0)
      return set_dqblk(device, id, type, SET_QLIMIT | QUOTA_SYSCALL, dqblk);
   else
      return -ESRCH;
}

int get_quota(dev_t dev, int id, int type, struct dqblk * dqblk)
{
   register struct device_list *device;
   register struct dquot *dquot;
   struct dqblk exp_times;
   int error;

   if ((device = lookup_device(dev, type)) != (struct device_list *) 0) {
      if (dqblk == (struct dqblk *) 0)
         return -EFAULT;

      if ((error = verify_area(VERIFY_WRITE, dqblk, sizeof(struct dqblk))) != 0)
         return (error);

      if (id > ID_NO_QUOTA &&
         (dquot = lookup_dquot(device, id, type)) != (struct dquot *) 0) {
         memcpy_tofs(dqblk, (char *) &dquot->dq_dqb, sizeof(struct dqblk));
         unlock_dquot(dquot);
         return 0;
      } else {
         if (id == 0) {
            /*
             * Special case for getting of expiration times.
             */
            memset(&exp_times, 0, sizeof(struct dqblk));
            exp_times.dqb_btime = device->dq_iexp;
            exp_times.dqb_itime = device->dq_bexp;
            memcpy_tofs(dqblk, &exp_times, sizeof(struct dqblk));
            return 0;
         }
         return -ESRCH;
      }
   } else
      return -ESRCH;
}

/*
 * Sync quota on a device. Dev == 0 ==> sync all quotafiles.
 * Type == -1 ==> sync all types.
 */
int sync_quota(dev_t dev, int type)
{
   register struct device_list *device;
   int cnt, retval = 0;

   if (dev) {
      if (type != -1) {
         if ((device = lookup_device(dev, type)) == (struct device_list *) 0)
            return -ESRCH;
         return sync_device(device, type);
      } else {
         for (cnt = 0; cnt < MAXQUOTAS; cnt++) {
            if ((device = lookup_device(dev, cnt)) == (struct device_list *) 0)
               continue;
            retval |= sync_device(device, cnt);
         }
         return retval;
      }
   } else {
      for (cnt = 0; cnt < MAXQUOTAS; cnt++) {
         device = devicelist[cnt];
         while (device != (struct device_list *) 0) {
            retval |= sync_device(device, cnt);
            device = device->next;
         }
      }
      return retval;
   }
   /* NOTREACHED */
}

/*
 * Turn quota off on a device. type == -1 ==> quotaoff for all types (umount)
 */
int quota_off(dev_t dev, int type)
{
   register struct device_list *device;
   register struct inode *inode = (struct inode *) 0;
   int cnt;

   if (type != -1) {
      if ((device = lookup_device(dev, type)) == (struct device_list *) 0)
         return -ESRCH;

      if (!device_can_be_removed(device))
         return -EBUSY;

      (void) sync_device(device, type);   /* Yes, just like Q_SYNC... */

      if (device->dq_file.f_op->release)
         device->dq_file.f_op->release(device->dq_file.f_inode,
                                       &device->dq_file);

      if ((inode = device->dq_file.f_inode) != (struct inode *) 0)
         iput(device->dq_file.f_inode);   /* Now release the inode */

      remove_device(dev, type);
   } else {
      for (cnt = 0; cnt < MAXQUOTAS; cnt++) {
         if ((device = lookup_device(dev, cnt)) == (struct device_list *) 0)
            continue;

         if (!device_can_be_removed(device))
            return -EBUSY;

         (void) sync_device(device, cnt);   /* Yes, just like Q_SYNC... */

         if (device->dq_file.f_op->release)
            device->dq_file.f_op->release(device->dq_file.f_inode,
                                          &device->dq_file);

         if ((inode = device->dq_file.f_inode) != (struct inode *) 0)
            iput(device->dq_file.f_inode);   /* Now release the inode */

         remove_device(dev, cnt);
      }
   }
   return 0;
}

int quota_on(dev_t dev, const char *devicename, int type, char *path)
{
   register struct device_list *device;
   struct dqblk dq_dqb;
   struct inode *inode;
   char *tmp;
   int error, maxid, id = ID_NO_QUOTA;
   unsigned short  fs;

   /* Quota already enabled */
   if ((device = lookup_device(dev, type)) != (struct device_list *) 0)
      return -EBUSY;

   if ((error = getname(path, &tmp)) != 0)
      return (error);

   error = open_namei(tmp, O_RDWR, 0600, &inode, 0);
   putname(tmp);

   if (error)
      return (error);

   if (!S_ISREG(inode->i_mode)) {
      iput(inode);
      return -EACCES;
   }

   if ((device = add_device(dev, devicename, type)) == (struct device_list *) 0) {
      iput(inode);
      return -EUSERS;
   }

   fs = get_fs();
   if (!inode->i_op || !inode->i_op->default_file_ops)
      goto end_quotaon;

   device->dq_file.f_mode = 3;
   device->dq_file.f_flags = 0;
   device->dq_file.f_count = 1;
   device->dq_file.f_inode = inode;
   device->dq_file.f_pos = 0;
   device->dq_file.f_reada = 0;
   device->dq_file.f_op = inode->i_op->default_file_ops;

   if (device->dq_file.f_op->open)
      if (device->dq_file.f_op->open(device->dq_file.f_inode, &device->dq_file))
         goto end_quotaon;

   if (!device->dq_file.f_op->read)
      goto end_quotaon;

   maxid = inode->i_size / sizeof(struct dqblk);

   set_fs(KERNEL_DS);

   /* First read expire times */
   if (device->dq_file.f_op->read(inode, &device->dq_file,
       (char *)&dq_dqb, sizeof(struct dqblk)) != sizeof(struct dqblk))
      goto end_quotaon;
   set_dqblk(device, 0, type, SET_QUOTA, &dq_dqb);

   /* Seek to first real entry */
   if (device->dq_file.f_op->lseek) {
      if (device->dq_file.f_op->lseek(inode, &device->dq_file,
          dqoff(id), 0) != dqoff(id))
         goto end_quotaon;
   } else
      device->dq_file.f_pos = dqoff(id);

   while (id < maxid) {
      if (device->dq_file.f_op->read(inode, &device->dq_file, (char *)&dq_dqb,
          sizeof(struct dqblk)) != sizeof(struct dqblk))
         goto end_quotaon;
      set_dqblk(device, id++, type, SET_QUOTA, &dq_dqb);
   }
   set_fs(fs);

   return 0;

end_quotaon:
   set_fs(fs);
   iput(inode);
   remove_device(device->dq_dev, type);
   return -EIO;
}

void _quota_init(void)
{
   unsigned short  cnt;

   for (cnt = 0; cnt < MAXQUOTAS; cnt++) {
      devicelist[cnt] = (struct device_list *) 0;
      mru_dquot[cnt] = (struct dquot *) 0;
      mru_device[cnt] = (struct device_list *) 0;
   }
}

asmlinkage int quota_quotactl(int cmd, const char *special, int id, caddr_t * addr)
{
   int cmds = 0, type = 0;
   struct inode *ino;
   dev_t dev;

   cmds = cmd >> SUBCMDSHIFT;
   type = cmd & SUBCMDMASK;
   switch (cmds) {
      case Q_SYNC:
	 if (type<0 && suser()) 
		 return sync_quota(id, type);
         break;
      case Q_GETQUOTA:
         if (((type == USRQUOTA && current->uid != id) ||
              (type == GRPQUOTA && current->gid != id)) && !suser())
            return -EPERM;
         break;
      case Q_QUOTAOFF:
	 if (type<0 && suser()) 
		 return quota_off(id, type);
      default:
         if (!suser())
            return -EPERM;
   }

   if (special == (char *)0 && cmds == Q_SYNC)
      dev = 0;
   else {
      if (namei(special, &ino))
         return -EINVAL;
      dev = ino->i_rdev;
      if (!S_ISBLK(ino->i_mode)) {
         iput(ino);
         return -ENOTBLK;
      }
      iput(ino);
   }

   if ((u_int) type >= MAXQUOTAS)
      return -EINVAL;

   switch (cmds) {
      case Q_QUOTAON:
         return quota_on(dev, special, type, (char *) addr);
      case Q_QUOTAOFF:
         return quota_off(dev, type);
      case Q_GETQUOTA:
         return get_quota(dev, id, type, (struct dqblk *) addr);
      case Q_SETQUOTA:
         return set_quota(dev, id, type, (struct dqblk *) addr);
      case Q_SETUSE:
         return set_use(dev, id, type, (struct dqblk *) addr);
      case Q_SETQLIM:
         return set_qlimit(dev, id, type, (struct dqblk *) addr);
      case Q_SYNC:
         return sync_quota(dev, type);
      default:
         return -EINVAL;
   }
   /* NOTREACHED */
}

int lookup(struct inode *, const char *, int, struct inode **);

static int quota_vfs_write(struct inode *ino, struct file *file,
                            char *addr, size_t bytes)
{
   size_t written;
   u_long cur_blocks, wanted_blocks, avail_blocks;

   if (S_ISREG(ino->i_mode)) {
      cur_blocks = isize_to_blocks(ino->i_size, ino->i_blksize);
      wanted_blocks = isize_to_blocks(file->f_pos + bytes, ino->i_blksize) -
                      cur_blocks;
      if (quota_alloc(ino->i_dev, ino->i_uid, ino->i_gid, 0,
                      wanted_blocks, &avail_blocks) == NO_QUOTA)
         return -EDQUOT;
      if (avail_blocks < wanted_blocks)
         bytes = blocks_to_isize((cur_blocks + avail_blocks), ino->i_blksize) -
                 file->f_pos;
      if ((written = file->f_op->write(ino, file, addr, bytes)) != bytes) {
         quota_remove(ino->i_dev, ino->i_uid, ino->i_gid, 0, avail_blocks -
                     (isize_to_blocks(ino->i_size, ino->i_blksize) -
                      isize_to_blocks((ino->i_size - written), ino->i_blksize)));
      }
      if (avail_blocks < wanted_blocks)
         return -EDQUOT;

      return written;
   } else
      return file->f_op->write(ino, file, (char *)addr, bytes);
}

static int quota_vfs_create(struct inode *dir, const char *basename,
                             int namelen, int mode, struct inode **res_ino)
{
   int error;

   if (quota_alloc(dir->i_dev, current->euid, current->egid, 1, 0,
                  (u_long *)0) == NO_QUOTA)
      return -EDQUOT;
   error = dir->i_op->create(dir, basename, namelen, mode, res_ino);
   if (error)
      quota_remove(dir->i_dev, current->euid, current->egid, 1, 0);

   return error;
}

static int quota_vfs_truncate(struct inode *ino, size_t lenght)
{
   int error;
   size_t old_isize;

   old_isize = ino->i_size;
   ino->i_size = lenght;
   if (ino->i_op && ino->i_op->truncate)
      ino->i_op->truncate(ino);
   ino->i_ctime = ino->i_mtime = CURRENT_TIME;
   ino->i_dirt = 1;
   if ((error = notify_change(NOTIFY_SIZE, ino))) {
      return error;
   }
   quota_remove(ino->i_dev, ino->i_uid, ino->i_gid, 0,
                isize_to_blocks(old_isize, ino->i_blksize));

   return error;
}

static int quota_vfs_mknod(struct inode *dir, const char *basename,
                            int namelen, int mode, dev_t dev)
{
   int error;

   if (quota_alloc(dir->i_dev, current->euid, current->egid, 1, 0,
      (u_long *)0) == NO_QUOTA) {
      iput(dir);
      return -EDQUOT;
   }
   dir->i_count++; /* mknod uses up dir */
   error = dir->i_op->mknod(dir, basename, namelen, mode, dev);
   if (error)
      quota_remove(dir->i_dev, current->euid, current->egid, 1, 0);
   iput(dir);

   return error;
}

static inline int quota_vfs_mkdir(struct inode *dir, const char *basename,
                            int namelen, int mode)
{
   int error;

   if (quota_alloc(dir->i_dev, current->euid, current->egid, 1, 1,
                  (u_long *)0) == NO_QUOTA) {
      iput(dir);
      return -EDQUOT;
   }
   dir->i_count++; /* mkdir uses up dir */
   error = dir->i_op->mkdir(dir, basename, namelen, mode);
   if (error)
      quota_remove(dir->i_dev, current->euid, current->egid, 1, 1);
   iput(dir);

   return error;
}

static int quota_vfs_rmdir(struct inode *dir, const char *basename,
                            int namelen)
{
   int error;
   struct inode inode, *ino;

   /* Need inode entry of directory */
   dir->i_count++; /* lookup uses up dir */
   if ((error = lookup(dir, basename, namelen, &ino))) {
      iput(dir);
      return error;
   }
   inode = *ino;
   iput(ino);
   if (!(error = dir->i_op->rmdir(dir, basename, namelen)))
      quota_remove(inode.i_dev, inode.i_uid, inode.i_gid, 1, 1);

   return error;
}

static int quota_vfs_unlink(struct inode *dir, const char *basename,
                             int namelen)
{
   int error;
   struct inode inode, *ino;

   /* Need inode info for quota operations */
   dir->i_count++; /* lookup uses up dir */
   if ((error = lookup(dir, basename, namelen, &ino))) {
      iput(dir);
      return error;
   }
   inode = *ino;
   iput(ino);
   error = dir->i_op->unlink(dir, basename, namelen);
   /* Remove block and inode. Only if link-count was 1 ! */
   if (!error && inode.i_nlink == 1)
      quota_remove(inode.i_dev, inode.i_uid, inode.i_gid, 1,
                   isize_to_blocks(inode.i_size, inode.i_blksize));

   return error;
}

static int quota_vfs_symlink(struct inode *dir, const char *basename,
                              int namelen, const char *oldname)
{
   int error;

   if (quota_alloc(dir->i_dev, current->euid, current->egid, 1, 1,
      (u_long *)0) == NO_QUOTA) {
      iput(dir);
      return -EDQUOT;
   }
   dir->i_count++; /* symlink uses up dir */
   if (!(error = dir->i_op->symlink(dir, basename, namelen, oldname)))
      quota_remove(dir->i_dev, current->euid, current->egid, 1, 1);
   iput(dir);

   return error;
}

static int quota_vfs_chown(struct inode *ino, uid_t uid, gid_t gid)
{
   int error;
   uid_t olduid;
   gid_t oldgid;

   olduid = ino->i_uid;
   oldgid = ino->i_gid;
   if (quota_transfer(ino->i_dev, olduid, uid, oldgid, gid, 1,
                      isize_to_blocks(ino->i_size, ino->i_blksize)) == NO_QUOTA)
      return -EDQUOT;
   ino->i_uid = uid;
   ino->i_gid = gid;
   ino->i_ctime = CURRENT_TIME;
   ino->i_dirt = 1;
   error = notify_change(NOTIFY_UIDGID, ino);
   if (error)
      quota_transfer(ino->i_dev, uid, olduid, gid, oldgid, 1,
                     isize_to_blocks(ino->i_size, ino->i_blksize));

   return error;
}

static struct inode_operations quota_iops = {
	NULL,
	quota_vfs_create,
	(void*)quota_vfs_write,
	(void*)quota_quotactl,
	quota_vfs_unlink,
	quota_vfs_symlink,
	quota_vfs_mkdir,
	quota_vfs_rmdir,
	(void*)quota_vfs_mknod,
	NULL,
	NULL,
	NULL,
	NULL,
	(void*)quota_vfs_truncate,
	(void*)quota_vfs_chown
};


static struct file_system_type filesys = { "quota", (void*)&quota_iops, NULL, 0 };

asmlinkage unsigned long quota_init(void)
{	
	if (register_filesystem( &filesys )<0)
		return(-EINVAL);
	_quota_init();
	return MODULE_OK(0);
}

asmlinkage void quota_cleanup(void)
{
	unregister_filesystem( "quota" );
}

