/*
 *  linux/fs/filesystems.c
 *
 *  Copyright (C) 1991, 1992  Linus Torvalds
 *
 *  table of configured filesystems
 */

#include <linux/config.h>
#include <linux/fs.h>
#include <linux/errno.h>
#include <linux/kernel.h>
#include <linux/quota.h>
#include <linux/sched.h>
#include <linux/fileio.h>
#include <linux/module.h>
#if CONFIG_MINIX_FS
#include <linux/minix_fs.h>
extern unsigned long minix_init(void);
#endif
#if CONFIG_XIA_FS
#include <linux/xia_fs.h>
extern unsigned long xiafs_init(void);
#endif
#if CONFIG_PROC_FS
#include <linux/proc_fs.h>
extern unsigned long proc_init(void);
#endif
#if CONFIG_EXT2_FS
#include <linux/ext2_fs.h>
extern unsigned long ext2_init(void);
#endif
#if CONFIG_EXT_FS
#include <linux/ext_fs.h>
extern unsigned long ext_init(void);
#endif
#if CONFIG_MSDOS_FS
#include <linux/msdos_fs.h>
extern unsigned long msdos_init(void);
#endif
#if CONFIG_NFS_FS
#include <linux/nfs_fs.h>
extern unsigned long nfs_init(void);
#endif
#if CONFIG_ISO9660_FS
#include <linux/iso_fs.h>
extern unsigned long isofs_init(void);
#endif
#if CONFIG_HPFS_FS
#include <linux/hpfs_fs.h>
extern unsigned long hpfs_init(void);
#endif
#if CONFIG_SYSV_FS
#include <linux/sysv_fs.h>
extern unsigned long sysv_init(void);
#endif
#if CONFIG_IFS_FS
#include <linux/ifs_fs.h>
extern unsigned long ifs_init(void);
#endif
#if CONFIG_DBLE_FS
#include <linux/double_fs.h>
#endif


void filesystems_init(void)
{
#if CONFIG_MINIX_FS
	minix_init();
#endif
#if CONFIG_EXT2_FS
	ext2_init();
#endif
#if CONFIG_XIA_FS
	xiafs_init();
#endif
#if CONFIG_EXT_FS
	ext_init();
#endif
#if CONFIG_MSDOS_FS
	msdos_init();
#endif
#if CONFIG_PROC_FS
	proc_init();
#endif
#if CONFIG_NFS_FS
	nfs_init();
#endif
#if CONFIG_ISO9660_FS
	isofs_init();
#endif
#if CONFIG_SYSV_FS
	sysv_init();
#endif
#if CONFIG_HPFS_FS
	hpfs_init();
#endif
#if CONFIG_IFS_FS
	ifs_init();
#endif
#if CONFIG_QUOTA
	quota_init();
#endif
#if (CONFIG_DBLE_FS && CONFIG_DOUBLE)
	dble_init();
#endif
}

static int def_vfs_truncate(struct inode *ino, size_t lenght)
{
   int error;

   ino->i_size = lenght;
   if (ino->i_op && ino->i_op->truncate)
      ino->i_op->truncate(ino);
   if ((error = notify_change(NOTIFY_SIZE, ino))) {
      return error;
   }
   ino->i_dirt = 1;

   return error;
}

static int def_vfs_chown(struct inode *ino, uid_t uid, gid_t gid)
{
   ino->i_uid = uid;
   ino->i_gid = gid;
   ino->i_ctime = CURRENT_TIME;
   ino->i_dirt = 1;
   return notify_change(NOTIFY_UIDGID, ino);
}

static int def_vfs_write(struct inode *ino, struct file *file, char *addr, int bytes)
{	return file->f_op->write(ino, file, addr, bytes); }

static int def_vfs_create(struct inode *dir, const char *basename, int namelen, int mode, 
	struct inode **res_inode) 
{	return dir->i_op->create(dir, basename, namelen, mode, res_inode); }

static int def_vfs_mknod(struct inode *dir, const char *basename, int namelen, int mode, int dev) 
{	return dir->i_op->mknod(dir,basename,namelen,mode,dev); }

static int def_vfs_mkdir(struct inode *dir, const char *basename, int namelen, int mode) 
{	return dir->i_op->mkdir(dir,basename,namelen,mode); }

static int def_vfs_rmdir(struct inode *dir, const char *basename, int namelen) 
{	return dir->i_op->rmdir(dir,basename,namelen); }

static int def_vfs_unlink(struct inode *dir, const char *basename, int namelen) 
{	return dir->i_op->unlink(dir,basename,namelen); }

static int def_vfs_symlink(struct inode *dir, const char *basename, int namelen, 
	const char *oldname)
{	return dir->i_op->symlink(dir,basename,namelen,oldname); }

int sys_quotactl(int cmd, const char *special, int id, char *addr)
{
	if (vfs_quota_ops->link)
		return ((int (*)(int cmd, const char *special, int id, char *addr))vfs_quota_ops->link)
			(cmd, special, id, addr);
	else
		return -ENOSYS;
}

static struct inode_operations def_vfs_iops = {
	NULL,
	def_vfs_create,
	(void*)def_vfs_write,
	NULL,
	def_vfs_unlink,
	def_vfs_symlink,
	def_vfs_mkdir,
	def_vfs_rmdir,
	def_vfs_mknod,
	NULL,
	NULL,
	NULL,
	NULL,
	(void*)def_vfs_truncate,
	(void*)def_vfs_chown
};

struct inode_operations* vfs_quota_ops= &def_vfs_iops;

struct file_system_type *file_systems = NULL;

int register_filesystem( struct file_system_type* fst )
{	
	if (!strcmp("quota", fst->name))
	{
		struct inode_operations* ip = (struct inode_operations*)fst->next;
		if (ip)
			vfs_quota_ops=ip;
		return 0;
	}
	return generic_register((register_item*)fst, (register_item**)&file_systems);
}
int unregister_filesystem( char *name )
{
	if (!strcmp("quota", name))
	{	vfs_quota_ops= &def_vfs_iops;
		return 0;
	}
	return generic_unregister(name, (register_item**)&file_systems);
}

