#define AUTOSENSE
#define PSEUDO_DMA

/*
 * This driver adapted from Drew Eckhardt's Trantor T128 driver
 *
 * Copyright 1993, Drew Eckhardt
 *	Visionary Computing
 *	(Unix and Linux consulting and custom programming)
 *	drew@colorado.edu
 *      +1 (303) 666-5836
 *
 *  ( Based on T128 - DISTRIBUTION RELEASE 3. ) 
 *
 * Modified to work with the Pro Audio Spectrum/Studio 16
 * by John Weidman.
 *
 *
 * For more information, please consult 
 *
 * Media Vision
 * (510) 770-8600
 * (800) 348-7116
 * 
 * and 
 *
 * NCR 5380 Family
 * SCSI Protocol Controller
 * Databook
 *
 * NCR Microelectronics
 * 1635 Aeroplaza Drive
 * Colorado Springs, CO 80916
 * 1+ (719) 578-3400
 * 1+ (800) 334-5454
 */

/*
 * Options : 
 * AUTOSENSE - if defined, REQUEST SENSE will be performed automatically
 *      for commands that return with a CHECK CONDITION status. 
 *
 * PSEUDO_DMA - enables PSEUDO-DMA hardware, should give a 3-4X performance
 * increase compared to polled I/O.
 *
 * PARITY - enable parity checking.  Not supported.
 * 
 * SCSI2 - enable support for SCSI-II tagged queueing.  Untested.
 *
 *
 * UNSAFE - leave interrupts enabled during pseudo-DMA transfers.  You
 *          only really want to use this if you're having a problem with
 *          dropped characters during high speed communications, and even
 *          then, you're going to be better off twiddling with transfersize.
 *
 * USLEEP - enable support for devices that don't disconnect.  Untested.
 *
 * The card is detected and initialized in one of several ways : 
 * 1.  Autoprobe (default) - There are many different models of
 *     the Pro Audio Spectrum/Studio 16, and I only have one of
 *     them, so this may require a little tweaking.  An interrupt
 *     is triggered to autoprobe for the interrupt line.  Note:
 *     with the newer model boards, the interrupt is set via
 *     software after reset using the default_irq for the
 *     current board number.
 *
 *
 * 2.  With command line overrides - pas16=port,irq may be 
 *     used on the LILO command line to override the defaults.
 *     NOTE:  untested.
 *
 * 3.  With the PAS16_OVERRIDE compile time define.  This is 
 *     specified as an array of address, irq tupples.  Ie, for
 *     one board at the default 0x388 address, IRQ10, I could say 
 *     -DPAS16_OVERRIDE={{0x388, 10}}
 *     NOTE:  Also untested.
 *	
 *     Note that if the override methods are used, place holders must
 *     be specified for other boards in the system.
 * 
 */
 
#include <asm/system.h>
#include <linux/signal.h>
#include <linux/sched.h>
#include <linux/module.h>
#include <asm/io.h>
#include "../block/blk.h"
#include "scsi.h"
#include "hosts.h"
#include "pas16.h"
#define AUTOPROBE_IRQ
#include "NCR5380.h"
#include "constants.h"

char pas16_version[] = UTS_RELEASE;
char pas16_mod_dependancies[] = SCSI_MOD_DEPENDANCIES;

int scsi_irq_translate[] =
	{ 0,  0,  1,  2,  3,  4,  5,  6, 0,  0,  7,  8,  9,  0, 10, 11 };

/* The default_irqs array contains values used to set the irq into the
 * board via software (as must be done on newer model boards without
 * irq jumpers on the board).  The first value in the array will be
 * assigned to logical board 0, the next to board 1, etc.
 */
int default_irqs[] = {  PAS16_DEFAULT_BOARD_1_IRQ,
			PAS16_DEFAULT_BOARD_2_IRQ,
			PAS16_DEFAULT_BOARD_3_IRQ,
			PAS16_DEFAULT_BOARD_4_IRQ
		     };

static struct override {
    unsigned short io_port;
    int  irq;
} overrides 
#ifdef PAS16_OVERRIDE
    [] = PAS16_OVERRIDE;
#else
    [4] = {{0,IRQ_AUTO}, {0,IRQ_AUTO}, {0,IRQ_AUTO},
        {0,IRQ_AUTO}};
#endif

#define NO_OVERRIDES (sizeof(overrides) / sizeof(struct override))

static struct base {
    unsigned short io_port;
    int noauto;
} bases[] = { {PAS16_DEFAULT_BASE_1, 0},
	      {PAS16_DEFAULT_BASE_2, 0},
	      {PAS16_DEFAULT_BASE_3, 0},
	      {PAS16_DEFAULT_BASE_4, 0}
	    };

#define NO_BASES (sizeof (bases) / sizeof (struct base))

unsigned short  pas16_offset[ 8 ] =
    {
        0x1c00,    /* OUTPUT_DATA_REG */
        0x1c01,    /* INITIATOR_COMMAND_REG */
        0x1c02,    /* MODE_REG */
        0x1c03,    /* TARGET_COMMAND_REG */
        0x3c00,    /* STATUS_REG ro, SELECT_ENABLE_REG wo */
        0x3c01,    /* BUS_AND_STATUS_REG ro, START_DMA_SEND_REG wo */
        0x3c02,    /* INPUT_DATA_REGISTER ro, (N/A on PAS16 ?)
                    * START_DMA_TARGET_RECIEVE_REG wo
                    */
        0x3c03,    /* RESET_PARITY_INTERRUPT_REG ro,
                    * START_DMA_INITIATOR_RECIEVE_REG wo
                    */
    };



/*
 * Function : enable_board( int  board_num, unsigned short port )
 *
 * Purpose :  set address in new model board
 *
 * Inputs : board_num - logical board number 0-3, port - base address
 *
 */

void	enable_board( int  board_num,  unsigned short port )
{
    outb( 0xbc + board_num, MASTER_ADDRESS_PTR );
    outb( port >> 2, MASTER_ADDRESS_PTR );
}



/*
 * Function : init_board( unsigned short port, int irq )
 *
 * Purpose :  Set the board up to handle the SCSI interface
 *
 * Inputs : port - base address of the board,
 *	    irq - irq to assign to the SCSI port
 *
 */

void	init_board( unsigned short io_port, int irq )
{
	unsigned int	tmp;

        /* Initialize the SCSI part of the board */

        outb( 0x30, io_port + P_TIMEOUT_COUNTER_REG );  /* Timeout counter */
        outb( 0x01, io_port + P_TIMEOUT_STATUS_REG_OFFSET );   /* Reset TC */
        outb( 0x01, io_port + WAIT_STATE );   /* 1 Wait state */

        NCR5380_read( RESET_PARITY_INTERRUPT_REG );

	/* Set the SCSI interrupt pointer without mucking up the sound
	 * interrupt pointer in the same byte.
	 */
	tmp = inb( io_port + IO_CONFIG_3 );
	tmp = (  tmp & 0x0f ) | ( scsi_irq_translate[irq] << 4 );
	outb( tmp, io_port + IO_CONFIG_3 );

	/* Set up the drive parameters and enable 5380 interrupts */
	outb( 0x6d, io_port + SYS_CONFIG_4 );
}


/*
 * Function : pas16_hw_detect( unsigned short board_num )
 *
 * Purpose : determine if a pas16 board is present
 * 
 * Inputs : board_num - logical board number ( 0 - 3 )
 *
 * Returns : 0 if board not found, 1 if found.
 */

int     pas16_hw_detect( unsigned short  board_num )
{
    unsigned char	board_rev, tmp;
    unsigned short	port = bases[ board_num ].io_port;

    /* See if we can find a PAS16 board at the address associated
     * with this logical board number.
     */

    /* First, attempt to take a newer model board out of reset and
     * give it a base address.  This shouldn't affect older boards.
     */
    enable_board( board_num, port );

    /* Now see if it looks like a PAS16 board */
    board_rev = inb( port + PCB_CONFIG );

    if( board_rev == 0xff )
	return 0;

    tmp = board_rev ^ 0xe0;

    outb( tmp, port + PCB_CONFIG );
    tmp = inb( port + PCB_CONFIG );
    outb( board_rev, port + PCB_CONFIG );

    if( board_rev != tmp ) 	/* Not a PAS-16 */
	return 0;

    if( ( inb( port + OPERATION_MODE_1 ) & 0x03 ) != 0x03 ) 
	return 0;  	/* return if no SCSI interface found */

    return 1;
}


/*
 * Function : pas16_setup(char *str, int *ints)
 *
 * Purpose : LILO command line initialization of the overrides array,
 * 
 * Inputs : str - unused, ints - array of integer paramters with ints[0]
 *	equal to the number of ints.
 *
 */

void pas16_setup(char *str, int *ints) {
    static int commandline_current = 0;
    int i;
    if (ints[0] != 2) 
        printk("pas16_setup : usage pas16=io_port,irq\n");
    else 
        if (commandline_current < NO_OVERRIDES) {
            overrides[commandline_current].io_port = (unsigned short) ints[1];
            overrides[commandline_current].irq = ints[2];
            for (i = 0; i < NO_BASES; ++i)
                if (bases[i].io_port == (unsigned short) ints[1]) {
 		    bases[i].noauto = 1;
		    break;
		}
	    ++commandline_current;
	}
}

static struct sigaction pas16_sigaction =  { pas16_intr, 0, SA_INTERRUPT , NULL };

/* 
 * Function : int pas16_detect(int hostno)
 *
 * Purpose : detects and initializes PAS16 controllers
 *	that were autoprobed, overriden on the LILO command line, 
 *	or specified at compile time.
 *
 * Inputs : hostno - id of this SCSI adapter.
 * 
 * Returns : 1 if a host adapter was found, 0 if not.
 *
 */

int pas16_detect(int hostno) {
    static int current_override = 0;
    static unsigned short current_base = 0;
    struct Scsi_Host *instance;
    unsigned short io_port;
    int  count;

    for (count = 0; current_override < NO_OVERRIDES; ++current_override) {
	io_port = 0;

	if (overrides[current_override].io_port)
	{
	    io_port = overrides[current_override].io_port;
	    enable_board( current_override, io_port );
	    init_board( io_port, overrides[current_override].irq );
	}
	else
	    for (; !io_port && (current_base < NO_BASES); ++current_base) {
#if (PDEBUG & PDEBUG_INIT)
    printk("scsi%d : probing io_port %04x\n", hostno, (unsigned int) bases[current_base].io_port);
#endif
	        if ( !bases[current_base].noauto &&
		     pas16_hw_detect( current_base ) ){
		        io_port = bases[current_base].io_port;
			init_board( io_port, default_irqs[ current_base ] ); 
#if (PDEBUG & PDEBUG_INIT)
		        printk("scsi%d : detected board.\n", hostno);
#endif
		}
    }


#if defined(PDEBUG) && (PDEBUG & PDEBUG_INIT)
	printk("scsi%d : io_port = %04x\n", hostno, (unsigned int) io_port);
#endif

	if (!io_port)
	    break;

	instance = scsi_register (hostno, sizeof(struct NCR5380_hostdata));
	instance->io_port = io_port;

	NCR5380_init(instance);

	if (overrides[current_override].irq != IRQ_AUTO)
	    instance->irq = overrides[current_override].irq;
	else 
	    instance->irq = NCR5380_probe_irq(instance, PAS16_IRQS);

	if (instance->irq != IRQ_NONE) 
	    if (irqaction (instance->irq, &pas16_sigaction)) {
		printk("scsi%d : IRQ%d not free, interrupts disabled\n", 
		    hostno, instance->irq);
		instance->irq = IRQ_NONE;
	    } 

	if (instance->irq == IRQ_NONE) {
	    printk("scsi%d : interrupts not enabled. for better interactive performance,\n", hostno);
	    printk("scsi%d : please jumper the board for a free IRQ.\n", hostno);
	}

#if defined(PDEBUG) && (PDEBUG & PDEBUG_INIT)
	printk("scsi%d : irq = %d\n", hostno, instance->irq);
#endif

	printk("scsi%d : at 0x%04x", instance->host_no, (int) 
	    instance->io_port);
	if (instance->irq == IRQ_NONE)
	    printk (" interrupts disabled");
	else 
	    printk (" irq %d", instance->irq);
	printk(" options CAN_QUEUE=%d  CMD_PER_LUN=%d release=%d",
	    CAN_QUEUE, CMD_PER_LUN, PAS16_PUBLIC_RELEASE);
	NCR5380_print_options(instance);
	printk("\n");

	++current_override;
	++count;
	++hostno;
    }
    return count;
}

/*
 * Function : int pas16_biosparam(int size, int dev, int *ip)
 *
 * Purpose : Generates a BIOS / DOS compatable H-C-S mapping for 
 *	the specified device / size.
 * 
 * Inputs : size = size of device in sectors (512 bytes), dev = block device
 *	major / minor, ip[] = {heads, sectors, cylinders}  
 *
 * Returns : allways 0 (success), initializes ip
 *	
 */

/* 
 * XXX Most SCSI boards use this mapping, I could be incorrect.  Some one
 * using hard disks on a trantor should verify that this mapping corresponds
 * to that used by the BIOS / ASPI driver by running the linux fdisk program
 * and matching the H_C_S coordinates to what DOS uses.
 */

int pas16_biosparam(int size, int dev, int * ip)
{
  ip[0] = 64;
  ip[1] = 32;
  ip[2] = size >> 11;
  return 0;
}

/*
 * Function : int NCR5380_pread (struct Scsi_Host *instance, 
 *	unsigned char *dst, int len)
 *
 * Purpose : Fast 5380 pseudo-dma read function, transfers len bytes to 
 *	dst
 * 
 * Inputs : dst = destination, len = length in bytes
 *
 * Returns : 0 on success, non zero on a failure such as a watchdog 
 * 	timeout.
 */

static inline int NCR5380_pread (struct Scsi_Host *instance, unsigned char *dst,
    int len) {
    register unsigned char  *d = dst;
    register unsigned short reg = (unsigned short) (instance->io_port + 
        P_DATA_REG_OFFSET);
    register i = len;

    while ( inb(instance->io_port + P_STATUS_REG_OFFSET) & P_ST_RDY );

    for (; i; --i) 
	*d++ = (unsigned char) inb(reg);

    if ( inb(instance->io_port + P_TIMEOUT_STATUS_REG_OFFSET) & P_TS_TIM) {
        outb( P_TS_CT, instance->io_port + P_TIMEOUT_STATUS_REG_OFFSET);
	printk("scsi%d : watchdog timer fired in NCR5480_pread()\n",
	    instance->host_no);
	return -1;
    } else
	return 0;
}

/*
 * Function : int NCR5380_pwrite (struct Scsi_Host *instance, 
 *	unsigned char *src, int len)
 *
 * Purpose : Fast 5380 pseudo-dma write function, transfers len bytes from
 *	src
 * 
 * Inputs : src = source, len = length in bytes
 *
 * Returns : 0 on success, non zero on a failure such as a watchdog 
 * 	timeout.
 */

static inline int NCR5380_pwrite (struct Scsi_Host *instance, unsigned char *src,
    int len) {
    register unsigned char *s = src;
    register unsigned short reg = (instance->io_port + P_DATA_REG_OFFSET);
    register i = len;

    while ( ( inb( instance->io_port + P_STATUS_REG_OFFSET ) ) & P_ST_RDY );
    for (; i; --i)
        outb( *s++, reg );

    if (inb(instance->io_port + P_TIMEOUT_STATUS_REG_OFFSET) & P_TS_TIM) {
        outb( P_TS_CT, instance->io_port + P_TIMEOUT_STATUS_REG_OFFSET);
	printk("scsi%d : watchdog timer fired in NCR5480_pwrite()\n",
	    instance->host_no);
	return -1;
    } else 
	return 0;
}

/*
 * Function : const char *pas16_info(void)
 *
 * Purpose : provide furthur information about this driver.
 *
 * Returns : an empty string.
 */

const char *pas16_info (void) {
    static const char string[]="";
    return string;
}

#include "NCR5380.c"

#undef PAS16_H
#define HOSTS_C
#include "pas16.h"

static int templ_ind;

void pas16_cleanup(void)
{
	scsi_template_unregister(templ_ind);
}

int pas16_init(unsigned int start_memory)
{
	Scsi_Host_Template template = MV_PAS16;
	if ((templ_ind=scsi_template_register(&template))<0)
		return MODULE_ERROR(start_memory);
	return MODULE_OK(start_memory);
		
}
