/*
 *	diehl.c - low-level driver for Diehl ISDN cards
 *
 *	Copyright (c) 1993
 *		Tilo Schuerer <tilo@cs.tu-berlin.de>
 *		Michael Riepe <riepe@ifwsn4.ifw.uni-hannover.de>
 *		Henrik Hempelmann <marsu@palumbia.in-berlin.de>
 *
 *	Parts of this code were derived from Diehl's original MS-DOS loader
 *	which is Copyright (c) Diehl Elektronik GmbH 1988-1991.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2, or (at your option)
 *	any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 */

#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/errno.h>
#include <linux/signal.h>
#include <linux/malloc.h>
#include <linux/string.h>
#include <linux/ptrace.h>

#include <asm/system.h>
#include <asm/segment.h>

#include <linux/diehl.h>
#include <linux/isdn.h>

#define DIEHL_DEBUG
#define DIEHL_MEMTEST	1

extern int d_level;

/*
 * some important hardware registers
 */

#define intack(dev)	(((__volatile__ u_char*)(dev)->shm_addr)[0x3fe])
#define srespc(dev)	(((__volatile__ u_char*)(dev)->shm_addr)[0x400])
#define crespc(dev)	(((__volatile__ u_char*)(dev)->shm_addr)[0x401])

static int diehl_init(struct isdn_dev*);
static int diehl_loados(struct isdn_dev*, u_char*, int);
static int diehl_send(struct isdn_dev*, struct isdn_buf*);
static void handle_indication(struct channel *chan ,struct isdn_buf *buf);
static void queue_ind_parm(struct channel *chan, struct isdn_buf *buf);

void print_dev_buf(struct isdn_dev *dev);

struct isdn_dev isdn_devices[NR_ISDN] = {{
	NULL, NULL, NULL, NULL,
	ISDN_ADR, ISDN_IRQ,
	NULL,				/* irq handler will be added by diehl_init */
	diehl_init,
	diehl_loados,
	diehl_send,
	HZ/10,				/* request-wait timeout */
	0, 0,
	N_ISDN_B * N_SUB_CHAN , NULL,	/* additional channel memory is added by isdn_init */
	0, 0,
	0, 0, 0, 0, 0, 0, 0, "", "", "",
}};

/*
 * irq -> device mapping
 */

static struct isdn_dev *irq_lookup[16] = {
	NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
	NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
};

/*
 * low level IRQ handler (common for all devices)
 */

static void
diehl_irq(int reg_ptr)
{
	/* find out which device to serve */
	int irq = -(((struct pt_regs *)reg_ptr)->orig_eax + 2);
	struct isdn_dev *dev = irq_lookup[irq];

	if (dev) {
		u_char tmp;

		/* clear interrupt line */
		tmp = intack(dev);

		if (dev->irq_handler) {
			/* pass control to the device's IRQ handler */
			(*dev->irq_handler)(dev);
		}
		else {
			printk("diehl_irq: no handler for IRQ%d - huh?\n", irq);
		}

		/* re-enable interrupt generation for this device */
		intack(dev) = 0;
	}
	else {
		printk("diehl_irq: bogus irq (%d) - huh?\n", irq);
	}
}

/*
 * IRQ probing. This is only used by diehl_loados().
 */

static int
diehl_probe_irq(struct isdn_dev *dev)
{
	__volatile__ struct dual *com = (struct dual*)dev->shm_addr;
	u_char tmp;

	tmp = com->Rc;
	com->Rc = 0;

	if (dev->wait)
		wake_up_interruptible(&dev->wait);
	else
		printk("diehl_probe_irq: nothing to wakeup\n");
	return 0;
}

/*
 * device interrupt handler
 * will be installed as dev->irq_handler once everything is running.
 */

static int
diehl_interrupt(struct isdn_dev *dev)
{
	__volatile__ struct dual *com = (struct dual*)dev->shm_addr;
	struct isdn_buf *buf;
	u_long flags;
	u_char tmp;
	int i;

	if (!dev->running) {
		PRINTD(DEBUG_ERR,"diehl_interrupt: got interrupt for uninitialized device\n");
		return 0;
	}

	/* handle return codes */
	if ((tmp = com->Rc) != 0) {

		if (dev->wait)		
			wake_up_interruptible(&dev->wait);	/* card not busy any more */

		/* check for assign command */
		if ((tmp & 0xf0) == 0xe0) {
			if (com->Req != ASSIGN) {
				/* huh? we didn't even send an ASSIGN request! */
				PRINTD(DEBUG_ERR,"diehl_interrupt: bogus assign return code\n");
			}
			else if ((buf = dev->assign) && buf->rri_code == ASSIGN) {
				/* we've been waiting for this to come */
				dev->assign = NULL;
				buf->rri_code = tmp;		/* saved com->Rc */
				buf->rri_id = com->RcId;
				buf->rri_ch = com->RcCh;
				com->Req = 0;
				PRINTD(DEBUG_FYI,"diehl_interrupt: waking up task after assign\n");
				wake_up(&buf->wait);
			}
			else {
				/* the assign buffer is invalid - race condition? */
				PRINTD(DEBUG_ERR,"diehl_interrupt: bogus assign buffer\n");
				com->Req = 0;
			}
		}
		else {
			/* return code for `normal' (non-assign) request */
			PRINTD(DEBUG_FYI,"diehl_interrupt: get com->RcId: %02x\n", com->RcId); 
			if ((buf = dev->buffers) != NULL) {

				/* find request buffer for this entity */
				/* maybe we should use hashing here to speed up things? */
				do {
					if (buf->rri_id == com->RcId) {

						/* got it! */
						buf->rri_code = tmp;		/* saved com->Rc */
						buf->rri_ch = com->RcCh;

						/* now dequeue the request buffer */
						save_flags(flags);
						cli();
						if (buf->next == buf) {
							dev->buffers = NULL;
						}
						else {
							if (buf == dev->buffers) {
								dev->buffers = buf->next;
							}
							buf->next->prev = buf->prev;
							buf->prev->next = buf->next;
						}

						restore_flags(flags);

						PRINTD(DEBUG_FYI,"diehl_interrupt: waking up task after request\n");
						if(buf->wait)
							wake_up(&buf->wait);
						else
							printk("diehl_interrupt: buffer without waiting_queue - panic ?\n");

						goto done;
					}
				} while ((buf = buf->next) != dev->buffers);
			}
			else
				PRINTD(DEBUG_ERR,"diehl_interrupt: no dev->buffers\n");
			/* no buffer for this id - ??? */
			PRINTD(DEBUG_ERR,"diehl_interrupt: bogus return code id (%d)\n", com->RcId);
done:
		}
		/* return code accepted */
		com->Rc = 0;
	}
	if ((tmp = com->Ind) != 0) {
		struct channel *chan = NULL;

		/* find channel for this indication */
		for (i = 0; i < dev->num_channels; ++i) {
			if (com->IndId == dev->channels[i].D3_Id ||
				com->IndId == dev->channels[i].B2_Id) {

				chan = &dev->channels[i];
				break;
			}
		}
		if (chan == NULL) {
			printk("diehl_interrupt: no indication channel (%d/%d)\n", com->Ind, com->IndId);
			goto fail;
		}

		/* allocate buffer */
		buf = (struct isdn_buf*)kmalloc(sizeof(*buf) + com->RBuffer.length, GFP_ATOMIC);
		if (buf == NULL) {
			printk("diehl_interrupt: indication: out of memory\n");
			goto fail;
		}

		/* copy data to buffer */
		buf->wait = NULL;
		buf->data = (u_char*)(buf + 1);
		buf->data_len = com->RBuffer.length;
		buf->buf_len = sizeof(*buf) + buf->data_len;
		memcpy(buf->data, com->RBuffer.P, buf->data_len);
		buf->rri_code = com->Ind;
		buf->rri_id = com->IndId;
		buf->rri_ch = com->IndCh;

		/* queue buffer for higher layers */
		save_flags(flags);
		cli();
		if (chan->backlog) {
			buf->next = chan->backlog;
			buf->prev = chan->backlog->prev;
			buf->prev->next = buf;
			buf->next->prev = buf;
		}
		else {
			buf->next = buf;
			buf->prev = buf;
			chan->backlog = buf;
		}
		chan->backlog_size++;
		if ( chan->backlog_size > MAX_BACKLOG )
		{
			/* dequeue and trash oldest buffer */
			if (buf->next == buf) {
				chan->backlog = NULL;
				kfree_s(buf, buf->buf_len);
			}
			else {
				struct isdn_buf *delbuf;

				delbuf=chan->backlog;
				chan->backlog = chan->backlog->next;
				delbuf->prev->next = delbuf->next;
				delbuf->next->prev = delbuf->prev;
				kfree_s(delbuf, delbuf->buf_len);
			}
			chan->backlog_size--;
			PRINTD(DEBUG_LOSS,".");
		}

		restore_flags(flags);
		
		queue_ind_parm(chan,buf);

		/* now wake up the receiver */
		if (chan->ind_wait)
		{
			wake_up(&chan->ind_wait);
		}
		else	/* nobody is waiting for indication */
		{
			if (buf->rri_code != LL_DATA)
			{
				handle_indication(chan,buf);
			}
		}

fail:	com->Ind = 0;	/* indication accepted */
	}
	return 0;
}

#define DELAY(ticks)	for(timeout=jiffies+(ticks);timeout>=jiffies;)

/*
 * boot time initialization (called once for each device)
 */
 
static int
diehl_init(struct isdn_dev *dev)
{
	static u_char dnload_code[1024] = {
#include "diehl_dnload.h"		/* download code */
	};
	static char *type_names[] = {
		"S",
		"SX",
		"SCOM",
		"QUADRO"
	};

	__volatile__ union dram *ram;
	struct sigaction sa;
	u_long timeout;
	u_char tmp;
	u_long flags;

	save_flags(flags);
	sti();
	printk("diehl_init: ");

	if (!dev) {
		printk("bad device\n");
		return 1;
	}
	if (dev->present) {
		/* force a reset - maybe this will change later. */
		printk("re-initializing... ");
	}
	dev->present = 0;
	dev->running = 0;

	if (dev->shm_addr < 0xc0000 ||
		dev->shm_addr >= 0xf0000 ||
		(dev->shm_addr & 0x1ffff)) {
		printk("bad shm_addr: 0x%08x\n", dev->shm_addr);
		return 1;
	}
	if (dev->irq < 0 || dev->irq >= 16 || ((1 << dev->irq) & 0xe343)) {
		/* only IRQs 2,3,4,5,7,10,11,12 are allowed */
		printk("bad irq: %d\n", dev->irq);
		return 1;
	}

	ram = (union dram*)dev->shm_addr;

	/* reset the card */

	tmp = intack(dev);
	srespc(dev) = 0;
	intack(dev) = 0;
	tmp = intack(dev);

	/* load the boot program */

	memcpy((void*)ram, &dnload_code, sizeof(dnload_code));

	if (memcmp((void*)ram, dnload_code, sizeof(dnload_code) - 4)) {
		printk("shared memory not found at 0x%08x\n", dev->shm_addr);
		return 1;
	}

	DELAY(HZ/9);

#if DIEHL_MEMTEST
	ram->load.ctrl = 1;
#else
	ram->load.ctrl = 2;
#endif

	/* start the card */

	crespc(dev) = 0;

	DELAY(HZ/18);

	if (ram->load.ctrl != 0 && ram->load.ctrl != 3) {
		printk("processor test failed\n");
		return 1;
	}

#if DIEHL_MEMTEST
	DELAY(HZ*22)
#else
	DELAY(HZ*2)
#endif
		if (ram->load.ctrl == 0)
			break;

	if (ram->load.ctrl != 0 && ram->load.ctrl != 3) {
		printk("processor stopped during memory test\n");
		return 1;
	}

#if DIEHL_MEMTEST
	/* check for memory errors */

	if (ram->load.ebit) {
		printk("adapter memory failure (0x%04x at 0x%08x)\n",
			ram->load.ebit, *(u_long*)ram->load.eloc);
		return 1;
	}
#endif

	/* check card type and memory size */

	dev->card = ram->load.card;
	if (dev->card > 3) {
		printk("ISDN adapter type not recognized\n");
		return 1;
	}

	tmp = ram->load.msize;
	if (tmp != 8 && tmp != 16 && tmp != 24 &&
		tmp != 32 && tmp != 48 && tmp != 60) {
		printk("ISDN adapter memory size is invalid (%d)\n", tmp);
		return 1;
	}

	/* install the low-level IRQ handler */

	sa.sa_handler = diehl_irq;	/* Interrupt handler */
	sa.sa_flags = 0;
	sa.sa_mask = 0;
	sa.sa_restorer = NULL;
	if (irqaction(dev->irq, &sa)) {
		printk("unable to get IRQ%d\n", dev->irq);
		return 1;
	}

	/* remember which device belongs to this IRQ */

	irq_lookup[dev->irq] = dev;

	/* we're through */

	printk("ISDN-%s adapter (%d KB RAM) at 0x%08x using IRQ%d\n",
		type_names[dev->card], tmp * 16, dev->shm_addr, dev->irq);
	dev->present = 1;

	restore_flags(flags);
	return 0;
}

/*
 * load operating system
 * will be called by a user level process
 */
 
static int
diehl_loados(struct isdn_dev *dev, u_char *os_buf, int buf_len)
{
	__volatile__ struct dual *com;
	__volatile__ union dram *ram;
	u_char version[40];
	u_long timeout;
	int i, count;
	u_char tmp;

	printk("diehl_loados: ");

	if (!dev) {
		printk("bad device\n");
		return -EINVAL;
	}
	if (!dev->present) {
		printk("device not present\n");
		return -EINVAL;
	}
	if (!dev->channels) {
		printk("channels not initialized\n");
		return -EINVAL;
	}
	if (dev->running) {
		printk("operating system already loaded\n");
		return -EINVAL;
	}

	/* we assume that os_buf and buf_len have already been checked */
	/* by the calling function. */

	ram = (union dram*)dev->shm_addr;

	/* get software version and id */

	for (i = 0; i < sizeof(version); i++)
		if (!(version[i] = get_fs_byte(os_buf+4+i)))
			break;

	dev->software_id = get_fs_long((unsigned long*)(os_buf+5+i));

	/* now load the card's operating system */

	count = buf_len;
	while (count > 0)
	{
		int len = count;

		if (len > sizeof(struct buf))
			len = sizeof(struct buf);

		memcpy_fromfs((void*)&ram->load.b, os_buf, len);

		ram->load.ctrl = 1;

		DELAY(HZ/9)
			if (ram->load.ctrl == 0)
				break;
		if (ram->load.ctrl) 	
		{
			PRINTD(DEBUG_ERR,"download timeout after %d bytes\n", buf_len - count);
			dev->present = 0;	/* avoid subsequent calls */
			return -ENOSYS;
		}

		os_buf += len;
		count -= len;
	}

	printk("software id %d, version `%s' loaded.\n", dev->software_id, version);

	/* supply additional parameters */
	/* this should be user configurable */

	ram->c[8] = dev->tei;
	ram->c[9] = dev->nt2;
	ram->c[10] = 0;
	ram->c[11] = dev->watchdog;
	ram->c[12] = dev->permanent;
	ram->c[13] = dev->xinterface;
	ram->c[14] = dev->stablel2;
	ram->c[15] = dev->noordercheck;
	memcpy(&ram->c[32], dev->oad, 32);
	memcpy(&ram->c[64], dev->osa, 32);
	memcpy(&ram->c[96], dev->spid, 20);

	/* start the card's operating system */

	ram->load.ctrl = 2;

	DELAY(HZ*1)
		if (ram->load.signature)
			break;
	if (ram->load.signature != SIGNATURE) {
		printk("diehl_loados: adapter self-test failed\n");
		dev->present = 0;	/* avoid subsequent calls */
		return -EIO;
	}

#if defined(DIEHL_DEBUG)
	printk("diehl_loados: %d bytes free", *(u_long*)&ram->c[0x3f4]);
	if (dev->card == 2) {
		printk(", OS serial # = %u", *(u_long*)&ram->c[0x3f0]);
	}
	printk("\n");
#endif
		
	*(u_long*)&ram->c[0x3e8] = 0;
	*(u_long*)&ram->c[0x3ec] = 0;
	*(u_long*)&ram->c[0x3f8] = dev->software_id;
	ram->c[0x3fc] = dev->card;

	com = (struct dual*)dev->shm_addr;

	/* test the card's IRQ line */

	dev->irq_handler = diehl_probe_irq;

	com->Rc = 0;
	com->ReqId = 0xff;
	com->Req = 1;

	current->timeout = jiffies + HZ*1;
	interruptible_sleep_on(&dev->wait);

	if (!current->timeout) {
		dev->irq_handler = NULL;
		tmp = intack(dev);
		printk("diehl_loados: interrupt test failed\n");
		dev->present = 0;
		return -EIO;
	}

	current->timeout = 0;

	/* install the `real' interrupt handler */

	dev->irq_handler = diehl_interrupt;
	com->Int = dev->irq;      /* set interrupt-number */

	/* we're done */

	dev->running = 1;
	return buf_len;
}

/*
 * send request to a device
 */
 
static int
diehl_send(struct isdn_dev *dev, struct isdn_buf *buf)
{
	__volatile__ struct dual *com;
	int timeout = dev->timeout;		/* default timeout for this device */
	u_long flags;

	if (!dev) {
		PRINTD(DEBUG_ERR,"diehl_send: bad device\n");
		return -EINVAL;
	}
	if (!dev->running) {
		PRINTD(DEBUG_ERR,"diehl_send: device inoperable\n");
		return -EINVAL;
	}
	if (!buf) {
		PRINTD(DEBUG_ERR,"diehl_send: bad message buffer\n");
		return -EINVAL;
	}
	if (buf->data_len > 270) {
		PRINTD(DEBUG_ERR,"diehl_send: bad message buffer length (%d)\n", buf->data_len);
		return -EINVAL;
	}

	com = (struct dual*)dev->shm_addr;

	/* wait if the card is busy */

again:
	save_flags(flags);
	cli();
	if (com->Req || com->XLock) {

		restore_flags(flags);
		if (timeout) {

			/* wait a short while, then try again */

			current->timeout = jiffies + timeout;
			printk("busy_sleep - ");
			interruptible_sleep_on(&dev->wait);
			printk("... wakeup\n");
			if (current->signal & ~current->blocked) {
				return -EINTR;
			}
			if (!current->timeout) {
				/* if we don't make it next time, give up */
				timeout = 0;	
			}
			else {
				/* reset timeout */
				current->timeout = 0;
			}
			goto again;
		}

		PRINTD(DEBUG_ERR,"diehl_send: device busy\n");
		return -EBUSY;
	}
	com->XLock = 1;				/* lock the card for other applications */

	/* queue the request buffer */

	buf->wait = NULL;
	if ((buf->rri_code == ASSIGN) && ((buf->rri_id == DSIG_ID) || 
	                                  (buf->rri_id == NL_ID) ||
	                                  (buf->rri_id == BLLC_ID) ||
	                                  (buf->rri_id == TASK_ID) ||
	                                  (buf->rri_id == TIMER_ID) ||
	                                  (buf->rri_id == TEL_ID)))
	{
		dev->assign = buf;
	}
	else if (dev->buffers) {
		buf->next = dev->buffers;
		buf->prev = dev->buffers->prev;
		buf->prev->next = buf;
		dev->buffers->prev = buf;
	}
	else {
		buf->next = buf->prev = buf;
		dev->buffers = buf;
	}

	/* copy request parameters to the card */

	if (buf->data_len) {
		memcpy((void*)&com->XBuffer.P, buf->data, buf->data_len);
	}
	com->XBuffer.P[buf->data_len] = 0;
	com->XBuffer.length = buf->data_len;

	/* now send the request */

	com->ReqCh = buf->rri_ch;
	com->ReqId = buf->rri_id;

	PRINTD(DEBUG_FYI,"diehl_send: sending request (%02x), sleeping...\n", buf->rri_code);

	/* prevent ``IRQ-before-sleep'' race */
	save_flags(flags);
	cli();

	com->Req = buf->rri_code;

	/* wait for completition */
	
	if (current != task[0])		/* Never sleep task[0] ! */
		sleep_on(&buf->wait);

	restore_flags(flags);

	/* return code should now be in request buffer */
	return 0;
}

static void
handle_indication(struct channel *chan ,struct isdn_buf *buf)
{
	if (buf->rri_id == chan->D3_Id) {
		switch (buf->rri_code) {
			case HANGUP:	/* our peer doesn't like us any more ;-) */
				PRINTD(DEBUG_FYI,"handle_indication(): peer sent HANGUP\n");
				chan->D3_conn=0;
			default:
				/*
				 * TODO: handle other indications
				 */
				break;
		}
	}
	else {	/* B2 indication */
		switch (buf->rri_code) {
			case LL_RELEASE:	/* our peer doesn't like us any more ;-) */
				PRINTD(DEBUG_FYI,"handle_indication(): peer sent LL_RELEASE\n");
				chan->B2_conn=0;
			default:
				/*
				 * TODO: handle other indications
				 */
				break;
		}
	}
}

static void
queue_ind_parm(struct channel *chan, struct isdn_buf *buf)
{
	int pos;

	PRINTD(DEBUG_FUNC,"queue_ind_parm\n");
	if (buf->rri_id == chan->D3_Id) {
	if (buf->data_len > 0) {
		pos = 0;
		while ((buf->data[pos] != 0) && (pos < buf->data_len))
		{
			switch (buf->data[pos])
			{
				case BC:
					chan->info.bc.length=buf->data[pos+1];
					memcpy((void*)&chan->info.bc.data,(void*)&buf->data[pos+2],chan->info.bc.length);
					break;
				case CAI:
					chan->info.cai.length=buf->data[pos+1];
					memcpy((void*)&chan->info.cai.data,(void*)&buf->data[pos+2],chan->info.cai.length);
					break;
				case CAD:
					chan->info.cad.length=buf->data[pos+1];
					memcpy((void*)&chan->info.cad.data,(void*)&buf->data[pos+2],chan->info.cad.length);
					break;
				case CAU:
					chan->info.cau.length=buf->data[pos+1];
					memcpy((void*)&chan->info.cau.data,(void*)&buf->data[pos+2],chan->info.cau.length);
					break;
				case CHI:
					chan->info.chi.length=buf->data[pos+1];
					memcpy((void*)&chan->info.chi.data,(void*)&buf->data[pos+2],chan->info.chi.length);
					break;
				case CIF:
					chan->info.cif.length=buf->data[pos+1];
					memcpy((void*)&chan->info.cif.data,(void*)&buf->data[pos+2],chan->info.cif.length);
					break;
				case CL:
					chan->info.cl.length=buf->data[pos+1];
					memcpy((void*)&chan->info.cl.data,(void*)&buf->data[pos+2],chan->info.cl.length);
					break;
				case CPS:
					chan->info.cps.length=buf->data[pos+1];
					memcpy((void*)&chan->info.cps.data,(void*)&buf->data[pos+2],chan->info.cps.length);
					break;
				case CPN:
					chan->info.cpn.length=buf->data[pos+1];
					memcpy((void*)&chan->info.cpn.data,(void*)&buf->data[pos+2],chan->info.cpn.length);
					break;
				case DATE:
					chan->info.date.length=buf->data[pos+1];
					memcpy((void*)&chan->info.date.data,(void*)&buf->data[pos+2],chan->info.date.length);
					break;
				case DSA:
					chan->info.dsa.length=buf->data[pos+1];
					memcpy((void*)&chan->info.dsa.data,(void*)&buf->data[pos+2],chan->info.dsa.length);
					break;

				case ESC:
					switch (buf->data[pos + 2])
					{
						case 8:
							chan->info.esccau.length=buf->data[pos+1];
							memcpy((void*)&chan->info.esccau.data,(void*)&buf->data[pos+2],chan->info.esccau.length);
						case 0: /* statistics */
							chan->info.escstat.length=buf->data[pos+1];
							memcpy((void*)&chan->info.escstat.data,(void*)&buf->data[pos+2],chan->info.escstat.length);

						default:
							break;
					}
					break;
				case HLC:
					chan->info.hlc.length=buf->data[pos+1];
					memcpy((void*)&chan->info.hlc.data,(void*)&buf->data[pos+2],chan->info.hlc.length);
					break;
				case LLC:
					chan->info.llc.length=buf->data[pos+1];
					memcpy((void*)&chan->info.llc.data,(void*)&buf->data[pos+2],chan->info.llc.length);
					break;
				case OAD:
					chan->info.oad.length=buf->data[pos+1];
					memcpy((void*)&chan->info.oad.data,(void*)&buf->data[pos+2],chan->info.oad.length);
					break;
				case OSA:
					chan->info.osa.length=buf->data[pos+1];
					memcpy((void*)&chan->info.osa.data,(void*)&buf->data[pos+2],chan->info.osa.length);
					break;
				case SIN:
					chan->info.sin.length=buf->data[pos+1];
					memcpy((void*)&chan->info.sin.data,(void*)&buf->data[pos+2],chan->info.sin.length);
					break;
				case UUI:
					chan->info.uui.length=buf->data[pos+1];
					memcpy((void*)&chan->info.uui.data,(void*)&buf->data[pos+2],chan->info.uui.length);
					break;
				case 0x96:	/* switch codeset (ignored) */
					break;
				default:
					printk("unknown Parameter: %02x\n", buf->data[pos]);
					break;
			}
			if ((unsigned char)buf->data[pos] >= 0x80)
				pos++;
			else
				pos+=buf->data[pos+1]+2;
		}
	}
	}
}
