/*
 * linux/kernel/chr_drv/mouse.c
 *
 * Generic mouse open routine by Johan Myreen
 *
 * Based on code from Linus
 *
 * Teemu Rantanen's Microsoft Busmouse support and Derrick Cole's
 *   changes incorporated into 0.97pl4
 *   by Peter Cervasio (pete%q106fm.uucp@wupost.wustl.edu) (08SEP92)
 *   See busmouse.c for particulars.
 *
 * Made things a lot mode modular - easy to compile in just one or two
 * of the mouse drivers, as they are now completely independent. Linus.
 */

#include <linux/fs.h>
#include <linux/errno.h>
#include <linux/mouse.h>
#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/major.h>
#include <linux/proc_fs.h>

extern unsigned long busmouse_init(unsigned long);
extern unsigned long psaux_init(unsigned long);
extern unsigned long msmouse_init(unsigned long);
extern unsigned long atixlmouse_init(unsigned long);

#define MAX_MICE 10

struct device_struct {
	const char * name;
	struct file_operations * fops;
};

static struct device_struct mouse_minor_fops[MAX_MICE] = { { 0, 0 } };

static int mouse_ioctl (struct inode * inode, struct file * filp, unsigned int cmd,
		unsigned long arg)
{	
	int minor;
	if (PROC_DEV_NAME==cmd)
		if ((minor=PROC_DEV_MINOR(arg)) >= MAX_MICE) return -E2BIG;
		else if (mouse_minor_fops[minor].fops)
		{	strcpy((char*)arg, mouse_minor_fops[minor].name );
			return 0;
		}
		else return -ENODEV;
}

static int mouse_open(struct inode * inode, struct file * file)
{
	int minor = MINOR(inode->i_rdev);

	if ((minor >= 0) && (minor <MAX_MICE) && (mouse_minor_fops[minor].fops))
		file->f_op = mouse_minor_fops[minor].fops;
	else
		return -ENODEV;
        return file->f_op->open(inode,file);
}

int register_mouse( int minor, char *name, struct file_operations *fops )
{
	if ((minor >= 0) && (minor <MAX_MICE) && (!mouse_minor_fops[minor].fops))
	{	mouse_minor_fops[minor].fops = fops;
		mouse_minor_fops[minor].name = name;
	}
	else
		return 0;
	return 1;
}

int unregister_mouse( int minor, char *name )
{
	if ((minor >= 0) && (minor <MAX_MICE) && (mouse_minor_fops[minor].fops))
		mouse_minor_fops[minor].fops = NULL;
	else
		return 0;
	return 1;
}

static struct file_operations mouse_fops = {
        NULL,		/* seek */
	NULL,		/* read */
	NULL,		/* write */
	NULL,		/* readdir */
	NULL,		/* select */
	mouse_ioctl,	/* ioctl */
	NULL,		/* mmap */
        mouse_open,
        NULL		/* release */
};

unsigned long mouse_init(unsigned long kmem_start)
{
#if CONFIG_BUSMOUSE
	kmem_start = busmouse_init(kmem_start);
#endif
#if CONFIG_PSMOUSE || CONFIG_QUICKPORT_MOUSE
	kmem_start = psaux_init(kmem_start);
#endif
#if CONFIG_MS_BUSMOUSE
	kmem_start = msmouse_init(kmem_start);
#endif
#if CONFIG_ATIXL_BUSMOUSE
 	kmem_start = atixlmouse_init(kmem_start);
#endif
	if (register_chrdev(MOUSE_MAJOR,"mouse",&mouse_fops))
		printk("unable to get major %d for mouse devices\n",
		       MOUSE_MAJOR);
	return kmem_start;
}
