/*
	linux/kernel/blk_drv/lmscd.c - Laser Magnetic Storage International
				       CDROM driver

	Copyright (C) 1993  Kai Petzke (wpp@marie.physik.tu-berlin.de)

	Parts taken from: linux/kernel/blk_drv/mcd.c

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2, or (at your option)
	any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

	HISTORY

	0.3	Some cleanups for a new release for 99pl14
	0.2.3	Better recovery after CD_RAM_OVERFLOW
	0.2.2	Speedup - up to 150 kB/sec
	0.2.1	Adapted to kernel 99pl13r
	0.2	Do a bit of buffer prefetching - about 70 kB/sec
	0.1	First attempt - slow, but working (about 10 kB/sec)
*/

/* #include <linux/config.h> */

#include <linux/errno.h>
#include <linux/sched.h>
#include <linux/timer.h>
#include <linux/fs.h>
#include <linux/kernel.h>
#include <linux/cdrom.h>
#include <linux/string.h>
#include <linux/malloc.h>
#include <linux/module.h>

#include <asm/system.h>
#include <asm/io.h>
#include <asm/segment.h>

char lmscd_version[] = UTS_RELEASE;

/*************************************************************************
 **********************  USER CONFIGURATION SECTION  *********************
 *************************************************************************/

/*
 * To change the major number, that this driver uses, change the
 * value of LMS_CDROM_MAJOR in include/linux/major.h.
 */
#ifndef LMS_CDROM_MAJOR
#define LMS_CDROM_MAJOR 28
#endif
#define MAJOR_NR LMS_CDROM_MAJOR
#include "blk.h"

/*
 * Adapt these to the settings on your adapter card.
 */
int LMSCD_INTR_NR=5;
int LMSCD_PORT=0x340;

void lmscd_setup(char *str, int *p)
{
	if (p[0]>0 && p[1]>=0)
		LMSCD_INTR_NR=p[1];
	if (p[0]>1 && p[2]>=0)
		LMSCD_PORT=p[1];
}

/* to turn all debugging off, define DLEVEL to -1, CLEVEL and DFUNC as 0 */
#define DLEVEL 0		/* debugging level */
#define CLEVEL 1		/* data integrity check level */
#define DFUNC 0			/* print functions along debugging data? */

/*
 * If you have problems compiling this driver, try to define AVOIDASM.
 */
/* #define AVOIDASM 1 */

/*************************************************************************
 ******************  END OF USER CONFIGURATION SECTION  ******************
 *************************************************************************/

/*
 * This is the number of sectors guaranteed to be on one track.
 * If a read request comes in for a sector ONE_TRACK or less
 * sectors ahead, continue with the current read request, as
 * we can't reach the information faster anyway ...
 */
#define ONE_TRACK 5

/* too much typing all the time: */
typedef unsigned char uch;

/*
 * These are bits set in the word, which is read by lms_state()
 * (see below)
 */

/* CD ROM ready to receive commands */
#define CD_CMD_READY	0x0001

/* CD ROM acknowledges receiption of command byte */
#define CD_CMD_RCVD	0x0002

/* CD ROM signals error */
#define CD_ERROR	0x0010

/* CD ROM signals error on receiption of command */
#define CD_CMD_ERROR	0x000c

/* CD ROM finished to output data */
#define CD_DATA_END	0x0040

/* CD ROM tells, that it is ready to output track information */
#define CD_TRACK_READY	0x0080

/* a sector has been read */
#define CD_SECTOR_READY	0x1000

/*
 * Overflow of the RAM of the CD ROM drive (in other words: we were
 * too slow with reading ...
 */
#define CD_RAM_OVERFLOW	0x2000

/* non fatal read errors ?? */
#define CD_READ_ERROR	0xc000
#define CD_READ_ERROR_1	0x4000
#define CD_READ_ERROR_2	0x8000

#define PORT(x) (cd -> cd_port + (x))

/*
 * Possible values for cd_state
 */
#define CD_STATE_ERROR	0x01	/* drive is not ready (originally 0x80) */
#define CD_STATE_UNIDEN	0x02	/* drive type not identified (orig 0x1) */
#define CD_STATE_NODISK	0x04	/* no disk present (0x40) */
#define CD_STATE_OPEN	0x08	/* door open (0x20) */
#define CD_STATE_LOCKED	0x10	/* door locked (0x10) */
#define CD_STATE_MEDIA	0x20	/* media change (0x02) */
#define CD_STATE_PAUSED	0x40	/* audio play paused (0x04) */
#define CD_STATE_AUDIO	0x80	/* drive is in audio mode (0x08) */

/* all those things we do not like when accessing the disk: */
#define CD_STATE_BAD	(CD_STATE_ERROR | CD_STATE_UNIDEN | CD_STATE_NODISK | \
			 CD_STATE_OPEN | CD_STATE_MEDIA)

/*
 * I hope, I have put enough volatiles here.  Most functions access the
 * data in this structure not before they ensured, that they are alone
 * doing so.  But some fields are read by code, which may be interrupted
 * by a timer or device interrupts, and these fields have to be declared
 * volatile.
 */
struct lmscd {
    short cd_port;
    short cd_irq;
    dev_t cd_dev;		/* device */
    short cd_adap_sect;		/* how many sectors fit into adapter memory? */

    /* all the functions, which to call on interrupt and other events */
    struct timer_list cd_timer;		/* timer */
    void (*cd_irq_handler)(struct lmscd *, unsigned short);
					/* interrupt request handler */
    void (*cd_success)(struct lmscd *);/* function to call from the irq handler
					   after completion of command */
    void (*cd_retry)(struct lmscd *);	/* function to call after timeout */
    void (*cd_abort)(struct lmscd *);	/* to call after too many timeouts */

    /* retry count and timeout */
    volatile short cd_retry_cnt;/* # of retries left */
    short cd_timeout;		/* counted in 1/100 seconds (jiffies) */

    /* this is the stuff for serializing access to the CD-ROM */
    volatile uch cd_sync;	/* set by any code, which communicates with
				 * the CD ROM directly. */
    volatile uch cd_kernel;	/* set while processing kernel requests */
    volatile uch cd_u_irqs;	/* number of unprocessed interrupts */
    volatile uch cd_u_timer;	/* is there an unprocessed timer interrupt? */

    /*
     * Kernel interface.  The first wait queue is for serializing access
     * to the drive, the second is for waiting for actions to finish.
     */
    struct wait_queue *cd_kernel_queue;
    struct wait_queue *cd_command_queue;
    unsigned short cd_open;	/* number of open files */

    /* state of CD-ROM drive and adapter */
    unsigned short cd_state;	/* see #define's above */
    uch cd_error_seen;		/* any of CD_READ_ERROR seen? */
    volatile signed char cd_type;/* type of CD ROM drive (LMS CM205/202) */
    char cd_read_state;		/* internal information for read_state() */

    uch cd_drive_error;		/* last "physical" error (eg. "no disk") */
    uch cd_adapter_error;	/* last "logical" error (eg. "illegal cmd") */
    uch cd_save_state;		/* needed by lms_read_state() */

    /* some information about the current CD: */
    unsigned long cd_size;	/* size (in frames/tracks with 2048 byte each)*/
    uch cd_size_fsm[3];		/* the same value as size - coded differently */
    uch cd_first_track;
    uch cd_last_track;
    uch *cd_track_info;		/* Information about all tracks */

    /* Information about command currently transferred to CD-ROM.  Actually,
     * there may be up to two commands. */
    const uch *cd_command;	/* pointer to command to send */
    const uch *cd_command2;	/* pointer to another command to send */
    short cd_cmd_sent;		/* number of bytes already send */
    short cd_cmd_len;		/* number of bytes to send */
    short cd_cmd_len2;		/* saved length */
    short cd_timeout2;		/* saved timeout */
    uch cd_cmd_check;		/* error check after command transfer */
    uch cd_echo[24];		/* echo of command */

    /* Information about current reads */
    long cd_sector;		/* next sector to read */
    long cd_sectors;		/* number of sectors to read */
    short cd_read_retry;	/* retry count for read command */
    volatile signed char cd_identify;	/* flag for identify_drive() */
    char cd_request;		/* CURRENT request is in process */
    short cd_unread;		/* number of sectors unread in adapter memory */
    short cd_error_free;	/* number of error free unread sectors */

    /*
     * The smallest Linux buffers are (hopefully) 1 kB, but CD blocks are 2 kB.
     * Buffer up to one kilobyte.
     */
    unsigned long cd_buf_sector;
    uch cd_buffer[1024];
};

static struct lmscd *cd0;
int cd_max_minor;		/* maximum minor for CD ROM drives */

/*
 * Translate an device number to a pointer to struct lmscd (this is used once
 * we support multiple CD-ROM drives on one MAJOR device no).
 */
#define GETCD(x) (MINOR(x) < cd_max_minor ? cd0 : 0)


/* commands to send to the CD-ROM: */
static uch cmd_clear_error[] = { 0x4e };	/* clear error condition */
static uch cmd_seek_over_end[] = { 0x59, 0x74, 0x59, 0x99 };	/* abort read */
static uch cmd_2d_state[] = {
    0x2d, 0x9c, 0x9c, 0x9c, 0x9c, 0x9c, 0x9c, 0x9c, 0x9c, 0x9c, 0x9c, 0x9c, 0x9c
};
static uch cmd_3a_state[] = {
    0x3a, 0x9c, 0x9c, 0x9c, 0x9c, 0x9c, 0x9c, 0x9c, 0x9c, 0x9c, 0x9c, 0x9c, 0x9c
};

unsigned long lmscd_init(unsigned long mem_start, unsigned long mem_end);
static void lms_interrupt(int sig);
static void lms_timer(unsigned long data);

/* send a command to drive */
static void lms_command(struct lmscd *const cd, const uch *cmd, int len,
			 int timeout);
static void lms_start_command(struct lmscd *const cd, const uch *cmd, int len,
			       int timeout);
static void lms_retry_command(struct lmscd *const cd);
static void lms_send_byte(struct lmscd *const cd, unsigned short state);
static void lms_read_echo(struct lmscd *const cd, unsigned short state);

/* send a command and check for error */
static void lms_command_check(struct lmscd *const cd, const uch *cmd, int len,
			       int timeout);
static void lms_do_check(struct lmscd *const cd);

/* initialize adapter */
static void lms_adinit(struct lmscd *const cd);
static void lms_retry_adinit(struct lmscd *const cd);
static void lms_check_adinit(struct lmscd *const cd, unsigned short state);
static void lms_do_adinit(struct lmscd *const cd);
static void lms_after_adinit(struct lmscd *const cd, unsigned short state);

/* reset drive */
static void lms_reset_drive(struct lmscd *const cd);
static void lms_poll_for_error(struct lmscd *const cd);

/* identify drive */
static void lms_identify_drive(struct lmscd *const cd);
static void lms_bad_identify(struct lmscd *const cd);

/* read drive state */
static void lms_set_all_zero(struct lmscd *const cd);
static void lms_read_state(struct lmscd *const cd);
static void lms_error_state(struct lmscd *const cd);

/* check for media change */
int check_lms_media_change(int dev, int flag);

/* open() */
static int lms_open(struct inode *ip, struct file *fp);
static void lms_release(struct inode *ip, struct file *fp);

/* handling read requests */
static void lms_request(struct lmscd *const cd, int flag);
static void lms_start_read(struct lmscd *const cd);
static void lms_wait_read(struct lmscd *const cd);
static void lms_abort_read(struct lmscd *const cd);
static void lms_do_abort(struct lmscd *const cd, int from_where);
static void lms_cont_abort(struct lmscd *const cd);
static void lms_retry_abort(struct lmscd *const cd);
static void lms_irq_read(struct lmscd *const cd, unsigned short state);
static int lms_read(struct lmscd *const cd);
static int lms_discard(struct lmscd *const cd);

#if DFUNC 
static inline char *lms_irq_handler(struct lmscd *const cd);
static inline char *lms_retry(struct lmscd *const cd);
static inline char *lms_abort(struct lmscd *const cd);
static inline char *lms_success(struct lmscd *const cd);
static void lms_print_func(struct lmscd *const cd, const char *str);

static inline char *lms_irq_handler(struct lmscd *const cd)
{
    if(cd -> cd_irq_handler == 0) return "(not installed)";
    if(cd -> cd_irq_handler == lms_after_adinit) return "send_byte";
    if(cd -> cd_irq_handler == lms_send_byte) return "send_byte";
    if(cd -> cd_irq_handler == lms_read_echo) return "read_echo"; 
    if(cd -> cd_irq_handler == lms_check_adinit) return "check_adinit"; 
    return "(unknown)";
}
static inline char *lms_retry(struct lmscd *const cd)
{
    if(cd -> cd_retry == 0) return "(not installed)";
    if(cd -> cd_retry == lms_retry_command) return "retry_command";
    if(cd -> cd_retry == lms_retry_adinit) return "retry_adinit"; 
    if(cd -> cd_retry == lms_abort_read) return "retry_adinit"; 
    if(cd -> cd_retry == lms_poll_for_error) return "poll_for_error"; 
    return "(unknown)";
}
static inline char *lms_abort(struct lmscd *const cd)
{
    if(cd -> cd_abort == 0) return "(not installed)";
    if(cd -> cd_abort == lms_bad_identify) return "bad_identify";
    if(cd -> cd_abort == lms_error_state) return "error_state"; 
    if(cd -> cd_abort == lms_abort_read) return "abort_read"; 
    if(cd -> cd_abort == lms_cont_abort) return "abort_read"; 
    if(cd -> cd_abort == lms_retry_abort) return "retry_abort"; 
    return "(unknown)";
}
static inline char *lms_success(struct lmscd *const cd)
{
    if(cd -> cd_success == 0) return "(not installed)";
    if(cd -> cd_success == lms_identify_drive) return "identify_drive";
    if(cd -> cd_success == lms_read_state) return "read_state"; 
    if(cd -> cd_success == lms_abort_read) return "abort_read"; 
    if(cd -> cd_success == lms_cont_abort) return "abort_read"; 
    if(cd -> cd_success == lms_wait_read) return "wait_read"; 
    return "(unknown)";
}
static void lms_print_func(struct lmscd *const cd, const char *str)
{
    if(strchr(str, '\n'))
	printk("LMSCD: scs = %s, abt = %s, rty = %s, irq = %s\n",
	       lms_success(cd), lms_abort(cd), lms_retry(cd),
	       lms_irq_handler(cd));
}
/*
 * Unfortunately, this can't be done as inline.
 */
static const char *first_arg(const char *str, ...)
{
    return str;
}
#define PRINTF(x)	lms_print_func(cd, first_arg x);
#else
#define PRINTF(x)
#endif

#if DLEVEL >= 0
#define debug_0(x) ({ printk x; PRINTF(x) })
#else
#define debug_0(x) (void) 0
#endif

#if DLEVEL >= 1
#define debug_1(x) ({ printk x; PRINTF(x) })
#else
#define debug_1(x) (void) 0
#endif

#if DLEVEL >= 2
#define debug_2(x) printk x
#else
#define debug_2(x) (void) 0
#endif

#if DLEVEL >= 3
#define debug_3(x) printk x
#else
#define debug_3(x) (void) 0
#endif


/* Usefull inlines: */
/* bcd to binary: */
static inline uch bcd2bin(uch bcd)
{
    return bcd - (((unsigned) (bcd & 0xf0) / 8) * 3);
}

/*
 * The assembler version of this function caused compile problems for
 * some people, so I also supply a non-assembler version of bin2bcd.
 * Both convert a number between 0 and 99 to bcd (binary coded decimal).
 */
static inline uch bin2bcd(uch bin)
{
#ifdef AVOIDASM
    return (bin / 10) * 16 + bin % 10;
#else
    unsigned char bcd;
    /* unsigned short b = bin; */

__asm__ ("idivb %b2\n\t"
    "shlb $4,%b0\n\t"
    "orb %%ah,%b0"
    : "=a" (bcd) : "a" ((unsigned short) bin) , "q" ((uch) 10) : "a");
    return bcd;
#endif
}

/* FSM (Frame - Second - Minute) to sector number: */
static inline unsigned long fsm2sec(uch *fsm)
{
    long val;
    val = fsm[0] + 75 * (fsm[1] - 2 + 60 * fsm[2]);
    if(val < 0)
	return 0;
    return val;
}

#if 0
/* FSM coded in bcd to sector number: */
static unsigned long fsm_bcd2sec(uch *fsm)
{
    long val;
    val = bcd2bin(fsm[0]) + 75 * (bcd2bin(fsm[1]) - 2 + 60 * bcd2bin(fsm[2]));
    if(val < 0)
	return 0;
    return val;
}
#endif

/* MSF coded in bcd to sector number: */
static unsigned long msf_bcd2sec(uch *msf)
{
    long val;
    val = bcd2bin(msf[2]) + 75 * (bcd2bin(msf[1]) - 2 + 60 * bcd2bin(msf[0]));
    if(val < 0)
	return 0;
    return val;
}

/* sector number to FSM */
static inline void sec2fsm(unsigned long sec, uch *fsm)
{
    int tmp;
    tmp = sec / 75 + 2;
    fsm[0] = sec % 75;
    fsm[1] = tmp % 60;
    fsm[2] = tmp / 60;
}

/* sector number to FSM coded in bcd */
static void sec2fsm_bcd(unsigned long sec, uch *fsm)
{
    int tmp;
    tmp = sec / 75 + 2;
    fsm[0] = bin2bcd(sec % 75);
    fsm[1] = bin2bcd(tmp % 60);
    fsm[2] = bin2bcd(tmp / 60);
}

/* read the state of the CD ROM */
static inline unsigned short lms_state(const struct lmscd *const cd)
{
    unsigned short s;
__asm__ __volatile__ ("inb %w1,%b0\n\t"
    "movb %%al,%%ah\n\t"
    "addw $3,%w1\n\t"
/*    "jmp 1f\n1:\t" */
    "inb %w1,%b0"
    : "=a" (s):"d" (cd -> cd_port) : "dx");
    return s;
}

/* read only the lower or upper byte of the state, respective */
#define lms_state_low(cd)	(inb((cd) -> cd_port + 3))
#define lms_state_high(cd)	(inb((cd) -> cd_port))

/*
 * Read a junk of data (stolen from mcd.h, which got it from hd.c)
 * This macro won't compile for some people.  We can't avoid using
 * assembler though, but the problems went away, when READ_DATA was
 * defined as inline function.
 */
#ifdef AVOIDASM
static inline void READ_DATA(unsigned short port, uch *buf, unsigned nr)
{
    __asm__("cld;rep;insb": :"d" (port),"D" (buf),"c" (nr):"cx","di");
}
#else
#define READ_DATA(port, buf, nr) \
    __asm__("cld;rep;insb": :"d" (port),"D" (buf),"c" (nr):"cx","di")
#endif

/*
 * add_timer() does not copy the timer struct.  Calling add_timer() twice
 * therefore provokes really bad things.  So check, that the timer was really
 * clear, before adding a new one, if CLEVEL is set.
 */

#define CLEAR_TIMER \
	(del_timer(&cd -> cd_timer), \
	 cd -> cd_u_timer = 0)

#if CLEVEL >= 1
#   define SET_TIMER(jifs) ({ \
	if(del_timer(&cd -> cd_timer)) \
	    debug_0(("LMSCD: File " __FILE__ ", Line %d: Timer already set!\n", \
		     __LINE__)); \
	cd -> cd_timer.expires = (jifs); \
	add_timer(&cd -> cd_timer); })

#   define CHECK_TIMER_CLEAR ({ \
	if(del_timer(&cd -> cd_timer)) \
	    debug_0(("LMSCD: File " __FILE__ ", Line %d: Timer shouldn't be set!\n", \
		     __LINE__)); \
	})

#   define CHECK_TIMER_SET ({ \
	if(cd -> cd_request && ! cd -> cd_u_timer && ! ask_timer(&cd -> cd_timer)) { \
	    debug_0(("LMSCD: File " __FILE__ ", Line %d: Timer should be set!\n", \
		     __LINE__)); \
	    SET_TIMER(cd -> cd_timeout > 0 ? cd -> cd_timeout : 1); \
	} })

#else

#   define SET_TIMER(jifs) \
	(cd -> cd_timer.expires = (jifs), \
	 add_timer(&cd -> cd_timer))

#   define CHECK_TIMER_CLEAR
#   define CHECK_TIMER_SET

#endif

/* test a value and set it (atomic!) (stolen from system.h) */
static inline uch tas_vol(volatile uch * m)
{
	uch res;

	__asm__("xchgb %0,%1":"=q" (res),"=m" (*m):"0" (0x1));
	return res;
}

/* test a value and clear it (atomic!) (stolen from system.h) */
static inline uch tac_vol(volatile uch * m)
{
	uch res;

	__asm__("xchgb %0,%1":"=q" (res),"=m" (*m):"0" (0x0));
	return res;
}

/*
 * synchronize procession of:
 * - device interrupts
 * - timer interrupts
 * - kernel requests
 * by using two registers, which may be set with tas only:
 * - cd_sync (for all three types of requests)
 * - cd_kernel (for kernel requests against each other)
 * cd_sync should always be clear, after cd_kernel has been tested against
 * zero and set to one.
 */

#define SYNC_TIMER \
    ({  if(tas_vol(&cd -> cd_sync)) { \
	    debug_2(("lmscd: TIMER interrupt, while cd_sync set!\n")); \
	    cd -> cd_u_timer = 1; return; \
	} \
    })

/*
 * Handle interrupts, which came in, while cd_sync was set.
 * We have to be carefulf.  Testing cd_u_irqs or cd_u_timer, before we
 * set cd_sync to zero can yield problems, if another interrupt comes in
 * right after this test.
 *
 * Perform timer checking as well, if DLEVEL is set.
 */
#define RESYNC \
    ({  uch cnt; \
	CHECK_TIMER_SET; \
	cd -> cd_sync = 0; \
	for(cnt = tac_vol(&cd -> cd_u_irqs); cnt > 0; cnt--) \
	    lms_interrupt(cd -> cd_irq); \
	if(tac_vol(&cd -> cd_u_timer)) \
	    (* cd -> cd_timer.function)((long) cd); \
    })

#if 0
/* slow read (with data correction information ???) */
static int lms_slow_read(struct lmscd *const cd, uch *buf)
{
    int cnt;

    for(cnt = 0; cnt < 0x93; cnt++) {
	outb(0x31, PORT(4));
	buf[0x930 + cnt] = inb(PORT(2));
	outb(0x21, PORT(4));
	if(lms_state_low(cd) & CD_DATA_END) {
	    debug_1(("LMSCD: lms_slow_read(): error\n"));
	    return 1;
	}
	READ_DATA(PORT(2), buf + (cnt * 16), 16);
    }
    return 0;
}


/* fast read */
static inline int lms_fast_read(struct lmscd *const cd, uch *buf)
{
    READ_DATA(PORT(2), buf, 0x930);
    memset(buf + 0x930, 0, 0x93);
}
#endif


/*
 * Interrupt Handler.  Apparently, most of the work is done in interrupt
 * routines for the LMS CD drive.  We run it with all interrupts enabled.
 * We have to be carefull about multiple interrupts, etc.
 */

int pid = -1;

static void lms_interrupt(int sig)
{
    unsigned short state;
    struct lmscd *const cd = cd0;

    /* deliver interrupt to the process, who registered for it ... */
    if(pid != 0) {
	struct task_struct *p;
	for_each_task(p) {
	    if (p && p->pid == pid) {
		send_sig(SIGUSR1,p,0);
		return;
	    }
	}
    }

    /*
     * The original LMS - CD - Interrupt handler did re-read the status
     * at the end of the interrupt routine, than checked it for certain
     * bits to be set. (CD_CMD_RCVD | CD_TRACK_READY | CD_SECTOR_READY |
     * CD_RAM_OVERFLOW | CD_READ_ERROR_2).
     *
     * This implementation counts the number of interrupts received, while
     * still not finished with the old interrupts.  The result hopefully is
     * the same.
     */

    if(tas_vol(&cd -> cd_sync)) {
	cd -> cd_u_irqs++;
	debug_2(("pending interrupts now: %d\n", cd -> cd_u_irqs));
	if(cd -> cd_u_irqs >= 5)
	    debug_1(("LMSCD: pending interrupts now: %d\n", cd -> cd_u_irqs));
	return;
    }

    /* read state register of CD-ROM */
    state = lms_state(cd);

    /* acknowledge reception of state: */
    outb(0x21, PORT(4));
    debug_3(("interrupt: state = %x\n", state));
    debug_3(("state after outb(0x21, PORT(4)): %x\n", lms_state(cd)));

    /* save error conditions */
    if(state & CD_READ_ERROR)
	cd -> cd_error_seen = 1;

    /* overflow of the RAM of the CD ROM: */
    if(state & CD_RAM_OVERFLOW) {
	debug_1(("LMSCD: CD_RAM_OVERFLOW - bit set: state = %x, unread = %d!\n",
		 state, cd -> cd_unread));
	if(cd -> cd_irq_handler != lms_check_adinit) {
	    cd -> cd_unread++;
	    cd -> cd_irq_handler = lms_after_adinit;
	    /* if(cd -> cd_sectors >= 0)
		cd -> cd_retry = lms_abort_read; */
	    lms_do_adinit(cd);
	    RESYNC;
	    return;
	}
    }

    /* deliver interrupt to the correct function */
    if(state & CD_SECTOR_READY) {
	if(state & 0x0f00)
	    debug_1(("LMSCD: state = %x\n", state));
	state |= (cd -> cd_error_seen) ? CD_READ_ERROR : 0;
	cd -> cd_error_seen = 0;
	lms_irq_read(cd, state);
    }

    if(state & (CD_CMD_RCVD | CD_TRACK_READY) ||
       ! (state & CD_SECTOR_READY)) {
	if(cd -> cd_irq_handler)
	    (* cd -> cd_irq_handler)(cd, state);
	else
	    debug_1(("LMSCD: unexpected interrupt: state = %x!\n", state));
    }

    RESYNC;

    /*
     * Warn, if we might miss something important. However, while initializing
     * the adapter, the CD_RAM_OVERFLOW bit is typically set.  Ignore it here.
     */
#if CLEVEL >= 2
    {
	unsigned short new_state;
	new_state = lms_state(cd);
	if(new_state & (CD_CMD_RCVD | CD_TRACK_READY | CD_SECTOR_READY |
			CD_RAM_OVERFLOW | CD_READ_ERROR_2) &&
	   cd -> cd_irq_handler != lms_check_adinit) {
	    debug_1(("LMSCD: after interrupt: state = %x (was %x on entry)!\n",
		     new_state, state));
	}
    }
#endif
}


/*
 * Timer interrupt.
 */
static void lms_timer(unsigned long data)
{
#define cd ((struct lmscd *) data)
    SYNC_TIMER;
    outb(0x21, PORT(4));
    if(cd -> cd_retry_cnt <= 3)
	debug_1(("LMSCD: timeout: %d retries left\n", cd -> cd_retry_cnt - 1));
    if(--cd -> cd_retry_cnt > 0) {
	if(cd -> cd_retry)
	    (*cd -> cd_retry)(cd);
	SET_TIMER(cd -> cd_timeout);
    } else {
	if(cd -> cd_abort)
	    (*cd -> cd_abort)(cd);
    }
    RESYNC;
#undef cd
}


/*
 * Retry after initializing the adapter after an CD_RAM_OVERFLOW.
 * Procession is mostly identical to that of an timer interrupt.
 */
static void lms_after_adinit(struct lmscd *const cd, unsigned short state)
{
    CLEAR_TIMER;
    debug_1(("LMSCD: after adinit: %d retries left\n", cd -> cd_retry_cnt - 1));
    cd -> cd_irq_handler = 0;
    if(--cd -> cd_retry_cnt > 0) {
	if(cd -> cd_retry)
	    (*cd -> cd_retry)(cd);
	SET_TIMER(cd -> cd_timeout);
    } else {
	if(cd -> cd_abort)
	    (*cd -> cd_abort)(cd);
    }
}


/*
 * Send a command to the CD ROM drive. This function should only be called,
 * if no other processes or interrupts can access the drive (typically by
 * setting the cd_sync - field of the lmscd structure).
 */
static void lms_command(struct lmscd *const cd, const uch *cmd, int len,
			 int timeout)
{
    cd -> cd_cmd_check = 0;
    lms_start_command(cd, cmd, len, timeout);
}


/* send a command and check for error afterwards */
static void lms_command_check(struct lmscd *const cd, const uch *cmd, int len,
			       int timeout)
{
    cd -> cd_cmd_check = 1;
    cd -> cd_command2  = cmd;
    cd -> cd_cmd_len2  = len;
    cd -> cd_timeout2  = timeout;
    lms_start_command(cd, cmd, len, timeout);
}


static void lms_start_command(struct lmscd *const cd, const uch *cmd, int len,
			       int timeout)
{
    CLEAR_TIMER;
#if CLEVEL >= 1
    if(cd -> cd_cmd_sent < cd -> cd_cmd_len && cd -> cd_retry_cnt == 3) {
	int i;
	debug_0(("LMSCD: lmsi_start_command() called in middle of command transfer!\n"));
	debug_0(("LMSCD: old command ="));
	for(i = 0; i < cd -> cd_cmd_len; i++)
	    debug_0((" %x", cd -> cd_command[i]));
	debug_0(("\nLMSCD: new command ="));
	for(i = 0; i < len; i++)
	    debug_0((" %x", cmd[i]));
	debug_0(("\n"));
    }
#endif
    cd -> cd_retry_cnt   = 3;
    cd -> cd_timeout     = timeout;
    cd -> cd_irq_handler = lms_send_byte;
    cd -> cd_retry       = lms_retry_command;
    /* cd_abort and cd_success are hopefully set by the calling function */
    cd -> cd_cmd_sent    = 0;
    cd -> cd_cmd_len     = len;
    cd -> cd_command     = cmd;

    SET_TIMER(timeout);

    /* get things rolling: */
    outb(0x1, PORT(4));
}


/*
 * After an timeout, retry to send a command.
 */
static void lms_retry_command(struct lmscd *const cd)
{
#if DLEVEL >= 1
    { int i;
    debug_1(("LMSCD: lms_retry_command() called, after %d of %d command bytes sent\n",
	     cd -> cd_cmd_sent, cd -> cd_cmd_len));
    debug_1(("LMSCD: command ="));
    for(i = 0; i < cd -> cd_cmd_len; i++)
	debug_1((" %x", cd -> cd_command[i]));
    debug_1(("\n"));
    }
#endif
    cd -> cd_cmd_sent    = 0;
    cd -> cd_irq_handler = lms_send_byte;

    /*
     * !!! After an overflow of adapter RAM, we should perform an adapter
     * init (to clear the dirty RAM ...)
     */

    outb(0x1, PORT(4));
}


/*
 * Send a command byte on receiption of a device interrupt.
 */
static void lms_send_byte(struct lmscd *const cd, unsigned short state)
{
    if(state & CD_CMD_READY) {
	outb(cd -> cd_command[cd -> cd_cmd_sent], PORT(5));
	cd -> cd_irq_handler = lms_read_echo;
    } else {
	/* signal CD ROM again, that a command is pending */
	debug_2(("signal_handler: trying to start command again!\n"));
	outb(1, PORT(4));
    }
}


/*
 * Test the byte, which was just send.
 */
static void lms_read_echo(struct lmscd *const cd, unsigned short state)
{
    uch echo, orig;

    /* Is there a command byte echoed from the CD-ROM? */
    if(! (state & CD_CMD_RCVD)) {
	debug_1(("LMSCD: CD-ROM did not echo command byte!\n"));
	return;
    }

    cd -> cd_echo[cd -> cd_cmd_sent] = echo = inb(PORT(1));
    orig = cd -> cd_command[cd -> cd_cmd_sent];

    /* check for transmission errors */
    if((state & CD_CMD_ERROR) || (orig != 0x9c && echo != orig)) {
	debug_1(("LMSCD: error in sending command: sent = %x; echo = %x\n",
		 cd -> cd_command[cd -> cd_cmd_sent], echo));
	/*
	 * Simulate timer interrupt to force retry/abortion (the interrupt
	 * is processed upon exit of lms_interrupt()
	 */
	cd -> cd_irq_handler = 0;
	CLEAR_TIMER;
	lms_timer((unsigned long) cd);
	return;
    }

    /* bytes left to transfer? */
    if(++cd->cd_cmd_sent < cd->cd_cmd_len) {
	if(state & CD_CMD_READY) {
	    outb(cd->cd_command[cd->cd_cmd_sent], PORT(5));
	    return;
	} else {
	    cd -> cd_irq_handler = lms_send_byte;
	    outb(1, PORT(4));
	    return;
	}
    }

    /*
     * Complete command transferred.  Any error checks left?
     */
    CLEAR_TIMER;
    if(cd -> cd_cmd_check) {
	lms_do_check(cd);
	if(cd -> cd_cmd_check)
	    return;
    }

    /* The command has been transferred successfully: */
    cd -> cd_irq_handler = 0;
    if(cd -> cd_success)
	(*cd -> cd_success)(cd);
    return;
}


/*
 * After a command has been transferred, check for error:
 */
static void lms_do_check(struct lmscd *const cd)
{
    /*
     * meanings of cd_cmd_check:
     * 1: just tried to send the command, check for the error
     *    bit in lms_state()
     * 2: just requested drive state, test error codes of them.
     * 3: just cleared the error, request the drive state again.
     * 4: just re-requested the drive state.  Abort, if still errors
     *    left, or the earlier errors were fatal.  Otherwise,
     *    retry to send command.
     * 5: exactly as 1, but after first retry ...
     * 9: again exactly as 1, but after second retry ...
     * 12: terminate in any case!
     */
    debug_3(("checking for command errors: cd_cmd_check = %d\n",
	     cd -> cd_cmd_check));
    cd -> cd_cmd_check++;
    switch((cd -> cd_cmd_check) % 5) {
    case 2:
	/*
	 * 1: just tried to send the command, check for the error
	 *    bit in lms_state()
	 */
	if(lms_state_low(cd) & CD_ERROR) {
	    debug_1(("LMSCD: command_check: state = %x after transmission of command!\n",
		     lms_state(cd)));
	    lms_start_command(cd, cmd_3a_state, 9, 110);
	    return;
	}
	cd -> cd_drive_error = 0;
	cd -> cd_adapter_error = 0;
	break;

	/* 2: just requested drive state, test error codes of them. */
    case 3:
	debug_1(("LMSCD: command_check: drive_error = %d, adapter_error = %d!\n",
		 cd -> cd_echo[7], cd -> cd_echo[8]));
	if((cd -> cd_drive_error = cd -> cd_echo[7]) |
	   (cd -> cd_adapter_error = cd -> cd_echo[8])) {
	    cd -> cd_sectors = 0;
	    lms_start_command(cd, cmd_clear_error, 1, 162);
	    return;
	}
	break;

	/* 3: just cleared the error, request the drive state again. */
    case 4:
	lms_start_command(cd, cmd_3a_state, 9, 110);
	return;

	/* 4: initialize adapter. */
    case 0:
	lms_adinit(cd);
	return;

	/*
	 * 5: just re-requested the drive state.  Abort, if still errors
	 *    left, or the earlier errors were fatal.  Otherwise,
	 *    retry to send command.
	 */
    case 1:
	if(! (cd->cd_echo[7] | cd->cd_echo[8]) && cd->cd_cmd_check <= 11) {
	    /*
	     * Adapter errors 1/4/17 appear, if illegal commands, etc. were
	     * transmitted.  They can appear, if a new command is started,
	     * while a command transfer is still in progress.
	     */
	    switch(cd -> cd_adapter_error) {
	    case 1:
	    case 4:
	    case 17:
#if DLEVEL >= 1
		{ int i;
		debug_1(("LMSCD: command_check: trying to restart command!\n"));
		debug_1(("LMSCD: command ="));
		for(i = 0; i < cd -> cd_cmd_len2; i++)
		    debug_1((" %x", cd -> cd_command2[i]));
		debug_1(("\n"));
		}
#endif
		lms_start_command(cd, cd -> cd_command2, cd -> cd_cmd_len2,
				   cd -> cd_timeout2);
		return;
	    }
	}

	if(cd -> cd_drive_error == 0x1e ||
	   cd -> cd_drive_error == 0x1f)
	    cd -> cd_state |= CD_STATE_NODISK;

	cd -> cd_irq_handler = 0;
	cd -> cd_retry_cnt = 0;
	if(cd -> cd_abort)
	    (*cd -> cd_abort)(cd);
	return;
    }
    cd -> cd_cmd_check = 0;
}


/*
 * Initialize the CM 250 CD-ROM adapter.
 * The caller should have synchronized procession by setting cd_sync.
 * The lms_init() function does not need to do so, because it is called,
 * while Linux is still in single user mode.
 */
static void lms_adinit(struct lmscd *const cd)
{
    /* safety first: */
    CLEAR_TIMER;
    debug_3(("lms_adinit()\n"));

    cd -> cd_retry_cnt   = 3;
    cd -> cd_timeout     = 2;
    cd -> cd_irq_handler = lms_check_adinit;
    cd -> cd_retry       = lms_retry_adinit;
    lms_do_adinit(cd);
    SET_TIMER(2);
}


static void lms_do_adinit(struct lmscd *const cd)
{
    cd -> cd_error_seen  = 0;
    cd -> cd_unread      = 0;
    cd -> cd_error_free  = 0;
    outb(0x61, PORT(4));
}


/*
 * This is called after timeout of the adapter init ...
 */
static void lms_retry_adinit(struct lmscd *const cd)
{
    debug_1(("LMSCD: lms_retry_adinit() called!\n"));
    outb(0x61, PORT(4));
}


/*
 * And this on receiption of a device interrupt ...
 */
static void lms_check_adinit(struct lmscd *const cd, unsigned short state)
{
    CLEAR_TIMER;
    debug_3(("lms_check_adinit()\n"));
    if((state & 0xffef) == 0x0041) {
	cd -> cd_retry_cnt   = -1;
	cd -> cd_irq_handler = 0;
	if(cd -> cd_cmd_check) {
	    lms_do_check(cd);
	    return;
	}
	if(cd -> cd_success)
	    (* cd -> cd_success)(cd);
	return;
    } else {
	debug_1(("LMSCD: lms_check_adinit() failed: state = %x!\n", state));
	lms_timer((unsigned long) cd);
	return;
    }
}


/*
 * Reset the CD-ROM drive.  The adapter sends an IRQ for any sh**, but
 * not, if the CD_ERROR status bit gets active.  A successfull reset_drive
 * always ends with an error condition, however!
 *
 * In other words: this is the only function, which uses the timer for
 * polling.
 */
#define RETRY_RESET 500
static void lms_reset_drive(struct lmscd *const cd)
{
    /* safety first: */
    CLEAR_TIMER;
    debug_1(("LMSCD: resetting drive; cd_read_retry = %d\n",
	     cd -> cd_read_retry));

    cd -> cd_identify    = 0;	/* type of CD-ROM drive not known yet */
    cd -> cd_retry_cnt   = RETRY_RESET;
    cd -> cd_timeout     = 1;
    cd -> cd_retry       = lms_poll_for_error;
    cd -> cd_success     = lms_identify_drive;
    cd -> cd_abort       = lms_bad_identify;
    cd -> cd_irq_handler = 0;

    cd -> cd_sectors     = 0;
    cd -> cd_error_seen  = 0;
    cd -> cd_unread      = 0;
    cd -> cd_error_free  = 0;

    SET_TIMER(1);
    outb(0x21, PORT(4));
}


/*
 * Poll the drive for the error bit to go on ...
 */
static void lms_poll_for_error(struct lmscd *const cd)
{
    /* debug_3(("lms_poll_for_error() called!\n")); */
    if(cd -> cd_retry_cnt == RETRY_RESET - 1)
	outb(0xa1, PORT(4));
    else if(cd -> cd_retry_cnt == RETRY_RESET - 2)
	outb(0x21, PORT(4));
    else if((lms_state_low(cd) & CD_ERROR)) {
	CLEAR_TIMER;
	cd -> cd_retry_cnt = -1;
	if(cd -> cd_success)
	    (*cd -> cd_success)(cd);
    }
}


/*
 * Error in resetting or identifying the drive ...
 */
static void lms_bad_identify(struct lmscd *const cd)
{
    debug_1(("LMSCD: resetting/identifying drive failed!\n"));
    debug_1(("LMSCD: cd_identify = %d\n", cd -> cd_identify));
    cd -> cd_identify = -1;
    cd -> cd_state |= CD_STATE_UNIDEN;
    if(cd -> cd_read_retry) {
	lms_request(cd, 0);
    }
}


/* Offset of member MEMBER in a struct of type TYPE.  */
#ifndef offsetof
#define offsetof(TYPE, MEMBER) ((size_t) &((TYPE *)0)->MEMBER)
#endif

/*
 * Clear all information about last CD.
 */
static void lms_set_all_zero(struct lmscd *const cd)
{
    if(cd -> cd_track_info)
	kfree(cd -> cd_track_info);

    /* fast and painless: */
    memset(&cd -> cd_drive_error, 0,
	   sizeof(*cd) - offsetof(struct lmscd, cd_drive_error));

    /* anything else is an error! */
    cd -> cd_buf_sector = (unsigned) -1;
}


/*
 * Try to identify drive ...
 * This is done immediately after a reset.  We first read the drive
 * state.  It should be set to error 0xe.  Then we clear the error
 * register, and read the reply to the 2d - command.
 */
static void lms_identify_drive(struct lmscd *const cd)
{
    debug_3(("lms_identify_drive() called!\n"));

    switch(++cd -> cd_identify) {
    case 1:
	lms_command(cd, cmd_3a_state, 9, 110);
	/*
	 * It looks crazy, but we need it (because this function is called
	 * from a retry branch, and lms_timer() will set the timer again.
	 */
	CLEAR_TIMER;
	break;

    case 2:
	if(cd -> cd_echo[7] != 0xe) {
	    debug_1(("LMSCD: reset_drive() did not set correct error!\n"));
	    lms_bad_identify(cd);
	} else {
	    lms_command(cd, cmd_clear_error, 1, 162);
	}
	break;

    case 3:
	if(cd -> cd_read_retry)
	    lms_start_read(cd);
	else
	    lms_command_check(cd, cmd_2d_state, sizeof(cmd_2d_state), 81);
	break;

    case 4:
	/* test result of the 0x2d command */
	if(cd->cd_echo[2]==4 && ((cd->cd_echo[1]==4 && cd->cd_echo[5]>=10) ||
				 cd->cd_echo[1] == 5)) {
	    if(cd -> cd_echo[1] == 5)
		cd -> cd_type = 1;
	    else
		cd -> cd_type = 2;
	    cd -> cd_state = CD_STATE_NODISK;
	    lms_set_all_zero(cd);
	    /* TODO: assign normal audio input/output and volumes */
	    /* mov ..., 0 */
	} else {
	    debug_1(("LMSCD: Drive identification failed\n"));
	    cd -> cd_state |= CD_STATE_UNIDEN;
	}
	cd -> cd_identify = -1;
	break;
    }
}


/*
 * Read the state of the CD-ROM drive.  This functions checks, whether
 * a disk is in, etc.  If the disk was changed, this functions reads in
 * the new information about size and tracks.
 */
static void lms_read_state(struct lmscd *const cd)
{
    /* safety first: */
    CLEAR_TIMER;

    debug_2(("lms_read_state(): cd_read_state = %d\n", cd -> cd_read_state));
    switch(++cd -> cd_read_state) {
    case 1:
	cd -> cd_success     = lms_read_state;
	cd -> cd_abort       = lms_error_state;
    case 3:
	lms_command(cd, cmd_3a_state, 9, 110);
	break;

    case 2:
	if((cd -> cd_save_state    = cd -> cd_echo[6]) & 0x10 ||
	   (cd -> cd_drive_error   = cd -> cd_echo[7]) ||
	   (cd -> cd_adapter_error = cd -> cd_echo[8])) {
	    /* error seen: clear error, then re-read state */
	    lms_command(cd, cmd_clear_error, 1, 162);
	    break;
	}
	cd -> cd_read_state = 4;

    case 4:
	/*
	 * Don't allow multiple errors.
	 */
	if((cd -> cd_save_state    = cd -> cd_echo[6]) & 0x10 ||
	   (cd -> cd_drive_error   = cd -> cd_echo[7]) ||
	   (cd -> cd_adapter_error = cd -> cd_echo[8])) {
	    lms_error_state(cd);
	    return;
	}
	cd -> cd_state &= ~CD_STATE_ERROR;

	/*
	 * Has the CD been changed, the door been opened, etc.?
	 */
	if(cd -> cd_echo[6] & 0x18 ||
	   cd -> cd_save_state & 0x10 ||
	   cd -> cd_state & CD_STATE_NODISK) {
	    lms_set_all_zero(cd);
	    lms_adinit(cd);
	    break;
	}
	cd -> cd_read_state = 5;

    case 5:
	/*
	 * Door currently open, or no CD in?
	 */
	if(cd -> cd_echo[6] & 0x18) {
	    if(cd -> cd_echo[6] & 0x10)
		cd -> cd_state |= CD_STATE_OPEN;
	    else
		cd -> cd_state &= ~CD_STATE_OPEN;
	    cd -> cd_state = (cd -> cd_state | CD_STATE_NODISK) &
		    ~CD_STATE_AUDIO & ~CD_STATE_MEDIA & ~CD_STATE_PAUSED;
	} else {
	    /*
	     * There is a disk in the drive.  But has it been changed?
	     */
	    if(cd -> cd_state & CD_STATE_NODISK || cd -> cd_save_state & 0x10) {
		/* if(lms_new_cd(cd))
		       goto error; */
		invalidate_buffers(cd -> cd_dev);
		cd -> cd_state |= CD_STATE_MEDIA;
		/*
		 * To get the size of the CD, we have to produce an error first
		 * by seeking past the end of the CD.
		 */
		lms_command(cd, cmd_seek_over_end, sizeof(cmd_seek_over_end),
			     162);
		break;
	    }
	    cd -> cd_state &= ~CD_STATE_NODISK & ~CD_STATE_OPEN;

	    /*
	     * Audio play finished:
	     */
	    if(cd -> cd_state & CD_STATE_AUDIO && ! (cd -> cd_echo[6] & 1)) {
		cd -> cd_state &= ~CD_STATE_PAUSED & ~ CD_STATE_AUDIO;
		/* TODO: set information about last audio play to zero  */
	    }
	}
	wake_up(&cd -> cd_command_queue);
	break;

    case 6:
	/*
	 * After we seeked past the end, we now request the drive state ...
	 */
	cd -> cd_state &= ~CD_STATE_NODISK & ~CD_STATE_OPEN;
	lms_command(cd, cmd_3a_state, 14, 110);
	break;

    case 7:
	/*
	 * And clear the error again ...
	 */
	lms_command(cd, cmd_clear_error, sizeof(cmd_clear_error), 162);
	break;

    case 8:
	/*
	 * Ready to read the information after an disk change ...
	 */
	/* audio tracks: */
	cd -> cd_first_track = bcd2bin(cd -> cd_echo[12]);
	cd -> cd_last_track = bcd2bin(cd -> cd_echo[13]) /*  - 1 ??? */;

	/* copy fsm information, and add 1 Frame */
	cd -> cd_size_fsm[0] = bcd2bin(cd -> cd_echo[9]);
	cd -> cd_size_fsm[1] = bcd2bin(cd -> cd_echo[10]);
	cd -> cd_size_fsm[2] = bcd2bin(cd -> cd_echo[11]);
	if(++cd -> cd_size_fsm[0] == 75) {
	    cd -> cd_size_fsm[0] = 0;
	    if(++cd -> cd_size_fsm[1] == 60) {
		cd -> cd_size_fsm[1] = 0;
		++cd -> cd_size_fsm[2];
	    }
	}

	cd -> cd_size = fsm2sec(cd -> cd_size_fsm);

	debug_1(("LMSCD: new cd: size = %d/%d/%d fsm = %ld decimal\nfirst/last track = %d/%d\n",
		 cd -> cd_size_fsm[0],cd -> cd_size_fsm[1],cd -> cd_size_fsm[2],
		 cd -> cd_size, cd -> cd_first_track, cd -> cd_last_track));

	/* lms_read_track_info(cd); */
	wake_up(&cd -> cd_command_queue);
	break;
    }
}


/*
 * Error in reading the state of the drive.
 */
static void lms_error_state(struct lmscd *const cd)
{
    cd -> cd_state |= CD_STATE_ERROR;
    wake_up(&cd -> cd_command_queue);
}


/*
 * Test, whether the disk changed.
 */
int check_lms_media_change(int dev, int flag)
{
    struct lmscd * const cd = GETCD(dev);
    int retval;

    /* kernel calls may (and sometimes must) sleep: */
    while(tas_vol(&cd -> cd_kernel))
	interruptible_sleep_on(&cd -> cd_kernel_queue);

    /*
     * Interrupts must never call "normal" kernel routines.  And we just
     * ensured, that we are the only process accessing the drive.  But
     * this means, that cd_sync should be clear, when we come here:
     */
    if(tas_vol(&cd -> cd_sync))
	debug_1(("LMSCD: cd_sync set in normal kernel call!"));

    /*
     * Now, we may safely read the drive state:
     */
    cd -> cd_read_state = 0;
    lms_read_state(cd);
    RESYNC;
    sleep_on(&cd -> cd_command_queue);

    if((retval = (cd -> cd_state & CD_STATE_MEDIA) / CD_STATE_MEDIA))
	invalidate_buffers(cd -> cd_dev);

    if (!flag)
    {
	cd -> cd_state &= ~CD_STATE_MEDIA;
    }

    cd -> cd_kernel = 0;
    wake_up(&cd -> cd_kernel_queue);

    return retval;
}


/*
 * Open the device special file.  Check that a disk is in.
 */
static int lms_open(struct inode *ip, struct file *fp)
{
    struct lmscd * const cd = GETCD(ip -> i_rdev);

    debug_1(("LMSCD: lms_open() called!\n"));
    if (! cd)
	return -ENXIO;			/* no hardware */

#if 0
    /*
     * If the CD ROM special device has already been opened, we do not
     * test for disk changes.
     */
    if(! cd -> cd_open)
#endif
	check_disk_change(ip -> i_rdev);
    if(cd -> cd_state & CD_STATE_BAD) {
	if(cd -> cd_state & CD_STATE_NODISK)
	    debug_1(("LMSCD: no disk in drive!\n"));
	return -EIO;
    }

#if 0
    /*
     * The number of open file descriptors should also be in
     * fp -> f_open.  Can we reliably use this count, or may there
     * be races (eg. two opens at the same time, and the second
     * reaches lms_open() first).
     */
    cd -> cd_open++;
#endif
    return 0;
}


/*
 * After a close, we flush all blocks from buffer cache.
 */
static void lms_release(struct inode *ip, struct file *fp)
{
#if DLEVEL >= 1 && DFUNC
    struct lmscd * const cd = GETCD(ip -> i_rdev);
    debug_1(("LMSCD: lms_release() called!\n"));
#endif

    sync_dev(ip -> i_rdev);		/* needed??? */
    invalidate_buffers(ip -> i_rdev);
}


/*
 * Synchronize, then start to handle read requests.
 */
static void do_lms_request(void)
{
    struct lmscd *cd;

retry:
    if(CURRENT == NULL || CURRENT -> sector == -1)
	return;

    cd = GETCD(CURRENT -> dev);

#if CLEVEL >= 1
    if(cd -> cd_request)
	debug_0(("LMSCD: cd_request already set on call to do_lms_request()!\n"));
#endif

    while(tas_vol(&cd -> cd_kernel))
	interruptible_sleep_on(&cd -> cd_kernel_queue);

    if(tas_vol(&cd -> cd_sync))
	debug_1(("LMSCD: cd_sync weirdness!"));

    if(! cd) {
	debug_1(("LMSCD: Illegal minor device %d\n", MINOR(CURRENT -> dev)));
	end_request(0);
	cd -> cd_request = 0;
	goto retry;
    }

    lms_request(cd, -1);

    RESYNC;
    cd -> cd_kernel = 0;
    wake_up(&cd -> cd_kernel_queue);
}


/*
 * The second arg determines, whether:
 *  1: the current request has been terminated successfully
 *  0: error in processing the current request
 * -1: the current request should not be changed.
 */
static void lms_request(struct lmscd *const cd, int flag)
{
    long sect;

    debug_2(("lms_request() called!\n"));
    /* CLEAR_TIMER; */

    if(flag >= 0)
	end_request(flag);

#if CLEVEL >= 1
    else if(cd -> cd_request) {
	debug_0(("LMSCD: lms_request() maybe called recursive!\n"));
    }
#endif

    while(1) {
	cd -> cd_request = 0;
	INIT_REQUEST;

	if (CURRENT == NULL || CURRENT -> sector == -1)
	    return;

	cd -> cd_request = 1;
	if (CURRENT -> cmd != READ) {
	    debug_1(("LMSCD: bad cmd %d\n", CURRENT -> cmd));
	    end_request(0);
	    continue;
	}

	if (CURRENT -> nr_sectors & 1) {
	    debug_1(("LMSCD: sector count odd!\n"));
	    end_request(0);
	    continue;
	}

	if (CURRENT -> sector & 1) {
	    debug_1(("LMSCD: start sector odd!\n"));
	    end_request(0);
	    continue;
	}

	/*
	 * Is the data already there?
	 */
	if(CURRENT -> sector == cd -> cd_buf_sector) {
	    memcpy(CURRENT -> buffer, cd -> cd_buffer, 1024);
	    CURRENT -> nr_sectors -= 2;
	    CURRENT -> sector     += 2;
	    CURRENT -> buffer     += 1024;
	}

	if (CURRENT -> nr_sectors == 0)
	{
	    end_request(1);
	    continue;
	}

	/*
	 * lms_start_read always reads ahead some sectors into the memory
	 * of the adapter.  Check, whether we can at least partly suffice
	 * the current request from these sectors:
	 */
	cd -> cd_read_retry = 3;
	sect = CURRENT -> sector / 4;
	if(cd->cd_sector <= sect && cd->cd_sector + ONE_TRACK > sect &&
	   cd->cd_sector + cd->cd_sectors > sect) {
	    while(cd -> cd_error_free > 0 && CURRENT -> nr_sectors > 0) {
		if(cd -> cd_sector == CURRENT -> sector / 4) {
		    /*
		     * If lms_read() does hit an error, it automatically
		     * clears the error, then retries the read ...
		     */
		    if(lms_read(cd))
			return;
		} else {
		    if(lms_discard(cd))
			return;
		}
		cd -> cd_unread--;
		cd -> cd_error_free--;
	    }
	    if (CURRENT -> nr_sectors == 0)
	    {
		end_request(1);
		continue;
	    }
	} else if(cd -> cd_sectors > 0) {
	    lms_do_abort(cd, 2);
	    return;
	}

	/*
	 * Issue a read command only, if nothing else is going on.
	 */
	if(cd -> cd_sectors == 0) {
	    if(cd -> cd_unread)
		lms_do_abort(cd, 3);
	    else
		lms_start_read(cd);
	}
	return;
    }
}


/*
 * Start a read command.
 */
static void lms_start_read(struct lmscd *const cd)
{
    static uch cmd[] = {
	0xa6, 0, 0, 0, 0, 0, 0
    };
    long first;
    long cnt;
    int tmp;

    if (CURRENT == NULL || CURRENT -> sector == -1) {
	CHECK_TIMER_CLEAR;
	return;
    }

#if CLEVEL >= 1
    if(cd -> cd_sectors > 0 && cd -> cd_read_retry == 3)
	debug_0(("LMSCD: lms_start_read() called with cd_sectors > 0!\n"));
    if(cd -> cd_sectors < 0)
	debug_0(("LMSCD: lms_start_read() called with cd_sectors < 0!\n"));
#endif

    if(--cd -> cd_read_retry < 0) {
	debug_1(("LMSCD: read failed!\n"));
	lms_request(cd, 0);
	return;
    }

    if (CURRENT -> nr_sectors == 0)
    {
	lms_request(cd, 1);
	return;
    }

    first = CURRENT -> sector / 4;
    cnt = cd -> cd_size - first;
    if(cnt <= 0) {
	debug_1(("LMSCD: read beyond end of disk!\n"));
	lms_request(cd, 0);
	return;
    }
    if(cnt >= 10000)
	cnt = 9999;

    debug_2(("read request: sector = %d; nr_sectors = %d; buffer = %p\n",
	     CURRENT -> sector, CURRENT -> nr_sectors, CURRENT -> buffer));

    sec2fsm_bcd(first, cmd + 1);
    cmd[4] = bin2bcd(cnt % 100);
    tmp = cnt / 100;
    cmd[5] = bin2bcd(tmp % 100);
    cmd[6] = bin2bcd(tmp / 100);

    debug_2(("lms_start_read: %ld sectors from sector %ld(F: %x, S: %x, M: %x) on\n",
	     cnt, first, cmd[1], cmd[2], cmd[3]));

    cd -> cd_sectors = cnt;
    cd -> cd_sector  = first;
    cd -> cd_abort   = lms_abort_read;
    cd -> cd_success = lms_wait_read;
    lms_command_check(cd, cmd, sizeof(cmd), 182);
}


/*
 * After the read command has been transferred successfully, prepare to read
 * the data.
 */
static void lms_wait_read(struct lmscd *const cd)
{
    cd -> cd_retry_cnt   = 1;
    /* cd -> cd_abort       = lms_abort_read; */
    cd -> cd_timeout     = 218;
    outb(0x21, PORT(4));

    SET_TIMER(cd -> cd_timeout);
}


/*
 * Abort read.
 */
static void lms_abort_read(struct lmscd *const cd)
{
    lms_do_abort(cd, 4);
}


static void lms_do_abort(struct lmscd *const cd, int where)
{
    debug_2(("lms_abort_read() called\n"));
    if(cd -> cd_sectors >= 0) {
	cd -> cd_success    = lms_cont_abort;
	cd -> cd_abort      = lms_retry_abort;
	cd -> cd_sectors    = -6;
	lms_command(cd, cmd_3a_state, 9, 110);
    } else {
	/*
	 * Do nothing but ensure, that a timer is running!
	 */
	debug_1(("LMSCD: call from location %d to lms_do_abort() while abort in step %ld!\n",
	         where, cd -> cd_sectors));
	/*
	CLEAR_TIMER;
	SET_TIMER(cd -> cd_timeout);
	*/
	if(cd -> cd_sectors >= -6)
	    cd -> cd_sectors--;
	lms_retry_abort(cd);
    }
}


/*
 * To abort a read, we need a few commands.  After each has been sent
 * successfully, come back to this function for the next.  If something
 * fails, lms_retry_abort() will be called, which tries to initialize
 * the adapter, and then continues with the abort.
 */
static void lms_cont_abort(struct lmscd *const cd)
{
    debug_2(("lms_cont_abort: cd_sectors = %d\n", cd -> cd_sectors));
    cd -> cd_success = lms_cont_abort;
    cd -> cd_abort   = lms_retry_abort;

    switch(++cd -> cd_sectors) {
    default:
	debug_0(("LMSCD: lms_cont_abort(): sectors set to illegal value %ld\n",
		 cd -> cd_sectors - 1));
	cd -> cd_sectors    = -6;

    case -6:
    case -3:
	/*
	 * After we saw an error: read the error state.
	 */
	cd -> cd_echo[7] = 0;
	cd -> cd_echo[8] = 0;
	lms_command(cd, cmd_3a_state, 9, 110);
	break;

    case -2:
    case -5:
	/*
	 * Test the error code, then clear the error.
	 */
	cd -> cd_drive_error   = cd -> cd_echo[7];
        cd -> cd_adapter_error = cd -> cd_echo[8];

	if(cd -> cd_drive_error == 0x1e ||
	   cd -> cd_drive_error == 0x1f)
	    cd -> cd_state |= CD_STATE_NODISK;
	lms_command(cd, cmd_clear_error, 1, 162);
	break;

    case -4:
	/*
	 * Seek over disk end to abort the current read.
	 */
	lms_command(cd, cmd_seek_over_end, sizeof(cmd_seek_over_end), 162);
	break;

    case -1:
	lms_adinit(cd);
	break;

    case 0:
	/*
	 * Retry the read.
	 */
	lms_start_read(cd);
	break;
    }
}


/*
 * Reset adapter, if abortion of an read command failed.
 */
static void lms_retry_abort(struct lmscd *const cd)
{
    debug_1(("LMSCD: lms_retry_abort() called: abort step = %ld!\n",
	     cd -> cd_sectors));

    /* Does this help in case of echoed, but unread command bytes ??? */
    inb(PORT(1));
    inb(PORT(1));

    /*
     * Best idea right now is to initialize the adapter.  But it
     * will very likely end up with state 1 instead of 41, so try
     * it only once!
     */
    cd -> cd_abort     = lms_cont_abort;
    lms_adinit(cd);
    cd -> cd_retry_cnt = 1;
    if(cd -> cd_sectors > -3)
	cd -> cd_sectors = -5;
}


/*
 * Received an interrupt, which should tell us: we are ready to
 * read a sector.
 */
static void lms_irq_read(struct lmscd *const cd, unsigned short state)
{
    int sect;
    int unread;

    /* CD - ROM ready to send a sector? */
    if(state & CD_SECTOR_READY && cd -> cd_sectors > 0) {
	/*
	 * Stop timer by now, restart it afterwards, if sectors left.
	 */
	CLEAR_TIMER;

	/*
	 * Abortion in process, but drive is still sending sectors.
	 */
	if(cd -> cd_sectors < 0) {
	    cd -> cd_unread++;
	    return;
	}

#if CLEVEL >= 1
	if(cd -> cd_unread >= cd -> cd_sectors) {
	    cd -> cd_unread++;
	    debug_0(("LMSCD: drive is sending unrequested sectors!\n"));
	    lms_do_abort(cd, 5);
	    return;
	}
#endif

	if(state & CD_READ_ERROR) {
	    cd -> cd_unread++;
	    debug_1(("LMSCD: CD_READ_ERROR - bit set: state = %d!\n", state));
	    lms_do_abort(cd, 6);
	    return;
	}

	/*
	 * If we are currently processing a request, we must either read or
	 * discard the sector.
	 */
	if(cd -> cd_request) {
#if CLEVEL >= 1
	    if(cd -> cd_unread) {
		debug_0(("LMSCD: cd_unread set, though processing request!\n"));
	    }
#endif
	    if(cd -> cd_sector == CURRENT -> sector / 4) {
		if(lms_read(cd))
		    return;
	    } else {
#if CLEVEL >= 1
		if(cd->cd_sector > CURRENT->sector / 4 ||
		   cd->cd_sector + cd->cd_sectors <= CURRENT->sector / 4 ||
		   cd->cd_sector + ONE_TRACK <= CURRENT->sector / 4) {
		    debug_0(("LMSCD: CURRENT request outside of read window!\n"));
		    lms_do_abort(cd, 7);
		    return;
		}
#endif
		if(lms_discard(cd))
		    return;
	    }
	} else {
	    if(cd -> cd_unread == cd -> cd_error_free)
		cd -> cd_error_free++;
	    cd -> cd_unread++;
	    if(cd -> cd_unread >= cd -> cd_adap_sect - 2) {
		debug_2(("LMSCD: aborting read-ahead!\n"));
		lms_do_abort(cd, 8);
		return;
	    }
	    sect = cd -> cd_sectors;
	}

	sect = cd -> cd_sectors;
	unread = cd -> cd_unread;
	if(cd -> cd_request && CURRENT -> nr_sectors <= 0) {
	    lms_request(cd, 1);

	} else if(sect <= 0 && cd -> cd_request && CURRENT -> nr_sectors)
	    lms_start_read(cd);

	if(sect > unread && cd -> cd_sectors > 0) {
	    /* restart timer again for next data block */
	    SET_TIMER(cd -> cd_timeout);
	}
    }
}


/*
 * Read a sector from the adapter.  The sector read should match the
 * current request.
 */
static int lms_read(struct lmscd *const cd)
{
    uch header[16];
    uch ext_header[8];
    uch tail[0x120];
    uch *adr;

    debug_2(("in read interrupt: sector = %d; nr_sectors = %d; buffer = %p\n",
	     CURRENT -> sector, CURRENT -> nr_sectors, CURRENT -> buffer));
    if(CURRENT -> nr_sectors < 2) {
	debug_1(("LMSCD: data read though no data requested!\n"));
	lms_do_abort(cd, 9);
	return 1;
    }

    /* read header */
    READ_DATA(PORT(2), header, 0x10);

    /* check sector number */
    if(msf_bcd2sec(header + 12) != cd -> cd_sector) {
	debug_1(("LMSCD: wrong sector read!\n"));
	lms_do_abort(cd, 10);
	return 1;
    }

    /* read extended header, if necessary */
    if(header[15] == 2)
	READ_DATA(PORT(2), ext_header, 0x08);

    /* read first kilobyte */
    if(CURRENT -> sector == cd -> cd_sector * 4) {
	adr = (uch *) CURRENT -> buffer;
	CURRENT -> nr_sectors -= 2;
	CURRENT -> sector     += 2;
	CURRENT -> buffer     += 1024;
    } else {
	adr = cd -> cd_buffer;
	cd -> cd_buf_sector = cd -> cd_sector * 4;
    }
    READ_DATA(PORT(2), adr, 0x400);

    /* second kilobyte */
    if(CURRENT -> sector != cd -> cd_sector * 4 + 2) {
	debug_1(("LMSCD: sectors in request and in CD data struct different!\n"));
	lms_do_abort(cd, 11);
	return 1;
    }
    if(CURRENT -> nr_sectors >= 2) {
	adr = (uch *) CURRENT -> buffer;
	CURRENT -> nr_sectors -= 2;
	CURRENT -> sector     += 2;
	CURRENT -> buffer     += 1024;
    } else {
	adr = cd -> cd_buffer;
	cd -> cd_buf_sector = cd -> cd_sector * 4 + 2;
    }
    READ_DATA(PORT(2), adr, 0x400);

    /* read the rest of the raw data */
    READ_DATA(PORT(2), tail, 0x120 - (header[15] == 2 ? 8 : 0));

    /* Any sector, not only every read may be retried 3 times */
    cd -> cd_read_retry = 3;
    cd -> cd_sector++;
    cd -> cd_sectors--;
    return 0;
}


/*
 * Discard a sector from adapter memory.
 */
static int lms_discard(struct lmscd *const cd)
{
    int i;

    debug_2(("discarding sector!\n"));
    for(i = 0; i < 0x930; i++)
	inb(PORT(2));

    cd -> cd_read_retry = 3;
    cd -> cd_sector++;
    cd -> cd_sectors--;

    return 0;
}


static struct file_operations lms_fops = {
	NULL,			/* lseek - default */
	block_read,		/* read - general block-dev read */
	block_write,		/* write - general block-dev write */
	NULL,			/* readdir - bad */
	NULL,			/* select */
	NULL /* lms_ioctl */,		/* ioctl */
	NULL,			/* mmap */
	lms_open,		/* open */
	lms_release		/* release */
};


/*
 * LMSCD interrupt descriptor
 */
static struct sigaction lms_sigaction = {
    lms_interrupt,
    0,
    0,
    NULL
};


static char *drv_type[] = {
    "unknown", "CM205", "CM202"
};


void lmscd_cleanup(void)
{
    free_irq(LMSCD_INTR_NR);
    unregister_blkdev(MAJOR_NR, "lmscd");
}

/*
 * This routine is called once after booting.  Check, that we can
 * access the driver, etc.
 */
unsigned long lmscd_init(unsigned long mem_start, unsigned long mem_end)
{
    struct lmscd cd_;
    struct lmscd *const cd = &cd_;

    memset(cd, 0, sizeof(*cd));

    cd -> cd_port = LMSCD_PORT;
    cd -> cd_irq = LMSCD_INTR_NR;
    cd -> cd_dev = MAJOR_NR << 8;
    cd -> cd_timer.data = (long) cd;
    cd -> cd_timer.function = lms_timer;
    cd -> cd_adap_sect = 11;

    /*
     * Before we request an irq and try to send commands to the drive, we
     * make a quick check, that there is something connected to the I/O - Port:
     */
    if(lms_state(cd) == 0xffff) {
	printk("Initialization of LMS CD-ROM failed: port %x empty\n",
		cd -> cd_port);
	return MODULE_ERROR(mem_start);
    }

    cd0 = cd;
    if (irqaction(cd -> cd_irq, &lms_sigaction))
    {
	printk("Unable to get IRQ%d for Laser Magnet Storage International CD-ROM\n", LMSCD_INTR_NR);
	cd0 = 0;
	return MODULE_ERROR(mem_start);
    }

    debug_1(("resetting CM250 CD-ROM adapter: "));
    cd -> cd_abort   = 0;
    cd -> cd_success = 0;
    lms_adinit(cd);

    /* loop until we see something happen ... */
    while(cd -> cd_retry_cnt > 0)
	;

    if(cd -> cd_retry_cnt == 0) {
	printk("failed\n");
	free_irq(cd -> cd_irq);
	cd0 = 0;
	return MODULE_ERROR(mem_start);
    }

    printk("successfull\n");

    debug_1(("resetting CD-ROM drive: "));
    lms_reset_drive(cd);

    /* loop until we see something happen ... */
    while(cd -> cd_identify >= 0)
	;

    if(cd -> cd_type == 0) {
	debug_1(("failed\n"));
	free_irq(cd -> cd_irq);
	cd0 = 0;
	return MODULE_ERROR(mem_start);
    }

    debug_1(("successfull\n"));

    if (register_blkdev(MAJOR_NR, "lmscd", &lms_fops) != 0)
    {
	debug_0(("LMSCD: Cannot register for major %d!\n", MAJOR_NR));
	return MODULE_ERROR(mem_start);
    }

    blk_dev[MAJOR_NR].request_fn = DEVICE_REQUEST;
    read_ahead[MAJOR_NR] = 16;

    /*
     * Now copy the cd data structure to permament memory.  At this point, no
     * timer may be running, or bad things will happen.
     */
    if(del_timer(&cd -> cd_timer))
	debug_1(("LMS-Timer was running, though it should be clear!\n"));

    memcpy((void *) mem_start, (void *) cd, sizeof(*cd));
    (cd0 = (struct lmscd *) mem_start) -> cd_timer.data = mem_start;
    mem_start += sizeof(*cd);
    cd_max_minor = 1;

    debug_0(("LMS %s CD-ROM ready\n", drv_type[cd -> cd_type]));

    return MODULE_OK(mem_start);
}
