/*
 *  linux/drivers/block/double.c
 *
 *  Adapted from loop.c (Copyright 1993 by Theodore Ts'o)
 *  by Jean-Marc Verbavatz 12 Jan 1994 <jmv@receptor.mgh.harvard.edu>
 *
 *  Copyright 1994 by Jean-Marc Verbavatz. Redistribution of this file
 *  is permitted under the GNU Public License.
 * 
 */

#include <linux/config.h>
#include <linux/stat.h>
#include <linux/malloc.h>
#include <linux/string.h>
#include <linux/errno.h>
#include <asm/segment.h>
#include <linux/double.h>
#include <linux/module.h>
#include "blk.h"

char dble_version[] = UTS_RELEASE;

struct v_buffer  {
	long block;
	u_char db_dev;
	char data[DB_MAXSZ];
} **db_vb;

struct dble_device dble_dev[MAX_DBLE];
static int dble_sizes[MAX_DBLE];

/* Synchronize requests (ioctl vs kernel ...) */
static int in_request = 0;
static struct wait_queue *wait_on_request;

extern char *db_buffer;
extern long *db_mapbuf;

extern void db_write_buffers(void);
extern int iLZW(void *);
extern int ilzrw2(void *);
extern int ilzrw3a(void *);
extern int ipred(void *);
extern int dbtools_debug(void);

int db_debug(void)
{
	int i;

	if((i=dbtools_debug())) {
		printk("TOOLS memory corruption %d\n", i);
		return 1;
	}
	return 0;
}

/* Process synchronization; at any given time, only one process can
 * use the driver
 */
void lock_db(void)
{
	cli();
	while(in_request) sleep_on(&wait_on_request);
	in_request = 1;
	sti();
}

void unlock_db(void)
{
	cli();
	in_request = 0;
	wake_up(&wait_on_request);
	sti();
}

/* Get buffer for request */
struct v_buffer *get_v_buffer(u_char device, long block)
{
	int i;
	struct v_buffer *v_buffer;

	for(i = 0; i < DB_NR_VB; i++)
		if(db_vb[i]->db_dev == device && db_vb[i]->block == block) break;
	if(i < DB_NR_VB) dble_stat.ohit++;
	else {
		dble_stat.onohit++;
		db_vb[--i]->block = -1;
		db_vb[i]->db_dev = device;
	}
	v_buffer = db_vb[i];
	for(; i > 0; i--)
		db_vb[i] = db_vb[i-1];
	db_vb[0] = v_buffer;
	return v_buffer;
}

/* DouBle driver request loop */
static void do_db_request(void)
{
	int     offset, len, size;
	long    block, total_len;
	char    *dest_addr;
	struct dble_device *db;
	struct buffer_head *bh;
	struct v_buffer *v_buffer;

/* Synchronize with ioctl/other calls */
	lock_db();
repeat:
/* INIT_REQUEST */
	if(!CURRENT) { /* No requests pending */
		for(len = 0; len < MAX_DBLE; len++) {
			db = &dble_dev[len];
			if(db->db_inode) db_flush(db);
		}
		unlock_db();
		return;
	}
	if(MAJOR(CURRENT->dev) != MAJOR_NR)
		panic(DEVICE_NAME ": request list destroyed.");
	if(CURRENT->bh && !CURRENT->bh->b_lock)
		panic(DEVICE_NAME ": block not locked.");
/* Additional checking */
	if ((MINOR(CURRENT->dev)&0x7f) >= MAX_DBLE)
		goto error_out;
	db = &dble_dev[MINOR(CURRENT->dev)&0x7f];
	if (!db->db_inode)
		goto error_out;
	if ((CURRENT->cmd != WRITE) && (CURRENT->cmd != READ)) {
		printk("DouBle: unknown command (%d)", CURRENT->cmd);
		goto error_out;
	}
	if(db->debug >= 3) printk("INIT REQ\n");

/* Compute device block/offset */
	if (db->osize < 512) { /* Somewhat unlikely, but ... */
		block = CURRENT->sector * (512/db->osize);
		offset = 0;
	} else {
		block = CURRENT->sector / (db->osize>>9);
		offset = (CURRENT->sector % (db->osize>>9))<<9;
	}
	dest_addr = CURRENT->buffer;
	total_len = CURRENT->nr_sectors<<9;
	if(CURRENT->bh) /* Are we working block by block ? */
		len = CURRENT->bh->b_size;
	else len = total_len; /* Only one buffer ... */

/* Transfer data */
	while (total_len > 0) {
		v_buffer = get_v_buffer(db->db_number, block);
		if (v_buffer->block != block &&
		(CURRENT->cmd == READ || offset || total_len < db->osize)) {
			if(!read_oblock(db, block, v_buffer->data)) {
				printk("DouBle: read error ");
				goto error_out;
			}
		}
		v_buffer->block = -1;	/* Not up to date */
	/* Transfer data within a single cluster */
		do {
			size = db->osize - offset;
			if (size > len) size = len;
			if(CURRENT->cmd == READ) {
				dble_stat.rvirtual++;
				memcpy(dest_addr, v_buffer->data+offset, size);
			} else {
				dble_stat.wvirtual++;
				memcpy(v_buffer->data+offset, dest_addr, size);
			}
			total_len -= size;
			offset += size;
		/* Check end of cluster */
			if(offset >= db->osize || total_len <=0) break;
		/* Next buffer if any */
			if(!(bh = CURRENT->bh)) break;
			CURRENT->bh = bh->b_reqnext;
			bh->b_reqnext = NULL;
			bh->b_uptodate = 1;
			unlock_buffer(bh);
			bh = CURRENT->bh;
			if(bh == NULL) break;
			dest_addr = CURRENT->buffer = bh->b_data;
			len = CURRENT->bh->b_size;
		} while(total_len > 0);

		if(CURRENT->cmd == WRITE &&
		   !write_oblock(db, block, v_buffer->data)) {
			printk("DouBle: write error\n");
			goto error_out;
		}
		v_buffer->block = block;
/* Get ready for next cluster */
		len -= size;
		dest_addr += size;
		offset = 0;
		block++;
	}
	/* Write buffers and give a chance to other processes */
	db_write_buffers();
	db_debug();
	if(db->debug >= 3) printk("REQ DONE\n");
	end_request(1);
	goto repeat;
error_out:
	db_debug();
	end_request(0);
	goto repeat;
}

/* Update block map in regular file used as device
 * The first isize/sizeof(long) block are loaded in memory by rheader
 * block map starts on iblock # 1.
 */
int write_bmap(struct dble_device *db)
{
	register int i, j, n;
	long block, save;

	n = db->isize/sizeof(long);
	save = dble_stat.wphysical;
	for(i = 1; i < db->addr_bitmap; i++) {
		memset(db->BAT_block, 0, db->isize);
		for(j = 0; j < n;)
			if(!(block = bmap(db->db_inode, i*n+j)))
				break;  
			else ((long *)db->BAT_block)[j++] = block;
		db_rw_iblock(db, WRITE, i, db->BAT_block);
	}
	dble_stat.wphysical = save;
	return 0;
}

/* Mount device on DouBle device */
static int dble_set_fd(struct dble_device *db, struct inode * inode)
{
	if (db->db_inode) return -EBUSY;
	if (!inode) {
		printk("dble_set_fd: NULL inode?!?\n");
		return -EINVAL;
	}
	db->db_flags = 0;
	if (S_ISREG(inode->i_mode)) {
		db->db_device = inode->i_dev;
		db->db_flags |= DB_BMAP;
	} else if (S_ISBLK(inode->i_mode)) 
			db->db_device = inode->i_rdev;
	else return -EINVAL;
	if(inode->i_blksize > BLOCK_SIZE) return -EINVAL;
	db->db_inode = inode;
	if(!db_rheader(db)) {
		db->db_inode = 0;
		printk("DouBle: error reading header\n");
		return -EINVAL;
	}
	if(db->db_flags & DB_BMAP)
		write_bmap(db);
	dble_sizes[db->db_number] = db->oblocks*db->osize/1024;
	printk("DouBle Alpha Version 0.2\n");
	printk("double%d: %ld clusters of %d bytes (%dK).\n", db->db_number,
	db->oblocks, db->osize, dble_sizes[db->db_number]);
	return 0;
}

/* release device from DouBle device */
static int dble_clr_fd(struct dble_device *db)
{
	int i;
	struct inode *ino;

	if (db->db_refcnt > 0) return -EBUSY;
	ino = db->db_inode;
	db_flush(db);
	iput(db->db_inode);
	db->db_inode = 0;
	dble_sizes[db->db_number] = 0;
/* invalidate internal buffers */
	for(i=0; i<DB_NR_BH; i++)
		if(db_bh[i]->b_dev == db->db_device)
			db_bh[i]->b_dev = -1;
	for(i=0; i<DB_NR_VB; i++)
		if(db_vb[i]->db_dev == db->db_number)
			db_vb[i]->db_dev = -1;
	invalidate_buffers(MKDEV(MAJOR_NR, db->db_number));
	invalidate_buffers(MKDEV(MAJOR_NR, 0x80 | db->db_number));
	db->db_device = 0;
	kfree_s(db->bitmap, db->isize);
	kfree_s(db->BAT_block, db->isize);
	if(db->db_flags & DB_BMAP) kfree_s(db->bmap0, db->isize);
	return 0;
}

/* Function called by sys_mount via get_super, when using mount
 * with DBLE_FS support instead of the ioctl call
 */ 
void *dble_open(dev_t dev, struct inode * inode)
{
	struct dble_device *db;
	int err = 0;

	if (MAJOR(dev) != MAJOR_NR) {
		printk("db_ioctl: pseudo-major != %d\n", MAJOR_NR);
		return NULL;
	}
	dev = MINOR(dev)&0x7f;
	if (dev >= MAX_DBLE) return NULL;
	db = &dble_dev[dev];
	lock_db();
	err = dble_set_fd(db, inode);
	unlock_db();
	if(err) return NULL;
	db->db_mounted = 1;
	return (void *) db;
}

/* Called by sys_umount via put_super, when using CONFIG_DBLE_FS */
int dble_close(struct dble_device *db)
{
	int err;

	lock_db();
	if(!db->db_inode) return -ENXIO;
	if(!db->db_mounted) return -EBUSY;
	err = dble_clr_fd(db);
	unlock_db();
	return err;
}
	

/* Change parameters (currently only default compression method) */
static int dble_set_status(struct dble_device *db, struct dble_info *arg)
{
	struct dble_info info;

	if (!db->db_inode) return -ENXIO;
	if (!arg) return -EINVAL;
	memcpy_fromfs(&info, arg, sizeof(info));
	db->code = info.code;
	return 0;
}

static int dble_get_status(struct dble_device *db, struct dble_info *arg)
{
	struct dble_info info;
	
	if (!db->db_inode) return -ENXIO;
	if (!arg) return -EINVAL;
	memset(&info, 0, sizeof(info));
	info.oblocks = db->oblocks;
	info.iblocks = db->iblocks;
	info.osize = db->osize;
	info.isize = db->isize;
	info.code = db->code;
	info.db_device = db->db_inode->i_dev;
	info.db_inode = db->db_inode->i_ino;
	info.db_rdevice = db->db_device;
	info.db_flags = db->db_flags;
	memcpy_tofs((struct dble_info *) arg, &info, sizeof(info));
	return 0;
}

static int db_ioctl(struct inode * inode, struct file * file,
	unsigned int cmd, unsigned long arg)
{
	struct dble_device *db;
	int dev, err = 0;

	if (!inode) return -EINVAL;
	if (!S_ISBLK(inode->i_mode)) return -ENXIO;
	if (MAJOR(inode->i_rdev) != MAJOR_NR) {
		printk("db_ioctl: pseudo-major != %d\n", MAJOR_NR);
		return -ENODEV;
	}
	dev = MINOR(inode->i_rdev)&0x7f;
	if (dev >= MAX_DBLE) return -ENODEV;
	db = &dble_dev[dev];
	lock_db();
	switch (cmd) {
	case DBLE_SET_FD:
		if (arg >= NR_OPEN || !current->filp[arg]) return -EBADF;
		err = dble_set_fd(db, current->filp[arg]->f_inode);
		if(! err && db->db_inode) {
			db->db_inode->i_count++;
			db->db_mounted = 0;
		}
		break;
	case DBLE_CLR_FD:
		db->db_refcnt--;	/* Was opened for ioctl */
		if (!db->db_inode)
			err = -ENXIO;
	/* Was it mounted or dbmounted ? */
		else if(db->db_mounted)
			err = -EBUSY;	/* Mounted */
		else	/* dbmounted; we can go ahead */
			err = dble_clr_fd(db);
		db->db_refcnt++;
		break;
	case DBLE_SET_STATUS:
		err = dble_set_status(db, (struct dble_info *) arg);
		break;
	case DBLE_GET_STATUS:
		err = dble_get_status(db, (struct dble_info *) arg);
		break;
	case DBLE_FLUSH:
		err = db_flush(db);
		break;
	case DBLE_STAT:
		memcpy_tofs((struct dble_stat *) arg, &dble_stat, sizeof(dble_stat));
		break;
	case BLKGETSIZE:   /* Return device size */
		if (!db->db_inode) {
			err = -ENXIO;
			break;
		}
		if (!arg) {
			err = -EINVAL;
			break;
		}
		err = verify_area(VERIFY_WRITE, (long *) arg, sizeof(long));
		if (err) break;
		put_fs_long(dble_sizes[db->db_number] << 1, (long *) arg);
		break;
	case DBLE_DEBUG: /* Set debug parameter */
		db->debug = arg;
		break;
	default:
			err = -EINVAL;
	}
	unlock_db();
	return err;
}

static int db_open(struct inode *inode, struct file *file)
{
	struct dble_device *db;
	int     dev;

	if (!inode) return -EINVAL;
	if (MAJOR(inode->i_rdev) != MAJOR_NR) {
		printk("db_open: pseudo-major != %d\n", MAJOR_NR);
		return -ENODEV;
	}
	dev = MINOR(inode->i_rdev)&0x7f;
	if (dev >= MAX_DBLE) return -ENODEV;
	db = &dble_dev[dev];
	if(db->db_refcnt++ > 0)
		printk("Warning: double%d is already opened\n", db->db_number);
	return 0;
}

static void db_release(struct inode *inode, struct file *file)
{
	struct dble_device *db;
	int     dev;

	if (!inode) return;
	if (MAJOR(inode->i_rdev) != MAJOR_NR) {
		printk("db_release: pseudo-major != %d\n", MAJOR_NR);
		return;
	}
	dev = MINOR(inode->i_rdev)&0x7f;
	if (dev >= MAX_DBLE) return;
	db = &dble_dev[dev];
/* Synchronize with ioctl/request calls */
	lock_db();
	if(db->db_inode) db_flush(db);
	unlock_db();
	if (db->db_refcnt-- <= 0) {
		printk("db_release: refcount < 0\n");
		db->db_refcnt = 0;
	}
	return;
}

int db_fsync(struct inode *inode, struct file *file)
{
	struct dble_device *db;
	int     dev;

	printk("DBfsync called ... Why ???\n");
	if (!inode) return -EINVAL;
	if (MAJOR(inode->i_rdev) != MAJOR_NR) {
		printk("db_fsync: pseudo-major != %d\n", MAJOR_NR);
		return -ENODEV;
	}
	dev = MINOR(inode->i_rdev)&0x7f;
	if (dev >= MAX_DBLE) return -ENODEV;
	db = &dble_dev[dev];
/* Synchronize with ioctl/request calls */
	lock_db();
	if(db->db_inode) db_flush(db);
	unlock_db();
	return 0;
}

static struct file_operations db_fops = {
	NULL,                   /* lseek - default */
	block_read,             /* read - general block-dev read */
	block_write,            /* write - general block-dev write */
	NULL,                   /* readdir - bad */
	NULL,                   /* select */
	db_ioctl,               /* ioctl */
	NULL,                   /* mmap */
	db_open,                /* open */
	db_release,             /* release */
	db_fsync                /* fsync: when is it called ? */
};

struct super_block *dble_read_super(struct super_block *s, 
	struct inode *inode, int silent);

static struct file_system_type filesys = { "dble", NULL, dble_read_super, 1 };

/*
 * Returns mem_start increased by the amount of memory which needs to be
 *  reserved.
 */
unsigned long dble_init(unsigned long mem_start, unsigned long mem_end)
{
	int     i, l;
	long 	mem0;
	void * addr;

	mem0 = mem_start;

#ifdef CONFIG_DBLE_FS
	if (register_filesystem( &filesys )<0)
		return(-EINVAL);
#endif

	if (register_blkdev(MAJOR_NR, "dble", &db_fops)) {
		printk("Unable to get major %d for DouBle device\n",MAJOR_NR);
		return MODULE_ERROR(mem_start);
	}
	blk_dev[MAJOR_NR].request_fn = DEVICE_REQUEST;
	for (i=0; i < MAX_DBLE; i++) {
		memset(&dble_dev[i], 0, sizeof(struct dble_device));
		dble_dev[i].db_number = i;
	}
	dble_stat.ihit = dble_stat.inohit =
	dble_stat.ihit = dble_stat.inohit = 
	dble_stat.roblock = dble_stat.woblock =
	dble_stat.rvirtual = dble_stat.rphysical =
	dble_stat.wvirtual = dble_stat.wphysical = 0;
	memset(&dble_sizes, 0, sizeof(dble_sizes));
	blk_size[MAJOR_NR] = dble_sizes;
	db_bh = (struct buffer_head **) dmalloc(DB_NR_BH*sizeof(struct buffer_head *), &mem_start);
	for(i=0; i<DB_NR_BH; i++) {
		db_bh[i] = (struct buffer_head *) dmalloc(sizeof(struct buffer_head), &mem_start);
		db_bh[i]->b_data = (char *) dmalloc(BLOCK_SIZE, &mem_start);
		db_bh[i]->b_dirt = db_bh[i]->b_count = 0;
		db_bh[i]->b_dev = -1;
	}
	db_vb = (struct v_buffer **) mem_start;
	mem_start += DB_NR_VB*sizeof(struct v_buffer *);
	for(i=0; i<DB_NR_VB; i++) {
		db_vb[i] = (struct v_buffer *) mem_start;
		db_vb[i]->db_dev = -1;
		mem_start += sizeof(struct v_buffer);
	}
	db_buffer = (char *) dmalloc( DB_MAXSZ+4, &mem_start);
	db_mapbuf = (long *) dmalloc( BLOCK_SIZE, &mem_start);
	l = iLZW((void *)mem_start);
	i = ilzrw2((void *)mem_start);
	if(i > l) l = i;
	i = ilzrw3a((void *)mem_start);
	if(i > l) l = i;
	if (l>0)
	{	addr = (void*)dmalloc(l,&mem_start);
		iLZW(addr);
		ilzrw2(addr);
		ilzrw3a(addr);
	}
	i = ipred((void *)mem_start);
	addr = (void*)dmalloc(i, &mem_start);
	ipred(addr);
	printk("DouBle: %d devices, %d buffers %dK.\n", MAX_DBLE, DB_NR_BH+DB_NR_VB, (int)((mem_start-mem0)/1024));
	return MODULE_OK(mem_start);
}

void dble_cleanup(void)
{
	unregister_blkdev(MAJOR_NR, "dble" );
#ifdef CONFIG_DBLE_FS
	unregister_filesystem( "dble" );
#endif
}
