/*
 * linux/drivers/block/dbsuper.c
 *
 * Support for DOUBLE_FS; 
 * Copyright (c) 1994 Jean-Marc Verbavatz
 */

#include <linux/config.h>
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/double_fs.h>
#include <linux/double.h>
#include <linux/locks.h>
#include <asm/system.h>
#include <asm/segment.h>

#ifdef CONFIG_DBLE_FS
static void dble_statfs(struct super_block *, struct statfs *);
extern void *dble_open(dev_t, struct inode *);
extern int dble_close(void *);
extern int get_bit(struct dble_device *, long);
extern void lock_db(void);
extern void unlock_db(void);

extern struct dble_device dble_dev[];

void dble_put_super(struct super_block *s)
{
	if(dble_close(s->u.dble_sb.db))
		return;
	lock_super(s);
	s->s_dev = 0;
	unlock_super(s);
}

static struct super_operations dble_sops = {
	NULL,
	NULL,
	NULL,
	NULL,
	dble_put_super,
	NULL,
	dble_statfs,
	NULL
};

struct super_block *dble_read_super(struct super_block *s,
				struct inode *inode, int silent)
{
	s->u.dble_sb.db = dble_open(s->s_dev, inode);
	if(!s->u.dble_sb.db) {
		s->s_dev = 0;
		printk("DouBle device mount failed.\n");
		return NULL;
	}
	lock_super(s);
	s->s_blocksize = 1024;
	s->s_blocksize_bits = 10;
	s->s_magic = DBLE_SUPER_MAGIC;
	s->s_op = &dble_sops;
	unlock_super(s);
	return s;
}

static void dble_statfs(struct super_block *s, struct statfs *buf)
{
	struct dble_device *db;
	long free, total;

	db = s->u.dble_sb.db;
	put_fs_long((long)DBLE_SUPER_MAGIC, &buf->f_type);
	put_fs_long((long)db->isize, &buf->f_bsize);
	lock_db();
	for(total = free = 0; total < db->iblocks; total++)
		if(get_bit(db, total)) free++;
	unlock_db();
	put_fs_long(total, &buf->f_blocks);	/* Total blocks */
	put_fs_long(free, &buf->f_bfree);	/* Total free blocks */
	put_fs_long(free, &buf->f_bavail);	/* Total available blocks */
	put_fs_long(1, &buf->f_files);
	put_fs_long(0, &buf->f_ffree);
	put_fs_long(0, &buf->f_namelen);
}

int dble_may_umount(dev_t dev)
{
	if(MAJOR(dev) != MAJOR_NR) /* I don't know what this is */
		return 0;
	dev = MINOR(dev)&0x7f;
	if(dev >= MAX_DBLE)
		return 0;
	if(dble_dev[dev].db_refcnt > 0)
		return 0;
	return 1;
}

#endif
