/* Framebuffer Graphics Libary for Linux, Copyright 1993 Harm Hanemaayer */
/* grlib.c	Main module */


#include <stdlib.h>
#include <vga.h>
#include "inlstring.h"		/* include inline string operations */

#include "vgagl.h"
#include "def.h"
#include "driver.h"



/* Global variables */

GraphicsContext currentcontext;


/* Framebuffer function pointers */

static framebufferfunctions ff8 = {
	driver8_setpixel,
	driver8_getpixel,
	driver8_hline,
	driver8_fillbox,
	driver8_putbox,
	driver8_getbox,
	driver8_putboxmask,
	driver8_putboxpart,
	driver8_getboxpart,
	driver8_copybox
};

static framebufferfunctions ff16 = {
	driver16_setpixel,
	driver16_getpixel,
	driver16_hline,
	driver16_fillbox,
	driver16_putbox,
	driver16_getbox,
	driver16_putboxmask,
	driver16_putboxpart,
	driver16_getboxpart,
	driver16_copybox
};

static framebufferfunctions ff24 = {
	driver24_setpixel,
	driver24_getpixel,
	driver24_hline,
	driver24_fillbox,
	driver24_putbox,
	driver24_getbox,
	driver24_putboxmask,
	driver24_putboxpart,
	driver24_getboxpart,
	driver24_copybox
};

static framebufferfunctions ff8paged = {
	driver8p_setpixel,
	driver8p_getpixel,
	driver8p_hline,
	driver8p_fillbox,
	driver8p_putbox,
	driver8p_getbox,
	driver8p_putboxmask,
	driver8p_putboxpart,
	driver8p_getboxpart,
	driver8p_copybox
};

static framebufferfunctions ff16paged = {
	driver16p_setpixel,
	driver16p_getpixel,
	driver16p_hline,
	driver16p_fillbox,
	driver16p_putbox,
	driver16p_getbox,
	driver16p_putboxmask,
	driver16p_putboxpart,
	driver16p_getboxpart,
	driver16p_copybox
};

static framebufferfunctions ff24paged = {
	driver24p_setpixel,
	driver24p_getpixel,
	driver24p_hline,
	driver24p_fillbox,
	driver24p_putbox,
	driver24p_getbox,
	driver24p_putboxmask,
	driver24p_putboxpart,
	driver24p_getboxpart,
	driver24p_copybox
};


/* Initialization and graphics contexts */

#define SCREENSIZE(gc) (gc.bytewidth * gc.height * gc.bytesperpixel)

static int colorbits( int c ) {
	switch (c) {
	case 256 : return 8;
	case 32768 : return 15;
	case 65536 : return 16;
	case 256 * 65536 : return 24;
	}
}

int gl_setcontextvga( int m ) {
	framebufferfunctions *ff;
	vga_modeinfo *modeinfo;
	if (!vga_hasmode(m))
		return -1;
	modeinfo = vga_getmodeinfo(m);
	/* Set graphics context */
	WIDTH = modeinfo->width;
	HEIGHT = modeinfo->height;
	BYTESPERPIXEL = modeinfo->bytesperpixel;
	COLORS = modeinfo->colors;
	BITSPERPIXEL = colorbits(COLORS);
	BYTEWIDTH = modeinfo->linewidth;
	VBUF = graph_mem;
	__clip = 0;
	ff = &(currentcontext.ff);
	if (m == G320x200x256) {
		MODETYPE = CONTEXT_LINEAR;
		switch (BYTESPERPIXEL) {
			case 1 : currentcontext.ff = ff8; break;
			case 2 : currentcontext.ff = ff16; break;
			case 3 : currentcontext.ff = ff24; break;
		}
	}
	else {
		MODETYPE = CONTEXT_PAGED;
		switch (BYTESPERPIXEL) {
			case 1 : currentcontext.ff = ff8paged; break;
			case 2 : currentcontext.ff = ff16paged; break;
			case 3 : currentcontext.ff = ff24paged; break;
		}
		if (BYTESPERPIXEL == 1) {
			if (modeinfo->haveblit & HAVE_FILLBLIT)
				currentcontext.ff.driver_fillbox_func =
					driver8a_fillbox;
			if (modeinfo->haveblit & HAVE_BITBLIT)
				currentcontext.ff.driver_copybox_func =
					driver8a_copybox;
		}
	}
}

int gl_setcontextvgavirtual( int m ) {
	vga_modeinfo *modeinfo;
	if (!vga_hasmode(m))
		return -1;
	modeinfo = vga_getmodeinfo(m);
	/* Set graphics context */
	WIDTH = modeinfo->width;
	HEIGHT = modeinfo->height;
	BYTESPERPIXEL = modeinfo->bytesperpixel;
	COLORS = modeinfo->colors;
	BITSPERPIXEL = colorbits(COLORS);
	BYTEWIDTH = modeinfo->linewidth;
	VBUF = malloc(SCREENSIZE(currentcontext));
	MODETYPE = CONTEXT_VIRTUAL;
	__clip = 0;
	switch (BYTESPERPIXEL) {
		case 1 : currentcontext.ff = ff8; break;
		case 2 : currentcontext.ff = ff16; break;
		case 3 : currentcontext.ff = ff24; break;
	}
}

void gl_setcontextvirtual( int w, int h, int bpp, int bitspp, void *v ) {
	WIDTH = w;
	HEIGHT = h;
	BYTESPERPIXEL = bpp;
	BITSPERPIXEL = bitspp;
	COLORS = 1 << bitspp;
	BYTEWIDTH = WIDTH * BYTESPERPIXEL;
	VBUF = v;
	MODETYPE = CONTEXT_VIRTUAL;
	switch (BYTESPERPIXEL) {
		case 1 : currentcontext.ff = ff8; break;
		case 2 : currentcontext.ff = ff16; break;
		case 3 : currentcontext.ff = ff24; break;
	}
	__clip = 0;
}

void gl_setcontext( GraphicsContext *gc ) {
	currentcontext = *gc;
}

void gl_getcontext( GraphicsContext *gc ) {
	*gc = currentcontext;
}

void gl_freecontext( GraphicsContext *gc ) {
	if (gc->modetype == CONTEXT_VIRTUAL)
		free(gc->vbuf);
}


/* Clipping */

void gl_setclippingwindow( int x1, int y1, int x2, int y2 ) {
	__clip = 1;
	__clipx1 = x1;
	__clipy1 = y1;
	__clipx2 = x2;
	__clipy2 = y2;
}

void gl_enableclipping() {
	__clip = 1;
	__clipx1 = 0;
	__clipy1 = 0;
	__clipx2 = WIDTH - 1;
	__clipy2 = HEIGHT - 1;
}

void gl_disableclipping() {
	__clip = 0;
}


/* Primitive functions */

void gl_setpixel( int x, int y, int c ) {
	if (__clip && outside(x, y))
		return;
	setpixel(x, y, c);
}

int gl_getpixel( int x, int y ) {
	if (__clip && outside(x, y))
		return -1;
	return getpixel(x, y);
}

void gl_hline( int x1, int y, int x2, int c ) {
	if (__clip) {
		if (y_outside(y))
			return;
		clipxleft(x1);
		clipxright(x2);
		if (x1 > x2)
			return;
	}
	hline(x1, y, x2, c);
}

#define ADJUSTBITMAPBOX() \
	nw = w; nh = h; nx = x; ny = y;				\
	if (nx + nw < __clipx1 || nx > __clipx2)		\
		return;						\
	if (ny + nh < __clipy1 || ny > __clipy2)		\
		return;						\
	if (nx < __clipx1) {		/* left adjust */	\
		nw += nx - __clipx1;				\
		nx = __clipx1;					\
	}							\
	if (ny < __clipy1) {		/* top adjust */	\
		nh += ny - __clipy1;				\
		ny = __clipy1;					\
	}							\
	if (nx + nw > __clipx2)		/* right adjust */	\
		nw = __clipx2 - nx + 1;				\
	if (ny + nh > __clipy2)		/* bottom adjust */	\
		nh = __clipy2 - ny + 1;				\

void gl_fillbox( int x, int y, int w, int h, int c ) {
	if (w <= 0 || h <= 0)
		return;
	if (__clip) {
		if (x + w < __clipx1 || x > __clipx2)		
			return;						
		if (y + h < __clipy1 || y > __clipy2)		
			return;
		if (x < __clipx1) {
			w -= __clipx1 - x;
			x = __clipx1;
		}
		if (y < __clipy1) {
			h -= __clipy1 - y;
			y = __clipy1;
		}
		if (x + w > __clipx2 + 1)
			w = __clipx2 - x + 1;
		if (y + h > __clipy2 + 1)
			h = __clipy2 - y + 1;
	}
	fillbox(x, y, w, h, c);
}

void gl_putboxpart( int x, int y, int w, int h, int ow, int oh, void *b,
int ox, int oy ) {
	putboxpart(x, y, w, h, ow, oh, b, ox, oy);
}

void gl_putbox( int x, int y, int w, int h, void *b ) {
	uchar *bp = b;
	if (w <= 0 || h <= 0)
		return;
	if (__clip) {
		int nx, ny, nw, nh;
		ADJUSTBITMAPBOX();
		if (nw != w || nh != h) {
			putboxpart(nx, ny, nw, nh, w, h, bp, nx - x, ny - y);
			return;
		}
	}
	putbox(x, y, w, h, bp, w);
}

static void emulate_putboxmask( int x, int y, int w, int h, void *b ) {
	void *box;
	GraphicsContext gc;
	box = alloca(w * h * BYTESPERPIXEL);
	gl_getbox(x, y, w, h, box);	/* does clipping */

	gl_getcontext(&gc);		/* save context */

	/* create context that is only the box */
	gl_setcontextvirtual(w, h, BYTESPERPIXEL, BITSPERPIXEL, box);
	gl_putboxmask(0, 0, w, h, b);

	gl_setcontext(&gc);		/* restore context */
	gl_putbox(x, y, w, h, box);
}

void gl_putboxmask( int x, int y, int w, int h, void *b ) {
	if (w <= 0 || h <= 0)
		return;
	if (__clip) {
		if (x + w < __clipx1 || x > __clipx2)
			return;
		if (y + h < __clipy1 || y > __clipy2)		
			return;
		if (x < __clipx1 || y < __clipy1
		|| x + w > __clipx2 + 1 || y + h > __clipy2 + 1) {
			/* clipping is not directly implemented */
			emulate_putboxmask(x, y, w, h, b);
			return;
		}
	}
	if (MODETYPE == CONTEXT_PAGED)
		/* paged primitive is not implemented */
		emulate_putboxmask(x, y, w, h, b);
	else
		putboxmask(x, y, w, h, b);
}

void gl_getbox( int x, int y, int w, int h, void *b ) {
	if (__clip) {
		int nx, ny, nw, nh;
		ADJUSTBITMAPBOX();
		if (nw != w || nh != h) {
			getboxpart(nx, ny, nw, nh, w, h, b, nx - x, ny - y);
			return;
		}
	}
	getbox(x, y, w, h, b, w);
}


/* Miscellaneous functions */

void gl_clearscreen( int c ) {
	gl_fillbox(0, 0, WIDTH, HEIGHT, c);
}

void gl_setpixelrgb( int x, int y, int r, int g, int b ) {
/* Color components range from 0 to 255 */
	if (__clip && outside(x, y))
		return;
	switch (BITSPERPIXEL) {
	case 8 :
		/* assumes RGB palette at index 0-255 */
		/* bits 0-2 = blue (3 bits) */
		/*	3-5 = green (3 bits) */
		/*	6-7 = red (2 bits) */
		setpixel(x, y, (r & 0xc0) + ((g & 0xe0) >> 2) + (b >> 5));
		break;
	case 15 :
		setpixel(x, y, ((r & 0xf8) << 7) + ((g & 0xf8) << 2) + 
			(b >> 3));
		break;
	case 16 :
		setpixel(x, y, ((r & 0xf8) << 8) + ((g & 0xfc) << 3) +
			(b >> 3));
		break;
	case 24 :
		setpixel(x, y, (r << 16) + (g << 8) + b);
		break;
	}
}

void gl_getpixelrgb( int x, int y, int *r, int *g, int *b ) {
	int c;
	if (__clip && outside(x, y)) {
		*r = *g = *b = -1;
		return;
	}
	c = getpixel(x, y);
	switch (BITSPERPIXEL) {
	case 8 :
		*b = (c & (1+2+4)) << 5;	/* bits 0-2 */
		*g = (c & (8+16+32)) << 2;	/* bits 3-5 */
		*r = (c & (64 + 128));		/* bits 6-7 */
		break;
	case 15 :
		*b = (c & (1+2+4+8+16)) << 3;
		*g = (c & (32+64+128+256+512)) >> 2;
		*r = (c & (1024+2048+4096+8192+16384)) >> 7;
		break;
	case 16 :
		*b = (c & (1+2+4+8+16)) << 3;
		*g = (c & (32+64+128+256+512+1024)) >> 3;
		*r = (c & (2048+4096+8192+16384+32768)) >> 8;
		break;
	case 24 :
		*b = c & 0xff;
		*g = (c & 0xff00) >> 8;
		*r = c >> 16;
		break;
	}
}

void gl_setdisplaystart( int x, int y ) {
	vga_setdisplaystart(y * BYTEWIDTH + x * BYTESPERPIXEL);
}


/* Screen copying */

void gl_copyscreen( GraphicsContext *gc ) {
	int size = SCREENSIZE(currentcontext);
	int offset = 0;
	void *svp, *dvp;
	while (offset < size) {
		int schunk, dchunk;
		int count;
		schunk = driver_setread(&currentcontext, offset, &svp);
		dchunk = driver_setwrite(gc, offset, &dvp);
		count = min(min(schunk, dchunk), size - offset);
		memcpy(dvp, svp, count);
		offset += count;
	}
}
