/*
 * ps.c		- show process status
 *
 * Copyright (c) 1992 Branko Lankester
 *
 * Snarfed and HEAVILY modified for the YAPPS (yet another /proc ps)
 * by Michael K. Johnson, johnsonm@stolaf.edu.  What is used is what
 * is required to have a common interface.
 */

#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <pwd.h>
#include <linux/sched.h>
#include <linux/tty.h>
#include "ps.h"


#define	PS_D	0	/* default format (short) */
#define	PS_L	1	/* long format */
#define	PS_U	2	/* user format */
#define	PS_J	3	/* jobs format */
#define	PS_S	4	/* signal format */
#define	PS_V	5	/* vm format */
#define	PS_M	6	/* mem. stuff */
#define	PS_X	7	/* regs etc., for testing */

char *hdrs[] = {
"  PID TT STAT  TIME COMMAND",
" F   UID   PID  PPID PRI NI SIZE  RSS WCHAN      STAT TT   TIME COMMAND",
"USER        PID %CPU %MEM SIZE  RSS TT STAT START   TIME COMMAND",
" PPID   PID  PGID   SID TT TPGID  STAT   UID   TIME COMMAND",
"  UID   PID SIGNAL   BLOCKED  IGNORED  CATCHED  STAT TT   TIME COMMAND",
"  PID TT STAT  TIME  PAGEIN TSIZ DSIZ  RSS   LIM %MEM COMMAND",
"  PID TT MAJFLT MINFLT  TRS  DRS SIZE SWAP  RSS SHRD  LIB  DT COMMAND",
"NR   PID    STACK      ESP      EIP TMOUT ALARM STAT TT   TIME COMMAND"
};

extern void (*fmt_fnc[])();	/* forward declaration */
void prtime(unsigned long t, unsigned long rel);
void read_globals();
void usage(void);
void show_procs(void);
void show_time(struct ps_proc * this);
void set_maxcmd(int w_opts);

/*
 * command line options
 */
int fmt;
int all;
int kern_comm;
int no_ctty;
int run_only;
char *ctty;
pid_t pid = -1;
int show_env;
int no_header = 0;
int num_outp;		/* numeric fields for user or wchan */
int first_task = 1;	/* don't show task 0 */
int pg_shift = 2;	/* default: show k instead of pages */
int Sum;

/* Globals */
int current_time, maxcmd;
unsigned int main_mem;
long time_now;

int main(int argc, char **argv)
{
    char *p;
    int fopt = 0;
    int width = 0;
    int Update = 0;


repeat:
    if (argc > 1) {
	for (p = argv[1]; *p; ++p) {
	    switch (*p) {
		case 'l': fmt = PS_L; ++fopt; break;
		case 'u': fmt = PS_U; ++fopt; break;
		case 'j': fmt = PS_J; ++fopt; break;
		case 's': fmt = PS_S; ++fopt; break;
		case 'v': fmt = PS_V; ++fopt; break;
		case 'm': fmt = PS_M; ++fopt; break;
		case 'X': fmt = PS_X; ++fopt; break; /* regs */
		case 'a': all = 1; break;
		case 'c': kern_comm = 1; break;
		case '0': first_task = 0; /*falltrough*/
		case 'x': no_ctty = 1; break;
		case 't': ctty = p + 1; break;
		case 'r': run_only = 1; break;
		case 'e': show_env = 1; break;
		case 'w': ++width; break;
		case 'h': no_header = 1; break;
		case 'n': num_outp = 1; break;
		case 'S': Sum = 1; break;
		case 'p': pg_shift = 0; break;
		case 'g':	/* old flag, ignore */
		case '-': break;
		default:
		    if (*p >= '0' && *p <= '9') {
			pid = atoi(p);
		    } else
			usage();
	    }
	    if (ctty || pid != -1)
		break;		/* pid and tty always last */
	}
	if (fopt > 1) {
	    fprintf(stderr, "ps: specify only one of j,l,s,u,v,m,X\n");
	    exit(1);
	}
    }
    if (argc > 2) {
	++argv;
	--argc;
	goto repeat;
    }

    set_maxcmd(width);
    read_globals();
    show_procs();
    return 0;
}


void usage(void)
{
    fprintf(stderr, "usage:  ps acehjlnrsSuvwx{t<tty>,#} \n");
    exit(1);
}


/*
 * set maximum chars displayed on a line
 */
void set_maxcmd(int w_opts)
{
    struct winsize win;
    int cols = 80;

    if (ioctl(1, TIOCGWINSZ, &win) != -1 && win.ws_col > 0)
	cols = win.ws_col;

    switch (w_opts) {
	case 0: break;
	case 1: cols += 52; break;
	case 2: cols *= 2; break;
	default: cols = MAXCMD;
    }
    maxcmd = cols - strlen(hdrs[fmt]) + 6;
}


void print_cmdline(char *cmdline)
{
  if (strlen(cmdline) > maxcmd)
    cmdline[maxcmd - 1] = 0;
  printf("%s\n", cmdline);
}


void show_procs(void)
{
    struct ps_proc *this;

    int tty, uid, pid;

    uid = getuid();
    pid = getpid();

    if (ctty)
	tty = tty_to_dev(ctty);

    this = take_snapshot((all | (uid==0)), fmt==PS_U, no_ctty, fmt==PS_M);
    if (!no_header)
	puts(hdrs[fmt]);
    for (; this != NULL; this = this->next) {
      if(!ctty || (this->tty == tty)){
	(fmt_fnc[fmt])(this);
	if (fmt != PS_V && fmt != PS_M)
	  show_time(this);
	print_cmdline(strlen(this->cmdline) ? this->cmdline : this->cmd);
      }
    }
}



void show_short(struct ps_proc *this)
{
    printf("%5d %s %c   ",
	this->pid,
	this->ttyc,
	this->state);
}

void show_long(struct ps_proc *this)
{
  char wchan[10];

  sprintf(wchan, "%-9x", this->wchan);
  printf("%2x %5d %5d %5d %3d %2d %4d %4d %-10.10s %c    %s ",
	 this->flags, /* the used_math element will /always/ be set,
			 because crt0.s checks the math emulation,
			 so it isn't worth including here, which is
			 why I didn't include it in the output format
			 from the stat file... */
	 this->uid,
	 this->pid,
	 this->ppid,
	 this->counter,
	 this->priority,
	 this->vsize / 1024,
	 this->rss * 4,
	 wchan,
	 this->state,
	 this->ttyc);
}

void show_jobs(struct ps_proc *this)
{
    printf("%5d %5d %5d %5d %s %5d  %c    %5d ",
	this->ppid,
	this->pid,
	this->pgrp,
	this->session,
	this->ttyc,
	this->tpgid,
	this->state,
	this->uid);
}

void show_user(struct ps_proc *this)
{
  int pcpu, pmem, start, total_time;

  if (num_outp)
    printf("%5d    ", this->uid);
  else
    printf("%-8s ", this->user);
  start = time_now - (((current_time * 100) - this->start_time) / HZ);
  total_time = (this->utime + this->stime +
		(Sum ? this->cutime + this->cstime : 0));
  pcpu = (total_time > (HZ-1)) ?
         (total_time * 10) / (current_time - (this->start_time / HZ)) :
         0;
  pmem = this->rss * 1000 / (main_mem / 4096);
  printf(" %5d %2d.%d %2d.%d %4d %4d %s %c   %.6s ",
	 this->pid,
	 pcpu / 10, pcpu % 10,
	 pmem / 10, pmem % 10,
	 this->vsize / 1024,
	 this->rss * 4,
	 this->ttyc,
	 this->state,
	 ctime(&start) + (time_now - start > 3600*24 ? 4 : 10));
}

void show_sig(struct ps_proc *this)
{

    printf("%5d %5d %08x %08x %08x %08x %c    %s ",
	this->uid,
	this->pid,
	this->signal,
	this->blocked,
	this->sigignore,
	this->sigcatch,
	this->state,
	this->ttyc);
}

void show_vm(struct ps_proc *this)
{
    int pmem;

    printf("%5d %2s %c   ",
	   this->pid,
	   this->ttyc,
	   this->state);
    show_time(this);
    printf(" %6d %4d %4d %4d ",
	   this->maj_flt + (Sum ? this->cmaj_flt : 0),
	   this->end_code / 1024,
	   (this->vsize - this->end_code) / 1024,
	   this->rss * 4);
    if(this->rss_rlim == RLIM_INFINITY)
      printf("   xx ");
    else
      printf("%5d ", this->rss_rlim / 1024);
    pmem = this->rss * 1000 / (main_mem / 4096);
    printf("%2d.%d ", pmem / 10, pmem % 10);
}


void show_m(struct ps_proc *this)
{

  printf("%5d %2s %6d %6d %4d %4d %4d %4d %4d %4d %4d %3d ", 
	 this->pid,
	 this->ttyc,
	 this->maj_flt + (Sum ? this->cmaj_flt : 0),
	 this->min_flt + (Sum ? this->cmin_flt : 0),
	 this->statm.trs << pg_shift,
	 this->statm.drs << pg_shift,
	 this->statm.size << pg_shift,
	 (this->statm.size - this->statm.resident) << pg_shift,
	 this->statm.resident << pg_shift,
	 this->statm.share << pg_shift,
	 this->statm.lrs << pg_shift,
	 this->statm.dt);
}

void show_regs(struct ps_proc *this)
{
    printf("%2d %5d %8x %8x %8x ",
	this->start_code >> 26,
	this->pid,
	this->start_stack,
	this->kstk_esp,
	this->kstk_eip);

    prtime(this->timeout, current_time * 100);
    prtime(this->it_real_value, 0);

    printf("%c    %2s ",
	this->state,
	this->ttyc);
}

void prtime(unsigned long t, unsigned long rel)
{
    if (t == 0) {
	printf("      ");
	return;
    }
    if ((long) t == -1) {
	printf("   xx ");
	return;
    }
    if ((long) (t -= rel) < 0)
	t = 0;
    
    if (t > 9999)
	printf("%5d ", t / 100);
    else
	printf("%2d.%02d ", t / 100, t % 100);
}

void (*fmt_fnc[])() = {
    show_short,
    show_long,
    show_user,
    show_jobs,
    show_sig,
    show_vm,
    show_m,
    show_regs
};


void show_time(struct ps_proc * this)
{
    unsigned t;
    t = (this->utime + this->stime) / HZ;
    if (Sum) t += (this->cutime + this->cstime) / HZ;
    printf("%3d:%02d ", t / 60, t % 60);
}


void read_globals()
{
  char uptime[30], memory[300];
  int fd;

  fd = open("/proc/uptime", O_RDONLY, 0);
  if (fd == -1) {
    fprintf(stderr, "/proc must be mounted\n");
    perror("ps.c:read_globals:uptime");
    exit(1);
  }
  read(fd,uptime,29);
  close(fd);
  current_time = atoi(uptime);
  fd = open("/proc/meminfo", O_RDONLY, 0);
  if(fd == -1) {
    fprintf(stderr, "/proc must be mounted\n");
    perror("ps.c:read_globals:meminfo");
    exit(1);
  }
  read(fd,memory,299);
  close(fd);
  sscanf(memory, "%*s %*s %*s %*s %*s %*s %u", &main_mem);
  time_now = time(0L);
}
