/*
 * testfs.c		- Test race conditions on a filesystem
 *
 * Copyright (C) 1993  Remy Card <card@masi.ibp.fr>
 *
 * This file can be redistributed under the terms of the GNU General
 * Public License
 */

/*
 * History:
 * 93/11/28	- Creation
 */

#include <fcntl.h>
#include <memory.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <sys/param.h>

#define DEFAULT_ALARM	600	/* 10 minutes */
#define NR_FILES	32
#define NR_PROCESSES	16

char * prog_name = "testfs";

volatile void usage (void)
{
	fprintf (stderr, "Usage: %s [-f file_count] [-d delay] [-p proc_count]\n",
			 prog_name);
	exit (1);
}

void alarm_handler (int sig)
{
	exit (0);
}

void test (int files_count)
{
	int d;
	int i;
	int r;
	char buffer1 [4096];
	char buffer2 [4096];
	char path [MAXPATHLEN];

	memset (buffer1, 'x', sizeof (buffer1));
	while (1)
		for (i = 1; i <= files_count; i++)
		{
			sprintf (path, "file%d", i);
			d = open (path, O_CREAT | O_EXCL | O_WRONLY, 0644);
			if (d == -1)
				continue;
			r = write (d, buffer1, sizeof (buffer1));
			if (r == -1)
				perror ("write");
			else if (r < sizeof (buffer1))
				printf ("Only wrote %d bytes\n", r);
			close (d);
			d = open (path, O_RDONLY);
			if (d == -1)
				continue;
			r = read (d, buffer2, sizeof (buffer2));
			if (r == -1)
				perror ("read");
			else if (r < sizeof (buffer2))
				printf ("Only read %d bytes\n", r);
			close (d);
			unlink (path);
		}
}

void main (int argc, char ** argv)
{
	char c;
	int i;
	int pid;
	int delay = DEFAULT_ALARM;
	int file_count = NR_FILES;
	int proc_count = NR_PROCESSES;
	char * tmp;

	if (argc && *argv)
		prog_name = *argv;
	while ((c = getopt (argc, argv, "d:f:p:")) != EOF)
		switch (c)
		{
			case 'd':
				delay = strtol (optarg, &tmp, 0);
				if ((*tmp) || delay < 1)
				{
					printf ("Bad delay: %s\n", optarg);
					usage ();
				}
				break;
			case 'f':
				file_count = strtol (optarg, &tmp, 0);
				if ((*tmp) || file_count < 1)
				{
					printf ("Bad file count: %s\n", optarg);
					usage ();
				}
				break;
			case 'p':
				proc_count = strtol (optarg, &tmp, 0);
				if ((*tmp) || proc_count < 1)
				{
					printf ("Bad process count: %s\n", optarg);
					usage ();
				}
				break;
			default:
				usage ();
		}
	if (optind != argc)
		usage ();
	for (i = 0; i < proc_count; i++)
	{
		pid = fork ();
		if (pid == -1)
			perror ("fork");
		else if (pid == 0)
		{
			signal (SIGALRM, alarm_handler);
			alarm (delay);
			test (file_count);
		}
	}
	exit (0);
}
