
/*
 * fcall.c
 *
 * This program uses SAMAPI to look up callsign data in the
 * SAM database. New version 2 features for accessing optional
 * data (eg. expiration date), are demonstrated.
 *
 * To run:
 *    FCALL callsign
 *
 * fcall depends on the API to the sam db provided by SAMAPI.EXE.
 *
 * to compile with Borland C++
 *  bcc findcall.c samlib.c
 *
 * Other C compilers can be used, but a "pack structs" option may be
 * required.
 *
 * Copyright 1993 by Tom Thompson, Athens, AL and by RT Systems, Inc.
 *                   [n4yos]                         1-800-723-6922
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dos.h>

#include "samapi.h"			/* samapi interface spec */

/*
 * functions in samlib.c
 */

int LocateSam(void);		// determine if API available
int CallSam(int cmd, void far *cmdbuf, void far *rspbuf);  // access API

/*
 * decls for functions contained here
 */

char *cDate(int year, int jday, int type);


/*******************
 * main
 *******************
 *
 * in: argv[1] is callsign
 *
 * returns (or exits()) non-zero if error, zero if call found and displayed
 */

main(int argc, char *argv[])
{
	int apiLoaded, apiVersion;	// loaded flag and SAMAPI version
	int apiOptions = 0;			// option bits (eg. license expire info)
	SamCmdBufUnion cmd;			// command buffer
	SamRespBufUnion resp;		// response buffer
	int err;
	int findPrevFlag = 1;		// find previous call flag
	datarec_t *d;
	int isCanada;


	apiLoaded = !LocateSam();

	if (apiLoaded)
	{
		CallSam(SamGetVersion, &cmd, &resp);
		apiVersion = resp.version.version;
		if (apiVersion >= 2)
		{
			CallSam(SamGetAvailOptions, &cmd, &resp);
			apiOptions = resp.option.optmask;
		}
	}
	if (argc < 2 || !apiLoaded || apiVersion < 2)
	{
		// we don't have everything we need so tell user about program

		printf("FCALL Version 1.0\n"
			   "\n"
			   "usage: FCALL callsign\n"
			   "\n");
		if (!apiLoaded)
		{
			printf("You must load SAMAPI before using FCALL\n");
		}
		else if (apiVersion < 2)
		{
			printf("You must use a later version of SAMAPI\n");
		}
		else
		{
			// we got API and its a good version
			printf("SAMAPI is loaded");
			if (apiOptions)
			{
				printf(" and the available options are:\n");
				if (apiOptions & OB_BIRTHDAY)
					printf("   - Birthday\n");
				if (apiOptions & OB_LICENSE)
					printf("   - Date first licensed\n");
				if (apiOptions & OB_EXPIRE)
					printf("   - License expiration date\n");
				if (apiOptions & OB_NEWCALL)
					printf("   - Find current call from old call\n");
				if (apiOptions & OB_OLDCALL)
					printf("   - Find old call from current call\n");
				if (apiOptions & OB_COUNTY)
					printf("   - County\n");
			}
			printf("\n");
		}
		return 1;
	}

	// if we get here, then the api is ready and we have a call in argv[1]

	strupr(argv[1]);		// make callsign uppercase

	strncpy(cmd.call.call, argv[1], 6);
	cmd.call.call[6] = 0;
	cmd.call.packflags = 0;
	err = CallSam(SamFindCall, &cmd, &resp);
	if (err)
	{
		// call not found see if there is new call data available
		strncpy(cmd.newcall.call, argv[1], 6);
		cmd.newcall.call[6] = 0;
		err = CallSam(SamFindNewCall, &cmd, &resp);
		if (!err)
		{
			// we got a new call for the ham
			printf("%s is now %s\n\n", argv[1], resp.newcall.call);
			findPrevFlag = 0;
			strcpy(cmd.call.call, resp.newcall.call);
			cmd.call.packflags = 0;
			err = CallSam(SamFindCall, &cmd, &resp);
		}
	}

	if (err)
	{
		printf("No data for %s\n", argv[1]);
		return 1;
	}


	d = &resp.drec.d;

	isCanada = (6 == strlen(d->Zip));


	// display the basic info
	printf("\n%-10.10s", d->Call);
	if (d->FirstName[0] > ' ' || strcmp(" ", d->FirstName))
		printf("%s ", d->FirstName);
	if (d->MidInitial[0] > ' ')
		printf("%c ", d->MidInitial[0]);
	printf("%s\n", d->LastName);
	printf("          %s\n", d->Address);
	printf("          %s, %s ", d->City, d->State);
	if (isCanada)
		printf("%3.3s %s\n", d->Zip, d->Zip+3);
	else
		printf("%s\n", d->Zip);

	printf("\n");
	if (d->Class[0] > ' ')
		printf("Class: %s\n", d->Class);

	// print birthday info
	if (strchr("NTGAE", d->Class[0]) && d->Dob[0] > ' ')
	{
		if (d->BirthDay)
			printf("Birthday: %s\n", cDate(d->DobNumber, d->BirthDay, 0));
		else
			printf("Year Born: %s\n", d->Dob);
	}
	// now expire date
	if (d->ExpireDay)
		printf("Expires : %s\n", cDate(d->ExpireYear, d->ExpireDay, 2));
	// now first licensed
	if (d->LicenseDay)
	{
		if (d->LicenseYear == 81 && d->LicenseDay == (31+27))
			printf("Licensed: Before Mar '81\n");
		else
			printf("Licensed: %s\n", cDate(d->LicenseYear, d->LicenseDay, 1));
	}

	// check for any user edits

	if (d->Flags & FL_ADDED)
		printf("This record has been added\n");
	else if (d->Flags & FL_EDITED)
		printf("This record has been edited\n");

	// get the county
	if (!isCanada)
	{
		strcpy(cmd.county.zip, d->Zip);
		cmd.county.reserved[0] = cmd.county.reserved[1] = 0;
		err = CallSam(SamFindCounty, &cmd, &resp);
		if (!err)
			printf("County: %s\n", resp.county.county);
	}

	// check for old calls (unless we already showed it

	if (!isCanada && findPrevFlag)
	{
		cmd.oldcall.maxreturn = 1;	// current database only has one anyway
		strncpy(cmd.oldcall.call, argv[1], 6);
		cmd.oldcall.call[6] = 0;
		err = CallSam(SamFindOldCall, &cmd, &resp);
		if (!err)
		{
			printf("Previous call: %s\n", resp.oldcall.call[0]);
		}
	}

	return 0;
}





//
// cDate converts year and day of year to human readable string
//
// returns char * to static string
//
//
// type = 0 for a birthday, 1 for a recent past date, 2 for a future date

char *cDate(int year, int jday, int type)
{
	static char buf[] = "Jan 21, 93    ";
	static char *ms[] = { "Jan", "Feb", "Mar", "Apr", "May", "Jun",
						  "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };
	static int mtab[] = { 31,28,31,30,31,30,31,31,30,31,30,31,-1 };
	int mon, day, i;

	if (year < 0 || year > 99 || jday < 1 || jday > 366)
	{
		buf[0] = 0;
		return buf;
	}

	mon = 0;
	day = 0;

	mtab[1] = (year & 3) ? 28 : 29;

	for (i=0; mtab[i] > 0; i++)
	{
		if (jday <= mtab[i])
			break;
		mon++;
		jday -= mtab[i];
	}
	day = jday;
	if (mtab[i] < 0 || mon > 11)
	{
		buf[0] = 0;
		return buf;		// something went wrong
	}

	switch (type)
	{
		case 0:
			year += (year > 90) ? 1800 : 1900;
			break;
		case 1:
			year += 1900;
			break;
		case 2:
			year += (year < 50) ? 2000 : 1900;
			break;
	}

	sprintf(buf, "%s %d, %d", ms[mon], day, year);
	return buf;
}

