/*****************************************************************************
** COLORSQR.PRG
**
** ProVision:Windows v1.20 example of subclassing pwCtrl with Class(y)
**
** by J. David Reynolds
**
** Copyright 1992 SofDesign International, Inc.
** All rights reserved
**
** tab spacing = 3
**
*****************************************************************************/

#include "class(y).ch"
#include "pw.ch"



/*****************************************************************************
** CLASS ColorSquare
*****************************************************************************/
CREATE CLASS ColorSquare FROM pwCtrl

   EXPORT:

      // Declare exported methods.
      METHOD   init
      METHOD   display
      METHOD   event
      METHOD   setValue

   HIDDEN:

      // Declare hidden class variables and methods.
      CLASS VAR      colors
      CLASS METHOD   colorToNum

      // Declare hidden instance variables and methods.
      VAR      currentColor
      METHOD   nextColor
      METHOD   prevColor

   INIT CLASS:

      // Initialize the class variables.
      ::colors := { "N", "B", "G", "BG", "R", "RB", "GR", "W" }

   END CLASS // ColorSquare



/*****************************************************************************
** CLASS METHOD colorToNum( cColor ) --> nColor
*****************************************************************************/
METHOD FUNCTION colorToNum( cColor )

   // Return the matching color number, or 0 if the color is invalid.
   RETURN( ASCAN( ::colors, cColor ) )
   // END colorToNum( cColor )



/*****************************************************************************
** METHOD init( nRow, nCol, nHeight, nWidth ) --> self
*****************************************************************************/
METHOD FUNCTION init( nRow, nCol, nHeight, nWidth )


   // Let the superclass initialize the control.
   ::super:init()

   // Set the position and size of the control.
   ::row    := nRow
   ::col    := nCol
   ::height := nHeight
   ::width  := nWidth

   // Set the colors for the control.  These instance variables are
   // numeric for the standard controls, but we will use them as
   // character values.
   ::color        := "+W/B"
   ::focusColor   := "N/W"

   // Set the current and initial values to N.
   ::currentColor := ::class:colorToNum( "N" )
   ::initialValue := "N"
   ::super:setValue( "N" )

   // Return a reference to self.
   RETURN( self )
   // END init( nRow, nCol, nHeight, nWidth )



/*****************************************************************************
** METHOD display() --> lSuccess
*****************************************************************************/
METHOD FUNCTION display()

   LOCAL idMsg := PWMSG_DISPLAY

   LOCAL oOutput


   // Let the filterBlock, if any, have a shot at the message.
   ::filterBlock:eval( @idMsg,,,,, self )

   // If the message still exists, continue.
   IF idMsg != PWMSG_NONE

      // Make sure the window is valid (that is, that the control has been
      // attached).
      IF ::wnd != NIL

         // Make sure output is directed to the proper window.
         oOutput  := pw():setOutput( ::wnd )

         // Display an outline.  Note the use of the instance variables
         // hasFocus, focusColor, and color.
         pwWriteBox( PWRW_TEXTATTR, ::row, ::col, ;
            ::row + ::height - 1, ::col + ::width - 1, ;
            "Ŀ", IIF( ::hasFocus, ::focusColor, ::color ) )

         // Fill the outline with the current color.
         pwWriteBox( PWRW_TEXTATTR, ::row + 1, ::col + 1, ;
            ::row + ::height - 2, ::col + ::width - 2, ;
            "         ", ;
            ::class:colors[9 - ::currentColor] + "/" + ::getValue() )

         // Restore the previous output window.
         pw():setOutput( oOutput )

         // Signal that the message was successfully handled.
         idMsg := PWMSG_NONE

      END IF // ::wnd != NIL

   END IF // idMsg != PWMSG_NONE

   // Return whether the message was successfully handled.
   RETURN( idMsg == PWMSG_NONE )
   // END display()



/*****************************************************************************
** METHOD event( @idMsg, @xParam1, @xParam2, @xParam3, @xParam4 ) --> NIL
*****************************************************************************/
METHOD FUNCTION event( idMsg, xParam1, xParam2, xParam3, xParam4 )

   LOCAL nWndCol, ;
         nWndRow

   LOCAL oCtrl


   // Let the filterBlock, if any, have a shot at the message.
   ::filterBlock:eval( @idMsg, @xParam1, @xParam2, @xParam3, @xParam4, ;
      self )

   // If the message still exists, continue.
   IF idMsg != PWMSG_NONE

      DO CASE
      // Test for keypresses.
      CASE idMsg == PWMSG_KEYPRESS

         // Test for interesting keys.
         IF ( xParam1 == K_SPACE ) .OR. ( xParam1 == ASC( "+" ) )

            // Display the next color in sequence.
            ::nextColor()

            // Signal that the message was successfully handled.
            idMsg := PWMSG_NONE

         ELSEIF xParam1 == ASC( "-" )

            // Display the previous color in sequence.
            ::prevColor()

            // Signal that the message was successfully handled.
            idMsg := PWMSG_NONE

         END IF // ( xParam1 == K_SPACE ) .OR. ( xParam1 == ASC( "...

      // Test for mouse button activity.        
      CASE ( idMsg == PWMSG_MOUSEPRESS ) .OR. ( idMsg == PWMSG_MOUSEHELD )

         // Convert the mouse cursor position to a window position.
         ::wnd:screenToWnd( xParam2, xParam3, @nWndRow, @nWndCol )

         // Get the control, if any, at the window position.
         oCtrl := ::wnd:ctrlAtPos( nWndRow, nWndCol )

         // Test whether the mouse button activity was on the
         // control.
         IF oCtrl == self

            IF xParam1 == PMBUTTON_LEFT

               // Display the next color in sequence.
               ::nextColor()

            ELSE

               // Display the previous color in sequence.
               ::prevColor()

            END IF // xParam1 == PMBUTTON_LEFT

            // Signal that the message was successfully handled.
            idMsg := PWMSG_NONE

         END IF // oCtrl == self

      END CASE

   END IF // idMsg != PWMSG_NONE

   // Return whether the message was successfully handled.
   RETURN( idMsg == PWMSG_NONE )
   // END event( idMsg, xParam1, xParam2, xParam3, xParam4 )



/*****************************************************************************
** METHOD nextColor() --> NIL
*****************************************************************************/
METHOD FUNCTION nextColor()


   // Increment the current color with wrapping.
   IF ::currentColor == LEN( ::class:colors )
      ::setValue( ::class:colors[1] )
   ELSE
      ::setValue( ::class:colors[::currentColor + 1] )
   END IF // ::currentColor == LEN( ::class:colors )

   // Return NIL.
   RETURN( NIL )
   // END nextColor()



/*****************************************************************************
** METHOD prevColor() --> NIL
*****************************************************************************/
METHOD FUNCTION prevColor()


   // Decrement the current color with wrapping.
   IF ::currentColor == 1
      ::setValue( ATAIL( ::class:colors ) )
   ELSE
      ::setValue( ::class:colors[::currentColor - 1] )
   END IF // ::currentColor == 1

   // Return NIL.
   RETURN( NIL )
   // END prevColor()



/*****************************************************************************
** METHOD setValue( cColor ) --> NIL
*****************************************************************************/
METHOD FUNCTION setValue( cColor )

   // Get the new color character value.
   LOCAL nColor   := ::class:colorToNum( cColor )


   // If the new value was valid, continue.
   IF nColor != 0

      // Set the new value.
      ::currentColor := nColor

      // Let the superclass know about the new value.
      ::super:setValue( cColor )

   END IF // nColor != 0

   // Return NIL.
   RETURN( NIL )
   // END setValue( cColor )

