// 
// THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY
// KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR 
// PURPOSE.
// 
// Copyright 1993 Microsoft Corporation, all rights reserved.
// 
// 
/****************************************************************************/
/*
    Dialer ---- Windows TAPI sample application created as an illustration of the usage of Windows TAPI
    
    Dialer does the following 3 things :
    
    (1) initiates/drops calls
    (2) handles simple TAPI requests for other applications; initiating/dropping calls on their behalf
    (3) monitors incoming/outgoing calls and keeps a call log based on the user's request.
    
    dialer.c : contains dialer's main code module and all the related UI code.
*/

/****************************************************************************/
/* include files */

#include <windows.h>
#include <windowsx.h>
#include <io.h>
#include <errno.h>
#include <string.h>
#include <tapi.h>
#include <math.h>
#include <malloc.h>

#include "dialer.h"

/****************************************************************************/
/* constants definitions */

#define cLogFieldMax            10      /* maximum number of log fields we allow */
#define cchLogLineMax           128     /* maximum line length (in chars) = all fields plus tabs */
#define cchLogFileMax           32000   /* maximum number of characters we care to store */

#define CTL3D_REGISTER          12      /* ordinals for using CTL3D.DLL */
#define CTL3D_AUTOSUBCLASS      16
#define CTL3D_SUBCLASSDLG       2
#define CTL3D_UN_REGISTER       13
#define CTL3D_COLOR_CHANGE      6
#define CTL3D_ALL				     0xffff  /* was defined in ctl3d.h, moved so we won't need it */

/****************************************************************************/
/* struct definitions and the corresponding global declarations */

typedef struct tagDWS                   /* Dialer Window State */ 
    {
    HANDLE          hInst;              /* instance handle */    
    HANDLE          haccel;             /* handle to our accelerator table */
    HWND            hwnd;               /* handle to our main window */ 
    HWND            hdlgCallStatus;     /* handle to the call status window */
    HWND            hdlgCallLog;        /* handle to the call logging window */
    HWND            hwndDummy    ;      /* handle to our dummy top level window */
    
    /* various resources we use */                                    
    HFONT           hfntBtn;            /* the font that is used to display speed dial buttons and dial pad buttons */
    HFONT           hfntBtnText;        /* the font that is used to display the first line text in the dial pad buttons */
    HBRUSH          hbrGray;            
    HBRUSH          hbrBtnFace;
    HBRUSH          hbrFocusPat;
    HPEN            hpenBlack;
    HPEN            hpenWhite;
    HPEN            hpenBtnFace;
    HPEN            hpenBtnShadow;
    
    /* various flags we keep */
    BOOL        fCommandFromLogWindow;  /* tells me whether the Dial/LogOptions commands are issued from the log window's menu */
    BOOL        fSDBtnRightClicked;     /* tells me whether the mouse click on one of the speed dial buttons in the main window is a right click */
    BOOL        fLogIncomingCalls;      /* tells me whether incoming calls should be logged */
    BOOL        fLogOutgoingCalls;      /* tells me whether outgoing calls should be logged */
    BOOL        fCallLogVisible;        /* tells me whether the call log window is visible */
    BOOL        fCallLogDirty;          /* tells me whether the content of the call log window is out of date */
    BOOL        f24HourTime;            /* tells me whether we are on 24 hours time system */

    /* misc variables used for call logging */
    int rgcchFieldMax[cLogFieldMax];    /* array of field widths computed from the ikszCallLogMask string */
    int cLogField;                      /* number of fields for each log entry */
    /* 
    These are used for remembering what we've read into log window.
    */
    LONG logLengthPrev;
    LONG curLogData;
    
    /* misc fields */                                                
    WORD        cfDialer;               /* our private clipboard format. We use it as a flag to tell us whether the clipboard is from our log window */             
    DWORD       dwSelPos;               /* text selection in the Number to Dial combo box */
    int         hidCallStatus;          /* the help topic to bring up for the Call Status dialog */
    char        szAM[6];                /* call logging related strings */
    char        szPM[6];
    char        szDate[2];
    char        szTime[2];
    char        szDateFormat[20];
    char        szDialerName[32];       /* name of the application */ 
    RECT        rcCallStatusHelpBtn;    /* the original rect of the help button in the Call Status Dialog */ 

    /* for the ctl3d library */
    HINSTANCE   hCtl3DInst;             /* library handle for the ctl3d.dll library */
    } DWS;

DWS vdws = {NULL,NULL,NULL,NULL,NULL,NULL, NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL, FALSE,FALSE,FALSE,FALSE,FALSE,TRUE,TRUE, {0},0,0,0, NULL,0,0};

/****************************************************************************/
/* function declarations */

/* utility functions */

void CenterDlg(HWND hwndDlg);
void DoDialog(FARPROC pfnDlgProc,WORD iddlog,HINSTANCE hinst,HWND hwnd,LONG lref);
int IkszFromErrCode(DWORD errCode);
void DialerFatalExit(DWORD errCode);
void DisableDialerDesktop(BOOL fDisable);
void DialerReInitTapi(void);
void DialerErrMessageBox(int stringID);

/* main code module */

static BOOL FDialerTranslateMessage(MSG *pmsg);
static void HandleDialerWndCmds(WPARAM wParam,LPARAM lParam);
static void PainDialerWnd(HWND hwndDialer);
static VOID DialPad(char chNum);

LRESULT FAR PASCAL _export      DialerWndProc(HWND hwnd,UINT uMsg,WPARAM wParam,LPARAM lParam);
LRESULT FAR PASCAL _export      DummyWndProc(HWND hwnd,UINT uMsg,WPARAM wParam,LPARAM lParam);
 
/* dialog procs for all the dialog boxes we bring up */

BOOL FAR PASCAL _export         ProgramSpeedDialDlgProc(HWND hDlg,UINT uMsg,WPARAM wParam,LPARAM lParam);
BOOL FAR PASCAL _export         ProgramSpeedDialButtonDlgProc(HWND hDlg,UINT uMsg,WPARAM wParam,LPARAM lParam);
BOOL FAR PASCAL _export         DialingOptionDlgProc(HWND hDlg,UINT uMsg,WPARAM wParam,LPARAM lParam);
BOOL FAR PASCAL _export         CallLogOptionDlgProc(HWND hDlg,UINT uMsg,WPARAM wParam,LPARAM lParam);
BOOL FAR PASCAL _export         AboutDialerDlgProc(HWND hDlg,UINT uMsg,WPARAM wParam,LPARAM lParam);
BOOL FAR PASCAL _export         LineInUseDlgProc(HWND hDlg,UINT uMsg,WPARAM wParam,LPARAM lParam);
BOOL FAR PASCAL _export         ChangeOptionDlgProc(HWND hDlg,UINT uMsg,WPARAM wParam,LPARAM lParam);
BOOL FAR PASCAL _export         DialingPromptDlgProc(HWND hDlg,UINT uMsg,WPARAM wParam,LPARAM lParam);

/* call status dialog related function */

static BOOL FCreateCallStatusDlg();
void ShowCallStatusDlg(LPCSTR szName,LPCSTR szNumber,LPCSTR szLocation,LPCSTR szCallingCard,LPCSTR szTranslatedSddress);
void UpdateCallStatusDlg(BOOL fCallIsActive,LPCSTR szName,LPCSTR szNumber);
void HideCallStatusDlg(void);

BOOL FAR PASCAL _export         CallStatusProc(HWND hDlg,UINT uMsg,WPARAM wParam,LPARAM lParam);

/* Call logging related functions */

BOOL FAR PASCAL _export CallLogProc(HWND hDlg,UINT uMsg,WPARAM wParam,LPARAM lParam);
BOOL FLogCall(BOOL,LPSTR,LPSTR,time_t,time_t);

static void ShowCallLogDlg(BOOL fDoNotShow);
static BOOL FReadCallLog(HWND,BOOL);                                                 
static int  SetCallLogTabs(HWND hwndList);
static void LoadIntlFormats(void);
static void GetSzNameNumFromSzLine(LPSTR szLine,LPSTR szName,LPSTR szNum);
static BOOL FGetNameOfNumberFromCallLog(LPSTR szNumber,LPSTR szName);
static void CopyLogToClipboard(HWND hwndList);
static BOOL FRemoveLogEntries(HWND hwndList);
static void PasteLogClipboardIntoEditCtrl(HWND hwndCtrl,BOOL fPasteName);

static WORD CplConfigDlg(HWND hDlg,WORD wDlg,LPSTR lpszName);

/****************************************************************************/
/*macro definitions */

/* Returns TRUE if preset (Quick Dial) is empty   */
#define FIsPresetEmpty(preset)  (!((preset)[0]))

/* Returns TRUE iff we have a dialable number in didDialerComboNumDial. We define this as just a non-empty string */
#define FNumberDialable()   (SendMessage(GetDlgItem(vdws.hwnd,didDialerComboNumDial),WM_GETTEXTLENGTH,0,0) > 0)

/****************************************************************************/
/* function definitions */
/****************************************************************************/
/* %%Function:CenterDlg  */
/* this function centers the dialog hwndDlg with respect to vdws.hwnd. */

void CenterDlg(HWND hwndDlg)
{
    RECT rcDlg, rcDialerWnd, rcScreen;
    int dxd, dyd;

    GetWindowRect(hwndDlg,&rcDlg);
    OffsetRect(&rcDlg,-rcDlg.left,-rcDlg.top);

    GetWindowRect(GetDesktopWindow(),&rcScreen);
    if (vdws.fCommandFromLogWindow)
        GetWindowRect(vdws.hdlgCallLog,&rcDialerWnd);  
   else if (!IsIconic(vdws.hwnd))
        GetWindowRect(vdws.hwnd,&rcDialerWnd);  
    else
        /* center on screen */
        rcDialerWnd = rcScreen;
        
    dxd = max(0,(rcDialerWnd.right + rcDialerWnd.left - rcDlg.right)/2);
    dyd = max(0,(rcDialerWnd.bottom + rcDialerWnd.top - rcDlg.bottom)/2);

    if (rcDlg.right + dxd > rcScreen.right)
        dxd = rcScreen.right - rcDlg.right;
    if (rcDlg.bottom + dyd > rcScreen.bottom)
        dyd = rcScreen.bottom - rcDlg.bottom;
    
    OffsetRect(&rcDlg,dxd,dyd);                              
    MoveWindow(hwndDlg, rcDlg.left, rcDlg.top, 
               rcDlg.right - rcDlg.left,
               rcDlg.bottom - rcDlg.top,FALSE);

} /* CenterDlg */

/****************************************************************************/
/* %%Function:DoDialog */
/*
    General cover proc for running a modal dialog
*/

VOID DoDialog(FARPROC pfnDlgProc,WORD iddlog,HINSTANCE hinst,HWND hwnd,LONG lref)
{
    DLGPROC dlgproc;

    dlgproc = (DLGPROC)MakeProcInstance((FARPROC)pfnDlgProc,hinst);
    DialogBoxParam(hinst, MAKEINTRESOURCE(iddlog),hwnd,dlgproc,lref);
    FreeProcInstance((FARPROC)dlgproc);
    
} /* DoDialog */

/****************************************************************************/
/* %%Function:IkszFromErrCode */
/*  
    translate errCode to its corresponding iksz string ID.
*/

int IkszFromErrCode(DWORD errCode)

{
    int ikszErr;
    
    switch (errCode) 
        {
        case errNoVoiceLine:
            {     
            ikszErr = ikszErrNoVoiceLine;
            break;
            }
        case LINEERR_NODRIVER:
            {
            ikszErr = ikszErrLineInitNoDriver;
            break;
            }   
        case LINEERR_INIFILECORRUPT:
            {
            ikszErr = ikszErrLineInitBadIniFile;
            break;
            }
        case LINEERR_NOMEM:
            {
            ikszErr = ikszErrLineInitOOM;
            break;
            }
        case LINEERR_INCOMPATIBLEAPIVERSION:
            {
            ikszErr = ikszErrLineInitWrongDrivers;
            break;
            }
        case LINEERR_OPERATIONFAILED:
            {
            ikszErr = ikszErrTAPI;
            break;
            }
        case LINEERR_INVALLINEHANDLE:
            {
            ikszErr = ikszErrInvalLineHandle;
            break;
            }
        case LINEERR_INVALADDRESS:
            {
            ikszErr = ikszErrInvalAddress;
            break;
            }
        case LINEERR_INVALPARAM:
            {
            ikszErr = ikszErrInvalParam;
            break;
            }
        case LINEERR_INVALCALLPARAMS:
            {
            ikszErr = ikszErrInvalCallParam;
            break;
            }
        case LINEERR_ADDRESSBLOCKED:
            {
            ikszErr = ikszErrAddrBlocked;
            break;
            }
        case LINEERR_BILLINGREJECTED:
            {
            ikszErr = ikszErrBillingRejected;
            break;
            }
        case LINEERR_RESOURCEUNAVAIL:
            {
            ikszErr = ikszErrResUnavail;
            break;
            }
        case LINEERR_OPERATIONUNAVAIL:
            {
            ikszErr = ikszErrOperationUnavail;
            break;
            }
        default: 
            {
            ikszErr = ikszErrAppStart; 
            break;
            } 
        } /* switch */
    return ikszErr;    

} /* IkszFromErrCode */

/****************************************************************************/
/* %%Function:DialerFatalExit */
/* 
    called when we encounter a fatal error and we need to exit the application. tells user
    what's wrong by putting up a message box and then calls DestroyWindow to end the
    world.  
*/   

VOID FAR DialerFatalExit(DWORD errCode)
{
    int  ikszErr = IkszFromErrCode(errCode);  
    char szT[cchSzMax];   
    
    LoadString(vdws.hInst,ikszErr,szT,sizeof(szT));
    MessageBox(vdws.hwnd,szT,vdws.szDialerName,MB_APPLMODAL | MB_ICONEXCLAMATION);

    TapiDone();
    DestroyWindow(vdws.hwnd);   
    
} /* DialerFatalExit */

/****************************************************************************/
/* %%Function:HwndDialerMain */
/*
    returns vdws.hwnd.
*/

HWND HwndDialerMain()

{ 
    return vdws.hwnd;
    
} /* HwndDialerMain */

/****************************************************************************/
/* %%Function:DisableDialerDesktop */
/* 
    disables/enables Dialer's dialing functionality while waiting for the line state
    to return to idle.
*/

void DisableDialerDesktop(BOOL fDisable)

{   
    int did;
    
    EnableWindow(GetDlgItem(vdws.hwnd,didDialerBtnDial),!fDisable);
    for (did = didDialerBtnSpeedDialFirst; did <= didDialerBtnSpeedDialLast; ++did)
        EnableWindow(GetDlgItem(vdws.hwnd,did),!fDisable);
        
} /* DisableDialerDesktop */

/****************************************************************************/
/* %%Function:DialerReInitTapi */
/*
    called to reinit tapi after telephon.ini changes.
*/

void DialerReInitTapi(void)

{   
    DWORD errCode;
          
    if ((errCode = ErrStartTapi(vdws.hInst,vdws.szDialerName)) != errNone) 
        {/* bail out */
        DialerFatalExit(errCode);
        return;
        } /* if */

   ErrInitCallingCardInfo(vdws.hInst);
    
} /* DialerReInitTapi */
 
/****************************************************************************/
/* %%Function:DialerErrMessageBox */
/*   
    Puts up a simple warning message when something non-tragic goes wrong.
    stringID - ID to pass to LoadString for text of warning message
*/

void DialerErrMessageBox(int stringID)
{
    char szText[128];
    char szTitle[60];
    
    if (!LoadString(vdws.hInst,stringID,szText,sizeof(szText)) || !LoadString(vdws.hInst,ikszWarningTitle,szTitle,sizeof(szTitle)))
        {
        MessageBeep(MB_ICONEXCLAMATION);  
        return;
        } /* if */
        
    MessageBox(vdws.hwnd,szText,szTitle,MB_ICONEXCLAMATION | MB_OK);
    
} /* DialerErrMessageBox */

/****************************************************************************/
/* main code module */
/****************************************************************************/
/* %%Function:FDialerTranslateMessage */
/*
    translate VK_TAB and VK_RETURN keyboard messages passed to vdws.hwnd's children so that these key strokes have the same
    effects has controls in a dialog box.
*/

static BOOL FDialerTranslateMessage(MSG *pmsg)  

{   
    HWND hwndCtrl;

    /* the following code deals with the fact that we can't get the text selection of the combo box when it does not
       have focus. We need to know the text selection for the function DialPad 
    */
    if (pmsg->hwnd != NULL && GetParent(pmsg->hwnd) == GetDlgItem(vdws.hwnd,didDialerComboNumDial) && GetFocus() == pmsg->hwnd)
        vdws.dwSelPos = SendMessage(GetDlgItem(vdws.hwnd,didDialerComboNumDial),CB_GETEDITSEL,0,0L);

    /* the following code converts right mouse actions on any speed dial button in the main window into
       the corresponding left mouse actions so that a BN_CLICKED message will be generated for it. we set       
       the global variable vdws.fSDBtnRightClicked to fTrue to distinguish between the right click form the
       left click in the WM_COMMAND processing and the global varibale is reset there too. On a right
       click, we let the user edit the speed dialing settings associated with the button while on a left
       click, we dial the number associated with the speed dial button.
    */
    if (pmsg->message >= WM_MOUSEFIRST && pmsg->message <= WM_MOUSELAST && GetParent(pmsg->hwnd) == vdws.hwnd
        && GetDlgCtrlID(pmsg->hwnd) >= didDialerBtnSpeedDialFirst && GetDlgCtrlID(pmsg->hwnd) <= didDialerBtnSpeedDialLast)
        if (pmsg->message == WM_RBUTTONDOWN) 
            {
            pmsg->message = WM_LBUTTONDOWN;
            vdws.fSDBtnRightClicked = TRUE; 
            } /* if */
        else if (pmsg->message == WM_RBUTTONUP)
            pmsg->message = WM_LBUTTONUP; 
    
    if (pmsg->message != WM_CHAR || pmsg->hwnd != NULL && GetParent(hwndCtrl = pmsg->hwnd) != vdws.hwnd && GetParent(pmsg->hwnd) != (hwndCtrl = GetDlgItem(vdws.hwnd,didDialerComboNumDial)))
        return FALSE;
    
    if (pmsg->wParam == VK_TAB)
        {    
        HWND hwndNextCtrl;
        
        if (hwndNextCtrl = GetNextDlgTabItem(vdws.hwnd,hwndCtrl,GetKeyState(VK_SHIFT) < 0)) 
            {
            SetFocus(hwndNextCtrl);
            return TRUE;
            } /* if */
        } /* if */
    else if (pmsg->wParam == VK_RETURN)
        {
        int didCtrl; 
        
        if ((didCtrl = GetDlgCtrlID(hwndCtrl)) == didDialerComboNumDial) 
            SendMessage(vdws.hwnd,WM_COMMAND,didDialerBtnDial,MAKELONG(GetDlgItem(vdws.hwnd,didDialerBtnDial),BN_CLICKED)); 
        else
            SendMessage(vdws.hwnd,WM_COMMAND,didCtrl,MAKELONG(pmsg->hwnd,0));
        return TRUE; 
        } /* else if */
            
    return FALSE;
                     
} /* DialerTranslateMessage */

/****************************************************************************/
/* %%Function:DummyWndProc */
/*
    Window proc for vdws.hwndDummy.
*/    

LRESULT FAR PASCAL _export DummyWndProc(HWND hwnd,UINT uMsg,WPARAM wParam,LPARAM lParam)

{ 
    return DefWindowProc(hwnd,uMsg,wParam,lParam);
    
} /* DummyWndProc */

/****************************************************************************/
/* %%Function:WinMain */
/*
    entry point to the Dialer program.
*/

int PASCAL WinMain(HINSTANCE hinst, HINSTANCE hprvinst, LPSTR lpszcmdline, int ncmdshow)
{
    static WORD _based(_segname("_CODE")) krgBits[] = {0xAA, 0x55, 0xAA, 0x55, 0xAA, 0x55, 0xAA, 0x55};
    char        szDialerClass[32], szDummyClass[32];
    char        szClipbrdFormat[cchSzMax];
    HBITMAP     hbmp = NULL, hbmpFocusPat = NULL;
    MSG         msg;
    WNDCLASS    wndclass, wndclassDummy;
    RECT        rc;
    HDC         hdcScreen; 
    DWORD       errCode;
    FARPROC     lpfnCtl3DSubclassDlg = NULL;
    
    /* get the global strings */
    LoadString(hinst,ikszAppClass,szDialerClass,sizeof(szDialerClass));
    LoadString(hinst,ikszDummyClass,szDummyClass,sizeof(szDummyClass));
    LoadString(hinst,ikszAppName,vdws.szDialerName,sizeof(vdws.szDialerName));

    /* allow only one instance */
    if (hprvinst)  
        {
        /* Find the first instance main window, and post a message to it to tell it to activate itself */
        HWND hwnd;
        
        ShowWindow(hwnd = FindWindow(szDialerClass,NULL),SW_RESTORE);
        BringWindowToTop(hwnd);
        return FALSE;
        } /* if */
    vdws.hInst = hinst;
    
    /* init 3D control stuff */
    if ( LOBYTE(LOWORD( GetVersion())) < 4 )      /* only for win3.1 and below */
      {
      WORD  wOldErrMode;

      wOldErrMode = SetErrorMode( SEM_NOOPENFILEERRORBOX );      /* turn off that it can't find the library! */
      vdws.hCtl3DInst = LoadLibrary("CTL3D.DLL");
      SetErrorMode( wOldErrMode );                               /* restore the error mode! */

      if (vdws.hCtl3DInst <= HINSTANCE_ERROR)
         {
         vdws.hCtl3DInst = NULL;    /* set back to default */
         }
      else
         {
         FARPROC  lpfnCtl3DRegister;
         FARPROC  lpfnCtl3DAutoSubclass;

         /* get all the procs that we need and call them */
         lpfnCtl3DRegister = GetProcAddress( vdws.hCtl3DInst, MAKEINTRESOURCE( CTL3D_REGISTER ));
         lpfnCtl3DAutoSubclass = GetProcAddress( vdws.hCtl3DInst, MAKEINTRESOURCE( CTL3D_AUTOSUBCLASS ));
         lpfnCtl3DSubclassDlg = GetProcAddress( vdws.hCtl3DInst, MAKEINTRESOURCE( CTL3D_SUBCLASSDLG ));

         if ((lpfnCtl3DRegister == NULL) || (lpfnCtl3DAutoSubclass == NULL) || (lpfnCtl3DSubclassDlg == NULL))
            {
            FreeLibrary( vdws.hCtl3DInst );  /* ah, didn't work, don't worry too much about it */
            vdws.hCtl3DInst = NULL;
            }
         else
            {      
            (*lpfnCtl3DRegister)( vdws.hInst );     /* register the stuff */
            (*lpfnCtl3DAutoSubclass)( vdws.hInst );
            }  /* if */
         }  /* if */
      } /* if */

    /* we can still run the program without being able to load the accelerator */
    vdws.haccel = LoadAccelerators(vdws.hInst,MAKEINTRESOURCE(aidDialer));

    /* init the pens and brushes we use */
    if ((hbmp = CreateBitmap(8,8,1,1,(LPSTR)krgBits)) == NULL 
        || (hbmpFocusPat = LoadBitmap(vdws.hInst,MAKEINTRESOURCE(bidFocusPat))) == NULL
        || (vdws.hbrGray = CreatePatternBrush(hbmp)) == NULL
        || (vdws.hbrFocusPat = CreatePatternBrush(hbmpFocusPat)) == NULL
        || (vdws.hbrBtnFace = CreateSolidBrush(GetSysColor(COLOR_BTNFACE))) == NULL
        || (vdws.hpenBtnFace = CreatePen(PS_SOLID,1,GetSysColor(COLOR_BTNFACE))) == NULL
        || (vdws.hpenBtnShadow = CreatePen(PS_SOLID,1,GetSysColor(COLOR_BTNSHADOW))) == NULL)
        goto LOOM;
    vdws.hpenBlack = GetStockObject(BLACK_PEN);
    vdws.hpenWhite = GetStockObject(WHITE_PEN);

    /* get international date/time formats */
    LoadIntlFormats();
    
    /* Dialer window class definition */
    wndclass.style          = CS_HREDRAW | CS_VREDRAW;
    wndclass.lpfnWndProc    = DialerWndProc;
    wndclass.cbClsExtra     = 0;
    wndclass.cbWndExtra     = DLGWINDOWEXTRA;
    wndclass.hInstance      = hinst;
    wndclass.hIcon          = LoadIcon(hinst, MAKEINTRESOURCE(icoDialer));
    wndclass.hCursor        = LoadCursor(NULL, IDC_ARROW);
    wndclass.hbrBackground  = GetStockObject(LTGRAY_BRUSH);
    wndclass.lpszMenuName   = NULL;
    wndclass.lpszClassName  = szDialerClass; 

    /* Dummy top level window class definition */
    wndclassDummy.style          = CS_HREDRAW | CS_VREDRAW;
    wndclassDummy.lpfnWndProc    = DummyWndProc;
    wndclassDummy.cbClsExtra     = 0;
    wndclassDummy.cbWndExtra     = 0;
    wndclassDummy.hInstance      = hinst;
    wndclassDummy.hIcon          = NULL;
    wndclassDummy.hCursor        = NULL;
    wndclassDummy.hbrBackground  = NULL;
    wndclassDummy.lpszMenuName   = NULL;
    wndclassDummy.lpszClassName  = szDummyClass; 
    
    /* register window class; load accelerators; and create the Dialer Dialing window. return if OOM */
    if (RegisterClass(&wndclass) == 0 
        || RegisterClass(&wndclassDummy) == 0
        || (vdws.hwndDummy = CreateWindow(szDummyClass,vdws.szDialerName,WS_POPUP,0,0,0,0,NULL,NULL,hinst,NULL)) == NULL
        || (vdws.hwnd = CreateDialog(hinst,MAKEINTRESOURCE(dlgDialer),vdws.hwndDummy,NULL)) == NULL
        || !FCreateCallStatusDlg())
        {
        char szT[256];   

        if (vdws.hwnd)
            DestroyWindow(vdws.hwnd);
        if (vdws.hwndDummy)
            DestroyWindow(vdws.hwndDummy);
            
        LoadString(vdws.hInst,ikszErrOOM,szT,sizeof(szT));
        MessageBox(NULL,szT,vdws.szDialerName,MB_SYSTEMMODAL | MB_ICONEXCLAMATION);
        return FALSE;
        } /* if */
    ShowWindow(vdws.hwndDummy,SW_SHOW);
    if ( lpfnCtl3DSubclassDlg != NULL )      /* ctl3d subclass the dialog */
       (*lpfnCtl3DSubclassDlg)(vdws.hwnd,CTL3D_ALL);
    
    SetCursor(LoadCursor(NULL,IDC_WAIT));

    /* set up line,address and last number(s) dialed pull downs */
    GetAllINIQuickDials(vdws.hInst,vdws.hwnd); 
    GetLastDialedNumbers(vdws.hInst,vdws.hwnd);

    /* set the position of the window as it was before */
    GetWindowRect(vdws.hwnd,&rc);
    GetSetINIScreenPos(vdws.hInst,(POINT *)&rc,TRUE);
    SetWindowPos(vdws.hwnd,NULL,rc.left,rc.top,0,0,SWP_NOSIZE | SWP_NOACTIVATE | SWP_NOREDRAW |SWP_NOZORDER);
    
    /* set the font for the speed dial/dial pad buttons */
    hdcScreen = GetDC(GetDesktopWindow());
    vdws.hfntBtn = CreateFont((-8)*GetDeviceCaps(hdcScreen,LOGPIXELSY)/72,0,0,0,FW_NORMAL,FALSE,FALSE,FALSE,
        ANSI_CHARSET,OUT_DEFAULT_PRECIS,CLIP_DEFAULT_PRECIS,PROOF_QUALITY,VARIABLE_PITCH|FF_SWISS,(LPSTR)"Helv");
    vdws.hfntBtnText = CreateFont((-6)*GetDeviceCaps(hdcScreen,LOGPIXELSY)/72,0,0,0,FW_NORMAL,FALSE,FALSE,FALSE,
        ANSI_CHARSET,OUT_DEFAULT_PRECIS,CLIP_DEFAULT_PRECIS,PROOF_QUALITY,VARIABLE_PITCH|FF_SWISS,NULL);
    ReleaseDC(GetDesktopWindow(),hdcScreen);
    if (vdws.hfntBtn)
        {
        int did;
        
        for (did = didDialerBtnSpeedDialFirst; did <= didDialerBtnSpeedDialLast; ++did)  
            SendMessage(GetDlgItem(vdws.hwnd,did),WM_SETFONT,vdws.hfntBtn,0L); 
        for (did = didDialerBtnPadFirst; did <= didDialerBtnPadLast; ++did)  
            SendMessage(GetDlgItem(vdws.hwnd,did),WM_SETFONT,vdws.hfntBtn,0L);
             
        SendMessage(GetDlgItem(vdws.hwnd,didDialerComboNumDial),WM_SETFONT,vdws.hfntBtn,0L);      
        SendMessage(GetDlgItem(vdws.hwnd,didDialerGrpSpeedDial),WM_SETFONT,vdws.hfntBtn,0L);      
        SendMessage(GetDlgItem(vdws.hwnd,didDialerSTextDial),WM_SETFONT,vdws.hfntBtn,0L);     
        } /* if */
     
    ShowWindow(vdws.hwnd,ncmdshow);
    UpdateWindow(vdws.hwnd);
    
    /* Limit text in Number field to TAPIMAXDESTADDRESSSIZE */
    SendDlgItemMessage(vdws.hwnd,didDialerComboNumDial,CB_LIMITTEXT,(WPARAM)TAPIMAXDESTADDRESSSIZE,0);
        
    /*  Initialize tapi, Negotiate TAPI versions for all line devices.Get lineDevCaps for all lines, and init address info. */ 
    if ((errCode = ErrStartTapi(vdws.hInst,vdws.szDialerName)) != errNone) 
        {/* bail out */
        DialerFatalExit(errCode);
        return FALSE;
        } /* if */
    
    /* init calling card and location info. */
    errCode = ErrInitCallingCardInfo(vdws.hInst);
    if (errCode != errNone)
        {
        int ikszErr = IkszFromErrCode(errCode);
        DialerErrMessageBox((ikszErr == ikszErrAppStart) ? ikszErrBadTAPIAddr : ikszErr);
        } /* if */
        
    /* register ourselves to be simple TAPI recipient */
    if (!FRegisterSimpleTapi(TRUE))
        DialerErrMessageBox(ikszWarningRegisterSTapi);

    /* register our clipboard format. go on if we fail */ 
    LoadString(vdws.hInst,ikszDialerClipbrdFormatName,szClipbrdFormat,sizeof(szClipbrdFormat));
    if (szClipbrdFormat[0])
        vdws.cfDialer = RegisterClipboardFormat(szClipbrdFormat);

    SetCursor(LoadCursor(NULL,IDC_ARROW));

    /* get the call logging options */
    vdws.fLogIncomingCalls = WGetDialerProfileInt(vdws.hInst,ikszSecCallLogging,ikszFieldCLIncoming,TRUE);
    vdws.fLogOutgoingCalls = WGetDialerProfileInt(vdws.hInst,ikszSecCallLogging,ikszFieldCLOutgoing,TRUE);
    
    /* show the call logging window according to the ini setting */
    if (WGetDialerProfileInt(vdws.hInst,ikszSecCallLogging,ikszFieldCLVisible,FALSE) != 0) 
        {
        ShowCallLogDlg(FALSE);
        BringWindowToTop(vdws.hwnd);
        } /* if */
        
    /* main message loop */
    while (GetMessage(&msg,NULL,0,0)) 
        {
        if (FDialerTranslateMessage(&msg)
            || vdws.fCallLogVisible && (GetActiveWindow() == vdws.hdlgCallLog && TranslateAccelerator(vdws.hdlgCallLog,vdws.haccel,&msg) || IsDialogMessage(vdws.hdlgCallLog,&msg))
            || vdws.hdlgCallStatus != NULL && IsWindowVisible(vdws.hdlgCallStatus) && IsDialogMessage(vdws.hdlgCallStatus,&msg)             
            || TranslateAccelerator(vdws.hwnd,vdws.haccel,&msg))
            continue;
        
        TranslateMessage(&msg);
        DispatchMessage(&msg);
        
        ProcessNextQueuedSTAPIRequest();
        } /* while */
    
    if (vdws.hdlgCallLog)
        DestroyWindow(vdws.hdlgCallLog);
    if (vdws.hdlgCallStatus)
        DestroyWindow(vdws.hdlgCallStatus);
    if (vdws.hwndDummy)
        DestroyWindow(vdws.hwndDummy);
    
    /* unregister and line close */
    TapiDone();

LOOM:
    /* close the 3d stuff */
    if ( vdws.hCtl3DInst != NULL )      /* does not happen very often */
       {         
       FARPROC   lpfnCtl3DUnregister;
       lpfnCtl3DUnregister = GetProcAddress( vdws.hCtl3DInst, MAKEINTRESOURCE( CTL3D_UN_REGISTER ));

       if ( lpfnCtl3DUnregister != NULL )
          (*lpfnCtl3DUnregister)( vdws.hInst );     // let him know
          
       FreeLibrary( vdws.hCtl3DInst );
       vdws.hCtl3DInst = NULL;
       } /* if */
    
    /* kill the pens and brushes we have created */
    if (hbmp)
        DeleteObject(hbmp);
    if (hbmpFocusPat)
        DeleteObject(hbmpFocusPat);
    if (vdws.hbrGray)
        DeleteObject(vdws.hbrGray);
    if (vdws.hbrFocusPat)
        DeleteObject(vdws.hbrFocusPat);
    if (vdws.hbrBtnFace)
        DeleteObject(vdws.hbrBtnFace);
    if (vdws.hpenBtnFace)
        DeleteObject(vdws.hpenBtnFace);
    if (vdws.hpenBtnShadow)
        DeleteObject(vdws.hpenBtnShadow);
    
    /* kill the font we created */
    if (vdws.hfntBtn)
        DeleteObject(vdws.hfntBtn);
    if (vdws.hfntBtnText)
        DeleteObject(vdws.hfntBtnText);   
        
    return (int)msg.wParam; 
    
} /* WinMain */

/****************************************************************************/
/* %%Function:DialPad */
/*
    called when the user presses the numeric key buttons. simulate typing the asciiCode 
    into number. window proc will generate DTMF digit if we are already dialing (this 
    triggers off EN_CHANGE notification from number.
*/

static VOID DialPad(char chDigit)
{
    char szNumDial[64];
    WORD ichSelStart, ichSelEnd;
    HWND hwndNumDial = GetDlgItem(vdws.hwnd,didDialerComboNumDial);
     
    ichSelStart = LOWORD(vdws.dwSelPos); 
    ichSelEnd = HIWORD(vdws.dwSelPos);
    
    GetWindowText(hwndNumDial,szNumDial,63); 
    _fmemmove(szNumDial + ichSelStart + 1,szNumDial + ichSelEnd,strlen(szNumDial + ichSelEnd)+1);
    
    szNumDial[ichSelStart] = chDigit; 
    SetWindowText(hwndNumDial,szNumDial); 
    
    SetFocus(hwndNumDial); 
    SendMessage(hwndNumDial,CB_SETEDITSEL,0,MAKELPARAM(ichSelStart+1,ichSelStart+1));
    vdws.dwSelPos = SendMessage(GetDlgItem(vdws.hwnd,didDialerComboNumDial),CB_GETEDITSEL,0,0L);
    EnableWindow(GetDlgItem(vdws.hwnd,didDialerBtnDial),TRUE);
    
} /* DialPad */                          

/****************************************************************************/
/* %%Function:replaceNumber */
/*
    replaces contents of number edit control with newNumber.
*/

static VOID replaceNumber(char *newNumber)
{   
    HWND numberHwnd = GetDlgItem(vdws.hwnd,didDialerComboNumDial);
    SetWindowText(GetDlgItem(vdws.hwnd,didDialerComboNumDial),newNumber);
    UpdateWindow(numberHwnd);

    EnableWindow(GetDlgItem(vdws.hwnd,didDialerBtnDial),TRUE);
    
} /* replaceNumber */                          


/****************************************************************************/
/* %%Function:getLast */
/*
    Returns selected redial number
*/

static VOID getLast(char *redialNumber)
{
    HWND numberHwnd = GetDlgItem(vdws.hwnd,didDialerComboNumDial);
    SendMessage(numberHwnd,CB_GETLBTEXT,(WPARAM)SendMessage(numberHwnd,CB_GETCURSEL,0,0),(LPARAM)((LPSTR)redialNumber));     

} /* getLast */


/****************************************************************************/
/* %%Function:addToLast */
/*
    Adds number to redial list.  if list is full, moves list down 1 and replaces first item.
*/

static VOID addToLast(char *newNumber)
{
    #define cnumRedial      20          /* maximum number of redial number we care to remember */
    WORD clistItems; 
    
    HWND numberHwnd = GetDlgItem(vdws.hwnd,didDialerComboNumDial);
    
    clistItems = (WORD)SendMessage(numberHwnd,CB_GETCOUNT, 0,0);
    if (clistItems != 0)
        { 
        DWORD dwSearch = SendMessage(numberHwnd,CB_FINDSTRING,0,(LONG)(LPCSTR)newNumber);
        if (dwSearch != CB_ERR) /* remove duplicate */
            SendMessage(numberHwnd,CB_DELETESTRING,LOWORD(dwSearch),0);
        else if (clistItems == cnumRedial)
            SendMessage(numberHwnd,CB_DELETESTRING,cnumRedial-1,0); 
        } /* else */
            
    SendMessage(numberHwnd,CB_INSERTSTRING,0,(LPARAM)((LPSTR)newNumber)); 
    SendMessage(numberHwnd,CB_SETCURSEL,0,0L); 
    UpdateWindow(numberHwnd);

    EnableWindow(GetDlgItem(vdws.hwnd,didDialerBtnDial),TRUE);
        
} /* addToLast */

/****************************************************************************/
/* %%Function:FDialerInitiateCall */
/*
    adds szNumber to the combo box and calls FInitiateCall to do the real work.
*/

BOOL FDialerInitiateCall(char *szNumber,char *szName)

{
    addToLast(szNumber);
    SetFocus(GetDlgItem(vdws.hwnd,didDialerBtnDial));
    return FInitiateCall(szNumber,szName); 
    
} /* FDialerInitiateCall */

/****************************************************************************/
/* %%Function:HandleDialerWndCmds */
/*
    called to process the WM_COMMAND message for vdws.hwnd. This message can be sent
    for either menu command or action to one of our dialog controls.
*/

void HandleDialerWndCmds(WPARAM wParam,LPARAM lParam)

{    
    char szName[TAPIMAXCALLEDPARTYSIZE];
    char szNumber[TAPIMAXDESTADDRESSSIZE];

    switch (wParam)
        { 
        /********************************************************/
        /* File menu */
        /* exit application */      
        case midFileExit:
            {
            DestroyWindow(vdws.hwnd);
            break;
            }        
        
        /********************************************************/
        /* Edit menu */ 
        /* Cut,Copy,Paste,Delete for Number Edit Control */
        case midEditCut: 
        case midEditCopy: 
        case midEditPaste:
        case midEditDelete:
            {
            if (wParam == midEditPaste && vdws.cfDialer != NULL && IsClipboardFormatAvailable(vdws.cfDialer))
                { 
                PasteLogClipboardIntoEditCtrl(GetDlgItem(vdws.hwnd,didDialerComboNumDial),FALSE);
                SendMessage(vdws.hwnd,WM_COMMAND,didDialerComboNumDial,MAKELPARAM(GetDlgItem(vdws.hwnd,didDialerComboNumDial),CBN_EDITCHANGE));
                } /* if */
            else
                SendDlgItemMessage(vdws.hwnd,didDialerComboNumDial,WM_CUT + (wParam-midEditCut),0,0);
            break; 
            }
        /* Calls up panel to edit Quick Dial information. */
        case midEditSpeedDialButtons:  
            {
            DoDialog(ProgramSpeedDialDlgProc,dlgProgSD,vdws.hInst,vdws.hwnd,1);
            SetFocus(GetDlgItem(vdws.hwnd, didDialerBtnDial));
            break; 
            }           

        /********************************************************/
        /* Options menu */
        /* Brings up the options dialog for dialing */
        case midOptionsDialing:
            { 
            DoDialog(DialingOptionDlgProc,dlgDialingOption,vdws.hInst,vdws.hwnd,0);
            break;
            }        
        /* Brings up the options dialog for call logging */
        case midOptionsLog:
            {
            DoDialog(CallLogOptionDlgProc,dlgLogOption,vdws.hInst,vdws.hwnd,0);
            break;
            }
        /* Brings up the options dialog for locations or calling cards */
        case midSetupLocation:
        case midSetupCallingCard:
            {
            CplConfigDlg(vdws.hwnd,wParam,NULL);
            break;
            }
                           
        /********************************************************/
        /* Log menu */  
        /* Show Call Log Menu item */
        case midOptionsViewLog:
            {
            if (!vdws.fCallLogVisible) 
                ShowCallLogDlg(FALSE);
            else
                SendMessage(vdws.hdlgCallLog,WM_CLOSE,0,0);
            break;
            }  
            
        /********************************************************/
        /* Help menu */                 
        /* Help */
        case midHelpContents:
        case midAccelHelp: 
            {
            WinHelp(vdws.hwnd,"dialer.hlp",HELP_CONTENTS,0);
            break; 
            }
        /* About Dialer Dialog */
        case midHelpAbout:
            {
            DoDialog(AboutDialerDlgProc,dlgAbout,vdws.hInst,vdws.hwnd,0);
            break;
            }
        /* does the accelerator action processing */
        case midAccelSelectNumToDial:
            {     
            if (GetActiveWindow() == vdws.hwnd)
                SetFocus(GetDlgItem(vdws.hwnd,didDialerComboNumDial));
            break;
            } 
        
        /* Phone number box. On change, if we are calling already, we generate DTMF for new digits entered. */
        case didDialerComboNumDial: 
            {
            if (GetFocus() == GetDlgItem(vdws.hwnd,didDialerComboNumDial))
                vdws.dwSelPos = SendMessage(GetDlgItem(vdws.hwnd,didDialerComboNumDial),CB_GETEDITSEL,0,0L);                     

            if (HIWORD(lParam) == CBN_SELENDOK && SendDlgItemMessage(vdws.hwnd,didDialerComboNumDial,CB_GETCOUNT,0,0) > 0) 
                {
                getLast(szNumber);
                replaceNumber(szNumber);
                } /* if */
            
            if (HIWORD(lParam) == CBN_EDITCHANGE || HIWORD(lParam) == CBN_SELENDOK)
                EnableWindow(GetDlgItem(vdws.hwnd,didDialerBtnDial),GetWindowTextLength(GetDlgItem(vdws.hwnd,didDialerComboNumDial)) > 0);  
                
            break;
            }
        /* Dial/Hangup button. If currently dialing, button says hangup, line choice 
           popup will also be disabled (along with most other controls). */         
        case didDialerBtnDial:
            {
            if (FCallInProgress())  
                {/* hangup */
                SetFocus(GetDlgItem(vdws.hwnd,didDialerComboNumDial));
                FDropCurrentCall();
                } /* if */
            else if (FNumberDialable() && GetDlgItemText(vdws.hwnd,didDialerComboNumDial,(LPSTR)szNumber,sizeof(szNumber)))
                {
                char szName[cchSzMax], szNum[cchSzMax];
                
                /* scan the call log for a name that corresponds to szNumber */
                if (!FGetNameOfNumberFromCallLog(szNumber,szName))
                    {/* search all the speed dial buttons for szNumber */
                    int isd;
                    
                    for (isd = 0; isd <= didDialerBtnSpeedDialLast - didDialerBtnSpeedDialFirst; ++isd)
                        {
                        GetSetINIQuickDial(vdws.hInst,isd,szName,szNum,TRUE);
                        if (lstrcmp(szNum,szNumber) == 0)
                            break; 
                        } /* for */
                    
                    if (isd > didDialerBtnSpeedDialLast - didDialerBtnSpeedDialFirst)
                        szName[0] = 0;
                    } /* if */
                    
                FDialerInitiateCall(szNumber,szName);          
                } /* else if */
            break;
            }
        /* dial pad emulation */                
        case didDialerBtnPad1:
        case didDialerBtnPad2:
        case didDialerBtnPad3:
        case didDialerBtnPad4:
        case didDialerBtnPad5:
        case didDialerBtnPad6:
        case didDialerBtnPad7:
        case didDialerBtnPad8:
        case didDialerBtnPad9:
        case didDialerBtnPad0:
            {
            DialPad((char)('0' + (1+wParam-didDialerBtnPad1)%10));
            break;
            } 
        case didDialerBtnPadStar: 
        case didDialerBtnPadPound:
            {
            DialPad((char)((wParam == didDialerBtnPadStar) ? '*' : '#'));
            break; 
            }
        /* These are the preset buttons - dial the number hit if preset has setting. */
        case didDialerBtnSpeedDial1:
        case didDialerBtnSpeedDial2:
        case didDialerBtnSpeedDial3:
        case didDialerBtnSpeedDial4:
        case didDialerBtnSpeedDial5:
        case didDialerBtnSpeedDial6:
        case didDialerBtnSpeedDial7:
        case didDialerBtnSpeedDial8:
        case didDialerBtnSpeedDial9:
        case didDialerBtnSpeedDial10:
            {
            GetSetINIQuickDial(vdws.hInst,wParam-didDialerBtnSpeedDial1,szName,szNumber,TRUE);
            if (!vdws.fSDBtnRightClicked && !FIsPresetEmpty(szNumber)) 
                FDialerInitiateCall(szNumber,szName);          
            else 
                {
                DoDialog(ProgramSpeedDialButtonDlgProc,dlgProgSDB,vdws.hInst,vdws.hwnd,MAKELPARAM(wParam,0));
                vdws.fSDBtnRightClicked = FALSE;
                } /* else */
            break;
            }
        } /* switch */
    
} /* HandleDialerWndCmds */

/****************************************************************************/
/* %%Function:DrawButton */
/*
    draws a 3D button within rcBtn on hdc.
*/

static void DrawButton(HDC hdc,RECT rcBtn,BOOL fDown);
static void DrawButton(HDC hdc,RECT rcBtn,BOOL fDown)
{
    HPEN        hpenPrev;
    HBRUSH      hbrPrev;
    int         ropPrev;
    int         dzw = fDown ? 2 : 4;

    --rcBtn.right;
    --rcBtn.bottom;
    
    hpenPrev = SelectObject(hdc,vdws.hpenBlack);
    ropPrev  = SetROP2(hdc,R2_COPYPEN);
    hbrPrev  = SelectObject(hdc,vdws.hbrBtnFace);
    PatBlt(hdc,rcBtn.left+2,rcBtn.top+2,rcBtn.right-rcBtn.left-dzw,rcBtn.bottom-rcBtn.top-dzw,PATCOPY);
    
    MoveTo(hdc,rcBtn.left + 1,rcBtn.top);
    LineTo(hdc,rcBtn.right,rcBtn.top);
    MoveTo(hdc,rcBtn.right,rcBtn.top+1);
    LineTo(hdc,rcBtn.right,rcBtn.bottom);
    MoveTo(hdc,rcBtn.right - 1,rcBtn.bottom);
    LineTo(hdc,rcBtn.left,rcBtn.bottom);
    MoveTo(hdc,rcBtn.left,rcBtn.bottom - 1);
    LineTo(hdc,rcBtn.left,rcBtn.top);

    if (fDown)
        {
        int mmPrev = SetMapMode(hdc,MM_TEXT);
        DWORD clrTextPrev = SetTextColor(hdc,RGB(0,0,0));
        DWORD clrBkPrev = SetBkColor(hdc,RGB(255,255,255));

        SelectObject(hdc,vdws.hbrGray);
        PatBlt(hdc,rcBtn.left+1,rcBtn.top+1,rcBtn.right-rcBtn.left-1,rcBtn.bottom-rcBtn.top-1,0xFA0089L);
        SetTextColor(hdc,clrTextPrev);
        SetBkColor(hdc,clrBkPrev);
        SetMapMode(hdc,mmPrev);
 
        SelectObject(hdc,vdws.hpenBtnShadow);
        MoveTo(hdc,rcBtn.left+1,rcBtn.bottom-1);
        LineTo(hdc,rcBtn.left+1,rcBtn.top+1);
        LineTo(hdc,rcBtn.right,rcBtn.top+1);
        
        {
        MoveTo(hdc,rcBtn.left+2,rcBtn.bottom-1);
        LineTo(hdc,rcBtn.left+2,rcBtn.top+2);
        LineTo( hdc,rcBtn.right,rcBtn.top+2);
        }
        } /* if */
    else
        { /* up state */
        /* Draw Edges */
        SelectObject(hdc,vdws.hpenWhite);
        MoveTo(hdc,rcBtn.left+1,rcBtn.bottom-2);
        LineTo(hdc,rcBtn.left+1,rcBtn.top+1);
        LineTo(hdc,rcBtn.right-1,rcBtn.top+1);

        MoveTo(hdc,rcBtn.left+2,rcBtn.bottom-3);
        LineTo(hdc,rcBtn.left+2,rcBtn.top+2);
        LineTo(hdc,rcBtn.right-2,rcBtn.top+2);

        SelectObject(hdc, vdws.hpenBtnShadow);
        MoveTo(hdc,rcBtn.right-1,rcBtn.top+1);
        LineTo(hdc,rcBtn.right-1,rcBtn.bottom-1);
        LineTo(hdc,rcBtn.left,rcBtn.bottom-1);
        
        MoveTo(hdc,rcBtn.left+2,rcBtn.bottom-2);
        LineTo(hdc,rcBtn.right-2,rcBtn.bottom-2);
        LineTo(hdc,rcBtn.right-2,rcBtn.top+1);
        } /* else */

    SetROP2(hdc,ropPrev);
    SelectObject(hdc,hbrPrev);
    SelectObject(hdc,hpenPrev); 
    
} /* DrawButton */

/****************************************************************************/
/* %%Function:DrawButtonText */
/*
    draws the text associated with didBtn. This whole mess is necessary since Windows
    standard buttons do not support multiple lines of text in them. 
    
    expects the text associated with the button contains a single '\n'(chr(10)) character
    which breaks the text to be drawn into two lines.
*/

void DrawButtonText(HDC hdc,RECT rcBtn,WORD didBtn);
void DrawButtonText(HDC hdc,RECT rcBtn,WORD didBtn)

{   
    BOOL fDown = ((WORD)SendDlgItemMessage(vdws.hwnd,didBtn,BM_GETSTATE,0,0)) & 0x0004;
    BOOL fHasFocus = ((WORD)SendDlgItemMessage(vdws.hwnd,didBtn,BM_GETSTATE,0,0)) & 0x0008;
    BOOL fSingleLineText;
    int bkModePrev;
    char szText[cchSzMax];
    char szDelimiter[2] = {10,0};
    WORD ichDelimiter;
    TEXTMETRIC tm;
    RECT rcText;
    int dxdText = 0, dydText = 0, dxdText2;
    
    if (fDown)
        {
        rcBtn.left += 3;
        rcBtn.top += 3;
        } /* if */ 
    rcText = rcBtn;
        
    bkModePrev = SetBkMode(hdc,TRANSPARENT);
    
    GetDlgItemText(vdws.hwnd,didBtn,szText,sizeof(szText));
    ichDelimiter = _fstrcspn(szText,szDelimiter);
    fSingleLineText = (ichDelimiter == strlen(szText));

    /* draws the first line */
    if (ichDelimiter > 0)
        {
        HFONT hfntPrev = NULL;
        
        if (vdws.hfntBtnText && !fSingleLineText)
            hfntPrev = SelectObject(hdc,vdws.hfntBtnText);
        GetTextMetrics(hdc,&tm);
        dxdText = LOWORD(GetTextExtent(hdc,(LPCSTR)szText,ichDelimiter));
        
        rcText.bottom = (rcBtn.bottom + rcBtn.top)/2;
        rcText.top = rcText.bottom - (dydText = tm.tmHeight - 1);
        if (fSingleLineText)
            OffsetRect(&rcText,0,(rcText.bottom - rcText.top)/2);
        DrawText(hdc,szText,ichDelimiter,&rcText,DT_SINGLELINE | DT_CENTER);
        
        if (hfntPrev && !fSingleLineText)  
            SelectObject(hdc,hfntPrev);
        } /* if */

    if(!fSingleLineText)
        {/* draws the second line */
        GetTextMetrics(hdc,&tm);
        dxdText2 = LOWORD(GetTextExtent(hdc,(LPCSTR)(szText+ichDelimiter+1),strlen(szText+ichDelimiter+1)));
        if (dxdText2 > dxdText)
            dxdText = dxdText2;
     
        rcText.top = (rcBtn.bottom + rcBtn.top)/2;
        rcText.bottom = rcText.top + tm.tmHeight;
        DrawText(hdc,szText+ichDelimiter+1,-1,&rcText,DT_SINGLELINE | DT_CENTER);
        dydText += tm.tmHeight;
        } /* if */
    
    /* draws the focus marking */
    if (fHasFocus) 
        {
        COLORREF crForPrev, crBackPrev;
        HPEN hpenPrev;
        HBRUSH hbrPrev;
        RECT rcFocus = rcBtn;
        int rop2Prev;
        
        InflateRect(&rcFocus,-((rcFocus.right - rcFocus.left) - dxdText)/2 + 1 + 2*(ichDelimiter == 0),-((rcFocus.bottom - rcFocus.top) - dydText)/2 + 1);
        if (ichDelimiter == 0)
            OffsetRect(&rcFocus,0,(rcFocus.bottom - rcFocus.top)/2);
        else
            rcFocus.top += 2;

        rop2Prev = SetROP2(hdc,R2_COPYPEN);
        hpenPrev = SelectObject(hdc,GetStockObject(NULL_PEN));
        hbrPrev = SelectObject(hdc,vdws.hbrFocusPat);
        crForPrev = SetTextColor(hdc,RGB(0,0,0));
        crBackPrev = SetBkColor(hdc,RGB(255,255,255)); 
    
        PatBlt(hdc,rcFocus.left,rcFocus.top,1,rcFocus.bottom -rcFocus.top,PATCOPY);
        PatBlt(hdc,rcFocus.right,rcFocus.top,1,rcFocus.bottom -rcFocus.top + 1,PATCOPY);
        PatBlt(hdc,rcFocus.left,rcFocus.top,rcFocus.right -rcFocus.left,1,PATCOPY);
        PatBlt(hdc,rcFocus.left,rcFocus.bottom,rcFocus.right-rcFocus.left,1,PATCOPY);
    
        SetROP2(hdc,rop2Prev);
        SetTextColor(hdc,crForPrev);
        SetBkColor(hdc,crBackPrev);
        SelectObject(hdc,hbrPrev);
        SelectObject(hdc,hpenPrev);
        } /* if */

    SetBkMode(hdc,bkModePrev);
        
} /* DrawButtonText */

/****************************************************************************/
/* %%Function:DialerWndProc */
/*
    Window proc for vdws.hwnd.
*/    

LRESULT FAR PASCAL _export DialerWndProc(HWND hwnd,UINT uMsg,WPARAM wParam,LPARAM lParam)
{
    switch (uMsg)
        { 
        /* Handle menu commands and actions to dialog controls */
        case WM_COMMAND:
            {
            HandleDialerWndCmds(wParam,lParam);
            break;
            }
        /* Enable/Disable Menu items */
        case WM_INITMENUPOPUP:
            {   
            HMENU mEdit;
            WORD wEnable;
             
            if (LOWORD(lParam) == 2) /* Options menu */
                {
                char szMenuText[cchSzMax];
                
                LoadString(vdws.hInst,vdws.fCallLogVisible ? ikszOptionsMenuHideLog : ikszOptionsMenuShowLog,szMenuText,sizeof(szMenuText));
                ModifyMenu((HMENU)wParam,midOptionsViewLog,MF_STRING | MF_BYCOMMAND,midOptionsViewLog,szMenuText);
                break;
                } /* else if */
            else if (LOWORD(lParam) != 1) /* not Edit menu */ 
                break;
            
            mEdit = (HMENU)wParam;
            EnableMenuItem(mEdit,midEditPaste,IsClipboardFormatAvailable(CF_TEXT)? MF_ENABLED:MF_GRAYED);
            if (GetParent(GetFocus()) != GetDlgItem(vdws.hwnd,didDialerComboNumDial)) 
                {
                wEnable = MF_GRAYED;
                EnableMenuItem(mEdit,midEditPaste,MF_GRAYED);
                } /* if */
            else    
                { 
                LONG lSelect = SendDlgItemMessage(vdws.hwnd,didDialerComboNumDial,CB_GETEDITSEL,0,0);
                wEnable = HIWORD(lSelect) != LOWORD(lSelect) ? MF_ENABLED:MF_GRAYED;
                } /* else */
            EnableMenuItem(mEdit,midEditCut,wEnable);
            EnableMenuItem(mEdit,midEditCopy,wEnable);
            EnableMenuItem(mEdit,midEditDelete,wEnable); 
            
            break;
            }
        case WM_DRAWITEM:
            {            
            LPDRAWITEMSTRUCT lpdis = (LPDRAWITEMSTRUCT)lParam;
            
            DrawButton(lpdis->hDC,lpdis->rcItem,((WORD)SendDlgItemMessage(vdws.hwnd,lpdis->CtlID,BM_GETSTATE,0,0)) & 0x0004);
            DrawButtonText(lpdis->hDC,lpdis->rcItem,lpdis->CtlID);
            break;
            } 
        /* Set custom colors for dialing window */
        case WM_CTLCOLOR:
            {
            switch(HIWORD(lParam))
                {
                case CTLCOLOR_DLG:
                case CTLCOLOR_MSGBOX:
                case CTLCOLOR_BTN:
                case CTLCOLOR_STATIC:
                case CTLCOLOR_LISTBOX:
                    {
                    SetBkColor(wParam,RGB(192,192,192));
                    return GetStockObject(LTGRAY_BRUSH);
                    }
                default:
                    return NULL;
                }
            break;
            }
        /* 3d control stuff */
        case  WM_SYSCOLORCHANGE:
         {
         /* 3D control stuff */
         if ( vdws.hCtl3DInst != NULL )      /* does not happen very often */
            {         
            FARPROC  lpfnCtl3DColorChange;

            lpfnCtl3DColorChange = GetProcAddress( vdws.hCtl3DInst, MAKEINTRESOURCE( CTL3D_COLOR_CHANGE ));
         
            if ( lpfnCtl3DColorChange != NULL )
               (*lpfnCtl3DColorChange)();     /* let him know */
            }  /* if */
         break;
         }
        /* Reflect changes in International date/time formating */
        case WM_WININICHANGE:
            {
            if (lParam == 0 || lstrcmp((LPSTR)lParam,"Intl"))
                LoadIntlFormats();
            break;
            }
        /* redraw Dialer window */
        /* Dialer being activated */  
        case WM_ACTIVATE:
            {
            if (wParam != WA_INACTIVE && !HIWORD(lParam))
                SetFocus(GetDlgItem(hwnd,didDialerComboNumDial));
            break;
            }
        case WM_ACTIVATEAPP:
            {
            if (wParam && !HIWORD(lParam))
                SetFocus(GetDlgItem(hwnd,didDialerComboNumDial));
            break;
            }
        case WM_LBUTTONDOWN:  
        case WM_MOVE:
            {/* hide the number to dial drop-down list box */
            SendDlgItemMessage(vdws.hwnd,didDialerComboNumDial,CB_SHOWDROPDOWN,(WPARAM)FALSE,0);
            break;          
            }
         /* quitting application */
        case WM_DESTROY:
            {
            RECT    rc;
            
            /* terminate outstanding calls */
            TerminateCalls(vdws.hInst);
            
            /* save screen location */
            GetWindowRect(hwnd,&rc);
            GetSetINIScreenPos(vdws.hInst,(POINT *)&rc,FALSE);
            SaveLastDialedNumbers(vdws.hInst,vdws.hwnd);
            
            /* do our part to unload help */
            WinHelp(hwnd,"dialer.hlp", HELP_QUIT, 0);
            
            /* get rid of log modeless dialog */
            if (vdws.hdlgCallLog) 
                {
                DestroyWindow(vdws.hdlgCallLog);
                vdws.fCallLogVisible = 0;
                vdws.fCallLogDirty = FALSE;
                vdws.hdlgCallLog = 0;
                } /* if */ 
            /* get rid of the call status dialog */ 
            if (vdws.hdlgCallStatus) 
                {
                DestroyWindow(vdws.hdlgCallStatus);
                vdws.hdlgCallStatus = 0;
                } /* if */ 
                
            PostQuitMessage(wParam);
            break;
            }
        default:
            return DefWindowProc(hwnd,uMsg,wParam,lParam);
        } /* switch */
    
    return FALSE;  
    
} /* DialerWndProc */

/****************************************************************************/
/* %%Function:DoubleUpAmpersandsInSz */
/* 
    looks at each character in sz and doubles up each '&' we run into.
    cchSzBuffer contains the TOTAL available size of the buffer used to contain sz.
*/

void DoubleUpAmpersandsInSz(char *sz,int cchSzBuffer)

{   
    int cchSz = strlen(sz) + 1;
    int cchFree = cchSzBuffer - cchSz;
    int cchAmpersand = 0;
    int ich = -1;
    
    while (++ich < cchSz)
        {
        if (sz[ich] != '&')
            continue;
        if (cchAmpersand >= cchFree)
            break;
        
        _fmemmove(sz + ich + 1,sz + ich,cchSz - ich);
        ++cchSz;
        ++ich;
        ++cchAmpersand;
        } /* while */
        
} /* DoubleUpAmpersandsInSz */

/****************************************************************************/
/* %%Function:SingleDownAmpersandsInSz */
/* 
    looks at each character in sz and converts each TWO '&'s into a single '&'.
*/

static void SingleDownAmpersandsInSz(char *sz)

{ 
    int cchSz = strlen(sz) + 1;
    int ich = -1;
    
    while (++ich < cchSz)
        {
        if (sz[ich] != '&' || sz[ich+1] != '&')
            continue;
        
        _fmemmove(sz + ich,sz + ich + 1,cchSz - ich - 1);
        --cchSz;
        } /* while */
    
} /* SingleDownAmpersandsInSz */

static FARPROC lpfnDefEditProcS = NULL;
static FARPROC lpfnSmartPasteEditProcS = NULL;

/****************************************************************************/
/* %%Function:NameNumEditCtrlProc */ 
/*
    sub-classed window-proc for edit controls that are used to contain name or
    phone number. This is necessary to implementing the smart paste when the clipboard
    contains the call log entry. We trap SHIFT+INSERT/CTRL+V and do the right thing.
*/

LRESULT FAR PASCAL _export NameNumEditCtrlProc(HWND hwndCtrl,UINT uMsg,WPARAM wParam,LPARAM lParam);
LRESULT FAR PASCAL _export NameNumEditCtrlProc(HWND hwndCtrl,UINT uMsg,WPARAM wParam,LPARAM lParam)

{   
    int did;

     if (vdws.cfDialer == NULL || !IsClipboardFormatAvailable(vdws.cfDialer)
        || !(uMsg == WM_KEYDOWN && wParam == VK_INSERT && GetKeyState(VK_SHIFT) < 0)
        && !(uMsg == WM_KEYUP && (wParam == 'V' || wParam == 'v') && GetKeyState(VK_CONTROL) < 0))
        return CallWindowProc(lpfnDefEditProcS,hwndCtrl,uMsg,wParam,lParam); 
    
    did = GetDlgCtrlID(hwndCtrl);
    PasteLogClipboardIntoEditCtrl(hwndCtrl,did == didProgSDEditName || did == didProgSDBEditName);
    return TRUE;
    
} /* NameNumEditCtrlProc */

/****************************************************************************/
/* %%Function:ProgramSpeedDialDlgProc */
/*
    Dialog Proc for the "Program Speed Dial Buttons..." dialog.
*/    

BOOL FAR PASCAL _export ProgramSpeedDialDlgProc(HWND hDlg,UINT uMsg,WPARAM wParam,LPARAM lParam)
{   
    static WORD didBtnCurS;
    
    switch (uMsg) 
        {
        case WM_INITDIALOG:
            {
            char szName[TAPIMAXCALLEDPARTYSIZE];
            char szNumber[TAPIMAXDESTADDRESSSIZE];
            WORD did, didBtnEmptyFirst = (WORD)-1;
            
            CenterDlg(hDlg);
            
            /* get names from INI file */
            for (did = didProgSDBtnSpeedDialFirst; did <= didProgSDBtnSpeedDialLast; ++did) 
                {
                GetSetINIQuickDial(vdws.hInst,(WORD)(did-didProgSDBtnSpeedDialFirst),szName,szNumber,TRUE); 
                /*  remember first empty preset */
                if (didBtnEmptyFirst == ((WORD)-1) && FIsPresetEmpty(szName)) 
                    didBtnEmptyFirst = did;
                        
                /* init the button with szName and set its font */
                DoubleUpAmpersandsInSz(szName,(int)TAPIMAXCALLEDPARTYSIZE);
                SetDlgItemText(hDlg,did,(LPCSTR)szName);  
                SetDlgItemText(hDlg,didProgSDSTextSpeedDialFirst + did - didProgSDBtnSpeedDialFirst,(LPCSTR)szNumber);  
                if (vdws.hfntBtn)
                    SendDlgItemMessage(hDlg,did,WM_SETFONT,vdws.hfntBtn,0L);
                } /* for */

            if (vdws.hfntBtn) 
                {
                SendDlgItemMessage(hDlg,didProgSDEditName,WM_SETFONT,vdws.hfntBtn,0L);  
                SendDlgItemMessage(hDlg,didProgSDEditNumber,WM_SETFONT,vdws.hfntBtn,0L);  
                } /* if */
            
            /* sub-class the two edit controls for smart paste */
            lpfnSmartPasteEditProcS = MakeProcInstance((FARPROC)NameNumEditCtrlProc,vdws.hInst);
            if (lpfnSmartPasteEditProcS != NULL)
                {
                lpfnDefEditProcS = (FARPROC)GetWindowLong(GetDlgItem(hDlg,didProgSDEditName),GWL_WNDPROC);
                SetWindowLong(GetDlgItem(hDlg,didProgSDEditName),GWL_WNDPROC,(LONG)lpfnSmartPasteEditProcS);
                SetWindowLong(GetDlgItem(hDlg,didProgSDEditNumber),GWL_WNDPROC,(LONG)lpfnSmartPasteEditProcS);
                } /* if */
            
            /* select the first empty name  */
            didBtnEmptyFirst = (didBtnEmptyFirst == ((WORD)-1) ? didProgSDBtnSpeedDialFirst : didBtnEmptyFirst);
            SendMessage(hDlg,WM_COMMAND,didBtnEmptyFirst,MAKELPARAM(GetDlgItem(hDlg,didBtnEmptyFirst),BN_CLICKED));     
            return FALSE;
            }
       /* Set custom colors for dialing window */
        case WM_CTLCOLOR:
            {/* do the same thing as in the Dialer main window */
            return LOWORD(DialerWndProc(hDlg,WM_CTLCOLOR,wParam,lParam));
            }
        /* action processing */
        case WM_COMMAND:
            {
            char szName[TAPIMAXCALLEDPARTYSIZE];
            char szNumber[TAPIMAXDESTADDRESSSIZE];

            switch (wParam)
                {
                case IDOK:
                    {
                    WORD did;
                    
                    for (did = didProgSDBtnSpeedDialFirst; did <= didProgSDBtnSpeedDialLast; ++did) 
                        {
                        GetDlgItemText(hDlg,did,szName,sizeof(szName)); 
                        SingleDownAmpersandsInSz(szName);
                        GetDlgItemText(hDlg,didProgSDSTextSpeedDialFirst + did - didProgSDBtnSpeedDialFirst,szNumber, sizeof(szNumber));  
                        
                        /*  save screen settings to INI file */
                        GetSetINIQuickDial(vdws.hInst,did-didProgSDBtnSpeedDialFirst,szName,szNumber,FALSE);
                        
                        /*  update the corresponding button in the main window */
                        GetSetINIQuickDial(vdws.hInst,did-didProgSDBtnSpeedDialFirst,szName,szNumber,TRUE); 
                        DoubleUpAmpersandsInSz(szName,(int)TAPIMAXCALLEDPARTYSIZE);
                        SetDlgItemText(vdws.hwnd,didDialerBtnSpeedDial1 + did - didProgSDBtnSpeedDialFirst,(LPCSTR)szName);                       
                        } /* for */
                    
                    /* fall through is intentional */
                    } 
                case IDCANCEL:
                    { 
                    if (lpfnSmartPasteEditProcS)
                        {
                        SetWindowLong(GetDlgItem(hDlg,didProgSDEditName),GWL_WNDPROC,(LONG)lpfnDefEditProcS);
                        SetWindowLong(GetDlgItem(hDlg,didProgSDEditNumber),GWL_WNDPROC,(LONG)lpfnDefEditProcS);
                        FreeProcInstance(lpfnSmartPasteEditProcS);
                        } /* if */
                    EndDialog(hDlg,wParam == IDOK);
                    return TRUE;
                    break;
                    } 
                case didProgSDBtnSpeedDial1: 
                case didProgSDBtnSpeedDial2: 
                case didProgSDBtnSpeedDial3: 
                case didProgSDBtnSpeedDial4: 
                case didProgSDBtnSpeedDial5: 
                case didProgSDBtnSpeedDial6: 
                case didProgSDBtnSpeedDial7: 
                case didProgSDBtnSpeedDial8: 
                case didProgSDBtnSpeedDial9: 
                case didProgSDBtnSpeedDial10:
                    {                    
                    /* save away the current selection */
                    didBtnCurS = wParam;
                    
                    /* update the two edit boxes to reflect the new selection */
                    GetDlgItemText(hDlg,didBtnCurS,szName,sizeof(szName));
                    SingleDownAmpersandsInSz(szName);
                    GetDlgItemText(hDlg,didProgSDSTextSpeedDialFirst + didBtnCurS - didProgSDBtnSpeedDialFirst,szNumber, sizeof(szNumber));  
                    SetDlgItemText(hDlg,didProgSDEditName,(LPCSTR)szName);  
                    SetDlgItemText(hDlg,didProgSDEditNumber,(LPCSTR)szNumber); 
                    SetFocus(GetDlgItem(hDlg,didProgSDEditName));
                    SendDlgItemMessage(hDlg,didProgSDEditName,EM_SETSEL,0,MAKELPARAM(0,-1)); 
                    break;
                    }
                case didProgSDEditName:
                    {  
                    if (HIWORD(lParam) != EN_CHANGE)
                        break;
                    
                    GetDlgItemText(hDlg,didProgSDEditName,szName,sizeof(szName)); 
                    DoubleUpAmpersandsInSz(szName,(int)TAPIMAXCALLEDPARTYSIZE);
                    SetDlgItemText(hDlg,didBtnCurS,(LPCSTR)szName);
                    break;  
                    } 
                case didProgSDEditNumber:
                    {  
                    if (HIWORD(lParam) != EN_CHANGE)
                        break;
                    
                    GetDlgItemText(hDlg,didProgSDEditNumber,szNumber,sizeof(szNumber));
                    SetDlgItemText(hDlg,didProgSDSTextSpeedDialFirst + didBtnCurS - didProgSDBtnSpeedDialFirst,(LPCSTR)szNumber);
                    break;  
                    } 
                case didHelp:
                    { 
                    WinHelp(hDlg,"dialer.hlp",HELP_CONTEXT,hidEditSpeedDial);
                    break;
                    } 
                } /* switch wParam */
            break;
            }
        } /* switch */
    
    return FALSE;
    
} /* ProgramSpeedDialDlgProc */
  
/****************************************************************************/
/* %%Function:ProgramSpeedDialButtonDlgProc */
/*
    Dialog Proc for the dialog we put up when the user clicks on an empty speed dial button or when he/she
    right clicks on a non-empty speed dial button.
*/    

BOOL FAR PASCAL _export ProgramSpeedDialButtonDlgProc(HWND hDlg,UINT uMsg,WPARAM wParam,LPARAM lParam)
{   
static WORD didDialerBtnSDClickedS = (WORD)-1;    /* the button the user clicked to program its speed dial setting */

    switch (uMsg) 
        {
        case WM_INITDIALOG:
            {
            char szName[TAPIMAXCALLEDPARTYSIZE];
            char szNumber[TAPIMAXDESTADDRESSSIZE];

            CenterDlg(hDlg);
            didDialerBtnSDClickedS = LOWORD(lParam);
            
            /* get names from INI file */
            GetSetINIQuickDial(vdws.hInst,(WORD)(didDialerBtnSDClickedS-didDialerBtnSpeedDialFirst),szName,szNumber,TRUE); 
                        
            /* init the edit boxes with szName and szNumber and set their font */
            SetDlgItemText(hDlg,didProgSDBEditName,(LPCSTR)szName);  
            SetDlgItemText(hDlg,didProgSDBEditNumber,(LPCSTR)szNumber);  

            if (vdws.hfntBtn) 
                {
                SendDlgItemMessage(hDlg,didProgSDBEditName,WM_SETFONT,vdws.hfntBtn,0L);  
                SendDlgItemMessage(hDlg,didProgSDBEditNumber,WM_SETFONT,vdws.hfntBtn,0L);  
                } /* if */

            /* sub-class the two edit controls for smart paste */
            lpfnSmartPasteEditProcS = MakeProcInstance((FARPROC)NameNumEditCtrlProc,vdws.hInst);
            if (lpfnSmartPasteEditProcS != NULL)
                {
                lpfnDefEditProcS = (FARPROC)GetWindowLong(GetDlgItem(hDlg,didProgSDBEditName),GWL_WNDPROC);
                SetWindowLong(GetDlgItem(hDlg,didProgSDBEditName),GWL_WNDPROC,(LONG)lpfnSmartPasteEditProcS);
                SetWindowLong(GetDlgItem(hDlg,didProgSDBEditNumber),GWL_WNDPROC,(LONG)lpfnSmartPasteEditProcS);
                } /* if */

            SetFocus(GetDlgItem(hDlg,didProgSDBEditName));
            SendDlgItemMessage(hDlg,didProgSDBEditName,EM_SETSEL,0,MAKELPARAM(0,-1));   
            return FALSE;
            } 
        /* Set custom colors for dialing window */
        case WM_CTLCOLOR:
            {/* do the same thing as in the Dialer main window */
            return LOWORD(DialerWndProc(hDlg,WM_CTLCOLOR,wParam,lParam));
            }
        /* action processing */
        case WM_COMMAND:
            {
            char szName[TAPIMAXCALLEDPARTYSIZE];
            char szNumber[TAPIMAXDESTADDRESSSIZE];

            switch (wParam)
                {
                case IDOK:
                case didProgSDBBtnSaveDial:
                    {
                    /* get the edited settings */
                    GetDlgItemText(hDlg,didProgSDBEditName,szName,sizeof(szName));
                    GetDlgItemText(hDlg,didProgSDBEditNumber,szNumber,sizeof(szNumber));  
                        
                    /*  save screen settings to INI file */
                    GetSetINIQuickDial(vdws.hInst,didDialerBtnSDClickedS-didDialerBtnSpeedDialFirst,szName,szNumber,FALSE);
                        
                    /*  update the corresponding button in the main window */
                    GetSetINIQuickDial(vdws.hInst,didDialerBtnSDClickedS-didDialerBtnSpeedDialFirst,szName,szNumber,TRUE);
                    DoubleUpAmpersandsInSz(szName,(int)TAPIMAXCALLEDPARTYSIZE);
                    SetDlgItemText(vdws.hwnd,didDialerBtnSDClickedS,(LPCSTR)szName);                      
                    
                    /* execute the updated speed dial button action */
                    if (wParam == didProgSDBBtnSaveDial)
                        PostMessage(vdws.hwnd,WM_COMMAND,didDialerBtnSDClickedS,MAKELPARAM(GetDlgItem(vdws.hwnd,didDialerBtnSDClickedS),BN_CLICKED));
                        
                    /* fall through is intentional */
                    } 
                case IDCANCEL:
                    {
                    if (lpfnSmartPasteEditProcS)
                        {
                        SetWindowLong(GetDlgItem(hDlg,didProgSDBEditName),GWL_WNDPROC,(LONG)lpfnDefEditProcS);
                        SetWindowLong(GetDlgItem(hDlg,didProgSDBEditNumber),GWL_WNDPROC,(LONG)lpfnDefEditProcS);
                        FreeProcInstance(lpfnSmartPasteEditProcS);
                        } /* if */
                    EndDialog(hDlg,wParam == IDOK);
                    return TRUE;
                    break;
                    }
                case didProgSDBEditName:
                case didProgSDBEditNumber:
                    {
                    if (HIWORD(lParam) != EN_CHANGE)
                        EnableWindow(GetDlgItem(hDlg,didProgSDBBtnSaveDial),GetWindowTextLength(GetDlgItem(hDlg,didProgSDBEditNumber)) > 0);
                    break;
                    } 
                case didHelp:
                    { 
                    WinHelp(hDlg,"dialer.hlp",HELP_CONTEXT,hidProgramSpeedDial);
                    break;
                    } 
                } /* switch wParam */
            break;
            }
        } /* switch */
    
    return FALSE;
    
} /* ProgramSpeedDialButtonDlgProc */

/****************************************************************************/
/* %%Function:CplConfigDlg */
/* 
    Calls into the control panel dll and brings up the config dialogs.
    Will dynamically load the control panel dll dialog stuff. 
    
    the return value from the org function (0 = success)
*/

static WORD CplConfigDlg(HWND hWnd,WORD wDlg,LPSTR lpszName)
{
    WORD  wResult;
    FARPROC  lpfnCplFunc;
    HINSTANCE hCplInstance;
    static char _based(_segname("_CODE")) gszCplApplet[] = "telephon.cpl";
    
    hCplInstance = LoadLibrary(gszCplApplet);
    if (hCplInstance <= HINSTANCE_ERROR)
        return((WORD)-1);     // oops this should not happen!
    
   /* 201 and 202 are the control panel applet location/calling card setup function ordinals */
   lpfnCplFunc = GetProcAddress(hCplInstance,MAKEINTRESOURCE((wDlg == midSetupLocation) ? 201 : 202));

   if (lpfnCplFunc)
      wResult = (*lpfnCplFunc)(hWnd,lpszName,TRUE);
   else     // should never happen!
      wResult = (WORD)-1;
            
   /* close telephon.cpl if we had it open */
   FreeLibrary(hCplInstance);
      
   return wResult;
         
} /* CplConfigDlg */

/****************************************************************************/
/* %%Function:DialingOptionDlgProc */
/*
    Dialog Proc for "Options..." menus command in "Phone" menu.
*/    

BOOL FAR PASCAL _export DialingOptionDlgProc(HWND hDlg,UINT uMsg,WPARAM wParam,LPARAM lParam)
{   

    switch (uMsg) 
        {
        case WM_INITDIALOG:
            {
            BOOL fEnable;
            BOOL fTapiAddrDLLPresent = FTapiAddrDLLPresent();
            
            CenterDlg(hDlg); 
                                                                                                                                                   
            EnableWindow(GetDlgItem(hDlg,didDialingOptionLBoxLine),fEnable = FInitLineLBox(hDlg)); 
            EnableWindow(GetDlgItem(hDlg,didDialingOptionLBoxAddress),fEnable = fEnable && FInitAddressLBox(vdws.hInst,hDlg)); 
            EnableWindow(GetDlgItem(hDlg,didDialingOptionLBoxLocation),fTapiAddrDLLPresent && (fEnable = fEnable && FInitLocationLBox(hDlg,didDialingOptionLBoxLocation))); 
            EnableWindow(GetDlgItem(hDlg,didDialingOptionLBoxCallingCard),fTapiAddrDLLPresent && (fEnable = fEnable && FInitCallingCardLBox(hDlg,didDialingOptionLBoxLocation,didDialingOptionLBoxCallingCard))); 
            EnableWindow(GetDlgItem(hDlg,IDOK),fEnable);
            
            return FALSE;
            } 
        /* Set custom colors for dialing window */
        case WM_CTLCOLOR:
            {/* do the same thing as in the Dialer main window */
            return LOWORD(DialerWndProc(hDlg,WM_CTLCOLOR,wParam,lParam));
            }
        /* action processing */
        case WM_COMMAND:
            {
            switch (wParam)
                {
                case didDialingOptionLBoxLine:
                    { 
                    if (HIWORD(lParam) != CBN_SELENDOK)
                        break;
                    
                    /* update the address list box */
                    if (!FInitAddressLBox(vdws.hInst,hDlg))
                        {/* just disable the appropriate controls */
                        WORD did;

                        for (did = didDialingOptionLBoxFirst; did <= didDialingOptionLBoxLast; ++did)
                            EnableWindow(GetDlgItem(hDlg,did),FALSE); 
                        EnableWindow(GetDlgItem(hDlg,IDOK),FALSE);
                        } /* if */
                    break;
                    }
                case didDialingOptionLBoxLocation:
                    {
                    if (HIWORD(lParam) != CBN_SELENDOK)
                        break;
                    
                    UpdateCallingCardLBoxSelection(hDlg,didDialingOptionLBoxLocation,didDialingOptionLBoxCallingCard);
                    break;
                    }
                case didDialingOptionLBoxCallingCard:
                    {     
                    if (HIWORD(lParam) != CBN_SELENDOK)
                        break;

                    UpdatePreferedCardForLocation(hDlg,didDialingOptionLBoxLocation,didDialingOptionLBoxCallingCard);
                    break;
                    }
                case IDOK:
                    {
                    UpdateDialingOptionSettings(vdws.hInst,hDlg); 
                    /* fall through is intentional */
                    } 
                case IDCANCEL:
                    {
                    EndDialog(hDlg,wParam == IDOK);
                    return TRUE;
                    } 
                case didHelp:
                    { 
                    WinHelp(hDlg,"dialer.hlp",HELP_CONTEXT,hidDialingOptions);
                    break;
                    } 
                } /* switch */
            }
        } /* switch */
    
    return FALSE;
    
} /* DialingOptionDlgProc */


/****************************************************************************/
/* %%Function:CallLogOptionDlgProc */
/*
    Dialog Proc for "Call Log Options..." menus command in "Log" menu.
*/    

BOOL FAR PASCAL _export CallLogOptionDlgProc(HWND hDlg,UINT uMsg,WPARAM wParam,LPARAM lParam)
{   
    switch (uMsg) 
        {
        case WM_INITDIALOG:
            {
            CenterDlg(hDlg);
            CheckDlgButton(hDlg,didLogOptionChkBoxIncoming,vdws.fLogIncomingCalls); 
            CheckDlgButton(hDlg,didLogOptionChkBoxOutgoing,vdws.fLogOutgoingCalls); 
    
            return FALSE;
            } 
        /* Set custom colors for dialing window */
        case WM_CTLCOLOR:
            {/* do the same thing as in the Dialer main window */
            return LOWORD(DialerWndProc(hDlg,WM_CTLCOLOR,wParam,lParam));
            }
        /* action processing */
        case WM_COMMAND:
            {
            if (wParam == didHelp)
                {
                WinHelp(hDlg,"dialer.hlp",HELP_CONTEXT,hidCallLogOptions);
                break;
                } /* if */ 
                
            if (wParam != IDOK && wParam != IDCANCEL)
                break;
            
            if (wParam == IDOK)
                {
                vdws.fLogIncomingCalls = (SendDlgItemMessage(hDlg,didLogOptionChkBoxIncoming,BM_GETCHECK,0,0L) != 0);  
                SetDialerProfileInt(vdws.hInst,ikszSecCallLogging,ikszFieldCLIncoming,vdws.fLogIncomingCalls);

                vdws.fLogOutgoingCalls = (SendDlgItemMessage(hDlg,didLogOptionChkBoxOutgoing,BM_GETCHECK,0,0L) != 0);  
                SetDialerProfileInt(vdws.hInst,ikszSecCallLogging,ikszFieldCLOutgoing,vdws.fLogOutgoingCalls);
                } /* if */  
                
            EndDialog(hDlg,wParam == IDOK);
            return TRUE;
            }
        } /* switch */
    
    return FALSE;
    
} /* CallLogOptionDlgProc */


/****************************************************************************/
/* %%Function:AboutDialerDlgProc */
/*
    dialog proc for "About Dialer..." in "Help" menu.
*/    

BOOL FAR PASCAL _export AboutDialerDlgProc(HWND hDlg,UINT uMsg,WPARAM wParam,LPARAM lParam)
{
    switch (uMsg) 
        {
        case WM_INITDIALOG:
            {
            WORD w;
            DWORD dw;   
            char sz[256];
            char szlabel[256];

            CenterDlg(hDlg);
    
            /* sets up the version number for Windows */
            w = LOWORD(GetVersion());
            GetDlgItemText(hDlg,didAboutSTextTitle,sz,sizeof(sz));
            wsprintf(szlabel,sz,w & 0xFF,HIBYTE(w) == 10 ? 1 : 0);
            SetDlgItemText(hDlg,didAboutSTextTitle,szlabel);
            
            /* sets up version number for Dialer */
            GetDlgItemText(hDlg,didAboutSTextVersion,sz,sizeof(sz));
            wsprintf(szlabel,sz,VER_MAJOR,VER_MINOR,VER_BUILD);
            SetDlgItemText(hDlg,didAboutSTextVersion,szlabel);
            
            /* set windows mode information */
            dw = GetWinFlags();
            if (dw & WF_ENHANCED)
                w = ikszAboutModeEnhanced;
            else if (dw & WF_STANDARD)
                w = ikszAboutModeStandard;
            else if (dw & WF_WLO)
                w = ikszAboutModeWLO;
            else
                w = ikszAboutModeUndef;
            LoadString(vdws.hInst,w,sz,sizeof(sz));
            SetDlgItemText(hDlg,didAboutSTextWinMode,sz);
            
            /* get free memory information */
            GetDlgItemText(hDlg,didAboutSTextFreeMem,sz,sizeof(sz));
            wsprintf(szlabel,sz,GetFreeSpace(0)>>10);
            SetDlgItemText(hDlg,didAboutSTextFreeMem,szlabel);
            
            /* get free resources information */
            w = GetFreeSystemResources(0);
            GetDlgItemText(hDlg,didAboutSTextResource,sz,sizeof(sz));
            wsprintf(szlabel,sz,w);
            SetDlgItemText(hDlg,didAboutSTextResource,szlabel);
    
            return TRUE;
            }
        /* Set custom colors for dialing window */
        case WM_CTLCOLOR:
            {/* do the same thing as in the Dialer main window */
            return LOWORD(DialerWndProc(hDlg,WM_CTLCOLOR,wParam,lParam));
            }
        case WM_COMMAND: 
            {
            if (wParam != IDOK)
                break;
                
            EndDialog(hDlg,TRUE);
            return TRUE;
            }
        } /* switch */
    
    return FALSE;  
    
} /* AboutDialerDlgProc */

/****************************************************************************/
/* %%Function:LineInUseDlgProc */
/*
    dialog proc for "Line In Use" dialog.
*/    

BOOL FAR PASCAL _export LineInUseDlgProc(HWND hDlg,UINT uMsg,WPARAM wParam,LPARAM lParam)
{
    switch (uMsg) 
        {
        case WM_INITDIALOG:
            {
            CenterDlg(hDlg);
            return TRUE;
            }
        /* Set custom colors for dialing window */
        case WM_CTLCOLOR:
            {/* do the same thing as in the Dialer main window */
            return LOWORD(DialerWndProc(hDlg,WM_CTLCOLOR,wParam,lParam));
            }
        case WM_COMMAND: 
            {
            if (wParam == didHelp)
                {
                WinHelp(hDlg,"dialer.hlp",HELP_CONTEXT,hidLineInUse);
                break;
                } /* if */ 

            if (wParam != IDOK)
                break;
                
            EndDialog(hDlg,TRUE);
            return TRUE;
            }
        } /* switch */
    
    return FALSE;  
    
} /* LineInUseDlgProc */

/****************************************************************************/
/* %%Function:DisplayLineInUseDlg */
/*
    shows the modal dialog "Line In Use" to inform the user that the line is busy.
    called after lineMakeCall fails.
*/    

void FAR DisplayLineInUseDlg()

{                                                                            
    
    DoDialog(LineInUseDlgProc,dlgLineInUse,vdws.hInst,vdws.hdlgCallStatus,0);
    
} /* DisplayLineInUseDlg */

/****************************************************************************/
/* %%Function:DialingPromptDlgProc */
/*
    dialog proc for "Dialing Prompt" dialog.
*/    

BOOL FAR PASCAL _export DialingPromptDlgProc(HWND hDlg,UINT uMsg,WPARAM wParam,LPARAM lParam)
{
    switch (uMsg) 
        {
        case WM_INITDIALOG:
            {
            int iksz;
            char szText[cchSzMax];
            
            CenterDlg(hDlg); 
            
            switch (lParam)
                {
                case LINEERR_DIALBILLING: 
                    {
                    iksz = ikszDialingPromptSTextBilling;
                    break;
                    }
                case LINEERR_DIALDIALTONE: 
                    {
                    iksz = ikszDialingPromptSTextTone;
                    break;
                    }
                case LINEERR_DIALQUIET: 
                    {
                    iksz = ikszDialingPromptSTextQuite;
                    break;
                    }
                default: 
                    {
                    iksz = ikszDialingPromptSTextPrompt;
                    break;
                    }
                } /* switch */ 
            LoadString(vdws.hInst,iksz,szText,sizeof(szText));
            SetDlgItemText(hDlg,didDialingPromptSText,szText);
            return TRUE;
            }
        /* Set custom colors for dialing window */
        case WM_CTLCOLOR:
            {/* do the same thing as in the Dialer main window */
            return LOWORD(DialerWndProc(hDlg,WM_CTLCOLOR,wParam,lParam));
            }
        /* action processing */
        case WM_COMMAND:
            {
            if (wParam == didHelp)
                {
                WinHelp(hDlg,"dialer.hlp",HELP_CONTEXT,hidDialingPaused);
                break;
                } /* if */ 

            EndDialog(hDlg,wParam == IDOK);
            if (wParam == didDialingPromptBtnOption || wParam == IDCANCEL) 
                { 
                int did = (wParam == didDialingPromptBtnOption) ? didCallStatusBtnOption : IDCANCEL; 
                PostMessage(vdws.hdlgCallStatus,WM_COMMAND,did,MAKELPARAM(GetDlgItem(vdws.hdlgCallStatus,did),BN_CLICKED));
                AbortTapiCallInProgress();
                } /* if */
            break; 
            }
        } /* switch */
    
    return FALSE;  
    
} /* DialingPromptDlgProc */

/****************************************************************************/
/* %%Function:DisplayDialingPromptDlg */
/*
    shows the modal dialog "Dialing Prompt" to prompt the user to continue the dialing
    process when appropriate. dwLineErrCode is one of the following 4 values :
    LINEERR_DIALBILLING, LINEERR_DIALDIALTONE, LINEERR_DIALQUIET, and LINEERR_DIALPROMPT.
*/    

void FAR DisplayDialingPromptDlg(DWORD dwLineErrCode)

{
    DoDialog(DialingPromptDlgProc,dlgDialingPrompt,vdws.hInst,vdws.hdlgCallStatus,dwLineErrCode);
    
} /* DisplayDialingPromptDlg */


/****************************************************************************/
/* %%Function:ChangeOptionDlgProc */
/*
    dialog proc for "Change Options and Redial" dialog.
*/    

BOOL FAR PASCAL _export ChangeOptionDlgProc(HWND hDlg,UINT uMsg,WPARAM wParam,LPARAM lParam)
{
    switch (uMsg) 
        {
        case WM_INITDIALOG:
            {
            char szNumber[cchSzMax];
            DWORD dwTranslateResults;
            BOOL fTollListBtnEnabled;
            BOOL fTapiAddrDLLPresent = FTapiAddrDLLPresent();
            
            CenterDlg(hDlg); 

            GetCurCallTranslatedNumber(szNumber,sizeof(szNumber),&dwTranslateResults); 
                    
            EnableWindow(GetDlgItem(hDlg,didChangeOptionLBoxLocation),fTapiAddrDLLPresent && FInitLocationLBox(hDlg,didChangeOptionLBoxLocation)); 
            EnableWindow(GetDlgItem(hDlg,didChangeOptionLBoxCallingCard),fTapiAddrDLLPresent && FInitCallingCardLBox(hDlg,didChangeOptionLBoxLocation,didChangeOptionLBoxCallingCard));
            EnableWindow(GetDlgItem(hDlg,didChangeOptionBtnTollList),fTollListBtnEnabled = fTapiAddrDLLPresent
                && ((dwTranslateResults & LINETRANSLATERESULT_INTOLLLIST) || (dwTranslateResults & LINETRANSLATERESULT_NOTINTOLLLIST)));
            if (fTollListBtnEnabled) 
                {
                char szText[cchSzMax];
                
                LoadString(vdws.hInst,(dwTranslateResults & LINETRANSLATERESULT_INTOLLLIST) ? ikszChangeOptionTollListRemove : ikszChangeOptionTollListAdd,szText,sizeof(szText));
                SetDlgItemText(hDlg,didChangeOptionBtnTollList,szText);
                } /* if */

            SetDlgItemText(hDlg,didChangeOptionEditBoxNumber,szNumber);
            SetFocus(GetDlgItem(hDlg,didChangeOptionEditBoxNumber));
            SendDlgItemMessage(hDlg,didChangeOptionEditBoxNumber,EM_SETSEL,0,MAKELPARAM(0,-1)); 
            return FALSE;
            }
        /* Set custom colors for dialing window */
        case WM_CTLCOLOR:
            {/* do the same thing as in the Dialer main window */
            return LOWORD(DialerWndProc(hDlg,WM_CTLCOLOR,wParam,lParam));
            }
        /* action processing */
        case WM_COMMAND:
            {
            switch (wParam)
                {
                case didChangeOptionLBoxLocation:
                    {
                    if (HIWORD(lParam) != CBN_SELENDOK)
                        break;
                    
                    UpdateCallingCardLBoxSelection(hDlg,didChangeOptionLBoxLocation,didChangeOptionLBoxCallingCard);
                    break;
                    }
                case didChangeOptionLBoxCallingCard:
                    {     
                    if (HIWORD(lParam) != CBN_SELENDOK)
                        break;

                    UpdatePreferedCardForLocation(hDlg,didChangeOptionLBoxLocation,didChangeOptionLBoxCallingCard);
                    break;
                    }
                case IDOK:
                    {
                    char szNumber[cchSzMax]; 
                    
                    UpdateCurrentLocation(hDlg,didChangeOptionLBoxLocation);
                    UpdateCurrentCallingCard(vdws.hInst,hDlg,didChangeOptionLBoxCallingCard);
                    GetDlgItemText(hDlg,didChangeOptionEditBoxNumber,szNumber,sizeof(szNumber)); 
                    replaceNumber(szNumber);
                    
                    ShowWindow(hDlg,SW_HIDE);
                    EndDialog(hDlg,TRUE); 
                    PostMessage(vdws.hwnd,WM_COMMAND,didDialerBtnDial,MAKELPARAM(GetDlgItem(vdws.hwnd,didDialerBtnDial),BN_CLICKED));
                    return TRUE;
                    } 
                case IDCANCEL:
                    {
                    EndDialog(hDlg,FALSE);
                    return TRUE;
                    }
                case didChangeOptionBtnTollList:
                    {
                    char szNumber[cchSzMax];
                    DWORD dwTranslateResults;
                    char szText[cchSzMax];
                    
                    AddRemoveCurCallFromTollList();                  
                    GetCurCallTranslatedNumber(szNumber,sizeof(szNumber),&dwTranslateResults);
                     
                    LoadString(vdws.hInst,(dwTranslateResults & LINETRANSLATERESULT_INTOLLLIST) ? ikszChangeOptionTollListRemove : ikszChangeOptionTollListAdd,szText,sizeof(szText));
                    SetDlgItemText(hDlg,didChangeOptionBtnTollList,szText);
    
                    SetDlgItemText(hDlg,didChangeOptionEditBoxNumber,szNumber);
                    SetFocus(GetDlgItem(hDlg,didChangeOptionEditBoxNumber));
                    SendDlgItemMessage(hDlg,didChangeOptionEditBoxNumber,EM_SETSEL,0,MAKELPARAM(0,-1)); 
                    break;
                    }
                case didHelp:
                    { 
                    WinHelp(hDlg,"dialer.hlp",HELP_CONTEXT,hidChangeOptions);
                    break;
                    } 
                } /* switch */
            }
        } /* switch */
    
    return FALSE;  
    
} /* ChangeOptionDlgProc */


/****************************************************************************/
/* %%Function:CallStatusProc */
/*
    dialog proc for the call status dialog.
*/    

BOOL FAR PASCAL _export CallStatusProc(HWND hDlg,UINT uMsg,WPARAM wParam,LPARAM lParam)
{
    switch (uMsg) 
        {
        case WM_DESTROY:
            {
            vdws.hdlgCallStatus = NULL;
            return TRUE;
            } 
        case WM_CLOSE: 
            {
            SendMessage(hDlg,WM_COMMAND,IDCANCEL,MAKELPARAM(GetDlgItem(hDlg,IDCANCEL),BN_CLICKED));
            return TRUE;
            }  
        /* Set custom colors for dialing window */
        case WM_CTLCOLOR:
            {/* do the same thing as in the Dialer main window */
            return LOWORD(DialerWndProc(hDlg,WM_CTLCOLOR,wParam,lParam));
            }
        case WM_COMMAND: 
            {
            if (wParam == didCallStatusEditBoxLogName)
                break;
            else if (wParam == didHelp)
                {
                WinHelp(hDlg,"dialer.hlp",HELP_CONTEXT,vdws.hidCallStatus);
                break;
                } /* else if */
            
            /* since the line could have been busy and we didn't even get a chance to get connected */
            if (IsWindowVisible(GetDlgItem(vdws.hdlgCallStatus,didCallStatusEditBoxLogName)))
                {/* we need to shrink the window back to its original height etc. */
                HWND hwndCtrl;
                RECT rc, rc2;
                WORD dyd, dxdBtn, dydBtn;
                char szName[cchSzMax];
                
                GetWindowRect(GetDlgItem(vdws.hdlgCallStatus,didCallStatusSTextLogName),&rc);
                GetWindowRect(GetDlgItem(vdws.hdlgCallStatus,didCallStatusEditBoxLogName),&rc2);
                dyd = rc2.bottom - rc.top + 7;
                
                /* move the OK button up */
                GetWindowRect(hwndCtrl = GetDlgItem(vdws.hdlgCallStatus,IDOK),&rc);
                dxdBtn = rc.right - rc.left;
                dydBtn = rc.bottom - rc.top;
                ScreenToClient(GetParent(hwndCtrl),(POINT *)&rc); 
                MoveWindow(hwndCtrl,rc.left,rc.top - dyd,dxdBtn,dydBtn,FALSE);
                
                /* move the Help button back */
                MoveWindow(GetDlgItem(vdws.hdlgCallStatus,didHelp),vdws.rcCallStatusHelpBtn.left,
                vdws.rcCallStatusHelpBtn.top,dxdBtn,dydBtn,FALSE);
                
                /* shrink the call status dialog by dyd */
                dyd -= dydBtn;
                GetWindowRect(vdws.hdlgCallStatus,&rc);
                MoveWindow(vdws.hdlgCallStatus,rc.left,rc.top,rc.right - rc.left,rc.bottom - rc.top - dyd,TRUE);
                
                /* update the name entered */ 
                GetDlgItemText(vdws.hdlgCallStatus,didCallStatusEditBoxLogName,szName,sizeof(szName));
                SetCurCallName(szName);
                } /* if */
                    
            ShowWindow(hDlg,SW_HIDE); 
            EnableWindow(vdws.hwnd,TRUE); 
            if (FCallInProgress()) /* drop the call */ 
            	SendMessage(vdws.hwnd,WM_COMMAND,didDialerBtnDial,MAKELPARAM(GetDlgItem(vdws.hwnd,didDialerBtnDial),BN_CLICKED));
            if (wParam == didCallStatusBtnOption)
                DoDialog(ChangeOptionDlgProc,dlgChangeOption,vdws.hInst,vdws.hwnd,0);
            return TRUE;
            }
        } /* switch */
    
    return FALSE;  
    
} /* CallStatusProc */

/****************************************************************************/
/* %%Function:FCreateCallStatusDlg */
/*
    create the dialing dialog and initialize vhdlgDialing.
*/

static BOOL FCreateCallStatusDlg()

{
    DLGPROC dlgproc;
    
    dlgproc = (DLGPROC)MakeProcInstance((FARPROC)CallStatusProc,vdws.hInst);
    vdws.hdlgCallStatus = CreateDialog(vdws.hInst,MAKEINTRESOURCE(dlgCallStatus),vdws.hwnd,dlgproc);
    if (vdws.hdlgCallStatus == NULL)
        FreeProcInstance((FARPROC)dlgproc);

    GetWindowRect(GetDlgItem(vdws.hdlgCallStatus,didHelp),&(vdws.rcCallStatusHelpBtn));
    ScreenToClient(vdws.hdlgCallStatus,(POINT FAR *)&(vdws.rcCallStatusHelpBtn));
    ScreenToClient(vdws.hdlgCallStatus,(POINT FAR *)&(vdws.rcCallStatusHelpBtn) + 1);
    return (vdws.hdlgCallStatus != NULL);  
        
} /* FCreateCallStatusDlg */    

/****************************************************************************/
/* %%Function:StrReplace */
/* 
    upon entry, szOut contains szTok. upon exit, szOut's szTok portion is replaced
    with szCpntent. The maximum number of characters that can be used is cchContentMax.
    if szContent is longer than cchContentMax, it is truncated and "..." is added at the
    end.
*/

static void StrReplace(LPSTR szTok,LPCSTR szContent,LPSTR szOut,int cchContentMax)

{
    LPSTR lpsz;
    char szBuffer[cchSzMax];
    int ich, cchContent;
    
    _fmemmove(szBuffer,szOut,lstrlen(szOut) + 1);
    
    /* figure out where the token starts in szOut */
    lpsz = _fstrstr(szOut,szTok);
    ich = lpsz - szOut;

    /* copy szContent to where szTok is in szOut */
    cchContent = lstrlen(szContent);
    _fmemmove(szOut + ich,szContent,min(cchContent,cchContentMax));
    if (cchContent > cchContentMax)
        {
        cchContent = cchContentMax;
        szOut[ich + cchContent - 1] = '.';
        szOut[ich + cchContent - 2] = '.';
        szOut[ich + cchContent - 3] = '.';
        } /* if */
    
    _fmemmove(szOut + ich + cchContent,szBuffer + ich + lstrlen(szTok),strlen(szBuffer) - ich - lstrlen(szTok) + 1);
        
} /* StrReplace */

/****************************************************************************/
/* %%Function:StrReplaceInSTextItem */
/* 
    same as StrReplace except we don't want the string to be displayed in hwndCtrl is more than one
    line long. calls StrReplace to do the real work.
*/

static void StrReplaceInSTextItem(LPSTR szTok,LPCSTR szContent,LPSTR szOut,HWND hwndCtrl)

{
    int cchAvail;
    TEXTMETRIC tm;
    HDC  hdcCtrl;
    RECT rcText;

    /* figure out the maximum number of chars we can display */
    hdcCtrl = GetDC(hwndCtrl); 
    GetTextMetrics(hdcCtrl,&tm);
    ReleaseDC(hwndCtrl,hdcCtrl); 
    
    GetClientRect(hwndCtrl,&rcText);
    cchAvail = min((rcText.right - rcText.left)/tm.tmAveCharWidth - 2,cchSzMax-1);

    cchAvail -= (lstrlen(szOut) - lstrlen(szTok));
    StrReplace(szTok,szContent,szOut,cchAvail);
    
} /* StrReplaceInSTextItem */

/****************************************************************************/
/* %%Function:SetUpDialingText */
/* 
    sets up the text in didCallStatusSTextNameNum in the call status dialog.        
*/

static void SetUpDialingText(int ikszFormat,LPCSTR szName,LPCSTR szNum,LPSTR szText,int cchSzText);
static void SetUpDialingText(int ikszFormat,LPCSTR szName,LPCSTR szNum,LPSTR szText,int cchSzText)

{   
    HWND hwndCtrl = GetDlgItem(vdws.hdlgCallStatus,didCallStatusSTextNameNum);
    char szNameFormat[cchSzMax];
    char szNumFormat[cchSzMax];
    char szEmpty[1] = {0};
    int cchAvail, cchNameFormat, cchNum;
    TEXTMETRIC tm;
    HDC  hdcCtrl;
    RECT rcText;

    /* figure out the maximum number of chars we can display */
    hdcCtrl = GetDC(hwndCtrl); 
    GetTextMetrics(hdcCtrl,&tm);
    ReleaseDC(hwndCtrl,hdcCtrl); 
    
    GetClientRect(hwndCtrl,&rcText);
    cchAvail = min((rcText.right - rcText.left)/tm.tmAveCharWidth - 2,cchSzText-1);
     
    /* load in the strings we use */
    LoadString(vdws.hInst,ikszNameFormat,szNameFormat,sizeof(szNameFormat));
    LoadString(vdws.hInst,ikszNumFormat,szNumFormat,sizeof(szNumFormat));
    LoadString(vdws.hInst,ikszFormat,szText,cchAvail);
    cchNameFormat = strlen(szNameFormat);
    cchNum = lstrlen(szNum);
    
    /* don't even have enough space for number alone. forget about it */
    if (cchAvail < (int)strlen(szText) - cchNameFormat - (int)strlen(szNumFormat) + cchNum)
        {/* just remove szNumFormat and szNameFormat from szFormat */
        StrReplace(szNumFormat,szEmpty,szText,1);
        StrReplace(szNameFormat,szEmpty,szText,1);
        return;
        } /* if */

    /* replace the szNumFormat portion in szFormat with the actual address szNum */
    StrReplace(szNumFormat,szNum,szText,cchNum);
    
    /* no space for name at all, forget it */
    if (cchAvail - (lstrlen(szText) - cchNameFormat) <= 3)
        {
        StrReplace(szNameFormat,szEmpty,szText,1);
        return;
        } /* if */
    
    /* replace the szNameFormat portion in szText with the actual name szName */
    StrReplace(szNameFormat,szName,szText,cchAvail - (lstrlen(szText) - cchNameFormat));
    
} /* SetUpDialingText */

/****************************************************************************/
/* %%Function:ShowCallStatusDlg */
/*
    create the dialing dialog and initialize vhdlgDialing.
*/

void ShowCallStatusDlg(LPCSTR szName,LPCSTR szNumber,LPCSTR szLocation,LPCSTR szCallingCard,LPCSTR szTranslatedNumber)

{   HWND hwndCtrl;
    char szText[cchSzMax], szText2[cchSzMax];
   
    /* set up the dialog title */ 
    LoadString(vdws.hInst,ikszCallStatusDialing,szText,sizeof(szText));
    SetWindowText(vdws.hdlgCallStatus,szText);

    /* set up the name/number text */
    SetUpDialingText(ikszCallStatusNameFormatDialing,szName,szNumber,szText,sizeof(szText)); 
    SetDlgItemText(vdws.hdlgCallStatus,didCallStatusSTextNameNum,szText); 
    
    /* set up location text */
    LoadString(vdws.hInst,ikszCallStatusLocationFormat,szText,sizeof(szText));
    LoadString(vdws.hInst,ikszLocationFormat,szText2,sizeof(szText2));
    StrReplaceInSTextItem(szText2,szLocation,szText,GetDlgItem(vdws.hdlgCallStatus,didCallStatusSTextLocation));
    SetDlgItemText(vdws.hdlgCallStatus,didCallStatusSTextLocation,szText); 
    
    /* set up calling card text */
    LoadString(vdws.hInst,ikszCallStatusCallingCardFormat,szText,sizeof(szText));
    LoadString(vdws.hInst,ikszCallingCardFormat,szText2,sizeof(szText2));
    StrReplaceInSTextItem(szText2,szCallingCard,szText,GetDlgItem(vdws.hdlgCallStatus,didCallStatusSTextCallingCard));
    SetDlgItemText(vdws.hdlgCallStatus,didCallStatusSTextCallingCard,szText); 
    
    /* set up translated number */
    LoadString(vdws.hInst,ikszCallStatusTranslatedNumberFormat,szText,sizeof(szText));
    LoadString(vdws.hInst,ikszNumFormat,szText2,sizeof(szText2)); 
    StrReplaceInSTextItem(szText2,szTranslatedNumber,szText,GetDlgItem(vdws.hdlgCallStatus,didCallStatusSTextTranslatedNum));
    SetDlgItemText(vdws.hdlgCallStatus,didCallStatusSTextTranslatedNum,szText); 
            
    /* show/hide the appropriate buttons */
    ShowWindow(hwndCtrl = GetDlgItem(vdws.hdlgCallStatus,IDOK),SW_HIDE);
    EnableWindow(hwndCtrl,FALSE);  
    ShowWindow(hwndCtrl = GetDlgItem(vdws.hdlgCallStatus,didCallStatusEditBoxLogName),SW_HIDE);
    EnableWindow(hwndCtrl,FALSE);  
    ShowWindow(hwndCtrl = GetDlgItem(vdws.hdlgCallStatus,didCallStatusSTextLogName),SW_HIDE);

    ShowWindow(hwndCtrl = GetDlgItem(vdws.hdlgCallStatus,IDCANCEL),SW_SHOW);
    EnableWindow(hwndCtrl,TRUE);  
    ShowWindow(hwndCtrl = GetDlgItem(vdws.hdlgCallStatus,didCallStatusBtnOption),SW_SHOW);
    EnableWindow(hwndCtrl,TRUE);  
    
    vdws.hidCallStatus = hidDialing;
    
    /* show the dialog */
    EnableWindow(vdws.hwnd,FALSE);
    CenterDlg(vdws.hdlgCallStatus); 
    ShowWindow(vdws.hdlgCallStatus,SW_SHOW); 
    UpdateWindow(vdws.hdlgCallStatus);
        
} /* ShowCallStatusDlg */

/****************************************************************************/
/* %%Function:UpdateCallStatusDlg */
/*
    update the call status dialog display to reflect the current call state.
    fCallIsActive == TRUE iff the call has just become active. Otherwise, the 
    remote end has just disconnected.
*/

void UpdateCallStatusDlg(BOOL fCallIsActive,LPCSTR szName,LPCSTR szNumber)

{   
    char szText[cchSzMax];
   
    if (!IsWindowVisible(vdws.hdlgCallStatus))
        return;
        
    /* set up the dialog title */ 
    LoadString(vdws.hInst,fCallIsActive ? ikszCallStatusConntected : ikszCallStatusDisconnected,szText,sizeof(szText));
    SetWindowText(vdws.hdlgCallStatus,szText);

    /* set up the name/number text */
    SetUpDialingText(fCallIsActive ? ikszCallStatusNameFormatConnected : ikszCallStatusNameFormatDisconnected,szName,szNumber,szText,sizeof(szText)); 
    SetDlgItemText(vdws.hdlgCallStatus,didCallStatusSTextNameNum,szText); 

    /* show/hide the appropriate buttons */
    if (fCallIsActive)
        {
        HWND hwndCtrl;
        RECT rc, rc2;
        WORD dyd, dxdBtn, dydBtn, xdCancel, xdOption;
        
        /* hide away the Cancel/Change Option buttons */
        ShowWindow(hwndCtrl = GetDlgItem(vdws.hdlgCallStatus,IDCANCEL),SW_HIDE);
        EnableWindow(hwndCtrl,FALSE);  
        ShowWindow(hwndCtrl = GetDlgItem(vdws.hdlgCallStatus,didCallStatusBtnOption),SW_HIDE);
        EnableWindow(hwndCtrl,FALSE); 
        
        /* ishow the edit box */
        ShowWindow(hwndCtrl = GetDlgItem(vdws.hdlgCallStatus,didCallStatusSTextLogName),SW_SHOW);
        GetWindowRect(hwndCtrl,&rc);
        ShowWindow(hwndCtrl = GetDlgItem(vdws.hdlgCallStatus,didCallStatusEditBoxLogName),SW_SHOW);
        GetWindowRect(hwndCtrl,&rc2);
        dyd = rc2.bottom - rc.top + 7;
            
        EnableWindow(hwndCtrl,TRUE); 
        SetFocus(hwndCtrl);  
        if (szName)
            {
            SetDlgItemText(vdws.hdlgCallStatus,didCallStatusEditBoxLogName,szName);
            SendDlgItemMessage(vdws.hdlgCallStatus,didCallStatusEditBoxLogName,EM_SETSEL,0,MAKELPARAM(0,-1));
            } /* if */

        GetWindowRect(hwndCtrl = GetDlgItem(vdws.hdlgCallStatus,IDCANCEL),&rc);
        ScreenToClient(GetParent(hwndCtrl),(POINT *)&rc);
        xdCancel = rc.left; 
        GetWindowRect(hwndCtrl = GetDlgItem(vdws.hdlgCallStatus,didCallStatusBtnOption),&rc);
        ScreenToClient(GetParent(hwndCtrl),(POINT *)&rc);
        xdOption = rc.left; 

        /* move the OK button and the Help button */
        GetWindowRect(hwndCtrl = GetDlgItem(vdws.hdlgCallStatus,IDOK),&rc);
        dxdBtn = rc.right - rc.left;
        dydBtn = rc.bottom - rc.top;
        ScreenToClient(GetParent(hwndCtrl),(POINT *)&rc); 
        MoveWindow(hwndCtrl,xdCancel,rc.top + dyd,dxdBtn,dydBtn,FALSE);
        MoveWindow(GetDlgItem(vdws.hdlgCallStatus,didHelp),xdOption,rc.top + dyd,dxdBtn,dydBtn,TRUE);

        /* enlarge the call status dialog by dyd */
        dyd -= dydBtn;
        GetWindowRect(vdws.hdlgCallStatus,&rc);
        MoveWindow(vdws.hdlgCallStatus,rc.left,rc.top,rc.right - rc.left,rc.bottom - rc.top + dyd,TRUE);
            
        ShowWindow(hwndCtrl = GetDlgItem(vdws.hdlgCallStatus,IDOK),SW_SHOW);
        EnableWindow(hwndCtrl,TRUE);  
        } /* if */ 

    vdws.hidCallStatus = (fCallIsActive ? hidActiveCall : hidDisconnectedCall);

    LoadString(vdws.hInst,fCallIsActive ? ikszCallStatusBtnHangup : ikszCallStatusBtnOK,szText,sizeof(szText));
    SetDlgItemText(vdws.hdlgCallStatus,IDOK,szText); 

} /* UpdateCallStatusDlg */

/****************************************************************************/
/* %%Function:HideCallStatusDlg */
/*
    called when the call status dialog is visible. it hides it by issuing the
    "Cancel" button command. 
*/

void HideCallStatusDlg()

{ 
    SendMessage(vdws.hdlgCallStatus,WM_COMMAND,IDCANCEL,MAKELPARAM(GetDlgItem(vdws.hdlgCallStatus,IDCANCEL),BN_CLICKED));
    
} /* HideCallStatusDlg */

/****************************************************************************/
/* call logging related functions */
/****************************************************************************/
/* %%Function:CchFile */
/*
    Returns the length of hFile.
*/

static long CchFile(int hFile)
{
    long lCurrentPos = _lseek(hFile,0L,SEEK_CUR);
    long lFileLength = _lseek(hFile,0L,SEEK_END);
    
    _lseek(hFile,lCurrentPos,SEEK_SET);
    return lFileLength;
    
} /* CchFile */

/****************************************************************************/
/* %%Function:CchFreeLogBytes */
/*
    Frees log entries to make room for trailing cchLogFileMax bytes of file.
    Any log entry whose data doesn't come from within cchLogFileMax
    bytes of lastByte is tossed. We return total number of bytes tossed.  
    if fResetFile is true, then all entries are tossed.
*/

static LONG CchFreeLogBytes(HWND hwndList,LONG eofByte,BOOL fResetFile)
{
    LONG cchFreed = 0;
    DWORD ifc;
    DWORD lineLen;
    DWORD bound = (eofByte < cchLogFileMax) ? 0 : (eofByte - cchLogFileMax);
    
    if (bound != 0 || fResetFile) 
        {
        LONG item;
        WORD nItems;
        
        nItems = (WORD) SendMessage(hwndList,LB_GETCOUNT,0,0);
        for (item = (LONG)nItems-1; item >= 0; item--) 
            {
            lineLen = SendMessage(hwndList,LB_GETTEXTLEN,(WPARAM)item,0);
            if (fResetFile)
                cchFreed += lineLen + 2;
            else 
                {
                ifc = SendMessage(hwndList,LB_GETITEMDATA,(WPARAM)item,0);
                if (ifc < bound) {
                    SendMessage(hwndList,LB_DELETESTRING,(WPARAM)item,0);
                    cchFreed += lineLen+2;
                }
            } /* for */  
        }  /* if */
        
        if (fResetFile)
            SendMessage(hwndList,LB_RESETCONTENT,0,0);
        }
        
    return cchFreed;
    
} /* CchFreeLogBytes */

/****************************************************************************/
/* %%Function:FPrependLogLine */
/* 
    Prepends a line to the list. returns TRUE if successful.
*/

static BOOL FPrependLogLine(HWND hwndList,LPSTR szLine,LONG ifc,WORD iItemInsert)
{
    DWORD item;
    LPSTR lpch, lpchEnd;
    
    /* we care to trim off the extra characters in the name field */
    lpch = _fstrchr(szLine,'\t');      
    if (lpch)
        lpchEnd = _fstrchr(lpch + 1,'\t'); 
    if (lpch && lpchEnd && (lpchEnd - lpch) > vdws.rgcchFieldMax[1])
        {
        lpch += vdws.rgcchFieldMax[1];
        *(lpch - 1) = *(lpch - 2) = *(lpch - 3) = '.';
        _fmemmove(lpch,lpchEnd,lstrlen(lpchEnd) + 1);
        } /* if */
    
    item = SendMessage(hwndList,LB_INSERTSTRING,iItemInsert,(LPARAM)szLine);
    if (item == LB_ERR) 
        FALSE;
        
    SendMessage(hwndList,LB_SETITEMDATA,(WPARAM)item,(LPARAM)ifc);
    return TRUE;
      
} /* FPrependLogLine */

/****************************************************************************/
/* %%Function:FcOpenCallLog */
/*  
    Opens the call log file, either for read/share or write/exclusive, 
    depending on fWrite. creates the log file if it does not exist.
*/

static int FcOpenCallLog(BOOL fWrite)
{
    char szBuffer[256];
    int cchBuffer;
    int fcLogFile;
    int retry = 2;
    int openMode;
    
    GetWindowsDirectory(szBuffer,144);
    cchBuffer = strlen(szBuffer);
    if (szBuffer[cchBuffer-1] != '\\') 
        {
        szBuffer[cchBuffer++] = '\\';
        szBuffer[cchBuffer] = '\0';
        }  /* if */
    
    LoadString(vdws.hInst,ikszCallLogFile,&szBuffer[cchBuffer],sizeof(szBuffer) - cchBuffer); 
    while (retry--) 
        { 
        openMode = fWrite ? (OF_READWRITE | OF_SHARE_EXCLUSIVE) : (OF_READ | OF_SHARE_DENY_WRITE);
        fcLogFile = _lopen((LPSTR)szBuffer,openMode);
        if (fcLogFile == HFILE_ERROR && retry) 
            {
            if (_access(szBuffer,0) < 0 && errno == ENOENT) 
                {
                fcLogFile = _lcreat((LPSTR)szBuffer,0 /* NORMAL */); 
                if (fcLogFile != HFILE_ERROR)
                    fcLogFile = _lclose(fcLogFile);
                } /* if */
            if (fcLogFile == HFILE_ERROR)
                retry = 0;
            } 
        else
            retry = 0;
        } /* while */
        
    return fcLogFile;
    
} /* FcOpenCallLog */

/****************************************************************************/
/* %%Function:FReadCallLog */
/*
    Reads the call log (starting on line boundary) into list window in CallLog 
    Dialog. if fResetFile is TRUE, the local cache is ignored and the file is
    read in. if fResetFile is FALSE there are two cases:
    A.  We've added to log so read the new piece at the end of file (possibly
        need to toss oldest rows to make room
    B.  We've removed some rows and adjusted the cache.  Might need to read in
        some additional older rows.  

    Need to do following:
    If file is longer than MAX_SHOWLOG, a message is put at the top of
    the edit window followed by MAX_SHOWLOG-length of the message-2
    bytes (2 added for \n\r).
*/

static BOOL FReadCallLog(HWND hwndList,BOOL fResetFile)
{
    BOOL fres = FALSE;
    char szBuffer[cchLogLineMax];
    LPSTR readBuff = NULL;
    int fcLogFile = HFILE_ERROR;
    LONG fLength;
    LONG toRead;
    UINT nRead;
    LONG readAt;
    LPSTR pc,psrc,pbnd;
    BOOL fInvalid = FALSE;
    int cchBuffer;
    LONG filePosition;
    BOOL readOldRows = FALSE;  
    WORD cEntryTotal;
    WORD at = 0;
    
    /* open the log file */ 
    fcLogFile = FcOpenCallLog(FALSE);
    if (fcLogFile == HFILE_ERROR)
        goto LDone;

    /*  Read the new (unread) portion of the file, starting from the end.
        Stop when cchLogFileMax bytes are in list or when we've run out of
        new data.  Toss oldest rows (those at end of list) if necessary
        to make room.  The intent here is to limit memory taken up by
        log and to avoid reading portions of the log that haven't changed.
    */
    if (fResetFile)
        vdws.logLengthPrev = 0;
    fLength = CchFile(fcLogFile);
    toRead = fLength - vdws.logLengthPrev;
    
    /*  If we have current data in window up to current eof, check if
        we need to read some old rows that will now be viewable due to
        deletion of some rows via cut or delete operation.
    */
    if (toRead == 0) 
        {
        cEntryTotal = (WORD) SendMessage(hwndList,LB_GETCOUNT,0,0);
        if (cEntryTotal == 0)
            filePosition = vdws.logLengthPrev;
        else
            filePosition = SendMessage(hwndList,LB_GETITEMDATA,(WPARAM)(cEntryTotal-1),0);
        if (vdws.curLogData >= cchLogFileMax || filePosition == 0)
            goto exitSuccess;
            
        readOldRows = TRUE;
        toRead = cchLogFileMax - vdws.curLogData;
        if (toRead > filePosition)
            toRead = filePosition;
        readAt = filePosition - toRead;
        at = cEntryTotal;
        }  /* if */
        
    if (toRead > cchLogFileMax)
        toRead = cchLogFileMax;
    readBuff = (LPSTR)GlobalAllocPtr(GHND,toRead);
    if (readBuff == NULL)
        goto LDone;
    if (!readOldRows)
        readAt = fLength-toRead;
    if (_lseek(fcLogFile,readAt,SEEK_SET) < 0)
        goto LDone;
    nRead = _lread(fcLogFile,readBuff,(UINT)toRead);
    if (nRead != (UINT)toRead)
        goto LDone;
    
    fInvalid = TRUE;
    SendMessage(hwndList,WM_SETREDRAW,0,0);
        
    /* Free enough items to allow us to add vdws.curLogData bytes of new log */
    if (!readOldRows)
        vdws.curLogData -= CchFreeLogBytes(hwndList,nRead+readAt,fResetFile);
    pbnd = readBuff+toRead;
    pc = readBuff;

    /* Scan to a line start if we just read starting at random place in file */
    if (readAt != 0 && (readOldRows || readAt != vdws.logLengthPrev)) 
        {
        while (pc < pbnd && *pc != '\n' && *pc != '\r')
            pc++;
        while (pc < pbnd && (*pc == '\n' || *pc == '\r'))
            pc++;
        } /* for */
    
    while (pc < pbnd) 
        {/* Find the next line and add it to list (truncate as required) */
        psrc = pc;
        while (pc < pbnd && *pc != '\n' && *pc != '\r')
            pc++;
        
        cchBuffer = pc-psrc;
        if (cchBuffer >= sizeof(szBuffer))
            cchBuffer = sizeof(szBuffer)-1;
        if (cchBuffer) 
            {
            _fmemcpy(szBuffer,psrc,cchBuffer);
            szBuffer[cchBuffer] = '\0';
            filePosition = readAt+(LONG)(psrc-readBuff);     
            if (!FPrependLogLine(hwndList,szBuffer,filePosition,at))
                goto LDone;
            vdws.curLogData += cchBuffer + 2;
            while (pc < pbnd && (*pc == '\n' || *pc == '\r'))
                pc++;
            } /* if */
        } /* while */
        
exitSuccess:
    if (!readOldRows)
        vdws.logLengthPrev = fLength;
    fres = TRUE;
    vdws.fCallLogDirty = FALSE; 
    
LDone:
    if (fcLogFile != NULL)
        _lclose(fcLogFile);
    if (readBuff)
        GlobalFreePtr(readBuff);
    if (fInvalid) 
        {
        SendMessage(hwndList,WM_SETREDRAW,1,0);
        InvalidateRect(hwndList,NULL,TRUE);
        } /* if */
    return fres; 
    
} /* FReadCallLog */

/****************************************************************************/
/* %%Function:GetSzNameNumFromSzLine */
/* 
    szLine contains a log entry. it separates out the name and number portion from
    the log entry and put them into szName and szNum respectively.
*/

static void GetSzNameNumFromSzLine(LPSTR szLine,LPSTR szName,LPSTR szNum)

{
    LPSTR lpchNameStart, lpchNumStart, lpchTimeStart;
    
    szName[0] = szNum[0] = 0;
    
    lpchNameStart = _fstrchr(szLine,'\t');
    if (lpchNameStart == NULL)
        return;
             
    lpchNumStart = _fstrchr(lpchNameStart + 1,'\t');
    if (lpchNumStart == NULL)
        return;      

    lpchTimeStart = _fstrchr(lpchNumStart + 1,'\t');
    if (lpchTimeStart == NULL)
        return;      

    lstrcpyn(szName,lpchNameStart + 1,lpchNumStart - lpchNameStart);
    szName[lpchNumStart - lpchNameStart] = 0;
    lstrcpyn(szNum,lpchNumStart + 1,lpchTimeStart - lpchNumStart);
    szNum[lpchTimeStart - lpchNumStart] = 0;
    
} /* GetSzNameNumFromSzLine */

/****************************************************************************/
/* %%Function:FGetNameOfNumberFromCallLog */
/*
    given szNumber, searches the call log to find the log entry that has the 
    szNumber as its number and returns in szName the name contained in the
    call log. empty string is returned if not found.
*/
static BOOL FGetNameOfNumberFromCallLog(LPSTR szNumber,LPSTR szName)

{   
    char szLine[cchSzMax],szNum[cchSzMax];
    int iLine, cLine; 
    
    if (!vdws.hdlgCallLog || vdws.fCallLogDirty)
        ShowCallLogDlg(TRUE);

    cLine = LOWORD(SendDlgItemMessage(vdws.hdlgCallLog,didCallLogSTextLog,LB_GETCOUNT,0,0));
    for (iLine = 0; iLine < cLine; ++iLine)
        {
        SendDlgItemMessage(vdws.hdlgCallLog,didCallLogSTextLog,LB_GETTEXT,iLine,(LPARAM)((LPSTR)szLine));
        GetSzNameNumFromSzLine(szLine,szName,szNum);
        if (lstrcmp(szNum,szNumber) == 0)
            return TRUE;
        } /* for */        

    szName[0] = 0;
    return FALSE;
            
} /* FGetNameOfNumberFromCallLog */

/****************************************************************************/
/* %%Function:ShowCallLogDlg */
/*
    Makes CallLog Non-Modal Dialog visible (creates it, if necessary).
    insures that it is big enough to show whole log line.
*/    

static VOID ShowCallLogDlg(BOOL fDoNotShow)
{  
    if (!vdws.hdlgCallLog) 
        {
        DLGPROC dlgproc;
        HFONT   hFont;
        int     logWidth;
        RECT    logRect;
        RECT    listRect;
        HWND    logText;
        int     deltaH;
        int     newLeft,newRight;
        int     newWidth;

        dlgproc = (DLGPROC)MakeProcInstance((FARPROC)CallLogProc,vdws.hInst);
        vdws.hdlgCallLog = CreateDialog(vdws.hInst,MAKEINTRESOURCE(dlgCallLog),vdws.hwndDummy,dlgproc);
        if (!vdws.hdlgCallLog)
            { 
            FreeProcInstance((FARPROC)dlgproc);
            /* should put up an error message */
            return;
            } /* if */     
            
        hFont = (HFONT)GetStockObject(ANSI_FIXED_FONT);
        logText = GetDlgItem(vdws.hdlgCallLog,didCallLogSTextLog);
        SendMessage(logText,WM_SETFONT,(WPARAM)hFont,0);
        logWidth = SetCallLogTabs(logText);
    
        /*  Size the log window to show whole log */
        GetWindowRect(vdws.hdlgCallLog,&logRect);
        GetClientRect(logText,&listRect);
        deltaH = logWidth - listRect.right;
        if (deltaH != 0) 
            {
            newLeft = max(0,logRect.left - deltaH/2);
            newRight = logRect.right + (deltaH - (logRect.left-newLeft));
            newWidth = newRight - newLeft;
            MoveWindow(vdws.hdlgCallLog,logRect.top,newLeft,newWidth,logRect.bottom-logRect.top,TRUE);
            } /* if */
        GetSetCallLogWinRect(vdws.hInst,vdws.hdlgCallLog,TRUE);
        } /* if */
        
    if (!vdws.fCallLogVisible) 
        {
        if (!FReadCallLog(GetDlgItem(vdws.hdlgCallLog,didCallLogSTextLog),TRUE))
            DialerErrMessageBox(ikszWarningCallLog);
        if (!fDoNotShow)
            {
            ShowWindow(vdws.hdlgCallLog,SW_SHOW);
            vdws.fCallLogVisible = TRUE;
            } /* if */
        } /* if */
            
} /* ShowCallLogDlg */

/****************************************************************************/
/* %%Function:CallLogProc */
/*
    Handles CallLog Non-Modal Dialog. Note we use fixed font so we can
    accurately compute tab stops from max chars per field.
*/    

BOOL FAR PASCAL _export CallLogProc(HWND hDlg,UINT uMsg,WPARAM wParam,LPARAM lParam)
{
    switch (uMsg)
        {
        case WM_DESTROY:
            {
            SetDialerProfileInt(vdws.hInst,ikszSecCallLogging,ikszFieldCLVisible,vdws.fCallLogVisible);
            GetSetCallLogWinRect(vdws.hInst,vdws.hdlgCallLog,FALSE);
            vdws.hdlgCallLog = NULL;
            vdws.fCallLogVisible = FALSE;
            return TRUE;
            } 
            
        case WM_INITDIALOG:
            return FALSE;
            
        case WM_CLOSE: 
            {
            vdws.fCallLogVisible = FALSE;
            ShowWindow(vdws.hdlgCallLog,SW_HIDE);
            return TRUE;
            }  

        case WM_INITMENUPOPUP:
            {   
            HMENU mEdit;
            WORD wEnable;
            LONG nSelected;
            
            if (LOWORD(lParam) == 0) 
                {/* Edit menu */
                mEdit = (HMENU)wParam;
                nSelected = SendDlgItemMessage(vdws.hdlgCallLog,didCallLogSTextLog,LB_GETSELCOUNT,0,0);
                wEnable = nSelected != 0 ? MF_ENABLED:MF_GRAYED;
                EnableMenuItem(mEdit, midEditCut, wEnable);
                EnableMenuItem(mEdit, midEditCopy, wEnable);
                EnableMenuItem(mEdit, midEditDelete, wEnable);
                }
            else
                {/* Log menu */
                EnableMenuItem((HMENU)wParam,midLogDial,(SendDlgItemMessage(vdws.hdlgCallLog,didCallLogSTextLog,LB_GETSELCOUNT,0,0) == 1) ? MF_ENABLED:MF_GRAYED);
                } 
            break;
            }
    
        case WM_SIZE:  
            {
            MoveWindow(GetDlgItem(hDlg,didCallLogSTextLog),0,0,LOWORD(lParam),HIWORD(lParam),TRUE);
            break;
            }
        
        case WM_VKEYTOITEM: /* Delete key processing */
            {
            if (wParam == VK_CLEAR || wParam == VK_DELETE)
                {         
                SendMessage(hDlg,WM_COMMAND,midEditDelete,0);
                return -2;
                } /* if */
            return -1;
            }   
        case WM_COMMAND:
            {
            switch (wParam)
                {
                case midEditCut: 
                    {
                    if (vdws.hdlgCallLog != 0) 
                        {
                        CopyLogToClipboard(GetDlgItem(vdws.hdlgCallLog,didCallLogSTextLog));
                        FRemoveLogEntries(GetDlgItem(vdws.hdlgCallLog,didCallLogSTextLog));
                        }
                    break; 
                    } 
                case midEditCopy: 
                    {
                    if (vdws.hdlgCallLog != 0)
                        CopyLogToClipboard(GetDlgItem(vdws.hdlgCallLog,didCallLogSTextLog));
                    break;
                    }
                case midEditDelete:
                    {
                    if (vdws.hdlgCallLog != 0)
                        FRemoveLogEntries(GetDlgItem(vdws.hdlgCallLog,didCallLogSTextLog));
                    break;
                    } 
                case midLogDial:
                    {
                    char szLine[cchSzMax], szName[cchSzMax], szNum[cchSzMax];

                    SendDlgItemMessage(vdws.hdlgCallLog,didCallLogSTextLog,LB_GETTEXT,
                        (WPARAM)SendDlgItemMessage(vdws.hdlgCallLog,didCallLogSTextLog,LB_GETCURSEL,0,0),
                        (LPARAM)((LPSTR)szLine));
                    GetSzNameNumFromSzLine(szLine,szName,szNum);
                    
                    vdws.fCommandFromLogWindow = TRUE;
                    FDialerInitiateCall(szNum,szName);
                    vdws.fCommandFromLogWindow = FALSE;
                    break;
                    } 
                /* Brings up the options dialog for call logging */
                case midOptionsLog:
                    {
                    vdws.fCommandFromLogWindow = TRUE;
                    DoDialog(CallLogOptionDlgProc,dlgLogOption,vdws.hInst,vdws.hdlgCallLog,0);
                    vdws.fCommandFromLogWindow = FALSE;
                    break;
                    }
                case midHelpContents:
                    {
                    WinHelp(hDlg,"dialer.hlp",HELP_CONTENTS,0);
                    break;
                    }
                case midHelpCallLog:
                case midAccelHelp: 
                    {
                    WinHelp(hDlg,"dialer.hlp",HELP_CONTEXT,hidCallLog);
                    break;
                    }
                /* if we get a double click in the list box, dial the associated number */
                case didCallLogSTextLog:
                    {     
                    if (HIWORD(lParam) == CBN_DBLCLK && SendDlgItemMessage(vdws.hdlgCallLog,didCallLogSTextLog,LB_GETSELCOUNT,0,0) == 1)
                        SendMessage(hDlg,WM_COMMAND,midLogDial,0);
                    break;
                    } 
                default:
                    return TRUE;
                }
            break;
            }
        } /* switch */
    
    
    return FALSE;

} /* CallLogProc */

/****************************************************************************/
/* %%Function:SetCallLogTabs */
/*                                   
    sets up the globals that contain the formatting info about a call log entry.
*/

static int SetCallLogTabs(HWND hwndList)
{
    double duCharWidth = 0.0;
    char szMask[cchLogLineMax];
    int tabStops[cLogFieldMax];
    char *pch,*pchStart;
    int fld;
    int stopPos = 0;
    TEXTMETRIC tm;
    WORD baseUnits;
    
    if (hwndList) 
        {
        HDC hdc = GetDC(hwndList);
        HFONT hFontOld = NULL;
        HFONT hFont;
        RECT rcDlg;
        
        /* get the text metric */
        hFont = (HFONT)SendMessage(hwndList,WM_GETFONT,0,0);
        if (hFont)
            hFontOld = SelectObject(hdc,hFont);
        GetTextMetrics(hdc,&tm);
        if (hFontOld)
            SelectObject(hdc,hFontOld);
        ReleaseDC(hwndList,hdc);
        
        /* calculate duCharWidth */
        SetRect(&rcDlg,0,0,1000,1000);
        MapDialogRect(vdws.hdlgCallLog,&rcDlg);
        duCharWidth = (double)tm.tmAveCharWidth*1000.0/(double)rcDlg.right;
        } /* if */
        
    LoadString(vdws.hInst,ikszCallLogMask,szMask,sizeof(szMask));
    pch = szMask;
    for (fld = 0; fld < cLogFieldMax; fld++) 
        {
        int cchMax;
        
        pchStart = pch;
        while (*pch != '\0' && *pch != ' ')
            pch++; 
            
        cchMax = pch - pchStart;
        vdws.rgcchFieldMax[fld] = cchMax;
        stopPos += (int)ceil(duCharWidth*(cchMax+1));
        tabStops[fld] = stopPos;
        if (*pch == '\0') 
            {
            vdws.cLogField = fld+1;
            break;
            } /* if */  
        pch++;
        } /* for */
        
    if (hwndList) 
        {
        SendMessage(hwndList,LB_SETTABSTOPS,(WPARAM)vdws.cLogField-1,(LPARAM) (int FAR*)tabStops); 
        baseUnits = LOWORD(GetDialogBaseUnits());
        stopPos = (int) ceil(baseUnits*stopPos/4);
        } /* if */
        
    return stopPos;
             
} /* SetCallLogTabs */
 
/****************************************************************************/
/* %%Function:SzLogCat */
/*
    cat szDate onto szLog taking into account the size of szLog and the field
    position specified by iField.
*/

static void SzLogCat(LPSTR szLog,LPCSTR szData,int iField)
{
    int cchLog = lstrlen(szLog);
    int cchLimit = cchLogLineMax - 4 - cchLog; /* leave room for \t, \r\n & \0 */
    
    if (iField >= vdws.cLogField)
        return;
        
    if (cchLimit > vdws.rgcchFieldMax[iField])
        cchLimit = vdws.rgcchFieldMax[iField];
    if (iField) 
        {
        lstrcpy((LPSTR)(&szLog[cchLog]),"\t");
        cchLog++;
        } /* if */
        
    lstrcpyn((LPSTR)(&szLog[cchLog]),szData,cchLimit+1);
    szLog[cchLog+cchLimit] = '\0'; 
    
} /* SzLogCat */

/****************************************************************************/
/* %%Function:LoadIntlFormats */
/*  
    loads in international date/time format as stored in win.ini "Intl" section.
*/

static void LoadIntlFormats()
{    
    GetProfileString("Intl","s1159","am",vdws.szAM,sizeof(vdws.szAM));
    GetProfileString("Intl","s2359","pm",vdws.szPM,sizeof(vdws.szPM));
    GetProfileString("Intl","sDate","/",vdws.szDate,sizeof(vdws.szDate));
    GetProfileString("Intl","sTime",":",vdws.szTime,sizeof(vdws.szTime));
    GetProfileString("Intl","sShortDate","mm/dd/yy",vdws.szDateFormat,sizeof(vdws.szDateFormat));
    vdws.f24HourTime = GetProfileInt("Intl","iTime",0);
    
}  /* LoadIntlFormats */

/****************************************************************************/
/* %%Function:GetSzDateFromTm */
/* 
    gets the date string corresponding to the date stored in the TM struct. 
    assumes szDate is big enough.
*/
 
static void GetSzDateFromTm(struct tm *tm,char *szDate)
{
    char *pch;
    char chSeg;
    char chSegLast = '\0';
    
    szDate[0] = '\0';
    for (pch = vdws.szDateFormat;*pch;pch++) 
        {
        int data = 0;
        switch (*pch) 
            {
            case 'm':
            case 'M': 
                {
                chSeg = 'M';
                data = tm->tm_mon+1;
                break;
                }
            case 'd':
            case 'D': 
                {
                chSeg = 'D';
                data = tm->tm_mday;
                break; 
                }
            case 'Y':
            case 'y': 
                {
                chSeg = 'Y';
                data = tm->tm_year;
                break;
                }
            } /* switch */
            
        if (data && chSegLast != chSeg) 
            {
            if (chSegLast)
                lstrcat(szDate,vdws.szDate);
            wsprintf(&szDate[strlen(szDate)],"%02i",data);
            chSegLast = chSeg;
            } /* if */
        } /* for */
        
} /* GetSzDateFromTm */

/****************************************************************************/
/* %%Function:GetSzTimeFromTm */
/* 
    gets the date string corresponding to the date stored in the TM struct. 
    assumes szTime is big enough.
*/

static void GetSzTimeFromTm(struct tm *tm,char *szTime)
{
    int hour;
    char *pch;
    
    hour = tm->tm_hour;
    pch = vdws.szPM;
    if (!vdws.f24HourTime) 
        {
        if (hour <= 11)
            pch = vdws.szAM;
        else if (hour > 12)
            hour -= 12;
        } /* if */
    wsprintf(szTime,"%02i%s%02i%s",hour,(LPSTR)vdws.szTime,tm->tm_min,pch); 
             
} /* GetSzTimeFromTm */

/****************************************************************************/
/* %%Function:FLogCall */
/* 
    creates a log entry based on the input. return TRUE if success.
*/

BOOL FLogCall(BOOL in,LPSTR name,LPSTR number,time_t tBegin,time_t tEnd)
{
    char szLog[cchLogLineMax];
    char szBuffer[32], szMin[32];
    struct tm *tm;
    int fcLogFile;
    WORD cchLog;
    BOOL fSuccess = FALSE;
     
    /* a call that needn't be logged */
    if (in && !vdws.fLogIncomingCalls || !in && !vdws.fLogOutgoingCalls)
        return FALSE;
    
    /* load the header string */        
    LoadString(vdws.hInst,in ? ikszCallLogFrom : ikszCallLogTo,szLog,sizeof(szLog));
     
    /*  make sure max fields sizes, etc computed already */
    SetCallLogTabs(0);

    /* append name and number */        
    SzLogCat(szLog,name,1);
    SzLogCat(szLog,number,2);

    /* append date string */
    tm = localtime(&tBegin);
    GetSzDateFromTm(tm,szBuffer);   
    SzLogCat(szLog,szBuffer,3);
    
    /* append time string. lop off seconds and include am/pm as required */
    GetSzTimeFromTm(tm,szBuffer);     
    SzLogCat((LPSTR)szLog,(LPSTR)szBuffer,4);

     /* append duration */
    LoadString(vdws.hInst,ikszCallLogMinute,szMin,sizeof(szMin));
    wsprintf(szBuffer,"%d %s",(int)ceil(difftime(tEnd,tBegin)/60.0),szMin);
    SzLogCat((LPSTR)szLog,(LPSTR)szBuffer,5); 

    /* SzLogCat preserved 2 bytes at end for us! */
    lstrcat(szLog,"\r\n");
        
    /* append log entry to file */
    fcLogFile = FcOpenCallLog(TRUE);    
    if (fcLogFile == HFILE_ERROR)
        goto LDone;
    if (_lseek(fcLogFile,0L,SEEK_END) < 0)
        goto LDone;
    cchLog = strlen(szLog);
    if (cchLog != _lwrite(fcLogFile,szLog,cchLog))
        goto LDone;
        
    fSuccess = TRUE;
    
LDone:
    if (fcLogFile != HFILE_ERROR)
        fSuccess = (_lclose(fcLogFile) == 0);
 
    /* read in all the log entry and update the log display */
    if (vdws.fCallLogVisible)
        FReadCallLog(GetDlgItem(vdws.hdlgCallLog,didCallLogSTextLog),FALSE);
    else
        vdws.fCallLogDirty = TRUE;
        
    return fSuccess;
    
} /* FLogCall */

/****************************************************************************/
/* %%Function:FRemoveLogEntries */
/*
    deletes the selected entries and reloads the display as necessary.
    this code depends on there being no additions to log that haven't been
    read by FReadCallLog, so that hwndList has the current tail of the log.
*/

static BOOL FRemoveLogEntries(HWND hwndList)
{
    WORD cEntry, cEntryTotal; 
    int iEntry;
    LONG ifc, ifcSmallest = 0x7fffffff;
    int FAR *lprgIEntrySel;
    DWORD cchSel = 0;
    WORD iItemSmallestIfc;
    LPSTR lpData = NULL, lpDataCur;
    LONG dataSize;
    BOOL fSuccess = FALSE;
    char xxxx[2];
    int fcLogFile;
    UINT cchEntry;
    BOOL fInvalid = FALSE;
    
    /* sets up cEntry and cEntryTotal */
    cEntry = (WORD)SendMessage(hwndList,LB_GETSELCOUNT,0,0);
    cEntryTotal = (WORD)SendMessage(hwndList,LB_GETCOUNT,0,0);
    if (cEntry == 0)
        return TRUE; 
    
    /* figure out the amount of date we are removing and the lowest file offset */
    lprgIEntrySel = (int FAR *) _fmalloc(cEntry*sizeof(int));
    if (lprgIEntrySel == NULL)
        return FALSE; 
    SendMessage(hwndList,LB_GETSELITEMS,(WPARAM)cEntry,(LPARAM)lprgIEntrySel);
    for (iEntry = 0; iEntry < (int)cEntry; iEntry++) 
        {
        WORD iItem = lprgIEntrySel[iEntry];
        cchSel += SendMessage(hwndList,LB_GETTEXTLEN,iItem,0) + 2;
        ifc = SendMessage(hwndList,LB_GETITEMDATA,iItem,0);
        if (ifc < ifcSmallest) 
            {
            ifcSmallest = ifc;
            iItemSmallestIfc = iItem;
            } /* if */    
        } /* for */
    _ffree(lprgIEntrySel);
    
    /* open the log file */
    fcLogFile = FcOpenCallLog(TRUE);    
    if (fcLogFile == HFILE_ERROR)
        return FALSE;
    if (_lseek(fcLogFile,ifcSmallest,SEEK_SET) < 0)
        goto LDone;
    
    /*  Compute the size of the new piece of the log that starts at the
        first deleted line.  Add 2 bytes fudge to account for possible 
        missing \r\n at eof 
    */
    dataSize = vdws.logLengthPrev - ifcSmallest - cchSel + 2;
    vdws.logLengthPrev = ifcSmallest;
    fInvalid = TRUE;
    SendMessage(hwndList,WM_SETREDRAW,0,0);
    
    /*  If this isn't just a truncate, put the new text in file.  Delete
        the selected rows and update the fileOffsets of other rows.
    */
    if (dataSize > 2) 
        {
        lpData = GlobalAllocPtr(GHND,dataSize);
        if (lpData == NULL)
            goto LDone;
        lpDataCur = lpData;
        iEntry= iItemSmallestIfc + 1;
        while (--iEntry >= 0) 
            {
            if (SendMessage(hwndList,LB_GETSEL,(WPARAM)iEntry,0) != 0) 
                SendMessage(hwndList,LB_DELETESTRING,(WPARAM)iEntry,0);
            else
                {
                SendMessage(hwndList,LB_GETTEXT,(WPARAM)iEntry,(LPARAM)lpDataCur);
                ifc = ifcSmallest + (lpDataCur-lpData);
                SendMessage(hwndList,LB_SETITEMDATA,(WPARAM)iEntry,ifc);
                lpDataCur = lpDataCur + lstrlen(lpDataCur);
                *lpDataCur++ = '\r';
                *lpDataCur++ = '\n';
                } /* else */
            } /* while */
            
        cchEntry = (UINT)(lpDataCur-lpData);
        vdws.logLengthPrev += cchEntry;
        if (cchEntry != 0 && cchEntry != _lwrite(fcLogFile,lpData,cchEntry))
            goto LDone;
        } 
    else 
        {
        /* Truncate only.  Delete selected rows */
        iEntry = iItemSmallestIfc + 1;
        while (--iEntry >= 0) 
            {
            if (SendMessage(hwndList,LB_GETSEL,(WPARAM)iEntry,0) != 0)
                SendMessage(hwndList,LB_DELETESTRING,(WPARAM)iEntry,0);
            } /* while */
        } /* else */
    vdws.curLogData -= cchSel;
    
    /* Write of 0 truncates the file  */
    if (0 != _lwrite(fcLogFile,xxxx,0))
        goto LDone;
        
    fSuccess = TRUE;
    
LDone:
    if (lpData)
        GlobalFreePtr(lpData); 
    if (fcLogFile != HFILE_ERROR)
        fSuccess = (_lclose(fcLogFile) != 0);

    FReadCallLog(hwndList,FALSE);
    if (fInvalid) 
        {
        SendMessage(hwndList,WM_SETREDRAW,1,0);
        InvalidateRect(hwndList,NULL,TRUE);
        } /* if */
        
    return fSuccess;

} /* FRemoveLogEntries */

/****************************************************************************/
/* %%Function:CopyLogToClipboard */
/*  
    copies the selected call log entries into the clipboard as the CF_TEXT format. 
    we also set clipboard of format vdws.cfDialer to notify ourselves that the clipboard
    is of our own format for the smart pasting of log entry.
*/

static void CopyLogToClipboard(HWND hwndList)
{
    WORD cEntry;
    int FAR *lprgIEntrySel = NULL;
    DWORD cchTotal = 0;
    HGLOBAL ghData = 0;
    LPSTR lpData;
    WORD iEntry;
    
    /* get the number of selected entries */
    cEntry = (WORD) SendMessage(hwndList,LB_GETSELCOUNT,0,0);
    if (cEntry == 0)
        return;
    
    /* init lpgrCchItem */      
    lprgIEntrySel = (int FAR *) _fmalloc(cEntry*sizeof(int));
    if (lprgIEntrySel == NULL)
        goto LDone; 
    SendMessage(hwndList,LB_GETSELITEMS ,(WPARAM)cEntry, (LPARAM)lprgIEntrySel);
    for (iEntry = 0; iEntry < cEntry; iEntry++)
        cchTotal += SendMessage(hwndList,LB_GETTEXTLEN,lprgIEntrySel[iEntry],0) + 2; 
    
    /* init ghData that will contain the dtat for the clipboard */    
    ghData = GlobalAlloc(GHND,cchTotal+1);  
    if (ghData == 0)
        goto LDone;
     lpData = GlobalLock(ghData);
    for (iEntry = 0; iEntry < cEntry; iEntry++) 
        {
        SendMessage(hwndList,LB_GETTEXT,lprgIEntrySel[iEntry],(LPARAM)lpData);
        lpData = lpData + lstrlen(lpData);
        *lpData++ = '\r';
        *lpData++ = '\n';
        } /* for */
    GlobalUnlock(ghData);
    
    _ffree(lprgIEntrySel);
    lprgIEntrySel = NULL;
    
    /* give data to the clipboard */
    OpenClipboard(vdws.hwnd);
    EmptyClipboard();
    
    SetClipboardData(CF_TEXT,ghData);
    /* also put on the clipboard vdws.cfDialer so that we know our log entry is in the clipboard */
    SetClipboardData(vdws.cfDialer,NULL); 
        
    CloseClipboard();
    ghData = NULL; /* Owned by ClipBoard now! */
    
LDone:
    if (lprgIEntrySel)
        _ffree(lprgIEntrySel);
    if (ghData)
        GlobalFree(ghData); 
        
} /* CopyLogToClipboard */

/****************************************************************************/
/* %%Function:PasteLogClipboardIntoEditCtrl */
/* 
    called when our call log entry is on the clipboard. It pastes either the name 
    OR the number portion of the log entry into the specified edit/combo control based
    on fPasteName.
*/

static void PasteLogClipboardIntoEditCtrl(HWND hwndCtrl,BOOL fPasteName)

{
    HGLOBAL ghData;
    LPSTR   lpData; 
    char    szDelimiter[2] = {'\t',0}, chSave;
    int     ichDataStart, ichDataEnd, cchData;
    
    if (!IsClipboardFormatAvailable(CF_TEXT))
        return;
    
    OpenClipboard(vdws.hwnd); 
    
    ghData = GetClipboardData(CF_TEXT); 
    if (ghData == NULL)
        goto LOOM;
                
    lpData = GlobalLock(ghData); 
    if (lpData == NULL)
        goto LOOM;
    
    cchData = lstrlen(lpData);
    ichDataStart = _fstrcspn(lpData,szDelimiter) + sizeof(szDelimiter) - 1;
    if (ichDataStart == cchData) /* the format is wrong */
        goto LUnlok;

    ichDataEnd = _fstrcspn(lpData + ichDataStart,szDelimiter) + ichDataStart;
    if (ichDataEnd == cchData) /* the format is wrong */
        goto LUnlok;

    if (!fPasteName)
        {
        ichDataStart = ichDataEnd + sizeof(szDelimiter) - 1;
        ichDataEnd = _fstrcspn(lpData + ichDataStart,szDelimiter) + ichDataStart;
        } /* if */

    chSave = lpData[ichDataEnd];
    lpData[ichDataEnd] = 0;
    /* this is a hack since SetWindowText wouldn't work for a combo box */
    if (GetDlgCtrlID(hwndCtrl) != didDialerComboNumDial)
        SetWindowText(hwndCtrl,lpData + ichDataStart);
    else
        {
        SendDlgItemMessage(GetParent(hwndCtrl),didDialerComboNumDial,CB_INSERTSTRING,0,(LPARAM)(lpData + ichDataStart));
        SendDlgItemMessage(GetParent(hwndCtrl),didDialerComboNumDial,CB_SETCURSEL,0,0);
        } /* else */
    lpData[ichDataEnd] = chSave;
        
LUnlok:
    GlobalUnlock(ghData);
LOOM:
    CloseClipboard();
            
} /* PasteLogClipboardIntoEditCtrl */

