// This is a part of the Microsoft Foundation Classes C++ library.
// Copyright (C) 1993 Microsoft Corporation
// All rights reserved.
//
// This source code is only intended as a supplement to the
// Microsoft Foundation Classes Reference and Microsoft
// QuickHelp and/or WinHelp documentation provided with the library.
// See these sources for detailed information regarding the
// Microsoft Foundation Classes product.


#include "stdafx.h"

#ifdef AFX_OLE4_SEG
#pragma code_seg(AFX_OLE4_SEG)
#endif

#ifdef _DEBUG
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

#define new DEBUG_NEW

/////////////////////////////////////////////////////////////////////////////
// COleDropSource implementation

UINT COleDropSource::nDragMinDist;
UINT COleDropSource::nDragDelay;

COleDropSource::COleDropSource()
{
	m_bDragStarted = FALSE;

	static BOOL bInitialized;
	if (!bInitialized)
	{
		// get drag metrics from win.ini
		static char BASED_CODE szWindows[] = "windows";
		static char BASED_CODE szDragMinDist[] = "DragMinDist";
		static char BASED_CODE szDragDelay[] = "DragDelay";

		nDragMinDist = GetProfileInt(szWindows, szDragMinDist, DD_DEFDRAGMINDIST);
		nDragDelay = GetProfileInt(szWindows, szDragDelay, DD_DEFDRAGDELAY);

		// now initialized, no need to call Initialize any more
		bInitialized = TRUE;
	}

	ASSERT_VALID(this);
}

SCODE COleDropSource::QueryContinueDrag(BOOL bEscapePressed, DWORD dwKeyState)
{
	ASSERT_VALID(this);

	// check escape key or right button -- and cancel
	if (bEscapePressed || (dwKeyState & MK_RBUTTON))
	{
		m_bDragStarted = FALSE; // avoid unecessary cursor setting
		return DRAGDROP_S_CANCEL;
	}

	// check left-button up to end drag/drop and do the drop
	if ((dwKeyState & MK_LBUTTON) == 0)
		return m_bDragStarted ? DRAGDROP_S_DROP : DRAGDROP_S_CANCEL;

	// otherwise, keep polling...
	return S_OK;
}

SCODE COleDropSource::GiveFeedback(DROPEFFECT dropEffect)
{
	ASSERT_VALID(this);

	// don't change the cursor until drag is officially started
	return m_bDragStarted ? DRAGDROP_S_USEDEFAULTCURSORS : NOERROR;
}

BOOL COleDropSource::OnBeginDrag(CWnd* pWnd)
{
	ASSERT_VALID(this);

	DWORD dwLastTick = GetTickCount();
	pWnd->SetCapture();
	m_bDragStarted = FALSE;

	while (!m_bDragStarted)
	{
		// some applications steal capture away at random times
		if (CWnd::GetCapture() != pWnd)
			break;

		// peek for next input message
		MSG msg;
		if (PeekMessage(&msg, NULL, WM_MOUSEFIRST, WM_MOUSELAST, PM_REMOVE) ||
			PeekMessage(&msg, NULL, WM_KEYFIRST, WM_KEYLAST, PM_REMOVE))
		{
			// check for button cancellation
			if (msg.message == WM_LBUTTONUP || msg.message == WM_RBUTTONDOWN)
				break;
			// check for keyboard cancellation
			if (msg.message == WM_KEYDOWN && msg.wParam == VK_ESCAPE)
				break;

			// check for drag start transition
			m_bDragStarted = !m_rectStartDrag.PtInRect(msg.pt);
		}

		// if the user sits here long enough, we eventually start the drag
		if (GetTickCount() - dwLastTick > nDragDelay)
			m_bDragStarted = TRUE;
	}

	ReleaseCapture();
	return m_bDragStarted;
}

BEGIN_INTERFACE_MAP(COleDropSource, CCmdTarget)
	INTERFACE_PART(COleDropSource, IID_IDropSource, DropSource)
END_INTERFACE_MAP()

STDMETHODIMP_(ULONG) COleDropSource::XDropSource::AddRef()
{
	METHOD_PROLOGUE(COleDropSource, DropSource)
	return (ULONG)pThis->ExternalAddRef();
}

STDMETHODIMP_(ULONG) COleDropSource::XDropSource::Release()
{
	METHOD_PROLOGUE(COleDropSource, DropSource)
	return (ULONG)pThis->ExternalRelease();
}

STDMETHODIMP COleDropSource::XDropSource::QueryInterface(
	REFIID iid, LPVOID far* ppvObj)
{
	METHOD_PROLOGUE(COleDropSource, DropSource)
	return (HRESULT)pThis->ExternalQueryInterface(&iid, ppvObj);
}

STDMETHODIMP COleDropSource::XDropSource::QueryContinueDrag(
	THIS_ BOOL fEscapePressed, DWORD dwKeyState)
{
	METHOD_PROLOGUE(COleDropSource, DropSource)
	ASSERT_VALID(pThis);

	return ResultFromScode(pThis->QueryContinueDrag(fEscapePressed, dwKeyState));
}

STDMETHODIMP COleDropSource::XDropSource::GiveFeedback(THIS_ DWORD dwEffect)
{
	METHOD_PROLOGUE(COleDropSource, DropSource)
	ASSERT_VALID(pThis);

	return ResultFromScode(pThis->GiveFeedback(dwEffect));
}

/////////////////////////////////////////////////////////////////////////////
// helper for doing drag/drop with COleDataSource object

DROPEFFECT COleDataSource::DoDragDrop(DWORD dwEffects,
	LPCRECT lpRectStartDrag, COleDropSource* pDropSource)
{
	ASSERT_VALID(this);
	if (pDropSource != NULL)
		ASSERT_VALID(pDropSource);
	ASSERT(lpRectStartDrag == NULL ||
		AfxIsValidAddress(lpRectStartDrag, sizeof(RECT), FALSE));

	// use standard drop source implementation if one not provided
	COleDropSource dropSource;
	if (pDropSource == NULL)
		pDropSource = &dropSource;

	// setup drag/drop sensitivity rect
	pDropSource->m_bDragStarted = FALSE;

	if (lpRectStartDrag != NULL)
	{
		// set drop source drag start rect to parameter provided
		pDropSource->m_rectStartDrag.CopyRect(lpRectStartDrag);
	}
	else
	{
		// otherwise start with default empty rectangle
		CPoint ptCursor;
		GetCursorPos(&ptCursor);
		pDropSource->m_rectStartDrag.SetRect(
			ptCursor.x, ptCursor.y, ptCursor.x, ptCursor.y);
	}
	if (pDropSource->m_rectStartDrag.IsRectEmpty())
	{
		// use drag sensitivity rect passed as parameter
		pDropSource->m_rectStartDrag.CopyRect(lpRectStartDrag);

		if (pDropSource->m_rectStartDrag.IsRectNull())
		{
			// null rect specifies no OnBeginDrag wait loop
			pDropSource->m_bDragStarted = TRUE;
		}
		else
		{
			// empty rect specifies drag drop around starting point
			pDropSource->m_rectStartDrag.InflateRect(
				COleDropSource::nDragMinDist, COleDropSource::nDragMinDist);
		}
	}
	ASSERT_VALID(pDropSource);

	// before calling OLE drag/drop code, wait for mouse to move outside
	//  the rectangle
	ASSERT_VALID(AfxGetMainWnd());
	if (!pDropSource->OnBeginDrag(AfxGetMainWnd()))
		return DROPEFFECT_NONE;

	// call global OLE api to do the drag drop
	LPDATAOBJECT lpDataObject = (LPDATAOBJECT)GetInterface(&IID_IDataObject);
	LPDROPSOURCE lpDropSource =
		(LPDROPSOURCE)pDropSource->GetInterface(&IID_IDropSource);
	DWORD dwResultEffect = DROPEFFECT_NONE;
	::DoDragDrop(lpDataObject, lpDropSource, dwEffects, &dwResultEffect);
	return dwResultEffect;
}

/////////////////////////////////////////////////////////////////////////////
// COleDropSource diagnostics

#ifdef _DEBUG
void COleDropSource::Dump(CDumpContext& dc) const
{
	CCmdTarget::Dump(dc);
	AFX_DUMP1(dc, "\nm_bDragStarted = ", m_bDragStarted);
	AFX_DUMP1(dc, "\nm_rectStartDrag.left = ", m_rectStartDrag.left);
	AFX_DUMP1(dc, "\nm_rectStartDrag.top = ", m_rectStartDrag.top);
	AFX_DUMP1(dc, "\nm_rectStartDrag.right = ", m_rectStartDrag.right);
	AFX_DUMP1(dc, "\nm_rectStartDrag.bottom = ", m_rectStartDrag.bottom);
}
#endif

/////////////////////////////////////////////////////////////////////////////
