/******************************************************************************
      (C) Copyright 1992 by Autodesk, Inc.

      This program is copyrighted by Autodesk, Inc. and is  licensed
      to you under the following conditions.  You may not distribute
      or  publish the source code of this program in any form.   You
      may  incorporate this code in object form in derivative  works
      provided  such  derivative  works  are  (i.) are  designed and 
      intended  to  work  solely  with  Autodesk, Inc. products, and 
      (ii.)  contain  Autodesk's  copyright  notice  "(C)  Copyright  
      1992 by Autodesk, Inc."

      AUTODESK  PROVIDES THIS PROGRAM "AS IS" AND WITH  ALL  FAULTS.
      AUTODESK  SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTY OF  MER-
      CHANTABILITY OR FITNESS FOR A PARTICULAR USE.  AUTODESK,  INC.
      DOES  NOT  WARRANT THAT THE OPERATION OF THE PROGRAM  WILL  BE
      UNINTERRUPTED OR ERROR FREE.

*******************************************************************************/

/* For the Metaware High C and High C/C++ compilers, turn off an
   unwanted warning message */
#ifdef __HIGHC__
pragma Offwarn(67); /* kills "switch statement has no cases" warning */
pragma Off(Prototype_conversion_warn); /* turns off warnings about non standard conversions */
#endif

#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include "sxp.h"

#define LIM255(c) (((c)>=255)?255:(c))

/* Variable definitions */
#define SIZE 1
#define R1 2
#define R2 3
#define DX 6
#define DY 7
#define DZ 8
#define COL1 10
#define COL2 11
/* The above definitions are the LineID numbers for the dialog lines in which the variables will be set. 
The integers assigned to the dialog lines containing variables are selected by the IPAS programmer and 
used by 3D Studio to identify the variables. */

/************************/

/* Dialog description */
DlgEntry cdialog[]={
	0,"TITLE=\"3D Wood\"",					/*Text to appear in dialog box */
    SIZE, "LFLOAT=\"Size:\",0.0001,10000.0,5",         /*  Setup for 'limited float' type
								variable and associated range 
								values */
	R1, "-LFLOAT=\" R1:\",0.0,10000.0,5",			/* Limited float variable */
	R2, "-LFLOAT=\" R2:\",0.0,10000.0,5",			/* Limited float variable */
	0,"TITLE=\"\"",						/* Blank line for spacing */
	DX, "LFLOAT=\"Offset:\",0.0,10000.0,5",			/* Limited float variable */
	DY, "-LFLOAT=\" \",0.0,10000.0,5",			/* Limited float variable */
	DZ, "-LFLOAT=\" \",0.0,10000.0,5",			/* Limited float variable */
	0,"TITLE=\"\"",						/* Blank line for spacing */
	COL1,"COLOR=\"Color 1\"",				/* Setup for standard color triple 	
								slider dialog*/
	COL2,"-COLOR=\"Color 2\"",				/* Setup for standard color triple 	
								slider dialog*/
	0,"TITLE=\"\"",						/* Blank line for spacing */
	0,NULL
	};


/* State Structure definition */
/* Any variables whose values are set via dialog interface must be declared within the 'State' struct. */	
typedef struct {
	ulong version;
	float size;
	float dx,dy,dz;
	float r1;
	float r2;
	Col24 col1,col2;
	} WoodState;
/* the "state" struct MUST start with a "ulong" which is the version#,
	to prevent using data from old versions of this program.
	This verification is performed automatically. */
/* Version test value */
/* Every external process must have a unique version number  */
#define WOOD_VERS 0x3AE2

#define INIT_WOOD { WOOD_VERS,10.0,0.0,0.0,0.0,1.0,1.0,{115,46,11},{13,3,2}}

static WoodState init_state = INIT_WOOD; /* default, used for resetting */
static WoodState state = INIT_WOOD;		/* the current state */


/*----------------------------------------------------------------*/

#define NOISE_DIM 20    
#define FNOISE_DIM 20.0

static float noise_table[NOISE_DIM+1][NOISE_DIM+1][NOISE_DIM+1];

void init_noise() {
	int i,j,k,ii,jj,kk;
	for (i=0; i<=NOISE_DIM; i++)
		for (j=0; j<=NOISE_DIM; j++)
			for (k=0; k<=NOISE_DIM; k++) {
				noise_table[i][j][k] = (float)(rand()&0x7FFF);
				ii = (i==NOISE_DIM)?0:i; 
				jj = (j==NOISE_DIM)?0:j; 
				kk = (k==NOISE_DIM)?0:k; 
				noise_table[i][j][k] = noise_table[ii][jj][kk];
				}
	}

float noise1(float x) {
    int ix /**** ,iy,iz*/;
	float fx,mx;
	float n0,n1;
	mx = fmod(x,FNOISE_DIM); 
	if (mx<0) mx += FNOISE_DIM;
	ix = (int)mx;
	fx = fmod(mx,1.0);
	n0 = noise_table[ix][0][0];
	n1 = noise_table[ix+1][0][0];
	return(((float)(n0+fx*(n1-n0)))/32768.0);
	}

float noise(float x,float y,float z) {
	int ix,iy,iz;
	float fx,fy,fz,mx,my,mz;
	float n,n00,n01,n10,n11,n0,n1;
	mx = fmod(x,FNOISE_DIM); if (mx<0) mx += FNOISE_DIM;
	my = fmod(y,FNOISE_DIM); if (my<0) my += FNOISE_DIM;
	mz = fmod(z,FNOISE_DIM); if (mz<0) mz += FNOISE_DIM;
	ix = (int)mx;
	iy = (int)my;
	iz = (int)mz;
	fx = fmod(mx,1.0);
	fy = fmod(my,1.0);
	fz = fmod(mz,1.0);
	n = noise_table[ix][iy][iz];
	n00 = n + fx*(noise_table[ix+1][iy][iz]-n);
	n = noise_table[ix][iy][iz+1];
	n01 = n + fx*(noise_table[ix+1][iy][iz+1]-n);
	n = noise_table[ix][iy+1][iz];
	n10 = n + fx*(noise_table[ix+1][iy+1][iz]-n);
	n = noise_table[ix][iy+1][iz+1];
	n11 = n + fx*(noise_table[ix+1][iy+1][iz+1]-n);
	n0 = n00 + fy*(n10-n00);
	n1 = n01 + fy*(n11-n01);
	return(((float)(n0+fz*(n1-n0)))/32768.0);
	}


static void lerp_color(Col24 *c, Col24 *a, Col24 *b, float f) {
	int alph,ialph;
	if (f>1.0) f = 1.0;
	alph = (int)4096*f;
	ialph = 4096-alph;
	c->r = (ialph*a->r + alph*b->r)>>12;
	c->g = (ialph*a->g + alph*b->g)>>12;
	c->b = (ialph*a->b + alph*b->b)>>12;
	}

/* smooth step function with hermite interpolation*/
float smoothstep(float x0, float x1, float v) {
	if (v<=x0) return(0.0);
	else if (v>=x1) return(1.0);
	else {
		float u = (v-x0)/(x1-x0);
		return(u*u*(3-2*u));
		}
	}

float wood(float x, float y, float z) {
	float	r;
	x = state.dx + x/state.size;
	y = state.dy + y/state.size;
	z = state.dz + z/state.size;
	x += noise1(x)*state.r1;
	y += noise1(y)*state.r1;
	z += noise1(z)*state.r1;
	r = sqrt(x*x+z*z);
	r += noise1(r)+state.r2*noise1(y/4.0);
	r = fmod(r,1.0); /* be periodic */
	r = smoothstep(0,0.8,r) - smoothstep(0.83,1.0,r);
	return(r);
	}

/* The following function is a 'Client' procedure that defines a 3-dimensional function to modify the 
diffuse color and/or the shininess. This function is passed the SXPDATA structure contained in the 
EXPbuf, but has no access to packet commands or to other EXPbuf data structures. */

void Client3DText(Sxpdata *sd) {
	float i;
	i = wood(sd->p[0],sd->p[1],sd->p[2]);
	lerp_color(&sd->col,&state.col1,&state.col2,i);
	}

/* The following function is a 'Client' procedure that defines a 3-dimensional function to compute an 
opacity value. This function is passed the SXPDATA structure contained in the EXPbuf, but has no access 
to packet commands or to other EXPbuf data structures. */

void Client3DOpac(Sxpdata *sd) {
	float i;
	i = wood(sd->p[0],sd->p[1],sd->p[2]);
	lerp_color(&sd->col,&state.col1,&state.col2,i);
	}

/* The following function is a 'Client' procedure that defines a 3-dimensional function to compute a 
normal perturbation value. This function is passed the SXPDATA structure contained in the EXPbuf, but 
has no access to packet commands or to other EXPbuf data structures. */

void Client3DBump(Sxpdata *sd) {
	float del,d,x,y,z;
	x = sd->p[0];
	y = sd->p[1];
	z = sd->p[2];
	d = wood(x,y,z);
	del = .1;
	sd->norm_pert[0] = (wood(x+del,y,z) - d)/del;
	sd->norm_pert[1] = (wood(x,y+del,z) - d)/del;
	sd->norm_pert[2] = (wood(x,y,z+del) - d)/del;
	}

/* Following function sets the appropriate 'State' variables based
   on the input from the dialog */
void ClientSetStateVar(int id, void *ptr) {
	OVL o;				/* Union of possible values that the dialog can return */
	ulong *ul;

	ul=(ulong *)ptr;
	o.ul = *ul;
	switch(id) {
		case SIZE:  state.size = o.f; break;			/* State variables set here based on
								values stored in the fields of the 
								Union type .*/
		case R1: state.r1 = o.f; break;
		case R2: state.r2 = o.f; break;
		case DX: state.dx = o.f; break;
		case DY: state.dy = o.f; break;
		case DZ: state.dz = o.f; break;
		case COL1: 					/* Set each field within the COL 
								variable in RGB format */
			state.col1.r = o.c.r;
			state.col1.g = o.c.g;
			state.col1.b = o.c.b;
			break;
		case COL2: 					/* Set each field within the COL 
								variable in RGB format */
			state.col2.r = o.c.r;
			state.col2.g = o.c.g;
			state.col2.b = o.c.b;
			break;
		}
	}

/* Following function sets the appropriate value of the Union 'OVL' variable
   to return values stored in the 'State' variable */
ulong ClientGetStateVar(int id) {
	OVL o;
	switch(id) {
		case SIZE: o.f = state.size; break;		/* Union variables set to appropriate value
							from 'state' variables */
		case R1: o.f = state.r1; break;
		case R2: o.f = state.r2; break;
		case DX: o.f = state.dx; break;
		case DY: o.f = state.dy; break;
		case DZ: o.f = state.dz; break;
		case COL1: 
			o.c.r = state.col1.r;
			o.c.g = state.col1.g;
			o.c.b = state.col1.b;
			break;
		case COL2: 
			o.c.r = state.col2.r;
			o.c.g = state.col2.g;
			o.c.b = state.col2.b;
			break;
		}
	return(o.ul);					/* the variable requested by 3D Studio is 
							returned to 3D Studio via the 'ulong' field
							of the 'Union' structure 'OVL'. */
	}

/* ---For most applications, the following functions will not change */

/* Following function allows 3D Studio to determine the size of
   the 'State' structure and returns a pointer to the 'State' struct */

char  *ClientGetState(int *size) {
	*size = sizeof(WoodState);
	return((char *)&state);
	}

/* Following function allows 3D Studio to reset the state of the
   external process */

void ClientResetState(void) {	
	state = init_state;	
	}

/* The following function performs any calculations necessary to the operation of 3DTEXT, 3DBUMP, or 
3DOPAC that cannot be performed within any of these functions. The 'ClientStartUp' procedure does not 
have access to packet-commands. */

void ClientStartup(void) {
	/* allocate data structures, compute lookup tables, etc */
	init_noise();
	}

void ClientTerminate(void) { 
	/* free any data structures, etc. */
	}

/* The following function allows 3D Studio to get the address of a string containing a variable. The IPAS
programmer will not have to call this function or modify the following code, but will have to include it */

DlgEntry *ClientDialog(int n) {	
	return(&cdialog[n]); 
	}

/* Following function allows 3D Studio to determine the size of
   any variable in the dialog */

int ClientVarSize(int id) {
	switch(id) {
		default:
			return(1);
		}
	}




