/******************************************************************************
      (C) Copyright 1992 by Autodesk, Inc.

      This program is copyrighted by Autodesk, Inc. and is  licensed
      to you under the following conditions.  You may not distribute
      or  publish the source code of this program in any form.   You
      may  incorporate this code in object form in derivative  works
      provided  such  derivative  works  are  (i.) are  designed and 
      intended  to  work  solely  with  Autodesk, Inc. products, and 
      (ii.)  contain  Autodesk's  copyright  notice  "(C)  Copyright  
      1992 by Autodesk, Inc."

      AUTODESK  PROVIDES THIS PROGRAM "AS IS" AND WITH  ALL  FAULTS.
      AUTODESK  SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTY OF  MER-
      CHANTABILITY OR FITNESS FOR A PARTICULAR USE.  AUTODESK,  INC.
      DOES  NOT  WARRANT THAT THE OPERATION OF THE PROGRAM  WILL  BE
      UNINTERRUPTED OR ERROR FREE.

*******************************************************************************/

/* ------------------------------------------------- */
/*  VBLUR: Vertical Image Blurring            	*/
/*    Removes those nasty video artifacts		*/
/*      By Dan Silva 2-7-92		       		*/
/* ------------------------------------------------- */

#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include "ixp.h"

void getline(int y, Color_24 *buf);
void putline(int y, Color_24 *buf);

/* Variables */

#define SHFT 12
#define SCFACTOR 4096

static Color_24 _far *inpix;	/* Input image data */
static Color_24 _far *outpix;	/* Output image data */
static int dev_width,dev_height,iy;
static long frac,frac0;

Color_24 *inbuf[3] = {0,0,0};
Color_24 *outbuf = 0;

/* Function declarations */
void freebufs();
void getline(int y, Color_24 *buf);
void putline(int y, Color_24 *buf);


/* Configuration dialog contents */

#define AMOUNT 1

/* The above definitions are the LineID numbers for the dialog lines in which the variables will be set. The integers assigned to the dialog lines containing variables are selected by the IPAS programmer and used by 3D Studio to identify the variables. */

DlgEntry cdialog[]={
	0,"TITLE=\"Autodesk 3D Studio Vertical Blur Process\"",	/* Text to appear in dialog box */
	0,"TITLE=\"by Dan Silva, 2/7/92\"",			/* Text to appear in dialog box */
	0,"TITLE=\"\"",						/* Blank line for spacing */
	AMOUNT,"LFLOAT=\"Amount (0 to 1.0) :\",0.0,1.0",	/*  Setup for 'limited float' type 
								variable and associated range 
								values */
	0,"TITLE=\"\"",						/* Blank line for spacing */
	0,NULL
	};

/* Version test value */
/* Every external process must have a unique version number  */

#define VERSION 0x8456

/* State Structure definition */
/* Any variables whose values are set via dialog interface must be declared within the 'State' struct. */	
	
typedef struct {
	ulong version;			/* required to be first field within 'State' struct */
	float amount;
	} VBlurState;			

/* The "state" struct MUST start with a "ulong" which is the version#,
	to prevent using data from old versions of this program.
	This verification is performed automatically. If you change the
	state structure, change the version#. */

static VBlurState init_state = { VERSION,0.6};	/* Initial state settings */
static VBlurState state = { VERSION,0.6};		/* Default state settings */

/*----------------------------------------------------------------*/
/*Following function contains all the routines particular to the image processing done by the IXP. The passed parameters contain all the data from an image that an IXP can access and modify. */

int ClientProcImage(FrameParm *fp,FrameInfo *fi,int phase,Report *rpt) {
    int /****pixels,lowy,highy,*/i,ix;
	Color_24 *prevbuf,*thisbuf,*nextbuf,pprev,pthis,pnext;
	Color_24 pixel,*sv;

	/* The integer variable 'phase' records the state of the 'ClientProcImage' function and thus 	determines flow of control. The 'phase' variable has two possible values 0 or 1 representing a 	'start' or 'continue' state respectively. */						

	switch(phase) {
		case PHASE_START:
			dev_width= fp->dev_width;	/* data is recorded from the parameters */
			dev_height=fp->dev_height;
	
			inpix = fp->inpix;		/* pointer to input image */
			outpix = fp->outpix;		/* pointer to output image */
		
			for (i=0; i<3; i++) {
				inbuf[i] = (Color_24 *)malloc(dev_width*sizeof(Color_24));
				}
			outbuf = (Color_24 *)malloc(dev_width*sizeof(Color_24));
	
			if (inbuf[0]==0||inbuf[1]==0||inbuf[2]==0||outbuf==0) {
				freebufs();
				return(IXP_ERROR);	/* Value of  0 is returned to 3D Studio to 								signal error */
				}

			/* weight for adjacent pixels */
			frac = (int)(SCFACTOR*state.amount/(1.0 + 2*state.amount));
	
			/* weight for center pixel */
			frac0 = (int)(SCFACTOR/(1.0 + 2*state.amount));
				
			iy=0;
			getline(0,inbuf[2]);
			goto proc_a_line;			/* process the first  image line */
			

		case PHASE_CONTINUE:		/* Test for 'continue' value of 'phase' var. */
			line_loop:
			iy++;

			if(iy>=dev_height) {
				freebufs();
				return(IXP_EXIT);	/* Return a 'terminate' without error 								command */
				}
	
			proc_a_line:			/* image line processing algorithm */

			/* rotate buffers */
			sv = inbuf[0];
			inbuf[0] = inbuf[1];
			inbuf[1] = inbuf[2];
			inbuf[2] = sv;

			if (iy+1<dev_height)
				getline(iy+1,inbuf[2]);

   		/* Image-process one line here */
			prevbuf = inbuf[0];
			thisbuf = inbuf[1];
			nextbuf = inbuf[2];
			if (iy==0) prevbuf = thisbuf;; 
			if (iy==dev_height-1) nextbuf = thisbuf;
			for(ix=0; ix<dev_width; ++ix) {
				pprev = prevbuf[ix];
				pthis = thisbuf[ix];
				pnext = nextbuf[ix];
				/* The color value for each pixel is set in RGB format */
				pixel.r = (pprev.r*frac + pthis.r*frac + pnext.r*frac0)>>SHFT;
				pixel.g = (pprev.g*frac + pthis.g*frac + pnext.g*frac0)>>SHFT;
				pixel.b = (pprev.b*frac + pthis.b*frac + pnext.b*frac0)>>SHFT;
				outbuf[ix] = pixel;
				}
			putline(iy,outbuf);
			if((iy%10)!=0)
				goto line_loop;
			break;
		}
	
  /* Send back progress report */
/* After every iteration of 'ClientProcImage' the progress bar should be updated, and a 'continue' value should be sent so that 3D Studio can automatically set phase to 'continue' after the first call. */

	strcpy(rpt->title,"Filtering Image");	/* Title to appear above progress bar */
	rpt->max=dev_height;			/* max value for progress bar */
	rpt->current=iy;				/* amount by which to increment progress bar */
	rpt->recall_state = PHASE_CONTINUE;	/* 'Continue' value for 3D Studio to place in 'phase'*/
	return(IXP_REPORT);			/* Signals an 'update progress bar' event and 							continue status */
	}

/* Following function allows 3D Studio to determine the size of
   any variable in the dialog */

int ClientVarSize(int id) {
	switch(id) {
		default:
			return(1);
		}
	}

/* Following function sets the appropriate 'State' variables based
   on the input from the dialog */

void ClientSetStateVar(int id, void *ptr) {
	OVL o;				/*Union of possible values that the dialog can return */
	ulong *ul;
	ul=(ulong *)ptr;
	o.ul = *ul;
	switch(id) {
		case AMOUNT: state.amount=o.f; break;		/* State variables set here based on
								values stored in the fields of the 
								Union type .*/
		}
	}

/* Following function sets the appropriate value of the Union 'OVL' variable
   to return values stored in the 'State' variable */

ulong ClientGetStateVar(int id) {
	OVL o;
	switch(id) {
		case AMOUNT: o.f = state.amount; break;		/* Union variables set to 
								appropriate value from 'state'
								variables */
		}
	return(o.ul);					/* the variable requested by 3D Studio is 
							returned to 3D Studio via the 'ulong' field
							of the 'Union' structure 'OVL'. */
	}

/* Following function allows 3D Studio to determine the size of
   the 'State' structure and returns a pointer to the 'State' struct */

char  *ClientGetState(int *size) {
	*size = sizeof(VBlurState);
	return((char *)&state);
	}

/* Following function allows 3D Studio to reset the state of the
   external process */

void ClientResetState() { state = init_state; }

/* Following function performs any calculations or packet-commands
   necessary before the EXP can begin. Note: IXPs do not have ability to send packet commands as the EXPbuf is not passed in to the 'ClientStartup' procedure, and there is no 'ClientUserCode' function. */

void ClientStartup(void) {
	/* allocate data structures, compute lookup tables, etc */
	}

void ClientTerminate(void) { 
	/* free any data structures, etc. */
	freebufs();
	}

/* The following function allows 3D Studio to get the address of a string containing a variable. The IPAS
programmer will not have to call this function or modify the following code, but will have to include it */

DlgEntry *ClientDialog(int n)
 	{	
	return(&cdialog[n]); 
	}

/* The following procedure returns the name of the IPAS process */
char *ClientName(void) {
	return("VBLUR.IXP");		/* Put your process name here */
	}

/* The following functions were written by the IPAS programmer for this specific routine, and therefore, are not examples of required 'Client' functions */

void freebufs() {
	int i;
	for (i=0; i<3; i++) {
		if (inbuf[i])
			free(inbuf[i]); 
		inbuf[i] = 0;
		}
	if (outbuf)
		free(outbuf);
	outbuf = 0;
	}

void getline(int y, Color_24 *buf) {
	far_to_near(buf,&inpix[y*dev_width],dev_width*3);
	}

void putline(int y, Color_24 *buf) {
	near_to_far(&outpix[y*dev_width],buf,dev_width*3);
	}

