*   (C)Copyright LodeStone Consortium 1992-93 - All Rights Reserved
*	Compile with /m/n/w

#include "box.ch"
#include "cmx.ch"

	//	Flag indicating whether or not to use ClipMore (do by default)
static _cmFlag := .t.

	//	Main menu strings
static demoList := ;
	{"demo01()    List records which meet a condition (natural order)", ;
	 "demo02()    Same as demo01(), but in last name order", ;
	 "demo03()    dbedit() for records which meet a condition", ;
	 "demo04()    Copy records which meet a condition to _tmp.dbf", ;
	 "demo05()    UDF expression", ;
	 "demo06()    Relational example", ;
	 "demo07()    Ad hoc query (dbedit()'s the result, LAST order)", ;
	 "demo08()    Ad hoc query (cmReFilter() example)", ;
	 "cmOnOff()   Turn use of ClipMore by demo functions on or off", ;
	 "cmVer()     Get detailed ClipMore version", ;
	 "quitDemo()  Quit out of the demo" }

	//	Array of CodeBlocks which correspond to menu items above
static demoFun := ;
	{ {|| demo01()}, ;
	  {|| demo02()}, ;
	  {|| demo03()}, ;
	  {|| demo04()}, ;
	  {|| demo05()}, ;
	  {|| demo06()}, ;
	  {|| demo07()}, ;
	  {|| demo08()}, ;
	  {|| cmOnOff()}, ;
	  {|| cmVer()},  ;
	  {|| quitDemo()} }


*   demo.prg
*
*   action: Provides a brief demo of ClipMore's capabilities.
*           First, it creates all indexes which haven't yet been created.
*           Next, it provides a pick-list of functions to execute.
*
*   alters: o   May create indexes and temporary files
*

function main()
local choice

*	No scoreboard
set scoreboard off

*   Create all indexes not yet created
clear
GenInd()

*   Bring database into use
use demo index _last, _age, _lastcall, _salary, _udf alias customer

*   Keep going (user explicitly selects quitDemo() option to quit)
do while .t.

    *   Get the user's choice
    choice = mainMenu()

    *   Clear the screen and execute the appropriate function
    clear
	eval(demoFun[choice])
enddo
return


    *********************************************************
    ****** Routines callable from menu demoNN() series ******
    *********************************************************

*   demo01()
*
*   action: Same as demo02(), but sets order to 0 to show difference.
*
*   input:  None.
*
*   return: .t. (always)
*
*   alters: o   Writes to screen

function demo01()
local condition := "lastcall<ctod('01/01/72') .and. salary<90000"

*	Let user know what we're doing
cmStatus(demoList[1])

*   Set order to 0, since order doesn't matter, then ...
set order to 0

*   Filter 'em, list 'em out, and clean up
cmFilterP(condition)
list last, age, lastcall, salary

? "That was a list for " + condition
wait

cmClrFilter()
return nil


*   demo02()
*
*   action: Does a list for records which meet a certain condition, using
*           age as the master index.
*
*   input:  None.
*
*   return: .t. (always)
*
*   alters: o   Writes to screen

function demo02()
local condition := "lastcall<ctod('01/01/72') .and. salary<90000"

*	Let user know what we're doing
cmStatus(demoList[2])

*   last order
set order to 1

*   Filter 'em, list 'em out, and clean up
cmFilterP(condition)
list last, age, lastcall, salary

? "That was a list for " + condition
wait

cmClrFilter()
return nil


*   demo03()
*
*   action: Does a dbedit() for records which meet a certain condition.
*
*   input:  None.
*
*   return: .t. (always)
*
*   alters: o   Writes to screen

function demo03()
cmStatus(demoList[3])

*   age order
set order to 2

*   Get records which meet the condition, dbedit() 'em, & clean up
cmFilterP("age>=90 .and. last>='Q' .and. lastcall<=ctod('01/01/75')")
dbeditBar(3)
cmClrFilter()
return nil


*   demo04()
*
*   action: Copies to a temporary file for a condition.  Shows use of
*           cmFilter() in conjunction with a bulk command.
*
*   input:  None.
*
*   return: .t. (always)
*
*   alters: o   Writes to screen

function demo04()
cmStatus(demoList[4])

*   Copy 'em in salary order
*   (NOTE: This same logic would be employed for any bulk command such as
*   EDIT, REPLACE, DELETE, etc.)
*
set order to 4
cmFilterP("age <= 40 .and. last >= 'T' .and. salary >= 900000")
copy to _tmp

*   Let user look at the results
? "Copy is complete"
wait
delete file _tmp.dbf

cmClrFilter()
return nil


*   demo05()
*
*   action: Shows ClipMore optimizing a UDF expression.
*
*   input:  None.
*
*   return: .t. (always)
*
*   alters: o   Writes to the screen

function demo05()
cmStatus(demoList[5])

*   age order
set order to 2

*   dbedit() records which meet a UDF condition, in age order
cmFilterP("demoUDF(first, last) = demoUDF('Wiley', 'Morse')")
dbeditBar(3)
cmClrFilter()

return nil


*   demo06()
*
*   action: Shows using ClipMore with a relational query.  Demo.dbf has been
*           used in work area 1, pretending that it is a customer file. Demo
*           is then used again in work area 2, this time pretending that it
*           is an invoice file.  The salary field is treated as the customer
*           id which is used to link from the customer file to the invoice
*           file.
*
*           Then, a query is done against that file, and the results are
*           listed to the screen.
*
*   input:  None.
*
*   return: .t. (always)
*
*   alters: o   Writes to screen

function demo06()
cmStatus(demoList[6])

*   Bring demo into use a second time pretending it's an invoice database
*   (In this example, we act as if demo is a customer database)
sele 2
use demo index _salary, _lastcall alias invoice

*   Handle problem which results with this double-use "trick" if SHARE
*   is loaded
if (alias() <> "INVOICE")
    atSayCenter( 5, "SHARE or network shell is loaded")
    atSayCenter( 7, "This demo (demo06()) will NOT work in a multi-user")
    atSayCenter( 8, "environment because the same database is USE'd")
    atSayCenter( 9, "in two work areas simultaneously")
    atSayCenter(11, "Use in single-user mode (without SHARE or network")
    atSayCenter(12, "shell) to see this demo work")
    ?
    wait
    sele 1
    return .t.
endif

*   The net condition we want is:
*
*   All the customers over 80 who have a last name >= 'Royce' .and
*   whose invoice is dated before June of 1971
*
*   customer->last >= 'Royce' .and. customer->age >= 80
*   .and.
*   invoice->lastcall < ctod('06/01/71')
*

*   Set up the condition on invoice
sele invoice
cmFilterP("lastcall < ctod('06/01/71')")

*   Set up the condition on customer (demo)
sele customer
set order to 1 && last name order
cmFilterP("age >= 80 .and. last >= 'Royce'")

*   Set up the relation (pretend that salary is the customer id which
*   links the two files)
set relation to salary into invoice

*   Now, let's get out the records
go top
do while .not. eof()

    *   If we found an invoice record
    sele invoice
    if (.not. eof())

        *   Output it
        ? "Customer Name, Id & Age: ", customer->last,;
                customer->salary, customer->age

        *   Output related invoices indented
        do while (customer->salary = invoice->salary)
            ? "   Invoice Customer Id & Date: ",;
                    invoice->salary, invoice->lastcall
            skip
        enddo
    endif

    *   skip to next customer
    sele customer
    skip
enddo

*   Let user see results
wait

*   Clean up afterwards (clear filters for both work areas)
sele invoice
cmClrFilter()
use
sele customer
cmClrFilter()
return nil


*   demo07()
*
*   action: This allows input of an ad hoc query, and the query is
*           executed using cmFilterP().  This can be used to show
*           the effects of partial optimization.
*
*   input:  None.
*
*   return: .t. (always)
*
*   alters: o   Writes to screen

function demo07
static demo07cond := ;
	"                                                                 "

cmStatus(demoList[7])

set order to 1  && last name order

do while .t.
    atSayCenter(2, ;
        "Note: Last, Lastcall, Age, Salary have indexes (others don't)")
    @ 3, 0 say "Condition: " get demo07cond
    read
    clrLine(3)

    *   If it's a logical, break out of the loop
    if (type(demo07cond) = 'L')
        exit
    endif

    *   Tell user it's not OK, and try again
    @ 4, 0 say "Condition does not evaluate to a logical.  Please try again."
enddo

*   Do the filter based on the condition, dbedit() and clean up
cmFilterP(trim(demo07cond))
dbeditBar(3)
cmClrFilter()

return .t.


*   demo08()
*
*   action: This allows input of an increasingly selective ad hoc query
*           using cmReFilter().
*
*   input:  None.
*
*   return: .t. (always)
*
*   alters: o   Writes to screen

function demo08
local cond

cmStatus(demoList[8])

set order to 1  && last name order

*   Say what it is
@ 2, 0 say "    Condition: <None>"
@ 4, 0 say "      Matches: " + str(reccount())

do while .t.

    *   Get the new condition
    cond := space(60)
    do while .t.
        @ 3, 0 say "New Condition:" get cond
        read

        *   If we got a logical, or want to break out, then all done
        if (type(cond) = 'L') .or. (lastkey() = 27)
            exit
        endif

        *   Didn't get a logical
        atSayCenter(4, "Condition does not evaluate to a logical.  Please try again.")
    enddo

    *   Get out if user wants to exit
    if (lastkey() = 27)
        exit
    endif

    *   Do the condition
    @ 4, 0
    retVal = cmReFiltP(cond)

    *   Say what it is
    @ 2, 0 say "    Condition: " + iif(len(dbfilter()) <= 60,;
            dbfilter(), "... " + right(dbfilter(), 56))
    clrLine(4)
    @ 4, 0 say "      Matches: " + str(retVal)

    *   Bring up a dbedit()
    dbeditBar(5)
enddo

*   Clear the filter
cmClrFilter()

return .t.


*   cmOnOff()
*
*   action: Toggles whether or not use cmFilter (by toggling value of
*           _cmFlag).
*
*   input:  None.
*
*   return: New setting of _cmFlag (.T. if cmFilter() will be used,
*           .F. otherwise).
*
*   alters: o   _cmFlag

function cmOnOff
cmStatus(demoList[9])
_cmFlag := !_cmFlag
return _cmFlag


*   cmVer()
*
*   action: Print out the detailed ClipMore version.
*
*   input:  None.
*
*   return: .t. (always)
*
*   alters: None.

function cmVer
cmStatus(demoList[10])
atSayCenter(11, cmVersion(2))
wait
return .t.


*   quitDemo()
*
*   action: Quits the demo program.
*
*   input:  None.
*
*   return: None (doesn't).
*
*   alters: quits out of the program

function quitDemo
cmStatus(demoList[11])
? "Demo Exiting"
quit
return .t.


    *****************************************
    ****** Various support routines *********
    *****************************************

*   mainMenu()
*
*   action: Display the main menu of options (demoList), and get a selection
*           from the user.
*
*   input:  None.
*
*   return: o   Index of demoList choice (guaranteed to be a valid item
*               of demoList)
*
*   alters: o   Rewrites screen display
*
function mainMenu
private choice

*   Keep going while the user's pick is invalid
choice = 0
do while ((choice < 1) .or. (choice > len(demoList)))

    *   Display various status info
    clear
    cmStatus("ClipMore Demo Main Menu")
    atSayCenter(24, "Pick any selection (achoice())")
    @ 3, 0, maxRow() - 2, maxCol() box B_DOUBLE

    *   Let the user pick a choice
    choice = achoice(4, 1, maxRow() - 3, maxCol() - 1, demoList)
enddo

*   Return what he got
return choice


*   qindex()
*
*   action: Creates an index on the specified expression.  It will check
*           to see if the specified file exists before creating the index.
*           If it already exists then it will simply return assuming that
*           the index has already been created.
*
*   input:  expr:   Expression on which the index is to be created.  This
*                   parameter must be provided.
*           name:   Name of the index file to create.  If it is not provided,
*                   then an '_' prepended to expr is used as the name.
*                   (This will only work for simple field names)
*
*   return: .T. (always)
*
*   alters: o   If it needs to create the index:
*               o   Writes to screen.
*               o   Creates index.

function qindex

*   If a name is provided, use it, otherwise use the expression with an _
*   prepended as the name
if (pcount() = 2)
    parameters expr, name
else
    parameters expr
    private name
    name = "_" + expr
    if (len(name) > 8)
        name = substr(name, 1, 8)
    endif
endif

*   If the file doesn't already exist, then create the index,
*   letting user know what's going on
if (!file(name + indexExt()))
	initBar(expr, name)
	ordCondSet(nil, nil, nil, nil, {||dispBar()}, recCount() / 50)
	dbCreateIndex(name, expr, &("{||" + expr + "}"), nil)
endif
return .t.


*   initBar()
*
*   action: Does the initial display of the index bar (stuff that doesn't
*			change as the index is being created).
*
*   input:  expr:   Expression on which the index is to be created.
*           name:   Name of the index file to create.
*
*   return: NIL
*
*   alters: Writes to the screen.

function initBar(expr, name)
local str := "INDEX'ing ON " + expr + " TO " + name
@ 9, 10 clear to 13, 70
@ 9, 10, 13, 70 box B_DOUBLE
@ 11, 10 say chr(199) + replicate(chr(196), 59) + chr(182)
@ 10, (10 + (59 - len(str)) / 2) say str
return nil


*   dispBar()
*
*   action: Displays the percent complete with a reverse video bar going
*			through the percentage
*
*   input:  None.
*
*   return: .T. (always)
*
*   alters: Writes to the screen.

function dispBar()
local percent := recno() / recCount()
local str := space(27) + str(percent * 100, 3, 0) + "%" + space(28)
local numReverse := 59 * percent
set color to n/w
@ 12, 11 say left(str, numReverse)
set color to w/n
@ 12, 11 + numReverse say substr(str, numReverse + 1)
return .t.


*   GenInd()
*
*   action: Creates all indexes which aren't already created.
*
*   input:  None.
*
*   return: .T. (always)
*
*   alters: o   Writes to screen.
*           o   Creates indexes as needed.

function GenInd
atSayCenter(0, "Demo Initialization: Creating Indexes (using Comix)")
?

*	If the file doesn't exist, tell user what went wrong
if (!file("DEMO.DBF"))
	? "Unable to bring demo.dbf into use"
	? "Use dbgen to generate demo.dbf before running DEMO.EXE"
	?
	? "The syntax is:"
	?
	? "dbgen <number of records>"
	?
	quit
endif

*   Create all the indexes that we need
use demo
qindex("age")
qindex("last")
qindex("lastcall")
qindex("salary")
qindex("demoUDF(first, last)", "_udf")
return .t.


*   demoUDF()
*
*   action: A simple UDF to illustrate the use of UDF's in optimization.
*           Returns the first two characters of str1 + the first two
*           characters of str2.
*
*   input:  str1:   First string.
*           str2:   Second string.
*
*   return: A string per action.
*
*   alters: None.
*

function demoUDF
parameters str1, str2
return substr(str1, 1, 2) + substr(str2, 1, 2)


*   cmStatus()
*
*   action: Displays the given msg on line 0, and ClipMore's status
*           on line 1.
*
*   input:  msg:    Message to be displayed on line 0.
*
*   return: .t. always
*
*   alters: None.
*

function cmStatus
parameters msg
private status
atSayCenter(0, msg)
set color to n/w
status = iif(_cmFlag, " ON ", " OFF ")
atSayCenter(1, replicate(status, 3) + " " + cmVersion() + ;
        " (toggle with cmOnOff())  " + replicate(status, 3))
set color to w/n
return .t.


*   clrLine()
*
*   action: Clears the given line
*
*   input:  row:    Row to clear
*
*   return: .t. (always)
*
*   alters: o   Clears line per action

function clrLine
parameters row
@ row, 0 clear to row, maxCol()
return .t.


*   atSayCenter()
*
*   action: @ say's a message centered on the given row.
*
*   input:  row:    Row to display message.
*           msg:    Message to be displayed.
*
*   return: .t. (always)
*
*   alters: o   Writes to screen
*

function atSayCenter
parameters row, msg
clrLine(row)
@ row, (maxCol() - len(msg)) / 2 say msg
return .t.


    ******************************************************************
    ****** Special ClipMore functions (which use or do not use *******
    ****** ClipMore based on the setting of _cmFlag)           ******* 
    ******************************************************************

*   cmFilterP()
*
*   action: Shell around cmFilter() which will or will not use cmFilter()
*           depending on the setting of _cmFlag.
*
*   input:  condition:  Condition which records should meet.
*
*   return: .t. (always)
*
*   alters: o   Writes to screen
*

function cmFilterP
parameters condition
private retVal

*   Display what the condition is
atSayCenter(2, "condition: " + condition)

*   If we're to use ClipMore, call cmFilter(), otherwise, SET FILTER TO
if (_cmFlag)
    ? "cmFilter() in progress..."
    cmFilter(condition)
    ?? "Done"
else
    set filter to &condition
endif
go top
return 0


*   cmReFiltP()
*
*   action: Shell around cmReFilter() which will or will not use
*           cmReFilter() depending on the setting of _cmFlag.
*
*   input:  condition:  Condition which records should meet.
*
*   return: .t. (always)
*
*   alters: o   Writes to screen
*

function cmReFiltP
parameters cond

*   If we're to use ClipMore, call cmReFilter(), otherwise, SET FILTER TO
if (_cmFlag)
    ? "cmReFilter() in progress..."
    cmReFilter(cond)
	retVal = cmFiltCount()
    ?? "Done"
else
    cond = trim(cond)
    cond = iif(empty(dbfilter()), cond, dbfilter() + " .and. " + cond)
    set filter to &cond.

    *   Get the count
    count to retVal
endif

*   Go top
go top
return retVal


    ******************************************************************
    ****** Specialized dbedit (supporting reorder)              ******
    ******************************************************************

*   dbeditBar()
*
*   action: Does a dbedit().
*
*   input:  dbeditRow:  Row on which dbedit should begin.
*
*   return: Same as dbedit()
*
*   alters: o   Writes to screen
*
function dbeditBar
parameters dbeditRow

*   Set up prevRecno here for use in dbeditUDF (so it stays in scope during
*   multiple dbeditUDF invocations)
private prevRecno
prevRecno = 0

*   Draw the box around the dbedit() area
@ dbeditRow, 0, maxRow()-1, maxCol() box B_DOUBLE

*   Tell user valid keys
atSayCenter(maxRow(), "Alt-R: Reorder    ESC: Exit")

*   Now do the dbedit with dbeditUDF as the processing function
return dbedit(dbeditRow+1, 1, maxRow()-2, maxCol()-1, .t., [dbeditUDF])


*   dbeditUDF()
*
*   action: The UDF which is called by the dbedit() started by dbeditBar().
*           This does the following:
*           o   Displays the scroll bar as records are traversed.
*           o   Processes Alt-R to dynamically reorder.
*           o   Processes Alt-S to move the scroll bar (simulation of a
*               mouse).
*
*   expect: prevRecno: should be set up as previous record number by
*                       invoking function.
*
*   input:  status: Status of dbedit().
*           fld_ptr:    Not used.
*
*   return: 0:  to quit.
*           1:  to keep going.
*
*   alters: o   Displays scroll bar
*
function dbeditUDF
parameters status, fld_ptr
private retVal

*   Unknown key has been pressed
if (status = 4)

    *   Make sure the scroll bar gets adjusted and the screen is repainted
    *   after we do our changes (if really is an unknown key, retVal
    *   will be adjusted to exit in the otherwise clause below)
    prevRecno = 0
    retVal = 2

    do case

        *   if Alt-R (reorder records)
        case (lastkey() = 275)

            *   Do the re-ordering
            dbeditReOrder()
    
        otherwise
            *   Really is an unknown key, so quit
            retVal = 0
    endcase
else
    *   In all other cases, continue
    retVal = 1
endif

*   Return appropriate action
return retVal


*   dbeditReOrder()
*
*   action: Provides a data driven pop up menu of available indexes to
*           reorder by.
*
*   input:  None.
*
*   return: .t. (always)
*
*   alters: o   Writes to screen as needed
*
function dbeditReOrder
private scr, i, newOrder, indexCnt

*   Set up indexCnt
indexCnt = cmxIndexCount()
declare indexList[indexCnt]

*   Build the array of index keys
i = 1
do while (i <= indexCnt)
    indexList[i] = substr(indexKey(i)+space(40), 1, 40)
    i = i + 1
enddo

*   Save the screen
scr = savescreen(8, 19, 9 + indexCnt, 61)

*   Let the user pick a choice
@ 8, 19, 9 + indexCnt, 61 box B_DOUBLE
@ 9, 20 clear to 8 + indexCnt, 60
newOrder = achoice(9, 20, 8 + indexCnt, 60, indexList)

*   Set to the new ordering (unless user ESC'd out)
if (lastkey() <> 27)
    @ 9, 20 clear to 8 + indexCnt, 60
    @ 9 + (indexCnt / 2), 25 say "SET ORDER TO in progress..."
    set order to newOrder
    ?? "Done"
endif

*   Restore the screen
restscreen(8, 19, 9 + indexCnt, 61, scr)

return .t.
