/*
    csyobjec.prg - Class(y) Object class

    Copyright (c) 1992 Anton van Straaten

    All classes ultimately inherit from the Object class, since if no
    superclass is specified when creating a class, Object is used
    automatically.

    The Object class defines methods which are absolutely essential to the
    correct functioning of the object system.  These include 'new',
    'classH', and 'className', as well as a number of error handling
    methods.

    The new() message is mapped to the NULL method, which does nothing.  If
    a class or its superclasses do not implement their own 'new' method,
    this one will ultimately be invoked when Class's 'new' method sends a
    'new' message to the newly created instance.

    Mon  01-13-1992  11:18:49   avs - creation
    Thu  03-26-1992  12:46:55   avs
    Fri  06-19-1992  21:20:54   avs
*/

/*
    The following declaration forces linkage of a module which supports the
    Clipper debugger's standard object inspector's behavior.  If you plan to
    always use Class(y)'s enhanced object inspector, you can comment out this
    line.
*/
EXTERNAL CsyOldInsp


#include "class(y).ch"


CREATE CLASS Object FROM CsyNoClass METACLASS Class   // tbd: CREATE ROOT CLASS?
EXPORT:
    MESSAGE init        METHOD CsyNull

    METHOD  basicSize
    METHOD  become

    // classH and className are required by Clipper.  They are also
    // referred to by the Class(y) kernel, so the methods must be public.
    MESSAGE classH      METHOD _csyClassH
    MESSAGE className   METHOD _csyClsNam

    METHOD  isEqual
    METHOD  isKindOf
    METHOD  isScalar
    METHOD  copy                            // shallow copy
    METHOD  deepCopy
    MESSAGE null        METHOD CsyNull
    MESSAGE size        METHOD basicSize

    MESSAGE eval        METHOD CsyNull
    MESSAGE exec        METHOD CsyNull

    METHOD  error

    MESSAGE hash        METHOD deferred

    METHOD  deferred
    METHOD  protectErr
    METHOD  hiddenErr
    METHOD  msgNotFound
    METHOD  readOnlyErr
    METHOD  wrongClass
    METHOD  badMethod
    /*
        The following methods are special, and are declared for every
        class automatically.

        METHOD class
        METHOD super
        METHOD perform
    */
END CLASS


METHOD basicSize
RETURN LEN( self )


METHOD become( newSelf )
    ASIZE( self, LEN( newSelf ) )
    ACOPY( newSelf, self )
    CsySetDict( self, newSelf:classH )
RETURN self


FUNCTION _csyClsNam
RETURN QSELF():class:name


/*
    classH()

    This method will not be affected by reference objects, which means it
    will not allow hierarchy navigation in the debugger's native object
    inspector.  A C routine has been used instead.

METHOD classH
RETURN ::class:handle
*/


/*
    copy()

    Return a copy of the receiver which shares the receiver's instance
    variables.
*/

METHOD copy
RETURN ACOPY( self, ::class:basicNew() )


/*
    deepCopy()

    Return a copy of the receiver with shallow copies of each instance
    variable.  This is not as 'deep' as the kind of copy done by ACLONE(),
    but ACLONE() cannot handle self-referencing objects or arrays.
*/

METHOD deepCopy
    LOCAL i
    LOCAL nLen   := LEN(self)
    LOCAL newObj := ::class:basicNew()

    FOR i := 1 TO nLen
        // using :deepCopy() below instead of :copy() will result
        // in a complete clone, but self-referencing objects would
        // cause infinite recursion.
        newObj[i] := self[i]:copy()
    NEXT
RETURN newObj


METHOD isEqual( o )
RETURN self == o


/*
    The following two methods, isKindOf and isScalar, could normally be
    implemented by delegation in the class declaration, since all they do
    is pass a message through to the object's class.  However, the Object
    class is created too early on in the Class(y) initialization process to
    be able to use delegation.
*/

METHOD isKindOf( oClass )
RETURN ::class:inheritsFrom( oClass )


METHOD isScalar
RETURN ::class:isScalar


/*
    :null() (CsyNull())

    Do nothing.  This method can be mapped to from subclasses, if required,
    using a command of the form:

        MESSAGE foo IS NULL

    Note we cheat by not using the METHOD command, so we don't get the line
    'self := qself()', which would be useless in this case.
*/

FUNCTION CsyNull
RETURN NIL


#include "error.ch"
#include "csyerror.ch"

METHOD PROCEDURE error(subCode, desc, className, msg)
    local e := ErrorNew()

    // tbd: get rid of following tests?
    IF className == NIL .or. empty(className)
        className := ""
    ELSE
        className += ":"
    END

    IF msg == NIL
        msg := ""
    END

    e:genCode   := EG_NOMETHOD      // 5.01 also has EG_NOVARMETHOD ("No exported var")
    e:subSystem := "CLASS(Y)"
    e:subCode   := subcode
    e:description := desc
    e:operation := className + msg
    e:severity  := ES_ERROR

    // allow override eg. ::handleErr( e )?
    EVAL(ErrorBlock(), e)
RETURN


METHOD PROCEDURE deferred
    ::error(CSYERR_DEFERRED, "Message should be implemented by subclass in class", ::className)
RETURN

METHOD PROCEDURE protectErr( cMsg )
    ::error(CSYERR_PROTECTED, "Scope violation (protected)", ::className, cMsg)
RETURN

METHOD PROCEDURE hiddenErr( cMsg )
    ::error(CSYERR_HIDDEN, "Scope violation (hidden)", ::className, cMsg)
RETURN

METHOD PROCEDURE msgNotFound( cMsg )
    ::error(CSYERR_NOTFOUND, "Message not found", ::className, cMsg)
RETURN

METHOD PROCEDURE readOnlyErr( cMsg )
    ::error(CSYERR_READONLY, "Scope violation (read-only)", ::className, cMsg)
RETURN

METHOD PROCEDURE wrongClass( cMsg )
    ::error(CSYERR_WRONGCLASS, "Assigned value is wrong class", ::className, cMsg)
RETURN

METHOD PROCEDURE badMethod( cMsg )
    ::error(CSYERR_BADMETHOD, "Bad or missing method reference", ::className, cMsg)
RETURN


// eof csyobjec.prg
