/*
    csyarray.prg - Class(y) Array class

    Copyright (c) 1992 Anton van Straaten

    This Array class is automatically "attached" by Class(y) to all Clipper
    arrays, so that standard arrays will respond to the messages defined
    here and in the superclass, Object.

    The Class(y) kernel makes use of some of these methods, so the
    functionality of the existing methods in this class should not be
    changed.  However, methods can be added as desired.  This module is
    included in CLASSY.LIB; if you make changes, you will need to ensure
    that your changed module is linked in, either by explicitly specifying
    it to the linker, or by replacing CSYARRAY.OBJ in the library.  Such
    changes are made at your own risk!

    The Array class is not inherited from the ScalarObject class, since it
    behaves more like a real object in many ways than do other scalars.
    Specifically, the mechanism for copying of arrays is the same as for
    objects, so we do not want to inherit the copying methods defined in
    the ScalarObject class.

    Nonetheless, we may still wish to distinguish between Clipper arrays
    and "true" objects.  By creating the Array class as an instance of the
    metaclass ScalarClass, we ensure that arrays will respond to the
    'isScalar' message with .T.

    Operations within methods in this class are done, in most cases, using
    the Clipper primitive functions rather than the corresponding methods;
    for example 'AADD' is used directly instead of 'append' in the
    collect() method.  This provides better performance, although it makes
    it more difficult to inherit from the Array class, should you want to
    do that.
*/

#include "csylite.ch"


CREATE CLASS Array FUNCTION csyArray METACLASS ScalarClass
EXPORT:
    METHOD  init

    MESSAGE at      METHOD atIndex      // 'at' is a reserved word
    METHOD  atPut
    MESSAGE add     METHOD append       // 'add' and 'addAll' are
    METHOD  addAll                      // standard Collection messages
    METHOD  append
    METHOD  collect
    METHOD  copy
    METHOD  do
    // METHOD  doUntil
    MESSAGE delete  METHOD deleteElem   // 'delete' is a reserved word
    METHOD  indexOf
    METHOD  remove
    METHOD  scan
    METHOD  size
    METHOD  _size                       // assignment method
END CLASS


METHOD PROCEDURE init( nElements )
    ::size := IF(nElements == NIL, 0, nElements)
RETURN


// duplicate of Collection method
METHOD FUNCTION addAll( otherCollection )
    otherCollection:do( { |e| ::add(e) } )
RETURN self


METHOD FUNCTION atIndex( n )
RETURN self[ n ]


METHOD FUNCTION atPut( n, x )
RETURN self[ n ] := x


METHOD FUNCTION append( x )
    AADD( self, x )
RETURN .t.


METHOD FUNCTION collect( b )
    LOCAL i, currElem
    LOCAL result[0]
    LOCAL nElems := LEN(self)

    FOR i := 1 TO nElems
        currElem := self[i]
        IF EVAL( b, currElem )
            AADD( result, currElem )
        END
    NEXT
RETURN result


/*
    copy()

    The copy() method inherited from Object results in true objects of class
    Array.  To avoid confusing programs which rely on arrays being type 'A',
    we override copy() here.
*/

METHOD FUNCTION copy
RETURN ACOPY(self, ARRAY( LEN( self ) ) )


METHOD FUNCTION deleteElem( n )
    LOCAL nElems := LEN(self)

    IF n > 0 .AND. n <= nElems
        ADEL( self, n )
        ASIZE( self, nElems - 1 )
    END
RETURN self


/*
    do()

    We could use AEVAL( self, b ) here, but that would preclude 'b' from
    being anything other than a code block.
*/

METHOD FUNCTION do( b )
    LOCAL i

    // evaluating the length each time allows length to change during processing
    // although only changes after the current element will be noticed
    FOR i := 1 TO LEN(self)
        b:eval( self[ i ], i )
    NEXT
RETURN self


/*
METHOD doUntil( b )
RETURN ...
*/


/*
    :indexOf()

    We don't use ASCAN() below because it does an inexact match. This does
    not work when trying to match objects, unless a code block is used,
    which takes about 22% longer than the loop below.
*/

METHOD FUNCTION indexOf( x )
    LOCAL i
    LOCAL nElems := LEN(self)

    FOR i := 1 TO nElems
        IF self[i] == x
            RETURN i
        END
    NEXT
RETURN 0


METHOD PROCEDURE remove( e )
    ::delete( ::indexOf( e ) )
RETURN


METHOD FUNCTION scan( b )
RETURN ASCAN( self, b )


// Although this is defined in Object, it is
// required here for "chicken and egg" reasons.
METHOD FUNCTION size
RETURN LEN( self )


/*
    _size()

    This "assignment method" is executed in response to an assignment of
    the following form:

        obj:size := nLen

    This takes advantage of the internal implementation of assignment to
    instance variables in Clipper 5.01.  The next major version of Clipper
    is likely to explicitly provide such a capability; however, the current
    implementation - prepending an underscore to the message name to
    indicate an assignment method - is likely to change.

    This method allows arrays to be resized by assignment, as in the
    following code:

        LOCAL arr[10]

        arr:size := 5     // change array's size to 5!

    The sample program ARRLEN.PRG gives a simple demonstration of this.
*/

METHOD FUNCTION _size( newSize )
    ASIZE( self, newSize )
RETURN newSize  // so that assignment works according to standard rules


// eof csyarray.prg
