/*
    table.prg

    Copyright (c) 1992 Anton van Straaten

    The Table metaclass.  A metaclass is a class of classes.  Instances
    of such classes are themselves classes.

    What makes Table a metaclass is the fact that it is inherited from the
    Class class, which itself is a metaclass.

    We will call instances of the Table metaclass 'table class objects'.  A
    different table class object is required for each DBF table being used,
    since each table class object reflects the structure of a particular
    DBF table.

    As with any class, variables defined in Table will exist in every
    instance of the class.  For example, every table class object will
    contain its own copy of the instance variable nSelectArea.

    Since table class objects (instances of Table) are themselves classes,
    they in turn can have instances.  It is these instances which will be
    used to hold the information corresponding to a particular record.  We
    will call these instances 'record objects', although they are in fact
    subclasses of the Record class.  See the comments for the new() method.

    Thu  02-27-1992  10:38:51   avs
*/

#include "class(y).ch"


CREATE CLASS Table FROM Class METACLASS Class
EXPORT:
    VAR     nSelectArea READONLY
    VAR     fields      READONLY

    METHOD  init
    METHOD  alias
    MESSAGE fieldName   METHOD fldName      // FIELDNAME is a reserved word
    METHOD  numFields

    /*
        The following methods all relate to a table cursor.  DBF tables
        limit what we can do with a cursor to some extent.  In a more
        sophisticated system, it might make sense to have a separate Cursor
        class which might be associated with, say, a TupleSet class.  This
        would allow multiple views and traversals of the same database
        simultaneously, using multiple set objects referring to the same
        table object.
    */

    MESSAGE recNo       METHOD tableRecNo   // RECNO is a reserved word
    MESSAGE delete      METHOD tableDelete  // DELETE is a reserved word
    METHOD  isAtBof
    METHOD  isAtEof
    METHOD  goto
    METHOD  goTop
    METHOD  goBottom
    METHOD  skip
    MESSAGE lock    METHOD  lockAtCursor    // LOCK is a reserved word
    METHOD  unlock

END CLASS


/*
    :init()

    When the user executes a Table():new(filename), he receives a new
    table class object back which reflects the structure of the specified
    DBF table.

    This method opens the specified table, and initializes the table class
    object (self) to reflect the structure of that table (ie. by defining
    instance variables with the same names as the fields in the table).

    Note that in the call to the Class method initialize(), we set the
    superclass of the class being described (self) to Record.  This means
    that all record objects (instances of table class objects) will share
    the same methods, defined in their superclass, Record.  See RECORD.PRG
    for more information.
*/

METHOD PROCEDURE init( cTableName ), ( cTableName, Record() )
    LOCAL i, nScope                         // nScope reqd by EXPORT:

    SELECT 0
    USE (cTableName)

    ::nSelectArea := SELECT()
    ::fields := ARRAY( FCOUNT() )

    EXPORT:

    FOR i := 1 TO FCOUNT()
        ::fields[i] := FIELDNAME(i)
        VAR (FIELDNAME(i))
    NEXT i

    ::makeClass()                           // class is complete
RETURN


/*
    alias()

    The name of the class described by 'self' is the same as the alias.
    Of course, this does not allow for an alias with a different name
    from the table.  This could easily be corrected.  Using an object-based
    system, though, aliases should become virtually redundant.  Table
    objects themselves provides the means to access a table.  This method
    is only included as an example of the sort of method which might belong
    in the Table class.
*/

METHOD alias
RETURN ::name


/*
    :fieldName() (fldName())

    The Class class used to have an instance variable name 'variables' which
    contained the names of instance variables.  It no longer does, we we have
    saved them in the 'fields' variable in the Table class.
*/

METHOD fldName( n )
RETURN ::fields[n]


METHOD numFields
RETURN LEN(::fields)


/*
    Table 'cursor' methods follow.  Many of these methods just return the
    value of the underlying Clipper function (if any).  Some of these
    return values may not be meaningful.
*/

METHOD tableRecNo
RETURN (::nSelectArea)->(RECNO())

METHOD isAtBof
RETURN (::nSelectArea)->(BOF())

METHOD isAtEof
RETURN (::nSelectArea)->(EOF())

METHOD tableDelete
RETURN dbDelete()

METHOD goto( n )
RETURN dbGoTo(n)

METHOD goTop
RETURN (::nSelectArea)->(dbGoTop())

METHOD goBottom
RETURN (::nSelectArea)->(dbGoBottom())

METHOD skip( n )
RETURN (::nSelectArea)->(dbSkip( n ))


/*
    The lock() and unlock() methods here are cursor related, ie. they
    should lock or unlock the record referred to by the cursor in the
    current table.  Methods of the same name exist in the Record class;
    their job is to lock or unlock the records associated with specific
    record objects.

    This duplication might point out a design flaw, which is that the
    cursor should perhaps itself be a record object.  There are problems
    with this approach, however.  Needs further thought.
*/

METHOD lockAtCursor
    SELECT (::nSelectArea)
    WHILE !RLOCK()
    END
RETURN .T.


METHOD unlock
RETURN (::nSelectArea)->(dbUnlock())


// eof table.prg
