/*
    record.prg

    Copyright (c) 1992 Anton van Straaten

    The Record class is an abstract base class which other record classes
    inherit from, so that all record objects share the methods defined in
    this class.  The new() method in class Table ensures that Record is
    the superclass for all record objects.  See TABLE.PRG for more
    information.

    Methods such as read() and write() rely on the fact that subclasses of
    Record are created such that the indexes of their instance variables
    correspond to the ordinal positions of fields in the database.

    Methods in the Record class should apply to individual records rather
    than a table as a whole.  Methods which apply to a whole table, such as
    fieldName(), are implemented in the Table class.

    The recNo variable is included in this class mainly for illustration.
    Keeping a record number in every record object implies that multiple
    record objects from the same table will exist at one time.  Managing
    this successfully would require better handling of locking issues than
    has has been implemented here.

    The recNo variable is contained in the first element in the instance
    variable array.  This means that field numbers need to be offset by
    one to correspond to instance variable index numbers.

    The lock() and unlock() methods are also included for illustration
    only.  For safe operation, the ability to lock multiple records in the
    same table simultaneously would be required.  This facility does not
    exist in Clipper's native DBF implementation.

    Thu  02-27-1992  10:56:47   avs
*/

#include "class(y).ch"


CREATE CLASS Record
EXPORT:
    VAR     recNo   READONLY

    MESSAGE init    METHOD read // initialize new object by doing a read
    METHOD  read                // reads from current rec into object (scatter)
    METHOD  write               // writes from object into current rec (gather)
    MESSAGE lock    METHOD lockRecord   // LOCK is a reserved word
    METHOD  unlock
END CLASS


METHOD read
    LOCAL i

    SELECT (::class:nSelectArea)

    ::recNo := RECNO()

    /*
        We don't use Record's :lock() here because we know we're on the
        correct record.  This sort of thing reflects our weak attempt at
        encapsulating Clipper's single-active-record mentality into a more
        flexible system.  Ideally, we shouldn't have to check whether the
        cursor is in the right place all the time, as is done in
        lockRecord() below; we should just be able to pass the desired
        record number to a system level function.
    */
    ::class:lock()

    FOR i := 1 TO FCOUNT()
        self[i+1] := FIELDGET(i)
    NEXT i
RETURN self


METHOD write
    LOCAL i

    SELECT (::class:nSelectArea)

    IF RECNO() <> ::recNo
        GOTO ::recNo
        ::class:lock()
    END
    // If we were on the correct record number, we assume the record is
    // is locked - a dubious assumption given the way that locking has been
    // implemented here.

    FOR i := 1 TO FCOUNT()
        FIELDPUT(i, self[i+1])
    NEXT i

    ::unlock()

RETURN self


/*
    :lock()

    Has a serious drawback in that it will result in the currently locked
    record, if any, being unlocked.  Also needs failure/deadlock handling
    logic.
*/

METHOD lockRecord
    SELECT (::class:nSelectArea)

    IF ::recNo <> RECNO()
        GOTO ::recNo
    END
    WHILE !RLOCK()
    END
RETURN .t.


METHOD unlock
RETURN (::class:nSelectArea)->(dbUnlock())


// eof record.prg
