/*
    dbrowse.prg

    Adapted from Clipper's sample TBDEMO.PRG.

    Demonstrates inheriting from Clipper's predefined classes,
    in this case TBROWSE.

    Portions copyright Computer Associates.
*/

#include "class(y).ch"
#include "inkey.ch"
#include "setcurs.ch"


CREATE CLASS dBrowse FROM TBrowse
    VAR     appendMode

EXPORT:
    METHOD  init
    METHOD  autoFields
    METHOD  exec
    METHOD  goBottom
    METHOD  goTop
    METHOD  skipper
    METHOD  editCell
    METHOD  doGet
END CLASS


METHOD init( nTop, nLeft, nBottom, nRight ), ( nTop, nLeft, nBottom, nRight )
    ::headSep := ""
    ::colSep  := "  "

    // the skipBlock is still necessary, since there is no skip method.
    ::skipBlock := {|x| ::skipper(x) }

    // the caller might want to change this, and is free to do so
    // especially on a monochrome screen
    ::colorSpec := "N/W, N/BG, B/W, B/BG, B/W, B/BG, R/W, B/R"
RETURN self


/*
    :autoFields

    add a column for each field in the current workarea
*/

METHOD autoFields
    LOCAL n, cType

    // add a column for record number
    LOCAL column := TBColumn():new( "  Rec #", { || RecNo() } )
    ::addColumn( column )

    FOR n = 1 TO FCount()

        // make the new column
        column := TBColumn():new( FieldName(n), ;
                                  FieldWBlock(FieldName(n), Select()) )

        // evaluate the block once to get the field's data type
        cType := VALTYPE( EVAL( column:block ) )

        // if numeric, highlight with a color block
        IF cType == "N"
            column:defColor := { 5, 6 }
            column:colorBlock := { |x| if( x < 0, {7, 8}, {5, 6} ) }
        ELSE
            column:defColor := { 3, 4 }
        END

        ::addColumn( column )
    NEXT
RETURN self


METHOD goBottom
    GO BOTTOM
    SKIP ::rowPos - ::rowCount
    ::rowPos := ::rowCount
    ::refreshAll()
RETURN self


METHOD goTop
    GO TOP
    ::rowPos := 1
    ::refreshAll()
RETURN self


METHOD exec
    LOCAL nKey
    LOCAL nCursSave := SetCursor( 0 )
    LOCAL lMore := .T.

    ::appendMode := .F.

    WHILE lMore
        // don't allow cursor to move into frozen columns
        IF ::colPos <= ::freeze
            ::colPos := ::freeze + 1
        END

        // stabilize the display
        WHILE !::stabilize()
            nKey := INKEY()
            IF nKey <> 0
                EXIT            // abort if a key is waiting
            END
        END

        IF ::stable
            // display is stable
            IF ::hitBottom .AND. !::appendMode
                // banged against EOF; go into append mode
                ::appendMode := .t.
                nKey := K_DOWN
            ELSE
                IF ::hitTop .OR. ::hitBottom
                    Tone( 125, 0 )
                END

                // Make sure that the current record is showing up-to-date
                // data in case we are on a network.
                ::refreshCurrent()
                WHILE !::stabilize()
                END

                // everything's done; just wait for a key
                nKey := INKEY( 0 )
            END
        END

        // process key
        DO CASE
        CASE nKey == K_DOWN
            ::down()

        CASE nKey == K_UP
            ::up()

            IF ::appendMode
                ::appendMode := .f.
                ::refreshAll()
            END

        CASE nKey == K_PGDN
            ::pageDown()

        CASE nKey == K_PGUP
            ::pageUp()
            IF ::appendMode
                ::appendMode := .f.
                ::refreshAll()
            END

        CASE nKey == K_CTRL_PGUP
            ::goTop()
            ::appendMode := .f.

        CASE nKey == K_CTRL_PGDN
            ::goBottom()
            ::appendMode := .f.

        CASE nKey == K_RIGHT
            ::right()

        CASE nKey == K_LEFT
            ::left()

        CASE nKey == K_HOME
            ::home()

        CASE nKey == K_END
            ::end()

        CASE nKey == K_CTRL_LEFT
            ::panLeft()

        CASE nKey == K_CTRL_RIGHT
            ::panRight()

        CASE nKey == K_CTRL_HOME
            ::panHome()

        CASE nKey == K_CTRL_END
            ::panEnd()

        CASE nKey == K_ESC
            lMore := .f.

        CASE nKey == K_RETURN
            ::editCell()

        OTHERWISE
            KEYBOARD CHR( nKey )
            ::editCell()
        END
    END
    SetCursor(nCursSave)
RETURN self


/*
    skipper()
*/

METHOD skipper( n )
    LOCAL i := 0

    IF LastRec() <> 0
        IF n == 0
            SKIP 0
        ELSEIF n > 0 .AND. RecNo() <> LastRec() + 1
            WHILE i < n
                SKIP 1
                IF EOF()
                    IF ::appendMode
                        i++
                    ELSE
                        skip -1
                    END
                    EXIT
                END
                i++
            END
        ELSEIF n < 0
            WHILE i > n
                SKIP -1
                IF BOF()
                    EXIT
                END
                i--
            END
        END
    END
RETURN i


METHOD editCell
    // Save pertinent info about current record
    LOCAL xKeyVal := IF( EMPTY( IndexKey() ), NIL, &( IndexKey() ) )
    LOCAL nRec    := RecNo()

    ::doGet()
    ::appendMode := .F.

    IF EMPTY( IndexKey() ) .OR. ( xKeyVal == &( IndexKey() ) )
        // make sure browse is correctly updated
        ::refreshCurrent()
    ELSE
        // record may have moved relative to other records
        ::refreshAll()

        WHILE !::stabilize()
        END

        WHILE RecNo() <> nRec
            ::up()
            WHILE !::stabilize()
            END
        END
    END
RETURN self


METHOD doGet
    LOCAL column, get, nKey

    // save state
    LOCAL lScoreSave := Set( _SET_SCOREBOARD, .f. )
    LOCAL lExitSave  := Set( _SET_EXIT, .t. )
    LOCAL bInsSave   := SetKey( K_INS )

    // make sure browse is stable
    WHILE !::stabilize()
    END

    // if confirming new record, append blank
    IF ::appendMode .AND. RecNo() == LastRec() + 1
        APPEND BLANK
    END

    // set insert key to toggle insert mode and cursor
    SetKey( K_INS, ;
        { || SetCursor( if(ReadInsert(!ReadInsert()), SC_NORMAL, SC_INSERT)) };
          )

    // initial cursor setting
    SetCursor( IF( ReadInsert(), SC_INSERT, SC_NORMAL ) )

    // get column object from browse
    column := ::getColumn( ::colPos )

    // create a corresponding GET
    get := Get():new( ROW(), COL(), column:block, column:heading,, ::colorSpec )

    // read it
    ReadModal( { get } )

    // restore state
    SetCursor( 0 )
    Set( _SET_SCOREBOARD, lScoreSave )
    Set( _SET_EXIT, lExitSave )
    SetKey( K_INS, bInsSave )

    // check exit key from get
    nKey := LastKey()
    IF nKey == K_UP .OR. nKey == K_DOWN .OR. nKey == K_PGUP .OR. nKey == K_PGDN
        KEYBOARD CHR( nKey )
    END
RETURN self


// eof dbrowse.prg
