//===================================================================================
//
//     File:			EdEdText.cp
//     Release Version:	$Revision$
//
//     COPYRIGHT 1993 SYMANTEC CORPORATION. ALL RIGHTS RESERVED. UNPUBLISHED RIGHTS
//     RESERVED UNDER THE COPYRIGHT LAWS OF THE UNITED STATES. USE OF COPYRIGHT
//     NOTICE IS PRECAUTIONARY ONLY AND DOES NOT IMPLY PUBLICATION OR DISCLOSURE.
//
//     THIS SOFTWARE CONTAINS PROPRIETARY AND CONFIDENTIAL INFORMATION OF SYMANTEC
//     CORPORATION. USE, DISCLOSURE, OR REPRODUCTION IS PROHIBITED WITHOUT THE PRIOR
//     EXPRESS WRITTEN PERMISSION OF SYMANTEC CORPORATION.
//
//     RESTRICTED RIGHTS LEGEND
//     Use, duplication, or disclosure by the Government is subject to restrictions
//     as set forth in subparagraph (c)(l)(ii) of the Rights in Technical Data and
//     Computer Software clause at DFARS 252.227-7013.
//     Symantec Corporation, 10201 Torre Avenue, Cupertino, CA 95014.
//
//===================================================================================

#ifndef EDITTEXT_H
#include "EditText.h"
#endif

#ifndef EDEDTEXT_H
#include "EdEdText.h"
#endif

#ifndef EDAPP_H
#include "EdApp.h"
#endif

#ifndef BRMNUBAR_H
#include <BRMnuBar.h>
#endif

#ifndef BRDLGDIR_H
#include <BRDlgDir.h>
#endif

#ifndef BRSYSTEM_H
#include <BRSystem.h>
#endif

#ifndef BRSTRTOO_H
#include <BRStrToo.h>
#endif

DEFINE_CLASS(cEdEditText, BR_CEditText);

//-----------------------------------------------------------------------------------
// cEdEditText::cEdEditText
//
// The constructor for the cEdEditText class
//-----------------------------------------------------------------------------------

cEdEditText::cEdEditText(BR_CView					*enclosure,
                     	 BR_CBureaucrat				*supervisor,
						 BR_Distance				width,
						 BR_Distance				height,
						 BR_Coordinate				x,
						 BR_Coordinate				y,
						 ViewSizing					horzSizing,
						 ViewSizing					vertSizing,
						 ViewUnits					viewUnit,
						 const BR_CRectangle		&boundsRect)
			:BR_CEditText(enclosure, 
						  supervisor,
						  width,
						  height,
						  x,
						  y,
						  horzSizing,
						  vertSizing,
						  viewUnit,
						  boundsRect)
{						
}

//-----------------------------------------------------------------------------------
// cEdEditText::~cEdEditText
//
// Destructor for cEdEditText
//-----------------------------------------------------------------------------------

cEdEditText::~cEdEditText()
{
}

//-----------------------------------------------------------------------------------
// cEdEditText::UpdateMenus
//
// Activates appropriate menu options
//-----------------------------------------------------------------------------------

void cEdEditText::UpdateMenus(BR_CMenuBar &menuBar, BR_Boolean fCallSupervisor)
{
	BR_CEditText::UpdateMenus(menuBar, fCallSupervisor);

	menuBar.EnableCmd(cmdSelectAll);
	menuBar.EnableCmd(cmdFind);

	// Has a Find been performed?
	if (fFindText.GetLength() > 0)
   		menuBar.EnableCmd(cmdFindAgain);

	if (!IsReadOnly())
	{
		menuBar.EnableCmd(cmdReplace);

		// Has a Replace been performed?
		if (fReplaceFindText.GetLength() > 0)
			menuBar.EnableCmd(cmdReplaceAgain);
	}
	
	menuBar.EnableCmd(cmdGoToLine);
}

//-----------------------------------------------------------------------------------
// cEdEditText::DoCommand
//
// Called to process commands
//-----------------------------------------------------------------------------------

BR_Boolean cEdEditText::DoCommand(CommandId cmdId,
								  BR_CObject *notifier,
								  BR_Boolean fCallSupervisor)
{
   BR_Boolean commandHandled = TRUE;					// Return value

	switch (cmdId)
	{
	 	case cmdSelectAll:
			DoSelectAll();
			break;
			
	 	case cmdFind:
			DoFind();
			break;
			
 		case cmdFindAgain:
			DoFindAgain();
			break;
			
	 	case cmdReplace:
			DoReplace();
			break;
			
 		case cmdReplaceAgain:
			DoReplaceAgain();
			break;
			
 		case cmdGoToLine:
			DoGoToLine();
			break;
			
		default:
        	commandHandled = BR_CEditText::DoCommand(cmdId,
													 notifier,
													 fCallSupervisor);
			break;
	}
	return commandHandled;
}

//-----------------------------------------------------------------------------------
// cEdEditText::DoSelectAll
//
// Selects the entire text in the editor
//-----------------------------------------------------------------------------------

void cEdEditText::DoSelectAll()
{
	// Move the cursor to the last line and column
	SetPosition(GetLastLineNumber(), GetLine(GetLastLineNumber())->GetLength()); 

	// Select the entire text
	CreateSelection(0, 0, GetLineCount(), 0);
}

//-----------------------------------------------------------------------------------
// cEdEditText::DoFind
//
// Displays a find dialog and then searches for the specified text,starting from the 
// beginning of the document.
//-----------------------------------------------------------------------------------

void cEdEditText::DoFind()
{
	BR_CDialogDirector findDialog(this, NULL, *GetApplication(), IDV_FIND_DIALOG);
	findDialog.ShowDialog();
	
	if (findDialog.GetResult() == cmdOk)
	{
		findDialog.GetSubviewText(IDT_SEARCH_TEXT, fFindText);
		SearchText(fFindText, TRUE);
	}
}


//-----------------------------------------------------------------------------------
// cEdEditText::DoFindAgain
//
// Searches for the text that was previously found, starting from the current cursor
// position.
//-----------------------------------------------------------------------------------

void cEdEditText::DoFindAgain()
{
	SearchText(fFindText, FALSE);
}

//-----------------------------------------------------------------------------------
// cEdEditText::DoReplace
//
// Displays a replace dialog, searches for the specified text from the beginning of 
// the document, and replaces it with the new text.
//-----------------------------------------------------------------------------------

void cEdEditText::DoReplace()
{
	BR_CDialogDirector replaceDialog(this, NULL, *GetApplication(), IDV_REPLACE_DIALOG);
	replaceDialog.ShowDialog();
	
	if (replaceDialog.GetResult() == cmdOk)
	{
		replaceDialog.GetSubviewText(IDT_SEARCH_TEXT, fReplaceFindText);
		replaceDialog.GetSubviewText(IDT_REPLACE_TEXT, fReplacementText);
		
		ReplaceText(TRUE);
	}
}

//-----------------------------------------------------------------------------------
// cEdEditText::DoReplaceAgain
//
// Searches for the text that was previously replaced, starting from the current 
// cursor position, and replaces it with the new text.
//-----------------------------------------------------------------------------------

void cEdEditText::DoReplaceAgain()
{
	ReplaceText(FALSE);
}

//-----------------------------------------------------------------------------------
// cEdEditText::SearchText
//
// Searches the text for a specified string, starting from the beginning of the text
// or from the current cursor position.
//-----------------------------------------------------------------------------------

BR_Boolean cEdEditText::SearchText(const BR_TString<BR_Char> &searchString,
								   BR_Boolean fromStart)
{
	LineNumber			lineNo;			// the current line number
	BR_CDynamicCharString	lineString;	 	// the text in the current line
	ColumnNumber		stringOffset;	// the offset in the current line
							 			// where the substr was found
	BR_TMinimalStringTool<BR_Char> stringTool;

	// Get the starting column for searching the first line
	ColumnNumber startColumnNo = fromStart ? 0 : GetColumnNo();
	
	for (lineNo = fromStart ? 0 : GetLineNo(); lineNo < GetLineCount(); lineNo++)
	{
		// Get the line's text and search from the starting position
		GetLine(lineString, lineNo);
		
		if (stringTool.FindSubString(lineString, searchString, stringOffset, startColumnNo))
		{
			// Position the cursor after the replacement text
			SetPosition(lineNo, stringOffset + searchString.GetLength());

			// Select the found text
			SelectString(searchString, lineNo, stringOffset);
			return TRUE;
		}
		
		// Ensure that all lines after the first are searched from the beginning
		startColumnNo = 0;
	}

	// Couldn't find the text, so tell the user
	BR_CSystem::DoBeep();
	BR_CDialogDirector notFoundDialog(this, NULL, *GetApplication(), IDV_NOTFOUND_DIALOG);
	notFoundDialog.ShowDialog();
	
	return FALSE;
}

//-----------------------------------------------------------------------------------
// cEdEditText::ReplaceText
//
// Searches for a string and replaces it with another.
//-----------------------------------------------------------------------------------

void cEdEditText::ReplaceText(BR_Boolean fromStart)
{
	if (SearchText(fReplaceFindText, fromStart))
	{
		ColumnNumber startPos = GetColumnNo() - fReplaceFindText.GetLength();
		
		// Delete the text that was found
		BR_Boolean deletedEOL;
		DeleteCharacter(GetLineNo(),
						startPos,
						fReplaceFindText.GetLength(),
						deletedEOL);
					
		// Insert the replacement text
		InsertString(GetLineNo(), startPos, fReplacementText);
					 
		// Position the cursor after the replacement text
		SetPosition(GetLineNo(), startPos + fReplacementText.GetLength());
		
		// Select the replacement text
		SelectString(fReplacementText, GetLineNo(), startPos);

		// The text has been changed, so set the dirty bit
		GetDocument()->SetDirty(TRUE);
	}
}

//-----------------------------------------------------------------------------------
// cEdEditText::DoGoToLine
//
// Displays a GoTo dialog, and then goes to the specified line
//-----------------------------------------------------------------------------------

void cEdEditText::DoGoToLine()
{
	BR_CDialogDirector goToDialog(this, NULL, *GetApplication(), IDV_GOTOLINE_DIALOG);
	goToDialog.ShowDialog();

	if (goToDialog.GetResult() == cmdOk)
	{
		BR_CDynamicCharString lineStr;
		goToDialog.GetSubviewText(IDT_LINENO_TEXT, lineStr);

		// Check if something was entered, and if it's a valid, positive number
		BR_TMinimalStringTool<BR_Char> stringTool;
		long numberEntered = 1;
		BR_Boolean isNumber = TRUE;
		// Note 'stringTool' will have a GetLong() method to convert string to long:
		// 		isNumber = stringTool.GetLong(lineStr, numberEntered);
		if (isNumber && (numberEntered > 0))
			SetPosition(min(GetLineCount(), numberEntered) - 1, 0);
	}
}
