{*******************************************************************}
{                                                                   }
{               Copyright 1985 by IBS Corporation                   }
{                                                                   }
{*******************************************************************}
{                                                                   }
{   Program AT_SETUP will verify the time, date and other setup     }
{   options that are stored in the CMOS RAM of the Real Time        }
{   clock chip. If any of the options are incorrect then the user   }
{   will have the option to correct them.                           }
{                                                                   }
{*******************************************************************}

PROGRAM AT_Setup;


  CONST
    Register_D     = $0D;
    Register_E     = $0E;
    Diskette       = $10;
    Fixed_Disk     = $12;
    Equipment      = $14;
    Low_Base       = $15;
    High_Base      = $16;
    Low_Expansion  = $17;
    High_Expansion = $18;
    High_Check_Sum = $2E;
    Low_Check_Sum  = $2F;
    Information    = $33;


  TYPE
    Character_Set = SET OF CHAR;
    String_Name   = STRING[80];

    Regs = RECORD Case INTEGER OF
        1 :  (AX,BX,CX,DX,BP,SI,DI,DS,ES,Flags : INTEGER);
        2 :  (AL,AH,BL,BH,CL,CH,DL,DH : BYTE);
    END;


  VAR
    Response            : CHAR;
    System_Date         : String_Name;
    System_Time         : String_Name;
    Information_Byte    : BYTE;
    CMOS_RAM            : ARRAY [$0D..$3F] OF BYTE;
    Modified            : BOOLEAN;
    Monitor_Type        : INTEGER;
    Setup_Status        : BOOLEAN;

{*******************************************************************}
{                                                                   }
{   Ring the computers bell at a frequency of 600 Hertz for         }
{   500 milliseconds.                                               }
{                                                                   }
{*******************************************************************}

  PROCEDURE Ring_Bell;

     BEGIN
       SOUND(600);
       DELAY(300);
       NOSOUND;
     END;

{*******************************************************************}
{                                                                   }
{   String input routine that will not read more characters than    }
{   specified by LIMIT. The routine is terminated upon receiving    }
{   a carriage return (character 13).                               }
{                                                                   }
{   Entry : New_String - String to receive input data               }
{           Limit      - Maximum number of characters to read       }
{                        including the carriage return              }
{                                                                   }
{*******************************************************************}

  PROCEDURE Read_String(VAR New_String:String_Name;Limit:INTEGER);

   VAR  WasX : INTEGER;
        IsX  : INTEGER;
        CH   : CHAR;

   BEGIN
     New_String:='';
     WasX:=WhereX;
     IsX:=WasX;
     REPEAT
        READ(KBD,CH);
            IF (((CH) <> CHR(8)) OR ((IsX-WasX) <> 0))
            THEN
              BEGIN
                IF ((CH<>CHR(8)) AND ((IsX-WasX)=Limit))
                THEN
                  BEGIN
                    IF (CH <> CHR(13))
                    THEN Ring_Bell
                  END
                ELSE
                  BEGIN
                    WRITE(CH);
                    IF CH=CHR(13) THEN WRITE(CHR(10));
                  END;
                IF CH = CHR(8)
                THEN
                  BEGIN
                    WRITE(' ');
                    GotoXY(WhereX-1,WhereY);
                    New_String:=COPY(New_String,1,LENGTH(New_String)-1);
                  END
                ELSE
                  BEGIN
                    IF ((IsX-WasX <> Limit) OR (CH=CHR(13)))
                    THEN New_String:=CONCAT(New_String,CH);
                  END;
              END;
            IsX:=WhereX;
      UNTIL (CH=CHR(13));
   END;

{*******************************************************************}
{                                                                   }
{     Read a character from the keyboard and validate it against    }
{     the character set provided.                                   }
{                                                                   }
{     Exit:  Read_Response - Response character read from the       }
{                            keyboard ('Y' or 'N')                  }
{                                                                   }
{*******************************************************************}

  FUNCTION Read_Response:CHAR;

    VAR C_Input : String_Name;
        WasX    : INTEGER;
        IsX     : INTEGER;
        IsY     : INTEGER;
        Once    : BOOLEAN;

    BEGIN
      Once:=TRUE;
      WasX:=WhereX;
      IsX:=WasX;
      IsY:=WhereY;
      REPEAT
        GotoXY(WasX,IsY);
        WRITE(' ':IsX-WasX);
        GotoXY(WasX,IsY);
        IF NOT Once THEN Ring_Bell;
        Once:=FALSE;
        Read_String(C_Input,1);
        IsX:=WhereX;
      UNTIL (C_Input[1] IN ['Y','y','N','n']) AND
            (C_INPUT[2] = CHR(13));
      Read_Response:=UpCase(C_Input[1]);
    END;

{*******************************************************************}
{                                                                   }
{     Read an integer number from the keyboard and perform any      }
{     error checking needed.                                        }
{                                                                   }
{     Entry: Message  - Prompt Message                              }
{            Min      - Minimum allowed value                       }
{            Max      - Maximum allowed value                       }
{                                                                   }
{*******************************************************************}

  FUNCTION Read_Int(Message:String_Name;Min,Max:REAL):INTEGER;

    VAR WasX, Isx  : INTEGER;
        Result     : REAL;
        First_Time : BOOLEAN;

    BEGIN
      WRITE(Message);
      WasX:=WhereX;
      IsX:=WasX;
      First_Time:=TRUE;
      {$I-}
      REPEAT
        IF NOT(First_Time) THEN Ring_Bell;
        First_Time:=FALSE;
        GotoXY(WasX,WhereY);
        WRITE(' ':(IsX-WasX));
        GotoXY(WasX,WhereY);
        READ(Result);
        IsX:=WhereX;
      UNTIL (IOresult = 0) AND (Result >= Min) AND (Result <= Max) AND (IsX <> WasX);
      WRITELN;
      Read_Int:=TRUNC(Result);
      {$I+}
    END;

{*******************************************************************}
{                                                                   }
{    Read data from the specified port address.                     }
{                                                                   }
{    Entry : Port_Address - Port Address to read data from          }
{                                                                   }
{    Exit  : Data_Byte    - Data read from port address             }
{                                                                   }
{*******************************************************************}

  PROCEDURE Read_Byte(VAR Data_Byte:BYTE;Port_Address:INTEGER);

    BEGIN
      INLINE
        ($8B/$96/Port_Address/         {  MOV  DX,Port_Address[BP]  }
         $EC/                          {  IN   AL,DX                }
         $8B/$9E/Data_Byte/            {  MOV  BX,Data_Byte[BP]     }
         $88/$07);                     {  MOV  [BX],AL              }
    END;

{*******************************************************************}
{                                                                   }
{    Output data to the specified port address.                     }
{                                                                   }
{    Entry :  Data_Byte    - One byte value to output               }
{             Port_Address - Port Address to output to              }
{                                                                   }
{*******************************************************************}

  PROCEDURE Write_Byte(Data_Byte:BYTE;Port_Address:INTEGER);

    BEGIN
      INLINE
        ($8B/$96/Port_Address/         {  MOV  DX,Port_Address[BP]  }
         $8A/$86/Data_Byte/            {  MOV  AL,Data_Byte[BP]     }
         $EE);                         {  OUT  DX,AL                }
    END;

{*******************************************************************}
{                                                                   }
{   Read a byte from the CMOS clock ram. To read data you must      }
{   first write the CMOS ram address to port $70 and then read      }
{   the data from port $71.                                         }
{                                                                   }
{   Entry : CMOS_Address - CMOS ram address                         }
{                                                                   }
{   Exit  : Data_Byte    - Data read from CMOS ram                  }
{                                                                   }
{*******************************************************************}

  PROCEDURE Read_CMOS(VAR Data_Byte:BYTE;CMOS_Address:INTEGER);

    BEGIN
      Write_Byte(CMOS_Address,$70);
      Read_Byte(Data_Byte,$71);
    END;

{*******************************************************************}
{                                                                   }
{   Write a byte to the CMOS clock ram. To write data to the CMOS   }
{   ram you must first write the CMOS address to port $70 and then  }
{   write the data to port $71.                                     }
{                                                                   }
{   Entry : Data_Byte    - Data to write                            }
{           CMOS_Address - CMOS ram address                         }
{                                                                   }
{*******************************************************************}

  PROCEDURE Write_CMOS(Data_Byte:BYTE;CMOS_Address:INTEGER);

    BEGIN
      Write_Byte(CMOS_Address,$70);
      Write_Byte(Data_Byte,$71);
    END;

{*******************************************************************}
{                                                                   }
{   Read locations $10 through $20 of the Real Time Clock CMOS ram. }
{   This information will then be written back to the MC146818      }
{   every time a setup option is changed.                           }
{                                                                   }
{*******************************************************************}

  PROCEDURE Read_Setup_Values;

  VAR I          : INTEGER;
       Registers : Regs;


  BEGIN
    FOR I:= $0D TO $3F DO Read_CMOS(CMOS_RAM[I],I);
    Read_CMOS(Information_Byte,Information);
    Monitor_Type:=TRUNC((CMOS_RAM[Equipment] AND $30)/$10);
  END;

{*******************************************************************}
{                                                                   }
{  Write locations $10 through $20 of the Real Time Clock CMOS ram. }
{  After writing the setup data back to the CMOS ram, the checksum  }
{  will be written to CMOS locations $2E and $2F.                   }
{                                                                   }
{*******************************************************************}

  PROCEDURE Write_Setup_Values;

    VAR  I         : INTEGER;
         Check_Sum : INTEGER;

    BEGIN
      Check_Sum:=0;
      FOR I:=$10 TO $2D DO
        BEGIN
          Write_CMOS(CMOS_RAM[I],I);
          Check_Sum:=Check_Sum + CMOS_RAM[I];
        END;
      Write_CMOS(Check_Sum AND $FF,Low_Check_Sum);
      Write_CMOS(TRUNC(Check_Sum/256),High_Check_Sum);
      Modified:=TRUE;
   END;

{*******************************************************************}
{                                                                   }
{   Remove any leading spaces from the string specified             }
{                                                                   }
{   Entry : Parse_String - String to remove leading spaces          }
{                                                                   }
{*******************************************************************}

  PROCEDURE Parse_Spaces(VAR Parse_String:String_Name);
    BEGIN
      WHILE (Parse_String[1] = ' ') AND (LENGTH(Parse_String) <> 0) DO
               Parse_String:=COPY(Parse_String,2,LENGTH(Parse_String)-1);
    END;

{*******************************************************************}
{                                                                   }
{   Get the next character from the specified string. Parse_Char    }
{   is used by the date and time parse routines to insure that      }
{   the proper syntax is used. The next character in the string     }
{   is returned and the character is deleted from the start of      }
{   the string. If the string is empty then CHR(13) is returned.    }
{                                                                   }
{   Entry : Parse_String - Address of string                        }
{                                                                   }
{   Exit  : Parse_Char   - Next character from the string           }
{                                                                   }
{*******************************************************************}

  FUNCTION Parse_Char(VAR Parse_String:String_Name):CHAR;

    BEGIN
      IF LENGTH(Parse_String) = 0
      THEN Parse_Char:=CHR(13)
      ELSE
        BEGIN
          Parse_Spaces(Parse_String);
          Parse_Char:=Parse_String[1];
          Parse_String:=COPY(Parse_String,2,LENGTH(Parse_String)-1);
        END;
    END;

{*******************************************************************}
{                                                                   }
{   Display the current date stored in the real time clock.         }
{                                                                   }
{*******************************************************************}

  PROCEDURE Display_Date;

    VAR   Registers  :  Regs;

    BEGIN
      WITH Registers DO
        BEGIN
          AH:=04;
          INTR($1A,Registers);
          IF TRUNC(DH/16) <> 00 THEN WRITE(TRUNC(DH/16):1);
          WRITE(DH AND 15:1,'-');
          IF TRUNC(DL/16) <> 00 THEN WRITE(TRUNC(DL/16):1);
          WRITE(DL AND 15:1,'-');
          WRITELN(TRUNC(CH/16):1,CH AND 15:1,TRUNC(CL/16):1,CL AND 15:1);
        END;
    END;

{*******************************************************************}
{                                                                   }
{   Send the new date specified to the real time clock. Since       }
{   the BIOS provides support to read and write the real time       }
{   clock, the date is changed through INT $1A.                     }
{                                                                   }
{   Entry: New_Day   -  New Day must be a valid day for the month   }
{          New_Month -  Must be between 1 and 12                    }
{          New_Year  -  Must be between 1901 and 2099               }
{                                                                   }
{*******************************************************************}

  PROCEDURE Set_Date(New_Day,New_Month,New_Year:INTEGER);

    VAR  Registers  :  Regs;
         Temp       :  INTEGER;

    BEGIN
      WITH Registers DO
        BEGIN
          AH:=05;
          Temp:=TRUNC(New_Year/100);
          CH:=TRUNC(Temp/10)*16 + (Temp MOD 10);
          Temp:=New_Year MOD 100;
          CL:=TRUNC(Temp/10)*16 + (Temp MOD 10);
          DH:=TRUNC(New_Month/10)*16 + (New_Month MOD 10);
          DL:=TRUNC(New_Day/10)*16 + (New_Day MOD 10);
          INTR($1A,Registers);
        END;
    END;

{*******************************************************************}
{                                                                   }
{   Parse the date entered by the user and set the date in the      }
{   real time clock if the date is valid.                           }
{                                                                   }
{   Entry : New_Date   - Unparsed date string                       }
{                                                                   }
{   Exit :  Parse_Date - TRUE  if Date was valid                    }
{                        FALSE if invalid date specified            }
{                                                                   }
{*******************************************************************}

  FUNCTION Parse_Date(New_Date:String_Name):BOOLEAN;

    VAR Month        : String_Name;
        Year         : String_Name;
        Day          : String_Name;
        Month_Number : INTEGER;
        Year_Number  : INTEGER;
        Day_Number   : INTEGER;
        Status       : INTEGER;
        Max_Days     : INTEGER;
        CH           : CHAR;

    BEGIN
      Parse_Date:=FALSE;
      Month:='';
      Day:='';
      Year:='';
      CH:=Parse_Char(New_Date);
      WHILE CH IN ['0'..'9'] DO
        BEGIN
          Month:=CONCAT(Month,CH);
          CH:=Parse_Char(New_Date);
        END;
      IF (CH = '/') OR (CH = '-')
      THEN
        BEGIN
          CH:=Parse_Char(New_Date);
          WHILE CH IN ['0'..'9'] DO
            BEGIN
              Day:=CONCAT(Day,CH);
              CH:=Parse_Char(New_Date);
            END;
          IF (CH = '/') OR (CH = '-')
          THEN
            BEGIN
              CH:=Parse_Char(New_Date);
              WHILE CH IN ['0'..'9'] DO
                BEGIN
                  Year:=CONCAT(Year,CH);
                  CH:=Parse_Char(New_Date);
                END;
              IF (CH = CHR(13)) AND (Month <> '') AND
                 (Day <> '') AND (Year <> '')
              THEN
                BEGIN
                  VAL(Year,Year_Number,Status);
                  VAL(Month,Month_Number,Status);
                  VAL(Day,Day_Number,Status);
                  IF Year_Number < 100 THEN Year_Number:=Year_Number + 1900;
                  IF ((Month_Number >= 0) AND (Month_Number <= 12)) AND
                     ((Year_Number > 1900) AND (Year_Number < 2000))
                  THEN
                    BEGIN
                      CASE Month_Number OF
                        1 : Max_Days:=31;
                        2 : IF (Year_Number MOD 4) = 0
                            THEN Max_Days:=29
                            ELSE Max_Days:=28;
                        3 : Max_Days:=31;
                        4 : Max_Days:=30;
                        5 : Max_Days:=31;
                        6 : Max_Days:=30;
                        7 : Max_Days:=31;
                        8 : Max_Days:=31;
                        9 : Max_Days:=30;
                       10 : Max_Days:=31;
                       11 : Max_Days:=30;
                       12 : Max_Days:=31;
                      END;
                      IF Day_Number <= Max_Days
                      THEN
                        BEGIN
                          Set_Date(Day_Number,Month_Number,Year_Number);
                          Parse_Date:=TRUE;
                        END;
                    END;
                END;
            END;
        END;

    END;

{*******************************************************************}
{                                                                   }
{   Display the current time stored in the real time clock.         }
{                                                                   }
{*******************************************************************}

  PROCEDURE Display_Time;

    VAR   Registers  :  Regs;

    BEGIN
      WITH Registers DO
        BEGIN
          AH:=02;
          INTR($1A,Registers);
          IF TRUNC(CH/16) <> 00 THEN WRITE(TRUNC(CH/16):1);
          WRITE(CH AND 15:1,':');
          WRITE(TRUNC(CL/16):1,CL AND 15:1,':');
          WRITELN(TRUNC(DH/16):1,DH AND 15:1);
        END;
    END;

{*******************************************************************}
{                                                                   }
{   Send the new time specified to the real time clock. Since       }
{   the BIOS provides support to read and write the real time       }
{   clock, the time is changed through INT $1A.                     }
{                                                                   }
{   Entry: New_Hour     - Must be between 0 and 23                  }
{          New_Minutes  - Must be between 0 and 59                  }
{          New_Seconds  - Must be between 0 and 59                  }
{                                                                   }
{*******************************************************************}

  PROCEDURE Set_Time(New_Hour,New_Minutes,New_Seconds:INTEGER);

    VAR  Registers  :  Regs;

    BEGIN
      WITH Registers DO
        BEGIN
          AH:=03;
          CH:=TRUNC(New_Hour/10)*16 + (New_Hour MOD 10);
          CL:=TRUNC(New_Minutes/10)*16 + (New_Minutes MOD 10);
          DH:=TRUNC(New_Seconds/10)*16 + (New_Seconds MOD 10);
          DL:=00;
          INTR($1A,Registers);
        END;
    END;

{*******************************************************************}
{                                                                   }
{   Parse the time entered by the user and set the time in the      }
{   real time clock if the time is valid. All times must be         }
{   specified in military format (24 hour format).                  }
{                                                                   }
{   Entry : New_Time   - Unparsed time string                       }
{                                                                   }
{   Exit :  Parse_Time - TRUE  if time was valid                    }
{                        FALSE if invalid time specified            }
{                                                                   }
{*******************************************************************}

  FUNCTION Parse_Time(New_Time:String_Name):BOOLEAN;

    VAR Hour          : String_Name;
        Minute        : String_Name;
        Second        : String_Name;
        Hour_Number   : INTEGER;
        Minute_Number : INTEGER;
        Second_Number : INTEGER;
        Status        : INTEGER;
        CH            : CHAR;
        Good_Parse    : BOOLEAN;

    BEGIN
      Good_Parse:=FALSE;
      Hour:='';
      Minute:='';
      Second:='';
      Hour_Number:=0;
      Minute_Number:=0;
      Second_Number:=0;
      CH:=Parse_Char(New_Time);
      WHILE CH IN ['0'..'9'] DO
        BEGIN
          Hour:=CONCAT(Hour,CH);
          CH:=Parse_Char(New_Time);
        END;
      IF ((CH = ':') AND (LENGTH(Hour) < 5) AND (LENGTH(Hour) > 0))
      THEN
        BEGIN
          CH:=Parse_Char(New_Time);
          WHILE CH IN ['0'..'9'] DO
            BEGIN
              Minute:=CONCAT(Minute,CH);
              CH:=Parse_Char(New_Time);
            END;
          IF ((CH = ':') AND (LENGTH(Minute) < 5) AND (LENGTH(Minute) > 0))
          THEN
            BEGIN
              CH:=Parse_Char(New_Time);
              WHILE CH IN ['0'..'9'] DO
                BEGIN
                  Second:=CONCAT(Second,CH);
                  CH:=Parse_Char(New_Time);
                END;
              IF ((CH = CHR(13)) AND (LENGTH(Second) < 5) AND (LENGTH(Second) > 0))
              THEN Good_Parse:=TRUE;
            END
          ELSE IF ((CH=CHR(13)) AND (LENGTH(Minute) > 0) AND (LENGTH(Minute) < 5)) THEN Good_Parse:=TRUE;
        END
      ELSE IF ((CH=CHR(13)) AND (LENGTH(Hour) > 0) AND (LENGTH(Hour) < 5)) THEN Good_Parse:=TRUE;
      IF Good_Parse
      THEN
        BEGIN
          VAL(Hour,Hour_Number,Status);
          VAL(Minute,Minute_Number,Status);
          VAL(Second,Second_Number,Status);
          IF ((Hour_Number >= 0) AND (Hour_Number <= 23)) AND
             ((Minute_Number >= 0) AND (Minute_Number <=59)) AND
             ((Second_Number >= 0) AND (Second_Number <=59))
          THEN
            BEGIN
              Set_Time(Hour_Number,Minute_Number,Second_Number);
              Parse_Time:=TRUE;
            END;
       END;
      Parse_Time:=Good_Parse;
    END;

{*******************************************************************}
{                                                                   }
{    Draw a box at the specified cordinates.                        }
{                                                                   }
{    Entry :   X1,Y1  -  Upper left  corner cordinates              }
{              X2,Y2  -  Lower right corner cordinates              }
{                                                                   }
{*******************************************************************}

  PROCEDURE Draw_Box(X1,Y1,X2,Y2:INTEGER);

    VAR   I  :  INTEGER;

    BEGIN
      TextColor(15);
      GotoXY(X1,Y1);
      WRITE('');
      FOR I:= 1 TO ((X2-X1)-2) DO WRITE ('');
      WRITELN('');
      FOR I:= 1 TO ((Y2-Y1)-2) DO
        BEGIN
          GotoXY(X1,WhereY);
          WRITE('');
          GotoXY(X2-1,WhereY);
          WRITELN('');
        END;
      GotoXY(X1,WhereY);
      WRITE('');
      FOR I:= 1 TO ((X2-X1)-2) DO WRITE ('');
      WRITELN('');
      TextColor(7);
    END;

{*******************************************************************}
{                                                                   }
{   Display the diskette type at the current screen location.       }
{                                                                   }
{   Entry : Drive_Type - Current diskette type                      }
{                                                                   }
{*******************************************************************}

  PROCEDURE Diskette_Status(Drive_Type:BYTE);

    BEGIN
      CASE Drive_Type OF
        0 : WRITELN('Not Installed');
        1 : WRITELN('Double Sided');
        2 : WRITELN('High Capacity');
      ELSE  WRITELN('Invalid Type');
      END;
    END;

{*******************************************************************}
{                                                                   }
{   Display the fixed disk type at the current screen location      }
{                                                                   }
{   Entry : Drive_Type - Current Fixed Disk Type                    }
{                                                                   }
{*******************************************************************}

  PROCEDURE Fixed_Disk_Status(Drive_Type:BYTE);

    BEGIN
      CASE Drive_Type OF
        0     : WRITELN('Not Installed');
        1..14 : WRITELN('Drive Type ',Drive_Type:2);
        15    : WRITELN('Reserved');
      END;
    END;

{*******************************************************************}
{                                                                   }
{   Display the current monitor type at the screen location. The    }
{   monitor type is determined by the switch setting on the system  }
{   board. The value of this switch can be obtained through the     }
{   BIOS interface by using interrupt $11.                          }
{                                                                   }
{*******************************************************************}

  PROCEDURE Monitor_Status(X1,Y1:INTEGER);

    VAR  Registers  :  Regs;

    BEGIN
      CASE Monitor_Type OF
        0 : BEGIN
              WRITELN('Enhanced');
              GotoXY(X1+23,WhereY);
              WRITELN('Graphics');
              GotoXY(X1+23,WhereY);
              WRITELN('Adapter');
            END;
        1 : BEGIN
              WRITELN('Color Graphics');
              GotoXY(X1+23,WhereY);
              WRITELN('Adapter');
              GotoXY(X1+23,WhereY);
              WRITELN('40 Column');
            END;
        2 : BEGIN
              WRITELN('Color Graphics');
              GotoXY(X1+23,WhereY);
              WRITELN('Adapter');
              GotoXY(X1+23,WhereY);
              WRITELN('80 Columns');
            END;
        3 : BEGIN
              WRITELN('Monochrome');
              GotoXY(X1+23,WhereY);
              WRITELN('Adapter');
            END;
      END;
    END;

{*******************************************************************}
{                                                                   }
{   Verify that the system date and time are correct before         }
{   proceeding to the setup options.                                }
{                                                                   }
{   Exit: Correct_Time  - 'Y','y'  date and time are correct        }
{                       - 'N','n'  date and time are incorrect      }
{                                                                   }
{*******************************************************************}

  FUNCTION Correct_Time:CHAR;

    VAR  LX1  : INTEGER;
         LY1  : INTEGER;

    BEGIN
      LX1:=1;
      LY1:=1;
      ClrScr;
      Draw_Box(LX1,LY1,38,10);
      GotoXY(LX1+3,LY1+2);
      WRITE('Current system date : ');
      Display_Date;
      WRITELN;
      GotoXY(LX1+3,WhereY);
      WRITE('Current system time : ');
      Display_Time;
      WRITELN;
      GotoXY(LX1+3,WhereY);
      WRITE('Date and time correct [Y/N] ? ');
      Correct_Time:=Read_Response;
    END;

{*******************************************************************}
{                                                                   }
{   Display the current setup options on the screen and verify      }
{   the options are correct.                                        }
{                                                                   }
{   Exit : Correct_Setup - 'Y','y'  Setup is correct                }
{                        - 'N','n'  Setup is incorrect              }
{                                                                   }
{*******************************************************************}

  FUNCTION Correct_Setup:CHAR;

    VAR   LX1  : INTEGER;
          LY1  : INTEGER;

    BEGIN
      LX1:=1;
      LY1:=1;
      ClrScr;
      Draw_Box(LX1,LY1,40,17);
      GotoXY(LX1+2,LY1+2);
      WRITELN('  The following options are set:');
      WRITELN;
      GotoXY(LX1+2,WhereY);
      WRITE('  Diskette Drive A : ');
      Diskette_Status(TRUNC(CMOS_RAM[Diskette]/16));
      GotoXY(LX1+2,WhereY);
      WRITE('  Diskette Drive B : ');
      Diskette_Status(CMOS_RAM[Diskette] AND 15);
      GotoXY(LX1+2,WhereY);
      WRITE('Fixed Disk Drive C : ');
      Fixed_Disk_Status(TRUNC(CMOS_RAM[Fixed_Disk]/16));
      GotoXY(LX1+2,WhereY);
      WRITE('Fixed Disk Drive D : ');
      Fixed_Disk_Status(CMOS_RAM[Fixed_Disk] AND 15);
      GotoXY(LX1+2,WhereY);
      WRITE('  Base Memory Size : ');
      WRITELN(CMOS_RAM[High_Base] * $100 + CMOS_RAM[Low_Base]:5,' Kb');
      GotoXY(LX1+2,WhereY);
      WRITE('Expansion Mem Size : ');
      WRITELN(CMOS_RAM[High_Expansion] * $100 + CMOS_RAM[Low_Expansion]:5,' Kb');
      GotoXY(LX1+2,WhereY);
      WRITE('   Primary Display : ');
      Monitor_Status(LX1,LY1);
      GotoXY(LX1+2,WhereY+1);
      WRITE('   Are they correct [Y/N] ? ');
      Correct_Setup:=Read_Response;
    END;

{*******************************************************************}
{                                                                   }
{   Display copyright notice and verify that the user really        }
{   wanted to use this program.                                     }
{                                                                   }
{   Exit: Warning  -  'Y','y','N','n'                               }
{                                                                   }
{*******************************************************************}

  FUNCTION Warning:CHAR;

    VAR   LX1         : INTEGER;
          LY1         : INTEGER;
          Answer      : CHAR;
          Acknowledge : String_Name;

    BEGIN
      LX1:=1;
      LY1:=1;
      ClrScr;
      Draw_Box(LX1,LY1,37,14);
      GotoXY(LX1+3,LY1+2);
      WRITELN('IBS Setup Utility Version 1.30');
      GotoXY(LX1+3,WhereY);
      WRITELN('(C) Copyright IBS Corp. 1985');
      WRITELN;
      GotoXY(LX1+3,WhereY);
      WRITELN('Use this program only when');
      GotoXY(LX1+3,WhereY);
      WRITELN('the system time or date is');
      GotoXY(LX1+3,WhereY);
      WRITELN('incorrect, or when changing');
      GotoXY(LX1+3,WhereY);
      WRITELN('hardware options.');
      WRITELN;
      GotoXY(LX1+3,WhereY);
      WRITE('Continue [Y/N] ? ');
      Answer:=Read_Response;
      Warning:=Answer;
      IF (Answer = 'Y') AND (NOT(Setup_Status))
      THEN
        BEGIN
          ClrScr;
          Draw_Box(LX1,LY1,35,12);
          GotoXY(LX1+3,LY1+2);
          Ring_Bell;
          TextColor(0);
          TextBackground(7);
          WRITELN('          Warning          ');
          TextColor(7);
          TextBackground(0);
          GotoXY(LX1+3,WhereY+1);
          WRITELN('The current date, time, and');
          GotoXY(LX1+3,WhereY);
          WRITELN('SETUP options are probably ');
          GotoXY(LX1+3,WhereY);
          WRITELN('incorrect.');
          GotoXY(LX1+3,WhereY+1);
          WRITE('Press any key to continue ');
          Read_String(Acknowledge,1);
        END;
    END;

{*******************************************************************}
{
{   Verify the current system date. If the date is incorrect
{   then the user can set the current system date.
{
{*******************************************************************}

  PROCEDURE Get_Date;

    VAR   LX1        :  INTEGER;
          LY1        :  INTEGER;
          Right_Date : CHAR;
          First_Time : BOOLEAN;


    BEGIN
      LX1:=1;
      LY1:=1;
      ClrScr;
      Draw_Box(LX1,LY1,37,11);
      GotoXY(LX1+3,LY1+2);
      WRITE('Current date : ');
      Display_Date;
      WRITELN;
      GotoXY(LX1+3,WhereY);
      WRITE('Is the date correct [Y/N] ? ');
      Right_Date:=Read_Response;
      First_Time:=TRUE;
      IF Right_Date IN ['N']
      THEN
        REPEAT
          IF NOT(First_Time) THEN Ring_Bell;
          First_Time:=FALSE;
          GotoXY(LX1+3,LY1+6);
          WRITELN('Date format: MM-DD-YYYY');
          GotoXY(LX1+3,WhereY);
          WRITE('Enter correct date :            ');
          GotoXY(LX1+24,WhereY);
          Read_String(System_Date,11)
        UNTIL Parse_Date(System_Date);
    END;

{*******************************************************************}
{                                                                   }
{   Verify the current system time. If the time is incorrect        }
{   then the user can set the current system time.                  }
{                                                                   }
{*******************************************************************}

  PROCEDURE Get_Time;

    VAR   LX1        : INTEGER;
          LY1        : INTEGER;
          Right_Time : CHAR;
          First_Time : BOOLEAN;

    BEGIN
      LX1:=1;
      LY1:=1;
      ClrScr;
      Draw_Box(LX1,LY1,37,11);
      GotoXY(LX1+3,LY1+2);
      WRITE('Current time : ');
      Display_Time;
      WRITELN;
      GotoXY(LX1+3,WhereY);
      WRITE('Is the time correct [Y/N] ? ');
      Right_Time:=Read_Response;
      First_Time:=TRUE;
      IF Right_Time IN ['N','n']
      THEN
        REPEAT
          IF NOT(First_Time) THEN Ring_Bell;
          First_Time:=FALSE;
          GotoXY(LX1+3,LY1+6);
          WRITELN('24 hour time format: HH:MM:SS');
          GotoXY(LX1+3,WhereY);
          WRITE('Enter correct time :          ');
          GotoXY(LX1+24,WhereY);
          Read_String(System_Time,9)
        UNTIL Parse_Time(System_Time);
    END;

{*******************************************************************}
{                                                                   }
{   Determine the number of floppy drives available and setup       }
{   the proper drive types. Currently the only to valid drive       }
{   types are:                                                      }
{                                                                   }
{       0     - No drive present                                    }
{       1     - Double sided diskette drive (48 TPI)                }
{       2     - High Capacity diskette drive (96 TPI)               }
{       3..15 - Reserved                                            }
{                                                                   }
{*******************************************************************}

  PROCEDURE Get_Floppy_Drives;

    VAR  LX1           : INTEGER;
         LY1           : INTEGER;
         Right_Floppy  : CHAR;
         Drives        : INTEGER;
         Drive_Answer  : CHAR;
         First_Time    : BOOLEAN;
         Floppy_String : String_Name;
         Floppy_Status : INTEGER;

    BEGIN
      LX1:=1;
      LY1:=1;
      REPEAT
        ClrScr;
        Draw_Box(LX1,LY1,40,20);
        GotoXY(LX1+4,LY1+2);
        WRITELN('Your diskette drive types are:');
        GotoXY(LX1+4,WhereY+1);
        WRITE('Diskette Drive A : ');
        Diskette_Status(TRUNC(CMOS_RAM[Diskette]/16));
        GotoXY(LX1+4,WhereY);
        WRITE('Diskette Drive B : ');
        Diskette_Status(CMOS_RAM[Diskette] AND 15);
        GotoXY(LX1+4,WhereY+1);
        WRITE('Are they correct [Y/N] ? ');
        Right_Floppy:=Read_Response;
        IF Right_Floppy in ['N','n']
        THEN
          BEGIN
            First_Time:=TRUE;
              REPEAT
                GotoXY(LX1+4,LY1+9);
                IF NOT(First_Time) THEN Ring_Bell;
                First_Time:=FALSE;
                WRITELN('How many floppy drives');
                GotoXY(LX1+4,WhereY);
                WRITE('are attached [1/2] ?   ');
                GotoXY(WhereX-2,WhereY);
                Read_String(Floppy_String,1);
                IF   (LENGTH(Floppy_String) = 2) AND (Floppy_String[2]=CHR(13))
                THEN Floppy_String:=COPY(Floppy_String,1,LENGTH(Floppy_String)-1);
                Val(Floppy_String,Drives,Floppy_Status);
            UNTIL (Floppy_Status=0) AND ((Drives=1) OR (Drives=2));
            GotoXY(LX1+4,WhereY+2);
            WRITELN('Is Drive A a High');
            GotoXY(LX1+4,WhereY);
            WRITE('Capacity Drive [Y/N] ? ');
            Drive_Answer:=Read_Response;
            If Drive_Answer IN ['Y','y']
            THEN CMOS_RAM[Diskette]:=(CMOS_RAM[Diskette] AND 15) OR $20
            ELSE CMOS_RAM[Diskette]:=(CMOS_RAM[Diskette] AND 15) OR $10;
            IF Drives = 2
            THEN
              BEGIN
                GotoXY(LX1+4,WhereY+2);
                WRITELN('Is Drive B a High');
                GotoXY(LX1+4,WhereY);
                WRITE  ('Capacity Drive [Y/N] ? ');
                Drive_Answer:=Read_Response;
                If Drive_Answer IN ['Y','y']
                THEN CMOS_RAM[Diskette]:=(CMOS_RAM[Diskette] AND $F0) OR $2
                ELSE CMOS_RAM[Diskette]:=(CMOS_RAM[Diskette] AND $F0) OR $1
              END
            ELSE CMOS_RAM[Diskette]:=CMOS_RAM[Diskette] AND $F0;
            CMOS_RAM[Equipment]:=(CMOS_RAM[Equipment] AND $3E) + TRUNC((Drives-1) * $40) + 1;
          END;
      UNTIL Right_Floppy in ['Y','y'];
      Write_Setup_Values;
    END;

{*******************************************************************}
{                                                                   }
{   Determine the number of fixed disk drives available and setup   }
{   the proper drive types. The valid drive type are:               }
{                                                                   }
{       0     - No drive present                                    }
{       1     - 306 Cylinders, 4 Heads,                             }
{       2     - 615            4                                    }
{       3     - 615            6                                    }
{       4     - 940            8                                    }
{       5     - 940            6                                    }
{       6     - 615            4                                    }
{       7     - 462            8                                    }
{       8     - 733            5                                    }
{       9     - 900           15                                    }
{       10    - 820            3                                    }
{       11    - 855            5                                    }
{       12    - 855            7                                    }
{       13    - 306            8                                    }
{       14    - 733            7                                    }
{       15    - Reserved                                            }
{                                                                   }
{*******************************************************************}

  PROCEDURE Get_Fixed_Drives;

    VAR  LX1          : INTEGER;
         LY1          : INTEGER;
         Fixed_Right  : CHAR;
         Drives       : INTEGER;
         Drive_Answer : CHAR;
         Drive_Code   : INTEGER;
         First_Time   : BOOLEAN;
         Fixed_String : String_Name;
         Fixed_Status : INTEGER;

    BEGIN
      LX1:=1;
      LY1:=1;
      REPEAT
        ClrScr;
        Draw_Box(LX1,LY1,40,20);
        GotoXY(LX1+3,LY1+2);
        WRITELN('Your fixed disk drive types are:');
        GotoXY(LX1+3,WhereY+1);
        WRITE('Fixed Disk Drive C : ');
        Fixed_Disk_Status(TRUNC(CMOS_RAM[Fixed_Disk]/16));
        GotoXY(LX1+3,WhereY);
        WRITE('Fixed Disk Drive D : ');
        Fixed_Disk_Status(CMOS_RAM[Fixed_Disk] AND 15);
        GotoXY(LX1+3,WhereY+1);
        WRITE('Are they correct [Y/N] ? ');
        Fixed_Right:=Read_Response;
        IF Fixed_Right IN ['N','n']
        THEN
          BEGIN
            First_Time:=TRUE;
              REPEAT
                GotoXY(LX1+3,LY1+9);
                IF NOT(First_Time) THEN Ring_Bell;
                First_Time:=FALSE;
                WRITELN('How many fixed disks');
                GotoXY(LX1+3,WhereY);
                WRITE('are attached [0/1/2] ?   ');
                GotoXY(WhereX-2,WhereY);
                Read_String(Fixed_String,1);
                IF   (LENGTH(Fixed_String) = 2) AND (Fixed_String[2]=CHR(13))
                THEN Fixed_String:=COPY(Fixed_String,1,LENGTH(Fixed_String)-1);
                Val(Fixed_String,Drives,Fixed_Status);
            UNTIL (Fixed_Status=0) AND ((Drives=1) OR (Drives=2));
            IF Drives <> 0
            THEN
              BEGIN
                GotoXY(LX1+3,WhereY+2);
                WRITELN('Enter type for fixed');
                GotoXY(LX1+3,WhereY);
                Drive_Code:=Read_Int('disk C: [1..15] : ',1,15);
                CMOS_RAM[Fixed_Disk]:=(CMOS_RAM[Fixed_Disk] AND 15) OR (LO(Drive_Code) * $10);
                IF Drives = 2
                THEN
                  BEGIN
                    GotoXY(LX1+3,WhereY+1);
                    WRITELN('Enter type for fixed');
                    GotoXY(LX1+3,WhereY);
                    Drive_Code:=Read_Int('Disk D: [1..15] : ',1,15);
                    CMOS_RAM[Fixed_Disk]:=(CMOS_RAM[Fixed_Disk] AND $F0) OR LO(Drive_Code);
                  END
                ELSE CMOS_RAM[Fixed_Disk]:=CMOS_RAM[Fixed_Disk] AND $F0;
              END
            ELSE CMOS_RAM[Fixed_Disk]:=00;
          END;
      UNTIL Fixed_Right IN ['Y','y'];
      Write_Setup_Values;
    END;

{*******************************************************************}
{                                                                   }
{    Display the current primary display adapter and verify that    }
{    it is the correct adapter. If the user wants to change the     }
{    adapter (ie. change from monochrome to color graphics), then   }
{    he must turn of the computer, change the switch on the system  }
{    board, and then re-run the setup program. If the user has a    }
{    color graphics adapter then he has the option of selecting     }
{    40 or 80 column mode.                                          }
{                                                                   }
{*******************************************************************}

  PROCEDURE Get_Primary_Display;

    VAR  CMOS_Monitor_Type :  INTEGER;
         Screen_Size       :  INTEGER;
         Screen_String     :  STRING_Name;
         Screen_Status     :  INTEGER;
         First_Time        :  BOOLEAN;
         Registers         :  Regs;
         LX1               :  INTEGER;
         LY1               :  INTEGER;
         Answer            :  CHAR;
         CH                :  String_Name;

    BEGIN
      LX1:=1;
      LY1:=1;
      CMOS_Monitor_Type:=TRUNC((CMOS_RAM[Equipment] AND $30)/$10);
      REPEAT
        ClrScr;
        Draw_Box(LX1,LY1,40,19);
        GotoXY(LX1+3,LY1+2);
        WRITELN('Your primary display is : ');
        GotoXY(LX1+3,WhereY+1);
        CASE Monitor_Type OF
             0 : WRITELN('Enhanced Graphics Adapter');
           1,2 : WRITELN('Color Graphics Adapter');
             3 : WRITELN('Monochrome Display Adapter');
          END;
        GotoXY(LX1+3,WhereY+1);
        WRITE('Is this correct [Y/N] ? ');
        Answer:=Read_Response;
        IF Answer IN ['N','n']
        THEN
          BEGIN
            GotoXY(LX1+3,WhereY+2);
            WRITELN('The primary monitor types are:');
            GotoXY(LX1+3,WhereY+1);
            WRITELN('0 - Enhanced Graphics Adapter');
            GotoXY(LX1+3,WhereY);
            WRITELN('1 - Color Graphics 40 Columns');
            GotoXY(LX1+3,WhereY);
            WRITELN('2 - Color Graphics 80 Columns');
            GotoXY(LX1+3,WhereY);
            WRITELN('3 - Monochome Display Adapter');
            First_Time:=TRUE;
            REPEAT
              IF NOT(First_Time) then Ring_Bell;
              First_Time:=FALSE;
              GotoXY(LX1+3,LY1+15);
              WRITE('Enter your monitor type: ');
              Read_String(Screen_String,2);
               IF   ((LENGTH(Screen_String)=2) AND (Screen_String[2]=CHR(13)) )
              THEN Screen_String:=COPY(Screen_String,1,LENGTH(Screen_String)-1);
              Val(Screen_String,Monitor_Type,Screen_Status);
            UNTIL (Screen_Status=0) AND ((Monitor_Type >= 0) AND (Monitor_Type <= 3));
          END;
      UNTIL Answer IN ['Y','y'];
      IF (Monitor_Type <> 0)  AND (Monitor_Type <> CMOS_Monitor_Type)
      THEN
        BEGIN
          GotoXY(LX1+3,WhereY+2);
          WRITELN('To change your display type you');
          GotoXY(LX1+3,WhereY);
          WRITELN('must also change the display');
          GotoXY(LX1+3,WhereY);
          WRITELN('switch on the main system board.');
          GotoXY(LX1+3,WhereY);
          WRITE('Press any key to continue ');
          Read_String(CH,1);
        END;
      IF Monitor_Type <> CMOS_Monitor_Type
      THEN
        BEGIN
          CMOS_RAM[Equipment]:=(CMOS_RAM[Equipment] AND $CF) OR TRUNC(Monitor_Type * $10);
          Write_Setup_Values;
        END;
    END;

{*******************************************************************}
{                                                                   }
{  Verify the base memory size. If the memory size is incorrect     }
{  then the user can select the correct base memory size. Currently }
{  the only valid base memory sizes are:                            }
{                                                                   }
{     256K  -  On the system board                                  }
{     512K  -  On the system board                                  }
{     640K  -  512K on the system board                             }
{              128K on an expansion board                           }
{                                                                   }
{*******************************************************************}

  PROCEDURE Get_Base_Memory;

    VAR  LX1          : INTEGER;
         LY1          : INTEGER;
         Base_Size    : INTEGER;
         Answer       : CHAR;
         Base_String  : String_Name;
         Base_Status  : INTEGER;
         First_Time   : BOOLEAN;

    BEGIN
      LX1:=1;
      LY1:=1;
      REPEAT
        ClrScr;
        Draw_Box(LX1,LY1,35,18);
        GotoXY(LX1+3,LY1+2);
        WRITELN('Base memory options are:');
        GotoXY(LX1+3,WhereY+1);
        WRITELN('256K on main system board, or');
        GotoXY(LX1+3,WhereY);
        WRITELN('512K on main system board, or');
        GotoXY(LX1+3,WhereY);
        WRITELN('640K with 512Kb on the main');
        GotoXY(LX1+3,WhereY);
        WRITELN('     board and 128Kb on an');
        GotoXY(LX1+3,WhereY);
        WRITELN('     expansion board');
        GotoXY(LX1+3,WhereY+1);
        WRITE('Your base memory is : ');
        WRITELN(CMOS_RAM[High_Base] * $100 + CMOS_RAM[Low_Base]:3,' Kb');
        GotoXY(LX1+3,WhereY+1);
        WRITE('Is this correct [Y/N] ? ');
        Answer:=Read_Response;
        First_Time:=TRUE;
        IF Answer in ['N','n']
        THEN
          BEGIN
            REPEAT
              GotoXY(LX1+3,LY1+14);
              IF NOT(First_Time) THEN Ring_Bell;
              First_Time:=FALSE;
              WRITE('Enter base memory size :     ');
              GotoXY(WhereX-4,WhereY);
              Read_String(Base_String,4);
              IF ((LENGTH(Base_String)=4) AND (Base_String[4]=CHR(13)))
              THEN Base_String:=COPY(Base_String,1,LENGTH(Base_String)-1);
              Val(Base_String,Base_Size,Base_Status);
            UNTIL (Base_Status=0) AND ((Base_Size=256) OR (Base_Size=512) OR (Base_Size=640));
            CMOS_RAM[Low_Base]:=LO(Base_Size);
            CMOS_RAM[High_Base]:=HI(Base_Size);
            Write_Setup_Values;
          END;
      UNTIL Answer in ['Y','y','N','n'];
    END;

{*******************************************************************}
{                                                                   }
{  Verify the expanded memory size. If the expanded memory size is  }
{  incorrect then the user can select the correct expanded memory   }
{  size.  Expanded memory can be from 0 Kb to 15360 Kb in           }
{  increment of 64Kb.                                               }
{                                                                   }
{*******************************************************************}

  PROCEDURE Get_Extended_Memory;

    VAR  LX1              : INTEGER;
         LY1              : INTEGER;
         Expansion_Size   : INTEGER;
         Expansion_String : String_Name;
         Expansion_Status : INTEGER;
         First_Time       : BOOLEAN;
         Answer           : CHAR;

    BEGIN
      LX1:=1;
      LY1:=1;
      REPEAT
        ClrScr;
        Draw_Box(LX1,LY1,40,13);
        GotoXY(LX1+2,LY1+2);
        WRITELN('Extended memory ranges from 0 Kb');
        GotoXY(LX1+2,WhereY);
        WRITELN('to 15360 Kb in increments of 64 Kb');
        GotoXY(LX1+2,WhereY+1);
        WRITE('Your extended memory is : ');
        WRITELN(CMOS_RAM[High_Expansion] * $100 + CMOS_RAM[Low_Expansion]:5,' Kb');
        GotoXY(LX1+2,WhereY+1);
        WRITE('Is this correct [Y/N] ? ');
        Answer:=Read_Response;
        First_Time:=TRUE;
        IF Answer in ['N','n']
        THEN
          BEGIN
            REPEAT
              GotoXY(LX1+2,LY1+9);
              IF NOT(First_Time) THEN Ring_Bell;
              First_Time:=FALSE;
              WRITE('Enter expansion memory size :       ');
              GotoXY(WhereX-6,WhereY);
              Read_String(Expansion_String,6);
              IF  Expansion_String[LENGTH(Expansion_String)] = CHR(13)
              THEN Expansion_String:=COPY(Expansion_String,1,LENGTH(Expansion_String)-1);
              Val(Expansion_String,Expansion_Size,Expansion_Status);
            UNTIL (Expansion_Status=0) AND ((TRUNC(Expansion_Size/64)*64) = Expansion_Size);
            CMOS_RAM[Low_Expansion]:=LO(Expansion_Size);
            CMOS_RAM[High_Expansion]:=HI(Expansion_Size);
            Write_Setup_Values;
          END;
      UNTIL Answer in ['Y','y','N','n'];
    END;

{*******************************************************************}
{                                                                   }
{   Tell the user that the SETUP operation has been completed and   }
{   that his computer is now going to be reset. The reset is        }
{   generated by performing a long jump to the address F000:FFF0    }
{   in the BIOS area.                                               }
{                                                                   }
{*******************************************************************}

  PROCEDURE Adios;

    VAR   LX1  :  INTEGER;
          LY1  :  INTEGER;
          CH   :  String_Name;

    BEGIN
      IF NOT(Setup_Status)
      THEN Write_Setup_Values;
      LX1:=1;
      LY1:=1;
      ClrScr;
      GotoXY(LX1+4,LY1+2);
      WRITELN('SETUP program completed !');
      IF NOT(Modified)
      THEN
        BEGIN
          GotoXY(LX1+4,WhereY);
          WRITELN('No changes made.');
          Draw_Box(LX1,LY1,33,7)
        END
      ELSE
        BEGIN
          Draw_Box(LX1,LY1,33,9);
          GotoXY(LX1+4,LY1+3);
          WRITELN('Press any key and your');
          GotoXY(LX1+4,WhereY);
          WRITELN('system will reset with');
          GotoXY(LX1+4,WhereY);
          WRITE('all of your options set. ');
          Read_String(CH,1);
      INLINE
         ($EA/$F0/$FF/$00/$F0)           {  JMP  FAR F000:FFF0        }
        END;
    END;

{*******************************************************************}
{                                                                   }
{  Verify that the checksum of address $10 through $2D equal the    }
{  the checksum in address $2E and $2F. If they don't equal then    }
{  the CMOS RAM lost power or the system has been corrupted.        }
{                                                                   }
{*******************************************************************}

  PROCEDURE Check_Setup;

    VAR  Check_Sum  :  INTEGER;
         I          :  INTEGER;

    BEGIN
      Check_Sum:=0;
      FOR I:=$10 TO $2D DO Check_Sum:=Check_Sum + CMOS_RAM[I];
      IF TRUNC(CMOS_RAM[High_Check_Sum] * $100 + CMOS_RAM[Low_Check_Sum]) <> Check_Sum
      THEN
        BEGIN
          CMOS_RAM[$11]:=0;
          CMOS_RAM[$13]:=0;
          FOR I:=$19 TO $2D DO CMOS_RAM[I]:=0;
          Setup_Status:=FALSE;
        END;
    END;

{*******************************************************************}
{                                                                   }
{   Check and make sure that the CMOS RAM is in operation. If it    }
{   isn't then it could be either a dead or disconnected battery.   }
{                                                                   }
{   Exit : Check_Battery - TRUE  - CMOS RAM is OK                   }
{                        - FALSE - CMOS RAM is not functioning      }
{                                                                   }
{*******************************************************************}

  FUNCTION Check_Battery:BOOLEAN;

    VAR   LX1  :  INTEGER;
          LY1  :  INTEGER;

    BEGIN
      Check_Battery:=TRUE;
      IF (CMOS_RAM[Register_D] AND $80) = 00
      THEN
        BEGIN
          Check_Battery:=FALSE;
          LX1:=1;
          LY1:=1;
          ClrScr;
          Draw_Box(LX1,LY1,37,11);
          GotoXY(LX1+3,LY1+2);
          WRITELN('IBS Setup Utility Version 1.30');
          GotoXY(LX1+3,WhereY);
          WRITELN('(C) Copyright IBS Corp. 1985');
          WRITELN;
          GotoXY(LX1+3,WhereY);
          WRITELN('Your battery is either dead');
          GotoXY(LX1+3,WhereY);
          WRITELN('or disconnected. Correct the');
          GotoXY(LX1+3,WhereY);
          WRITELN('problem and then re-run SETUP');
          WRITELN;
          Ring_Bell;
       END
     ELSE
       BEGIN
         IF (CMOS_RAM[Register_E] AND $80) = $80
         THEN
           BEGIN
             CMOS_RAM[Register_E]:=CMOS_RAM[Register_E] AND $7F;
             Write_CMOS(CMOS_RAM[Register_E],Register_E);
             Setup_Status:=FALSE;
           END;
       END;
  END;

{*******************************************************************}
{                                                                   }
{                    M A I N   P R O G R A M                        }
{                                                                   }
{*******************************************************************}

  BEGIN
    Textcolor(07);
    ClrScr;
    Modified:=FALSE;
    Setup_Status:=TRUE;
    Read_Setup_Values;
    IF Check_Battery
    THEN
      BEGIN
        Check_Setup;
        Response:=Warning;
        IF Response IN ['Y','y']
        THEN
          BEGIN
            REPEAT
              Get_Date;
              Get_Time;
              Response:=Correct_Time;
            UNTIL Response IN ['Y','y'];
            IF   NOT(Setup_Status)
            THEN Response:='N'
            ELSE Response:=Correct_Setup;
            WHILE Response IN ['N','n'] DO
              BEGIN
                Get_Floppy_Drives;
                Get_Fixed_Drives;
                Get_Primary_Display;
                Get_Base_Memory;
                Get_Extended_Memory;
                Response:=Correct_Setup;
              END;
            Adios;
          END
        ELSE WRITELN(CHR(10));
      END;
  END.
