//
// MODULE NAME:  SMB.C.
//
// FUNCTIONAL DESCRIPTION.
//	This module handles packet summary and protocol decoding for Microsoft
//	SMB (Server Message Blocks).
//
//	There are two main entrypoints to this module.	PacketSummarySmb
//	returns a pointer to an ASCIIZ string allocated with malloc that
//	best describes the packet.  PacketDetailSmb returns a pointer to
//	a list of detail records (PDETREC) that describes the frame in detail.
//	If either function returns NULL, then no decoding is possible.
//
// MODIFICATION HISTORY.
//	S. E. Jones	92/04/09.	Original.
//	S. E. Jones	92/04/10.	Added LANMAN 1.0 SMB's.
//	S. E. Jones	92/07/13.	Fixed PacketFilterSmb's FILTER_SMB test.
//
// NOTICE:  Copyright (C) 1992-1993 General Software, Inc.  All rights reserved.
//

#include <stdlib.h>
#include <stdio.h>
#include <malloc.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <dos.h>
#include "..\inc\system.h"		// DOS operating system defns.
#include "..\cow\cow.h"                 // character-oriented windows.
#include "..\inc\ktypes.h"		// commonly-used types.
#include "analyzer.h"			// common stuff for all modules.

//
// SMB packet structures.
//

typedef struct _SMB {
    UCHAR Signature [4];		// 0xff, 'S', 'M', 'B'.
    UCHAR Cmd;				// command code.
    UCHAR ErrorClass;			// class of the error code.
    UCHAR Byte1;			// reserved (contains AH if DOS INT24 err).
    USHORT ErrorCode;			// status code.
    UCHAR Flags;			// flags, as follows.

#define SMB_FLAGS_RESPONSE	0x80	// this is a response SMB.

    USHORT ReservedWords [7];		// reserved words.
    USHORT Tid;                         // tree ID.
    USHORT Pid;                         // process ID.
    USHORT Uid;                         // user ID.
    USHORT Mid;                         // multiplex ID.
    UCHAR Wct;				// count of parameter words.
    USHORT Vwv [1];			// variable number of parameter words.

    //
    // Following this is a word containing the number of data bytes that
    // follow, followed by the data itself.
    //

} SMB, *PSMB;

//
// Core SMB protocol.
//

#define SMB_MKDIR		0x00
#define SMB_RMDIR		0x01
#define SMB_OPEN		0x02
#define SMB_CREATE		0x03
#define SMB_CLOSE		0x04
#define SMB_FLUSH		0x05
#define SMB_UNLINK		0x06
#define SMB_MV			0x07
#define SMB_GETATR		0x08
#define SMB_SETATR		0x09
#define SMB_READ		0x0a
#define SMB_WRITE		0x0b
#define SMB_LOCK		0x0c
#define SMB_UNLOCK		0x0d
#define SMB_CTEMP		0x0e
#define SMB_MKNEW		0x0f
#define SMB_CHKPTH		0x10
#define SMB_EXIT		0x11
#define SMB_LSEEK		0x12
#define SMB_TCON		0x70
#define SMB_TDIS		0x71
#define SMB_NEGPROT		0x72
#define SMB_DSKATTR		0x80
#define SMB_SEARCH		0x81
#define SMB_SPLOPEN		0xc0
#define SMB_SPLWR		0xc1
#define SMB_SPLCLOSE		0xc2
#define SMB_SPLRETQ		0xc3
#define SMB_SENDS		0xd0
#define SMB_SENDB		0xd1
#define SMB_FWDNAME		0xd2
#define SMB_CANCELF		0xd3
#define SMB_GETMAC		0xd4
#define SMB_SENDSTRT		0xd5
#define SMB_SENDEND		0xd6
#define SMB_SENDTXT		0xd7

//
// LANMAN 1.0 extensions.
//

#define SMB_LOCK_READ		0x13
#define SMB_WRITE_UNLOCK	0x14
#define SMB_READ_BLOCK_RAW	0x1a
#define SMB_READ_BLOCK_MPX	0x1b
#define SMB_READ_BLOCK_SEC	0x1c
#define SMB_WRITE_BLOCK_RAW	0x1d
#define SMB_WRITE_BLOCK_MPX	0x1e
#define SMB_WRITE_BLOCK_SEC	0x1f
#define SMB_WRITE_COMPLETE	0x20	// write complete RESPONSE.
#define SMB_SET_ATTR_EXPANDED	0x22
#define SMB_GET_ATTR_EXPANDED	0x23
#define SMB_LOCKING_AND_X	0x24
#define SMB_TRANSACTION         0x25
#define SMB_TRANSACTION_SEC	0x26
#define SMB_IOCTL		0x27
#define SMB_IOCTL_SEC		0x28
#define SMB_COPY		0x29
#define SMB_MOVE		0x2a
#define SMB_ECHO		0x2b
#define SMB_WRITE_CLOSE         0x2c
#define SMB_OPEN_AND_X		0x2d
#define SMB_READ_AND_X		0x2e
#define SMB_WRITE_AND_X         0x2f
#define SMB_SESSION_SETUP_AND_X 0x73
#define SMB_TCON_AND_X		0x75
#define SMB_FIND_FIRST		0x82
#define SMB_FIND_UNIQUE         0x83
#define SMB_FIND_CLOSE		0x84
#define SMB_INVALID		0xfe

//
// LANMAN 1.2 extensions.
//

#define SMB_TRANSACT2		0x32
#define SMB_TRANSACT2_SEC	0x33
#define SMB_FIND_CLOSE_12	0x34
#define SMB_FIND_NOTIFY_CLOSE	0x35
#define SMB_LOGOFF_AND_X	0x74

static UCHAR SmbSig [4] = { 0xff, 'S', 'M', 'B' };

static UCHAR *DataPtr;			// returned by NextBufStr.
static UCHAR DataType;			// returned by NextBufStr.

#define DATA_TYPE_RAWDATA	0x01	// format is 16-bit length & then data.
#define DATA_TYPE_DIALECT	0x02	// dialect string (ASCIIZ).
#define DATA_TYPE_PATHNAME	0x03	// pathname string (ASCIIZ).
#define DATA_TYPE_ASCII         0x04	// ASCIIZ string.
#define DATA_TYPE_VARIABLE	0x05	// just the same as RAWDATA.

//
// Routines in other modules.
//

//
// Routines in this module.
//

BOOLEAN PacketReqTypeSmb (PSMB Buffer, USHORT BufferSize)
{
    USHORT i;

    for (i=0; i<4; i++) {
	if (SmbSig [i] != Buffer->Signature [i]) {
	    return FALSE;		// this isn't an SMB.
	}
    }
    switch (Buffer->Cmd) {

	//
	// SMB CORE Protocol.
	//

	case SMB_MKDIR:         CountFileIoType (FILEIO_MKDIR); break;
	case SMB_RMDIR:         CountFileIoType (FILEIO_RMDIR); break;
	case SMB_OPEN:		CountFileIoType (FILEIO_OPEN); break;
	case SMB_MKNEW:
	case SMB_CREATE:	CountFileIoType (FILEIO_CREATE); break;
	case SMB_CLOSE:         CountFileIoType (FILEIO_CLOSE); break;
	case SMB_UNLINK:	CountFileIoType (FILEIO_DELETE); break;
	case SMB_MV:		CountFileIoType (FILEIO_RENAME); break;
	case SMB_READ:		CountFileIoType (FILEIO_READ); break;
	case SMB_WRITE:         CountFileIoType (FILEIO_WRITE); break;
	case SMB_LOCK:		CountFileIoType (FILEIO_LOCK); break;
	case SMB_UNLOCK:	CountFileIoType (FILEIO_UNLOCK); break;
	case SMB_CTEMP:         CountFileIoType (FILEIO_CREATE_TEMP); break;
	case SMB_EXIT:		CountFileIoType (FILEIO_DESTROY_PROCESS); break;
	case SMB_SEARCH:	CountFileIoType (FILEIO_SEARCH); break;
	case SMB_TCON:		CountFileIoType (FILEIO_LOGIN); break;
	case SMB_TDIS:		CountFileIoType (FILEIO_LOGOUT); break;

	case SMB_FLUSH:
	case SMB_GETATR:
	case SMB_SETATR:
	case SMB_CHKPTH:
	case SMB_LSEEK:
	case SMB_NEGPROT:
	case SMB_DSKATTR:
	case SMB_SPLOPEN:
	case SMB_SPLWR:
	case SMB_SPLCLOSE:
	case SMB_SPLRETQ:
	case SMB_SENDS:
	case SMB_SENDB:
	case SMB_FWDNAME:
	case SMB_CANCELF:
	case SMB_GETMAC:
	case SMB_SENDSTRT:
	case SMB_SENDEND:
	case SMB_SENDTXT:
	    break;

	//
	// LANMAN 1.0 protocol extensions.
	//

	case SMB_LOCK_READ:
	    CountFileIoType (FILEIO_READ);
	    CountFileIoType (FILEIO_WRITE);
	    break;
	case SMB_WRITE_UNLOCK:
	    CountFileIoType (FILEIO_WRITE);
	    CountFileIoType (FILEIO_UNLOCK);
	    break;
	case SMB_READ_BLOCK_RAW:
	case SMB_READ_BLOCK_MPX:
	case SMB_READ_BLOCK_SEC:
	    CountFileIoType (FILEIO_READ);
	    break;
	case SMB_WRITE_BLOCK_RAW:
	case SMB_WRITE_BLOCK_MPX:
	case SMB_WRITE_BLOCK_SEC:
	case SMB_WRITE_COMPLETE:
	    CountFileIoType (FILEIO_WRITE);
	    break;
	case SMB_MOVE:
	    CountFileIoType (FILEIO_RENAME);
	    break;
	case SMB_WRITE_CLOSE:
	    CountFileIoType (FILEIO_WRITE);
	    CountFileIoType (FILEIO_CLOSE);
	    break;
	case SMB_OPEN_AND_X:
	    CountFileIoType (FILEIO_OPEN);
	    break;
	case SMB_READ_AND_X:
	    CountFileIoType (FILEIO_READ);
	    break;
	case SMB_WRITE_AND_X:
	    CountFileIoType (FILEIO_WRITE);
	    break;
	case SMB_SET_ATTR_EXPANDED:
	case SMB_GET_ATTR_EXPANDED:
	case SMB_LOCKING_AND_X:
	case SMB_TRANSACTION:
	case SMB_TRANSACTION_SEC:
	case SMB_IOCTL:
	case SMB_IOCTL_SEC:
	case SMB_COPY:
	case SMB_ECHO:
	case SMB_SESSION_SETUP_AND_X:
	case SMB_INVALID:
	case 0xff:
	    break;
	case SMB_TCON_AND_X:
	    CountFileIoType (FILEIO_LOGIN);
	    break;
	case SMB_FIND_FIRST:
	case SMB_FIND_UNIQUE:
	case SMB_FIND_CLOSE:
	    CountFileIoType (FILEIO_SEARCH);
	    break;

	//
	// LANMAN 1.2 extensions.
	//

	case SMB_TRANSACT2:
	case SMB_TRANSACT2_SEC:
	    break;
	case SMB_FIND_CLOSE_12:
	case SMB_FIND_NOTIFY_CLOSE:
	    CountFileIoType (FILEIO_SEARCH);
	    break;
	case SMB_LOGOFF_AND_X:
	    CountFileIoType (FILEIO_LOGOUT);
    }
    return TRUE;			// we're returning SMB's and this is one.
} // PacketReqTypeSmb

BOOLEAN PacketFilterSmb (PSMB Buffer, USHORT BufferSize)
{
    USHORT i;

    if (!(CaptureFilter & FILTER_SMB)) {
	return FALSE;			// we aren't returning SMB's.
    }

    for (i=0; i<4; i++) {
	if (SmbSig [i] != Buffer->Signature [i]) {
	    return FALSE;		// this isn't an SMB.
	}
    }
    return TRUE;			// we're returning SMB's and this is one.
} // PacketFilterSmb

VOID AppendSmbAttr (PDETREC *q, UCHAR *Title, USHORT Value)
{
    Append8BitMask (q, "SMB:", Value,
		    Title,
		    "Read only",
		    "Hidden",
		    "System",
		    "Reserved",
		    "Directory",
		    "Archive",
		    NULL, NULL);
} // AppendSmbAttr

PUCHAR NextBufStr ()
{
    UCHAR *p, *q;

    q = DataPtr;			// remember this for the return.

    switch (DataType) {
	case DATA_TYPE_DIALECT:
	case DATA_TYPE_PATHNAME:
	case DATA_TYPE_ASCII:
	    for (p=DataPtr; *p; p++) ;	// find zero byte terminator.
	    p++;			// p = FWA, data type byte.
	    DataType = *(p++);		// save data type.
	    DataPtr = p;		// remember where we point to.
	    return q;
	case DATA_TYPE_RAWDATA:
	case DATA_TYPE_VARIABLE:
	default:
	    return q;
    }
} // NextBufStr

PUCHAR SmbCmdName (USHORT Cmd)
{
    switch (Cmd) {

	//
	// SMB CORE Protocol.
	//

	case SMB_MKDIR:         return "Make Directory";
	case SMB_RMDIR:         return "Remove Directory";
	case SMB_OPEN:		return "Open";
	case SMB_CREATE:	return "Create";
	case SMB_CLOSE:         return "Close";
	case SMB_FLUSH:         return "Flush";
	case SMB_UNLINK:	return "Delete";
	case SMB_MV:		return "Rename";
	case SMB_GETATR:	return "Get File Attributes";
	case SMB_SETATR:	return "Set File Attributes";
	case SMB_READ:		return "Read";
	case SMB_WRITE:         return "Write";
	case SMB_LOCK:		return "Lock";
	case SMB_UNLOCK:	return "Unlock";
	case SMB_CTEMP:         return "Create Temporary File";
	case SMB_MKNEW:         return "Make New File";
	case SMB_CHKPTH:	return "Check Path";
	case SMB_EXIT:		return "Process Exit";
	case SMB_LSEEK:         return "Seek";
	case SMB_TCON:		return "Tree Connect";
	case SMB_TDIS:		return "Tree Disconnect";
	case SMB_NEGPROT:	return "Negotiate Protocol";
	case SMB_DSKATTR:	return "Get Disk Attributes";
	case SMB_SEARCH:	return "Search";
	case SMB_SPLOPEN:	return "Open Spool File";
	case SMB_SPLWR:         return "Write Spool File";
	case SMB_SPLCLOSE:	return "Close Spool File";
	case SMB_SPLRETQ:	return "Return Print Queue";
	case SMB_SENDS:         return "Single Block Message";
	case SMB_SENDB:         return "Broadcast Message";
	case SMB_FWDNAME:	return "Forward Name";
	case SMB_CANCELF:	return "Cancel Forward";
	case SMB_GETMAC:	return "Get Machine Name";
	case SMB_SENDSTRT:	return "Message Start";
	case SMB_SENDEND:	return "Message End";
	case SMB_SENDTXT:	return "Message Text";

	//
	// LANMAN 1.0 protocol extensions.
	//

	case SMB_LOCK_READ:		return "Lock and read";
	case SMB_WRITE_UNLOCK:		return "Write and unlock";
	case SMB_READ_BLOCK_RAW:	return "Read block raw";
	case SMB_READ_BLOCK_MPX:	return "Read block multiplexed";
	case SMB_READ_BLOCK_SEC:	return "Read block secondary";
	case SMB_WRITE_BLOCK_RAW:	return "Write block raw";
	case SMB_WRITE_BLOCK_MPX:	return "Write block multiplexed";
	case SMB_WRITE_BLOCK_SEC:	return "Write block secondary";
	case SMB_WRITE_COMPLETE:	return "Write complete";
	case SMB_SET_ATTR_EXPANDED:	return "Set attributes expanded";
	case SMB_GET_ATTR_EXPANDED:	return "Get attributes expanded";
	case SMB_LOCKING_AND_X:         return "Locking and X";
	case SMB_TRANSACTION:		return "Transaction";
	case SMB_TRANSACTION_SEC:	return "Transaction secondary";
	case SMB_IOCTL:                 return "I/O control";
	case SMB_IOCTL_SEC:		return "I/O control secondary";
	case SMB_COPY:			return "Copy";
	case SMB_MOVE:			return "Move";
	case SMB_ECHO:			return "Echo";
	case SMB_WRITE_CLOSE:		return "Write and close";
	case SMB_OPEN_AND_X:		return "Open and X";
	case SMB_READ_AND_X:		return "Read and X";
	case SMB_WRITE_AND_X:		return "Write and X";
	case SMB_SESSION_SETUP_AND_X:	return "Session setup and X";
	case SMB_TCON_AND_X:		return "Tree connect and X";
	case SMB_FIND_FIRST:		return "Find first";
	case SMB_FIND_UNIQUE:		return "Find unique";
	case SMB_FIND_CLOSE:		return "Find close";
	case SMB_INVALID:		return "Invalid";
	case 0xff:			return "Empty";

	//
	// LANMAN 1.2 extensions.
	//

	case SMB_TRANSACT2:		return "Transact-2";
	case SMB_TRANSACT2_SEC:         return "Transact-2 Secondary";
	case SMB_FIND_CLOSE_12:         return "Find close LM1.2";
	case SMB_FIND_NOTIFY_CLOSE:	return "Find notify close";
	case SMB_LOGOFF_AND_X:		return "Logoff and X";

	default:			return "Undecoded";
    }
} // SmbCmdName

PUCHAR PacketSummarySmb (PSMB Buffer, USHORT BufferLength)
{
    USHORT i;
    UCHAR *p, cmd, *t;

    if (!(DisplayFilter & FILTER_SMB)) {
	return NULL;			// we aren't decoding SMB.
    }

    p = malloc (80);
    if (p == NULL) {
	return NULL;
    }

    for (i=0; i<4; i++) {
	if (SmbSig [i] != Buffer->Signature [i]) {
	    sprintf (p, "SMB %c Data", Buffer->Flags & SMB_FLAGS_RESPONSE ? 'R' : 'C');
	    return p;
	}
    }

    if (Buffer->Flags & SMB_FLAGS_RESPONSE) {
	if (Buffer->ErrorClass == 0) {
	    sprintf (p, "SMB R OK (%s)", SmbCmdName (Buffer->Cmd));
	} else {
	    sprintf (p, "SMB R %s Failed", SmbCmdName (Buffer->Cmd));
	}
    } else {
	sprintf (p, "SMB C %s", SmbCmdName (Buffer->Cmd));
    }
    return p;
} // PacketSummarySmb

VOID SmbTrailer (PDETREC *Ptr)
{
    AppendDetRec (Ptr, "SMB:");
} // SmbTrailer

VOID SmbDetailCommand (PDETREC *q, PSMB Buffer, USHORT BufferLength)
{
    USHORT smb_bcc;
    UCHAR *smb_buf;
    USHORT *smb_vwv;
    UCHAR *t;

    smb_vwv = Buffer->Vwv;
    smb_bcc = *(Buffer->Vwv + Buffer->Wct);
    smb_buf = (UCHAR *)(Buffer->Vwv + Buffer->Wct + 1);
    DataType = *smb_buf;
    DataPtr = smb_buf+1;

    switch (Buffer->Cmd) {
	case SMB_TCON:
	    sprintf (ScratchBuf, "SMB:  Path/Username = '%s'", NextBufStr ());
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Password = '%s'", NextBufStr ());
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Device Name = '%s'", NextBufStr ());
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_OPEN:
	    sprintf (ScratchBuf, "SMB:  Pathname = '%s'", NextBufStr ());
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Access = 0x%04x", smb_vwv [0]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Attribute = 0x%04x", smb_vwv [1]);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_CREATE:
	case SMB_UNLINK:
	case SMB_CTEMP:
	case SMB_MKNEW:
	    sprintf (ScratchBuf, "SMB:  Pathname = '%s'", NextBufStr ());
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Attribute = 0x%04x", smb_vwv [0]);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_CLOSE:
	case SMB_SPLCLOSE:
	    sprintf (ScratchBuf, "SMB:  Handle = 0x%04x", smb_vwv [0]);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_FLUSH:
	    sprintf (ScratchBuf, "SMB:  Handle = 0x%04x", smb_vwv [0]);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_READ:
	    sprintf (ScratchBuf, "SMB:  Handle = 0x%04x", smb_vwv [0]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Byte count = %u", smb_vwv [1]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Offset = %ld", smb_vwv [2], smb_vwv [3]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Bytes remaining = %u", smb_vwv [4]);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_WRITE:                 // CORE.
	case SMB_WRITE_UNLOCK:		// LANMAN 1.0.
	    sprintf (ScratchBuf, "SMB:  Handle = 0x%04x", smb_vwv [0]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Byte count = %u", smb_vwv [1]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Offset = %ld", smb_vwv [2], smb_vwv [3]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Bytes remaining = %u", smb_vwv [4]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Contains %u bytes of user data", smb_bcc);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_LSEEK:
	    switch (smb_vwv [1]) {
		case 0: t = "from start of file"; break;
		case 1: t = "from current position"; break;
		case 2: t = "from end of file"; break;
		default: t = "undecoded";
	    }
	    sprintf (ScratchBuf, "SMB:  Handle = 0x%04x", smb_vwv [0]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Mode = %u  (%s)", smb_vwv [1], t);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Offset = %ld", smb_vwv [2], smb_vwv [3]);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_MKDIR:
	case SMB_RMDIR:
	case SMB_CHKPTH:
	    sprintf (ScratchBuf, "SMB:  Pathname = '%s'", NextBufStr ());
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_MV:
	    sprintf (ScratchBuf, "SMB:  Old Pathname = '%s'", NextBufStr ());
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  New Pathname = '%s'", NextBufStr ());
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_GETATR:
	    sprintf (ScratchBuf, "SMB:  Pathname = '%s'", NextBufStr ());
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_SETATR:
	    sprintf (ScratchBuf, "SMB:  Pathname = '%s'", NextBufStr ());
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Attribute = 0x%04x", smb_vwv [0]);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_LOCK:
	case SMB_UNLOCK:
	    sprintf (ScratchBuf, "SMB:  Handle = 0x%04x", smb_vwv [0]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Byte count = %ld", smb_vwv [1], smb_vwv [2]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Offset = %ld", smb_vwv [3], smb_vwv [4]);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_NEGPROT:
	    sprintf (ScratchBuf, "SMB:  Dialect = '%s'", NextBufStr ());
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_SEARCH:		// CORE.
	case SMB_FIND_FIRST:		// LANMAN 1.0.
	case SMB_FIND_UNIQUE:		// LANMAN 1.0.
	case SMB_FIND_CLOSE:		// LANMAN 1.0.
	    sprintf (ScratchBuf, "SMB:  Pathname = '%s'", NextBufStr ());
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Max count = %u", smb_vwv [0]);
	    AppendDetRec (q, ScratchBuf);
	    AppendSmbAttr (q, "Search attributes", smb_vwv [1]);
	    break;
	case SMB_SPLOPEN:
	    switch (smb_vwv [1]) {
		case 0: t = "text mode"; break;
		case 1: t = "graphics mode"; break;
		default: t = "undecoded mode";
	    }
	    sprintf (ScratchBuf, "SMB:  Mode = 0x%04x (%s)", smb_vwv [1], t);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Identifier = '%s'", NextBufStr ());
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_SPLWR:
	    sprintf (ScratchBuf, "SMB:  Handle = 0x%04x", smb_vwv [0]);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_SPLRETQ:
	    sprintf (ScratchBuf, "SMB:  Max count = %u", smb_vwv [0]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Start index = %u", smb_vwv [1]);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_SENDS:
	case SMB_SENDB:
	case SMB_SENDSTRT:
	    sprintf (ScratchBuf, "SMB:  Source = '%s'", NextBufStr ());
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Destination = '%s'", NextBufStr ());
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_SENDTXT:
	case SMB_SENDEND:
	    sprintf (ScratchBuf, "SMB:  Message group ID = 0x%04x", smb_vwv [0]);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_FWDNAME:
	case SMB_CANCELF:
	    sprintf (ScratchBuf, "SMB:  Forwarded name = '%s'", NextBufStr ());
	    AppendDetRec (q, ScratchBuf);
	    break;

	//
	// LANMAN 1.0 protocol extensions.
	//

	case SMB_COPY:
	case SMB_MOVE:
	    sprintf (ScratchBuf, "SMB:  Destination Path TID = 0x%04x", smb_vwv [0]);
	    AppendDetRec (q, ScratchBuf);
	    Append8BitMask (q, "SMB:", smb_vwv [2],
			    "Copy flags",
			    "Destination must be a file",
			    "Destination must be a directory",
			    "Copy destination mode (0=binary, 1=ASCII)",
			    "Copy source mode (0=binary, 1=ASCII)",
			    "Verify writes",
			    "Tree copy",
			    NULL, NULL);
	    sprintf (ScratchBuf, "SMB:  Source file = '%s'", NextBufStr ());
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Destination file = '%s'", NextBufStr ());
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_ECHO:
	    sprintf (ScratchBuf, "SMB:  Reverb = %u", smb_vwv [0]);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_GET_ATTR_EXPANDED:
	    sprintf (ScratchBuf, "SMB:  Handle = 0x%04x", smb_vwv [0]);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_IOCTL:
	    sprintf (ScratchBuf, "SMB:  Handle = 0x%04x", smb_vwv [0]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Category = 0x%04x", smb_vwv [1]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Function = 0x%04x", smb_vwv [2]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Param bytes sent = %u, Data bytes sent = %u",
		     smb_vwv [3], smb_vwv [4]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Param bytes to return = %u, Data bytes to return = %u",
		     smb_vwv [5], smb_vwv [6]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Timeout (ms) = %ld", smb_vwv [7], smb_vwv [8]);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_IOCTL_SEC:
	    sprintf (ScratchBuf, "SMB:  Param bytes sent = %u, Data bytes sent = %u",
		     smb_vwv [0], smb_vwv [1]);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_LOCK_READ:
	    sprintf (ScratchBuf, "SMB:  Handle = 0x%04x", smb_vwv [0]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Bytes to lock and return = %u", smb_vwv [1]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Offset = %ld", smb_vwv [2], smb_vwv [3]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Bytes remaining = %u", smb_vwv [4]);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_LOCKING_AND_X:
	    sprintf (ScratchBuf, "SMB:  Piggy-back message = 0x%02x  (%s %s)",
		     (smb_vwv [0] & 0x00ff), SmbCmdName (smb_vwv [0] & 0x00ff),
		     Buffer->Flags & SMB_FLAGS_RESPONSE ? "Response" : "Command");
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Handle = 0x%04x", smb_vwv [2]);
	    AppendDetRec (q, ScratchBuf);
	    Append8BitMask (q, "SMB:", smb_vwv [3],
			    "Locking mode flags",
			    "Access (0=lock out all access, 1=permit other reads)",
			    "Single user total file unlock",
			    NULL, NULL, NULL, NULL, NULL, NULL);
	    sprintf (ScratchBuf, "SMB:  Timeout (ms) = %ld", smb_vwv [4], smb_vwv [5]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Unlocks/this request = %u", smb_vwv [6]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Locks/this request = %u", smb_vwv [7]);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_OPEN_AND_X:
	    sprintf (ScratchBuf, "SMB:  Piggy-back message = 0x%02x  (%s %s)",
		     (smb_vwv [0] & 0x00ff), SmbCmdName (smb_vwv [0] & 0x00ff),
		     Buffer->Flags & SMB_FLAGS_RESPONSE ? "Response" : "Command");
	    AppendDetRec (q, ScratchBuf);
	    Append8BitMask (q, "SMB:", smb_vwv [2],
			    "Open flags",
			    "Return additional information",
			    "Set single user total file lock",
			    "Opportunistic lock",
			    NULL, NULL, NULL, NULL, NULL);
	    sprintf (ScratchBuf, "SMB:  File open mode = 0x%04x", smb_vwv [3]);
	    AppendDetRec (q, ScratchBuf);
	    AppendSmbAttr (q, "Search attributes", smb_vwv [4]);
	    AppendSmbAttr (q, "File attributes", smb_vwv [5]);
	    switch (smb_vwv [8]) {
		case 0: t = "Fail if file exists"; break;
		case 1: t = "Open if file exists"; break;
		case 2: t = "Truncate if file exists"; break;
		default: t = "Undecoded";
	    }
	    sprintf (ScratchBuf, "SMB:  Open function = 0x%04x  (%s)",
		     smb_vwv [8], t);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Size = %ld", smb_vwv [9], smb_vwv [10]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Timeout (ms) = %ld", smb_vwv [11], smb_vwv [12]);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_READ_AND_X:
	    sprintf (ScratchBuf, "SMB:  Piggy-back message = 0x%02x  (%s %s)",
		     (smb_vwv [0] & 0x00ff), SmbCmdName (smb_vwv [0] & 0x00ff),
		     Buffer->Flags & SMB_FLAGS_RESPONSE ? "Response" : "Command");
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Handle = 0x%04x", smb_vwv [2]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Offset = %ldu", smb_vwv [3], smb_vwv [4]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Max bytes to return = %u", smb_vwv [5]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Min bytes to return = %u", smb_vwv [6]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Timeout (ms) = %ld", smb_vwv [7], smb_vwv [8]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Bytes remaining = %u", smb_vwv [9]);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_READ_BLOCK_MPX:
	case SMB_READ_BLOCK_RAW:
	    sprintf (ScratchBuf, "SMB:  Handle = 0x%04x", smb_vwv [0]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Offset = %ldu", smb_vwv [1], smb_vwv [2]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Max bytes to return = %u", smb_vwv [3]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Min bytes to return = %u", smb_vwv [4]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Timeout (ms) = %ld", smb_vwv [5], smb_vwv [6]);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_SESSION_SETUP_AND_X:
	    sprintf (ScratchBuf, "SMB:  Piggy-back message = 0x%02x  (%s %s)",
		     (smb_vwv [0] & 0x00ff), SmbCmdName (smb_vwv [0] & 0x00ff),
		     Buffer->Flags & SMB_FLAGS_RESPONSE ? "Response" : "Command");
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Consumer max buffer size = %u", smb_vwv [2]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Max mpx requests = %u", smb_vwv [3]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  VC ID = 0x%04x", smb_vwv [4]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Session key = 0x%08lx", smb_vwv [5], smb_vwv [6]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Account password = '%s'", NextBufStr ());
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Account name = '%s'", NextBufStr ());
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_SET_ATTR_EXPANDED:
	    sprintf (ScratchBuf, "SMB:  Handle = 0x%04x", smb_vwv [0]);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_TRANSACTION:		// LANMAN 1.0.
	case SMB_TRANSACT2:		// LANMAN 1.2.
	    sprintf (ScratchBuf, "SMB:  Transaction type = '%s'", NextBufStr ());
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Total param bytes sent = %u, Total data bytes sent = %u",
		     smb_vwv [0], smb_vwv [1]);
	    sprintf (ScratchBuf, "SMB:  Param bytes to return = %u, Data bytes to return = %u",
		     smb_vwv [2], smb_vwv [3]);
	    AppendDetRec (q, ScratchBuf);
	    Append8BitMask (q, "SMB:", smb_vwv [5],
			    "Additional flags",
			    "Disconnect this TID",
			    "One-way transaction",
			    NULL, NULL, NULL, NULL, NULL, NULL);
	    sprintf (ScratchBuf, "SMB:  Timeout (ms) = %ld", smb_vwv [6], smb_vwv [7]);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_TRANSACTION_SEC:	// LANMAN 1.0.
	case SMB_TRANSACT2_SEC:         // LANMAN 1.2.
	    sprintf (ScratchBuf, "SMB:  Total param bytes sent = %u, Total data bytes sent = %u",
		     smb_vwv [0], smb_vwv [1]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Param bytes sent/this msg = %u", smb_vwv [2]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Data bytes sent/this msg = %u", smb_vwv [5]);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_TCON_AND_X:
	    sprintf (ScratchBuf, "SMB:  Piggy-back message = 0x%02x  (%s %s)",
		     (smb_vwv [0] & 0x00ff), SmbCmdName (smb_vwv [0] & 0x00ff),
		     Buffer->Flags & SMB_FLAGS_RESPONSE ? "Response" : "Command");
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Password = '%s'", NextBufStr ());
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Server name = '%s'", NextBufStr ());
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Service name = '%s'", NextBufStr ());
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_WRITE_CLOSE:
	    sprintf (ScratchBuf, "SMB:  Handle = 0x%04x", smb_vwv [0]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Byte count = %u", smb_vwv [1]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Offset = %ld", smb_vwv [2], smb_vwv [3]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Contains %u bytes of user data", smb_bcc);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_WRITE_AND_X:
	    sprintf (ScratchBuf, "SMB:  Piggy-back message = 0x%02x  (%s %s)",
		     (smb_vwv [0] & 0x00ff), SmbCmdName (smb_vwv [0] & 0x00ff),
		     Buffer->Flags & SMB_FLAGS_RESPONSE ? "Response" : "Command");
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Handle = 0x%04x", smb_vwv [2]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Offset = %ld", smb_vwv [3], smb_vwv [4]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Timeout (ms) = %ld", smb_vwv [5], smb_vwv [6]);
	    AppendDetRec (q, ScratchBuf);
	    Append8BitMask (q, "SMB:", smb_vwv [7],
			    "Write mode flags",
			    "Complete write before return",
			    "Return remaining bytes to go",
			    "Use WriteRawNamedPipe",
			    "Start of pipe message",
			    NULL, NULL, NULL, NULL);
	    sprintf (ScratchBuf, "SMB:  Bytes remaining = %u", smb_vwv [8]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Contains %u bytes of user data", smb_bcc);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_WRITE_BLOCK_MPX:
	case SMB_WRITE_BLOCK_RAW:
	    sprintf (ScratchBuf, "SMB:  Handle = 0x%04x", smb_vwv [0]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Total write byte count = %u", smb_vwv [1]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Offset = %ld", smb_vwv [3], smb_vwv [4]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Timeout (ms) = %ld", smb_vwv [5], smb_vwv [6]);
	    AppendDetRec (q, ScratchBuf);
	    Append8BitMask (q, "SMB:", smb_vwv [7],
			    "Write mode flags",
			    "Complete write & send final response",
			    "Return remaining bytes to go",
			    NULL, NULL, NULL, NULL, NULL, NULL);
	    sprintf (ScratchBuf, "SMB:  Contains %u bytes of user data", smb_vwv [10]);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_READ_BLOCK_SEC:
	case SMB_WRITE_BLOCK_SEC:
	    sprintf (ScratchBuf, "SMB:  Raw data (%u bytes)", BufferLength);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_WRITE_COMPLETE:
	    sprintf (ScratchBuf, "SMB:  Total bytes written = %u", smb_vwv [0]);
	    AppendDetRec (q, ScratchBuf);
	    break;

	//
	// LANMAN 1.2 protocol extensions.
	//

	case SMB_FIND_CLOSE_12:
	case SMB_FIND_NOTIFY_CLOSE:
	    sprintf (ScratchBuf, "SMB:  Handle = 0x%04x", smb_vwv [0]);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_LOGOFF_AND_X:
	    sprintf (ScratchBuf, "SMB:  Piggy-back message = 0x%02x  (%s %s)",
		     (smb_vwv [0] & 0x00ff), SmbCmdName (smb_vwv [0] & 0x00ff),
		     Buffer->Flags & SMB_FLAGS_RESPONSE ? "Response" : "Command");
	    AppendDetRec (q, ScratchBuf);
	    break;

	//
	// The following protocols need no parameter interpretation.
	//

	case SMB_GETMAC:
	case SMB_DSKATTR:
	case SMB_EXIT:
	case SMB_TDIS:
	case SMB_INVALID:
	default:
	    return;
    }
} // SmbDetailCommand

VOID SmbDetailResponse (PDETREC *q, PSMB Buffer, USHORT BufferLength)
{
    USHORT smb_bcc;
    UCHAR *smb_buf;
    USHORT *smb_vwv;
    UCHAR *t;

    smb_vwv = Buffer->Vwv;
    smb_bcc = *(Buffer->Vwv + Buffer->Wct);
    smb_buf = (UCHAR *)(Buffer->Vwv + Buffer->Wct + 1);
    DataType = *smb_buf;
    DataPtr = smb_buf+1;

    switch (Buffer->Cmd) {
	case SMB_TCON:
	    sprintf (ScratchBuf, "SMB:  Maximum transmit size = %u", smb_vwv [0]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Returned Tree ID = 0x%04x", smb_vwv [1]);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_OPEN:
	    sprintf (ScratchBuf, "SMB:  Handle = 0x%04x", smb_vwv [0]);
	    AppendDetRec (q, ScratchBuf);
	    AppendSmbAttr (q, "File attributes", smb_vwv [1]);
	    sprintf (ScratchBuf, "SMB:  File size = %ld", smb_vwv [4], smb_vwv [5]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Access allowed = 0x%04x", smb_vwv [6]);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_CREATE:
	case SMB_MKNEW:
	case SMB_SPLOPEN:
	    sprintf (ScratchBuf, "SMB:  Handle = 0x%04x", smb_vwv [0]);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_READ:
	    sprintf (ScratchBuf, "SMB:  Byte count = %u", smb_vwv [0]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Contains %u bytes of user data", smb_bcc);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_WRITE:                 // CORE.
	case SMB_WRITE_CLOSE:		// LANMAN 1.0.
	case SMB_WRITE_UNLOCK:		// LANMAN 1.0.
	case SMB_WRITE_COMPLETE:	// LANMAN 1.0.
	case SMB_WRITE_BLOCK_SEC:	// LANMAN 1.0.
	    sprintf (ScratchBuf, "SMB:  Byte count = %u", smb_vwv [0]);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_LSEEK:
	    sprintf (ScratchBuf, "SMB:  Offset = %ld", smb_vwv [0], smb_vwv [1]);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_GETATR:
	    AppendSmbAttr (q, "File attributes", smb_vwv [0]);
	    sprintf (ScratchBuf, "SMB:  File size = %ld", smb_vwv [3], smb_vwv [4]);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_CTEMP:
	    sprintf (ScratchBuf, "SMB:  Handle = 0x%04x", smb_vwv [0]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  New pathname = '%s'", NextBufStr ());
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_DSKATTR:
	    sprintf (ScratchBuf, "SMB:  Allocation units = %u", smb_vwv [0]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Blocks/allocation unit = %u", smb_vwv [1]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Block size = %u", smb_vwv [2]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Free allocation units = %u", smb_vwv [3]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Media ID = 0x%04x", smb_vwv [4]);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_NEGPROT:
	    sprintf (ScratchBuf, "SMB:  Index = %u", smb_vwv [0]);
	    AppendDetRec (q, ScratchBuf);
	    if (Buffer->Wct >= 13) {
		sprintf (ScratchBuf, "SMB:  Security mode = 0x%04x", smb_vwv [1]);
		AppendDetRec (q, ScratchBuf);
		if (smb_vwv [1] & 0x0001) {
		    AppendDetRec (q, "SMB:      .... ...1 = User-level security selected");
		} else {
		    AppendDetRec (q, "SMB:      .... ...0 = Share-level security selected");
		}
		if (smb_vwv [1] & 0x0002) {
		    AppendDetRec (q, "SMB:      .... ..1. = Encrypt passwords");
		} else {
		    AppendDetRec (q, "SMB:      .... ..0. = Do NOT encrypt passwords");
		}
		sprintf (ScratchBuf, "SMB:  Max transmit size = %u", smb_vwv [2]);
		AppendDetRec (q, ScratchBuf);
		sprintf (ScratchBuf, "SMB:  Max simultaneous mux channels = %u", smb_vwv [3]);
		AppendDetRec (q, ScratchBuf);
		sprintf (ScratchBuf, "SMB:  Max virtual circuits = %u", smb_vwv [4]);
		AppendDetRec (q, ScratchBuf);
		Append8BitMask (q, "SMB:", smb_vwv [5],
				"Block mode flags",
				"Read Block Raw protocol supported",
				"Write Block Raw protocol supported",
				NULL, NULL, NULL, NULL, NULL, NULL);
		sprintf (ScratchBuf, "SMB:  Session key = 0x%08lx", smb_vwv [6], smb_vwv [7]);
		AppendDetRec (q, ScratchBuf);
		sprintf (ScratchBuf, "SMB:  Size of crypt key = %u", smb_bcc);
		AppendDetRec (q, ScratchBuf);
	    }
	    break;
	case SMB_SEARCH:		// CORE.
	case SMB_FIND_FIRST:		// LANMAN 1.0.
	case SMB_FIND_UNIQUE:		// LANMAN 1.0.
	    sprintf (ScratchBuf, "SMB:  Entries returned = %u", smb_vwv [0]);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_SPLRETQ:
	    sprintf (ScratchBuf, "SMB:  Count = %u", smb_vwv [0]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Restart index = %u", smb_vwv [1]);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_SENDSTRT:
	    sprintf (ScratchBuf, "SMB:  Message group ID = 0x%04x", smb_vwv [0]);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_GETMAC:
	    sprintf (ScratchBuf, "SMB:  Machine name = '%s'", NextBufStr ());
	    AppendDetRec (q, ScratchBuf);
	    break;

	//
	// LANMAN 1.0 protocol extensions.
	//

	case SMB_COPY:
	    sprintf (ScratchBuf, "SMB:  Number of files copied = %u", smb_vwv [0]);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_ECHO:
	    sprintf (ScratchBuf, "SMB:  ECHO Sequence Number = %u", smb_vwv [0]);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_GET_ATTR_EXPANDED:
	    sprintf (ScratchBuf, "SMB:  File end of data = %ld",
		     smb_vwv [6], smb_vwv [7]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  File allocation size = %ld",
		     smb_vwv [8], smb_vwv [9]);
	    AppendDetRec (q, ScratchBuf);
	    AppendSmbAttr (q, "File attributes", smb_vwv [10]);
	    break;
	case SMB_IOCTL_SEC:
	    sprintf (ScratchBuf, "SMB:  Total param bytes to return = %u", smb_vwv [0]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Total data bytes to return = %u", smb_vwv [1]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Param bytes returned/this buffer = %u", smb_vwv [2]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Data bytes returned/this buffer = %u", smb_vwv [5]);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_LOCK_READ:
	    sprintf (ScratchBuf, "SMB:  Locked bytes read = %u", smb_vwv [0]);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_MOVE:
	    sprintf (ScratchBuf, "SMB:  Number of files moved = %u", smb_vwv [0]);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_OPEN_AND_X:
	    sprintf (ScratchBuf, "SMB:  Piggy-back message = 0x%02x  (%s %s)",
		     (smb_vwv [0] & 0x00ff), SmbCmdName (smb_vwv [0] & 0x00ff),
		     Buffer->Flags & SMB_FLAGS_RESPONSE ? "Response" : "Command");
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Handle = 0x%04x", smb_vwv [2]);
	    AppendDetRec (q, ScratchBuf);
	    AppendSmbAttr (q, "File attributes", smb_vwv [3]);
	    sprintf (ScratchBuf, "SMB:  File size = %ld", smb_vwv [6], smb_vwv [7]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Access mode = 0x%04x", smb_vwv [8]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  File type = 0x%04x", smb_vwv [9]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  State of device = 0x%04x", smb_vwv [10]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Action taken = 0x%04x", smb_vwv [11]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Server-unique FILE ID = 0x%08lx",
		     smb_vwv [12], smb_vwv [13]);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_READ_AND_X:
	    sprintf (ScratchBuf, "SMB:  Piggy-back message = 0x%02x  (%s %s)",
		     (smb_vwv [0] & 0x00ff), SmbCmdName (smb_vwv [0] & 0x00ff),
		     Buffer->Flags & SMB_FLAGS_RESPONSE ? "Response" : "Command");
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Bytes remaining to be read = %u", smb_vwv [2]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Data bytes returned = %u", smb_vwv [5]);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_READ_BLOCK_MPX:
	    sprintf (ScratchBuf, "SMB:  Offset = %lu", smb_vwv [0], smb_vwv [1]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Total bytes being returned = %u", smb_vwv [2]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Bytes remaining to be read = %u", smb_vwv [3]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Bytes returned/this buffer = %u", smb_vwv [6]);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_SESSION_SETUP_AND_X:
	    sprintf (ScratchBuf, "SMB:  Piggy-back message = 0x%02x  (%s %s)",
		     (smb_vwv [0] & 0x00ff), SmbCmdName (smb_vwv [0] & 0x00ff),
		     Buffer->Flags & SMB_FLAGS_RESPONSE ? "Response" : "Command");
	    AppendDetRec (q, ScratchBuf);
	    Append8BitMask (q, "SMB:", smb_vwv [2],
			    "Session setup flags",
			    "Logged in as GUEST",
			    NULL, NULL, NULL, NULL, NULL, NULL, NULL);
	    break;
	case SMB_TRANSACTION_SEC:	// LANMAN 1.0.
	case SMB_TRANSACT2_SEC:         // LANMAN 1.2.
	    sprintf (ScratchBuf, "SMB:  Total param bytes to return = %u", smb_vwv [0]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Total data bytes to return = %u", smb_vwv [1]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Param bytes returned/this buffer = %u", smb_vwv [3]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Data bytes returned/this buffer = %u", smb_vwv [6]);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_TCON_AND_X:
	    sprintf (ScratchBuf, "SMB:  Piggy-back message = 0x%02x  (%s %s)",
		     (smb_vwv [0] & 0x00ff), SmbCmdName (smb_vwv [0] & 0x00ff),
		     Buffer->Flags & SMB_FLAGS_RESPONSE ? "Response" : "Command");
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Service responding = '%s'", NextBufStr ());
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_WRITE_AND_X:
	    sprintf (ScratchBuf, "SMB:  Piggy-back message = 0x%02x  (%s %s)",
		     (smb_vwv [0] & 0x00ff), SmbCmdName (smb_vwv [0] & 0x00ff),
		     Buffer->Flags & SMB_FLAGS_RESPONSE ? "Response" : "Command");
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Byte count = %u", smb_vwv [2]);
	    AppendDetRec (q, ScratchBuf);
	    sprintf (ScratchBuf, "SMB:  Bytes remaining = %u", smb_vwv [3]);
	    AppendDetRec (q, ScratchBuf);
	    break;
	case SMB_WRITE_BLOCK_MPX:
	case SMB_WRITE_BLOCK_RAW:
	    sprintf (ScratchBuf, "SMB:  Bytes remaining = %u", smb_vwv [0]);
	    AppendDetRec (q, ScratchBuf);
	    break;

	//
	// LANMAN 1.2 protocol extensions.
	//

	case SMB_LOGOFF_AND_X:
	    sprintf (ScratchBuf, "SMB:  Piggy-back message = 0x%02x  (%s %s)",
		     (smb_vwv [0] & 0x00ff), SmbCmdName (smb_vwv [0] & 0x00ff),
		     Buffer->Flags & SMB_FLAGS_RESPONSE ? "Response" : "Command");
	    AppendDetRec (q, ScratchBuf);
	    break;

	//
	// The following protocols need no parameter interpretation.
	//

	case SMB_TRANSACT2:
	case SMB_FIND_CLOSE_12:
	case SMB_FIND_NOTIFY_CLOSE:
	case SMB_READ_BLOCK_SEC:
	case SMB_TRANSACTION:
	case SMB_SET_ATTR_EXPANDED:
	case SMB_READ_BLOCK_RAW:
	case SMB_LOCKING_AND_X:
	case SMB_IOCTL:
	case SMB_FIND_CLOSE:
	case SMB_INVALID:
	case SMB_FWDNAME:
	case SMB_CANCELF:
	case SMB_SENDEND:
	case SMB_SENDTXT:
	case SMB_SENDS:
	case SMB_SENDB:
	case SMB_SPLWR:
	case SMB_SPLCLOSE:
	case SMB_CHKPTH:
	case SMB_EXIT:
	case SMB_LOCK:
	case SMB_UNLOCK:
	case SMB_SETATR:
	case SMB_UNLINK:
	case SMB_MV:
	case SMB_MKDIR:
	case SMB_RMDIR:
	case SMB_FLUSH:
	case SMB_CLOSE:
	case SMB_TDIS:
	default:
	    return;
    }
} // SmbDetailResponse

PDETREC PacketDetailSmb (PSMB Buffer, USHORT BufferLength)
{
    PDETREC r=NULL, q;
    UCHAR *p, cmd, *t;
    USHORT i;

    if (!(DisplayFilter & FILTER_SMB)) {
	return NULL;			// we aren't decoding SMB.
    }

    AppendHeader (&r, "SMB:  ----- SMB Message Header -----");

    for (i=0; i<4; i++) {
	if (SmbSig [i] != Buffer->Signature [i]) {
	    sprintf (ScratchBuf, "SMB:  Contains %u bytes of user data", BufferLength);
	    AppendDetRec (&r, ScratchBuf);
	    return r;
	}
    }

    sprintf (ScratchBuf, "SMB:  Message = 0x%02x  (%s %s)",
	     Buffer->Cmd, SmbCmdName (Buffer->Cmd),
	     Buffer->Flags & SMB_FLAGS_RESPONSE ? "Response" : "Command");
    AppendDetRec (&r, ScratchBuf);
    sprintf (ScratchBuf, "SMB:  Flags = 0x%02x", Buffer->Flags);
    AppendDetRec (&r, ScratchBuf);
    sprintf (ScratchBuf, "SMB:  Error code = 0x%02x, Error class = 0x%02x",
	     Buffer->ErrorCode, Buffer->ErrorClass);
    AppendDetRec (&r, ScratchBuf);
    sprintf (ScratchBuf, "SMB:  Tree ID = 0x%04x, Process ID = 0x%04x, User ID = 0x%04x",
	     Buffer->Tid, Buffer->Pid, Buffer->Mid);
    AppendDetRec (&r, ScratchBuf);

    if (Buffer->Flags & SMB_FLAGS_RESPONSE) {
	SmbDetailResponse (&r, Buffer, BufferLength);
    } else {
	SmbDetailCommand (&r, Buffer, BufferLength);
    }
    return r;
} // PacketDetailSmb
